
/*
 * NAME:        abt_emp.h - typedefs etc for User EMP interface
 * VERSION:     1.0
 *
 * COPYRIGHT:
 *
 *   (C) COPYRIGHT International Business Machines Corp. 1993, 2000
 *   All Rights Reserved
 *   Licensed Materials - Property of IBM
 *
 *   US Government Users Restricted Rights - Use, duplication or
 *   disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
 *
 *
 *
 *
 * DESCRIPTION: This file contains definitions for all the types used in the
 *              User EMP interface.
 */

#ifndef CICS_EMP_H

#define CICS_EMP_H



#include <cicstype.h>                   /* CICS-supplied datatypes      */
#include <cics_samples.h>		/* CICS-supplied macros		*/
#include <cics_eib.h>			/* CICS-supplied EIB		*/

#if !defined(_ERZ_cics_emp_clk_h_LOCK_)
#define _ERZ_cics_emp_clk_h_LOCK_
#include <cics_emp_clk.h>               /* Monitoring clocks            */
#endif

#if !defined(_ERZ_limits_h_LOCK_)
#define _ERZ_limits_h_LOCK_
#include <limits.h>
#endif



/*
 * OBJECT:  CICS_EMP_Return_t - The error codes returned by the User EMP
 * functions.
 */

enum CICS_EMP_Return
{
    CICS_EMP_RETURN_OK,
    CICS_EMP_RETURN_FAIL,
    CICS_EMP_RETURN_DISABLE,
    CICS_EMP_RETURN_WRITE
};

typedef enum CICS_EMP_Return CICS_EMP_Return_t;

/*
 * DESCRIPTION:
 * These error codes are returned by the User EMP interface functions.  They
 * are not defined using the SupER macros as they must be made publicly
 * available for users wishing to write their own EMP module.
 * :ul.
 * :li.
 * :hp2.CICS_EMP_RETURN_OK:ehp2. - The user function completed successfully.
 * :li.
 * :hp2.CICS_EMP_RETURN_FAIL:ehp2. - The user function detected an error.
 * User EMP are disabled for this task or region.
 * :li.
 * :hp2.CICS_EMP_RETURN_DISABLE:ehp2. - The user function request that
 * performance monitoring is disabled for this task or region.
 * :li.
 * :hp2.CICS_EMP_RETURN_WRITE:ehp2. - The user function request that
 * performance monitoring record is written.  This only applies to the
 * :hp4.CICS_EMP_EnterAPI:ehp4. function.
 * :eul.
 */

/*
 * OBJECT:  CICS_EMP_Finish - Parameter to CICS_EMP_ReStart.
 */

enum CICS_EMP_Finish
{
    CICS_EMP_FINISH_RESTART,
    CICS_EMP_FINISH_CLEANUP
};

/*
 * DESCRIPTION:
 * These codes are passed by the User EMP function :hp2.CICS_EMP_ReStart:ehp2.
 * to indicate the functionality required:
 * :ul.
 * :li.
 * :hp2.CICS_EMP_FINISH_RESTART:ehp2. - Restart any user clocks or counters
 * stopped by the preceding :hp2.CICS_EMP_WriteRecord:ehp2. call.
 * :li.
 * :hp2.CICS_EMP_FINISH_CLEANUP:ehp2. - Perform any cleanup functions required
 * to terminate the task.
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_ClockReturn_t - The return code from starting/stopping
 * a clock.
 */

enum CICS_EMP_ClockReturn
{
    CICS_EMP_CLOCK_RETURN_OK,
    CICS_EMP_CLOCK_RETURN_PHASE,
    CICS_EMP_CLOCK_RETURN_INVALID,
    CICS_EMP_CLOCK_RETURN_FAILURE
};

typedef enum CICS_EMP_ClockReturn       CICS_EMP_ClockReturn_t;

/*
 * DESCRIPTION:
 * These are the return codes from the :hp2.CICS_EMP_StartClock:ehp2. and
 * :hp2.CICS_EMP_StopClock:ehp2. functions.
 * :ul.
 * :li.
 * :hp2.CICS_EMP_CLOCK_RETURN_OK:ehp2. - the clock was stopped/started
 * successfully;
 * :li.
 * :hp2.CICS_EMP_CLOCK_RETURN_PHASE:ehp2. - the clock was stopped/started
 * successfully.  However, the start/stop was out of phase - the
 * appropriate inverse operation has not taken place.  In the case of
 * :hp2.CICS_EMP_StartClock:ehp2., a :hp2.CICS_EMP_StopClock:ehp2. call is
 * automatically performed.
 * :li.
 * :hp2.CICS_EMP_CLOCK_RETURN_INVALID:ehp2. - The ClockType is invalid.
 * :li.
 * :hp2.CICS_EMP_CLOCK_RETURN_FAILURE:ehp2. - The function could not
 * get the selected clock from the the operating system.
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_ClockType - The time to be recorded by a Monitor Clock.
 */

#define CICS_EMP_CLOCK_TYPE_MASK  0x0000F000  /* See Mask in CICS_EMP_Map  */

enum CICS_EMP_ClockType
{
    CICS_EMP_CLOCK_TYPE_NULL    = 0x00000000,
    CICS_EMP_CLOCK_TYPE_ELAPSED = 0x00009000, /* Record Elapsed time       */
    CICS_EMP_CLOCK_TYPE_BOTH    = 0x0000A000, /* Record user & system time */
    CICS_EMP_CLOCK_TYPE_SYSTEM  = 0x0000B000, /* Record system time        */
    CICS_EMP_CLOCK_TYPE_USER    = 0x0000C000  /* Record user time          */
};

/*
 * DESCRIPTION:
 * CICS_EMP_ClockType values define the times which can be recorded by a
 * Monitor Clock.
 */

/*
 * OBJECT: CICS_EMP_RegionData - Data supplied to the User EMP
 * Initialisation function to define the region.
 */

#define CICS_EMP_REGION_ID_LEN          8
#define CICS_EMP_RELEASE_ID_LEN         4
#define CICS_EMP_TRANSACTION_ID_LEN     4

struct CICS_EMP_RegionData
{
  cics_char_t          Region_Id[CICS_EMP_REGION_ID_LEN+1];
  cics_char_t          ReleaseId[CICS_EMP_RELEASE_ID_LEN+1];
  cics_char_t          TransactionId[CICS_EMP_TRANSACTION_ID_LEN+1];
};

/*
 * DESCRIPTION:
 * EMP_Init function is passed data describing the CICS region.  These
 * strings are in COBOL format.
 * :dl.
 * :dt.Region_Id
 * :dd.The region name.
 * :dt.ReleaseId
 * :dd.The release number.
 * :dt.TransactionId
 * :dd.The current transaction name.
 * :edl.
 */

/*
 * OBJECT: CICS_EMP_ENTER_API_Data - Buffer for passing EMP function addresses.
 */

#define CICS_EMP_ENTRY_NAME_LEN 8

struct CICS_EMP_ENTER_API_Data
{
   cics_ushort_t    ID;
   cics_char_t      EntryName [CICS_EMP_ENTRY_NAME_LEN];
   cics_ulong_t     Data1;
   cics_ulong_t     Data2;
};

/*
 * DESCRIPTION:
 * This structure is passed to the User EMP function
 * :hp4.CICS_EMP_EnterAPI:ehp4..  It contains the information from the
 * EXEC CICS ENTER API.
 * :ul.
 * :li.
 * :hp2.ID:ehp2. - The EMP identification number (0 to 255).
 * :li.
 * :hp2.EntryName:ehp2. - The EXEC CICS ENTER API ENTRYNAME parameter (COBOL
 * format).
 * :li.
 * :hp2.Data1:ehp2. - The DATA1 field from the EXEC CICS ENTER API.  This
 * is the most significant 32 bits from the FROM parameter.
 * :li.
 * :hp2.Data2:ehp2. - The DATA1 field from the EXEC CICS ENTER API.  This
 * is the least significant 32 bits from the FROM parameter.
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_HighWater_t - A :hp4.High Water:ehp4. value.
 */

struct CICS_EMP_HighWater
{
    cics_ulong_t    HighWater;
    cics_ulong_t    CurrentLevel;
};

typedef struct CICS_EMP_HighWater       CICS_EMP_HighWater_t;

/*
 * DESCRIPTION:
 * This structure contains the necessary information to record a
 * :hp4.High Water:ehp4. value.  The members of the structure are used as
 * follows:
 * :ul.
 * :li.
 * :hp2.HighWater:ehp2. - This member contains the maximum water level of the
 * member.
 * :li.
 * :hp2.CurrentLevel:ehp2. - This member contains the current water level.
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_Time_t - A :hp4.Time:ehp4. value.
 */

struct CICS_EMP_Time
{
    cics_slong_t tv_sec;
    cics_slong_t tv_usec;
};

typedef struct CICS_EMP_Time CICS_EMP_Time_t;

/*
 * DESCRIPTION:
 * This data type is used to represent time values.  The members of the
 * structure are as follows:
 * :ul.
 * :li.
 * :hp2.tv_sec:ehp2. - This member contains the number of seconds
 * :li.
 * :hp2.tv_usec:ehp2. - This member contains the number of microseconds
 * :eul.
 */



/*
 * OBJECT: CICS_EMP_OS_Clock_t - An :hp4.Operating System (OS) Clock:ehp4.
 * value.
 */

struct CICS_EMP_OS_Clock
{
    CICS_EMP_Time_t     StartValue;
    CICS_EMP_Time_t     EndValue;
};

typedef struct CICS_EMP_OS_Clock        CICS_EMP_OS_Clock_t;

/*
 * DESCRIPTION:
 * This structure contains the necessary information to record the change
 * in a clock maintained by the &os. operating system
 * between the start and end of a monitoring
 * period.  A monitoring period is the time between the start of a task or
 * the write of a performance record and the write of the first or next
 * record.  The difference in value between the start and end
 * is written to the Transient Data queue.  The members of the structure
 * are used as follows:
 * :ul.
 * :li.
 * :hp2.StartValue:ehp2. - This member contains the value of the clock at
 * the start of the monitoring period.
 * :li.
 * :hp2.EndValue:ehp2. - This member contains the value of the clock at
 * the end of the monitoring period.
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_OS_Count_t - An :hp4.Operating System (OS) Count:ehp4.
 * value.
 */

struct CICS_EMP_OS_Count
{
    cics_ulong_t    StartValue;
    cics_ulong_t    EndValue;
};

typedef struct CICS_EMP_OS_Count        CICS_EMP_OS_Count_t;

/*
 * DESCRIPTION:
 * This structure contains the necessary information to record the change
 * in an count maintained by &os. between the start and end of a monitoring
 * period.  The difference in value between the start and end is written to
 * the Transient Data queue.  The members of the structure are used as
 * follows:
 * :ul.
 * :li.
 * :hp2.StartValue:ehp2. - This member contains the value of the count at
 * the start of the monitoring period.
 * :li.
 * :hp2.EndValue:ehp2. - This member contains the value of the count at
 * the end of the monitoring period.
 * :eul.
 */


/*
 * OBJECT: CICS_EMP_EventTime_t - The Monitor Event Timer Structure.
 */

struct CICS_EMP_EventTime
{
    CICS_EMP_Time_t             StartTime;
    CICS_EMP_Clock_t            Clock;
    enum CICS_EMP_ClockType     Mode;
};

typedef struct CICS_EMP_EventTime       CICS_EMP_EventTime_t;

/*
 * DESCRIPTION:
 * This structure contains the necessary information to time an event.  The
 * structure is initialised by :hp4.CICS_EMP_StartClock:ehp4. and entered
 * into the appropriate Performance Monitor structure field by
 * :hp4.CICS_EMP_StopClock:ehp4..  The members of the structure are
 * used as follows:
 * :ul.
 * :li.
 * :hp2.StartTime:ehp2. - This member contains the value of the clock at
 * the time at which the timer was started.
 * :li.
 * :hp2.Clock:ehp2. - This member contains the clock accumulators.
 * :li.
 * :hp2.Mode:ehp2. - This member contains the system clocks which comprise
 * this timer.
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_Occupancy - A Performance Memory Occupancy structure.
 */

struct CICS_EMP_Occupancy
{
    CICS_EMP_Time_t     StartTime;
    cics_ulong_t        Amount;
    cics_uxlong_t       Occupancy;
};

typedef struct CICS_EMP_Occupancy       CICS_EMP_Occupancy_t;

/*
 * DESCRIPTION:
 * This structure contains the necessary information in order to generate
 * the memory occupancy statistics.  The members of the structure are
 * used as follows:
 * :ul.
 * :li.
 * :hp2.StartTime:ehp2. - This member contains the time at which the current
 * memory occupancy measurement started.  This field is set up by
 * :hp4.CICS_EMP_StartOccupancy:ehp4..
 * :li.
 * :hp2.Amount:ehp2. - This member contains the memory in use during the
 * occupancy measurement period.  This field is set up by
 * :hp4.CICS_EMP_StartOccupancy:ehp4..
 * :li.
 * :hp2.Occupancy:ehp2. - This member contains the accumulated
 * occupancy for the task.  This field is updated up by
 * :hp4.CICS_EMP_StopOccupancy:ehp4..
 * :eul.
 */

/*
 * OBJECT: CICS_EMP_TRANS_TYPE* - Transaction types.
 */

#define CICS_EMP_TRANS_TYPE_ATI   'A'   /* Attached by automatic        */
                                        /* transaction initiation (ATI) */
#define CICS_EMP_TRANS_TYPE_CONV  'C'   /* Second or subsequent part of */
                                        /* a conversational task        */
#define CICS_EMP_TRANS_TYPE_TRIG  'D'   /* Attached by transient data   */
                                        /* trigger level                */
#define CICS_EMP_TRANS_TYPE_SYS   'S'   /* System internal task         */
#define CICS_EMP_TRANS_TYPE_TERM  'T'   /* Attached from a terminal     */
#define CICS_EMP_TRANS_TYPE_PCONV 'Z'   /* Second or subsequent part of */
                                        /* a pseudo-conversation.       */

/*
 * DESCRIPTION:
 * CICS_EMP_TRANS_TYPE* defines the values of the :hp2.TransactionType:ehp2.
 * member in the CICS_EMP_PerformanceClass structure. Monitoring of the types
 * CICS_EMP_TRANS_TYPE_CONV and CICS_EMP_TRANS_TYPE_PCONV is not currently
 * supported.
 */

/*
 * OBJECT: CICS_EMP_REC_TYPE*(rectype) - Record types.
 */

#define CICS_EMP_REC_TYPE_CONV  'C'     /* Conversational record        */
#define CICS_EMP_REC_TYPE_ULM   'D'     /* ULM Request                  */
#define CICS_EMP_REC_TYPE_END   'T'     /* Task Termination             */
#define CICS_EMP_REC_TYPE_MIR   'M'     /* Long running mirror          */

/*
 * DESCRIPTION:
 * CICS_EMP_REC_TYPE* defines the values of the :hp2.RecordType:ehp2.
 * member in the CICS_EMP_PerformanceClass structure.
 */

/*
 * OBJECT: CICS_EMP_PerformanceClass - Performance Class Monitoring
 *                                     structure in TCA.
 */

#define CICS_EMP_OPERATOR_ID_SIZE       4
#define CICS_EMP_USER_ID_SIZE           8
#define CICS_EMP_TRANSACTION_ID_SIZE    4
#define CICS_EMP_TERMINAL_ID_SIZE       4
#define CICS_EMP_PROGRAM_ID_SIZE        8
#define CICS_EMP_SYSTEM_ID_SIZE         8
#define CICS_EMP_TID_SIZE               (8 + 11 + 2)
#define CICS_EMP_ABEND_SIZE             5

struct CICS_EMP_PerformanceClass
{

    CICS_EMP_EventTime_t        ElapsedTimePPCSNALink;
    CICS_EMP_EventTime_t        ElapsedTimeTCPLink;
    cics_ulong_t                ISCMessageInCount;
    cics_ulong_t                ISCMessageOutCount;


    cics_ulong_t                    StartRequests;


    CICS_EMP_EventTime_t        ElapsedTimeInFileMgr;


    CICS_EMP_EventTime_t        ElapsedTimeInFileIO;
    cics_ulong_t                FileGetCount;
    cics_ulong_t                FilePutCount;
    cics_ulong_t                FileBrowseCount;
    cics_ulong_t                FileAddCount;
    cics_ulong_t                FileDeleteCount;
    cics_ulong_t                FileTotal;


    CICS_EMP_EventTime_t        ElapsedTimeJournalIO;
    cics_ulong_t                JournalOutputCount;


    CICS_EMP_EventTime_t        ElapsedTimeTransientDataIO;
    cics_ulong_t                TransientDataGetCount;
    cics_ulong_t                TransientDataPutCount;
    cics_ulong_t                TransientDataPurgeCount;
    cics_ulong_t                TransientDataTotal;


    CICS_EMP_EventTime_t        ElapsedTimeTemporaryStorageIO;
    cics_ulong_t                TemporaryStorageGetCount;
    cics_ulong_t                TemporaryStorageAuxPutCount;
    cics_ulong_t                TemporaryStorageMainPutCount;
    cics_ulong_t                TemporaryStorageTotal;


    CICS_EMP_EventTime_t        ElapsedTimeInCICSSpace;
    CICS_EMP_EventTime_t        SystemUserTimeInCICSSpace;


    CICS_EMP_HighWater_t        TextSegmentSize;
    CICS_EMP_Occupancy_t        TextSegmentOccupancy;
    cics_ulong_t                LinkCount;
    cics_ulong_t                XctlCount;
    cics_ulong_t                LoadCount;
    CICS_EMP_EventTime_t        ElapsedProgramLoadTime;


    CICS_EMP_HighWater_t        DataSegmentSize;
    CICS_EMP_Occupancy_t        DataSegmentOccupancy;
    cics_ulong_t                TaskFlags;


    cics_ulong_t                    GetMainCount;
    cics_ulong_t                    FreeMainCount;
    cics_ulong_t                    GetMainSize;
    cics_ulong_t                    FreeMainSize;


    CICS_EMP_EventTime_t        ElapsedTimeInCompression;
    CICS_EMP_EventTime_t        ElapsedTimeInException;


    CICS_EMP_EventTime_t        ElapsedTimeSuspended;


    CICS_EMP_EventTime_t        ElapsedTimeInTransactionSch;
    cics_ulong_t                TransactionPriority;
    cics_ulong_t                TCASequenceNo;
    cics_char_t                 TransactionType;

    cics_char_t                 Program[CICS_EMP_PROGRAM_ID_SIZE];
    cics_char_t                 NetName[CICS_EMP_SYSTEM_ID_SIZE];


    cics_ulong_t                    SyncPointCount;
    cics_char_t                     TID[CICS_EMP_TID_SIZE];


    cics_ulong_t                BMSMapCount;
    cics_ulong_t                BMSInCount;
    cics_ulong_t                BMSOutCount;
    cics_ulong_t                BMSTotal;
    CICS_EMP_EventTime_t        ElapsedTimeTerminalIO;
    cics_ulong_t                PrimTermFacMessagesInCount;
    cics_ulong_t                PrimTermFacMessagesOutCount;
    cics_ulong_t                AltTermFacMessagesInCount;
    cics_ulong_t                AltTermFacMessagesOutCount;
    cics_ulong_t                PrimTermFacCharsInCount;
    cics_ulong_t                PrimTermFacCharsOutCount;
    cics_ulong_t                AltTermFacCharsInCount;
    cics_ulong_t                AltTermFacCharsOutCount;


    CICS_EMP_EventTime_t        ElapsedTimeTerminalMgr;
    CICS_EMP_EventTime_t        SystemUserTimeInTerminalMgr;


    cics_char_t                 FirstAbendCode[CICS_EMP_ABEND_SIZE];
    cics_char_t                 LatestAbendCode[CICS_EMP_ABEND_SIZE];


    CICS_EMP_Time_t             StartTime;
    CICS_EMP_Time_t             StopTime;
    CICS_EMP_OS_Count_t         PageCount;
    CICS_EMP_OS_Count_t         PageIOCount;
    CICS_EMP_OS_Count_t         SwapCount;
    CICS_EMP_OS_Count_t         ContextSwitch;
    CICS_EMP_OS_Count_t         IContextSwitch;
    CICS_EMP_OS_Count_t         SignalsCount;
    CICS_EMP_OS_Count_t         IOActions;
    CICS_EMP_OS_Clock_t         TaskElapsedTime;
    CICS_EMP_OS_Clock_t         TaskUserTime;
    CICS_EMP_OS_Clock_t         TaskSystemTime;
    CICS_EMP_EventTime_t        ElapsedTimeInULM;
    cics_char_t                 RecordType;
    cics_char_t                 TransactionID[CICS_EMP_TRANSACTION_ID_SIZE];
    cics_char_t                 OperatorID[CICS_EMP_OPERATOR_ID_SIZE];
    cics_char_t                 UserID[CICS_EMP_USER_ID_SIZE];
    cics_char_t                 TerminalID[CICS_EMP_TERMINAL_ID_SIZE];

    /* The following member is currently not supported. */

    CICS_EMP_EventTime_t        SystemUserTimeInFileMgr;



    CICS_EMP_EventTime_t        ElapsedTimeLocalSNALink;
};

/*
 * DESCRIPTION:
 * This structure contains the Performance Monitoring information for the
 * task. It is recommended that members of this structure are accessed using
 * the supplied CICS_EMP_Lookup function and not by direct reference to the
 * structure element names in the code. This is to enable monitoring programs
 * to run without needing to be changed or recompiled with future releases of
 * CICS.
 *
 */

/*
 * OBJECT: CICS_EMP_Map - A monitoring data dictionary field.
 */

struct CICS_EMP_Map
{
    int                 Mask;
    int                 Offset;
};

/* DESCRIPTION:
 * Forms the heart of the Dictionary structure InfEV_Mapping.
 */

/*
 * OBJECT: CICS_EMP_Functions - Buffer for passing EMP function addresses.
 */

struct CICS_EMP_Functions
{
    CICS_EMP_Return_t
        (*CICS_EMP_EnterAPI)(struct CICS_EMP_ENTER_API_Data *,
                             void **,
                             int *);

    CICS_EMP_Return_t
        (*CICS_EMP_WriteRecord)(void **, int *,
                                struct CICS_EMP_PerformanceClass *,
                                int [],
                                void **, int *, const struct CICS_EMP_Map *);

    CICS_EMP_Return_t
        (*CICS_EMP_ReStart)(enum CICS_EMP_Finish, void **, int *);
};

/*
 * DESCRIPTION:
 * This structure contains function addresses returned by the EMP
 * initialisation function.
 * :ul.
 * :li.
 * :hp2.CICS_EMP_EnterAPI:ehp2. - Process the EXEC CICS ENTER API function.
 * :li.
 * :hp2.CICS_EMP_WriteRecord:ehp2. - Return the user data to write to the
 * performance file.
 * :li.
 * :hp2.CICS_EMP_ReStart:ehp2. - Allows the user to restart any clocks
 * following the write of a monitoring record.
 * :eul.
 */

/* Function prototypes */

CICS_EMP_Return_t
CICS_EMP_Init        (    struct CICS_EMP_RegionData      *RegionData,
                       struct CICS_EMP_Functions       *Functions,
                       void                            **SaveSpace,
                         int                             *ErrorCode,
                          struct CICS_EMP_PerformanceClass *SystemData,
                          struct cics_eib                  *EIB,
                          const struct CICS_EMP_Map        Dictionary[]);

CICS_EMP_Return_t
CICS_EMP_EnterAPI    (    struct CICS_EMP_ENTER_API_Data   *EnterData,
                       void                             **SaveSpace,
                         int                              *ErrorCode);

CICS_EMP_Return_t
CICS_EMP_WriteRecord (   void                             **UserData,
                         int                              *UserDataLength,
                          struct CICS_EMP_PerformanceClass *SystemData,
                          int                              UnSupportedFlds[],
                       void                             **SaveSpace,
                         int                              *ErrorCode,
                          const struct CICS_EMP_Map        Dictionary[]);

CICS_EMP_Return_t
CICS_EMP_ReStart     (    enum CICS_EMP_Finish             Action,
                       void                             **SaveSpace,
                         int                              *ErrorCode);


CICS_EMP_ClockReturn_t
CICS_EMP_StopClock   ( CICS_EMP_EventTime_t             *Timer);


CICS_EMP_ClockReturn_t
CICS_EMP_StartClock  ( CICS_EMP_EventTime_t             *Timer,
                          enum CICS_EMP_ClockType          ClockType);

void
CICS_EMP_GetTime     ( CICS_EMP_Time_t                    *Now);

void
CICS_EMP_SubtractTime(  CICS_EMP_Time_t                    *T1,
                        CICS_EMP_Time_t                    *T2,
                       CICS_EMP_Time_t                    *Difference);
/*
 * The following struct is used to store a particular TraceId, function, and
 * Clock or Counter number for an EntryName
 */

struct TID
{
    int TraceId;
    char Function[9];
    int CNumber;
};

/*
 * The following struct is used to store the EntryName along with up to five TraceIds.
 * It also stores the clocks and counters for each entry point
 */

struct EMP
{
    char		EntryName[9];
    struct TID	ids[5];
    CICS_EMP_EventTime_t        Clock1;         /* Clock 1              */
    CICS_EMP_EventTime_t        Clock2;         /* Clock 2              */
    CICS_EMP_EventTime_t        Clock3;         /* Clock 3              */
    CICS_EMP_EventTime_t        Clock4;         /* Clock 4              */
    CICS_EMP_EventTime_t        Clock5;         /* Clock 5              */
    cics_slong_t                Counter1;       /* Counter 1            */
    cics_slong_t                Counter2;       /* Counter 2            */
    cics_slong_t                Counter3;       /* Counter 3            */
    cics_slong_t                Counter4;       /* Counter 4            */
    cics_slong_t                Counter5;       /* Counter 5            */
    int                         Active1;        /* Clock 1  active flag */
    int                         Active2;        /* Clock 2  active flag */
    int                         Active3;        /* Clock 3  active flag */
    int                         Active4;        /* Clock 4  active flag */
    int                         Active5;        /* Clock 5  active flag */
};

/*
 * Define the EMP information structure that is written to the record.
 * It contains the Clock Times and Counter Values along with which
 * clocks and counters were used
 */

struct EMPr
{
	char				EntryName[9];	/* Entry Name for EMP	*/
    cics_ulong_t        Clk_Counter1;   /* Clock 1 counter      */
    cics_ulong_t        Clk_Time1;      /* Clock 1 value        */
    cics_ulong_t        Clk_Counter2;   /* Clock 2 counter      */
    cics_ulong_t        Clk_Time2;      /* Clock 2 value        */
    cics_ulong_t        Clk_Counter3;   /* Clock 3 counter      */
    cics_ulong_t        Clk_Time3;      /* Clock 3 value        */
    cics_ulong_t        Clk_Counter4;   /* Clock 4 counter      */
    cics_ulong_t        Clk_Time4;      /* Clock 4 value        */
    cics_ulong_t        Clk_Counter5;   /* Clock 5 counter      */
    cics_ulong_t        Clk_Time5;      /* Clock 5 value        */
    cics_slong_t        Counter1r;      /* Counter 1            */
    cics_slong_t        Counter2r;      /* Counter 2            */
    cics_slong_t        Counter3r;      /* Counter 3            */
    cics_slong_t        Counter4r;      /* Counter 4            */
    cics_slong_t        Counter5r;      /* Counter 5            */
	int					Clock1Used;		/* Flag if Clock 1 used */
	int					Clock2Used;		/* Flag if Clock 2 used */
	int					Clock3Used;		/* Flag if Clock 3 used */
	int					Clock4Used;		/* Flag if Clock 4 used */
	int					Clock5Used;		/* Flag if Clock 5 used */
	int					Counter1Used;	/* Flag if Counter 1 used */
	int					Counter2Used;	/* Flag if Counter 2 used */
	int					Counter3Used;	/* Flag if Counter 3 used */
	int					Counter4Used;	/* Flag if Counter 4 used */
	int					Counter5Used;	/* Flag if Counter 5 used */
};

/*
 * Define the data structure for the ULM record that is written to the Monitoring TDQ.
 */

struct CICS_EMP_Record
{
	int					NumberOfEmps;	/* The number of EMP's  */
	struct EMPr			RecordEMPs[20];	/* EMP's from file		*/
};


#endif

