#ifndef _IPERSIST_
  #define _IPERSIST_

/*******************************************************************************
* FILE NAME: ipersist.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the class(es):                                              *
*    IPersistentObject - Persistent Object Abstract Base Class                 *
*    IPOManager        - Persistent Object Result Set Abstract Base Class      *
*                                                                              *
*******************************************************************************/

/***********************************************

    Licensed Materials - Property of IBM

    5622-880   5801-AAR

    (c) Copyright IBM Corp 1991, 1996.

************************************************/

#ifndef _IDAENV_
#include <idaenv.hpp>
#endif

#ifndef _ISTRING_
  #include <istring.hpp>
#endif

#ifndef _IDSEXC_HPP
  #include <idsexc.hpp>
#endif

#ifndef _ISTDNTFY_
  #include <istdntfy.hpp>
#endif

class IDatastoreBase;

/*----------------------------------------------------------------------------*/
/* Align classes on four byte boundary.                                       */
/*----------------------------------------------------------------------------*/
#pragma pack(4)

/******************************************************************************/
/* Class: IPersistentObject (Abstract)                                        */
/* Description: This abstract class defines the standard interface for a data */
/*              access class. This class provides data acess to a row of      */
/*              a table.                                                      */
/******************************************************************************/
class DAX_IMPORTEXPORT IPersistentObject : public IVBase
{

private:
/*----------------------------------------------------------------------------*/
/* data members                                                               */
/*----------------------------------------------------------------------------*/
  Boolean currentlyRetrievable;      // true if the object is retrievable
  const Boolean defaultReadOnly;     // true if the object is read-only by default.
  Boolean currentlyReadOnly;         // true if the set to read-only.

protected:

/*----------------------------------------------------------------------------*/
/* Constructors                                                               */
/*----------------------------------------------------------------------------*/
  IPersistentObject( const Boolean defReadonly = True );
  IPersistentObject(const IPersistentObject& partCopy);

/*----------------------------------------------------------------------------*/
/* Operator functions                                                         */
/*----------------------------------------------------------------------------*/
  IPersistentObject& operator= (const IPersistentObject& aIPersistentObject);
  Boolean operator== (const IPersistentObject& aIPersistentObject) const;

public:

/*----------------------------------------------------------------------------*/
/* Destructor                                                                 */
/*----------------------------------------------------------------------------*/
  virtual ~IPersistentObject();

/*----------------------------------------------------------------------------*/
/* Data Access functions: add, update, del, and retrieve.                     */
/* Add:      This function adds a row to a table.                             */
/* update:   This function updates a row.                                     */
/* del:      This function deletes a row.                                     */
/* retrieve: This function retrieves a row.                                   */
/*----------------------------------------------------------------------------*/
  virtual IPersistentObject& add();
  virtual IPersistentObject& update();
  virtual IPersistentObject& del();
  virtual IPersistentObject& retrieve();

/*----------------------------------------------------------------------------*/
/* Functions for object access support:                                       */
/* isRetrievable():     returns true if the object is retrievable.            */
/* isReadOnly():        returns true if the object is set to read only.       */
/* isDefaultReadOnly(): returns true if the object by default is read only.   */
/* setReadOnly(Boolean flag=true): if flag is true, the object is read only.  */
/*                       If flag is false, the object is read write.          */
/* setRetrievable(Boolean flag): set the retrievable attribute.               */
/*----------------------------------------------------------------------------*/
  Boolean isRetrievable() const;
  Boolean isReadOnly() const;
  Boolean isDefaultReadOnly() const;
  IPersistentObject& setReadOnly(Boolean flag = true);
  IPersistentObject& setRetrievable(Boolean flag = true);

/*----------------------------------------------------------------------------*/
/* Functions for object display:                                              */
/* asString     returns the object representation as an IString.              */
/* forDisplay   returns a tailored representation of the object as an IString.*/
/* The separator is used between the asString representation of each          */
/* attribute.                                                                 */
/*----------------------------------------------------------------------------*/

  virtual IString forDisplay( const char* separator = " " ) const = 0;

};

/*----------------------------------------------------------------------------*/
/* Class: IPOManager (Abstract)                                               */
/* Description: This abstract class defines the standard interface for a data */
/*              access class. This class provides data acess to a collection  */
/*              of rows from a table.                                         */
/*----------------------------------------------------------------------------*/
class DAX_IMPORTEXPORT IPOManager
{
protected:

/*----------------------------------------------------------------------------*/
/* Constructor                                                                */
/*----------------------------------------------------------------------------*/
  IPOManager();
  IPOManager(const IPOManager& partCopy);

/*----------------------------------------------------------------------------*/
/* Operator functions                                                         */
/*----------------------------------------------------------------------------*/
  IPOManager& operator= (const IPOManager& aIPOManager);

public:

/*----------------------------------------------------------------------------*/
/* Destructor                                                                 */
/*----------------------------------------------------------------------------*/
  virtual ~IPOManager();

/*----------------------------------------------------------------------------*/
/* Data Access functions: refresh and select.                                 */
/* refresh:  This function retrieves all the rows from a table.               */
/* select:   This function retrieves selected rows from a table based on      */
/*           the conditions described in the clause string.                   */
/*----------------------------------------------------------------------------*/
  virtual IPOManager& refresh () = 0;
  virtual IPOManager& select (const char* clause) = 0;

};

/*----------------------------------------------------------------------------*/
/* Resume compiler default packing.                                           */
/*----------------------------------------------------------------------------*/
#pragma pack()

#endif

