#ifndef _IPOATTR_
#define _IPOATTR_

/*******************************************************************************
* FILE NAME: IPOATTR.HPP                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the class(es):                                              *
*    Attribute definition for ODBC/CLI generation                              *
*                                                                              *
*******************************************************************************/

/***********************************************

    Licensed Materials - Property of IBM

    5622-880   5801-AAR

    (c) Copyright IBM Corp 1991, 1996.

************************************************/

#include <ibase.hpp>
#include <istring.hpp>
#include <wcstr.h>

//***************************************************************************
// base class for the null flag
class IPOAttributeBase
{
  private:
    Boolean theNull;
  public:
    IPOAttributeBase( Boolean aNull ) : theNull(aNull) {}
    IPOAttributeBase( const IPOAttributeBase& anAttribute ) :
         theNull(anAttribute.theNull) {}
    Boolean isNull() const { return theNull; }
    void setNull( const Boolean aNull = true ) { theNull = aNull; }
};


//***************************************************************************
// template class for numeric SQL types
// works for built in numeric types
template <class TYPE>
class IPONumericAttribute : public IPOAttributeBase
{
  protected:
    TYPE theValue;
  public:
    IPONumericAttribute() : IPOAttributeBase(true), theValue((TYPE)0) {}
    IPONumericAttribute( const IPONumericAttribute<TYPE>& anAttribute ) :
        IPOAttributeBase(anAttribute), theValue(anAttribute.theValue) {}
    const TYPE get() const { return theValue; }
    void set( const TYPE aValue ) { theValue = aValue; }
    Boolean operator==(const IPONumericAttribute<TYPE>& anAttribute) const
    {
      if ( isNull() != anAttribute.isNull() )
         return false;
      else if ( isNull() )
         return true;
      else
         return get() == anAttribute.get();
    }
    Boolean operator!=(const IPONumericAttribute<TYPE>& anAttribute) const
        { return ! ( *this == anAttribute ); }
    void set( const IString& aString );
    const IString asString() const { return isNull() ? IString("") : IString(theValue); }
    void getFrom( TYPE* pBuffer ) { theValue = *pBuffer; }
    const TYPE* pData() const { return &theValue; }
    long length() const { return sizeof(theValue); }
};

// instantiate the template for IString conversions - the only difference
// for the numeric types
inline void IPONumericAttribute<char>::set( const IString& aString )
  { theValue = *((const char*)aString); }
inline void IPONumericAttribute<short>::set( const IString& aString )
  { theValue = aString.asInt(); }
inline void IPONumericAttribute<unsigned short>::set( const IString& aString )
  { theValue = aString.asUnsigned(); }
inline void IPONumericAttribute<int>::set( const IString& aString )
  { theValue = aString.asInt(); }
inline void IPONumericAttribute<unsigned int>::set( const IString& aString )
  { theValue = aString.asUnsigned(); }
inline void IPONumericAttribute<long>::set( const IString& aString )
  { theValue = aString.asInt(); }
inline void IPONumericAttribute<unsigned long>::set( const IString& aString )
  { theValue = aString.asUnsigned(); }
inline void IPONumericAttribute<double>::set( const IString& aString )
  { theValue = aString.asDouble(); }

//***************************************************************************
// base class for strings
class IPOStringAttribute : public IPOAttributeBase
{
  protected:
    IString theString;
  public:
    IPOStringAttribute() : IPOAttributeBase(true), theString() {}
    IPOStringAttribute( const IPOStringAttribute& anAttribute ) :
        theString(anAttribute.theString), IPOAttributeBase(anAttribute) {}
    Boolean operator==(const IPOStringAttribute& anAttribute) const
    {
      if ( isNull() != anAttribute.isNull() )
         return false;
      else if ( isNull() )
         return true;
      else
         return theString == anAttribute.theString;
    }
    Boolean operator!=(const IPOStringAttribute& anAttribute) const
        { return ! ( *this == anAttribute ); }
    const IString asString() const { return isNull() ? IString() : theString; }
    void set( const IString& aValue ) { theString = aValue; }
    const IString& get() const { return theString; }
    unsigned putInto( void* pBuffer ) const
    {
      memcpy( pBuffer, (char*) theString, theString.length() );
      return theString.length();
    }
    unsigned putInto( char* pBuffer ) const
    {
      memcpy( pBuffer, (char*) theString, theString.length() );
      pBuffer[theString.length()] = '\0';
      return theString.length();
    }
    void getFrom( const char* pBuffer )
    {
      theString = pBuffer;
    }
    void getFrom( const char* pBuffer, unsigned length )
    {
      theString = IString(pBuffer,length);
    }
    unsigned putInto( wchar_t* pBuffer ) const
    {
      memcpy( pBuffer, (char*) theString, theString.length() );
      // set the last full character to the null terminator
      pBuffer[(theString.length()+1)/sizeof(wchar_t)] = L'\0';
      return theString.length();
    }
    void getFrom( const wchar_t* pBuffer )
    {
      theString = IString( pBuffer, wcslen(pBuffer) );
    }
    void getFrom( const wchar_t* pBuffer, unsigned length )
    {
      theString = IString( pBuffer, length*sizeof(wchar_t) );
    }
    const char* pData() const { return theString; }
    long length() const { return theString.length(); }
};

#endif
