/********************************************************************************/
/* File:  ipobuf.hpp                                                            */
/* Class: IPOBuffer                                                             */
/*                                                                              */
/* Abstract class IPOBuffer                                                     */
/********************************************************************************/

/***********************************************

    Licensed Materials - Property of IBM

    5622-880   5801-AAR

    (c) Copyright IBM Corp 1991, 1996.

************************************************/

#ifndef _IPOBUF_HPP_
#define _IPOBUF_HPP_

#include <istring.hpp>
#include "idssqlst.hpp"
#include "idstype.hpp"

class _Export IPOBufferBase
{
   private:
     Boolean fDataTruncated;
     long lPrecision;
     long lDataLength;
     short sScale;

   public:
     IPOBufferBase( Boolean aNull = true
                  , long aLength = 0
                  , long aPrecision = 0
                  , short aScale = 0 );
     virtual ~IPOBufferBase();

     virtual long theSqlCType() = 0;
     virtual long theSqlType() = 0;

     // for the buffer to bind itself to the statement
     void bindToColumn(short colNum, ISQLStatement* s);
     void bindToParameter(short parmNum, ISQLStatement* s);

/*---------------------- PO Buffer Functions -----------------------------------
| These methods deals with POBuffer functions:                                 |
|   bindToColumn - bind column in result set                                   |
|   bindToParameter - bind parameter marker                                    |
|   bufferSize      - get buffer size                                          |
|   buffer          - get buffer address                                       |
|   setNull         - set null                                                 |
|   setLength       - set data length                                          |
|   setPrecision    - set precision                                            |
|   setScale        - set scale                                                |
|    ---------------------- Used after fetch ------------------------------    |
|   length          - get amount of data read                                  |
|   precision       - get the precision                                        |
|   scale           - get the scale                                            |
|   dataTruncated   - returns true if data has been truncated                  |
|   dataNull        - returns true if data is null                             |
------------------------------------------------------------------------------*/

     // methods to get/set the buffer information
     virtual size_t bufferSize() const = 0;
     virtual void* buffer() const = 0;
     void setNull( Boolean aNull = true);
     void setLength( long aLength ) {lDataLength = aLength;}
     void setPrecision(long i) {lPrecision = i;}
     void setScale(short i) {sScale = i;}
     long precision() const {return lPrecision;}
     short scale() const {return sScale;}
     long length() const {return lDataLength;}
     virtual Boolean dataTruncated() const;
     Boolean dataNull() const;

};

// CPPTYPE is the c++ data type of the buffer
// SQLTYPE is the sql type in the DB
// SQLCTYPE is the sql c type in the DB

template < class SQLTYPE, class SQLCTYPE, class CPPTYPE >
class _Export IPONumericBuffer : public SQLTYPE, public SQLCTYPE, public IPOBufferBase
{
   private:
     CPPTYPE theData;
   public:
     IPONumericBuffer( Boolean aNull = true
                     , long aPrecision = 0
                     , short aScale = 0 )
         : IPOBufferBase(aNull,sizeof(CPPTYPE),aPrecision,aScale)
         {}
     ~IPONumericBuffer() {}
     size_t bufferSize() const { return sizeof(theData); }
     void* buffer() const { return (void*) &theData; }
     long theSqlCType() { return SQLCTYPE::sqlCType(); }
     long theSqlType() { return SQLTYPE::sqlType(); }
};

template < class SQLTYPE, class SQLCTYPE, class CPPTYPE >
class _Export IPOInBuffer : public SQLTYPE, public SQLCTYPE, public IPOBufferBase
{
  private:
     long lBufferSize;
     CPPTYPE* pBuffer;
   public:
     IPOInBuffer( long aNumberOfElements )
         : IPOBufferBase(true)
         , pBuffer(new CPPTYPE[aNumberOfElements])
         , lBufferSize(sizeof(CPPTYPE)*aNumberOfElements)
         {}
     ~IPOInBuffer() { delete pBuffer; }
     size_t bufferSize() const { return lBufferSize; }
     void* buffer() const { return (void*) pBuffer; }
     long theSqlCType() { return SQLCTYPE::sqlCType(); }
     long theSqlType() { return SQLTYPE::sqlType(); }
};

template < class SQLTYPE, class SQLCTYPE, class CPPTYPE >
class _Export IPOOutBuffer : public SQLTYPE, public SQLCTYPE, public IPOBufferBase
{
  private:
     long lBufferSize;
     const CPPTYPE* pBuffer;
   public:
     IPOOutBuffer( const CPPTYPE* aBuffer,
                   long aLength,
                   Boolean aNull = false,
                   long aPrecision = 0,
                   short aScale = 0 )
         : IPOBufferBase(aNull,aLength,aPrecision,aScale)
         , lBufferSize(aLength)
         , pBuffer(aBuffer)
         {}
     ~IPOOutBuffer(){}
     size_t bufferSize() const { return lBufferSize; }
     void* buffer() const { return (void*) pBuffer; }
     long theSqlCType() { return SQLCTYPE::sqlCType(); }
     long theSqlType() { return SQLTYPE::sqlType(); }
};

#endif
