#ifndef _IRecord_
#define _IRecord_
/*******************************************************************************
* FILE NAME: irecord.hpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the class(es):                                              *
*     IRecord - Base class for all record classes.                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM(R) VisualAge(TM) for C++                                               *
*   (C) Copyright International Business Machines Corporation 1991, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*   This program will not run in DOS mode.                                     *
*                                                                              *
* DISCLAIMER OF WARRANTIES:                                                    *
*   The following [enclosed] code is sample code created by IBM                *
*   Corporation.  This sample code is not part of any standard IBM product     *
*   and is provided to you solely for the purpose of assisting you in the      *
*   development of your applications.  The code is provided "AS IS",           *
*   without warranty of any kind.  IBM shall not be liable for any damages     *
*   arising out of your use of the sample code, even if they have been         *
*   advised of the possibility of such damages.                                *
*******************************************************************************/
#ifndef _IVBASE_
  #include <ivbase.hpp>
#endif

#ifndef _ISTRING_
  #include <istring.hpp>
#endif

/*-------------------------- Pragma Library Support --------------------------*/
#ifndef __NO_DEFAULT_LIBS__
  #ifdef __OS2__
    #ifdef __IMPORTLIB__
       #pragma library("CPPOV03I.LIB")
    #else
       #pragma library("CPPOV03.LIB")
    #endif
  #endif
  #ifdef __WINDOWS__
    #ifdef __IMPORTLIB__
       #pragma library("CPPWV03I.LIB")
    #else
       #pragma library("CPPWV03.LIB")
    #endif
  #endif
#endif

// Align classes on four byte boundary.
#pragma pack(4)

class IRecord : public IVBase {
/*******************************************************************************
*
* This class is the base class for all record classes.  A record class is
* generally used to wrapper a data structure required for a non-OO legacy
* system.  The record may be share between the OO and the legacy systems
* via communications, files, or any other medium of data exchange or sharing.
*
* This class also contains an IString that provides the storage for the
* data in the record.  The data muat always be maintained in this IString
* in the structure expected by the legacy system.  This way, with the
* exception of a data conversion step, it is always ready for exchange
* or sharing.
*
* The subclasses of IRecord should provide accessor functions for the
* fields they wish to expose to the OO code.  The IRecord class provides
* protected helper functions to aid in the writing of these functions.
* When using the member functions, remember that the first byte in the
* record is offset 1 (just like IString).
*
*******************************************************************************/

public:
/*------------------------- Constructors/Destructor ----------------------------
| You can contruct an instance of this class as follows:                       |
|   - Constructs an empty record of size 0.                                    |
|   - Constructs a record containing the 'recordData'.  The record size is     |
|     determined from the size of the IString.                                 |
|   - Copies the record data from the argument record.                         |
|-----------------------------------------------------------------------------*/
IRecord ( );

IRecord ( const IString & recordData );

IRecord ( const IRecord & aRecord );

virtual ~IRecord ( );

/*--------------------------------- Testing ------------------------------------
| The following functions test for record properties:                          |
|   isOrdered - Returns false.                                                 |
------------------------------------------------------------------------------*/
virtual IBoolean isOrdered ( ) const;

/*------------------------------- Conversion -----------------------------------
| The following functions permit the conversion of a record to various other   |
| data types.                                                                  |
|   asString - Returns the the IRecord contents as an IString.  A conversion   |
|              operator to IString is not provided to avoid ambiguity with the |
|              IString comparison operators (see IOrderedRecord).  Whenever an |
|              IOrderedRecord is involved in a comparison with an IString, we  |
|              want to make sure the IOrderedRecord operator is used.          |
------------------------------------------------------------------------------*/
IString asString ( ) const;

/*--------------------------- Assignment Operator ------------------------------
| The following operators allow the record's contents to be assigned from      |
| another record or an IString.                                                |
|   operator = - Replaces the contents of the record.  The target IRecord      |
|                size will be the source IRecord or IString size.              |
|-----------------------------------------------------------------------------*/
IRecord & operator = ( const IRecord & aRecord );
IRecord & operator = ( const IString & aRecord );

/*-------------------------------- Accessors -----------------------------------
| size   - returns the size of the record in bytes.                            |
|-----------------------------------------------------------------------------*/
unsigned long size ( ) const;

protected:
/*--------------------------- Protected Accessors ------------------------------
| setSize - Subclass default constructors, and constructors and assignment     |
|           operators that take an IString, must use this function in their    |
|           function body.  The default constructors must then set the fields  |
|           to their default values.  This function will pad the record with   |
|           '\x00' or truncate the record to the specified size.  A subclass   |
|           must only add to the size of its superclass.  It must never try to |
|           remove fields.  Using this technique will improve performance as   |
|           the fields are subsequently filled and ensure that size() always   |
|           returns the correct value.  The set size is returned.              |
|-----------------------------------------------------------------------------*/
unsigned long setSize ( unsigned long size );

/*------------------------ Accessor Helper Functions ---------------------------
| field    - There are four flavors of this function.                          |
|            1. Takes a pointer to a block of memory of at least the specified |
|               field size and copies size bytes from recordOffset.  If the    |
|               end of the record is encountered before size bytes are copied, |
|               the result is padded to size with padByte.  The same pointer   |
|               as the first parameter is returned from the function.          |
|            2. Takes an IString and copies size bytes from recordOffset.      |
|               If the end of the record is encountered before size bytes are  |
|               copied, the result is padded to size with padByte.  The same   |
|               IString as the first parameter is returned from the function.  |
|            3. Takes a pointer to memory (size + 1 bytes) for a null          |
|               terminated character string and copies size bytes from         |
|               recordOffset.  If the end of the record is encountered before  |
|               size bytes are copied, the result is padded to size with       |
|               padByte.  A terminating null is ALWAYS placed in the extra     |
|               byte provided in the buffer pointed to by the first parameter. |
|               The same pointer as the first parameter is returned from the   |
|               function.                                                      |
|            4. Takes a fundamental data type and copies the implied size      |
|               bytes from recordOffset.  If the end of the record is          |
|               encountered before size bytes are copied, the result is padded |
|               to size with padByte.  The result of the same type as the      |
|               first parameter is returned from the function.  The value of   |
|               the first parameter is not used.  It is needed to              |
|               differentiate the overloaded functions.  You can receive the   |
|               results into the same variable you pass as the first paramter, |
|               for example:                                                   |
|                   x = field( x, 20 );                                        |
|                                                                              |
| setField - There are four flavors of this function.  In all cases, if the    |
|            specified record offset is greater than the current size of the   |
|            record, the record is padded with '\x00' up to the offset.  The   |
|            padding of the field (if any) is as explained for each flavor.    |
|            1. Takes a pointer to a block of memory of at least the specified |
|               field size and copies size bytes to recordOffset.  The same    |
|               pointer as the first parameter is returned from the function.  |
|            2. Takes an IString and copies size bytes to recordOffset.        |
|               If (IString::size() < size), the field is padded to size with  |
|               padByte.  The same IString as the first parameter is returned  |
|               from the function.                                             |
|            3. Takes a pointer to a null terminated character string and      |
|               copies size bytes to recordOffset.  If (strlen(pData) < size), |
|               the field is padded to size with padByte.  The terminating     |
|               null byte is NOT copied.  The same pointer as the first        |
|               parameter is returned from the function.                       |
|            4. Takes a fundamental data type and copies the implied size      |
|               bytes to recordOffset.  The result of the same type as the     |
|               first parameter is returned from the function.  It will not    |
|               be changed.  You can receive the results into the same         |
|               variable you pass as the first paramter, for example:          |
|                   x = setField( x, 20 );                                     |
|-----------------------------------------------------------------------------*/
void * field ( void * pBuffer,
               unsigned long recordOffset,
               unsigned long size,
               unsigned char padByte = '\x00' ) const;

IString & field ( IString & buffer,
                  unsigned long recordOffset,
                  unsigned long size,
                  unsigned char padByte = '\x00' ) const;

char * field ( char * pBuffer,
               unsigned long recordOffset,
               unsigned long size,
               unsigned char padByte = '\x00' ) const;

char field ( char buffer,
             unsigned long recordOffset,
             unsigned char padByte = '\x00' ) const;
unsigned short field ( unsigned short buffer,
                       unsigned long recordOffset,
                       unsigned char padByte = '\x00' ) const;
unsigned long field ( unsigned long buffer,
                      unsigned long recordOffset,
                      unsigned char padByte = '\x00' ) const;
float field ( float buffer,
              unsigned long recordOffset,
              unsigned char padByte = '\x00' ) const;
double field ( double buffer,
               unsigned long recordOffset,
               unsigned char padByte = '\x00' ) const;
long double field ( long double buffer,
                    unsigned long recordOffset,
                    unsigned char padByte = '\x00' ) const;


const void * setField ( const void * pData,
                        unsigned long recordOffset,
                        unsigned long size );

const IString & setField ( const IString & data,
                           unsigned long recordOffset,
                           unsigned long size,
                           unsigned char padByte = '\x00' );

const char * setField ( const char * pData,
                        unsigned long recordOffset,
                        unsigned long size,
                        unsigned char padByte = '\x00' );

const char setField ( const char data,
                      unsigned long recordOffset );
const unsigned short setField ( const unsigned short data,
                                unsigned long recordOffset );
const unsigned long setField ( const unsigned long data,
                               unsigned long recordOffset );
const float setField ( const float data,
                       unsigned long recordOffset );
const double setField ( const double data,
                        unsigned long recordOffset );
const long double setField ( const long double data,
                             unsigned long recordOffset );

private:
friend class IOrderedRecord;

/*--------------------------- Private State Data -----------------------------*/
IString sRecordData;

}; // IRecord

// Resume compiler default packing.
#pragma pack()

#endif // _IRecord_
