/*******************************************************************************
* FILE NAME: ivbcnr.c                                                          *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in ivbcnr.h.                                                               *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#include  <ivbcnr.h>
#include  <icolvwd.hpp>
#include  <iseq.h>
#include  <ivseq.h>
#include  <ipartccl.h>
#include  <istring.hpp>
#include  <iexcept.hpp>
#include  <icconst.h>
#include  <inotifev.hpp>
#include  <imoushdr.hpp>
#include  <ipoint.hpp>


template <class ViewClass>
  class IVBContainerMouseHandler : public IMouseHandler {

public:
  IVBContainerMouseHandler(ViewClass* aVBContainer)
  {
    pVBContainer = aVBContainer;
    handleEventsFor(pVBContainer);
  }

virtual
  ~IVBContainerMouseHandler()
  {
    stopHandlingEventsFor(pVBContainer);
  }

protected:
  virtual Boolean mouseClicked(IMouseClickEvent& mouseEvent)
  {
    if ( (mouseEvent.mouseButton() == IMouseClickEvent::button2) &&
         (mouseEvent.mouseAction() == IMouseClickEvent::down) )
    {

      //Save the point away in the IVBContainerControl.
      pVBContainer->setButton2Point(
         IWindow::mapPoint(
            IWindow::pointerPosition(),   /* mouse pos in desktop coords */
            IWindow::desktopWindow()->handle(),           /* from window */
            pVBContainer->handle())                       /* to window   */
         );
    }
    return false;
  }

private:
  ViewClass * pVBContainer;

}; //IVBContainerMouseHandler

template <class Element, class ViewClass, class Collection>
  class IVBContainerControlData : public IBase {

public:

  IVBContainerControlData ( );

 ~IVBContainerControlData ( );

 ICollectionViewData< Element,
                     ViewClass,
                     Collection >
   *fCollectionViewData;
 IVSequence<Element> * pSelectedElements;
 IVSequence<Element> * pButton2Elements;
 IVSequence<IContainerObject *> * pSelectedCnrObjects;
 IVSequence<IContainerObject *> * pButton2CnrObjects;
 IVSequence<unsigned long> * pSelectedCollectionPositions;
 IVSequence<unsigned long> * pButton2CollectionPositions;
 IVBContainerMouseHandler< ViewClass > *pMouseHandler;
 IPoint fButton2Point;

private:
/*------------------------- Hidden Members -----------------------------------*/
  IVBContainerControlData
             ( const IVBContainerControlData< Element, ViewClass, Collection >& source );
IVBContainerControlData<Element,ViewClass, Collection>
 &operator=  ( const IVBContainerControlData< Element, ViewClass, Collection >& source );


};

template <class Element, class ViewClass, class Collection>
  IVBContainerControlData<Element, ViewClass, Collection>::
    IVBContainerControlData( )
{
  fCollectionViewData = new
        ICollectionViewData< Element,
                             ViewClass,
                             Collection > ( ) ;
}

template <class Element, class ViewClass, class Collection>
  IVBContainerControlData<Element,ViewClass,Collection>::
   ~IVBContainerControlData ( )
{
  delete pMouseHandler;
  delete pButton2CnrObjects;
  delete pSelectedCnrObjects;
  delete pButton2Elements;
  delete pSelectedElements;
  delete fCollectionViewData;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>::
    IVBContainerControl ( unsigned long                       identifier,
                          IWindow*                            parent,
                          IWindow*                            owner ,
                          const IRectangle&                   initial,
                          const IContainerControl::Style&     style,
                          const IContainerControl::Attribute& attribute )
    : IContainerControl( identifier, parent, owner, initial, style, attribute )
{
  initialize( );
}


template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>::
    IVBContainerControl ( unsigned long   identifier,
                          IWindow*        parent   )
    : IContainerControl( identifier, parent )
{
  initialize( );
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>::
    IVBContainerControl( const IWindowHandle&  handle  )
    : IContainerControl( handle )
{
  initialize( );
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>::
   ~IVBContainerControl ( )
{
  disableNotification();

  delete fVBContainerControlData;
}

template <class Element, class Collection, class CnrElement>
  INotificationId const IVBContainerControl<Element,Collection,CnrElement>::button2PointId =
    "IVBContainerControl::button2PointId";
template <class Element, class Collection, class CnrElement>
  INotificationId const IVBContainerControl<Element,Collection,CnrElement>::itemChangedId =
    "IVBContainerControl::itemChangedId";
template <class Element, class Collection, class CnrElement>
  INotificationId const IVBContainerControl<Element,Collection,CnrElement>::itemsId =
    "IVBContainerControl::itemsId";

template <class Element, class Collection, class CnrElement>
  Collection*  IVBContainerControl<Element,Collection,CnrElement>:: items( ) const
{
  return  (Collection *)fVBContainerControlData->fCollectionViewData->fCollectionObserver.collection();
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&   IVBContainerControl<Element,Collection,CnrElement>::
    setItems( Collection* collection )
{
  fVBContainerControlData->fCollectionViewData->fCollectionObserver.setCollection( collection );

  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement> :: elementsChanged ( )
{
  ICursor*       pCursor;
  Collection*    collection;
  collection   = items( );

 /*  Clear container and repopulate w/ current representations */
 if ( isWindowValid( this ) )  
 {
   Boolean refreshOn = isRefreshOn();
   setRefreshOff    ( );
   deleteAllObjects ( );

   if ( collection )
   {
     pCursor         = collection->newCursor( );
     forCursor( *pCursor )
     {
       addObject( new CnrElement( *( collection->elementAt( *pCursor ) ) ) );
     }
     delete pCursor;
   }

   setRefreshOn ( );
   refresh      ( );
   setRefreshOn ( refreshOn );
 }
 return *this;

}


template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      elementAdded ( unsigned long position, const Element&  element )
{
  IContainerObject* cnrObject;
  Collection*       collection;
  Boolean           refreshOn = isRefreshOn( );
  setRefreshOff    ( );

  collection      = items( );

  if (collection)
  {
    if ( position == IVBContainerControl<Element,Collection,CnrElement>::firstItem )
      cnrObject = 0;
    else
      cnrObject = objectAt(position - 2);

    addObjectAfter( new CnrElement( *(collection->elementAtPosition( position )) ),
                    cnrObject );

    if (refreshOn)
    {
      setRefreshOn( );
      refresh     ( );
    }
  }
  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      elementDeleted ( unsigned long position )
{

  Boolean refreshOn = isRefreshOn( );
  setRefreshOff( );
  delete  objectAt( position - 1 );
  if (refreshOn)
  {
    setRefreshOn( );
    refresh     ( );
  }
  return *this;
}


template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      elementChanged ( unsigned long  position, const Element&  element )
{
  IVBContainerObject* cnrObject;
  Boolean           selected;
  Boolean           cursored;
  Boolean           refreshOn = isRefreshOn( );

  setRefreshOff   ();
  cnrObject = ((IVBContainerObject * )objectAt  ( position - 1 ));

  cnrObject->refreshFromObject( );
  cnrObject->refresh          ( );

  if ( refreshOn )
  {
    setRefreshOn( );
    refresh     ( );
  }

  INotificationEvent  anEvent(
     IVBContainerControl<Element,Collection,CnrElement>::itemChangedId,
    *this,
     true ,
     IEventData((void *)&element) );
  notifyObservers( anEvent );

  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement> :: collectionReplaced( )
{
  INotificationEvent  anEvent(
     IVBContainerControl<Element,Collection,CnrElement>::itemsId,
    *this,
     true ,
     IEventData( (void *)items( ) ) );
  notifyObservers( anEvent );

  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      removeSelectedElements ( const Boolean deleteElements )
{
  IVSequence<unsigned long>         removePositions(*selectedCollectionPositions( ));
  Collection *                      pSequence = items( );

  IVSequence<unsigned long>::Cursor cursor(removePositions);
  Boolean                           refreshOn = isRefreshOn( ),
                                    itemRemoved = false;

  setRefreshOff   ( );

    fVBContainerControlData->pButton2Elements->removeAll();
    fVBContainerControlData->pSelectedElements->removeAll();
  for ( cursor.setToLast( );
        cursor.isValid ( );
        cursor.setToPrevious ( ) )
  {
    IPosition removePos = removePositions.elementAt ( cursor );
    Element removedElement = pSequence->elementAtPosition ( removePos );
    pSequence->removeAtPosition( removePos );
    itemRemoved = true;
    if (deleteElements)
    {
      delete removedElement;   //assuming pointers
    }
  }

  fVBContainerControlData->pSelectedCollectionPositions->removeAll();
  fVBContainerControlData->pButton2CollectionPositions->removeAll();

  if (itemRemoved)
  {
    INotificationEvent  anEvent(
       IVBContainerControl<Element,Collection,CnrElement>::selectId,
      *this );
    notifyObservers( anEvent );

    INotificationEvent  anEvent2(
       IVBContainerControl<Element,Collection,CnrElement>::button2PointId,
      *this );
    notifyObservers( anEvent2 );
  }

  if ( refreshOn )
  {
    setRefreshOn( );
    refresh     ( );
  }

  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      removeButton2Elements ( const Boolean deleteElements )
{
  IVSequence<unsigned long> *       removePositions = button2CollectionPositions( );
  Collection *                      pSequence = items( );

  IVSequence<unsigned long>::Cursor cursor(*removePositions);
  Boolean                           refreshOn = isRefreshOn( ),
                                    itemRemoved = false;

  setRefreshOff   ( );

  fVBContainerControlData->pButton2Elements->removeAll();
  fVBContainerControlData->pSelectedElements->removeAll();
  for ( cursor.setToLast( );
        cursor.isValid ( );
        cursor.setToPrevious ( ) )
  {
    IPosition removePos = removePositions->elementAt ( cursor );
    Element removedElement = pSequence->elementAtPosition ( removePos );
    pSequence->removeAtPosition( removePos );
    itemRemoved = true;
    if (deleteElements)
    {
      delete removedElement;   //assuming pointers
    }
  }

  fVBContainerControlData->pSelectedCollectionPositions->removeAll();
  fVBContainerControlData->pButton2CollectionPositions->removeAll();

  if (itemRemoved)
  {
    INotificationEvent  anEvent(
       IVBContainerControl<Element,Collection,CnrElement>::selectId,
      *this );
    notifyObservers( anEvent );

    INotificationEvent  anEvent2(
       IVBContainerControl<Element,Collection,CnrElement>::button2PointId,
      *this );
    notifyObservers( anEvent2 );
  }

  if ( refreshOn )
  {
    setRefreshOn( );
    refresh     ( );
  }

  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      removeAllElements ( const Boolean deleteElements )
{
  Collection *                      pSequence = items( );
  Boolean                           refreshOn = isRefreshOn( );

  if (pSequence) 
  {
    setRefreshOff   ( );

    fVBContainerControlData->pButton2Elements->removeAll();
    fVBContainerControlData->pSelectedElements->removeAll();

    while ( !(pSequence->isEmpty()) )
    {
      Element removedElement = pSequence->firstElement();
      pSequence->removeFirst();
      if (deleteElements)
      {
        //Notification must be disabled so that element observers don't trap on the deleteId
        removedElement->disableNotification();
        delete removedElement;   //assuming pointers
      }
    }

    fVBContainerControlData->pSelectedCollectionPositions->removeAll();
    fVBContainerControlData->pButton2CollectionPositions->removeAll();

    INotificationEvent  anEvent(
       IVBContainerControl<Element,Collection,CnrElement>::selectId,
      *this );
    notifyObservers( anEvent );

    INotificationEvent  anEvent2(
       IVBContainerControl<Element,Collection,CnrElement>::button2PointId,
      *this );
    notifyObservers( anEvent2 );

    if ( refreshOn )
    {
      setRefreshOn( );
      refresh     ( );
    }
  }
  return *this;
}

template <class Element, class Collection, class CnrElement>
  unsigned long IVBContainerControl<Element,Collection,CnrElement>::
    selectedCollectionPosition ( )
{
  unsigned long                    collectionPosition;
  IPartOrderedCollection<Element>* aCollection = items( );
  IVBContainerControl<Element,Collection,CnrElement>::ObjectCursor cursor( *this );
  Boolean                          proceed;
  IContainerObject*                cnrObject;

  if (aCollection) 
  {
    for ( cursor.setToFirst( ), collectionPosition = 0, proceed = true;
          cursor.isValid   ( ) && proceed;
          cursor.setToNext ( ), collectionPosition++ )
    {
      cnrObject = objectAt( cursor );
      if ( isSelected( cnrObject ) )
        proceed = false;
    }

    if ( ( aCollection->numberOfElements() < collectionPosition ) || proceed )
      collectionPosition =
        IVBContainerControl<Element,Collection,CnrElement>::noSelection;
  }
  else
  {
     collectionPosition =
        IVBContainerControl<Element,Collection,CnrElement>::noSelection;
  }
  return collectionPosition;

}

template <class Element, class Collection, class CnrElement>
  unsigned long IVBContainerControl<Element,Collection,CnrElement>::
    cursoredCollectionPosition ( )
{
  unsigned long      collectionPosition;
  IVBContainerControl<Element,Collection,CnrElement>::ObjectCursor cursor( *this );
  Boolean            proceed;
  IContainerObject*  cnrObject;

  for ( cursor.setToFirst( ), collectionPosition = 0, proceed = true;
        cursor.isValid   ( ) && proceed;
        cursor.setToNext ( ), collectionPosition++ )
  {
    cnrObject = objectAt( cursor );
    if ( isCursored( cnrObject ) )
      proceed = false;
  }

  if ( proceed )
    collectionPosition =
      IVBContainerControl<Element,Collection,CnrElement>::noSelection;

  return collectionPosition;

}

template <class Element, class Collection, class CnrElement>
  unsigned long IVBContainerControl<Element,Collection,CnrElement>::
    button2CollectionPosition ( )
{
  unsigned long      collectionPosition;
  IVBContainerControl<Element,Collection,CnrElement>::ObjectCursor cursor( *this );
  Boolean            proceed;
  IContainerObject*  cnrObject;
  IContainerObject*  button2CnrObject = objectUnderPoint( button2Point() );

  for ( cursor.setToFirst( ), collectionPosition = 0, proceed = true;
        cursor.isValid   ( ) && proceed;
        cursor.setToNext ( ), collectionPosition++ )
  {
    cnrObject = objectAt( cursor );
    if ( cnrObject == button2CnrObject )
      proceed = false;
  }

  if ( proceed )
    collectionPosition =
      IVBContainerControl<Element,Collection,CnrElement>::noSelection;

  return collectionPosition;
}

template <class Element, class Collection, class CnrElement>
 IVSequence<Element> *
   IVBContainerControl<Element,Collection,CnrElement>::
     selectedElements ( )
{
  return selectedElements(fVBContainerControlData->pSelectedElements);
}

template <class Element, class Collection, class CnrElement>
 IVSequence<Element> *
   IVBContainerControl<Element,Collection,CnrElement>::
     selectedElements ( IVSequence<Element> & aSequence)
{
  IPartOrderedCollection<Element>* pSequence = items                ( );
  IContainerControl::ObjectCursor  cursor( *this );

  unsigned long index = 0;
  aSequence.removeAll( );

  forCursor( cursor )
  {
    index++;
    IContainerObject *cnrObject = objectAt( cursor );
    if ( isSelected( cnrObject ) )
    {
      aSequence.add( pSequence->elementAtPosition( index ) );
    }
  }
  return &aSequence;
}

template <class Element, class Collection, class CnrElement>
 IVSequence<Element> *
   IVBContainerControl<Element,Collection,CnrElement>::
     selectedElements ( IVSequence<Element> * aSequence)
{
  return selectedElements( *aSequence );
}

template <class Element, class Collection, class CnrElement>
 IVSequence<Element> *
   IVBContainerControl<Element,Collection,CnrElement>::
     button2Elements ( )
{
  IVSequence<Element> * pBut2Col = fVBContainerControlData->pButton2Elements;

  if ( !(button2Point() == IPoint()) )
  {
    IContainerObject* button2CnrObject = objectUnderPoint(button2Point());
     if (button2CnrObject)
     {
       if ( isSelected(button2CnrObject) )
       {
         return selectedElements( pBut2Col );
       }
       else
       {
         pBut2Col->removeAll( );
         pBut2Col->add( button2Element() );
       }
     }
     else
       //clear the collection if no object is under the pointer
       pBut2Col->removeAll( );
  }
  return pBut2Col;
}

template <class Element, class Collection, class CnrElement>
  IVSequence<unsigned long> *
    IVBContainerControl<Element,Collection,CnrElement>::
      selectedCollectionPositions ( )
{
  return selectedCollectionPositions(fVBContainerControlData->pSelectedCollectionPositions);
}

template <class Element, class Collection, class CnrElement>
  IVSequence<unsigned long> *
    IVBContainerControl<Element,Collection,CnrElement>::
      selectedCollectionPositions ( IVSequence<unsigned long> * aSequence)
{
  IContainerControl::ObjectCursor  cursor ( *this );
  unsigned long index = 0;

  aSequence->removeAll( );

  forCursor( cursor )
  {
    index++;
    IContainerObject *cnrObject = objectAt( cursor );
    if ( isSelected( cnrObject ) )
      aSequence->add( index );
  }
  return aSequence;
}

template <class Element, class Collection, class CnrElement>
  IVSequence<unsigned long> *
    IVBContainerControl<Element,Collection,CnrElement>::
      button2CollectionPositions  ( )
{
  IVSequence<unsigned long> * pColPositions = fVBContainerControlData->pButton2CollectionPositions;

  if ( !(button2Point() == IPoint()) )
  {
    IContainerObject* button2CnrObject = objectUnderPoint(button2Point());
     if (button2CnrObject)
     {
       if ( isSelected(button2CnrObject) )
       {
         return selectedCollectionPositions( pColPositions );
       }
       else
       {
         pColPositions->removeAll( );
         pColPositions->add( button2CollectionPosition() );
       }
     }
     else
       //clear the collection if no object is under the pointer
       pColPositions->removeAll( );
  }
  return pColPositions;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      select ( unsigned long collectionPosition ,
               Boolean       select )
{
  IContainerObject* cnrObject;

  if ( ( collectionPosition > 0 ) &&
       ( collectionPosition <= IContainerControl::objectCount( ) ) )
  {
    cnrObject = objectAt( collectionPosition - 1 );
    setSelected( cnrObject, select );
  }

  return *this;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement>&
    IVBContainerControl<Element,Collection,CnrElement>::
      deselect ( unsigned long collectionPosition )
{
  return select( collectionPosition, false );
}

template <class Element, class Collection, class CnrElement>
  Element       IVBContainerControl<Element,Collection,CnrElement>::
    selectedElement ( )
{
  unsigned long                    selectedPosition;
  IPartOrderedCollection<Element>* aCollection = items( );
  IAutoPointer<ICursor>            pCursor( aCollection ? aCollection->newCursor( ) :
                                                          (ICursor*) 0,
                                            IINIT );
  
  if (!aCollection)
    ITHROWLIBRARYERROR( IC_NO_SELECTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );

  selectedPosition = selectedCollectionPosition( );
  if ( selectedPosition ==
         IVBContainerControl<Element,Collection,CnrElement>::noSelection )
  {
    ITHROWLIBRARYERROR( IC_NO_SELECTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  aCollection->setToPosition   ( selectedPosition, *pCursor );
  
  return aCollection->elementAt( *pCursor );
}

template <class Element, class Collection, class CnrElement>
  Element       IVBContainerControl<Element,Collection,CnrElement>::
    button2Element ( )
{
  unsigned long                    button2Position;
  IPartOrderedCollection<Element>* aCollection = items( );
  IAutoPointer<ICursor>            pCursor( aCollection ? aCollection->newCursor( ) :
                                                          (ICursor*) 0,
                                            IINIT );
 
 if (!aCollection)
   ITHROWLIBRARYERROR( IC_NO_SELECTION,
                     IBaseErrorInfo::invalidRequest,
                     IException::recoverable );


  button2Position = button2CollectionPosition( );
  if ( button2Position ==
         IVBContainerControl<Element,Collection,CnrElement>::noSelection )
  {
    ITHROWLIBRARYERROR( IC_NO_SELECTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  aCollection->setToPosition   ( button2Position, *pCursor );
  return aCollection->elementAt( *pCursor );
}

template <class Element, class Collection, class CnrElement>
  Element       IVBContainerControl<Element,Collection,CnrElement>::
    cursoredElement ( )
{
  unsigned long                    cursoredPosition;
  IPartOrderedCollection<Element>* aCollection = items( );
  IAutoPointer<ICursor>            pCursor( aCollection ? aCollection->newCursor( ) :
                                                          (ICursor*) 0,
                                            IINIT );
 
 if (!aCollection)
   ITHROWLIBRARYERROR( IC_NO_SELECTION,
                     IBaseErrorInfo::invalidRequest,
                     IException::recoverable );


  cursoredPosition = cursoredCollectionPosition( );
  if ( cursoredPosition ==
         IVBContainerControl<Element,Collection,CnrElement>::noSelection )
  {
    ITHROWLIBRARYERROR( IC_NO_SELECTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  aCollection->setToPosition   ( cursoredPosition, *pCursor );
  return aCollection->elementAt( *pCursor );
}

template <class Element, class Collection, class CnrElement>
  CnrElement*  IVBContainerControl<Element,Collection,CnrElement>::
    selectedCnrObject ( )
{
  IContainerControl::ObjectCursor  cursor( *this,
                                           IContainerObject::selected  );

  cursor.setToFirst   ( );
  if ( !cursor.isValid( ) )
  {
    ITHROWLIBRARYERROR( IC_NO_SELECTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  return (CnrElement *)objectAt( cursor );

}

template <class Element, class Collection, class CnrElement>
  IVSequence<IContainerObject *> *
    IVBContainerControl<Element,Collection,CnrElement>::
      selectedCnrObjects  ( IVSequence<IContainerObject *>& cnrObjects )
{
  IPartOrderedCollection<Element>* pSequence = items                ( );

  IContainerControl::ObjectCursor  cursor( *this,
                                           IContainerObject::selected  );

  cnrObjects.removeAll( );

  forCursor( cursor )
    cnrObjects.add( objectAt( cursor ) );

  return &cnrObjects;
}

template <class Element, class Collection, class CnrElement>
  IVSequence<IContainerObject *> *
    IVBContainerControl<Element,Collection,CnrElement>::
      selectedCnrObjects  ( IVSequence<IContainerObject *>* cnrObjects )
{
  return selectedCnrObjects( *cnrObjects );
}

template <class Element, class Collection, class CnrElement>
  IVSequence<IContainerObject *> *
    IVBContainerControl<Element,Collection,CnrElement>::
      selectedCnrObjects  ( )
{
  return selectedCnrObjects( fVBContainerControlData->pSelectedCnrObjects );
}

template <class Element, class Collection, class CnrElement>
  IVSequence<IContainerObject *> *
    IVBContainerControl<Element,Collection,CnrElement>::
      button2CnrObjects  ( )
{
  IVSequence<IContainerObject *> * pBut2Col = fVBContainerControlData->pButton2CnrObjects;

  if ( !(button2Point() == IPoint()) )
  {
    CnrElement * button2CnrObject = (CnrElement *) objectUnderPoint(button2Point());
    if (button2CnrObject)
     {
       if ( isSelected(button2CnrObject) )
       {
         return selectedCnrObjects( pBut2Col );
       }
       else
       {
         pBut2Col->removeAll( );
         pBut2Col->add( button2CnrObject );
       }
     }
     else
       //clear the collection if no object is under the pointer
       pBut2Col->removeAll( );
  }
  return pBut2Col;
}

template <class Element, class Collection, class CnrElement>
  unsigned long IVBContainerControl<Element,Collection,CnrElement>::
    numberOfSelections ( ) const
{
  unsigned long                    count = 0;
  IContainerControl::ObjectCursor  cursor( *this,
                                           IContainerObject::selected  );
  forCursor ( cursor )
    count++;

  return count;
}

template <class Element, class Collection, class CnrElement>
  unsigned long IVBContainerControl<Element,Collection,CnrElement>::
    numberOfButton2Objects ( ) const
{
  if ( !(button2Point() == IPoint()) )
  {
    IContainerObject* button2CnrObject = objectUnderPoint(button2Point());
     if (button2CnrObject)
     {
       if ( isSelected(button2CnrObject) )
         return numberOfSelections( );
       else
         return 1;
     }
  }
  return 0;
}

template <class Element, class Collection, class CnrElement>
  IVBContainerControl<Element,Collection,CnrElement> & IVBContainerControl<Element,Collection,CnrElement>::
    setButton2Point(const IPoint & aPoint)
{
  fVBContainerControlData->fButton2Point = aPoint;

  INotificationEvent  anEvent(
     IVBContainerControl<Element,Collection,CnrElement>::button2PointId,
     *this,
     true ,
     IEventData((void *) &(fVBContainerControlData->fButton2Point)) );
  notifyObservers( anEvent );
  return *this;
}

template <class Element, class Collection, class CnrElement>
  IPoint IVBContainerControl<Element,Collection,CnrElement>::
    button2Point() const
{
  return fVBContainerControlData->fButton2Point;
}

template <class Element, class Collection, class CnrElement>
  void IVBContainerControl<Element,Collection,CnrElement>::
    initialize ( )
{
  fVBContainerControlData = new
      IVBContainerControlData< Element, IVBContainerControl<Element,Collection,CnrElement>, Collection> ( );
  fVBContainerControlData->fCollectionViewData->fCollectionObserver.setViewer( this );
  fVBContainerControlData->pSelectedElements   = new IVSequence<Element>;
  fVBContainerControlData->pButton2Elements    = new IVSequence<Element>;
  fVBContainerControlData->pSelectedCnrObjects = new IVSequence<IContainerObject *>;
  fVBContainerControlData->pButton2CnrObjects  = new IVSequence<IContainerObject *>;
  fVBContainerControlData->pSelectedCollectionPositions = new IVSequence<unsigned long>;
  fVBContainerControlData->pButton2CollectionPositions  = new IVSequence<unsigned long>;
  fVBContainerControlData->pMouseHandler = new
    IVBContainerMouseHandler< IVBContainerControl<Element,Collection,CnrElement> >(this);
  enableNotification     ( );
  fVBContainerControlData->pSelectedElements->enableNotification();
  fVBContainerControlData->pButton2Elements->enableNotification();
  fVBContainerControlData->pSelectedCnrObjects->enableNotification();
  fVBContainerControlData->pButton2CnrObjects->enableNotification();
  fVBContainerControlData->pSelectedCollectionPositions->enableNotification();
  fVBContainerControlData->pButton2CollectionPositions->enableNotification();
  setDeleteObjectsOnClose( );
}

template <class Element, class Collection, class CnrElement>
  const unsigned long
    IVBContainerControl<Element,Collection,CnrElement>::firstItem   = 1;
template <class Element, class Collection, class CnrElement>
  const unsigned long
    IVBContainerControl<Element,Collection,CnrElement>::noSelection = 0;


