/*******************************************************************************
* FILE NAME: idocstor.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the classes:                                                *
*     IDocumentStorage                                                         *
*     IStructuredStorage                                                       *
*     IFlatFileStorage                                                         *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#ifndef _IDOCSTOR_
#define _IDOCSTOR_

#include <ivbase.hpp>
#include <istring.hpp>

// forward references
class IComponent;
class IModel;
class IEmbedderModel;
class CAswStream;
class CAswStorage;
class CAswDocFileStorage;

// Classes defined in this module
class IDocumentStorage;
class IFlatFileStorage;
class IStructuredStorage;

#pragma pack(4)


/*******************************************************************************
*                                                                              *
*   The IDocumentStorage class is an abstract base class for managing          *
*   the file representation of an IComponent.  Two concrete subclasses,        *
*   IStructuredStorage and IFlatFileStorage derive from IDocumentStorage.      *
*                                                                              *
*******************************************************************************/

class IDocumentStorage : public IVBase {
public:

/*----------------------- Construction/Destruction ---------------------------*/

    virtual                 ~IDocumentStorage();


/*------------------------------- File I/O -----------------------------------*/

    virtual void            loadFile(const IString& name) = 0;
    virtual void            saveFile(const IString& name,
                                     Boolean sameAsLoad,
                                     Boolean remember) = 0;


/*--------------------------- Title Components -------------------------------*/

    virtual void            setFileName(const IString& name);


/*----------------------------- Informational --------------------------------*/

    const IString&          fileName() const;
    IComponent&             component() const;


#ifdef IC_LIBRARYUSEONLY
    IString                 fileToUse( const char* name ) const;

    CAswStorage*            storage() const;
    virtual void            setStorage( CAswStorage* );
    virtual void            loadStorage( CAswStorage*, CAswStream* );
    virtual void            saveStorage( CAswStorage* stor,
                                         CAswStream* strm,
                                         Boolean sameAsLoad,
                                         Boolean remember ) const;
#endif // IC_LIBRARYUSEONLY

protected:
                            IDocumentStorage( IComponent& );
                            IDocumentStorage( const IDocumentStorage& );

#ifdef IC_LIBRARYUSEONLY
    // Helper functions
    virtual void            resetFile();

#endif // IC_LIBRARYUSEONLY

private:
    IDocumentStorage&       operator=( const IDocumentStorage& );

    IString                 fFileName;
    IComponent*             fpComponent;
    CAswStorage*            fpStorage;

};


/*******************************************************************************
*                                                                              *
*   The IFlatFileStorage class stores the document model in a flat file.       *
*   It can only be used for stand-alone servers that do not support            *
*   embedding (i.e. non containers).                                           *
*                                                                              *
*******************************************************************************/

class IFlatFileStorage : public IDocumentStorage {
public:

/*----------------------- Construction/Destruction ---------------------------*/

                            IFlatFileStorage( IComponent& );
    virtual                 ~IFlatFileStorage();


/*------------------------------- File I/O -----------------------------------*/

    virtual void            loadFile(const IString& name);
    virtual void            saveFile(const IString& name,
                                     Boolean sameAsLoad,
                                     Boolean remember);

protected:
                            IFlatFileStorage( const IFlatFileStorage& );

private:
    IFlatFileStorage&       operator=( const IFlatFileStorage& );

};


/*******************************************************************************
*                                                                              *
*   The IStructuredStorage class stores the document model in an OLE           *
*   Doc File.                                                                  *
*                                                                              *
*******************************************************************************/

class IStructuredStorage : public IDocumentStorage {
public:

/*----------------------- Construction/Destruction ---------------------------*/

                            IStructuredStorage( IComponent&, IEmbedderModel* );
    virtual                 ~IStructuredStorage();


/*------------------------------- File I/O -----------------------------------*/

    virtual void            loadFile(const IString& name);
    virtual void            saveFile(const IString& name,
                                     Boolean sameAsLoad,
                                     Boolean remember);

#ifdef IC_LIBRARYUSEONLY
    virtual void            setStorage( CAswStorage* );

    virtual void            loadStorage( CAswStorage*, CAswStream* );
    virtual void            saveStorage( CAswStorage* stor,
                                         CAswStream* strm,
                                         Boolean sameAsLoad,
                                         Boolean remember ) const;
#endif

protected:
                            IStructuredStorage( const IStructuredStorage& );

#ifdef IC_LIBRARYUSEONLY
    // Helper functions
    virtual void            resetFile();
    virtual void            loadEmbeddedComponents( CAswStorage* );
    virtual void            saveEmbeddedComponents( CAswStorage*,
                                                    Boolean,
                                                    Boolean ) const;
#endif

private:
    IStructuredStorage&     operator=( const IStructuredStorage& );

    CAswDocFileStorage*     fpDocFile;
    CAswStorage*            fpRootStorage;

};

  #include <idocstor.inl>


#pragma pack()


#endif // _IDOCSTOR_
