

#ifndef _IEXCEPT_
#define _IEXCEPT_
/*******************************************************************************
* FILE NAME: iexcept.hpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the classes                                                 *
*     IBaseErrorInfo                                                           *
*     IGUIErrorInfo                                                            *
*     ISystemErrorInfo                                                         *
*     ICLibErrorInfo                                                           *
*     IXLibErrorInfo                                                           *
*                                                                              *
*   This file also contains several macros and functions which can be used     *
*   to facilitate throwing exceptions.  This includes the IASSERTPARM and      *
*   IASSERTSTATE macros, and the IExcept__assertParameter and                  *
*   IExcept__assertState functions.                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#include <ivbase.hpp>
#include <iexcbase.hpp>

#pragma pack(4)



class IBaseErrorInfo : public IVBase {
typedef IVBase
  Inherited;
public:
/*------------------------------- Constructors ------------------------------ */
 IBaseErrorInfo  ( );

virtual
 ~IBaseErrorInfo ( );

/*---------------------------- Error Information -----------------------------*/
virtual
  operator const char* ( ) const = 0;

virtual const char
 *text ( ) const = 0;

virtual unsigned long
  errorId ( ) const = 0;

virtual Boolean
  isAvailable ( ) const = 0;

enum ExceptionType { accessError, deviceError, invalidParameter,
                     invalidRequest, outOfSystemResource, outOfWindowResource,
                     outOfMemory, resourceExhausted };

/*------------------------------ Throw Support -------------------------------*/
void
  throwError ( const IExceptionLocation&  location,
               ExceptionType              name = accessError,
               IException::Severity       severity = recoverable,
               IException::ErrorCodeGroup errorGroup =
                                             IException::baseLibrary );




}; // IBaseErrorInfo

#ifdef IUSE_IERRORINFO
  typedef IBaseErrorInfo IErrorInfo;
#endif

class IGUIErrorInfo : public IBaseErrorInfo {
typedef IBaseErrorInfo
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
  IGUIErrorInfo ( const char* GUIFunctionName = 0 );

virtual
 ~IGUIErrorInfo ( );

/*---------------------------- Error Information -----------------------------*/
virtual
  operator const char* ( ) const;

virtual const char
 *text ( ) const;

virtual unsigned long
  errorId ( ) const;

virtual Boolean
  isAvailable ( ) const;

/*------------------------------ Throw Support -------------------------------*/
static void
  throwGUIError  ( const char*               functionName,
                   const IExceptionLocation& location,
                   IBaseErrorInfo::ExceptionType name = accessError,
                   IException::Severity      severity = recoverable );

private:
/*----------------------------- Hidden Functions -----------------------------*/
  IGUIErrorInfo ( const IGUIErrorInfo& errorInformation );

IGUIErrorInfo
 &operator= ( const IGUIErrorInfo& errorInformation );

/*--------------------------------- Private ----------------------------------*/
  unsigned long
    ulClErrorId;


char
 *pClFuncNameText;




}; // IGUIErrorInfo

class ISystemErrorInfo : public IBaseErrorInfo {
typedef IBaseErrorInfo
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
  ISystemErrorInfo ( unsigned long systemErrorId,
                     const char* systemFunctionName = 0 );

virtual
 ~ISystemErrorInfo ( );

/*---------------------------- Error Information -----------------------------*/
virtual
  operator const char* ( ) const;

virtual const char
 *text ( ) const;

virtual unsigned long
  errorId ( ) const;

virtual Boolean
  isAvailable ( ) const;

/*------------------------------ Throw Support -------------------------------*/
static void
  throwSystemError ( unsigned long             systemErrorId,
                     const char*               functionName,
                     const IExceptionLocation& location,
                     IBaseErrorInfo::ExceptionType name = accessError,
                     IException::Severity      severity = recoverable );

private:
  ISystemErrorInfo ( const ISystemErrorInfo& errorInformation );

ISystemErrorInfo
 &operator= ( const ISystemErrorInfo& errorInformation );

/*--------------------------------- Private ----------------------------------*/
Boolean
  bClError;
unsigned long
  ulClErrorId;
char
 *pClErrorText;



}; // ISystemErrorInfo

class ICLibErrorInfo : public IBaseErrorInfo {
typedef IBaseErrorInfo
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
ICLibErrorInfo ( const char* CLibFunctionName = 0 );

virtual
 ~ICLibErrorInfo ( );

/*---------------------------- Error Information -----------------------------*/
virtual
operator const char* ( ) const;

virtual const char
 *text ( ) const;

virtual unsigned long
  errorId ( ) const;

virtual Boolean
  isAvailable ( ) const;

/*------------------------------ Throw Support -------------------------------*/
static void
  throwCLibError (   const char*               functionName,
                     const IExceptionLocation& location,
                     IBaseErrorInfo::ExceptionType name = accessError,
                     IException::Severity      severity = recoverable );

private:
/*----------------------------- Hidden Functions -----------------------------*/
  ICLibErrorInfo ( const ICLibErrorInfo& errorInformation );

ICLibErrorInfo
 &operator= ( const ICLibErrorInfo& errorInformation );

/*--------------------------------- Private ----------------------------------*/
Boolean
  bClError;
unsigned long
  ulClErrno;
char
 *pClErrorText;




}; // IClibErrorInfo




void
  IExcept__assertParameter    ( const char*               assertExpression,
                                const IExceptionLocation& location ),
  IExcept__assertState        ( const char*               assertExpression,
                                const IExceptionLocation& location ),
  IExcept__throwLibraryError  (
              unsigned long             messageId,
              const IExceptionLocation& location,
              IBaseErrorInfo::ExceptionType name =
                                            IBaseErrorInfo::invalidRequest,
              IException::Severity      severity = IException::recoverable,
              const char*               text = 0),
  IExcept__throwError (
              unsigned long              messageId,
              const IExceptionLocation&  location,
              const char*                messageFile,
              IException::ErrorCodeGroup errorGroup,
              IBaseErrorInfo::ExceptionType  name =
                                         IBaseErrorInfo::invalidRequest,
              IException::Severity       severity = IException::recoverable,
              const char*                substitutionText = 0);

#define IASSERTPARM(test)\
    if(!(test))\
    {\
       IExcept__assertParameter(#test, IEXCEPTION_LOCATION());\
    }

#define IASSERTSTATE(test)\
    if(!(test))\
    {\
       IExcept__assertState(#test, IEXCEPTION_LOCATION());\
    }

#define ITHROWGUIERROR(func)\
    IGUIErrorInfo::throwGUIError(func, IEXCEPTION_LOCATION())

#define ITHROWGUIERROR2(func, exType, sev)\
    IGUIErrorInfo::throwGUIError(func, IEXCEPTION_LOCATION(), exType, sev)

#define ITHROWSYSTEMERROR(id, func, exType, sev)\
    ISystemErrorInfo::throwSystemError(id, func, \
                                       IEXCEPTION_LOCATION(), exType, sev)

#define ITHROWCLIBERROR(func, exType, sev)\
    ICLibErrorInfo::throwCLibError(func, IEXCEPTION_LOCATION(), exType, sev)


#define ITHROWLIBRARYERROR(id, exType, sev)\
    IExcept__throwLibraryError(id, IEXCEPTION_LOCATION(), exType, sev)

#define ITHROWLIBRARYERROR1(id, exType, sev, text)\
    IExcept__throwLibraryError(id, IEXCEPTION_LOCATION(), exType, sev, text)

#define ITHROWERROR(messageId, name, severity, messageFile, errorGroup)\
    IExcept__throwError(messageId, IEXCEPTION_LOCATION(), messageFile, \
                        errorGroup, name, severity)

#define ITHROWERROR1(messageId, name, severity, messageFile,\
                     errorGroup, substitutionText)\
    IExcept__throwError(messageId, IEXCEPTION_LOCATION(), messageFile, \
                        errorGroup, name, severity, substitutionText)

#pragma pack()

#endif /* _IEXCEPT_ */


