#ifndef _INOTEBK_
#define _INOTEBK_
/*******************************************************************************
* FILE NAME: inotebk.hpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the classes:                                                *
*     IPageHandle                                                              *
*     INotebook                                                                *
*     INotebook::Cursor                                                        *
*     INotebook::PageSettings                                                  *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#include <icontrol.hpp>
#include <irect.hpp>
#include <istring.hpp>
#include <ihandle.hpp>
#include <ipoint.hpp>

class IResourceId;
class ITabBitmapMgr;
class INotebookData;
class INotebookCursorData;
class INotebookPageSettingsData;
class INotebookDrawItemEvent;
class IColor;

  class IPageClipWindow;
  class ITabCtrlPageSequence;

#pragma pack(4)

class IPageHandle : public IHandle {
typedef IHandle
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
  IPageHandle ( Value pageId = 0 );

/*-------------------------------- Page Handle -------------------------------*/
  operator Value ( ) const;

}; // class IPageHandle


class INotebook : public IControl {
typedef IControl
  Inherited;
public:
class Style;
/*------------------------------- Constructors -------------------------------*/
  INotebook   ( unsigned long        windowId,
                IWindow*             parent,
                IWindow*             owner,
                const IRectangle&    initial = IRectangle(),
                const Style&         style = defaultStyle() );

  INotebook   ( unsigned long        windowId,
                IWindow*             parentAndOwner );
  INotebook   ( const IWindowHandle& handle );

virtual
 ~INotebook   ( );

/*---------------------------------- Styles ----------------------------------*/
INESTEDBITFLAGCLASSDEF2(Style, INotebook, IWindow, IControl);
                                    // style class definition
static const Style
  IC_IMPORTU classDefaultStyle,
  IC_IMPORTU pmCompatible,
  IC_IMPORTU spiralBinding,
  IC_IMPORTU solidBinding,
  IC_IMPORTU backPagesBottomRight,
  IC_IMPORTU backPagesBottomLeft,
  IC_IMPORTU backPagesTopRight,
  IC_IMPORTU backPagesTopLeft,
  IC_IMPORTU majorTabsRight,
  IC_IMPORTU majorTabsLeft,
  IC_IMPORTU majorTabsTop,
  IC_IMPORTU majorTabsBottom,
  IC_IMPORTU squareTabs,
  IC_IMPORTU roundedTabs,
  IC_IMPORTU polygonTabs,
  IC_IMPORTU allTabsVisible,
  IC_IMPORTU handleDrawTabs,
  IC_IMPORTU statusTextLeft,
  IC_IMPORTU statusTextRight,
  IC_IMPORTU statusTextCenter,
  IC_IMPORTU tabTextLeft,
  IC_IMPORTU tabTextRight,
  IC_IMPORTU tabTextCenter;

static Style
  defaultStyle      ( );

static void
  setDefaultStyle   ( const Style&    style );

virtual unsigned long
  convertToGUIStyle ( const IBitFlag& style,
                      Boolean         extendedOnly = false ) const;

/*----------------------------- Status Text ------------------------------------*/
enum TextAlignment {
  left,
  right,
  center
};

virtual INotebook
 &setStatusTextAlignment ( TextAlignment       alignment ),
 &setStatusText          ( const IPageHandle&  referencePage,
                           const char*         statusText ),
 &setStatusText          ( const IPageHandle&  referencePage,
                           const IResourceId&  resourceId );

virtual TextAlignment
  statusTextAlignment    ( ) const;

/*------------------------------- Tab Support --------------------------------*/
enum TabShape {
  square,
  rounded,
  polygon
};

virtual INotebook
 &setTabShape         ( TabShape             tabShape ),
 &setMajorTabSize     ( const ISize&         sizeMajorTab ),
 &setMinorTabSize     ( const ISize&         sizeMinorTab );

virtual TabShape
  tabShape            ( ) const;

virtual INotebook
 &setTabTextAlignment ( TextAlignment        alignment ),
 &refreshTabs         ( ),
 &setTabText          ( const IPageHandle&   referencePage,
                        const char*          tabText ),
 &setTabText          ( const IPageHandle&   referencePage,
                        const IResourceId&   resourceId ),
 &setTabBitmap        ( const IPageHandle&   referencePage,
                        const IBitmapHandle& bitmap ),
 &setTabBitmap        ( const IPageHandle&   referencePage,
                        const IResourceId&   resourceId );

virtual TextAlignment
  tabTextAlignment    ( ) const;

Boolean
  areAllTabsVisible   ( ) const,
  isDrawTabsEnabled   ( ) const;

/*------------------------------ Page Button ---------------------------------*/
virtual INotebook
 &setPageButtonSize ( const ISize& sizePageButton );

/*----------------------------- Orientation ----------------------------------*/
enum Orientation {
  backpagesBottomTabsRight,
  backpagesTopTabsRight,
  backpagesBottomTabsLeft,
  backpagesTopTabsLeft,
  backpagesRightTabsTop,
  backpagesLeftTabsTop,
  backpagesRightTabsBottom,
  backpagesLeftTabsBottom
};

virtual INotebook
 &setOrientation   ( Orientation orientation );

virtual Orientation
  orientation      ( ) const;

/*----------------------------- Binding Support ------------------------------*/
enum Binding {
  spiral,
  solid
};

virtual INotebook
 &setBinding     ( Binding binding );

virtual Binding
  binding        ( ) const;

/*------------------------------ Colors --------------------------------------*/
#ifndef IC_WIN_FLAGNOP
virtual IColor
  backgroundColor              ( ) const;
#endif //!IC_WIN_FLAGNOP
#ifndef IC_MOTIFWIN_FLAGNOP
virtual IColor
  hiliteBackgroundColor        ( ) const,
#endif //!IC_MOTIFWIN_FLAGNOP
  pageBackgroundColor          ( ) const,
  majorTabBackgroundColor      ( ) const,
  minorTabBackgroundColor      ( ) const,
  majorTabForegroundColor      ( ) const,
  minorTabForegroundColor      ( ) const;

virtual INotebook
 &setPageBackgroundColor       ( const IColor& color ),
 &setMajorTabBackgroundColor   ( const IColor& color ),
 &setMinorTabBackgroundColor   ( const IColor& color ),
 &setMajorTabForegroundColor   ( const IColor& color ),
 &setMinorTabForegroundColor   ( const IColor& color );


virtual INotebook
 &resetPageBackgroundColor     ( ),
 &resetMajorTabBackgroundColor ( ),
 &resetMinorTabBackgroundColor ( ),
 &resetMajorTabForegroundColor ( ),
 &resetMinorTabForegroundColor ( );

/*--------------------------- Native Control Support -------------------------*/
Boolean
  isPMCompatible ( ) const;


class PageSettings : public IVBase {
typedef IVBase
  Inherited;
friend class INotebook;
public:
/*-------------------------------- Attributes --------------------------------*/
INESTEDBITFLAGCLASSDEF0(Attribute, INotebook::PageSettings);
static const Attribute
  IC_IMPORTU autoPageSize,
  IC_IMPORTU statusTextOn,
  IC_IMPORTU majorTab,
  IC_IMPORTU minorTab,
  IC_IMPORTU noAttribute;

/*------------------------------- Constructors -------------------------------*/
  PageSettings  ( );
  PageSettings  ( const Attribute&               attribute );
  PageSettings  ( const char*                    tabText,
                  const char*                    statusText = 0,
                  const Attribute&               pageAttribute = noAttribute );
  PageSettings  ( const IBitmapHandle&           tabBitmap,
                  const char*                    statusText = 0,
                  const Attribute&               pageAttribute = noAttribute );
  PageSettings  ( const INotebook::PageSettings& pageSettings );
  PageSettings
   &operator =  ( const INotebook::PageSettings& pageSettings );

virtual
  ~PageSettings ( );


/*----------------------------- Status Text ------------------------------------*/
virtual INotebook::PageSettings
 &setStatusText     ( const char*        statusText ),
 &setStatusText     ( const IResourceId& resourceId );

virtual IString
  statusText        ( ) const;

Boolean
  isStatusTextOn    ( ) const;

/*--------------------------------- Tabs -------------------------------------*/
virtual INotebook::PageSettings
 &setTabText        ( const char*          tabText ),
 &setTabText        ( const IResourceId&   resourceId ),
 &setTabBitmap      ( const IBitmapHandle& bitmap ),
 &setTabBitmap      ( const IResourceId&   resourceId );

virtual IString
  tabText           ( ) const;

virtual IBitmapHandle
  tabBitmap         ( ) const;

Boolean
  isMajorTab        ( ) const,
  isMinorTab        ( ) const;


/*---------------------------- Application Data ------------------------------*/
virtual INotebook::PageSettings
 &setUserData       ( unsigned long userData );

virtual unsigned long
  userData          ( ) const;

/*---------------------------- Automatic Sizing ------------------------------*/
Boolean
  isAutoSize       ( ) const;

private:
/*-------------------------------- Private -----------------------------------*/
Attribute
  pageStyle;
IBitmapHandle
  savedTabBitmap;
unsigned long
  savedUserData;
IString
  savedStatusText,
  savedTabText;

INotebookPageSettingsData
 *fPageSettingsData;
}; // INotebook::PageSettings


class Cursor : public IVBase {
typedef IVBase
  Inherited;
public :
/*------------------------------- Constructors -------------------------------*/
  Cursor  ( const INotebook& notebook );

virtual
  ~Cursor ( );

/*------------------------------ Page Iteration ------------------------------*/
virtual Boolean
  setToFirst    ( ),
  setToNext     ( ),
  setToPrevious ( ),
  setToLast     ( ),
  isValid       ( ) const;

virtual void
  invalidate    ( );

/*------------------------------- Page Retrieval -----------------------------*/
virtual IPageHandle
  first      ( ),
  next       ( ),
  previous   ( ),
  last       ( ),
  current    ( ) const;

void
  setCurrent ( const IPageHandle& current );


private:
/*----------------------------- Hidden Members -------------------------------*/
Cursor       ( const Cursor& cursor );
Cursor
 &operator=  ( const Cursor& cursor );

/*--------------------------------- Private ----------------------------------*/
const INotebook
 &nbCl;
unsigned long
  ulClCurrentPageId;
unsigned long
  ulClReferences;

INotebookCursorData
 *fCursorData;
}; // INotebook::Cursor


/*------------------------------ Page Addition -------------------------------*/
virtual IPageHandle
  addFirstPage      ( const PageSettings& pageInfo,
                      IWindow*            pageWindow = 0 ),
  addLastPage       ( const PageSettings& pageInfo,
                      IWindow*            pageWindow = 0 ),
  addPageBefore     ( const PageSettings& pageInfo,
                      const IPageHandle&  referencePage,
                      IWindow*            pageWindow = 0 ),
  addPageAfter      ( const PageSettings& pageInfo,
                      const IPageHandle&  referencePage,
                      IWindow*            pageWindow = 0 ),
  addPageBefore     ( const PageSettings& pageInfo,
                      const Cursor&       cursor,
                      IWindow*            pageWindow = 0 ),
  addPageAfter      ( const PageSettings& pageInfo,
                      const Cursor&       cursor,
                      IWindow*            pageWindow = 0 );

/*------------------------------- Page Removal -------------------------------*/
virtual INotebook
 &removePage        ( const IPageHandle&  page ),
 &removePage        ( const Cursor&       cursor ),
 &removeAllPages    ( ),
 &removeTabSection  ( const IPageHandle&  page ),
 &removeTabSection  ( const Cursor&       cursor );

/*---------------------------- Page Manipulation -----------------------------*/
virtual INotebook
 &turnToPage        ( const IPageHandle&  page ),
 &turnToPage        ( const Cursor&       cursor );

virtual IPageHandle
  topPage           ( ) const,
  firstPage         ( ) const,
  lastPage          ( ) const,
  nextPage          ( const IPageHandle&  referencePage ) const,
  previousPage      ( const IPageHandle&  referencePage ) const;

/*---------------------------- Page Information ------------------------------*/
PageSettings
  pageSettings      ( const Cursor&       cursor ) const,
  pageSettings      ( const IPageHandle&  page ) const;

/*---------------------------- Application Data ------------------------------*/
virtual INotebook
 &setUserData       ( const IPageHandle& referencePage,
                      unsigned long      userData );

/*------------------------- Page Window Association --------------------------*/
virtual IWindow
 *window            ( const Cursor&      cursor ) const,
 *window            ( const IPageHandle& page ) const;

virtual INotebook
 &setWindow         ( const Cursor&      cursor,
                            IWindow*     pageWindow = 0 ),
 &setWindow         ( const IPageHandle& referencePage,
                            IWindow*     pageWindow = 0 );


/*------------------------------- Page Support -------------------------------*/
virtual ISize
  pageSize        ( ) const,
  notebookSize    ( const IPageHandle& page ) const;

virtual unsigned long
  totalPages      ( ) const,
  pagesToMajorTab ( const IPageHandle& page ) const,
  pagesToMajorTab ( const Cursor&      cursor ) const,
  pagesToMinorTab ( const IPageHandle& page ) const,
  pagesToMinorTab ( const Cursor&      cursor ) const,
  pagesToEnd      ( const IPageHandle& page ) const,
  pagesToEnd      ( const Cursor&      cursor ) const;

virtual Boolean
  isEmpty         ( ) const;

/*-------------------------- Notification Members ----------------------------*/
static INotificationId const
  IC_IMPORTU pageBackgroundColorId,
  IC_IMPORTU majorTabBackgroundColorId,
  IC_IMPORTU minorTabBackgroundColorId,
  IC_IMPORTU majorTabForegroundColorId,
  IC_IMPORTU minorTabForegroundColorId,
  IC_IMPORTU orientationId;

virtual IWindow
 &enableNotification ( Boolean enable=true );

protected:
/*------------------------------ Layout Support ------------------------------*/
virtual ISize
  calcMinimumSize ( ) const;


private:
/*----------------------------- Hidden Members -------------------------------*/
INotebook   ( const INotebook& notebook );
INotebook
 &operator= ( const INotebook& notebook );

/*--------------------------------- Private ----------------------------------*/
INotebookData
 *fNotebookData;

static Style
  currentDefaultStyle;

//------------------------------------------------------------------------------  
// DEFECT 23265 (cloned from PORTAPAK)
// (but had to add member to notebook data, rather than notebook handler)
//------------------------------------------------------------------------------  
Boolean
  tabFromPage    ( void );
void
  tabFromPage    ( Boolean bFlag );
//------------------------------------------------------------------------------  

unsigned long
  queryPage      ( void* mp1, void* mp2 ) const,
  insertPage     ( void* mp1, void* mp2 );

Boolean
  deletePage     ( void* mp1, void* mp2 );

short
  queryPageCount ( void* mp1, void* mp2 ) const;

IPageHandle
  insertPageInfo2 ( const PageSettings& pageInfo,
                    const IPageHandle&  referencePage,
                    IWindow*            window,
                    unsigned long       fPosition ),
  insertPageInfo ( const PageSettings& pageInfo,
                   const IPageHandle&  referencePage,
                   IWindow*            window,
                   unsigned long       fPosition );


ITabBitmapMgr
 *bmClTabBitmapMgr;
unsigned long
  ulClValidate;

friend class INotebook::Cursor;
friend class INotebookHandler;

INESTEDBITFLAGCLASSDEF0(clrFlags, INotebook);
static const clrFlags
  IC_IMPORTU bgnPageColor,
  IC_IMPORTU bgnMajorColor,
  IC_IMPORTU bgnMinorColor,
  IC_IMPORTU fgnMajorColor,
  IC_IMPORTU fgnMinorColor;

INotebook::clrFlags
  colorFlags;

void setNotebookColors   ( unsigned long color,
                           unsigned long colorArea );
void storeNotebookColors ( unsigned long color,
                           unsigned long
                           colorArea );
IColor UnsignedLongAsRGB ( unsigned long color ) const;

struct NotebookColors
{
  unsigned long  pageBackgroundColor;
  unsigned long  majorTabBackgroundColor;
  unsigned long  minorTabBackgroundColor;
  unsigned long  majorTabForegroundColor;
  unsigned long  minorTabForegroundColor;
};
NotebookColors
 *pnotebookColors;


// Tab control related functions and members

friend INotebookDrawItemEvent;
friend IPageClipWindow;

static Boolean
  hasBeenRegistered;

IWindow
 *pageClippingWindow ( ) const;

Boolean
  processTabSelect   ( void*        pPageSelNotify,
                       Boolean      bSelectPending ) const;

void
  showTopPage        ( long         iPageIndex,
                       void*        pRect,
                       Boolean      bNotResizing = true ) const,
  adjustRect         ( Boolean      bTabCtrlSize,
                       void*        pRect ) const,
  tabControlResize   ( const ISize& newSize ) const;

ITabCtrlPageSequence
  *pTabCtrlPageSeqCl;
unsigned long
  ulClPagesInserted;

ITabCtrlPageSequence
 *tabPageHandleCollection ( ) const;

long
  tabPageIndex            ( const IPageHandle&  pageHandle ) const;

IPageHandle
  tabPageHandle           ( unsigned long       ulPosition ) const,
  insertTabPage           ( const PageSettings& pageInfo,
                            const IPageHandle&  referencePage,
                            unsigned long       ulPosition );

Boolean
  removeTabPage           ( const IPageHandle&  pageHandle );


}; // INotebook

INESTEDBITFLAGCLASSFUNCS(Style, INotebook);

#pragma pack()

  #include <inotebk.inl>

#endif /* _INOTEBK_ */
