#ifndef _IRECT_INL_
#define _IRECT_INL_ 0
/*******************************************************************************
* FILE NAME: irect.inl                                                         *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the definition of the inline functions for the          *
*   classes declared in irect.hpp.                                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#ifndef _IRECT_
  #undef  _IRECT_INL_
  #define _IRECT_INL_ 1
  #include <irect.hpp>
#endif

#if _IRECT_INL_
  #define inline
#endif

/*------------------------------- Constructors -------------------------------*/
inline IRectangle :: IRectangle ( )
  : origin( 0, 0 ), corner( 0, 0 )
  {
  }
inline IRectangle :: IRectangle ( const IPoint &point1,
                                  const IPoint &point2 )
  : origin( point1.minimum( point2 ) ), corner( point1.maximum( point2 ) )
  {
  }
inline IRectangle :: IRectangle ( const IPoint &aPoint,
                                  const ISize  &aSize )
  : origin( aPoint.minimum( aPoint+aSize ) ),
    corner( aPoint.maximum( aPoint+aSize ) )
  {
  }
inline IRectangle :: IRectangle ( Coord x1,
                                  Coord y1,
                                  Coord x2,
                                  Coord y2 )
   : origin( IPair( x1, y1 ).minimum( IPair( x2, y2 ) ) ),
     corner( IPair( x1, y1 ).maximum( IPair( x2, y2 ) ) )
  {
  }
#ifdef _WINDEF_
inline IRectangle :: IRectangle ( const RECT &aRECTL)
  : origin( IPair( aRECTL.left, aRECTL.bottom ).
      minimum( IPair( aRECTL.right, aRECTL.top ) ) ),
    corner( IPair( aRECTL.left, aRECTL.bottom ).
      maximum( IPair( aRECTL.right, aRECTL.top  ) ) )
  {
  }
inline IRectangle :: IRectangle ( const RECTL &aRECTL)
  : origin( IPair( aRECTL.left, aRECTL.bottom ).
      minimum( IPair( aRECTL.right, aRECTL.top ) ) ),
    corner( IPair( aRECTL.left, aRECTL.bottom ).
      maximum( IPair( aRECTL.right, aRECTL.top  ) ) )
  {
  }
#endif
#if defined(OS2DEF_INCLUDED)
inline IRectangle :: IRectangle ( const RECTL &aRECTL)
  : origin( IPair( aRECTL.xLeft, aRECTL.yBottom ).
      minimum( IPair( aRECTL.xRight, aRECTL.yTop ) ) ),
    corner( IPair( aRECTL.xLeft, aRECTL.yBottom ).
      maximum( IPair( aRECTL.xRight, aRECTL.yTop  ) ) )
  {
  }
#endif
inline IRectangle :: IRectangle ( Coord width,
                                  Coord height )
  : origin( IPair( width, height ).minimum( IPair() ) ),
    corner( IPair( width, height ).maximum( IPair() ) )
  {
  }
inline IRectangle :: IRectangle ( const IPair &aPair )
  : origin( aPair.minimum( IPair() ) ),
    corner( aPair.maximum( IPair() ) )
  {
  }
/*-------------------------------- Accessors ---------------------------------*/
inline IRectangle::Coord IRectangle :: minY ( ) const
  {
  return origin.y();
  }
inline IRectangle::Coord IRectangle :: bottom ( ) const
  {
  return minY();
  }
inline IRectangle::Coord IRectangle :: height ( ) const
  {
  return corner.y() - origin.y();
  }
inline IRectangle::Coord IRectangle :: minX ( ) const
  {
  return origin.x();
  }
inline IRectangle::Coord IRectangle :: left ( ) const
  {
  return minX();
  }
inline IRectangle::Coord IRectangle :: maxX ( ) const
  {
  return corner.x();
  }
inline IRectangle::Coord IRectangle :: right ( ) const
  {
  return maxX();
  }
inline IRectangle::Coord IRectangle :: maxY ( ) const
  {
  return corner.y();
  }
inline IRectangle::Coord IRectangle :: top ( ) const
  {
  return maxY();
  }
inline IRectangle::Coord IRectangle :: width ( ) const
  {
  return corner.x() - origin.x();
  }
inline IPoint IRectangle :: centerXMinY ( ) const
  {
  return IPoint( ( maxX() + minX() ) / 2,
                   minY() );
  }
inline IPoint IRectangle :: bottomCenter ( ) const
  {
  return centerXMinY( );
  }
inline IPoint IRectangle :: minXMinY ( ) const
  {
  return origin;
  }
inline IPoint IRectangle :: bottomLeft ( ) const
  {
  return minXMinY();
  }
inline IPoint IRectangle :: maxXMinY ( ) const
  {
  return IPoint( maxX(),
                 minY() );
  }
inline IPoint IRectangle :: bottomRight ( ) const
  {
  return maxXMinY( );
  }
inline IPoint IRectangle :: centerXCenterY ( ) const
  {
  return ( origin + corner ) / 2;
  }
inline IPoint IRectangle :: center ( ) const
  {
  return centerXCenterY( );
  }
inline IPoint IRectangle :: minXCenterY ( ) const
  {
  return IPoint( minX(),
                 ( minY() + maxY() ) / 2 );
  }
inline IPoint IRectangle :: leftCenter ( ) const
  {
  return minXCenterY( );
  }
inline IPoint IRectangle :: maxXCenterY ( ) const
  {
  return IPoint( maxX(),
                 ( minY() + maxY() ) / 2 );
  }
inline IPoint IRectangle :: rightCenter ( ) const
  {
  return maxXCenterY();
  }
inline IPoint IRectangle :: centerXMaxY ( ) const
  {
  return IPoint( ( minX() + maxX() ) / 2,
                 maxY() );
  }
inline IPoint IRectangle :: topCenter ( ) const
  {
  return centerXMaxY();
  }
inline IPoint IRectangle :: minXMaxY ( ) const
  {
  return IPoint( minX(),
                 maxY() );
  }
inline IPoint IRectangle :: topLeft ( ) const
  {
  return minXMaxY( );
  }
inline IPoint IRectangle :: maxXMaxY ( ) const
  {
  return corner;
  }
inline IPoint IRectangle :: topRight ( ) const
  {
  return maxXMaxY( );
  }
inline ISize IRectangle :: size ( ) const
  {
  return corner - origin;
  }
inline IRectangle::Coord IRectangle :: area ( ) const
  {
  return height() * width();
  }
/*--------------------------- Comparison Operators ---------------------------*/
inline IBase::Boolean IRectangle :: operator == ( const IRectangle &aRect ) const
  {
  return ( origin == aRect.origin
           &&
           corner == aRect.corner );
  }
inline IBase::Boolean IRectangle :: operator != ( const IRectangle& aRect ) const
  {
  return !( *this == aRect );
  }
/*------------------------------ Implementation ------------------------------*/
inline IRectangle &IRectangle :: validate ( )
  {
  if ( (corner.coord1() < origin.coord1()) ||
       (corner.coord2() < origin.coord2()) )
  {
    origin = corner = IPoint( 0, 0 );
  }
  return *this;
  }
/*-------------------------- Manipulation Operators --------------------------*/
inline IRectangle &IRectangle :: operator &= ( const IRectangle &aRect )
  {
  origin = origin.maximum( aRect.origin );
  corner = corner.minimum( aRect.corner );
  return validate();
  }
inline IRectangle IRectangle :: operator &  ( const IRectangle &aRect ) const
  {
  IRectangle result( *this );
  return ( result &= aRect );
  }
inline IRectangle &IRectangle :: operator |= ( const IRectangle &aRect )
  {
  origin = origin.minimum( aRect.origin ),
  corner = corner.maximum( aRect.corner );
  return *this;
  }
inline IRectangle IRectangle :: operator |  ( const IRectangle &aRect ) const
  {
  IRectangle result( *this );
  return result |= aRect;
  }
/*------------------------------- Manipulation -------------------------------*/
inline IRectangle &IRectangle :: moveBy ( const IPair &aPair )
  {
  origin += aPair;
  corner += aPair;
  return *this;
  }
inline IRectangle &IRectangle :: centerAt ( const IPoint &aPoint )
  {
  return moveBy( aPoint - centerXCenterY() );
  }
inline IRectangle IRectangle :: centeredAt ( const IPoint &aPoint ) const
  {
  IRectangle result( *this );
  return result.centerAt( aPoint );
  }
inline IRectangle &IRectangle :: expandBy ( const IPair &aPair )
  {
  origin -= aPair;
  corner += aPair;
  return validate();
  }
inline IRectangle &IRectangle :: expandBy ( Coord coord )
  {
  origin -= coord,
  corner += coord;
  return validate();
  }
inline IRectangle IRectangle :: expandedBy ( const IPair &aPair ) const
  {
  IRectangle result( *this );
  return result.expandBy( aPair );
  }
inline IRectangle IRectangle :: expandedBy ( Coord coord ) const
  {
  IRectangle result( *this );
  return result.expandBy( coord );
  }
inline IRectangle IRectangle :: movedBy ( const IPair &aPair ) const
  {
  IRectangle result( *this );
  return result.moveBy( aPair );
  }
inline IRectangle &IRectangle :: moveTo ( const IPoint &aPoint )
  {
  return moveBy( aPoint - origin );
  }
inline IRectangle IRectangle :: movedTo ( const IPoint &aPoint ) const
  {
  IRectangle result( *this );
  return result.moveTo( aPoint );
  }
inline IRectangle &IRectangle :: scaleBy ( const IPair &aPair )
  {
  origin *= aPair;
  corner *= aPair;
  return *this;
  }
inline IRectangle &IRectangle :: scaleBy ( Coord coord )
  {
  origin *= coord,
  corner *= coord;
  return *this;
  }
inline IRectangle &IRectangle :: scaleBy ( double amt )
  {
  origin *= amt;
  corner *= amt;
  return *this;
  }
inline IRectangle &IRectangle :: scaleBy ( double xamt, double yamt )
  {
  origin.scaleBy( xamt, yamt );
  corner.scaleBy( xamt, yamt );
  return *this;
  }
inline IRectangle IRectangle :: scaledBy ( const IPair &aPair ) const
  {
  IRectangle result( *this );
  return result.scaleBy( aPair );
  }
inline IRectangle IRectangle :: scaledBy ( Coord coord ) const
  {
  IRectangle result( *this );
  return result.scaleBy( coord );
  }
inline IRectangle IRectangle :: scaledBy ( double amt ) const
  {
  IRectangle result( *this );
  return result.scaleBy( amt );
  }
inline IRectangle IRectangle :: scaledBy ( double xamt, double yamt ) const
  {
  IRectangle result( *this );
  return result.scaleBy( xamt, yamt );
  }
inline IRectangle &IRectangle :: shrinkBy ( const IPair &aPair )
  {
  return expandBy( -aPair );
  }
inline IRectangle &IRectangle :: shrinkBy ( Coord coord )
  {
  return expandBy( -coord );
  }
inline IRectangle IRectangle :: shrunkBy ( const IPair &aPair ) const
  {
  return expandedBy( -aPair );
  }
inline IRectangle IRectangle :: shrunkBy ( Coord coord ) const
  {
  return expandedBy( -coord );
  }
inline IRectangle &IRectangle :: sizeTo ( const IPair &aPair )
  {
  corner = origin + aPair;
  return *this;
  }
inline IRectangle IRectangle :: sizedTo ( const IPair &aPair ) const
  {
  IRectangle result( *this );
  return result.sizeTo( aPair );
  }
inline IRectangle &IRectangle :: sizeBy ( Coord factor )
  {
  IPoint save( minXMinY() );
  moveTo( IPoint(0,0) ).scaleBy( factor ).moveTo( save );
  return *this;
  }
inline IRectangle &IRectangle :: sizeBy ( const IPair &aPair )
  {
  IPoint save( minXMinY() );
  moveTo( IPoint(0,0) ).scaleBy( aPair ).moveTo( save );
  return *this;
  }
inline IRectangle &IRectangle :: sizeBy ( double factor )
  {
  IPoint save( minXMinY() );
  moveTo( IPoint(0,0) ).scaleBy( factor ).moveTo( save );
  return *this;
  }
inline IRectangle &IRectangle :: sizeBy ( double xfact, double yfact )
  {
  IPoint save( minXMinY() );
  moveTo( IPoint(0,0) ).scaleBy( xfact, yfact ).moveTo( save );
  return *this;
  }
inline IRectangle IRectangle :: sizedBy ( Coord factor ) const
  {
  IRectangle result( *this );
  return result.sizeBy( factor );
  }
inline IRectangle IRectangle :: sizedBy ( const IPair &aPair ) const
  {
  IRectangle result( *this );
  return result.sizeBy( aPair );
  }
inline IRectangle IRectangle :: sizedBy ( double factor ) const
  {
  IRectangle result( *this );
  return result.sizeBy( factor );
  }
inline IRectangle IRectangle :: sizedBy ( double xfact, double yfact ) const
  {
  IRectangle result( *this );
  return result.sizeBy( xfact, yfact );
  }
/*--------------------------------- Testing ----------------------------------*/
inline IBase::Boolean IRectangle :: contains ( const IRectangle &aRect ) const
  {
  return ( origin <= aRect.origin
           &&
           aRect.corner <= corner );
  }
inline IBase::Boolean IRectangle :: contains ( const IPoint &aPoint ) const
  {
  return ( aPoint >= origin
           &&
           aPoint < corner );
  }
inline IBase::Boolean IRectangle :: intersects ( const IRectangle &aRect ) const
  {
  return ( origin < aRect.corner
           &&
           aRect.origin < corner );
  }
#endif /* _IRECT_INL_ */
