//******************************************************************************
//                                                                             *
//COPYRIGHT: Copyright (C) International Business Machines Corp., 1992,1994.   *
//                                                                             *
//DISCLAIMER OF WARRANTIES:                                                    *
//  The following [enclosed] code is sample code created by IBM                *
//  Corporation.  This sample code is not part of any standard IBM product     *
//  and is provided to you solely for the purpose of assisting you in the      *
//  development of your applications.  The code is provided "AS IS",           *
//  without warranty of any kind.  IBM shall not be liable for any damages     *
//  arising out of your use of the sample code, even if they have been         *
//  advised of the possibility of such damages.                                *
//                                                                             *
//******************************************************************************

#include "store.hpp"
#include <iomanip.h>
//******************************************************************************
// File Name: store.cpp                                                        *
//                                                                             *
// Note: This file illustrates how you can design your own object class        *
//       based on the table object class generated by the DAX Data Access      *
//       tool. This file keeps all the member functions of your own object     *
//       class. The database connection is established in the constructor      *
//       and is disconnected in the destructor.                                *
//                                                                             *
//******************************************************************************
//******************************************************************************
// Constructor: Establish the connection to the data base.                     *
//******************************************************************************
Store::Store() {
  try {

       ISESQL = getenv("ISESQL");

       if (ISESQL == "TRUE") {
          dsm3.connect("DAXSAMP");                                       // Embedded SQL Connection -  Only requires single connection
       }
       else {
          dsm.connect("DAXSAMP");                                        // CLI or ODBC Connection
          dsm1.connect("DAXSAMP");
          dsm2.connect("DAXSAMP");
       }

  } catch (IException &exc) {
    cout << "Failed: Connection" << endl;
    cout << "Error id: " << exc.errorId() << endl;
    cout << "Error Text: " << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
  } catch(...) {
    cout << "Unknown exception in connect" << endl;
  }
}

//******************************************************************************
// Constructor: Establish the connection to the data base.                     *
//******************************************************************************
Store::Store(IString usrid, IString pwd): userid(usrid), password(pwd) {
  try {

       ISESQL = getenv("ISESQL");

       if (ISESQL == "TRUE") {
          dsm3.connect("DAXSAMP", userid, password);                     // Embedded SQL Connection -  Only requires single connection
       }
       else {
          dsm.connect(userid, password);                                 // CLI or ODBC Connection
          dsm1.connect(userid, password);
          dsm2.connect(userid, password);
       }

  } catch (IException &exc) {
    cout << "Failed: Connection" << endl;
    cout << "Error id: " << exc.errorId() << endl;
    cout << "Error Text: " << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
  } catch(...) {
    cout << "Unknown exception in connect" << endl;
  }
}

//******************************************************************************
// Destructor: Disconnect the connection.                                      *
//******************************************************************************
Store::~Store() {
  try {
    // disconnect the connection
    if (ISESQL == "TRUE") {
       dsm3.disconnect();                           // Only single disconnection for Embedded SQL
    }
    else {
       dsm.disconnect();                            // CLI or ODBC case
       dsm1.disconnect();
       dsm2.disconnect();
    }
  } catch (IException &exc) {
    cout << "Failed: Disconnection" << endl;
    cout << "Error id: " << exc.errorId() << endl;
    cout << "Error Text: " << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
  } catch(...) {
    cout << "Unknown exception in disconnect" << endl;
  }
}

//******************************************************************************
// Function Name: generatePriceList
// Note : This member function generates a price list for the store.
//        The data printed in the list is a subset of the inventory
//        database.
//******************************************************************************
void Store::generatePriceList() {
  try {
    // call refresh to get a new copy of the price list.
    priceListMgr.refresh();

    // set up the cursor for going through each item.
    PrclistManager::sequenceType::Cursor cursor(*priceListMgr.items());

    // loop until the end of the list.
    cout << "Prod. ID   Description            Price " << endl;
    cout << "========   =====================  ===== " << endl;
    forCursor (cursor) {
      cout << cursor.element()->pnumber() << ' ';
      cout << cursor.element()->pdes() << "   $";

      cout << setiosflags(ios::fixed) << setprecision(2) << cursor.element()->lstprice() << endl;
    }
  } catch (IDSAccessError &exc) {
    cout << "Failed: generatePriceList " << endl;
    cout << "Error Text:" << exc.text() << endl;
    cout << "Error id:"  << exc.errorId() << endl;
    cout << "Error Class: " << exc.name() << endl;
    if (exc.errorId() == DAX_REF_SQLERR) {
      cout << "SQLCODE: " << exc.getSqlca().sqlcode << endl;
    }
  } catch(...) {
    cout << "Unknown exception in generatePriceList" << endl;
  }
}

//******************************************************************************
// Function Name: postingInventory                                             *
// Note: This member function updates the inventory table with the received    *
//       received order table.                                                 *
//******************************************************************************
void Store::postingInventory() {
  double newAveCost;                      // new average cost after update
  double newListingPrice;                 // new listing price after update
  long   newQuantity;                     // new quantity after update
  char   selectConditionArg[] = "pFlag = 0";  // selection condition

  try {
    // get a copy of all the rows that meet the select condition.
    receivedOrderMgr.select(selectConditionArg);

    // set up the cursor to go through each item.
    ReceiveManager::sequenceType::Cursor cursor(*receivedOrderMgr.items());

    // loop until the end of the list.
    forCursor (cursor) {


      // Use the product number as the key to locate the row of the item
      // in the inventory table.
      inventoryObj.setPnumber(cursor.element()->pnum());
      inventoryObj.retrieve();

      // calculate the new quantity and the new average cost
      newQuantity = inventoryObj.onhquan() +
                    cursor.element()->rquan();
      newAveCost  = (inventoryObj.avgcost() * inventoryObj.onhquan() +
                    cursor.element()->tcost()) / newQuantity;
      newListingPrice = newAveCost * 1.50;     // 50% mark up of the new cost

      // update the row with the new data
      inventoryObj.setOnhquan(newQuantity);
      inventoryObj.setAvgcost(newAveCost);
      inventoryObj.setLstprice(newListingPrice);
      inventoryObj.update();

      // Commit changes to the Inventory connection if CLI or ODBC
      if (ISESQL != "TRUE") {
         dsm1.commit();
      }

      // reset the posted flag so that it will not be used again.
      cursor.element()->setPflag(POSTED);
      cursor.element()->update();
    }
  } catch (IDSAccessError &exc) {
    cout << "Failed: postingInventory " << endl;
    cout << "Error id:"  << exc.errorId() << endl;
    cout << "Error Text:" << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
    if ((exc.errorId() == DAX_SEL_SQLERR) ||
        (exc.errorId() == DAX_RET_SQLERR) ||
        (exc.errorId() == DAX_UPD_SQLERR)) {
      cout << "SQLCODE: " << exc.getSqlca().sqlcode << endl;
    }
  } catch (IException &exc) {
    cout << "Failed: postingInventory" << endl;
    cout << "Error id: " << exc.errorId() << endl;
    cout << "Error Text: " << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
  } catch(...) {
    cout << "Unknown exception in generatePriceList" << endl;
  }
}

//******************************************************************************
// Function: addReceivedOrder                                                  *
// Note: Whenever an order is received, a new entry is added to the received   *
//       order table                                                           *
//******************************************************************************
void Store::addReceivedOrder(
       short   receivedNumberArg,     //received order number
       IString prodNumberArg,         // product number
       long    receivedQuantityArg,   // received order quantity
       double  totalCostArg ) {          // total cost of this order
  try {
    receivedOrderObj.setPflag           (NOT_POSTED);
    receivedOrderObj.setPnum            (prodNumberArg);
    receivedOrderObj.setRnumber         (receivedNumberArg);
    receivedOrderObj.setRquan           (receivedQuantityArg);
    receivedOrderObj.setTcost           (totalCostArg);
    receivedOrderObj.add();
  } catch (IDSAccessError &exc) {
    cout << "Failed: addReceivedOrder " << endl;
    cout << "Error id:"  << exc.errorId() << endl;
    cout << "Error Text:" << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
    if (exc.errorId() == DAX_ADD_SQLERR) {
      cout << "SQLCODE: " << exc.getSqlca().sqlcode << endl;
    }
  } catch(...) {
    cout << "Unknown exception in addReceiveOrder" << endl;
  }
}

//******************************************************************************
// Function: transaction
// Note: This function commits or rollbacks the activities between transactions
//******************************************************************************
void Store::transaction(commitflag flagArg) {  // commit or rollback
  try {
    if (flagArg == COMMIT) {
       if (ISESQL == "TRUE") {
          dsm3.commit();                           // Commit for Embedded SQL
       }
       else {
         dsm.commit();                             // Commit for ODBC or CLI
         dsm1.commit();
         dsm2.commit();
       }
    } else {
         if (ISESQL == "TRUE") {
            dsm3.rollback();                       // Rollback for Embedded SQL
         }
         else {
            dsm.rollback();                        // Rollback for CLI or ODBC
            dsm1.rollback();
            dsm2.rollback();
         }
    }
  } catch (IException &exc) {
    cout << "Failed: transaction " << endl;
    cout << "Error id:"  << exc.errorId() << endl;
    cout << "Error Text:" << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
  } catch(...) {
    cout << "Unknown exception in transaction" << endl;
  }
}

//******************************************************************************
// Function: addNewItem
// Note: This function add a new row to the inventory table
//******************************************************************************
void Store::addNewItem(
       IString prodNumberArg,       // product number
       IString prodDescriptionArg,  // product description
       long    onHandQuantityArg,   // on hand quantity
       double  averageCostArg,      // average cost per unit
       double  listingPriceArg) {   // listing price per unit
  try {
    inventoryObj.setPnumber         (prodNumberArg);
    inventoryObj.setPdes            (prodDescriptionArg);
    inventoryObj.setOnhquan         (onHandQuantityArg);
    inventoryObj.setAvgcost         (averageCostArg);
    inventoryObj.setLstprice        (listingPriceArg);
    inventoryObj.add();
  } catch (IDSAccessError &exc) {
    cout << "Failed: addNewItem " << endl;
    cout << "Error id:"  << exc.errorId() << endl;
    cout << "Error Text:" << exc.text() << endl;
    cout << "Error Class: " << exc.name() << endl;
  } catch(...) {
    cout << "Unknown exception in addNewItem" << endl;
  }
}

//******************************************************************************
// Function: displayItem
// Note: This function displays a new row in the inventory table
//******************************************************************************
void Store::displayItem() {
      cout << inventoryObj.pnumber() << ' ';
      cout << inventoryObj.pdes() << ' ';
      cout << inventoryObj.lstprice() << endl;
}


