//******************************************************************************
//                                                                             *
//COPYRIGHT: Copyright (C) International Business Machines Corp., 1992,1994.   *
//                                                                             *
//DISCLAIMER OF WARRANTIES:                                                    *
//  The following [enclosed] code is sample code created by IBM                *
//  Corporation.  This sample code is not part of any standard IBM product     *
//  and is provided to you solely for the purpose of assisting you in the      *
//  development of your applications.  The code is provided "AS IS",           *
//  without warranty of any kind.  IBM shall not be liable for any damages     *
//  arising out of your use of the sample code, even if they have been         *
//  advised of the possibility of such damages.                                *
//                                                                             *
//******************************************************************************
//******************************************************************************
// File Name: client.cpp                                                       *
//                                                                             *
// Note: This file keeps the SOM program which directly uses the methods from  *
//       SOM IDL CLass Library to connect and disconnect a datastore in DB2/2  *
//       and uses the generated methods to add, update, and retrieve.          *
//                                                                             *
//******************************************************************************
#include <iostream.h>
#include <stdlib.h>
#include <string.h>
#include <IString.hpp>
#include <iomanip.h>
#include "sdsmcon.xh"           // For SOM IDL Datastore Class
#include "sdsexc.xh"            // For SOM Exception Classes
#include "prclist.xh"          // For SOM IDL generated PriceList Class
#include "invento.xh"          // For SOM IDL generated Inventory Class
#include "receive.xh"          // For SOM IDL generated Receivedorder Class

static int exceptionCheck(void); // Routine to check if an exception has occurred.
static int generatePriceList();  // Generate a price list
static int postingInventory();  // update the inventory data with received order

static int addReceivedOrder(     // Add a received order to the database
    short  receivedNumberArg,    // received order number
    string prodNumberArg,        // product number
    long    receivedQuantityArg, // recieved quantity
    double  totalCostArg,        // total cost of the order
    string receivedDateArg);     // date of the received order

static int addNewItem(           // add a new item to the inventory table
    string prodNumberArg,        // product number
    string prodDescriptionArg,   // product description
    long    onHandQuantityArg,   // on hand quantity
    double  avarageCostArg,      // average cost per unit
    double  listingPriceArg);    // listing price per unit

//******************************************************************************
// Static variables                                                            *
//******************************************************************************
enum postedflagValue {NOT_POSTED, POSTED};
Environment                *ev                  = somGetGlobalEnvironment();
DatastoreFactory           *pidFactDS           = DatastoreNewClass(0,0);
PrclistFactory             *prclistFacPtr       = PrclistNewClass(0,0);
ReceiveFactory             *receiveFacPtr       = ReceiveNewClass(0,0);
InventoFactory             *inventoFacPtr       = InventoNewClass(0,0);

void main(int argc, char * argv[]) {
  Datastore            *dsm;
  PrclistDatastore     *dsm1;
  InventoDatastore     *dsm2;
  ReceiveDatastore     *dsm3;

  char *dstoreId;
  double listPrice, aveCost, totalCost;
  int rc;

  IString  ISESQL;

  // Establish connection to the datastore.

  ISESQL = getenv("ISESQL");
  if (ISESQL == "TRUE")  {

     dsm = pidFactDS->create_object(ev);

     if (argc < 3) {
       dsm->_set_datastore_name(ev, "DAXSAMP");
       dsm->connect_defaults(ev);
     }
     else {
       dsm->connect(ev,"DAXSAMP",argv[1],argv[2]);
     }
    rc = exceptionCheck();
  }
  else {

     dsm1 = prclistFacPtr->create_datastore(ev);
     dsm2 = inventoFacPtr->create_datastore(ev);
     dsm3 = receiveFacPtr->create_datastore(ev);

    if (argc == 3) {
      dsm1->_set_user_name(ev,argv[1]);
      dsm1->_set_authentication(ev,argv[2]);
      dsm2->_set_user_name(ev,argv[1]);
      dsm2->_set_authentication(ev,argv[2]);
      dsm3->_set_user_name(ev,argv[1]);
      dsm3->_set_authentication(ev,argv[2]);
    }
    dsm1->connect_defaults(ev);
    rc = exceptionCheck();
    dsm2->connect_defaults(ev);
    rc+= exceptionCheck();
    dsm3->connect_defaults(ev);
    rc+= exceptionCheck();

  }

  if (rc == 0) {

    // Generate a price list from the Price List object class.
    cout << "The following is a price list before update:" << endl << endl;
    rc = generatePriceList();
    if (rc == 0) {

      // Add new items to the inventory table.
      rc = addNewItem("HDR0025","Hard Drive 250MD ID", 12, 200.00, 239.45);
      rc += addNewItem("HDR0034","Hard Drive 340MD ID", 40, 220.00, 270.89);
      rc += addNewItem("CPU486a","486SLC-33 TI CPU,AM", 50, 105.45, 139.99);
      rc += addNewItem("CPUPTMa","Pentium 60/66 PCI25", 23, 350.85, 459.50);

      // continue if adding new inventory items is OK.
      if (rc == 0) {

        // Add new records to the receivedorder table.
        rc = addReceivedOrder(123,"HDR0025", 10, 1800.00, "01/01/1994");
        rc += addReceivedOrder(124,"RAM4-72", 30, 6001.35, "03/01/1994");
        rc += addReceivedOrder(125,"CPUPTMa",  5, 1904.75, "02/01/1994");
        rc += addReceivedOrder(126,"MONAD14", 40, 1538.00, "04/01/1994");

        // continue if adding new received orders is OK.
        if (rc == 0) {

          // The inventory has to be posted with the new received orders
          rc = postingInventory();
          if (rc == 0) {

            // Commit transaction activities after successful posting
            if (!strcmp(ISESQL,"TRUE"))  {
               dsm->commit(ev);
               rc+= exceptionCheck();
            }
            else {
               dsm1->commit(ev);
               rc+= exceptionCheck();
               dsm2->commit(ev);
               rc+= exceptionCheck();
               dsm3->commit(ev);
               rc+= exceptionCheck();
            }

            rc = exceptionCheck();
            if (rc == 0) {

              cout << endl << endl;
              cout << "The following is an updated price list:" << endl << endl;

              // Generate an updated report after posting.
              generatePriceList();

            }

          } else {

            // Rollback transaction activities because of error in posting
            if (!strcmp(ISESQL,"TRUE"))  {
               dsm->rollback(ev);
               rc+= exceptionCheck();
            }
            else {
               dsm1->rollback(ev);
               rc+= exceptionCheck();
               dsm2->rollback(ev);
               rc+= exceptionCheck();
               dsm3->rollback(ev);
               rc+= exceptionCheck();
            }
          }

        } else {

          // Rollback transaction activities because of error in adding receive orders
          if (!strcmp(ISESQL,"TRUE"))  {
             dsm->rollback(ev);
             rc+= exceptionCheck();
          }
          else {
             dsm1->rollback(ev);
             rc+= exceptionCheck();
             dsm2->rollback(ev);
             rc+= exceptionCheck();
             dsm3->rollback(ev);
             rc+= exceptionCheck();
          }
        }
      } else {

        // Rollback transaction activities because of error in adding new items
        if (!strcmp(ISESQL,"TRUE"))  {
           dsm->rollback(ev);
           rc+= exceptionCheck();
        }
        else {
           dsm1->rollback(ev);
           rc+= exceptionCheck();
           dsm2->rollback(ev);
           rc+= exceptionCheck();
           dsm3->rollback(ev);
           rc+= exceptionCheck();
        }
      }
    }

    // disconnect from the datastore.
    if (!strcmp(ISESQL,"TRUE"))  {
       dsm->disconnect(ev);
    }
    else {
       dsm1->commit(ev);
       dsm1->disconnect(ev);
       rc+= exceptionCheck();
       dsm2->commit(ev);
       dsm2->disconnect(ev);
       rc+= exceptionCheck();
       dsm3->commit(ev);
       dsm3->disconnect(ev);
       rc+= exceptionCheck();
    }
  }

  cout << endl;
  if (rc == 0) {
     cout << "Program completed successfully" << endl;
  } else {
     cout << "Error occured during program excution." << endl;
  } /* endif */

}

//******************************************************************************
// Function Name: generatePriceList                                            *
// Note : This member function generates a price list for the store.           *
//        The data printed in the list is a subset of the inventory            *
//        database.                                                            *
//******************************************************************************
int generatePriceList() {
  Prclist                      *priceListPtr;
  _IDL_SEQUENCE_PersistentObject itemList;
  int rc;

  itemList = prclistFacPtr->retrieveAll(ev);
  rc = exceptionCheck();
  if (rc != 0) {
    cout << "Failed: Exception occurs..." << endl;
  } else {
    // loop until the end of the list.
    cout << "Prod. ID   Description            Price " << endl;
    cout << "========   =====================  ===== " << endl;
    for (int j=0; j < sequenceLength(itemList); j++) {
      priceListPtr = (Prclist *) sequenceElement(itemList,j);
      cout << priceListPtr->_get_pnumber(ev) << ' ';
      cout << priceListPtr->_get_pdes(ev) << "   $";
      cout << setiosflags(ios::fixed) <<
              setprecision(2) << priceListPtr->_get_lstprice(ev) << endl;
    }
  }

  return rc;
}

//******************************************************************************
// Function Name: postingInventory                                             *
// Note: This member function updates the inventory table with the received    *
//       received order table.                                                 *
//******************************************************************************
int postingInventory() {
  Receive                  *receivedOrderPtr;
  Invento                  *inventoryPtr     = inventoFacPtr->create_object(ev);
  _IDL_SEQUENCE_PersistentObject itemList;
  double newAveCost;                      // new average cost after update
  double newListingPrice;                 // new listing price after update
  long   newQuantity;                     // new quantity after update
  char   selectArg[] = "pFlag = 0";  // selection condition
  char * currentProdNumber;
  int icnt, rc;

  itemList = receiveFacPtr->select(ev, selectArg);
  rc = exceptionCheck();
  if (rc != 0) {
    cout << "Error: Exception occurs..." << endl;
  } else {
    for (icnt = 0, rc = 0; icnt < sequenceLength(itemList) && rc ==0; icnt ++) {
      receivedOrderPtr = (Receive *) sequenceElement(itemList,icnt);
      currentProdNumber = receivedOrderPtr->_get_pnum(ev);
      inventoryPtr->_set_pnumber(ev,currentProdNumber);
      inventoryPtr->retrieve(ev);

      // calculate the new quantity, the new average cost, and new listing price
      newQuantity = inventoryPtr->_get_onhquan(ev) +
                    receivedOrderPtr->_get_rquan(ev);
      newAveCost  = (inventoryPtr->_get_avgcost(ev) * inventoryPtr->_get_onhquan(ev) +
                    receivedOrderPtr->_get_tcost(ev)) / newQuantity;
      newListingPrice = newAveCost * 1.5;

      // update the row with the new data
      inventoryPtr->_set_onhquan(ev, newQuantity);
      inventoryPtr->_set_avgcost(ev, newAveCost);
      inventoryPtr->_set_lstprice(ev, newListingPrice);
      inventoryPtr->update(ev);
      rc = exceptionCheck();

      // reset the posted flag so that it will not be used again.
      receivedOrderPtr->_set_pflag(ev, POSTED);
      receivedOrderPtr->update(ev);
      rc = exceptionCheck();
    }
  } /* endif */

  delete inventoryPtr;
  return rc;

}

//******************************************************************************
// Function: addReceivedOrder                                                  *
// Note: Whenever an order is received, a new entry is added to the received   *
//       order table                                                           *
//******************************************************************************
int addReceivedOrder(
       short   receivedNumberArg,     //received order number
       char *  prodNumberArg,         // product number
       long    receivedQuantityArg,   // received order quantity
       double  totalCostArg,          // total cost of this order
       char *  receivedDateArg) {     // date of the received order

  Receive              *receivedOrderPtr = receiveFacPtr->create_object(ev);
  int rc;
  // Set the attributes
  receivedOrderPtr->_set_pflag           (ev,NOT_POSTED);
  receivedOrderPtr->_set_pnum         (ev,prodNumberArg);
  receivedOrderPtr->_set_rnumber         (ev,receivedNumberArg);
//  receivedOrderPtr->_set_receiveddate    (ev,receivedDateArg);
  receivedOrderPtr->_set_rquan           (ev,receivedQuantityArg);
  receivedOrderPtr->_set_tcost           (ev,totalCostArg);

  // add a new row to the table
  receivedOrderPtr->add(ev);

  // Check for exception and return a value accordingly.
  rc =exceptionCheck();

  delete receivedOrderPtr;
  return rc;
}

//******************************************************************************
// Function: addNewItem                                                        *
// Note: This function add a new row to the inventory table                    *
//******************************************************************************
int addNewItem(
       char * prodNumberArg,       // product number
       char * prodDescriptionArg,  // product description
       long   onHandQuantityArg,   // on hand quantity
       double averageCostArg,      // average cost per unit
       double listingPriceArg) {   // listing price per unit

  Invento                  *inventoryPtr     = inventoFacPtr->create_object(ev);
  int rc;

  // Set the attributes
  inventoryPtr->_set_pnumber         (ev,prodNumberArg);
  inventoryPtr->_set_pdes            (ev,prodDescriptionArg);
  inventoryPtr->_set_onhquan         (ev,onHandQuantityArg);
  inventoryPtr->_set_avgcost     (ev,averageCostArg);
  inventoryPtr->_set_lstprice        (ev,listingPriceArg);

  // add a new row to the table
  inventoryPtr->add(ev);

  // Check for exception and return a value accordingly.
  rc = exceptionCheck();

  delete inventoryPtr;
  return rc;
}

//******************************************************************************
// Function: exceptionCheck                                                    *
// Note: This function add a new row to the inventory table                    *
//******************************************************************************
int exceptionCheck(void) {
   int rc = 0;
   char *exId;
   DA_DatastoreAccessError * er;

   switch (ev->_major) {
   case SYSTEM_EXCEPTION:
      cout << "system exception" << endl;
      rc ++;
      break;
   case USER_EXCEPTION:
      cout << "User Exception" << endl;
      char * exId;
      exId = somExceptionId(ev);
      cout << "Exception ID: " << somExceptionId(ev) << endl;
      er = (DA_DatastoreAccessError *) somExceptionValue(ev);
      cout << "error code:" << er->error_code << endl;
      cout << "error text:" << er->error_text << endl;
      cout << "error state:" << er->error_state << endl;
      cout << "error number:" << er->error_number << endl;
      rc ++;
      somExceptionFree(ev);
      break;
   case NO_EXCEPTION:
      break;
   }
   return rc;
}

