/******************************************************************************
* .FILE:         2d-bmap.cpp                                                  *
*                                                                             *
* .DESCRIPTION:  2D Bitmap Graphic Sample Program:  Implementation            *
*                                                                             *
* .CLASSES:      DrawingAreaPaintHandler                                      *
*                MainCommandHandler                                           *
*                DrawingArea                                                  *
*                MainWindow                                                   *
*                                                                             *
* .COPYRIGHT:                                                                 *
*    Licensed Material - Program-Property of IBM                              *
*    (C) Copyright IBM Corp. 1992, 1996 - All Rights Reserved                 *
*                                                                             *
* .DISCLAIMER:                                                                *
*   The following [enclosed] code is sample code created by IBM               *
*   Corporation.  This sample code is not part of any standard IBM product    *
*   and is provided to you solely for the purpose of assisting you in the     *
*   development of your applications.  The code is provided 'AS IS',          *
*   without warranty of any kind.  IBM shall not be liable for any damages    *
*   arising out of your use of the sample code, even if they have been        *
*   advised of the possibility of such damages.                               *
*                                                                             *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT THE SOURCE          *
*                                                                             *
******************************************************************************/

#include <ibase.hpp>
#include <ifiledlg.hpp>
#include <ihandle.hpp>
#include <igelipse.hpp>
#include <igrect.hpp>
#include <igpie.hpp>
#include <igrafctx.hpp>
#include <igregion.hpp>
#include <imenubar.hpp>
#include <ireslib.hpp>
#include <icoordsy.hpp>
#include "2dbmap.hpp"

/*****************************************************************************
* main  - Application entry point                                            *
*****************************************************************************/
int main()
{

  ICoordinateSystem::setApplicationOrientation(
        ICoordinateSystem::originLowerLeft );
  MainWindow  mainWindow(WND_MAIN);
  IApplication::current().run();

  return 0;
} /* end main */


/*******************************************************************************
* Class MainWindow::MainWindow - Constructor for the main window               *
*                                                                              *
* Define itself as a IFrameWindow                                              *
* Instatiate the viewport to display the bitmap.                               *
* Create a drawing area for the viewport.                                      *
* Instatiate the command handler to handle command events.                     *
*******************************************************************************/
MainWindow::MainWindow(unsigned long windowId)
  : MainWindow::Inherited(              //  Call Inherited constructor
    IFrameWindow::defaultStyle()        //  Use default plus
    | IFrameWindow::animated            //  Set to show with "animation"
    | IFrameWindow::menuBar             //  Frame has a menu bar
    | IFrameWindow::minimizedIcon,      //  Frame has an icon
    windowId),                          //  Main Window ID
    fViewPort( WND_VPORT, this, this ),
    fDrawingArea( WND_DRAW, &fViewPort, &fViewPort ),
    fMainCommandHandler()
{
/*-----------------------------------------------------------------------------|
| Begin handling events for MainWindow and set the client area                 |
------------------------------------------------------------------------------*/
  fMainCommandHandler.handleEventsFor(this);
  setClient( &fViewPort );

/*-----------------------------------------------------------------------------|
| Create the menu bar and disable menu items that require a loaded bitmap.     |                                                                           |
------------------------------------------------------------------------------*/
  IMenuBar menuBar(windowId, this);
  menuBar.setAutoDestroyWindow(false);
  menuBar.disableItem( IDM_REFLECTHORZ );
  menuBar.disableItem( IDM_REFLECTVERT );
  menuBar.disableItem( IDM_ROTATE90    );
  menuBar.disableItem( IDM_ROTATE180   );
  menuBar.disableItem( IDM_ROTATE270   );
  menuBar.disableItem( IDM_TRANSPOSE   );
  menuBar.disableItem( IDM_CLIPCIRCLES );
  menuBar.disableItem( IDM_CLIPSQUARES );
  menuBar.disableItem( IDM_CLIPRAD     );
  menuBar.disableItem( IDM_CLIPNONE    );

/*-----------------------------------------------------------------------------|
| Set the focus for the window and show the window on the screen.              |
------------------------------------------------------------------------------*/
  setFocus();
  show();
}

/*******************************************************************************
* Class MainWindow::~MainWindow - Destructor for the main window               *
*******************************************************************************/
MainWindow::~MainWindow()
{
/*-----------------------------------------------------------------------------|
| Stop handling events for the MainWindow                                      |
------------------------------------------------------------------------------*/
  fMainCommandHandler.stopHandlingEventsFor(this);
}

/*******************************************************************************
* Class MainWindow::modifyBitmap - modify the bitmap according to the command  *
*   event.                                                                     *
*******************************************************************************/
MainWindow& MainWindow::modifyBitmap( unsigned long eventId )
{
  IGBitmap* bitmap(fDrawingArea.bitmap());
  if (bitmap)
  {
    switch (eventId)
    {
      case  IDM_REFLECTHORZ:
        bitmap->reflectHorizontally();
      break;
      case  IDM_REFLECTVERT:
        bitmap->reflectVertically();
      break;
      case  IDM_ROTATE90:
        bitmap->rotateBy90();
      break;
      case  IDM_ROTATE180:
        bitmap->rotateBy180();
      break;
      case  IDM_ROTATE270:
        bitmap->rotateBy270();
      break;
      case  IDM_TRANSPOSE:
        bitmap->transposeXForY();
      break;
    } /* endswitch */
    ISize bitmapSize(bitmap->size());
    fDrawingArea.sizeTo(bitmapSize);
    fViewPort.setViewWindowSize(bitmapSize);
    fDrawingArea.refresh();
  }
  return *this;
}

/*******************************************************************************
* Class MainCommandHandler::command - Handles command events for MainWindow    *
*******************************************************************************/
IBase::Boolean MainCommandHandler::command( ICommandEvent& event )
{
  Boolean fProcessed = false;
  switch (event.commandId())
  {
    case  IDM_FILEOPEN:
    {
      MainWindow *mainWindow((MainWindow*)event.window());
      IResourceLibrary reslib;
      IFileDialog::Settings fsettings;      //                                     .
      fsettings.setTitle(TITLE_LD_IMG);//Set Open Dialog Title from Resource  .
      fsettings.setFileName(reslib.loadString(STR_FL_NAME));          //Set FileNames to *.hlo               .

      IFileDialog fd(                          //Create File Open Dialiog             .
        IWindow::desktopWindow(),              //  Parent is Desktop                  .
        mainWindow, fsettings);                //  Owner is me with settings          .
      if (fd.pressedOK())                      //Check if ok from file open dialog    .
      {
        mainWindow->loadImageFile(fd.fileName());
/*-----------------------------------------------------------------------------|
| Enable menubar items since a bitmap file has been loaded.                    |
------------------------------------------------------------------------------*/
        IMenuBar menuBar(mainWindow->id(), mainWindow);
        menuBar.setAutoDestroyWindow(false);
        menuBar.enableItem( IDM_REFLECTHORZ );
        menuBar.enableItem( IDM_REFLECTVERT );
        menuBar.enableItem( IDM_ROTATE90    );
        menuBar.enableItem( IDM_ROTATE180   );
        menuBar.enableItem( IDM_ROTATE270   );
        menuBar.enableItem( IDM_TRANSPOSE   );
        menuBar.enableItem( IDM_CLIPCIRCLES );
        menuBar.enableItem( IDM_CLIPSQUARES );
        menuBar.enableItem( IDM_CLIPRAD     );
        menuBar.enableItem( IDM_CLIPNONE    );
      }
    }
    break;
    case  IDM_QUIT:
    {
      IFrameWindow *frameWindow((IFrameWindow*)event.window());
      frameWindow->close();
    }
    break;
    case  IDM_REFLECTHORZ:
    case  IDM_REFLECTVERT:
    case  IDM_ROTATE90:
    case  IDM_ROTATE180:
    case  IDM_ROTATE270:
    case  IDM_TRANSPOSE:
    {
      MainWindow *mainWindow((MainWindow*)event.window());
      mainWindow->modifyBitmap( event.commandId() );
    }
    break;

    case  IDM_CLIPCIRCLES:
    case  IDM_CLIPSQUARES:
    case  IDM_CLIPRAD:
    case  IDM_CLIPNONE:
    {
      MainWindow *mainWindow((MainWindow*)event.window());
      mainWindow->drawingArea().setClipStyle( event.commandId() );
      mainWindow->drawingArea().refresh();
    }
    break;
  } /* endswitch */

  return fProcessed;
}

/*******************************************************************************
* Class MainWindow::loadImageFile - Loads an image file into a drawing canvas. *
*******************************************************************************/
MainWindow& MainWindow::loadImageFile( const IString& imageFile )
{
  fDrawingArea.loadBitmap( imageFile );
  fViewPort.setViewWindowSize( fDrawingArea.bitmap()->size());
  return *this;
}

/*******************************************************************************
* Class DrawingArea :: DrawingArea - Constructor for the drawing area          *
*******************************************************************************/

DrawingArea::DrawingArea( unsigned long id, IWindow* parent, IWindow* owner )
  : DrawingArea::Inherited( id, parent, owner, IRectangle(),
                            IDrawingCanvas::defaultStyle() & ~IDrawingCanvas::useDefaultPaintHandler ),
    fBitmap(0),
    fStyle(IDM_CLIPNONE)
{
  drawingAreaPaintHandler.handleEventsFor(this);
}


/*******************************************************************************
* Class DrawingArea::~DrawingArea - Destructor for the drawing area            *
*******************************************************************************/
DrawingArea::~DrawingArea()
{
  drawingAreaPaintHandler.stopHandlingEventsFor(this);
  if (fBitmap)
    delete fBitmap;
}


/*******************************************************************************
* Class DrawingArea::bitmap - Returns the IGBitmap currently being used        *
*******************************************************************************/
IGBitmap* DrawingArea::bitmap( ) const
{
  return fBitmap;
}

/*******************************************************************************
* Class DrawingArea::loadBitmap - Loads a new bitmap from a file               *
*******************************************************************************/
DrawingArea& DrawingArea::loadBitmap( const IString& imageFile )
{
  if (fBitmap)
    delete fBitmap;
  fBitmap = new IGBitmap( imageFile );
  sizeTo(fBitmap->size());
  refresh();
  return *this;
}


/*******************************************************************************
* Class DrawingArea::calcMinimumSize - Calculates the size of the bitmap       *
*******************************************************************************/
ISize DrawingArea::calcMinimumSize( ) const
{
  if (fBitmap)
    return fBitmap->size();
  else
    return ISize();
}

/*******************************************************************************
* Class DrawingAreaPaintHandler::paintWindow - Handles the repainting of the   *
*   window.                                                                    *
*******************************************************************************/
IBase::Boolean DrawingAreaPaintHandler::paintWindow( IPaintEvent& event )
{
/*-----------------------------------------------------------------------------|
  // Get a graphic context
------------------------------------------------------------------------------*/
  IGraphicContext gc(event.presSpaceHandle());

/*-----------------------------------------------------------------------------|
  // Get a pointer to the current bitmap if one exists.
------------------------------------------------------------------------------*/
  IGBitmap* bitmap(((DrawingArea*)event.window())->bitmap());

/*-----------------------------------------------------------------------------|
  // Get the dimensions of the window
------------------------------------------------------------------------------*/
  IRectangle windowRect(IPoint(),((DrawingArea*)event.window())->size());

/*-----------------------------------------------------------------------------|
  // paint the current background color
------------------------------------------------------------------------------*/
  event.clearBackground();

/*-----------------------------------------------------------------------------|
  // Query the current clipping style
------------------------------------------------------------------------------*/
  unsigned long clipStyle(((DrawingArea*)event.window())->clipStyle());

  IGRegion region;

/*-----------------------------------------------------------------------------|
  // Using an empty region, draw a clipping pattern.
  // Set the clipping region.
------------------------------------------------------------------------------*/
  switch (clipStyle)
  {
    case IDM_CLIPCIRCLES:
    {
      IGRectangle rect( windowRect );
      region -= rect;
      IGEllipse ellipse( windowRect );
      region += ellipse;
      windowRect.shrinkBy( 25 );
      ellipse.setEnclosingRect( windowRect );
      region -= ellipse;
      windowRect.shrinkBy( 25 );
      ellipse.setEnclosingRect( windowRect );
      region += ellipse;
      windowRect.shrinkBy( 25 );
      ellipse.setEnclosingRect( windowRect );
      region -= ellipse;
      windowRect.shrinkBy( 25 );
      ellipse.setEnclosingRect( windowRect );
      region += ellipse;
      windowRect.shrinkBy( 25 );
      ellipse.setEnclosingRect( windowRect );
      region -= ellipse;
      windowRect.shrinkBy( 25 );
      ellipse.setEnclosingRect( windowRect );
      region += ellipse;
      gc.setClipRegion( region );
      break;
    }
    case IDM_CLIPSQUARES:
    {
      IGRectangle rect( windowRect );
      region += rect;
      windowRect.shrinkBy( 25 );
      rect.setEnclosingRect( windowRect );
      region -= rect;
      windowRect.shrinkBy( 25 );
      rect.setEnclosingRect( windowRect );
      region += rect;
      windowRect.shrinkBy( 25 );
      rect.setEnclosingRect( windowRect );
      region -= rect;
      windowRect.shrinkBy( 25 );
      rect.setEnclosingRect( windowRect );
      region += rect;
      windowRect.shrinkBy( 25 );
      rect.setEnclosingRect( windowRect );
      region -= rect;
      windowRect.shrinkBy( 25 );
      rect.setEnclosingRect( windowRect );
      region += rect;
      gc.setClipRegion( region );
      break;
    }
    case IDM_CLIPRAD:
    {
      IGPie pie( windowRect, 0, 60 );
      region += pie;
      pie.setStartAngle( 120 );
      region += pie;
      pie.setStartAngle( 240 );
      region += pie;
      gc.setClipRegion( region );
      break;
    }
  } /* endswitch */

/*-----------------------------------------------------------------------------|
| Draw the bitmap if we have one                                               |
------------------------------------------------------------------------------*/
  if (bitmap)
  {
    bitmap->drawOn(gc);
  }

/*-----------------------------------------------------------------------------|
| Clear the clip region                                                        |
------------------------------------------------------------------------------*/
  gc.clearClipRegion( );

  return true;
}

