/******************************************************************************
* .FILE:         2d-draw.cpp                                                  *
*                                                                             *
* .DESCRIPTION:  2D Graphics Drawing Program :         Class Implementation   *
*                                                                             *
* .CLASSES:      MainWindow                                                   *
*                DrawingArea                                                  *
*                                                                             *
* .COPYRIGHT:                                                                 *
*    Licensed Material - Program-Property of IBM                              *
*    (C) Copyright IBM Corp. 1992, 1996 - All Rights Reserved                 *
*                                                                             *
* .DISCLAIMER:                                                                *
*   The following [enclosed] code is sample code created by IBM               *
*   Corporation.  This sample code is not part of any standard IBM product    *
*   and is provided to you solely for the purpose of assisting you in the     *
*   development of your applications.  The code is provided 'AS IS',          *
*   without warranty of any kind.  IBM shall not be liable for any damages    *
*   arising out of your use of the sample code, even if they have been        *
*   advised of the possibility of such damages.                               *
*                                                                             *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT THE SOURCE          *
*                                                                             *
******************************************************************************/
#ifndef _IBASE_                         //Make sure ibase.hpp is included
  #include <ibase.hpp>                  //  since that is where IC_<environ>
#endif                                  //  is defined.
#include <igline.hpp>
#include <igpyline.hpp>
#include <igelipse.hpp>
#include <igarc.hpp>
#include <igpie.hpp>
#include <igstring.hpp>
#include <igbitmap.hpp>
#include <icolor.hpp>
#include <iglist.hpp>
#include <math.h>
#include "2ddraw.hpp"

//*************************************************************************
// main  - Application entry point                                        *
//*************************************************************************
int main()
{
  IToolBarButton::setDefaultStyle( IToolBarButton::defaultStyle() |
                                   IToolBarButton::autoLatch      |
                                   IToolBarButton::useIdForText   |
                                   IWindow::visible );
  IToolBar::setDefaultStyle(IToolBar::classDefaultStyle |
                            IToolBar::buttonBitmapAndTextVisible);

  MainWindow  mainWindow(WND_MAIN);     //Create our main window on the desktop

  IApplication::current().run();        //Get the current application and
                                        // run it
  return 0;
} /* end main */


/*------------------------------------------------------------------------------
| MainWindow::MainWindow                                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
MainWindow::MainWindow(unsigned long windowId)
  : IFrameWindow (                      //Call IFrameWindow constructor
    IFrameWindow::defaultStyle()        //  Use default plus
    | IFrameWindow::animated            //  Set to show with "animation"
    | IFrameWindow::menuBar             //  Frame has a menu bar
    | IFrameWindow::minimizedIcon,      //  Frame has an icon
    windowId),                          //  Main Window ID
    drawingArea( WND_DRAW, this, this ),
    toolBar( WND_TOOLBAR,this, IToolBar::aboveClient),
    menuBar( this, IMenuBar::wrapper ),
    infoArea( this, WND_TEXT ),
    flyText( 1054, &menuBar ),

    // Set the initial delay for fly over help to 1 1/2 seconds and
    // set the regular delay to 1/3 seconds.
    flyOver( &flyText, &infoArea, 1500, 333 ),
    lastPenColorId( ID_COL_BLK ),
    lastFillColorId( ID_FLCOL_BLK ),
    lastPenPatternId( ID_PENPATTERN_SOLID ),
    lastFillPatternId( ID_FILLPATTERN_SOLID ),
    lastPenTypeId( ID_PENTYPE_SOLID ),
    lastPenWidthId( ID_PENWIDTH_1 ),
    lastDrawOperationId( ID_FILLANDFRAME ),
    normalButton(PALLET_NORM, &toolBar, &toolBar),
    lineButton(PALLET_LINE, &toolBar, &toolBar),
    drawButton(PALLET_DRAW, &toolBar, &toolBar),
    rectangleButton(PALLET_RECTANGLE, &toolBar, &toolBar),
    ellipseButton(PALLET_ELLIPSE, &toolBar, &toolBar),
    polylineButton(PALLET_POLYLINE, &toolBar, &toolBar),
    polygonButton(PALLET_POLYGON, &toolBar, &toolBar),
    arcButton(PALLET_ARC, &toolBar, &toolBar),
    pieButton(PALLET_PIE, &toolBar, &toolBar),
    chordButton(PALLET_CHORD, &toolBar, &toolBar),
    textButton(PALLET_TEXT, &toolBar, &toolBar),
    bitmapButton(PALLET_BITMAP, &toolBar, &toolBar)
{
  ICommandHandler::handleEventsFor(this);
  IMenuDrawItemHandler::handleEventsFor(this);

  // Allow the information area to wrap to a second line.
  infoArea.setLineCount( 2 );

  setExtensionSize(&infoArea, (unsigned long)(2*IFont(&infoArea).maxCharHeight()));

  menuBar.checkItem( lastPenColorId      )
         .checkItem( lastFillColorId     )
         .checkItem( lastPenPatternId    )
         .checkItem( lastFillPatternId   )
         .checkItem( lastPenTypeId       )
         .checkItem( lastPenWidthId      )
         .checkItem( lastDrawOperationId );

  toolBar.addAsLast( &normalButton    )
         .addAsLast( &lineButton      )
         .addAsLast( &drawButton      )
         .addAsLast( &rectangleButton )
         .addAsLast( &ellipseButton   )
         .addAsLast( &polylineButton  )
         .addAsLast( &polygonButton   )
         .addAsLast( &arcButton       )
         .addAsLast( &pieButton       )
         .addAsLast( &chordButton     )
         .addAsLast( &textButton      )
         .addAsLast( &bitmapButton    );
  normalButton.latch();

  // Set the fly over help for the client window to the
  // help information for the pointer (normal) button.

  flyOver.setHelpText( drawingArea.handle(),
                       IResourceId(0),
                       IResourceId(PALLET_NORM + LONG_OFFSET ));

  flyOver.handleEventsFor( this );
  flyOver.handleEventsFor( &drawingArea );
  flyOver.setLongStringTableOffset( LONG_OFFSET );

  setClient( &drawingArea );
  setFocus();
  show();
}

/*------------------------------------------------------------------------------
| MainWindow::MainWindow                                                       |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
MainWindow::~MainWindow()
{
}

/*------------------------------------------------------------------------------
| angleFromPoints                                                              |
|                                                                              |
| Given two points determine the angle between the horizontal line through     |
| the first point and the line from the first point to the second point.       |
------------------------------------------------------------------------------*/
double angleFromPoints( const IPoint& center, const IPoint& drop )
{
  IPoint temp;
  double angle;

  temp  = drop - center;
  angle = atan2((double)temp.y(), (double)temp.x());
  angle *= 57.295779;

  if ( angle < 0.0 )
    angle += 360.0;

#ifdef IC_PM
  return angle;
#else
  return -angle;
#endif
}

/*------------------------------------------------------------------------------
| DrawingArea::DrawingArea                                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
DrawingArea::DrawingArea( unsigned long windowId, IWindow* parent,
                          IWindow* owner, const IRectangle& initial )
  : IDrawingCanvas( windowId, parent, owner, initial,
                    IDrawingCanvas::defaultStyle() | IWindow::clipSiblings ),
    currentfont(),
    currentBundle(),
    animateBundle(),
    currentBitmap(),
    dState( notDrawing ),
    currentObj( pointer ),
    iGraphic(0),
    moveGraphic(0),
    moveRect(IRectangle()),
    startingPt(), previousPt(), tempPt(),
    pointCount(0)
{
  currentBundle.setPenColor(IGraphicContext::defaultPenColor())
               .setFillColor(IGraphicContext::defaultFillColor())
               .setMixMode(IGraphicContext::defaultMixMode())
               .setDrawOperation( IGraphicBundle::fillAndFrame );

  animateBundle.setMixMode( IGraphicBundle::bitExclusiveOr )
               .setPenColor( IColor::white )
               .setFillColor( IColor::white )
               .setDrawOperation( IGraphicBundle::frame );

  setGraphicList( new IGList() );

  // Load the pointers.

  IResourceLibrary reslib;
  ptrLine      = reslib.loadPointer( PALLET_LINE );
  ptrDraw      = reslib.loadPointer( PALLET_DRAW );
  ptrRectangle = reslib.loadPointer( PALLET_RECTANGLE );
  ptrEllipse   = reslib.loadPointer( PALLET_ELLIPSE );
  ptrPolyline  = reslib.loadPointer( PALLET_POLYLINE );
  ptrPolygon   = reslib.loadPointer( PALLET_POLYGON );
  ptrArc       = reslib.loadPointer( PALLET_ARC );
  ptrPie       = reslib.loadPointer( PALLET_PIE );
  ptrChord     = reslib.loadPointer( PALLET_CHORD );
  ptrText      = reslib.loadPointer( PALLET_TEXT );
  ptrBitmap    = reslib.loadPointer( PALLET_BITMAP );

  ((IMouseHandler*)this)->handleEventsFor(this);
}

/*------------------------------------------------------------------------------
| DrawingArea::DrawingArea                                                     |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
DrawingArea::~DrawingArea( )
{
  // Delete all the graphic objects in the drawing canvas.
  IGList::Cursor graphicsCursor( *graphicList() );
  for ( graphicsCursor.setToFirst();
        graphicsCursor.isValid();
        graphicsCursor.setToNext() )
  {
    IGraphic* graphic(&(graphicList()->graphicAt(graphicsCursor)));
    delete graphic;
  } /* endfor */
  delete graphicList();
}

/*------------------------------------------------------------------------------
| DrawingArea::mouseClicked                                                    |
|                                                                              |
| Translate the mouse clicked events.                                          |
------------------------------------------------------------------------------*/
IBase::Boolean DrawingArea::mouseClicked( IMouseClickEvent& event )
{
  Boolean bRc = false;
  if ( event.mouseButton() == IMouseClickEvent::button1 &&
       event.mouseAction() == IMouseClickEvent::down )
  {
    button1Down(event.mousePosition());
    bRc = false;
  }
  else if ( event.mouseButton() == IMouseClickEvent::button1 &&
            event.mouseAction() == IMouseClickEvent::up )
  {
    button1Up(event.mousePosition());
    bRc = true;
  }
  else if ( event.mouseButton() == IMouseClickEvent::button1 &&
            event.mouseAction() == IMouseClickEvent::doubleClick )
  {
    button1DoubleClick(event.mousePosition());
    bRc = true;
  }
  else if ( event.mouseButton() == IMouseClickEvent::button2 &&
            event.mouseAction() == IMouseClickEvent::down )
  {
    button2Down(event.mousePosition());
  }
  else if ( event.mouseButton() == IMouseClickEvent::button2 &&
            event.mouseAction() == IMouseClickEvent::up )
  {
    button2Up(event.mousePosition());
  }

  return bRc;
}


/*------------------------------------------------------------------------------
| DrawingArea::changeMousePointer                                              |
|                                                                              |
| Change the cursor to reflect the object being drawn.                         |
------------------------------------------------------------------------------*/
IBase::Boolean DrawingArea::mousePointerChange( IMousePointerEvent& event )
{
  Boolean fReturn(true);

  switch (drawObject())
  {
    case PALLET_NORM:
      return fReturn = false;
    break;
    case PALLET_LINE:
      event.setMousePointer( ptrLine );
    break;
    case PALLET_DRAW:
      event.setMousePointer( ptrDraw );
    break;
    case PALLET_RECTANGLE:
      event.setMousePointer( ptrRectangle );
    break;
    case PALLET_ELLIPSE:
      event.setMousePointer( ptrEllipse );
    break;
    case PALLET_POLYLINE:
      event.setMousePointer( ptrPolyline );
    break;
    case PALLET_POLYGON:
      event.setMousePointer( ptrPolygon );
    break;
    case PALLET_ARC:
      event.setMousePointer( ptrArc );
    break;
    case PALLET_PIE:
      event.setMousePointer( ptrPie );
    break;
    case PALLET_CHORD:
      event.setMousePointer( ptrChord );
    break;
    case PALLET_TEXT:
      event.setMousePointer( ptrText );
    break;
    case PALLET_BITMAP:
      event.setMousePointer( ptrBitmap );
    break;
    default:
      fReturn = false;
    break;
  } // end switch

  return fReturn;
}

/*------------------------------------------------------------------------------
| DrawingArea::button1Down                                                     |
|                                                                              |
| Handle button 1 down messages.  This event indicates a new graphic object is |
| to be created of additional data points to add to an existing graphic object.|
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::button1Down( const IPoint& point )
{
  IGraphicContext gc(this->handle());
  gc.setGraphicBundle(animateBundle);

  switch (currentObj)
  {
    case pointer:
    {
      // Change all objects to the current pen and fill color.
      IGList::Cursor cursor( *graphicList(), gc, point );
      for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext())
      {
        IGraphic& graphic(graphicList()->graphicAt( cursor ));
        this->refresh( graphic.boundingRect( gc ));//    .expandBy(1) );
        graphic.setGraphicBundle( currentBundle );
        graphic.drawOn( gc );
      } /* endfor */
    }
    break;
    case line:
      startingPt = point;
      previousPt = point;
      iGraphic = new IGLine( startingPt, previousPt );
      setDrawState();
    break;
    case freeHand:
      iGraphic = new IGPolygon(IPointArray());
      ((IGPolygon*)iGraphic)->addPoint( point );
      setDrawState();
    break;
    case rectangle:
      startingPt = point;
      previousPt = point;
      iGraphic = new IGRectangle(IRectangle(startingPt, previousPt));
      setDrawState();
    break;
    case ellipse:
      startingPt = point;
      previousPt = point;
      iGraphic = new IGEllipse( startingPt, 0L );
      setDrawState();
    break;
    case polyline:
    case polygon:
      if (drawState() == notDrawing)
      {
        startingPt = point;
        previousPt = point;
        if (currentObj == polyline)
          iGraphic = new IGPolyline(IPointArray());
        else
          iGraphic = new IGPolygon(IPointArray());

        ((IGPolyline*)iGraphic)->addPoint( startingPt );
        ((IGPolyline*)iGraphic)->addPoint( previousPt );
        setDrawState();
      }
      else
      {
        ((IGPolyline*)iGraphic)->IGPolyline::drawOn( gc );
        ((IGPolyline*)iGraphic)->addPoint( point );
        ((IGPolyline*)iGraphic)->IGPolyline::drawOn( gc );
      }
    break;
    case arc:
      pointCount++;
      if (drawState() == notDrawing)
      {
        iGraphic = new IG3PointArc( point, IPoint(0,0), IPoint(0,0) );
        startingPt = point;
        previousPt = point;
      }
      else if ( pointCount == 2 )
      {
        previousPt = point;
        IGLine tempLine( ((IG3PointArc*)iGraphic)->startingPoint(), point );
        tempLine.drawOn( gc );
      }
      else if ( pointCount == 3 )
      {
        IGLine tempLine( ((IG3PointArc*)iGraphic)->startingPoint(), previousPt );
        tempLine.drawOn( gc );
        previousPt = point;
        ((IG3PointArc*)iGraphic)->setEndingPoint( point );
        iGraphic->drawOn( gc );
      }
      setDrawState();
    break;
    case pie:
    case chord:
      pointCount++;
      if (drawState() == notDrawing)
      {
        if (currentObj == pie)
          iGraphic = new IGPie( IRectangle(), 0, 0);
        else
          iGraphic = new IGChord( IRectangle(), 0, 0);

        ((IGPie*)iGraphic)->setEnclosingRect(
                             ((IGPie*)iGraphic)->enclosingRect().centerAt( point ));
        startingPt = point;
        previousPt = point;
      }
      else if ( pointCount == 2 )
      {
        previousPt = point;
        IGLine tempLine( ((IGPie*)iGraphic)->enclosingRect().center(), point );
        tempLine.drawOn( gc );
      }
      else if ( pointCount == 3 )
      {
        IGLine tempLine( ((IGPie*)iGraphic)->enclosingRect().center(), previousPt );
        tempLine.drawOn( gc );
        previousPt = point;
        double sweep(angleFromPoints( ((IGPie*)iGraphic)->enclosingRect().center(), point ));
        if ( sweep < ((IGPie*)iGraphic)->startAngle() )
          ((IGPie*)iGraphic)->setSweepAngle( 360.0 -
                                ( ((IGPie*)iGraphic)->startAngle() - sweep ));
        else
          ((IGPie*)iGraphic)->setSweepAngle( sweep -
                                            ((IGPie*)iGraphic)->startAngle());
        iGraphic->drawOn( gc );
      }
      setDrawState();
    break;
    case text:
      setDrawState();
    break;
    case bitmap:
      setDrawState();
    break;
  } /* endswitch */
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::button2Down                                                     |
|                                                                              |
| Determine the object under the mouse and start moving.                       |
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::button2Down( const IPoint& point )
{
  if (drawState() == notDrawing)
  {
    IGraphicContext gc(this->handle());
    gc.setGraphicBundle(animateBundle);

    moveGraphic = graphicList()->topGraphicUnderPoint( point, gc );
    if ( moveGraphic )
    {
      moveRect.setEnclosingRect(moveGraphic->boundingRect( gc ));
      previousPt = point;
      startingPt = point;
      capturePointer();
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::mouseMoved                                                      |
|                                                                              |
| Handle button 1 down mouse move events.  This allows data points to be       |
| moved while the object is drawn with a rubber band effect.                   |
------------------------------------------------------------------------------*/
IBase::Boolean DrawingArea::mouseMoved( IMouseEvent& event )
{
  IGraphicContext gc(this->handle());
  gc.setGraphicBundle(animateBundle);

  IPoint point(event.mousePosition());
  if ( hasPointerCaptured() )
  {
    IRectangle windowRect(this->rect());
    windowRect.moveTo(IPoint(0,0));
    if (!windowRect.contains(point))
    {
      if ((short)point.x() < (short)windowRect.left())
        point.setX(windowRect.left());
      else if ((short)point.x() > (short)windowRect.right())
        point.setX(windowRect.right());
      else if ((short)point.y() < (short)windowRect.bottom())
        point.setY(windowRect.bottom());
      else if ((short)point.y() > (short)windowRect.top())
        point.setY(windowRect.top());

      IPoint mapPt( IWindow::mapPoint( point,
                                       this->handle(),
                                       IWindow::desktopWindow()->handle()));

      IWindow::movePointerTo( mapPt );
    }
  }

  // If we're not moving an object
  if (!moveGraphic)
  {
    if ( drawState() == drawing )
    {
      switch (currentObj)
      {
        case pointer:
        break;
        case line:
          ((IGLine*)iGraphic)->drawOn( gc );
          ((IGLine*)iGraphic)->setEndingPoint( point );
          ((IGLine*)iGraphic)->drawOn( gc );
        break;
        case freeHand:
          ((IGPolygon*)iGraphic)->addPoint( point );
          ((IGPolyline*)iGraphic)->IGPolyline::drawOn( gc );
        break;
        case rectangle:
        {
          IRectangle rc(((IGRectangle*)iGraphic)->enclosingRect());
          iGraphic->drawOn( gc );
          rc.sizeTo( rc.size() + point - previousPt );
          ((IGRectangle*)iGraphic)->setEnclosingRect( rc );
          iGraphic->drawOn( gc );
          previousPt = point;
        }
        break;
        case ellipse:
        {
          iGraphic->drawOn( gc );
          IPoint centerPt(((IGEllipse*)iGraphic)->enclosingRect().center());

          ((IGEllipse*)iGraphic)->setEnclosingRect(
            ((IGEllipse*)iGraphic)->enclosingRect().sizeTo( IPair(
                                                  abs(2*(point.x() - centerPt.x())),
                                                  abs(2*(point.y() - centerPt.y()))))
                                               .centerAt( centerPt ));
          iGraphic->drawOn( gc );
        }
        break;
        case polyline:
        case polygon:
          ((IGPolyline*)iGraphic)->IGPolyline::drawOn( gc );
          ((IGPolyline*)iGraphic)->setPoint(
                           ((IGPolyline*)iGraphic)->numberOfPoints()-1, point );
          ((IGPolyline*)iGraphic)->IGPolyline::drawOn( gc );
        break;
        case arc:
          if (drawState() != waitingForInput)
          {
            if ( pointCount == 2 )
            {
              IGLine tempLine( ((IG3PointArc*)iGraphic)->startingPoint(), previousPt );
              tempLine.drawOn( gc );
              tempLine.setEndingPoint( point );
              tempLine.drawOn( gc );
              previousPt = point;
            } else if ( pointCount == 3 ) {
              iGraphic->drawOn( gc );
              ((IG3PointArc*)iGraphic)->setEndingPoint( point );
              iGraphic->drawOn( gc );
            }
          } /* endif */
        break;
        case pie:
        case chord:
          if (drawState() != waitingForInput)
          {
            if ( pointCount == 2 )
            {
              IGLine tempLine( ((IGPie*)iGraphic)->enclosingRect().center(), previousPt );
              tempLine.drawOn( gc );
              tempLine.setEndingPoint( point );
              tempLine.drawOn( gc );
              previousPt = point;
            } else if ( pointCount == 3 ) {
              iGraphic->drawOn( gc );
              double sweep(angleFromPoints( ((IGPie*)iGraphic)->enclosingRect().center(), point ));
              if ( sweep < ((IGPie*)iGraphic)->startAngle() )
                ((IGPie*)iGraphic)->setSweepAngle( 360.0 -
                                    ( ((IGPie*)iGraphic)->startAngle() - sweep ));
              else
                ((IGPie*)iGraphic)->setSweepAngle( sweep -
                                                ((IGPie*)iGraphic)->startAngle());
              iGraphic->drawOn( gc );
            }
          } /* endif */
        break;
      } /* endswitch */
    }
  }
  else
  {
    moveRect.drawOn( gc );
    moveRect.translateBy( point - previousPt );
    moveRect.drawOn( gc );
    previousPt = point;
  }
  return false;
}

/*------------------------------------------------------------------------------
| DrawingArea::button1Up                                                       |
|                                                                              |
| Handle button 1 up events.  This indicates a data points final location.     |
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::button1Up( const IPoint& point )
{
  IGraphicContext gc(this->handle());
  gc.setGraphicBundle(animateBundle);

  if ( drawState() == drawing )
  {
    switch (currentObj)
    {
      case pointer:
      break;
      case line:
        ((IGLine*)iGraphic)->setEndingPoint( point );
        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
      break;
      case freeHand:
        ((IGPolygon*)iGraphic)->addPoint( point );
        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
      break;
      case rectangle:
      {
        IRectangle rc(((IGRectangle*)iGraphic)->enclosingRect());
        rc.sizeTo( rc.size() + point - previousPt );
        ((IGRectangle*)iGraphic)->setEnclosingRect( rc );
        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
      }
      break;
      case ellipse:
      {
        IPoint centerPt(((IGEllipse*)iGraphic)->enclosingRect().center());

        ((IGEllipse*)iGraphic)->setEnclosingRect(
          ((IGEllipse*)iGraphic)->enclosingRect().sizeTo( IPair(
                                                abs(2*(point.x() - centerPt.x())),
                                                abs(2*(point.y() - centerPt.y()))))
                                             .centerAt( centerPt ));

        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
      }
      break;
      case polyline:
      case polygon:
        previousPt = point;
        ((IGPolyline*)iGraphic)->setPoint(
                         ((IGPolyline*)iGraphic)->numberOfPoints()-1, point );
        setDrawState( waitingForInput );
      break;
      case arc:
      if ( pointCount == 2 )
      {
        ((IG3PointArc*)iGraphic)->setIntermediatePoint( point );
        gc.setMixMode( IGraphicBundle::bitExclusiveOr ).setPenColor( IColor::white );
        IGLine tempLine( ((IG3PointArc*)iGraphic)->startingPoint(), previousPt );
        tempLine.drawOn( gc );
        tempLine.setEndingPoint( point );
        tempLine.drawOn( gc );
        setDrawState( waitingForInput );
      }
      else if ( pointCount == 3 )
      {
        gc.setMixMode( IGraphicBundle::bitExclusiveOr ).setPenColor( IColor::white );
        iGraphic->drawOn( gc );
        ((IG3PointArc*)iGraphic)->setEndingPoint( point );
        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
        pointCount = 0;
      }
      break;
      case pie:
      case chord:
      if ( pointCount == 2 )
      {
        gc.setMixMode( IGraphicBundle::bitExclusiveOr ).setPenColor( IColor::white );
        setDrawState( waitingForInput );
        IPoint centerPt(((IGPie*)iGraphic)->enclosingRect().center());

        ((IGPie*)iGraphic)->setStartAngle( angleFromPoints( centerPt, point ));

        unsigned a(abs(centerPt.x()) - abs(point.x()));
        unsigned b(abs(centerPt.y()) - abs(point.y()));

        unsigned long radius((unsigned long)sqrt(a*a + b*b));

        ((IGPie*)iGraphic)->setEnclosingRect(
           ((IGPie*)iGraphic)->enclosingRect().expandBy(radius));
      }
      else if ( pointCount == 3 )
      {
        gc.setMixMode( IGraphicBundle::bitExclusiveOr ).setPenColor( IColor::white );
        iGraphic->drawOn( gc );

        double sweep(angleFromPoints(
                              ((IGPie*)iGraphic)->enclosingRect().center(), point ));

        if ( sweep < ((IGPie*)iGraphic)->startAngle() )
          ((IGPie*)iGraphic)->setSweepAngle( 360.0 -
                                ( ((IGPie*)iGraphic)->startAngle() - sweep ));
        else
          ((IGPie*)iGraphic)->setSweepAngle( sweep -
                                            ((IGPie*)iGraphic)->startAngle());

        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
        pointCount = 0;
      }
      break;
      case text:
      {
        IGString* text = new IGString( currentFont().name(), point, currentFont());
        text->setGraphicBundle( currentBundle );
        text->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *text );
      }
      break;
      case bitmap:
      {
        IGBitmap *bmp;
        if (bitmapFileName().length() > 0)
        {
          bmp = new IGBitmap(bitmapFileName());
        }
        else
          bmp = new IGBitmap(BMP_EAGLE);
        bmp->moveTo( point );
        bmp->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *bmp );
      }
      break;
    } /* endswitch */
  } /* endif */
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::button2Up                                                       |
|                                                                              |
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::button2Up( const IPoint& point )
{
  IGraphicContext gc(this->handle());
  gc.setGraphicBundle(animateBundle);

  if (moveGraphic)
  {
    moveRect.translateBy( point - previousPt );
    moveRect.drawOn( gc );
    moveRect.resetTransformMatrix();
    this->refresh( moveRect.boundingRect(gc).expandBy(1) );
    moveGraphic->translateBy( point - startingPt );
    moveGraphic->drawOn( gc );
    moveGraphic = 0;
    capturePointer(false);
  }
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::button1DoubleClick                                              |
|                                                                              |
| Handle button 1 up double click events.  In the case of polyline and polygon |
| a double click indicates the user has finished adding data points to the     |
| object.                                                                      |
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::button1DoubleClick( const IPoint& point )
{
  IGraphicContext gc(this->handle());
  gc.setGraphicBundle(animateBundle);

  if (drawState() == waitingForInput )
  {
    switch (currentObj)
    {
      case polyline:
      {
        iGraphic->drawOn( gc );
        unsigned lastPt = ((IGPolyline*)iGraphic)->numberOfPoints()-1;
        ((IGPolyline*)iGraphic)->removePoint( lastPt );
        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
      }
      break;
      case polygon:
      {
        ((IGPolyline*)iGraphic)->IGPolyline::drawOn( gc );
        unsigned lastPt = ((IGPolygon*)iGraphic)->numberOfPoints()-1;
        ((IGPolygon*)iGraphic)->removePoint( lastPt );
        iGraphic->setGraphicBundle( currentBundle );
        iGraphic->drawOn( gc );
        setDrawState( notDrawing );
        graphicList()->addAsLast( *iGraphic );
      }
      break;
    }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::setDrawState                                                    |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::setDrawState( const DrawState newState )
{
  dState = newState;
  if (dState == drawing)
  {
    if (!hasPointerCaptured())
      capturePointer();
  }
  else if (dState == notDrawing)
    capturePointer(false);
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::setFont                                                         |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
DrawingArea& DrawingArea::setCurrentFont( const IFont& font )
{
  currentfont = font;
  return *this;
}

/*------------------------------------------------------------------------------
| DrawingArea::font                                                            |
|                                                                              |
|                                                                              |
------------------------------------------------------------------------------*/
IFont DrawingArea::currentFont( ) const
{
  return currentfont;
}
