/******************************************************************************
* .FILE:         2d-menu.cpp                                                  *
*                                                                             *
* .DESCRIPTION:  2D Graphics Drawing Program :    Menu Class Implementation   *
*                                                                             *
* .CLASSES:      MainWindow                                                   *
*                                                                             *
* .COPYRIGHT:                                                                 *
*    Licensed Material - Program-Property of IBM                              *
*    (C) Copyright IBM Corp. 1992, 1995 - All Rights Reserved                 *
*                                                                             *
* .DISCLAIMER:                                                                *
*   The following [enclosed] code is sample code created by IBM               *
*   Corporation.  This sample code is not part of any standard IBM product    *
*   and is provided to you solely for the purpose of assisting you in the     *
*   development of your applications.  The code is provided 'AS IS',          *
*   without warranty of any kind.  IBM shall not be liable for any damages    *
*   arising out of your use of the sample code, even if they have been        *
*   advised of the possibility of such damages.                               *
*                                                                             *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT THE SOURCE          *
*                                                                             *
******************************************************************************/
#ifndef _IBASE_                         //Make sure ibase.hpp is included
  #include <ibase.hpp>                  //  since that is where IC_<environ>
#endif                                  //  is defined.
#include <igbitmap.hpp>
#include <ifontdlg.hpp>
#include <ifiledlg.hpp>
#include <igrect.hpp>
#include <igline.hpp>
#include <ireslib.hpp>
#include "2ddraw.hpp"

/*------------------------------------------------------------------------------
| MainWindow::command                                                          |
|                                                                              |
| Command handler for button bar and menu bar.                                 |
------------------------------------------------------------------------------*/
IBase::Boolean MainWindow::command( ICommandEvent& event )
{
  Boolean fProcessed = false;
  switch (event.commandId())
  {
    case  ID_BKCOL_WHT:
    case  ID_BKCOL_BLK:
    case  ID_BKCOL_DARKGRA:
    case  ID_BKCOL_PALEGRA:
    case  ID_BKCOL_BRO:
    case  ID_BKCOL_DARKRED:
    case  ID_BKCOL_RED:
    case  ID_BKCOL_YEL:
    case  ID_BKCOL_DARKGRE:
    case  ID_BKCOL_GRE:
    case  ID_BKCOL_DARKCYA:
    case  ID_BKCOL_CYA:
    case  ID_BKCOL_DARKBLU:
    case  ID_BKCOL_BLU:
    case  ID_BKCOL_DARKPIN:
    case  ID_BKCOL_PIN:
      menuBar.uncheckItem(lastPenColorId)
             .checkItem(lastPenColorId = event.commandId());
      drawingArea.setBackgroundColor(backColorFromId(event.commandId()))
                 .refresh();

      fProcessed = true;
    break;
    case PALLET_NORM:
    case PALLET_LINE:
    case PALLET_DRAW:
    case PALLET_RECTANGLE:
    case PALLET_ELLIPSE:
    case PALLET_POLYLINE:
    case PALLET_POLYGON:
    case PALLET_ARC:
    case PALLET_PIE:
    case PALLET_CHORD:
    case PALLET_TEXT:
    case PALLET_BITMAP:
      drawingArea.setDrawObject((DrawingArea::DrawObject)event.commandId());
      flyOver.setHelpText( drawingArea.handle(),
                           IResourceId(0),
                           IResourceId(event.commandId() + LONG_OFFSET ));
      fProcessed = true;
    break;
    case  ID_COL_WHT:
    case  ID_COL_BLK:
    case  ID_COL_DARKGRA:
    case  ID_COL_PALEGRA:
    case  ID_COL_BRO:
    case  ID_COL_DARKRED:
    case  ID_COL_RED:
    case  ID_COL_YEL:
    case  ID_COL_DARKGRE:
    case  ID_COL_GRE:
    case  ID_COL_DARKCYA:
    case  ID_COL_CYA:
    case  ID_COL_DARKBLU:
    case  ID_COL_BLU:
    case  ID_COL_DARKPIN:
    case  ID_COL_PIN:
      menuBar.uncheckItem(lastPenColorId)
             .checkItem(lastPenColorId = event.commandId());
      drawingArea.graphicBundle().setPenColor(penColorFromId(lastPenColorId));
      fProcessed = true;
    break;
    case ID_FLCOL_WHT:
    case ID_FLCOL_BLK:
    case ID_FLCOL_DARKGRA:
    case ID_FLCOL_PALEGRA:
    case ID_FLCOL_BRO:
    case ID_FLCOL_DARKRED:
    case ID_FLCOL_RED:
    case ID_FLCOL_YEL:
    case ID_FLCOL_DARKGRE:
    case ID_FLCOL_GRE:
    case ID_FLCOL_DARKCYA:
    case ID_FLCOL_CYA:
    case ID_FLCOL_DARKBLU:
    case ID_FLCOL_BLU:
    case ID_FLCOL_DARKPIN:
    case ID_FLCOL_PIN:
      menuBar.uncheckItem(lastFillColorId)
             .checkItem(lastFillColorId = event.commandId());
      drawingArea.graphicBundle().setFillColor(fillColorFromId(lastFillColorId));
      fProcessed = true;
    break;
    case ID_PENPATTERN_SOLID:
    case ID_PENPATTERN_NOSHADE:
    case ID_PENPATTERN_DIAG4:
    case ID_PENPATTERN_DIAG3:
    case ID_PENPATTERN_DIAG2:
    case ID_PENPATTERN_DIAG1:
    case ID_PENPATTERN_VERT:
    case ID_PENPATTERN_HORIZ:
    case ID_PENPATTERN_DENSE8:
    case ID_PENPATTERN_DENSE7:
    case ID_PENPATTERN_DENSE6:
    case ID_PENPATTERN_DENSE5:
    case ID_PENPATTERN_DENSE4:
    case ID_PENPATTERN_DENSE3:
    case ID_PENPATTERN_DENSE2:
    case ID_PENPATTERN_DENSE1:
      menuBar.uncheckItem(lastPenPatternId)
             .checkItem(lastPenPatternId = event.commandId());
      drawingArea.graphicBundle().setPenPattern(patternFromId(lastPenPatternId));
      fProcessed = true;
    break;
    case ID_FILLPATTERN_SOLID:
    case ID_FILLPATTERN_NOSHADE:
    case ID_FILLPATTERN_DIAG4:
    case ID_FILLPATTERN_DIAG3:
    case ID_FILLPATTERN_DIAG2:
    case ID_FILLPATTERN_DIAG1:
    case ID_FILLPATTERN_VERT:
    case ID_FILLPATTERN_HORIZ:
    case ID_FILLPATTERN_DENSE8:
    case ID_FILLPATTERN_DENSE7:
    case ID_FILLPATTERN_DENSE6:
    case ID_FILLPATTERN_DENSE5:
    case ID_FILLPATTERN_DENSE4:
    case ID_FILLPATTERN_DENSE3:
    case ID_FILLPATTERN_DENSE2:
    case ID_FILLPATTERN_DENSE1:
      menuBar.uncheckItem(lastFillPatternId)
             .checkItem(lastFillPatternId = event.commandId());
      drawingArea.graphicBundle().setFillPattern(patternFromId(lastFillPatternId));
      fProcessed = true;
    break;
    case ID_PENTYPE_ALTERNATE:
    case ID_PENTYPE_SOLID:
    case ID_PENTYPE_DASHDOUBLEDOT:
    case ID_PENTYPE_LONGDASH:
    case ID_PENTYPE_DOUBLEDOT:
    case ID_PENTYPE_DASHDOT:
    case ID_PENTYPE_SHORTDASH:
    case ID_PENTYPE_DOT:
      menuBar.uncheckItem(lastPenTypeId)
             .checkItem(lastPenTypeId = event.commandId());
      drawingArea.graphicBundle().setPenType(penTypeFromId(lastPenTypeId));
      fProcessed = true;
    break;
    case ID_PENWIDTH_1:
    case ID_PENWIDTH_2:
    case ID_PENWIDTH_3:
    case ID_PENWIDTH_4:
      menuBar.uncheckItem(lastPenWidthId)
             .checkItem(lastPenWidthId = event.commandId());
      drawingArea.graphicBundle().setPenWidth(penWidthFromId(lastPenWidthId));
    break;
    case ID_SETFONT:
    {
      IFont curFont(drawingArea.currentFont());
      curFont.useVectorOnly( );
      IFontDialog::Settings fsettings(&curFont);
      fsettings.setTitle(TITLE_DIALOG);    //Set Open Dialog Title from Resource  .

      IFontDialog fontd(                    //Create Font Open Dialiog             .
        (IWindow*)desktopWindow(),          //  Parent is Desktop                  .
        (IWindow*)this,
        IFontDialog::defaultStyle(),        //  Set default Style                  .
        fsettings);                         //  settings                           .
      if (fontd.pressedOK())                //Check if ok from Font open dialog    .
      {                                     //                                     .
        drawingArea.setCurrentFont( curFont );
      } /* endif */                         //                                     .
      fProcessed = true;
    }
    break;
    case ID_SETBITMAP:
    {
      IResourceLibrary reslib;
      IFileDialog::Settings fsettings;      //                                     .
      fsettings.setTitle(STR_LOAD_IMG);//Set Open Dialog Title from Resource  .
      fsettings.setFileName(reslib.loadString(STR_BMP_EXT)); //Set FileNames to *.bmp               .

      IFileDialog fd(                       //Create File Open Dialiog             .
        (IWindow*)desktopWindow(),          //  Parent is Desktop                  .
        (IWindow*)this, fsettings);         //  Owner is me with settings          .
      if (fd.pressedOK())                   //Check if ok from file open dialog    .
      {
        drawingArea.setBitmapFileName(fd.fileName());
      }
    }
    break;
    case ID_FILL:
      menuBar.uncheckItem(lastDrawOperationId)
             .checkItem(lastDrawOperationId = event.commandId());
      drawingArea.graphicBundle().setDrawOperation(IGraphicBundle::fill);
      break;
    case ID_FRAME:
      menuBar.uncheckItem(lastDrawOperationId)
             .checkItem(lastDrawOperationId = event.commandId());
      drawingArea.graphicBundle().setDrawOperation(IGraphicBundle::frame);
      break;
    case ID_FILLANDFRAME:
      menuBar.uncheckItem(lastDrawOperationId)
             .checkItem(lastDrawOperationId = event.commandId());
      drawingArea.graphicBundle().setDrawOperation(IGraphicBundle::fillAndFrame);
      break;
    break;

  } /* endswitch */

  return fProcessed;
}


/*------------------------------------------------------------------------------
| MainWindow::setSize                                                          |
|                                                                              |
| Overloaded function from IMenuDrawItemHandler.  Sets the width and height    |
| of the owner draw menu items.                                                |
------------------------------------------------------------------------------*/
IBase::Boolean MainWindow::setSize(IMenuDrawItemEvent& evt, ISize& newSize)
{
  switch (evt.itemId())
  {
    case ID_PENTYPE_ALTERNATE:
    case ID_PENTYPE_SOLID:
    case ID_PENTYPE_DASHDOUBLEDOT:
    case ID_PENTYPE_LONGDASH:
    case ID_PENTYPE_DOUBLEDOT:
    case ID_PENTYPE_DASHDOT:
    case ID_PENTYPE_SHORTDASH:
    case ID_PENTYPE_DOT:

    case ID_PENWIDTH_4:
    case ID_PENWIDTH_3:
    case ID_PENWIDTH_2:
    case ID_PENWIDTH_1:
      newSize.setWidth( 80 ).setHeight( 15 );
    break;
    default:
#ifdef IC_PM
      newSize.setWidth( 10 ).setHeight( 20 );
#else
      newSize.setWidth( 40 ).setHeight( 20 );
#endif
    break;
  }
  return true;
}

/*------------------------------------------------------------------------------
| MainWindow::draw                                                             |
|                                                                              |
| Overloaded function from IMenuDrawItemHandler.  Draws all of the owner       |
| draw menu items.                                                             |
------------------------------------------------------------------------------*/
IBase::Boolean MainWindow::draw(IMenuDrawItemEvent& evt, DrawFlag& flag)
{
  static int checkMarkHeight(0);

  IGraphicContext gc(evt.itemPresSpaceHandle());

  gc.setPenType( IGraphicBundle::invisible );

  if (evt.isSelected())
  {
    gc.setPenColor(IGUIColor(IGUIColor::menuHiliteText))
      .setFillColor(IGUIColor(IGUIColor::menuHiliteBgnd))
      .setBackgroundColor(IGUIColor(IGUIColor::menuHiliteBgnd));
  }
  else
  {
    gc.setPenColor(IGUIColor(IGUIColor::menuText))
      .setFillColor(IGUIColor(IGUIColor::menuBgnd))
      .setBackgroundColor(IGUIColor(IGUIColor::menuBgnd));
  }
  IGRectangle box( evt.itemRect());
  box.drawOn( gc );

  gc.setPenType( IGraphicBundle::solid );

  flag = IMenuDrawItemHandler::drewSelected;

  if (evt.isChecked())
  {
    flag |= IMenuDrawItemHandler::drewChecked;
    switch (evt.itemId())
    {
      default:
      {
        ISystemBitmapHandle temp = ISystemBitmapHandle( ISystemBitmapHandle::menuCheckMark );
        IGBitmap checkMark( temp );
        if (!checkMarkHeight)
          checkMarkHeight = checkMark.size().height();
        checkMark.moveTo( IPoint( evt.itemRect().minXMinY().x(),
                                  evt.itemRect().minXMinY().y() +
                                ( evt.itemRect().height() - checkMarkHeight) /2));
        checkMark.drawOn( gc );
      }
      break;
    } /* endswitch */
  }

  IRectangle rc(IPoint( evt.itemRect().bottomLeft().x()+20,
                        evt.itemRect().bottomLeft().y()+3 ),
                IPoint( evt.itemRect().topRight().x()-3,
                        evt.itemRect().topRight().y()-3 ));
  switch (evt.itemId())
  {
    case ID_FLCOL_WHT:
    case ID_FLCOL_BLK:
    case ID_FLCOL_DARKGRA:
    case ID_FLCOL_PALEGRA:
    case ID_FLCOL_BRO:
    case ID_FLCOL_DARKRED:
    case ID_FLCOL_RED:
    case ID_FLCOL_YEL:
    case ID_FLCOL_DARKGRE:
    case ID_FLCOL_GRE:
    case ID_FLCOL_DARKCYA:
    case ID_FLCOL_CYA:
    case ID_FLCOL_DARKBLU:
    case ID_FLCOL_BLU:
    case ID_FLCOL_DARKPIN:
    case ID_FLCOL_PIN:
      gc.setFillColor(fillColorFromId(evt.itemId()));
      box.setEnclosingRect(rc).drawOn(gc);
    break;
    case ID_BKCOL_WHT:
    case ID_BKCOL_BLK:
    case ID_BKCOL_DARKGRA:
    case ID_BKCOL_PALEGRA:
    case ID_BKCOL_BRO:
    case ID_BKCOL_DARKRED:
    case ID_BKCOL_RED:
    case ID_BKCOL_YEL:
    case ID_BKCOL_DARKGRE:
    case ID_BKCOL_GRE:
    case ID_BKCOL_DARKCYA:
    case ID_BKCOL_CYA:
    case ID_BKCOL_DARKBLU:
    case ID_BKCOL_BLU:
    case ID_BKCOL_DARKPIN:
    case ID_BKCOL_PIN:
    {
      gc.setFillColor(backColorFromId(evt.itemId()));
      box.setEnclosingRect(rc).drawOn(gc);
    }
    break;

    case ID_COL_WHT:
    case ID_COL_BLK:
    case ID_COL_DARKGRA:
    case ID_COL_PALEGRA:
    case ID_COL_BRO:
    case ID_COL_DARKRED:
    case ID_COL_RED:
    case ID_COL_YEL:
    case ID_COL_DARKGRE:
    case ID_COL_GRE:
    case ID_COL_DARKCYA:
    case ID_COL_CYA:
    case ID_COL_DARKBLU:
    case ID_COL_BLU:
    case ID_COL_DARKPIN:
    case ID_COL_PIN:
      gc.setFillColor(penColorFromId(evt.itemId()));
      box.setEnclosingRect(rc).drawOn(gc);
    break;
    case ID_PENPATTERN_SOLID:
    case ID_PENPATTERN_NOSHADE:
    case ID_PENPATTERN_DIAG4:
    case ID_PENPATTERN_DIAG3:
    case ID_PENPATTERN_DIAG2:
    case ID_PENPATTERN_DIAG1:
    case ID_PENPATTERN_VERT:
    case ID_PENPATTERN_HORIZ:
    case ID_PENPATTERN_DENSE8:
    case ID_PENPATTERN_DENSE7:
    case ID_PENPATTERN_DENSE6:
    case ID_PENPATTERN_DENSE5:
    case ID_PENPATTERN_DENSE4:
    case ID_PENPATTERN_DENSE3:
    case ID_PENPATTERN_DENSE2:
    case ID_PENPATTERN_DENSE1:
    {
      gc.setFillColor( IColor::blue )
        .setPenPattern(patternFromId(evt.itemId()))
        .setPenColor( IColor::blue )
        .setPenWidth( rc.height() );
      IGLine line(rc.leftCenter(),
                  IPoint( rc.rightCenter().x()-3,
                          rc.rightCenter().y() ));
      line.drawOn( gc );
    }
    break;
    case ID_FILLPATTERN_SOLID:
    case ID_FILLPATTERN_NOSHADE:
    case ID_FILLPATTERN_DIAG4:
    case ID_FILLPATTERN_DIAG3:
    case ID_FILLPATTERN_DIAG2:
    case ID_FILLPATTERN_DIAG1:
    case ID_FILLPATTERN_VERT:
    case ID_FILLPATTERN_HORIZ:
    case ID_FILLPATTERN_DENSE8:
    case ID_FILLPATTERN_DENSE7:
    case ID_FILLPATTERN_DENSE6:
    case ID_FILLPATTERN_DENSE5:
    case ID_FILLPATTERN_DENSE4:
    case ID_FILLPATTERN_DENSE3:
    case ID_FILLPATTERN_DENSE2:
    case ID_FILLPATTERN_DENSE1:
    {
      gc.setFillColor( IColor::blue )
        .setFillPattern(patternFromId(evt.itemId()))
        .setPenColor( IColor::blue )
        .setPenEndingStyle( IGraphicBundle::flat );
      IGRectangle rect(rc);
      rect.drawOn( gc );
    }
    break;
    case ID_PENTYPE_ALTERNATE:
    case ID_PENTYPE_SOLID:
    case ID_PENTYPE_DASHDOUBLEDOT:
    case ID_PENTYPE_LONGDASH:
    case ID_PENTYPE_DOUBLEDOT:
    case ID_PENTYPE_DASHDOT:
    case ID_PENTYPE_SHORTDASH:
    case ID_PENTYPE_DOT:
      gc.setPenType( penTypeFromId( evt.itemId()));
      {
        gc.setPenColor( IColor::blue );
        IGLine line( IPoint( evt.itemRect().bottomLeft().x()+20,
                             evt.itemRect().leftCenter().y() ),
                     IPoint( evt.itemRect().topRight().x()-3,
                             evt.itemRect().leftCenter().y() ));
        line.drawOn( gc );
      }
    break;
    case ID_PENWIDTH_4:
    case ID_PENWIDTH_3:
    case ID_PENWIDTH_2:
    case ID_PENWIDTH_1:
      gc.setPenType( IGraphicBundle::solid )
        .setPenColor( IColor::blue )
        .setPenWidth( penWidthFromId( evt.itemId()));
      {
        IGLine line( IPoint( evt.itemRect().bottomLeft().x()+20,
                             evt.itemRect().leftCenter().y() ),
                     IPoint( evt.itemRect().topRight().x()-3,
                             evt.itemRect().leftCenter().y() ));
        line.drawOn( gc );
      }
    break;
  }
  return true;
}

/*------------------------------------------------------------------------------
| MainWindow::backColorFromId                                                  |
|                                                                              |
| Static function which maps menu id values to their corresponding IColor      |
| equivalents.                                                                 |
------------------------------------------------------------------------------*/
IColor MainWindow::backColorFromId( unsigned long Identifier )
{
  IColor::Color col;
  switch (Identifier)
  {
    case ID_BKCOL_WHT:
      col = IColor::white;
    break;
    case ID_BKCOL_BLK:
      col = IColor::black;
    break;
    case ID_BKCOL_DARKGRA:
      col = IColor::darkGray;
    break;
    case ID_BKCOL_PALEGRA:
      col = IColor::paleGray;
    break;
    case ID_BKCOL_BRO:
      col = IColor::brown;
    break;
    case ID_BKCOL_DARKRED:
      col = IColor::darkRed;
    break;
    case ID_BKCOL_RED:
      col = IColor::red;
    break;
    case ID_BKCOL_YEL:
      col = IColor::yellow;
    break;
    case ID_BKCOL_DARKGRE:
      col = IColor::darkGreen;
    break;
    case ID_BKCOL_GRE:
      col = IColor::green;
    break;
    case ID_BKCOL_DARKCYA:
      col = IColor::darkCyan;
    break;
    case ID_BKCOL_CYA:
      col = IColor::cyan;
    break;
    case ID_BKCOL_DARKBLU:
      col = IColor::darkBlue;
    break;
    case ID_BKCOL_BLU:
      col = IColor::blue;
    break;
    case ID_BKCOL_DARKPIN:
      col = IColor::darkPink;
    break;
    case ID_BKCOL_PIN:
      col = IColor::pink;
    break;
  } /* endswitch */
  return IColor( col );
}

/*------------------------------------------------------------------------------
| MainWindow::fillColorFromId                                                  |
|                                                                              |
| Static function which maps menu id values to their corresponding IColor      |
| equivalents.                                                                 |
------------------------------------------------------------------------------*/
IColor MainWindow::fillColorFromId( unsigned long Identifier )
{
  IColor::Color col;
  switch (Identifier)
  {
    case ID_FLCOL_WHT:
      col = IColor::white;
    break;
    case ID_FLCOL_BLK:
      col = IColor::black;
    break;
    case ID_FLCOL_DARKGRA:
      col = IColor::darkGray;
    break;
    case ID_FLCOL_PALEGRA:
      col = IColor::paleGray;
    break;
    case ID_FLCOL_BRO:
      col = IColor::brown;
    break;
    case ID_FLCOL_DARKRED:
      col = IColor::darkRed;
    break;
    case ID_FLCOL_RED:
      col = IColor::red;
    break;
    case ID_FLCOL_YEL:
      col = IColor::yellow;
    break;
    case ID_FLCOL_DARKGRE:
      col = IColor::darkGreen;
    break;
    case ID_FLCOL_GRE:
      col = IColor::green;
    break;
    case ID_FLCOL_DARKCYA:
      col = IColor::darkCyan;
    break;
    case ID_FLCOL_CYA:
      col = IColor::cyan;
    break;
    case ID_FLCOL_DARKBLU:
      col = IColor::darkBlue;
    break;
    case ID_FLCOL_BLU:
      col = IColor::blue;
    break;
    case ID_FLCOL_DARKPIN:
      col = IColor::darkPink;
    break;
    case ID_FLCOL_PIN:
      col = IColor::pink;
    break;
  } /* endswitch */
  return IColor( col );
}

/*------------------------------------------------------------------------------
| MainWindow::penColorFromId                                                   |
|                                                                              |
| Static function which maps menu id values to their corresponding IColor      |
| equivalents.                                                                 |
------------------------------------------------------------------------------*/
IColor MainWindow::penColorFromId( unsigned long Identifier )
{
  IColor::Color col;
  switch (Identifier)
  {
    case ID_COL_WHT:
      col = IColor::white;
    break;
    case ID_COL_BLK:
      col = IColor::black;
    break;
    case ID_COL_DARKGRA:
      col = IColor::darkGray;
    break;
    case ID_COL_PALEGRA:
      col = IColor::paleGray;
    break;
    case ID_COL_BRO:
      col = IColor::brown;
    break;
    case ID_COL_DARKRED:
      col = IColor::darkRed;
    break;
    case ID_COL_RED:
      col = IColor::red;
    break;
    case ID_COL_YEL:
      col = IColor::yellow;
    break;
    case ID_COL_DARKGRE:
      col = IColor::darkGreen;
    break;
    case ID_COL_GRE:
      col = IColor::green;
    break;
    case ID_COL_DARKCYA:
      col = IColor::darkCyan;
    break;
    case ID_COL_CYA:
      col = IColor::cyan;
    break;
    case ID_COL_DARKBLU:
      col = IColor::darkBlue;
    break;
    case ID_COL_BLU:
      col = IColor::blue;
    break;
    case ID_COL_DARKPIN:
      col = IColor::darkPink;
    break;
    case ID_COL_PIN:
      col = IColor::pink;
    break;
  } /* endswitch */
  return IColor( col );
}


/*------------------------------------------------------------------------------
| MainWindow::patternFromId                                                    |
|                                                                              |
| Static function which maps menu id values to their corresponding pattern     |
| equivalents.                                                                 |
------------------------------------------------------------------------------*/
unsigned long MainWindow::patternFromId( unsigned long Identifier )
{
  unsigned long pattern;
  switch (Identifier)
  {
    case ID_PENPATTERN_SOLID:
    case ID_FILLPATTERN_SOLID:
      pattern = IGraphicBundle::filled;
    break;
    case ID_PENPATTERN_NOSHADE:
    case ID_FILLPATTERN_NOSHADE:
      pattern = IGraphicBundle::hollow;
    break;
    case ID_PENPATTERN_DIAG4:
    case ID_FILLPATTERN_DIAG4:
      pattern = IGraphicBundle::backwardDiag2;
    break;
    case ID_PENPATTERN_DIAG3:
    case ID_FILLPATTERN_DIAG3:
      pattern = IGraphicBundle::backwardDiag1;
    break;
    case ID_PENPATTERN_DIAG2:
    case ID_FILLPATTERN_DIAG2:
      pattern = IGraphicBundle::forewardDiag2;
    break;
    case ID_PENPATTERN_DIAG1:
    case ID_FILLPATTERN_DIAG1:
      pattern = IGraphicBundle::forewardDiag1;
    break;
    case ID_PENPATTERN_VERT:
    case ID_FILLPATTERN_VERT:
      pattern = IGraphicBundle::vertical;
    break;
    case ID_PENPATTERN_HORIZ:
    case ID_FILLPATTERN_HORIZ:
      pattern = IGraphicBundle::horizontal;
    break;
    case ID_PENPATTERN_DENSE8:
    case ID_FILLPATTERN_DENSE8:
      pattern = IGraphicBundle::dense8;
    break;
    case ID_PENPATTERN_DENSE7:
    case ID_FILLPATTERN_DENSE7:
      pattern = IGraphicBundle::dense7;
    break;
    case ID_PENPATTERN_DENSE6:
    case ID_FILLPATTERN_DENSE6:
      pattern = IGraphicBundle::dense6;
    break;
    case ID_PENPATTERN_DENSE5:
    case ID_FILLPATTERN_DENSE5:
      pattern = IGraphicBundle::dense5;
    break;
    case ID_PENPATTERN_DENSE4:
    case ID_FILLPATTERN_DENSE4:
      pattern = IGraphicBundle::dense4;
    break;
    case ID_PENPATTERN_DENSE3:
    case ID_FILLPATTERN_DENSE3:
      pattern = IGraphicBundle::dense3;
    break;
    case ID_PENPATTERN_DENSE2:
    case ID_FILLPATTERN_DENSE2:
      pattern = IGraphicBundle::dense2;
    break;
    case ID_PENPATTERN_DENSE1:
    case ID_FILLPATTERN_DENSE1:
      pattern = IGraphicBundle::dense1;
    break;
  }
  return pattern;
}

/*------------------------------------------------------------------------------
| MainWindow::patterFromId                                                     |
|                                                                              |
| Static function which maps menu id values to their corresponding pen type    |
| equivalents.                                                                 |
------------------------------------------------------------------------------*/
IGraphicBundle::PenType MainWindow::penTypeFromId( unsigned long Identifier )
{
  IGraphicBundle::PenType penType;
  switch (Identifier)
  {
    case ID_PENTYPE_ALTERNATE:
      penType = IGraphicBundle::alternate;
    break;
    case ID_PENTYPE_SOLID:
      penType = IGraphicBundle::solid;
    break;
    case ID_PENTYPE_DASHDOUBLEDOT:
      penType = IGraphicBundle::dashDoubleDot;
    break;
    case ID_PENTYPE_LONGDASH:
      penType = IGraphicBundle::longDash;
    break;
    case ID_PENTYPE_DOUBLEDOT:
      penType = IGraphicBundle::doubleDot;
    break;
    case ID_PENTYPE_DASHDOT:
      penType = IGraphicBundle::dashDot;
    break;
    case ID_PENTYPE_SHORTDASH:
      penType = IGraphicBundle::shortDash;
    break;
    case ID_PENTYPE_DOT:
      penType = IGraphicBundle::dot;
    break;
  }
  return penType;
}

/*------------------------------------------------------------------------------
| MainWindow::patterFromId                                                     |
|                                                                              |
| Static function which maps menu id values to their corresponding pen width   |
| equivalents.                                                                 |
------------------------------------------------------------------------------*/
unsigned long MainWindow::penWidthFromId( unsigned long Identifier )
{
  unsigned long width;
  switch (Identifier)
  {
    case ID_PENWIDTH_1:
      width = 1;
    break;
    case ID_PENWIDTH_2:
      width = 2;
    break;
    case ID_PENWIDTH_3:
      width = 4;
    break;
    case ID_PENWIDTH_4:
      width = 6;
    break;
  } /* endswitch */
  return width;
}
