///////////////////////////////////////////////////////////////////////////////
//                                              SAMPLE CODE
//
// FileName: ACDFVw2.cpp
//
// ClassName: ACompDocFwkView
//
// Description: Compound Document Framework (SimpleServer) View
//              This sample is to illustrates the use of notification and the use of
//              keyboard, mouse and command handlers in the server application. When
//              a notification is recieved from the model, if the handleNotification method
//              was NOT overridden the drawContents would have been called and the notification ID
//              not checked.
//              In this case the handleNotification method was overridden as a result different data
//              is set depending on the NotificationId
//              The view contains 2 entry fields to allow the user to change the model data
//              by entering the new data and hitting either newline or enter. Using the keyboard
//              handler to trap these events.
//              Doubleclick on either of the entry fields to change the string in MyObject
//              in the model, using the mouse handler trap this event.
//              The pushbutton and the menu option perform the same function which is
//              to update both the strings in the model.
///////////////////////////////////////////////////////////////////////////////
#include "ACDFMdl2.hpp"
#include "ACDFVw2.hpp"
#include "ACDFRes2.h"   // Resource id file

#include <iframe.hpp>   // this is need to set the icon
#include <igstring.hpp>
#include <igrect.hpp>
#include <igrafctx.hpp>
#include <imsgbox.hpp>
#include <assert.h>
#include <iguibndl.hpp> // must be include after windows.h


ACompDocFwkView::ACompDocFwkView(IGUIBundle& bundle ) :
    IView(bundle),
    fButton(ID_BUTTON,this,this),
    fDataStaticField1(ID_STATIC1, this, this),
    fDataEntryField1 (ID_ENTRY1, this, this),
    fDataStaticField2(ID_STATIC2, this, this),
    fDataEntryField2 (ID_ENTRY2, this, this),
    fMyObjectField (ID_OBJECT, this, this),
    fButtonHandler( *this, handleButtonPressed),
    fKeyboardHandler(*this, handleVirtualKeyPress),
    fMouseHandler( *this, handleMouseClicked)
{   IFUNCTRACE_DEVELOP();
    bundle.frameWindow().setIcon(IC_ACDF2);   // Set the window icon
}

ACompDocFwkView::~ACompDocFwkView()
// Destructor - stop all the handlers
{   IFUNCTRACE_DEVELOP();
    fButtonHandler.stopHandlingEventsFor( this );
    fKeyboardHandler.stopHandlingEventsFor(&fDataEntryField1 );
    fKeyboardHandler.stopHandlingEventsFor(&fDataEntryField2 );

    fMouseHandler.stopHandlingEventsFor( &fDataEntryField1 );
    fMouseHandler.stopHandlingEventsFor( &fDataEntryField2 );
}

ACompDocFwkModel* ACompDocFwkView::getModelPointer() const
// Get the pointer to the model
{   IFUNCTRACE_DEVELOP();
    ACompDocFwkModel* theModel = NULL;
    ::dynamicCastTo( theModel,model() ); // Downcasts the model pointer
    return theModel;
}

void ACompDocFwkView::initialize()
// Intialize code e.g build menus
{   IFUNCTRACE_DEVELOP();

    IView::initialize();
    bundle().objectView().setBackgroundColor(IColor(64,128,128)); // Set the color to green

    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    fButton.setText(STR_UPDATE);

    fDataStaticField1.setText(fOurModel->getString1());
    fDataStaticField2.setText(fOurModel->getString2());
    fMyObjectField.setText(fOurModel->getPointerMyString());

    // Start handling events for the keyboard, mouse and commands(buttons)
    fMouseHandler.handleEventsFor(&fDataEntryField1);
    fMouseHandler.handleEventsFor(&fDataEntryField2);

    fKeyboardHandler.setVirtualKeyMemberFn  (handleVirtualKeyPress);
    fKeyboardHandler.handleEventsFor(&fDataEntryField1);
    fKeyboardHandler.handleEventsFor(&fDataEntryField2);

    fButtonHandler.handleEventsFor( this );
}


ACompDocFwkView& ACompDocFwkView::sizeTo(const ISize& newSize)
{   IFUNCTRACE_DEVELOP();
    IView::sizeTo(newSize);
    SizeFieldsToScreen(newSize);
    return *this;
}

ACompDocFwkView& ACompDocFwkView::moveSizeTo(const IRectangle& newRect)
{  IFUNCTRACE_DEVELOP();
   IView::moveSizeTo(newRect);
   SizeFieldsToScreen(newRect.size());
   return *this;
}

void ACompDocFwkView::SizeFieldsToScreen(const ISize& myWindowSize)
// Resize the fields depending on the screen size
{   IFUNCTRACE_DEVELOP();

    int i = myWindowSize.coord1() / 2;

    fDataStaticField1.moveSizeTo(IRectangle(IPoint((i + 5),10), IPoint((i + i) ,40)));
    fDataEntryField1.moveSizeTo(IRectangle(IPoint(0,10), IPoint((i - 5) ,40)));

    fDataStaticField2.moveSizeTo(IRectangle(IPoint((i + 5),50), IPoint((i + i) ,80)));
    fDataEntryField2.moveSizeTo(IRectangle(IPoint(0,50), IPoint((i - 5) ,80)));

    fMyObjectField.moveSizeTo(IRectangle(IPoint((i - 60), 90),
                                IPoint((i  + 60), 120)));

    fButton.moveSizeTo(IRectangle(IPoint((i - 40), (myWindowSize.coord2() - 30)),
                                IPoint((i  + 40), myWindowSize.coord2())));
}

void ACompDocFwkView::drawContents( IPresSpaceHandle& hdl,
                  const IRectangle& invalidArea,
                  Boolean metaFile )
//  We can override "drawContents" if we want to optimize our drawing
//  or do things differently when rendering to a meta-file (where controls
//  won't be displayed).
{   IFUNCTRACE_DEVELOP();

    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    if (metaFile)
    {
    // Place the embedded image code here !!!!
        IString str1;
        IString str2;
        IString ptrstr;
        IGraphicContext ctxt ( hdl );
        ctxt.setFillColor(IColor::white);
        IGString myString1(fOurModel->getString1(),IPoint(100,20));
        IGString myString2(fOurModel->getString2(),IPoint(100,50));
        IGString myPointer(fOurModel->getPointerMyString(),IPoint(100,80));
        IGRectangle myRect(IRectangle(IPoint(10,10),IPoint(300,100)));

        if  (fOurModel->getString1().length() > 0)
            str1 = IApplication::current().userResourceLibrary().loadString(STR_STRING1);
        else
            str1 = IApplication::current().userResourceLibrary().loadString(STR_STRING1_EMPTY);

        if  (fOurModel->getString2().length() > 0)
            str2 = IApplication::current().userResourceLibrary().loadString(STR_STRING2);
        else
            str2 = IApplication::current().userResourceLibrary().loadString(STR_STRING2_EMPTY);

        if  (fOurModel->getPointerMyString().length() > 0)
            ptrstr = IApplication::current().userResourceLibrary().loadString(STR_POINTER);
        else
            ptrstr = IApplication::current().userResourceLibrary().loadString(STR_POINTER_EMPTY);

        IGString gStr1(str1,IPoint(20,20));
        IGString gStr2(str2,IPoint(20,50));
        IGString gPtrStr(ptrstr,IPoint(20,80));

        myRect.drawOn(ctxt);
        myString1.drawOn( ctxt );
        myString2.drawOn( ctxt );
        gStr1.drawOn( ctxt );
        gStr2.drawOn( ctxt );
        myPointer.drawOn( ctxt );
        gPtrStr.drawOn( ctxt );
    }

    fDataStaticField1.setText(fOurModel->getString1());
    fDataStaticField2.setText(fOurModel->getString2());
    fMyObjectField.setText(fOurModel->getPointerMyString());
}
void ACompDocFwkView::handleFileNew()
// This is to clear al the fields when file new is selected
{   IFUNCTRACE_DEVELOP();
    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    fDataEntryField1.removeAll();
    fDataEntryField2.removeAll();
}

void ACompDocFwkView::handleNotification( const INotificationEvent& event)
// Handle the notification events
{   IFUNCTRACE_DEVELOP();

    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    if (event.notificationId() == fOurModel->kDataStringChange1 )
    {
        fDataEntryField1.setText(fOurModel->getString1());
        fDataStaticField1.setText(fOurModel->getString1());
    }
    else if (event.notificationId() == fOurModel->kDataStringChange2 )
    {
        fDataEntryField2.setText(fOurModel->getString2());
        fDataStaticField2.setText(fOurModel->getString2());
    }
    else if (event.notificationId() == fOurModel->kDataPointerChange )
    {
        fMyObjectField.setText(fOurModel->getPointerMyString());
    }
    else
    refresh();  // will refresh the screen
}


Boolean ACompDocFwkView::handleButtonPressed(ICommandEvent& event)
// Catch the button pressed and menu commands
{   IFUNCTRACE_DEVELOP();
    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    switch(event.commandId())
    {
        case MI_CHANGE:
        case ID_BUTTON:
        {
            fOurModel->setString1(fDataEntryField1.text());
            fOurModel->setString2(fDataEntryField2.text());
            return true;
        }
    }

    return false;  //no processing was done

}

Boolean ACompDocFwkView::handleVirtualKeyPress( IKeyboardEvent &event )
// Catch the key pressed update model
{   IFUNCTRACE_DEVELOP();

    if (!event.isVirtual())
        return false;

    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    Boolean fboolean = false;
    switch (event.virtualKey())
    {
        case IKeyboardEvent::enter:
        case IKeyboardEvent::newLine:
        {
            IWindow *pWindow = windowWithHandle(event.controlHandle());
            unsigned long int winId = pWindow->id();
            switch (winId)
            {
                case ID_ENTRY1 :
                {
                    fOurModel->setString1(fDataEntryField1.text());
                    fboolean = true;
                    break;
                }
                case ID_ENTRY2 :
                {
                    fOurModel->setString2(fDataEntryField2.text());
                    fboolean = true;
                    break;
                }
            }
            break;
        }
    }
    return fboolean;
}

Boolean ACompDocFwkView::handleMouseClicked(IMouseClickEvent& event)
// Catch the mouse click to update pointer
{   IFUNCTRACE_DEVELOP();

    ACompDocFwkModel* fOurModel = getModelPointer();
    assert(fOurModel != NULL);

    Boolean bReturnBool = false;

    switch (event.mouseAction())
    {
        case IMouseClickEvent::doubleClick:
        {
            IWindow *pWindow = windowWithHandle(event.windowUnderPointer());
            unsigned long int winId = pWindow->id();
            switch (winId)
            {
                case ID_ENTRY1 :
                {
                    fOurModel->setString1(fDataEntryField1.text());
                    fOurModel->setPointerMyString(fDataEntryField1.text());
                    bReturnBool = true;
                    break;
                }
                case ID_ENTRY2 :
                {
                    fOurModel->setString2(fDataEntryField2.text());
                    fOurModel->setPointerMyString(fDataEntryField2.text());
                    bReturnBool = true;
                    break;
                }
            }
        }
    }
    return bReturnBool;
}

