///////////////////////////////////////////////////////////////////////////////
//                                              SAMPLE CODE
//
// FileName: ACDFVw4.cpp
//
// ClassName: ACompDocFwkView
//
// Description: Compound Document Framework Container View 
//              This is a container view which inherits from IView.
//              This sample illustrates how the embedded object can be
//              manipulated (shrink and grow)  on the screen. It also 
//              shows how to programmatically select,unselect and launch the 
//              embedded objects.                
///////////////////////////////////////////////////////////////////////////////
#include "ACDFMdl4.hpp"
#include "ACDFVw4.hpp"
#include "acdfres4.h"

#include <igstring.hpp>
#include <igrect.hpp>
#include <igrafctx.hpp>
#include <igelipse.hpp>
#include <igline.hpp>
#include <igarc.hpp>
#include <imsgbox.hpp>
#include <imenubar.hpp>
#include <itbar.hpp>
#include <iframe.hpp>   //This is needed to set the icon for the frame window
#include <itbarbut.hpp>
#include <ireslib.hpp>
#include <assert.h>

#include <iguibndl.hpp> // must be include after windows.h


ACompDocFwkView::ACompDocFwkView(IGUIBundle& bundle ) :
    IView(bundle),
    fCommandHandler( *this, handleCommand)
// Constructor - resize the window
{   IFUNCTRACE_DEVELOP();
    bundle.frameWindow().setIcon(IC_ACDF4);
}

ACompDocFwkView::~ACompDocFwkView()
// Destructor - stop all the handlers
{   IFUNCTRACE_DEVELOP();
    
    fCommandHandler.stopHandlingEventsFor( this );
    fCommandHandler.stopHandlingEventsFor( &bundle().objectToolBar() );
}

ACompDocFwkModel* ACompDocFwkView::getModelPointer() const
// Get the pointer to the model 
{   IFUNCTRACE_DEVELOP();
    ACompDocFwkModel* theModel = NULL;
    ::dynamicCastTo( theModel,model() );
    return theModel;
}

void ACompDocFwkView::initialize()
// Intialize code e.g build menus
{   IFUNCTRACE_DEVELOP();

    IView::initialize();
    bundle().objectView().setBackgroundColor(IColor(64,128,128)); // Set the color to light green

    // Manually adding the toolbar buttons to the existing toolbar
    IToolBar& myToolBar = bundle().objectToolBar();
    IToolBarButton* myButton;

    myButton = new IToolBarButton (TB_GROW, &myToolBar, &myToolBar);
    myToolBar.addAsLast( myButton, true);
    myButton = new IToolBarButton (TB_SHRINK, &myToolBar, &myToolBar);
    myToolBar.addAsLast( myButton, true);
    myButton = new IToolBarButton (TB_NEXT, &myToolBar, &myToolBar);
    myToolBar.addAsLast( myButton, true);
    myButton = new IToolBarButton (TB_PREV, &myToolBar, &myToolBar);
    myToolBar.addAsLast( myButton, true);
    myButton = new IToolBarButton (TB_LAUNCH, &myToolBar, &myToolBar);
    myToolBar.addAsLast( myButton, true);

    fCommandHandler.handleEventsFor( this );    // handle for menu's and pushbutton
    fCommandHandler.handleEventsFor( &myToolBar ); // handle floating toolbar
}

void ACompDocFwkView::drawContents( IPresSpaceHandle& hdl,
                      const IRectangle& invalidArea,
                      Boolean metaFile)
//  We can override "drawContents" if we want to optimize our drawing
//  or do things differently when rendering to a meta-file (where controls
//  won't be displayed).

{       IFUNCTRACE_DEVELOP();

    if (metaFile)
    {
    // Place the embedded image code here !!!!
        IGString myText (IApplication::current().userResourceLibrary().loadString(STR_METAFILE),IPoint(30,70));
        IGraphicContext ctxt ( hdl );
        IGEllipse myEllipse(IRectangle(IPoint(10,10),IPoint(210,50)));
        IGLine myLine1(IPoint(10,30),IPoint(40,150));
        IGLine myLine2(IPoint(210,30),IPoint(190,150));
        IG3PointArc myArc(IPoint(40,150),IPoint(75,170),IPoint(190,150));

        myEllipse.drawOn( ctxt) ;
        myLine1.drawOn( ctxt) ;
        myLine2.drawOn( ctxt) ;
        myArc.drawOn( ctxt) ;
        myText.drawOn( ctxt) ;
   }
}

void ACompDocFwkView::handleSelectAll(Boolean select)
// Handle select all by setting the background color to show that
// select all has been selected
{   IFUNCTRACE_DEVELOP();

    if (select)
        bundle().objectView().setBackgroundColor(IColor(0,0,128));  // Set the color to dark Blue
    else
        bundle().objectView().setBackgroundColor(IColor(64,128,128)); // re-Set the color back to light green
}

Boolean ACompDocFwkView::handleCommand(ICommandEvent& event)
// Catch the button pressed and menu bars selected
{   IFUNCTRACE_DEVELOP();
    IMessageBox msg(IWindow::desktopWindow());
    msg.setTitle(STR_VIEW);

    ACompDocFwkModel* ourModel = getModelPointer();
    assert(ourModel != NULL);

    switch( event.commandId() )
    {
        case MI_GROW:
        case TB_GROW:
        {
            // Get the current selected component from the embedder model
            IEmbeddedComponent* mySelectedComponent = ourModel->selectedComponent();
            if (mySelectedComponent )
            {
                IRectangle myComponentArea = mySelectedComponent->area();
                myComponentArea.sizeBy( 2.0 );

                mySelectedComponent ->resize( myComponentArea );
            }
            return true;
        }
        case MI_SHRINK:
        case TB_SHRINK:
        {
            IEmbeddedComponent* mySelectedComponent = ourModel->selectedComponent();
            if (mySelectedComponent )
            {
                IRectangle myComponentArea = mySelectedComponent->area();
                myComponentArea.sizeBy( 0.5 );

                mySelectedComponent ->resize( myComponentArea );
            }
            return true;
        }
        case MI_NEXT:
        case TB_NEXT:
        {
            // Get the component list from the embedder model
            IEmbedderModel::ComponentList myComponentList = ourModel->componentList();

            if ( myComponentList.isEmpty())
                return true;  // The list is emtpy therefore just exist

            // Get the number of components in the list (currently embedded)
            unsigned long myNumberElements = myComponentList.numberOfElements();
            ICursor* myCursor = myComponentList.newCursor();

            if (!myComponentList.setToFirst( *myCursor ))
            {
                msg.show(IResourceId(STR_INVALID_CURSOR),IMessageBox::okButton);
                return true;
            }
            // Find out which component is selected inorder to get the next
            IEmbeddedComponent* mySelectedComponent = ourModel->selectedComponent();
            if (!mySelectedComponent )
            {
                //Setting to the first component
                mySelectedComponent = myComponentList.elementAt( *myCursor);
                if ( mySelectedComponent )
                {
                    ourModel->selectComponent (mySelectedComponent);
                    refresh();
                    return true;
                }
            }
            
            for(int i=1;i <= myNumberElements;i++)
            {
                if (mySelectedComponent ==(IEmbeddedComponent*) myComponentList.elementAtPosition(i))
                {
                    myComponentList.setToPosition(i, *myCursor);
                    if (myComponentList.setToNext(*myCursor) )
                    {
                        mySelectedComponent = myComponentList.elementAt(*myCursor);
                    }
                    else
                    {
                        // Set to the first if there is no next
                        mySelectedComponent = (IEmbeddedComponent*)myComponentList.elementAtPosition(1);
                    }
                    ourModel->selectComponent (mySelectedComponent);
                    refresh();
                    return true;
                }
            }
        } 
        case MI_PREV:
        case TB_PREV:
        {
        // Get the component list from the embedder model
            IEmbedderModel::ComponentList myComponentList = ourModel->componentList();

            if ( myComponentList.isEmpty())
                return true;

            unsigned long myNumberElements = myComponentList.numberOfElements();
            ICursor* myCursor = myComponentList.newCursor();

            if (!myComponentList.setToFirst( *myCursor ))
            {
                msg.show(IResourceId(STR_INVALID_CURSOR),IMessageBox::okButton);
                return true;
            }

            IEmbeddedComponent* mySelectedComponent = ourModel->selectedComponent();
            if (!mySelectedComponent )
            {
                //Setting to the first component
                mySelectedComponent = myComponentList.elementAt( *myCursor);
                if ( mySelectedComponent )
                {
                    ourModel->selectComponent (mySelectedComponent);
                    refresh();  // Repaints the screen
                    return true;
                }
            }
            
            for(int i=1;i <= myNumberElements;i++)
            {
                if (mySelectedComponent ==(IEmbeddedComponent*) myComponentList.elementAtPosition(i))
                {
                    myComponentList.setToPosition(i, *myCursor);
                    if (myComponentList.setToPrevious(*myCursor) )
                    {
                        mySelectedComponent = myComponentList.elementAt(*myCursor);
                    }
                    else
                    {
                        // Set to last if no previous
                        mySelectedComponent = (IEmbeddedComponent*) myComponentList.elementAtPosition(myNumberElements);   
                    }
                    ourModel->selectComponent (mySelectedComponent);
                    refresh(); // Repaints the screen
                    return true;
                }
            }
        }
        case MI_UNSELECT:
        {
            // Unselect the selected object
            ourModel->selectComponent(NULL);
            refresh();  // Repaints the screen
            return true;
        }
        case MI_CASCADE:
        {
            // Get the component list from the embedder model
            IEmbedderModel::ComponentList myComponentList = ourModel->componentList();

            if ( myComponentList.isEmpty())
                return true;

            unsigned long myNumberElements = myComponentList.numberOfElements();
            ICursor* myCursor = myComponentList.newCursor();

            if (!myComponentList.setToFirst( *myCursor ))
            {
                msg.show(IResourceId(STR_INVALID_CURSOR),IMessageBox::okButton);
                return true;
            }

            IEmbeddedComponent* myComponent;
            for(int i=1;i <= myNumberElements;i++)
            {
                myComponent = (IEmbeddedComponent*)myComponentList.elementAtPosition(i);
                IRectangle myRect = myComponent->area();
                myRect.moveTo(IPoint((30*i),(30*i)));
                myComponent->resize(myRect);
            }
            return true;
        }
        case MI_HTILE: 
        {
        // Get the component list from the embedder model
            IEmbedderModel::ComponentList myComponentList = ourModel->componentList();

            if ( myComponentList.isEmpty())
                return true;

            ISize myWindowSize = this->size();
            IEmbeddedComponent* myComponent;
            unsigned long myNumberElements = myComponentList.numberOfElements();

            if (myNumberElements == 1)
            {
                myComponent = (IEmbeddedComponent*)myComponentList.elementAtPosition(1);
                IRectangle myRect = myComponent->area();
                myRect.sizeTo(myWindowSize);
                myComponent->resize(myRect);
            }
            ICursor* myCursor = myComponentList.newCursor();

            if (!myComponentList.setToFirst( *myCursor ))
            {
                msg.show(IResourceId(STR_INVALID_CURSOR),IMessageBox::okButton);
                return true;
            }

            // If there are more than 4 components screen gets to crowded 
            if (myNumberElements < 4)
            {
                for (int j=0,i=1;i <= myNumberElements;i++,j++)
                {
                    myComponent = (IEmbeddedComponent*)myComponentList.elementAtPosition(i);
                    IRectangle myRect = myComponent->area();
                    int verticalHeight = myWindowSize.height() / myNumberElements;
                    int verticalWidth = myWindowSize.width();
                    myRect=IRectangle(IPoint(0,(j*verticalHeight)),IPoint(verticalWidth,((j+1)*verticalHeight)));
                    myComponent->resize(myRect);
                }
            }
            else
                msg.show("There are more than 3 tiles, cannot tile");
            
            return true;
        }
        case MI_VTILE:
        {
        // Get the component list from the embedder model
            IEmbedderModel::ComponentList myComponentList = ourModel->componentList();

            if ( myComponentList.isEmpty())
                return true;

            ISize myWindowSize = this->size();
            IEmbeddedComponent* myComponent;
            unsigned long myNumberElements = myComponentList.numberOfElements();

            if (myNumberElements == 1)
            {
                myComponent = (IEmbeddedComponent*)myComponentList.elementAtPosition(1);
                IRectangle myRect = myComponent->area();
                myRect.sizeTo(myWindowSize);
                myComponent->resize(myRect);
            }
            ICursor* myCursor = myComponentList.newCursor();

            if (!myComponentList.setToFirst( *myCursor ))
            {
                msg.show(IResourceId(STR_INVALID_CURSOR),IMessageBox::okButton);
                return true;
            }

            if (myNumberElements < 4)
            {
                for (int j=0,i=1;i <= myNumberElements;i++,j++)
                {
                    myComponent = (IEmbeddedComponent*)myComponentList.elementAtPosition(i);
                    IRectangle myRect = myComponent->area();
                    int verticalHeight = myWindowSize.height() ;
                    int verticalWidth = myWindowSize.width() / myNumberElements;
                    myRect=IRectangle(IPoint((j*verticalWidth),0),IPoint(((j+1)*verticalWidth),verticalHeight));
                    myComponent->resize(myRect);
                }
            }
            else
                msg.show("There are more than 3 tiles, cannot tile");
            return true;
        }
        case MI_LAUNCH:
        case TB_LAUNCH:
        {
            // Get the currently selected component and start it
            IEmbeddedComponent* mySelectedComponent = ourModel->selectedComponent();
            if (mySelectedComponent )
            {
                mySelectedComponent ->goActive();
            }
            return true;
        }
    }               
    return false;  // Command not handled
}

