/******************************************************************************
* .FILE:         magnify.cpp                                                  *
*                                                                             *
* .DESCRIPTION:  Magnify Sample Program:  Class Implementation                *
*                                                                             *
* .CLASSES:      MagnifyWindow                                                *
*                MagnifyHandler                                               *
*                                                                             *
* .COPYRIGHT:                                                                 *
*    Licensed Material - Program-Property of IBM                              *
*    (C) Copyright IBM Corp. 1992, 1996 - All Rights Reserved                 *
*                                                                             *
* .DISCLAIMER:                                                                *
*   The following [enclosed] code is sample code created by IBM               *
*   Corporation.  This sample code is not part of any standard IBM product    *
*   and is provided to you solely for the purpose of assisting you in the     *
*   development of your applications.  The code is provided 'AS IS',          *
*   without warranty of any kind.  IBM shall not be liable for any damages    *
*   arising out of your use of the sample code, even if they have been        *
*   advised of the possibility of such damages.                               *
*                                                                             *
* .NOTE: WE RECOMMEND USING A FIXED SPACE FONT TO LOOK AT THE SOURCE          *
*                                                                             *
******************************************************************************/
#include <ibase.hpp>
#include <iapp.hpp>
#include <iframe.hpp>
#include <icmdhdr.hpp>
#include <igbitmap.hpp>
#include <igrafctx.hpp>
#include <ibmpctl.hpp>
#include <itimer.hpp>
#include <isetcv.hpp>
#include <icustbut.hpp>
#include <ispinnum.hpp>
#include <icoordsy.hpp>
#include "magnify.hpp"

/******************************************************************************
* main - Application entry point                                              *
******************************************************************************/
int main()
{
  ICoordinateSystem::setApplicationOrientation(
          ICoordinateSystem::originLowerLeft );
  MagnifyWindow win;
  IApplication::current().run();

  return 0;
}

/******************************************************************************
* Class MagnifyWindow::MagnifyWindow - Constructor for the main window        *
*                                                                             *
* Define itself as an IFrameWindow                                            *
* Create the title for the window                                             *
* Define a control for a bitmap                                               *
* Define a canvas to put the spin buttons, pushbuttons and text.              *
* Define the spin buttons, pushbuttons, and the text.                         *
******************************************************************************/
MagnifyWindow::MagnifyWindow ()
              :IFrameWindow (WND_MAIN),
               title(this),
               bitmapControl(0,this,this),
               canvas(100,this,this),
               button(101,&canvas,&canvas),
               xText(102,&canvas,&canvas),
               xSpin(103,&canvas,&canvas),
               yText(104,&canvas,&canvas),
               ySpin(105,&canvas,&canvas),
               bitmap(0)
{
/*----------------------------------------------------------------------------|
| Set the icon and title for the main window                                  |
-----------------------------------------------------------------------------*/
  setIcon( id() );
  title.setTitleText(TITLE_MAGNIFY);

/*----------------------------------------------------------------------------|
| Sets the bitmap as the client area                                          |
-----------------------------------------------------------------------------*/
  setClient(&bitmapControl);

/*----------------------------------------------------------------------------|
| Begin handling events for the push button                                   |
-----------------------------------------------------------------------------*/
  handler.handleEventsFor(this);

/*----------------------------------------------------------------------------|
| Enable latching for the button                                              |
| Set the text of the button                                                  |
| And. set it as a tab stop                                                   |
-----------------------------------------------------------------------------*/
  button.enableLatching().setText(STR_MAGNIFY).enableTabStop();

/*----------------------------------------------------------------------------|
| Set up the spin buttons with their text and the range of values             |
-----------------------------------------------------------------------------*/
  xText.setText(STR_X);
  yText.setText(STR_Y);
  xSpin.setRange(IRange(1,desktopWindow()->size().width())).setLimit(4);
  xSpin.setValue(32).enableTabStop();
  ySpin.setRange(IRange(1,desktopWindow()->size().height())).setLimit(4);
  ySpin.setValue(32).enableTabStop();

/*----------------------------------------------------------------------------|
| Define the margins and alignment of the set cavas                           |
| Add the set canvas as an extension of the main frame window                 |
| Resize the client area (the bitmap control) to fit the fram window          |
-----------------------------------------------------------------------------*/
  canvas.setMargin(ISize(2,2)).setAlignment(ISetCanvas::centerCenter);
  addExtension(&canvas,aboveClient,(unsigned long) (canvas.minimumSize().height()));
  moveSizeToClient(IRectangle(50,50,50+canvas.minimumSize().width(),
                                    50+canvas.minimumSize().width()));

/*----------------------------------------------------------------------------|
|  Set focus and show the window.                                             |
-----------------------------------------------------------------------------*/
  show();
  button.setFocus();
}


/******************************************************************************
* class MagnifyWindow::timerFunction - Called when the timer expires and grabs*
*   a new bitmap of the section of the window that is being magnified.        *
******************************************************************************/
void MagnifyWindow::timerFunction ()
{
/*----------------------------------------------------------------------------|
| Create a rectangle with the size specified by the spin buttons              |
-----------------------------------------------------------------------------*/
  unsigned long xSize = xSpin.value();
  unsigned long ySize = ySpin.value();
  IRectangle rect(0,0,xSize,ySize);

/*----------------------------------------------------------------------------|
| Reposition the rectangle so that the mouse is at the center of the rectangle|
-----------------------------------------------------------------------------*/
  ISize desktopSize = desktopWindow()->size();
  IPoint mousePosition = IWindow::pointerPosition();
  rect.centerAt(mousePosition);
  if (rect.left()<0)
    rect.moveTo(IPoint(0,rect.bottom()));
  if (rect.bottom()<0)
    rect.moveTo(IPoint(rect.left(),0));
  if (rect.right()>=desktopSize.width())
    rect.moveTo(IPoint(desktopSize.width()-xSize,rect.bottom()));
  if (rect.top()>= desktopSize.height())
    rect.moveTo(IPoint(rect.left(),desktopSize.height()-ySize));


/*----------------------------------------------------------------------------|
| Convert the rectangle to native coordinates.                                |
| For graphics, this function should be used for portable graphics.           |
-----------------------------------------------------------------------------*/
  rect = ICoordinateSystem::convertToNative( rect, desktopSize );

/*----------------------------------------------------------------------------|
| Create a new bitmap from the desktop in the area specified by the rectangle.|
| Delete the old bitmap and reset its pointer to the new bitmap.              |
-----------------------------------------------------------------------------*/
  IGBitmap* newBitmap = new IGBitmap(IGraphicContext(desktopWindow()->handle()),rect);
  bitmapControl.setBitmap(newBitmap->handle());
  if (bitmap)
    delete bitmap;
  bitmap = newBitmap;
}

/******************************************************************************
* MagnifyHandler::command - Command handler that turns the timer off and on.  *
******************************************************************************/
IBase::Boolean MagnifyHandler::command(ICommandEvent& event)
{
  MagnifyWindow *win = (MagnifyWindow*) event.window();
  if (win->button.isLatched())
    win->timer.start(new ITimerMemberFn0<MagnifyWindow>(*win,MagnifyWindow::timerFunction),200);
  else
    win->timer.stop();
  return true;
}

