/*******************************************************************************
* FILE NAME: irecord.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Functions to implement the class(es):                                      *
*     IRecord - Base class for all record classes.                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM(R) VisualAge(TM) for C++                                               *
*   (C) Copyright International Business Machines Corporation 1991, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*   This program will not run in DOS mode.                                     *
*                                                                              *
* DISCLAIMER OF WARRANTIES:                                                    *
*   The following [enclosed] code is sample code created by IBM                *
*   Corporation.  This sample code is not part of any standard IBM product     *
*   and is provided to you solely for the purpose of assisting you in the      *
*   development of your applications.  The code is provided "AS IS",           *
*   without warranty of any kind.  IBM shall not be liable for any damages     *
*   arising out of your use of the sample code, even if they have been         *
*   advised of the possibility of such damages.                                *
*******************************************************************************/
#include <irecord.hpp>

#include <string.h>

// Define the functions and static data members to be exported.
// Ordinals 350 through 399 are reserved for use by IRecord.
#pragma export(IRecord::asString() const,, 350)
#pragma export(IRecord::field(IString&,unsigned long,unsigned long,    \
                   unsigned char) const,, 351)
#pragma export(IRecord::field(char*,unsigned long,unsigned long,       \
                   unsigned char) const,, 352)
#pragma export(IRecord::field(char,unsigned long,unsigned char) const,, 353)
#pragma export(IRecord::field(double,unsigned long,unsigned char) const,, 354)
#pragma export(IRecord::field(float,unsigned long,unsigned char) const,, 355)
#pragma export(IRecord::field(long double,unsigned long,unsigned char) const,, 356)
#pragma export(IRecord::field(unsigned long,unsigned long,unsigned char) const,, 357)
#pragma export(IRecord::field(unsigned short,unsigned long,unsigned char) const,, 358)
#pragma export(IRecord::field(void*,unsigned long,unsigned long,       \
                   unsigned char) const,, 359)
#pragma export(IRecord::IRecord(),, 360)
#pragma export(IRecord::IRecord(const IRecord&),, 361)
#pragma export(IRecord::IRecord(const IString&),, 362)
#pragma export(IRecord::isOrdered() const,, 363)
#pragma export(IRecord::operator=(const IRecord&),, 364)
#pragma export(IRecord::operator=(const IString&),, 365)
#pragma export(IRecord::setField(const IString&,unsigned long,         \
                   unsigned long, unsigned char),, 366)
#pragma export(IRecord::setField(const char*,unsigned long,            \
                   unsigned long,unsigned char),, 367)
#pragma export(IRecord::setField(const char,unsigned long),, 368)
#pragma export(IRecord::setField(const double,unsigned long),, 369)
#pragma export(IRecord::setField(const float,unsigned long),, 370)
#pragma export(IRecord::setField(const long double,unsigned long),, 371)
#pragma export(IRecord::setField(const unsigned long,unsigned long),, 372)
#pragma export(IRecord::setField(const unsigned short,unsigned long),, 373)
#pragma export(IRecord::setField(const void*,unsigned long,unsigned long),, 374)
#pragma export(IRecord::setSize(unsigned long),, 375)
#pragma export(IRecord::size() const,, 376)
#pragma export(IRecord::~IRecord(),, 377)

// It's possible for the caller of the external entry points to have a
// different C library environment.  Make sure the exception handler for
// our library environment is registered on entry and deregistered on exit.
#ifdef __OS2__
#pragma handler(IRecord::asString())
#pragma handler(IRecord::field(IString&,unsigned long,unsigned long,   \
                    unsigned char))
#pragma handler(IRecord::field(char*,unsigned long,unsigned long,unsigned char))
#pragma handler(IRecord::field(char,unsigned long,unsigned char))
#pragma handler(IRecord::field(double,unsigned long,unsigned char))
#pragma handler(IRecord::field(float,unsigned long,unsigned char))
#pragma handler(IRecord::field(long double,unsigned long,unsigned char))
#pragma handler(IRecord::field(unsigned long,unsigned long,unsigned char))
#pragma handler(IRecord::field(unsigned short,unsigned long,unsigned char))
#pragma handler(IRecord::field(void*,unsigned long,unsigned long,unsigned char))
#pragma handler(IRecord::IRecord())
#pragma handler(IRecord::IRecord(const IRecord&))
#pragma handler(IRecord::IRecord(const IString&))
#pragma handler(IRecord::isOrdered())
#pragma handler(IRecord::operator=(const IRecord&))
#pragma handler(IRecord::operator=(const IString&))
#pragma handler(IRecord::setField(const IString&,unsigned long,        \
                    unsigned long, unsigned char))
#pragma handler(IRecord::setField(const char*,unsigned long,           \
                    unsigned long,unsigned char))
#pragma handler(IRecord::setField(const char,unsigned long))
#pragma handler(IRecord::setField(const double,unsigned long))
#pragma handler(IRecord::setField(const float,unsigned long))
#pragma handler(IRecord::setField(const long double,unsigned long))
#pragma handler(IRecord::setField(const unsigned long,unsigned long))
#pragma handler(IRecord::setField(const unsigned short,unsigned long))
#pragma handler(IRecord::setField(const void*,unsigned long,unsigned long))
#pragma handler(IRecord::setSize(unsigned long))
#pragma handler(IRecord::size())
#pragma handler(IRecord::~IRecord())
#endif


/*------------------------------------------------------------------------------
| Function Name: IRecord :: IRecord
|
| Implementation:
|   Default initialization of the record data IString is okay.
|-----------------------------------------------------------------------------*/
IRecord :: IRecord()
{
  ;
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: IRecord
|
| Implementation:
|   Copy the passed in record data.
|-----------------------------------------------------------------------------*/
IRecord :: IRecord ( const IString & recordData ) :
                   sRecordData( recordData )
{
  ;
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: IRecord
|
| Implementation:
|   Copy the passed in record data.
|-----------------------------------------------------------------------------*/
IRecord :: IRecord ( const IRecord & aRecord ) :
                   sRecordData( aRecord.sRecordData )
{
  ;
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: ~IRecord
|
| Implementation:
|   The implementation IString is automatically deleted.
|-----------------------------------------------------------------------------*/
IRecord :: ~IRecord()
{
  ;
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: isOrdered
|
| Implementation:
|   By default, IRecord objects are not ordered (see IOrderedRecord class).
------------------------------------------------------------------------------*/
IBoolean IRecord :: isOrdered ( ) const
{
  return( false );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: asString
|
| Implementation:
|   Return a copy of the implementation string.
------------------------------------------------------------------------------*/
IString IRecord :: asString ( ) const
{
  return( sRecordData );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: operator =
|
| Implementation:
|   Replace our data with the source IRecord data.
------------------------------------------------------------------------------*/
IRecord & IRecord :: operator = ( const IRecord & aRecord )
{
  // Check for assignment to self.
  if ( this != &aRecord )
    sRecordData = aRecord.sRecordData;
  return( *this );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: operator =
|
| Implementation:
|   Replace our data with the source IString data.
------------------------------------------------------------------------------*/
IRecord & IRecord :: operator = ( const IString & aRecord )
{
  sRecordData = aRecord;
  return( *this );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: size
|
| Implementation:
|   The IString size is the IRecord size.
------------------------------------------------------------------------------*/
unsigned long IRecord :: size ( ) const
{
  return( sRecordData.size( ) );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setSize
|
| Implementation:
|   Set the size of the IString to the requested size.
------------------------------------------------------------------------------*/
unsigned long IRecord :: setSize ( unsigned long size )
{
  if ( sRecordData.size( ) > size )
    sRecordData.remove( size + 1 );
  else if ( sRecordData.size( ) < size )
    sRecordData.insert( "", size, '\x00' );

  return( size );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract the field into the caller's buffer and return the buffer.
------------------------------------------------------------------------------*/
void * IRecord :: field ( void * pBuffer,
                          unsigned long recordOffset,
                          unsigned long size,
                          unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset, size, padByte );
  memcpy( pBuffer, (unsigned char *)theField, size );
  return( pBuffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract the field into the caller's IString and return the IString.
------------------------------------------------------------------------------*/
IString & IRecord :: field ( IString & buffer,
                             unsigned long recordOffset,
                             unsigned long size,
                             unsigned char padByte ) const
{
  buffer = sRecordData.subString( recordOffset, size, padByte );
  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract the field into the caller's null terminated character string
|   and return the string.
------------------------------------------------------------------------------*/
char * IRecord :: field ( char * pBuffer,
                          unsigned long recordOffset,
                          unsigned long size,
                          unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset, size, padByte );
  memcpy( pBuffer, (unsigned char *)theField, size );
  pBuffer[size] = '\x00';
  return( pBuffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract a character and return it to the caller.
------------------------------------------------------------------------------*/
char IRecord :: field ( char buffer,
                        unsigned long recordOffset,
                        unsigned char padByte ) const
{
  if ( recordOffset <= sRecordData.size() )
    buffer = sRecordData[recordOffset];
  else
    buffer = padByte;

  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract a short integer and return it to the caller.
------------------------------------------------------------------------------*/
unsigned short IRecord :: field ( unsigned short buffer,
                                  unsigned long recordOffset,
                                  unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset,
                                            sizeof(buffer),
                                            padByte );
  memcpy( &buffer, (unsigned char *)theField, sizeof(buffer) );
  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract a long integer and return it to the caller.
------------------------------------------------------------------------------*/
unsigned long IRecord :: field ( unsigned long buffer,
                                 unsigned long recordOffset,
                                 unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset,
                                            sizeof(buffer),
                                            padByte );
  memcpy( &buffer, (unsigned char *)theField, sizeof(buffer) );
  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract a float and return it to the caller.
------------------------------------------------------------------------------*/
float IRecord :: field ( float buffer,
                         unsigned long recordOffset,
                         unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset,
                                            sizeof(buffer),
                                            padByte );
  memcpy( &buffer, (unsigned char *)theField, sizeof(buffer) );
  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract a double and return it to the caller.
------------------------------------------------------------------------------*/
double IRecord :: field ( double buffer,
                          unsigned long recordOffset,
                          unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset,
                                            sizeof(buffer),
                                            padByte );
  memcpy( &buffer, (unsigned char *)theField, sizeof(buffer) );
  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: field
|
| Implementation:
|   Extract a long double and return it to the caller.
------------------------------------------------------------------------------*/
long double IRecord :: field ( long double buffer,
                               unsigned long recordOffset,
                               unsigned char padByte ) const
{
  IString theField = sRecordData.subString( recordOffset,
                                            sizeof(buffer),
                                            padByte );
  memcpy( &buffer, (unsigned char *)theField, sizeof(buffer) );
  return( buffer );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's buffer and return the passed buffer.
------------------------------------------------------------------------------*/
const void * IRecord :: setField ( const void * pData,
                                   unsigned long recordOffset,
                                   unsigned long size )
{
  IString theData( pData, size );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( pData );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's IString and return the passed IString.
------------------------------------------------------------------------------*/
const IString & IRecord :: setField ( const IString & data,
                                      unsigned long recordOffset,
                                      unsigned long size,
                                      unsigned char padByte )
{
  unsigned long copySize = size;
  unsigned long padSize  = 0;
  if ( data.size() < size )
  {
    copySize = data.size();
    padSize  = size - data.size();
  }
  IString theData( (unsigned char *)data,
                   copySize,
                   NULL,
                   padSize,
                   padByte );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's null terminated character string and
|   return the pointer to the passed character string.
------------------------------------------------------------------------------*/
const char * IRecord :: setField ( const char * pData,
                                   unsigned long recordOffset,
                                   unsigned long size,
                                   unsigned char padByte )
{
  unsigned long copySize = size;
  unsigned long padSize  = 0;
  if ( strlen( pData ) < size )
  {
    copySize = strlen( pData );
    padSize  = size - strlen( pData );
  }
  IString theData( pData,
                   copySize,
                   NULL,
                   padSize,
                   padByte );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( pData );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's character and return passed character.
------------------------------------------------------------------------------*/
const char IRecord :: setField ( const char data,
                                 unsigned long recordOffset )
{
  IString theData( &data, sizeof(data) );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's short integer and return passed
|   short integer.
------------------------------------------------------------------------------*/
const unsigned short IRecord :: setField ( const unsigned short data,
                                           unsigned long recordOffset )
{
  IString theData( &data, sizeof(data) );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's long integer and return passed
|   long integer.
------------------------------------------------------------------------------*/
const unsigned long IRecord :: setField ( const unsigned long data,
                                          unsigned long recordOffset )
{
  IString theData( &data, sizeof(data) );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's float and return passed float.
------------------------------------------------------------------------------*/
const float IRecord :: setField ( const float data,
                                  unsigned long recordOffset )
{
  IString theData( &data, sizeof(data) );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's double and return passed double.
------------------------------------------------------------------------------*/
const double IRecord :: setField ( const double data,
                                   unsigned long recordOffset )
{
  IString theData( &data, sizeof(data) );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}

/*------------------------------------------------------------------------------
| Function Name: IRecord :: setField
|
| Implementation:
|   Overlay the field with the caller's long double and return passed
|   long double.
------------------------------------------------------------------------------*/
const long double IRecord :: setField ( const long double data,
                                        unsigned long recordOffset )
{
  IString theData( &data, sizeof(data) );
  sRecordData.overlayWith( theData, recordOffset, '\x00' );
  return( data );
}
