--------------------------------------------------------------------------------
--
-- COPYRIGHT:
--   IBM WorkFrame - Project Smarts
--   (C) Copyright International Business Machines Corporation 1996
--   Licensed Material - Program-Property of IBM - All Rights Reserved.
--   US Government Users Restricted Rights - Use, duplication, or disclosure
--   restricted by GSA ADP Schedule Contract with IBM Corp.
--
--------------------------------------------------------------------------------

<include prologcp.tde>

/* UI Class Library headers */
#include <ibase.hpp>
#include <iaccel.hpp>
#include <iapp.hpp>
#include <iinfoa.hpp>
#include <ifont.hpp>
#include <ithread.hpp>
#include <ifiledlg.hpp>
#include <imsgbox.hpp>
#include <ihelp.hpp>
#include <iexcept.hpp>

/* Application headers */
#include "prodinfo.hpp"
#include "$FILE_NAME$.h"
#include "$FILE_NAME$.hpp"


/**************************************************************/
/* main() program entry point                                 */
/**************************************************************/
void main()
{
<if ($USE_IPF$)>
   IHelpWindow::setDefaultStyle( IHelpWindow::defaultStyle()
                                 | IHelpWindow::ipfCompatible );
</if>
   AppWindow appWindow(IDW_FRAME_WINDOW);
   IApplication::current().run();
}


/**************************************************************/
/* Application main window constructor.  Sets events handler, */
/* icon, window style and menu bar.  It also sets static text */
/* as the client area control.  You may want to set           */
/**************************************************************/
AppWindow :: AppWindow(unsigned long windowId)
  :IFrameWindow( IFrameWindow::defaultStyle() |   // Create frame window in default style
                 IFrameWindow::accelerator,
                 windowId)
  ,title( this, IDW_FRAME_WINDOW, IDS_NO_FILE ) // Create title bar
  ,menuBar( IResourceId( windowId ), this )     // Create menu bar
  ,mle(ID_MLE,this,this)                        // Create MLE for client area
  ,openFileName("")                             // No file open yet
  ,hasHelp(true)                                // Assume help is available for now
  ,infoarea( this )
  ,cmdHdr( this )
  ,hasFile( false )
{
   // Set up status line
   infoarea.setInactiveText( IDS_GENERAL_DESC);     // Set status line text

   // Set up menu bar
   menuBar.setMenu( IResourceId(IDW_FRAME_WINDOW));

   // Set up client area as MLE
   setClient(&mle);       // Set MLE control for client area

   // Set self as command event handler
   cmdHdr.handleEventsFor( this );

   // Set application icon
   setIcon(IDI_ICON);

   try
   {
     // Set up help, help settings
     IHelpWindow::Settings helpSettings;
     helpSettings.setMenuBar(IDM_HELP);
     helpSettings.setTitle(IDS_HELP_TITLE);
     helpSettings.setHelpTable(IDH_MAIN_HELPTABLE);
     helpSettings.setLibraries(HELP_FILE_NAME);
     help = new IHelpWindow(helpSettings,this);
   }
   catch( IAccessError &exc )
   {
      // Inform the user that help is not available
      IMessageBox errorMsg( this );
      errorMsg.setTitle( IResourceId(IDW_FRAME_WINDOW));
      errorMsg.show( IResourceId(IDS_ERROR_NO_HELP),
                     IMessageBox::catastrophic);

      // Set internal help flag to indicate that no help is available
      hasHelp = false;
   }

   // If there is no text in the clipboard, disable the Paste menuitem.
   if ( ! mle.clipboardHasTextFormat() )
      enablePaste( false );

   // Set focus to this window and show
   setFocus().show();
}


/***********************************************************/
/* Application main window destructor.  It stops the event */
/* handling and deletes any objects on the heap.           */
/***********************************************************/
AppWindow :: ~AppWindow ()
{
   cmdHdr.stopHandlingEventsFor( this );
   if ( isHelpAvailable() )
   {
      delete help;
   }
}


/*********************************************************************/
/* This productInfo function displays the product information dialog */
/*********************************************************************/
AppWindow& AppWindow :: productInfo()
{
   unsigned short rc;

   ProdInfoDialog prodInfo( this );
   prodInfo.showModally();
   rc = prodInfo.result();       //Get result (eg OK or Cancel)         .
   return *this;
}


/***********************************************************/
/* This function returns TRUE if there is currently no     */
/* help available to the application.                      */
/***********************************************************/
Boolean AppWindow :: isHelpAvailable() const
{
   return (hasHelp);
}

/***********************************************************/
/* Sets the information area text given a string           */
/***********************************************************/
AppWindow& AppWindow::setStatus( const char * string )
{
  infoarea.setText( string );
  return *this;
}

/***********************************************************/
/* Sets the information area text given a string resource  */
/***********************************************************/
AppWindow& AppWindow::setStatus( IResourceId id )
{
  infoarea.setText( id );
  return *this;
}

/***********************************************************/
/* Returns the editor                                      */
/***********************************************************/
IMultiLineEdit* AppWindow::editor()
{
  return &mle;
}

/***********************************************************/
/* Returns true if a file has already been selected        */
/***********************************************************/
Boolean AppWindow::hasFileName()
{
  return ( hasFile );
}

/***********************************************************/
/* Returns the current file name                           */
/***********************************************************/
IString AppWindow::fileName()
{
  return ( openFileName );
}

/***********************************************************/
/* Sets the current file name                              */
/***********************************************************/
AppWindow& AppWindow::setFileName( const char * file )
{
  hasFile = true;
  openFileName = file;
  title.setViewText( file );
  return *this;
}

/***********************************************************/
/* Returns the help window                                 */
/***********************************************************/
IHelpWindow* AppWindow::helpWin()
{
  return help;
}

/***********************************************************/
/* Enables (or disables) the paste menu item               */
/***********************************************************/
AppWindow& AppWindow::enablePaste( Boolean which )
{
  menuBar.enableItem( IDM_EDIT_PASTE, which );
  return *this;
}

/***********************************************************/
/* AppCommandHandler :: AppCommandHandler - constructor    */
/***********************************************************/
AppCommandHandler::AppCommandHandler( AppWindow* appWindow )
    :ICommandHandler()
    ,app( appWindow )
{
}

/***********************************************************/
/* AppCommandHandler :: command - handle comment events    */
/***********************************************************/
IBase::Boolean AppCommandHandler::command( ICommandEvent& event )
 {
   IResourceLibrary  resLib;

   switch ( event.commandId() )
   {
      case IDM_FILE_OPEN:
       {
        /*-------------------------------------------------------------*/
        /* When the user selects the File->Open menu item to open a    */
        /* file, display the standard "File open" dialog using         */
        /* IFileDialog.  The dialog is set to its most previous        */
        /* values.  Then, a thread is started to open the file and     */
        /* and perform any other application-specific file processing. */
        /*-------------------------------------------------------------*/

         IFileDialog::Settings fdSettings;  // Create file dialog settings object
         fdSettings.setOpenDialog();
         fdSettings.setOKButtonText(IResourceId(IDS_OPEN));
         fdSettings.setTitle(IResourceId(IDS_FILE_OPEN));
         fdSettings.setFileName("*.*");

         // Create and show File open dialog
         IFileDialog fileDlg( IWindow::desktopWindow(), // Parent is desktop
                              app,                      // Owner is main window
                              IFileDialog::helpButton, // Help button style
                              fdSettings);              // Here are my settings

         // Check results
         if ( fileDlg.pressedOK() )
          {
            app->setFileName( fileDlg.fileName() );    // Store selected file name
            app->editor()->importFromFile( app->fileName() );  // Load the file into the MLE
          }
          break;
       }

      case IDM_FILE_SAVE:
       {

         if ( app->hasFileName() )
         {
           app->editor()->exportToFile( app->fileName() );    // Save to file

           IString tmpText = resLib.loadString(IDS_SAVE_FILE);
           tmpText.change( "%s",
                           app->fileName(),
                           1, 1);
           app->setStatus( tmpText );
         }
         else
         {
           IString tmpText = resLib.loadString(IDS_ERROR_NO_FILE_LOADED);
           app->setStatus( tmpText );
         }
         break;
       }


      case IDM_FILE_SAVEAS:
       {
         /*-------------------------------------------------------------*/
         /* When the user selects the "File->Save as" menu item,        */
         /* display the standard "Save as" dialog using IFileDialog.    */
         /* Use the IFileDialog::Settings class to specify initial      */
         /* settings, such as the currently open file.                  */
         /*-------------------------------------------------------------*/

          // Specify Save as dialog settings
          IFileDialog::Settings fdSettings;
          fdSettings.setFileName( app->fileName() );
          fdSettings.setSaveAsDialog();
          fdSettings.setOKButtonText( IResourceId( IDS_SAVEAS ) );
          fdSettings.setTitle( IResourceId( IDS_SAVEAS ) );

          // Create and show Save as dialog
          IFileDialog fileDlg( IWindow::desktopWindow(), // Parent is desktop
                               app,                      // Owner is main window
                               IFileDialog::helpButton, // Help button style
                               fdSettings);              // Here are my settings

          // Check results
          if ( fileDlg.pressedOK() )
          {
             /*================================================*/
             /* Place processing to save file here.            */
             /*================================================*/

             app->setFileName( fileDlg.fileName() );
             app->editor()->exportToFile( app->fileName() );  // Save to file

             // Display status
             IString tmpStatus = resLib.loadString(IDS_FILE_SAVED);
             tmpStatus.change( "%s",
                               app->fileName(),
                               1, 1);
             app->setStatus( tmpStatus );
          }
        break;
       }

      case IDM_FILE_CLOSE:
       {
         // Display "Are you sure" message box
         IMessageBox closeMsg( app );
         closeMsg.setTitle( IResourceId(IDW_FRAME_WINDOW));  // Title is application title
         IMessageBox::Response msgRC;
         msgRC = closeMsg.show( IResourceId(IDS_EXIT),
                        IMessageBox::yesNoButton      |
                        IMessageBox::defButton1       |
                        IMessageBox::queryIcon        |
                        IMessageBox::applicationModal |
                        IMessageBox::moveable);


         if (msgRC == IMessageBox::yes)  // If user responded yes, close window
           app->close();
         break;
       }

      case IDM_EDIT_COPY:
      {
         if( app->editor()->hasSelectedText())
         {
           app->editor()->copy();
           app->enablePaste();
         }
         app->setStatus( IResourceId( IDS_COPY ));
         break;
      }

      case IDM_EDIT_CUT:
      {
         if( app->editor()->hasSelectedText())
         {
           app->editor()->cut();
           app->enablePaste();
         }
         app->setStatus( IResourceId( IDS_CUT ));
         break;
      }

      case IDM_EDIT_PASTE:
      {
         if ( app->editor()->clipboardHasTextFormat())
           app->editor()->paste();
         app->setStatus( IResourceId( IDS_PASTE ));
         break;
      }

      case IDM_HELP_USING:
      {
         // Show IPF Using Help panel
         if ( app->isHelpAvailable() )
           app->helpWin()->show(IHelpWindow::using);
         app->setStatus( IResourceId( IDS_HELP_USING ) );
         break;
      }

      case IDM_HELP_PRODINFO:
      {
         // Show product information dialog
         app->productInfo();
         app->setStatus( IResourceId( IDS_HELP_PRODINFO ) );
         break;
      }

   } /* end switch */
   return false;
}

