#ifndef _IBASSTRM_
#define _IBASSTRM_
/*******************************************************************************
* FILE NAME: ibasstrm.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the classes:                                                *
*     IBaseStream                                                              *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*   Portions or this component are based on work originating from Taligent.    *
*   (C) Copyright Taligent, Inc. 1996 - All rights reserved.                   *
*                                                                              *
*******************************************************************************/
#include <ivbase.hpp>
#include <istring.hpp>
#include <iseq.h>
#include <iptr.h>
#include <iacllct.h>

class IContext;

#pragma pack(4)


class IBaseStream : public IVBase {
typedef IBase
  Inherited;
public:
/*------------------------------- Type Definitions ---------------------------*/
typedef unsigned short  VersionNumber;
typedef unsigned long   Position;
typedef long            PositionDelta;

/*------------------------------- Constructors -------------------------------*/
virtual
 ~IBaseStream();


/*------------------------ Reading and Writing Primitives --------------------*/
virtual IBaseStream
 &read    ( void*           target,
            size_t          byteCount ),
 &read    ( char*           target ),
 &read    ( signed char&    target ),
 &read    ( unsigned char&  target ),
 &read    ( short&          target ),
 &read    ( unsigned short& target ),
 &read    ( long&           target ),
 &read    ( unsigned long&  target ),
 &read    ( float&          target ),
 &read    ( double&         target);

virtual IBaseStream
 &write   ( const void*     source,
            size_t          byteCount ),
 &write   ( const char*     source ),
 &write   ( signed char     source ),
 &write   ( unsigned char   source ),
 &write   ( short           source ),
 &write   ( unsigned short  source ),
 &write   ( long            source ),
 &write   ( unsigned long   source ),
 &write   ( float           source ),
 &write   ( double          source );

/*------------------------ Reading and Writing Arrays ------------------------*/
virtual IBaseStream
 &write   ( const signed char source[],    unsigned long count ),
 &write   ( const unsigned char source[],  unsigned long count ),
 &write   ( const short source[],          unsigned long count ),
 &write   ( const unsigned short source[], unsigned long count ),
 &write   ( const long source[],           unsigned long count ),
 &write   ( const unsigned long source[],  unsigned long count ),
 &write   ( const float source[],          unsigned long count ),
 &write   ( const double source[],         unsigned long count );

virtual IBaseStream
 &read    ( signed char target[],    unsigned long count ),
 &read    ( unsigned char target[],  unsigned long count ),
 &read    ( short target[],          unsigned long count ),
 &read    ( unsigned short target[], unsigned long count ),
 &read    ( long target[],           unsigned long count ),
 &read    ( unsigned long target[],  unsigned long count ),
 &read    ( float target[],          unsigned long count ),
 &read    ( double target[],         unsigned long count );


/*------------- Stream Positioning, Flushing, and Write Access ---------------*/
virtual Position
  position              ( ) const = 0,
  logicalEndOfStream    ( ) const,
  physicalEndOfStream   ( ) const = 0;

virtual IBaseStream
 &seek                  ( Position      position) = 0,
 &seekRelative          ( PositionDelta offset),
 &setLogicalEndOfStream ( Position      position),
 &setPhysicalEndOfStream ( Position position) = 0,
 &flush   ( );

virtual Boolean
  isWriteable    ( ) const;

/*------------------------------- Stream Context------------------------------*/
virtual IBaseStream
 &setContext     ( IContext* context );
IContext
 *context        ( ) const,
 *defaultContext ( ) const;

protected:
/*------------------------ Buffer Management Overrides -----------------------*/
virtual void
  handleReadBufferEmpty   ( void*       target,
                            size_t      byteCount ) = 0;

virtual void
  handleWriteBufferFull   ( const void* source,
                            size_t      byteCount ) = 0;


/*------------------------------- Constructors -------------------------------*/
  IBaseStream  ( );


/*---------------------------- Primitive Type Lengths ------------------------*/
enum TypeLength {
  kCharBytes = 1,
  kSignedCharBytes = 1,
  kUnsignedCharBytes = 1,
  kShortBytes = 2,
  kUnsignedShortBytes = 2,
  kUnsignedLongBytes = 4,
  kLongBytes = 4,
  kFloatBytes = 4,
  kDoubleBytes = 8};

private:
/*------------------------------ Hidden Members ------------------------------*/
  IBaseStream  ( const IBaseStream& baseStream);
const IBaseStream
 &operator=    ( const IBaseStream& baseStream);

/*----------------------------- Private --------------------------------------*/
IContext
 *fContext,
 *fDefaultContext;

};  // IBaseStream

/*------------------------- Builtin Types Streaming Operators-----------------*/
IBaseStream& operator<<=(char*           target, IBaseStream& baseStream);
IBaseStream& operator<<=(long&           target, IBaseStream& baseStream);
IBaseStream& operator<<=(short&          target, IBaseStream& baseStream);
IBaseStream& operator<<=(char&           target, IBaseStream& baseStream);
IBaseStream& operator<<=(signed char&    target, IBaseStream& baseStream);
IBaseStream& operator<<=(unsigned long&  target, IBaseStream& baseStream);
IBaseStream& operator<<=(unsigned short& target, IBaseStream& baseStream);
IBaseStream& operator<<=(unsigned char&  target, IBaseStream& baseStream);
IBaseStream& operator<<=(float&          target, IBaseStream& baseStream);
IBaseStream& operator<<=(double&         target, IBaseStream& baseStream);

IBaseStream& operator>>=(const char*     source, IBaseStream& baseStream);
IBaseStream& operator>>=(long            source, IBaseStream& baseStream);
IBaseStream& operator>>=(short           source, IBaseStream& baseStream);
IBaseStream& operator>>=(char            source, IBaseStream& baseStream);
IBaseStream& operator>>=(signed char     source, IBaseStream& baseStream);
IBaseStream& operator>>=(unsigned long   source, IBaseStream& baseStream);
IBaseStream& operator>>=(unsigned short  source, IBaseStream& baseStream);
IBaseStream& operator>>=(unsigned char   source, IBaseStream& baseStream);
IBaseStream& operator>>=(float           source, IBaseStream& baseStream);
IBaseStream& operator>>=(double          source, IBaseStream& baseStream);


/*------------------------- IString Streaming Operators-----------------------*/
// These will be converted to IString members in the future.
IBaseStream& operator<<=(IString&        target, IBaseStream& baseStream);
IBaseStream& operator>>=(const IString&  source, IBaseStream& baseStream);


/*------------------------- Collection Streaming Operators--------------------*/
// These will be converted to Collection members in the future.
template<class AType>
IBaseStream& operator<<=(IACollection<AType>&, IBaseStream& s);

template<class AType>
IBaseStream& operator>>=(const IACollection<AType>&, IBaseStream& s);



template<class AType>
IBaseStream& operator<<=( IElemPointer<AType>&, IBaseStream& baseStream );

template<class AType>
IBaseStream& operator>>=( const IElemPointer<AType>&, IBaseStream& baseStream );



/*------------------------- Version Writing and Reading ----------------------*/
void
  writeVersion( IBaseStream& toWhere, const IBaseStream::VersionNumber version = 0 );

IBaseStream::VersionNumber
  readVersion(IBaseStream& fromWhere);

#ifndef __TEMPINC__
  #include <ibasstrm.c>
#endif

  #include <ibasstrm.inl>


#pragma pack()

#endif /* _IBASSTRM_ */

