#ifndef _IGBUNDLE_
#define _IGBUNDLE_
/*******************************************************************************
* FILE NAME: igbundle.hpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the classes:                                                *
*     IGraphicBundle                                                           *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#include <ibase.hpp>
#include <ipoint.hpp>
#include <icolor.hpp>


class IGraphicContext;

#pragma pack(4)

class IGraphicBundle : public IBase {
typedef IBase
  Inherited;

public:
/*------------------------------- Constructors -------------------------------*/
  IGraphicBundle  ( );

  IGraphicBundle  ( const IGraphicBundle&  graphicBundle  );

  IGraphicBundle  ( const IGraphicContext& graphicContext );

IGraphicBundle
  &operator=      ( const IGraphicBundle&  graphicBundle  );

  ~IGraphicBundle ( );

/*---------------------------- Draw Operation --------------------------------*/
enum DrawOperation {
  fill,
  frame,
  fillAndFrame
  };

IGraphicBundle
  &setDrawOperation   ( DrawOperation drawOperation = fillAndFrame );

DrawOperation
  drawOperation       ( ) const;

IGraphicBundle
 &resetDrawOperation  ( );

IBase::Boolean
  hasDrawOperation    ( ) const;

/*---------------------------- Color Attributes ------------------------------*/
IGraphicBundle
  &setPenColor          ( const IColor& penColor  ),
  &setFillColor         ( const IColor& fillColor ),
  &setBackgroundColor   ( const IColor& backColor );

IColor
  penColor              ( ) const,
  fillColor             ( ) const,
  backgroundColor       ( ) const;

IGraphicBundle
  &resetPenColor        ( ),
  &resetFillColor       ( ),
  &resetBackgroundColor ( );

IBase::Boolean
  hasPenColor           ( ) const,
  hasFillColor          ( ) const,
  hasBackgroundColor    ( ) const;

/*---------------------------------- MixMode ---------------------------------*/
enum MixMode {
  bitOr,
  overPaint,
  bitExclusiveOr,
  leaveAlone,
  bitAnd,
  subtract,
  maskSourceNot,
  black,
  notMergeSource,
  notXorSource,
  invert,
  mergeSourceNot,
  notCopySource,
  mergeNotSource,
  notMaskSource,
  white
#if ( IC_OBSOLETE <= IC_OBSOLETE_2 )
#ifndef __DIGRAPHS__
  ,
  or  = bitOr,
  xor = bitExclusiveOr,
  and = bitAnd
#endif
#endif
  };

IGraphicBundle
 &setMixMode           ( MixMode mixMode = overPaint );

MixMode
  mixMode              ( ) const;

IGraphicBundle
 &resetMixMode         ( );

IBase::Boolean
  hasMixMode           ( ) const;

/*---------------------------- BackgroundMixMode -----------------------------*/
enum BackgroundMixMode {
  backLeaveAlone,
  backOverPaint,
  backOr,
  backXor
  };

IGraphicBundle
 &setBackgroundMixMode   ( BackgroundMixMode backgroundMixMode =
                                                               backLeaveAlone );
BackgroundMixMode
  backgroundMixMode      ( ) const;

IGraphicBundle
 &resetBackgroundMixMode ( );

IBase::Boolean
  hasBackgroundMixMode   ( ) const;

/*----------------------------- Pen Width ------------------------------------*/
IGraphicBundle
 &setPenWidth            ( unsigned long lineWidth = 1 );

unsigned long
  penWidth               ( ) const;

IGraphicBundle
 &resetPenWidth          ( );

IBase::Boolean
  hasPenWidth            ( ) const;

/*------------------------------ Pen Type ------------------------------------*/
enum PenType {
  solid,
  dot,
  shortDash,
  dashDot,
  doubleDot,
  longDash,
  dashDoubleDot,
  alternate,
  invisible
  };

IGraphicBundle
  &setPenType              ( PenType penType = solid );

PenType
  penType                  ( ) const;

IGraphicBundle
 &resetPenType             ( );

IBase::Boolean
  hasPenType               ( ) const;

/*------------------------------ PenEndingStyle ------------------------------*/
enum PenEndingStyle {
  flat,
  square,
  rounded
  };

IGraphicBundle
 &setPenEndingStyle        ( PenEndingStyle penEndingStyle = flat );

PenEndingStyle
  penEndingStyle           ( ) const;

IGraphicBundle
 &resetPenEndingStyle      ( );

IBase::Boolean
  hasPenEndingStyle        ( ) const;


/*------------------------------ PenJoiningStyle -----------------------------*/
enum PenJoiningStyle {
  bevel,
  round,
  miter
  };

IGraphicBundle
  &setPenJoiningStyle      ( PenJoiningStyle penJoiningStyle = bevel );

PenJoiningStyle
  penJoiningStyle          ( ) const;

IGraphicBundle
 &resetPenJoiningStyle     ( );

IBase::Boolean
  hasPenJoiningStyle       ( ) const;

/*--------------------------- Pen and Fill Patterns --------------------------*/
static const unsigned long
  IC_IMPORTU dense1,
  IC_IMPORTU dense2,
  IC_IMPORTU dense3,
  IC_IMPORTU dense4,
  IC_IMPORTU dense5,
  IC_IMPORTU dense6,
  IC_IMPORTU dense7,
  IC_IMPORTU dense8,
  IC_IMPORTU vertical,
  IC_IMPORTU horizontal,
  IC_IMPORTU forewardDiag1,
  IC_IMPORTU forewardDiag2,
  IC_IMPORTU backwardDiag1,
  IC_IMPORTU backwardDiag2,
  IC_IMPORTU hollow,
  IC_IMPORTU filled,
  IC_IMPORTU halftone,
  IC_IMPORTU hatched,
  IC_IMPORTU hatchedDiag;

IGraphicBundle
  &setPenPattern           ( unsigned long penPattern  = filled ),
  &setFillPattern          ( unsigned long fillPattern = filled );

//  &setPattern            ( const IBitmapHandle& handle       );

unsigned long
  penPattern               ( ) const,
  fillPattern              ( ) const;

IGraphicBundle
 &resetPenPattern          ( ),
 &resetFillPattern         ( );

IBase::Boolean
  hasPenPattern            ( ) const,
  hasFillPattern           ( ) const;

/*--------------------------- Pattern Origin ---------------------------------*/
IGraphicBundle
 &setPatternOrigin        ( const IPoint& point = IPoint(0,0) );

IPoint
  patternOrigin           ( ) const;

IGraphicBundle
 &resetPatternOrigin      ( );

IBase::Boolean
  hasPatternOrigin        ( ) const;

/*------------------------------ Comparison ----------------------------------*/
IBase::Boolean
  operator== ( const IGraphicBundle& graphicBundle ),
  operator!= ( const IGraphicBundle& graphicBundle );

private:
/*--------------------------------- Private ----------------------------------*/
enum Attribute {
  pColor = 0x0001,
  fColor = 0x0002,
  bColor = 0x0004,
  pMix   = 0x0008,
  bMix   = 0x0010,
  pWidth = 0x0020,
  pType  = 0x0040,
  pEnd   = 0x0080,
  pJoin  = 0x0100,
  pPat   = 0x0200,
  fPat   = 0x0400,
  pOrg   = 0x0800,
  drawOp = 0x1000,
  All    = 0x1fff
  };

IColor
  fPen,
  fFill,
  fBack;
DrawOperation
  fDrawOp;
MixMode
  fPmix;
BackgroundMixMode
  fBmix;
unsigned long
  fPpat,
  fFpat;
PenType
  fPtype;
PenEndingStyle
  fPend;
PenJoiningStyle
  fPjoin;
unsigned long
  fWidth;
IPoint
  fOrg;
unsigned long
  fMask;

}; // IGraphicBundle

#pragma pack()

/*--------------------------------- INLINES ----------------------------------*/
  #include <igbundle.inl>

#endif // _IGBUNDLE_
