/*******************************************************************************
* FILE NAME: ilistcv2.c                                                        *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of private classes/functions         *
*   declared in both ilistcvw.hpp and icombovw.hpp                             *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

#include <inotifev.hpp>

#ifdef IC_USE_CB
  #define IC_CLASSNAME     ICollectionViewComboBox<Element,Collection>
  #define IC_CLASSDATA     fCollectionViewComboBoxData
  #define IC_BASECLASSNAME IBaseComboBox
#else
  #define IC_CLASSNAME     ICollectionViewListBox<Element,Collection>
  #define IC_CLASSDATA     fCollectionViewListBoxData
  #define IC_BASECLASSNAME IBaseListBox
#endif


template <class Element, class Collection>
  Collection*  IC_CLASSNAME:: items( ) const
{
  return  (Collection *)IC_CLASSDATA->fCollectionViewData->fCollectionObserver.collection();
}

template <class Element, class Collection>
  IC_CLASSNAME&   IC_CLASSNAME::
    setItems( Collection* collection )
{
  IC_CLASSDATA->fCollectionViewData->fCollectionObserver.setCollection( collection );

  return *this;
}

template <class Element, class Collection>
  IStringGenerator<Element>&  IC_CLASSNAME::
    stringGenerator ( )
{
  return  IC_CLASSDATA->fCollectionViewData->fGenerator;
}

template <class Element, class Collection>
  IStringGenerator<Element>&
    IC_CLASSNAME::
      setStringGenerator ( const IStringGenerator<Element>&  newGenerator )
{
  IC_CLASSDATA->fCollectionViewData->fGenerator =
     (IStringGenerator<Element>&)newGenerator;

  IC_CLASSDATA->saveSelectionState = true;
  elementsChanged ( );
  IC_CLASSDATA->saveSelectionState = false;

  return  IC_CLASSDATA->fCollectionViewData->fGenerator;
}


template <class Element, class Collection>
  IC_CLASSNAME& IC_CLASSNAME :: elementsChanged ( )
{
  ICursor*                  pCursor;
  Collection*               collection;
#ifdef IC_USE_CB
  ISequence<unsigned long>  selectionIndexes( IC_BASECLASSNAME::numberOfSelections());
#else
  long                      selection;
#endif

  collection = items( );
#ifdef __WINDOWS__
  if ( isWindowValid( this ) )
  {
#endif
  IWindow::enableUpdate( false );

  /* Don't forget the case where collection is NULL  */
  if ( collection )
  {
    if ( ( collection->numberOfElements() > 0 ) &&
         IC_CLASSDATA->saveSelectionState )
    {
#ifdef IC_USE_CB
      IC_BASECLASSNAME::Cursor    cursor( *this );
      forCursor( cursor )
        selectionIndexes.add( cursor.asIndex());
#else
      selection  = IC_BASECLASSNAME::selection();
#endif
    }
    else   // this is a removeall so deselect list box items
    {
      IC_BASECLASSNAME::deselectAll( );
    }
  }

  /*  Clear list box and repopulate w/ current string representations */
  deleteAll( );
  if ( collection )
  {
    pCursor = collection->newCursor();
    forCursor( *pCursor )
    {
      insert( ICollectionViewConstants::listEnd,
        IC_CLASSDATA->fCollectionViewData->fGenerator.stringFor( collection->elementAt( *pCursor ) ) );
    }
    delete pCursor;
  }

  if ( !IC_BASECLASSNAME::isEmpty( ) &&
       IC_CLASSDATA->saveSelectionState )
#ifdef IC_USE_CB
    for ( unsigned long i = selectionIndexes.numberOfElements(); i; --i )
       IC_BASECLASSNAME::select( selectionIndexes.elementAtPosition(i), true );
#else
    if ( selection != ICollectionViewConstants::listNone )
      IC_BASECLASSNAME::select( selection, true );
#endif

  IWindow::enableUpdate( );
  IWindow::show( );
#ifdef __WINDOWS__
  }
#endif
  return *this;
}


template <class Element, class Collection>
  IC_CLASSNAME&  IC_CLASSNAME::
    elementAdded ( unsigned long position, const Element&  element )
{
  insert( position - 1,
    IC_CLASSDATA->fCollectionViewData->fGenerator.stringFor( element ) );
  return *this;
}

template <class Element, class Collection>
  IC_CLASSNAME&  IC_CLASSNAME::
    elementDeleted ( unsigned long position )
{
  deleteItem( position - 1 );

  return *this;
}


template <class Element, class Collection>
  IC_CLASSNAME&  IC_CLASSNAME::
    elementChanged ( unsigned long  position, const Element&  element )
{
  IC_BASECLASSNAME::setItemText(
     position - 1,
     IC_CLASSDATA->fCollectionViewData->fGenerator.stringFor(element) );
  INotificationEvent  anEvent(
     IC_CLASSNAME::itemChangedId,
    *this,
     true ,
     IEventData((void *)&element) );
  notifyObservers( anEvent );
  return *this;
}

template <class Element, class Collection>
  IC_CLASSNAME&  IC_CLASSNAME :: collectionReplaced( )
{
#ifdef __WINDOWS__
  if ( isWindowValid( this ) )
  {
#endif
    INotificationEvent  anEvent( IC_CLASSNAME::itemsId,
                                *this,
                                 true,
                                 IEventData( (void *)items( ) ) );
    notifyObservers( anEvent );
#ifdef __WINDOWS__
  }
#endif

  return *this;
}


template <class Element, class Collection>
  unsigned long IC_CLASSNAME::
    selectedCollectionPosition ( )
{
  return IC_BASECLASSNAME::selection( ) + 1;
}

template <class Element, class Collection>
  IC_CLASSNAME&
    IC_CLASSNAME::
      select ( unsigned long collectionPosition ,
               Boolean       select )
{
  if ( (collectionPosition > 0 ) &&
       (collectionPosition <= IC_BASECLASSNAME::count()) )
    IC_BASECLASSNAME::select( collectionPosition - 1, select );

  return *this;
}

template <class Element, class Collection>
  IC_CLASSNAME&
    IC_CLASSNAME::
      deselect ( unsigned long collectionPosition )
{
  return select( collectionPosition, false );
}

template <class Element, class Collection>
  Element       IC_CLASSNAME::
    selectedElement ( ) const
{
  unsigned long                    selectedPosition;
  IPartOrderedCollection<Element>* aCollection = items( );
  IAutoPointer<ICursor>            pCursor( aCollection->newCursor( ), IINIT );

  selectedPosition = IC_BASECLASSNAME::selection( );
  if ( selectedPosition == ICollectionViewConstants::listNone )
  {
    ITHROWLIBRARYERROR( IC_NO_SELECTION,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  aCollection->setToPosition( selectedPosition + 1, *pCursor );

  return aCollection->elementAt( *pCursor );

}

template <class Element, class Collection>
  unsigned long IC_CLASSNAME::
    insert( unsigned long index,
            const char*   text )
{
   incrementChangeCount( );   // add may result in an invalid cursor.
   unsigned long anIndex = count();
   if ((long)anIndex > (long)index)
     anIndex = index;

   IEventResult evt = handle().sendEvent(
#ifndef IC_USE_CB
                               ICollectionViewConstants::listInsert,
#else
                               ICollectionViewConstants::comboInsert,
#endif
                               IEventParameter1(anIndex),
                               IEventParameter2((unsigned long) text));

   if ( ( evt.asLong() == ICollectionViewConstants::listMemoryError ) ||
        ( evt.asLong() == ICollectionViewConstants::listError ) )
     ITHROWGUIERROR("ICollectionViewConstants::listInsert");

   return evt.asUnsignedLong();  // Return index of inserted item.
}


template <class Element, class Collection>
  unsigned long IC_CLASSNAME::
    deleteItem ( unsigned long index )
{
   Boolean selected = isSelected( index );

   incrementChangeCount( );   // add may result in an invalid cursor.
   IEventResult evt = handle().sendEvent(
#ifndef IC_USE_CB
                               ICollectionViewConstants::listDelete,
#else
                               ICollectionViewConstants::comboDelete,
#endif
                               IEventParameter1(index),
                               IEventParameter2(0));

   if ( selected )
   {
     INotificationEvent  anEvent( IC_CLASSNAME::selectId,
                                 *this );
     notifyObservers( anEvent );
   }


   return evt.asUnsignedLong();
}

template <class Element, class Collection>
  IC_CLASSNAME& IC_CLASSNAME::
      deleteAll ( )
{
   IEventResult evt = handle().sendEvent(
#ifndef IC_USE_CB
                               ICollectionViewConstants::listDeleteAll,
#else
                               ICollectionViewConstants::comboDeleteAll,
#endif
                               IEventParameter1(0),
                               IEventParameter2(0));
   incrementChangeCount( );   // add may result in an invalid cursor.
   return *this;
}
