#ifndef _IWINDOW_
#define _IWINDOW_
/*******************************************************************************
* FILE NAME: iwindow.hpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Declaration of the classes:                                                *
*     IWindow                                                                  *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1996       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#include <ibitflag.hpp>
#include <ievent.hpp>
#include <ihandle.hpp>
#include <inotify.hpp>
#include <ipoint.hpp>
#include <ibidiset.hpp>

extern "C" {
  extern void* __stdcall _pfnwpICWinProc ( void*,
                                           unsigned long,
                                           void*,
                                           void* );
}

class IAcceleratorTable;
class IAccelTblHandle;
class IColor;
class IDMItemProvider;
class IException;
class IFont;
class IHandler;
class INotificationEvent;
class IRectangle;
class IString;
class IChildCursorData;
class IWindowBidiData;
class IWindowData;
class IWindowNotifyHandler;

#pragma pack(4)

class IWindow : public INotifier {
typedef INotifier
  Inherited;

public:
/*------------------------------- Constructors -------------------------------*/
  IWindow ( unsigned long        identifier,
            IWindow*             parent );
  IWindow ( const IWindowHandle& handle );

virtual
 ~IWindow ( );

/*---------------------------------- Styles ----------------------------------*/
INESTEDBITFLAGCLASSDEF0( Style, IWindow );
static const Style
  IC_IMPORTU noStyle,
  IC_IMPORTU visible,
  IC_IMPORTU disabled,
  IC_IMPORTU clipChildren,
#ifndef IC_WIN_FLAGNOP
  IC_IMPORTU clipToParent,
  IC_IMPORTU saveBits,
  IC_IMPORTU synchPaint,
#endif
  IC_IMPORTU clipSiblings;

virtual unsigned long
  convertToGUIStyle ( const IBitFlag& style,
                      Boolean         extendedOnly = false ) const;

/*---------------------------- Window Positioning ----------------------------*/
virtual IWindow
 &moveTo     ( const IPoint&     newPosition ),
 &sizeTo     ( const ISize&      newSize ),
 &moveSizeTo ( const IRectangle& newSizeAndPosition );

virtual IPoint
  position   ( ) const;

virtual ISize
  parentSize ( ) const,
  size       ( ) const;

virtual IRectangle
  nativeRect ( ) const,
  rect       ( ) const;

static ISize
  parentSize ( const IWindowHandle& windowHandle );

/*-------------------------------- Attributes --------------------------------*/
virtual IWindow
 &setOwner                  ( const IWindow* newOwner ),
 &setParent                 ( const IWindow* newParent ),
 &setId                     ( unsigned long  newIdentifier ),
 &enable                    ( Boolean        enableWindow = true ),
 &disable                   ( ),
 &setFocus                  ( );

IWindow
 &setHelpId                 ( unsigned long  helpTopicId );

IWindow
 *parent                    ( ) const;

virtual IWindow
 *owner                     ( ) const;

Boolean
  isEnabled                 ( ) const;

virtual Boolean
  hasFocus                  ( ) const,
  isValid                   ( ) const,
  isGroup                   ( ) const,
  isTabStop                 ( ) const,
  isFrameWindow             ( ) const;

static IWindow
 *desktopWindow             ( );

static IWindow
 *objectWindow              ( );

virtual IWindowHandle
  handle                    ( ) const;

static IWindow
 *windowWithHandle          ( const IWindowHandle& windowHandle,
                              Boolean              allThreads = true ),
 *windowWithOwner           ( unsigned long        identifier,
                              const IWindow*       owner,
                              Boolean              allThreads = true ),
 *windowWithParent          ( unsigned long        identifier,
                              const IWindow*       parent,
                              Boolean              allThreads = true );

static IWindowHandle
  handleWithParent          ( unsigned long        identifier,
                              const IWindowHandle& parent );

static Boolean
  isWindowValid             ( const IWindow*       window );

static IWindowHandle
  handleWithPointerCaptured ( );

virtual unsigned long
  id                        ( ) const;

unsigned long
  helpId                    ( ) const;


/*------------------------- Capturing Pointer Events -------------------------*/
virtual IWindow
 &capturePointer     ( Boolean capture = true ),
 &releasePointer     ( );

virtual Boolean
  hasPointerCaptured ( ) const;

/*--------------------------- Pointer Positioning ----------------------------*/
static void
  movePointerTo   ( const IPoint& position );

static IPoint
  pointerPosition ( );

/*------------------------------ Sibling Order -------------------------------*/
enum SiblingOrder {
  onTopOfSiblings,
  behindSiblings
  };

virtual IWindow
 &positionBehindSiblings ( ),
 &positionOnSiblings     ( ),
 &positionBehindSibling  ( const IWindowHandle& siblingWindow );

static void
 setDefaultOrdering      ( SiblingOrder         order );
static SiblingOrder
 defaultOrdering         ( );

/*--------------------------- Event Send and Post ----------------------------*/
enum EventType {
  command=1,
  systemCommand,
  control,
  help,
  character
  };

virtual const IWindow
 &postEvent ( unsigned long           eventId,
              const IEventParameter1& parm1 = 0,
              const IEventParameter2& parm2 = 0 ) const,
 &postEvent ( const IEvent&           event )     const,
 &postEvent ( EventType               eventType,
              const IEventParameter1& parm1 = 0,
              const IEventParameter2& parm2 = 0 ) const;

virtual IEventResult
  sendEvent ( unsigned long           eventId,
              const IEventParameter1& parm1 = 0,
              const IEventParameter2& parm2 = 0 ) const,
  sendEvent ( const IEvent&           event )     const,
  sendEvent ( EventType               eventType,
              const IEventParameter1& parm1 = 0,
              const IEventParameter2& parm2 = 0 ) const;

/*----------------------------- Window Painting ------------------------------*/
enum RefreshType {
  paintAll,
  immediate,
  paintAllImmediate
  };

virtual IWindow
 &show               ( Boolean           showWindow = true ),
 &hide               ( ),
 &enableUpdate       ( Boolean           enableWindow = true ),
 &disableUpdate      ( ),
 &refresh            ( RefreshType       type=paintAll ),
 &refresh            ( const IRectangle& invalidRectangle,
                       Boolean           immediate=false ),
 &showSourceEmphasis ( Boolean           show = true ),
 &hideSourceEmphasis ( );

Boolean
  isVisible          ( ) const,
  isShowing          ( ) const;

virtual IPresSpaceHandle
  presSpace          ( ) const;

virtual void
  releasePresSpace   ( const IPresSpaceHandle& presentationSpaceHandle ) const;

/*------------------------- Accelerator Key Support --------------------------*/
IWindow
 &setAcceleratorTable  ( const IAcceleratorTable* acceleratorTable ),
 &setAcceleratorHandle ( const IAccelTblHandle&   handle );

IAcceleratorTable
  acceleratorTable     ( ) const;
IAccelTblHandle
  acceleratorHandle    ( ) const;

/*---------------------------- Compound Controls -----------------------------*/

/*----------------------------- Object Deletion ------------------------------*/
IWindow
 &setAutoDeleteObject  ( Boolean autoDelete = true ),
 &setAutoDestroyWindow ( Boolean autoDestroy = false );

Boolean
  isAutoDeleteObject   ( ) const,
  isAutoDestroyWindow  ( ) const;

/*-------------------------- Drag and Drop Support ---------------------------*/
IDMItemProvider
 *itemProvider    ( ) const;

IWindow
 &setItemProvider ( IDMItemProvider* dragProvider );


class ChildCursor : public IVBase {
typedef IVBase
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
  ChildCursor ( IWindow& parent );
virtual
 ~ChildCursor ( );

/*----------------------------- Cursor Functions -----------------------------*/
virtual Boolean
  setToFirst ( ),
  setToNext  ( ),
  isValid    ( ) const;

virtual void
  invalidate ( );

private:
/*----------------------------- Hidden Functions -----------------------------*/
  ChildCursor ( const ChildCursor& cursor );
ChildCursor
 &operator =  ( const ChildCursor& cursor );

/*--------------------------------- Private ----------------------------------*/
friend class IWindow;

IChildCursorData
 *pCursorData;

IEnumHandle
  enumHandle;


IWindowHandle
  hwnd,
  hwndParent;
}; // IWindow::ChildCursor

/*--------------------------- Child Cursor Support ---------------------------*/
IWindowHandle
  childAt ( const ChildCursor& cursor ) const;

/*------------------------------ Canvas Support ------------------------------*/
virtual IWindowHandle
  defaultPushButton  ( ) const,
  matchForMnemonic   ( unsigned short character ) const;

enum Layout {
  windowCreated=1,
  colorChanged=2,
  sizeChanged=4,
  minimumSizeChanged=8,
  childMinimumSizeChanged=16,
  fontChanged=32,
  fontPropogated=64,
  layoutChanged=128,
  immediateUpdate=256,
  childWindowCreated=512,
  windowDestroyed=1024,
  childWindowDestroyed=2048
  };

virtual IWindow
 &setLayoutDistorted ( unsigned long layoutAttributesOn,
                       unsigned long layoutAttributesOff );

virtual Boolean
  isLayoutDistorted  ( unsigned long layoutAttribute ) const;

IWindow
 &resetMinimumSize   ( ),
 &setMinimumSize     ( const ISize& size );

ISize
  minimumSize        ( Boolean windowCalculatedSize = false ) const;

virtual IRectangle
  layoutAdjustment   ( ) const,
  visibleRectangle   ( ) const;

IWindow
 &enableMinimumSizeCaching    ( Boolean enableCaching = true ),
 &disableMinimumSizeCaching   ( );
Boolean
  isMinimumSizeCachingEnabled ( ) const;

/*------------------------------ Miscellaneous -------------------------------*/
static IPoint
  mapPoint ( const IPoint&        point,
             const IWindowHandle& from,
             const IWindowHandle& to );

/*------------------------------- Diagnostics --------------------------------*/
virtual IString
  asString    ( ) const,
  asDebugInfo ( ) const;

class ExceptionFn : public IVBase {
typedef IVBase
  Inherited;
public:
/*------------------------------- Constructors -------------------------------*/
virtual
 ~ExceptionFn ( );

/*--------------------------- Exception Processing ---------------------------*/
virtual Boolean
  handleException ( IException& dispatcherException,
                    IEvent&     exceptionEvent ) = 0;
}; // IWindow::ExceptionFn


/*--------------------------- Exception Processing ---------------------------*/
static IWindow::ExceptionFn
 *setExceptionFunction ( IWindow::ExceptionFn* exceptionFunction ),
 *exceptionFunction    ( );

virtual Boolean
  handleException      ( IException& dispatcherException,
                         IEvent&     exceptionEvent );

/*----------------------------- Color Functions ------------------------------*/
virtual IColor
  foregroundColor              ( ) const,
  backgroundColor              ( ) const;
#ifndef IC_WIN_FLAGNOP
virtual IColor
  borderColor                  ( ) const;
#endif

virtual IWindow
 &setForegroundColor           ( const IColor& color ),
 &setBackgroundColor           ( const IColor& color );
#ifndef IC_WIN_FLAGNOP
virtual IWindow
 &setBorderColor               ( const IColor& color );
#endif

#ifndef IC_MOTIFWIN_FLAGNOP
virtual IColor
  shadowColor                  ( ) const,
  disabledForegroundColor      ( ) const,
  disabledBackgroundColor      ( ) const,
  hiliteForegroundColor        ( ) const,
  hiliteBackgroundColor        ( ) const,
  activeColor                  ( ) const,
  inactiveColor                ( ) const;

virtual IWindow
 &setShadowColor               ( const IColor& color ),
 &setDisabledForegroundColor   ( const IColor& color ),
 &setDisabledBackgroundColor   ( const IColor& color ),
 &setHiliteForegroundColor     ( const IColor& color ),
 &setHiliteBackgroundColor     ( const IColor& color ),
 &setActiveColor               ( const IColor& color ),
 &setInactiveColor             ( const IColor& color );
#endif // !IC_MOTIFWIN_FLAGNOP
virtual IWindow
 &resetForegroundColor         ( ),
 &resetBackgroundColor         ( );
#ifndef IC_MOTIFWIN_FLAGNOP
virtual IWindow
 &resetBorderColor             ( ),
 &resetShadowColor             ( ),
 &resetDisabledForegroundColor ( ),
 &resetDisabledBackgroundColor ( ),
 &resetHiliteForegroundColor   ( ),
 &resetHiliteBackgroundColor   ( ),
 &resetActiveColor             ( ),
 &resetInactiveColor           ( );
#endif //!IC_MOTIFWIN_FLAGNOP

/*------------------------------ Font Functions ------------------------------*/
virtual IWindow
 &setFont       ( const IFont& font );

virtual IWindow
 &resetFont     ( );

virtual IFont
  font          ( ) const;

ISize
  characterSize ( ) const;

/*-------------------------- Observer Notification ---------------------------*/
virtual IWindow
 &enableNotification       ( Boolean enable = true ),
 &disableNotification      ( );

virtual Boolean
  isEnabledForNotification ( ) const;

virtual IWindow
 &notifyObservers          ( const INotificationEvent& event );

/*------------------------------ Notifications -------------------------------*/
static INotificationId const
  IC_IMPORTU sizeId,
  IC_IMPORTU positionId,
  IC_IMPORTU visibleId,
  IC_IMPORTU enableId,
  IC_IMPORTU deleteId,
  IC_IMPORTU focusId,
  IC_IMPORTU foregroundColorId,
  IC_IMPORTU backgroundColorId,
#ifndef IC_WIN_FLAGNOP
  IC_IMPORTU borderColorId,
  IC_IMPORTU shadowColorId,
  IC_IMPORTU disabledForegroundColorId,
  IC_IMPORTU disabledBackgroundColorId,
  IC_IMPORTU activeColorId,
  IC_IMPORTU inactiveColorId,
  IC_IMPORTU hiliteForegroundColorId,
  IC_IMPORTU hiliteBackgroundColorId,
#endif
  IC_IMPORTU fontId,
  IC_IMPORTU commandId,
  IC_IMPORTU systemCommandId;

/*---------------------- Event Handling Implementation -----------------------*/
Boolean
  dispatchRemainingHandlers ( IEvent& event,
                              Boolean callDefProc = true );
static void
   enableFastWindowWithHandle   ( Boolean enable = true );
static Boolean
   isFastWindowWithHandleEnabled( );

protected:
/*------------------------------ Implementation ------------------------------*/
virtual IWindowHandle
 create            ( unsigned long         id,
                     const char*           text,
                     unsigned long         style,
                     const char*           windowClass,
                     const IWindowHandle&  parent,
                     const IWindowHandle&  owner,
                     const IRectangle&     initRect,
                     const void*           ctlData,
                     const void*           presParams,
                     IWindow::SiblingOrder ordering = defaultOrdering(),
                     unsigned long         extendedStyle = 0 );


IWindow
 &addHandler       ( IHandler* newHandler ),
 &removeHandler    ( IHandler* oldHandler );

Boolean
  isPrimaryWindow  ( ) const;

IWindow
 &setColor         ( unsigned long colorArea,
                     const IColor& color ),
 &resetColor       ( unsigned long colorArea );

virtual IWindow
 &setStyle         ( unsigned long style ),
 &setExtendedStyle ( unsigned long extendedStyle );

virtual unsigned long
  style            ( ) const,
  extendedStyle    ( ) const;

IColor
  color            ( unsigned long colorArea ) const,
  color            ( unsigned long colorArea,
                     const IColor& defaultColor ) const;
unsigned short
  windowUShort     ( long index ) const;

unsigned long
  windowULong      ( long index ) const;

IWindow
 &setWindowData    ( long           index,
                     unsigned short dataValue ),
 &setWindowData    ( long           index,
                     unsigned long  dataValue );


/*-------------------------- Protected Constructors --------------------------*/
  IWindow ( );

/*---------------------------------- Layout ----------------------------------*/
virtual ISize
  calcMinimumSize  ( ) const;
IWindow
 &saveMinimumSize  ( const ISize& size );
ISize
  savedMinimumSize ( ) const;

/*---------------------- Event Handling Implementation -----------------------*/
IWindow
 &startHandlingEventsFor ( const IWindowHandle& windowHandle ),
 &startHandlingEventsFor ( unsigned long        identifier,
                           IWindow*             parent );
static void
  addToWindowSet         ( IWindow*             window,
                           const IWindowHandle& windowHandle ),
  removeFromWindowSet    ( IWindow*             window );


Boolean
  dispatch               ( IEvent& event );

IWindow
 &defaultProcedure       ( IEvent& event );

Boolean
  deleteIsInProcess      ( ) const;


/*------------------- Observer Notification Implementation -------------------*/
virtual IWindow
 &addObserver            ( IObserver&             observer,
                           const IEventData&      userData = IEventData(0) ),
 &removeObserver         ( IObserver&             observer ),
 &removeAllObservers     ( );

IObserverList
 &observerList           ( ) const;

IWindow
 &setNotificationHandler ( IWindowNotifyHandler*  notifyHandler );

IWindowNotifyHandler
 *notificationHandler    ( ) const;

virtual IWindow
 &notifyObservers        ( const INotificationId& notification );

/*-------------------------- Drag and Drop Support ---------------------------*/
virtual Boolean
  isDragStarting ( IEvent &event );

private:
/*----------------------------- Hidden Functions -----------------------------*/
IWindow     ( const IWindow& window );
IWindow
 &operator= ( const IWindow& window );

/*--------------------------------- Private ----------------------------------*/
friend class IHandle;
friend class IHandler;
friend class IFileDialog;
friend class IFontDialog;


friend class IFont;
friend class IDMDragHandler;
friend class IMessageHandler;

enum State {
  autoDeleteObject=1,
  autoDestroyWindow=2,
  deleteInProcess=4,
  primaryWindow=8,
  addedToList=16,
  dispatchInProcess=32,
  needsDelete=64,
  updateRHtInProgress=128,
  neverBeenShown=256,
  forgetBitGravity=512,
  fontWasSet=1024,
  bkgndColorWasSet=2048,
  frgndColorWasSet=4096,
  cacheMinimumSize=8192
  };

friend unsigned long const
 &key             ( IWindow* const& window );


friend void* __stdcall
  _pfnwpICWinProc ( void*         hwnd,
                    unsigned long ulMsg,
                    void*         mp1,
                    void*         mp2 );


static void
  cleanUpHandler          ( IHandler& handler );


static Boolean
  useRealCoordinates      ( const IWindow* window );

IWindow
 &setDeleteInProcess      ( ),
 &cleanUp                 ( );

Boolean
  dispatchCleanup         ( IEvent& event,
                            Boolean nested );

IWinProc
 *setDefaultProcedure     ( IWinProc* newDefProc );

IWindowData
 *pWindowData;

unsigned long
  hwnd;

/*----------------------------- Obsolete Members -----------------------------*/
#if (IC_OBSOLETE <= IC_OBSOLETE_2)
public:
class BidiSettings;

Boolean
  isBidiSupported ( ) const;

enum BidiTextType {
  visual,
  implicit
  };

enum BidiTextOrientation {
  textLeftToRight,
  textRightToLeft,
  textContextual
  };

enum BidiLayout {
  layoutLeftToRight,
  layoutRightToLeft
  };

enum BidiNumeralType {
  arabic,
  asStored,
  national,
  contextual
  };

enum BidiTextShape {
  displayShaped,
  saveShaped,
  nominalShape,
  initialShape,
  middleShape,
  finalShape,
  isolatedShape
  };

virtual IWindow
 &applyBidiSettings ( const BidiSettings& settings,
                      Boolean             childInherit = true,
                      Boolean             refresh      = true );

class BidiSettings : public IVBase {
typedef IVBase
  Inherited;
public:
  BidiSettings ( const IWindow& window );

virtual
 ~BidiSettings ( );

virtual BidiSettings
 &setTextType                   ( BidiTextType        textType ),
 &setTextOrientation            ( BidiTextOrientation textOrientation ),
 &setWindowLayout               ( BidiLayout          windowLayout ),
 &setNumeralDisplay             ( BidiNumeralType     numeralDisplay ),
 &enableSymmetricSwapping       ( Boolean             enable = true ),
 &disableSymmetricSwapping      ( ),
 &enableWordByWordReordering    ( Boolean             enable = true ),
 &disableWordByWordReordering   ( ),
 &setTextShape                  ( BidiTextShape       textShape );

BidiTextType
  textType                      ( ) const;
BidiTextOrientation
  textOrientation               ( ) const;
BidiLayout
  windowLayout                  ( ) const;
BidiNumeralType
  numeralDisplay                ( ) const;
Boolean
  isSymmetricSwappingEnabled    ( ) const;
Boolean
  isWordByWordReorderingEnabled ( ) const;
BidiTextShape
  textShape                     ( ) const;

private:
friend class IWindow;
  BidiSettings ( const BidiSettings& settings );
BidiSettings
 &operator=    ( const BidiSettings& settings );
IBidiSettings
 *fBidiSettings;
}; // IWindow::BidiSettings

private:
friend class IWindow::BidiSettings;
#endif // IC_OBSOLETE

}; // IWindow

#pragma pack()

  #include <iwindow.inl>

#endif /* _IWINDOW_ */
