/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 05  1.18.1.7  source/albert/graph2d/imatrix.cpp, 2d, ioc.v400, 990114  
/*==========================================
||
||  File:   Matrix.C
||
||  What:   Defines IGrafMatrix class
||
||  Change History:
||
*/

#include <imatrix.hpp>
#include <inumeric.hpp>
#ifdef GRAPH2D_DEBUG
#include <stdio.h>
#endif
#include <igarea2d.hpp>
#include <ipt2darr.hpp>
#include <grassert.hpp>
#include <idatstrm.hpp>
#include <grtype.hpp>
#include <iprimlck.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _IMATRIX_CPP_
#include <ipagetun.h>
#endif

#include <isynchr.hpp>

/*================
||
||  General Declarations
||
*/

const long kDegrees = 0;
const long kCenterX = 1;
const long kCenterY = 2;
const long kSin = 3;
const long kCos = 4;

/*================
||
||  Forward Declarations
||
*/

void FirstCycle(GCoordinate& degrees);

/*================
||
|| Constants
||
*/

// matrixTypes for PreMatrixMult(above)

const kAffineAffine = 0;
const kGenericGeneric = 3;

struct IGrafMatrixTimeStamp {
	IGrafMatrixTimeStamp();
	unsigned long fGrafMatrixTimeStamp;
};

 IGrafMatrixTimeStamp::IGrafMatrixTimeStamp()
  : fGrafMatrixTimeStamp(0)
{
}

static IGrafMatrixTimeStamp* gStaticGrafMatrixTimeStamp = 0;
static IGrafMatrixTimeStamp *staticGrafMatrixTimeStamp()
{
	if (!gStaticGrafMatrixTimeStamp) {
		IPrimalLock lock;
		if (!gStaticGrafMatrixTimeStamp) {
			gStaticGrafMatrixTimeStamp = new IGrafMatrixTimeStamp;
			++(gStaticGrafMatrixTimeStamp->fGrafMatrixTimeStamp);
		}
	}
	else {
		IPrimalLock lock;
		++(gStaticGrafMatrixTimeStamp->fGrafMatrixTimeStamp);
	}
	return gStaticGrafMatrixTimeStamp;
}
		
const unsigned long& IGrafMatrix::staticTimeStamp()
{
	return staticGrafMatrixTimeStamp()->fGrafMatrixTimeStamp;
}	


void IGrafMatrix::updateTimeStamp()
{
    fTimeStamp = IGrafMatrix::staticTimeStamp();
}

void IGrafMatrix::resetTimeStamp()
{
    fTimeStamp = (unsigned long) 0;
}

/*================
||
||  IGrafMatrix::setTimeStamp
||
*/

void IGrafMatrix::setTimeStamp(
    unsigned long newTimeStamp)
{
    fTimeStamp = newTimeStamp;
}

/*================
||
||  Forward Declarations
||
*/

void FirstCycle(GCoordinate& degrees) {
    degrees = Remainder(degrees, 360.0);
    if (degrees < 0.0) degrees += 360.0;
}

/*================
||
|| Constants
||
*/

static ISynchronized<IGrafMatrix*> identityMatrix;

const IGrafMatrix& IGrafMatrix::identity()
{
    IGrafMatrix* matrix = identityMatrix.get();
    if (!matrix) {
        matrix = synchronizedInit(identityMatrix, new IGrafMatrix);
    }
    return *matrix;
}

/*================
||
|| Constructors and destructors
||
*/

IGrafMatrix::IGrafMatrix()
{
    resetTimeStamp();
    fAccelerator = kPrIdentity;
    fMatrix = 0 /*NIL*/;
}

IGrafMatrix::IGrafMatrix(const IGrafMatrix& matrix)
{
    fMatrix = 0 /*NIL*/;
    *this = matrix;
}

IGrafMatrix::IGrafMatrix(const GCoordinate matrix [])
{
    fAccelerator = GetAccelerator(matrix); // should be const in GetAccelerator
    fMatrix = 0 /*NIL*/;
    if (fAccelerator==kPrIdentity)
        resetTimeStamp();
    else {
        if (fAccelerator == kPrTranslate) {
            fTranslateX = matrix[kTranslateX];
            fTranslateY = matrix[kTranslateY];
        }
        else {
            const GCoordinate* src = matrix;
            GCoordinate* dest = fMatrix = new (::kSameHeap, this) GCoordinate[9];

            *dest++ = *src++;
            *dest++ = *src++;
            *dest++ = *src++;
            *dest++ = *src++;
            *dest++ = *src++;
            *dest++ = *src++;
            *dest++ = *src++;
            *dest++ = *src++;
            *dest = *src;
        }
        fRotateAccelerator = kSlowRotate;
        updateTimeStamp();
    }
}

IGrafMatrix::IGrafMatrix(
    const GCoordinate scaleX,
    const GCoordinate shearY,
    const GCoordinate perspectiveX,
    const GCoordinate shearX,
    const GCoordinate scaleY,
    const GCoordinate perspectiveY,
    const GCoordinate translateX,
    const GCoordinate translateY,
    const GCoordinate homogeneous)
{
    GCoordinate* mat = fMatrix = new (::kSameHeap, this) GCoordinate[9];
    *mat++ = scaleX;
    *mat++ = shearY;
    *mat++ = perspectiveX;
    *mat++ = shearX;
    *mat++ = scaleY;
    *mat++ = perspectiveY;
    *mat++ = translateX;
    *mat++ = translateY;
    *mat++ = homogeneous;
    fAccelerator = GetAccelerator(fMatrix);
    if (fAccelerator==kPrIdentity) {
        resetTimeStamp();
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
    else {
        if (fAccelerator == kPrTranslate) {
            fTranslateX = fMatrix[kTranslateX];
            fTranslateY = fMatrix[kTranslateY];
            delete [] fMatrix; fMatrix = 0 /*NIL*/;
        }
        else
            fRotateAccelerator = kSlowRotate;
        updateTimeStamp();
    }
}

IGrafMatrix::IGrafMatrix(const IGPoint2D& delta)
{
    fMatrix = 0 /*NIL*/;
    if (delta.fX != 0.0 || delta.fY != 0.0)
    {
        fTranslateX = delta.fX;
        fTranslateY = delta.fY;
        fAccelerator = kPrTranslate;
        updateTimeStamp();
    }
    else
    {
        fAccelerator = kPrIdentity;
        resetTimeStamp();
    }
}

IGrafMatrix::IGrafMatrix(
    const IGPoint2D& scale,
    const IGPoint2D& centerOfScale)
{
    if (scale.fX != 1.0 || scale.fY != 1.0)
    {
        GCoordinate* dest = fMatrix = new (::kSameHeap, this) GCoordinate[9];
        *dest++ = scale.fX;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = scale.fY;
        *dest++ = 0.0;
        *dest++ = centerOfScale.fX*(1.0-scale.fX);
        *dest++ = centerOfScale.fY*(1.0-scale.fY);
        *dest = 1.0;
        fAccelerator = kPrScale;
        updateTimeStamp();
    }
    else
    {
        fMatrix = 0 /*NIL*/;
        fAccelerator = kPrIdentity;
        resetTimeStamp();
    }
}

IGrafMatrix::IGrafMatrix(
    const GDegrees& angle,
    const IGPoint2D& centerOfRotate)
{
    updateTimeStamp();
    fAccelerator = kPrRotate;
    GCoordinate fcDegrees = angle;
    FirstCycle(fcDegrees);
    fRotateAccelerator = GetFastRotate(fcDegrees);
    if (fRotateAccelerator != kRotate0) {
        fMatrix = new (::kSameHeap, this) GCoordinate[9];
        fMatrix[kDegrees] = fcDegrees;
        fMatrix[kCenterX] = centerOfRotate.fX;
        fMatrix[kCenterY] = centerOfRotate.fY;
        ComputeCosAndSin();
    } else {
        fMatrix = 0 /*NIL*/;
        fAccelerator = kPrIdentity;
        resetTimeStamp();
    }
}

void IGrafMatrix::ComputePerspectiveMap(
    const IGQuadrilateral& fromQuadrilateral,
    const IGQuadrilateral& toQuadrilateral)
{
    IGPoint2D from0;
    IGPoint2D from1;
    IGPoint2D from2;
    IGPoint2D from3;
    fromQuadrilateral.points(from0,from1,from2,from3);

    fMatrix = new (::kSameHeap, this) GCoordinate[9];
    if (from0.fY==from1.fY && from2.fY==from3.fY
        &&  from1.fX==from2.fX && from3.fX==from0.fX)
    {

        // === from 0-1 and 2-3 are horz, 1-2 and 3-0 are vert ===

        RectToQuad(IGRect2D(from0, from2), toQuadrilateral, fMatrix);
        fAccelerator = kPrGeneric;
        normalize();
        fAccelerator = GetAccelerator(fMatrix);
    }
    else
    {
        if (from0.fX==from1.fX && from2.fX==from3.fX
        && from1.fY==from2.fY && from3.fY==from0.fY)
        {

                // === from 0-1 and 2-3 are vert, 1-2 and 3-0 are horz ===
        IGPoint2D to0;
        IGPoint2D to1;
        IGPoint2D to2;
        IGPoint2D to3;
        toQuadrilateral.points(to0, to1, to2, to3);
        IGQuadrilateral screenquad(to1, to2, to3, to0);
        RectToQuad(
                IGRect2D(from1, from3), screenquad, fMatrix);
        fAccelerator = kPrGeneric;
        normalize();
        fAccelerator = GetAccelerator(fMatrix);
        }
        else {
        GCoordinate mat [9];
        UnitToQuad(toQuadrilateral, mat);
        IGrafMatrix temp(mat);
        temp.fAccelerator = kPrGeneric;
        temp.normalize();
        UnitToQuad(fromQuadrilateral, fMatrix);
        fAccelerator = kPrGeneric;
        normalize();
        fAccelerator = GetAccelerator(fMatrix);
        invert();
        concatWith(temp);
        }
    }
    if (fAccelerator==kPrIdentity) {
        resetTimeStamp();
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
    else {
        if (fAccelerator == kPrTranslate) {
            fTranslateX = fMatrix[kTranslateX];
            fTranslateY = fMatrix[kTranslateY];
            delete [] fMatrix; fMatrix = 0 /*NIL*/;
        }
        else
            fRotateAccelerator = kSlowRotate;
        updateTimeStamp();
    }
}

IGrafMatrix::IGrafMatrix(
    const IGQuadrilateral& fromQuadrilateral,
    const IGQuadrilateral& toQuadrilateral)
{
    if (!fromQuadrilateral.isDegenerate() && !toQuadrilateral.isDegenerate())
        ComputePerspectiveMap(fromQuadrilateral,toQuadrilateral);
    else
        TransformException(IGraphicException::kSingularTransform);
}


#ifdef GRAPH2D_DEBUG

//void PrintQuad(const IGQuadrilateral& quad)
//{
//  IGPoint2D pt0;
//  IGPoint2D pt1;
//  IGPoint2D pt2;
//  IGPoint2D pt3;
//  quad.points(pt0,pt1,pt2,pt3);
//  printf("(%f,%f), ", pt0.fX, pt0.fY);
//  printf("(%f,%f), ", pt1.fX, pt1.fY);
//  printf("(%f,%f), ", pt2.fX, pt2.fY);
//  printf("(%f,%f)\n", pt3.fX, pt3.fY);
//}

#endif


void IGrafMatrix::UnitToQuad(
    const IGQuadrilateral& unitRectToQuad,
    GCoordinate dest [])
{
    IGPoint2D to0;
    IGPoint2D to1;
    IGPoint2D to2;
    IGPoint2D to3;
    unitRectToQuad.points(to0,to1,to2,to3);
    GCoordinate px = to0.fX - to1.fX + to2.fX - to3.fX;
    GCoordinate py = to0.fY - to1.fY + to2.fY - to3.fY;

    if (!px && !py) {
        dest[IGrafMatrix::kScaleX] = to1.fX - to0.fX;
        dest[IGrafMatrix::kShearX] = to2.fX - to1.fX;
        dest[IGrafMatrix::kTranslateX] = to0.fX;
        dest[IGrafMatrix::kShearY] = to1.fY - to0.fY;
        dest[IGrafMatrix::kScaleY] = to2.fY - to1.fY;
        dest[IGrafMatrix::kTranslateY] = to0.fY;
        dest[IGrafMatrix::kPerspectiveX] = 0.;
        dest[IGrafMatrix::kPerspectiveY] = 0.;
        dest[IGrafMatrix::kHomogeneous] = 1.;
        }
    else {                  /* projective */
        GCoordinate dx1 = to1.fX - to2.fX;
        GCoordinate dx2 = to3.fX - to2.fX;
        GCoordinate dy1 = to1.fY - to2.fY;
        GCoordinate dy2 = to3.fY - to2.fY;
        GCoordinate det = dx1*dy2 - dx2*dy1;
        if (det != 0.0)
        {
            dest[IGrafMatrix::kPerspectiveX] = (px*dy2 - dx2*py)/det;
            dest[IGrafMatrix::kPerspectiveY] = (dx1*py - px*dy1)/det;
            dest[IGrafMatrix::kScaleX] =
                to1.fX-to0.fX
                +dest[IGrafMatrix::kPerspectiveX]*to1.fX;
            dest[IGrafMatrix::kShearX] =
                to3.fX-to0.fX
                +dest[IGrafMatrix::kPerspectiveY]*to3.fX;
            dest[IGrafMatrix::kTranslateX] = to0.fX;
            dest[IGrafMatrix::kShearY] =
                to1.fY-to0.fY
                +dest[IGrafMatrix::kPerspectiveX]*to1.fY;
            dest[IGrafMatrix::kScaleY] =
                to3.fY-to0.fY
                +dest[IGrafMatrix::kPerspectiveY]*to3.fY;
            dest[IGrafMatrix::kTranslateY] = to0.fY;
            dest[IGrafMatrix::kHomogeneous] = 1.0;
        }
        else
            TransformException(IGraphicException::kSingularTransform);
    }
}

void IGrafMatrix::RectToQuad(
    const IGRect2D& fromRect, const IGQuadrilateral& toQuad,
    GCoordinate dest [])
{
    GCoordinate du = fromRect.fRight-fromRect.fLeft;
    GCoordinate dv = fromRect.fBottom-fromRect.fTop;

    if (du&&dv)
    {
        // define mapping from unit uv square to xy quadrilateral
        UnitToQuad(toQuad, dest);

        // concatenate transform from uv rectangle(u0, v0, u1, v1)
        // to unit square
        dest[IGrafMatrix::kScaleX] /= du;
        dest[IGrafMatrix::kShearX] /= dv;
        dest[IGrafMatrix::kTranslateX] -=
            dest[IGrafMatrix::kScaleX]*fromRect.fLeft
            + dest[IGrafMatrix::kShearX]*fromRect.fTop;
        dest[IGrafMatrix::kShearY] /= du;
        dest[IGrafMatrix::kScaleY] /= dv;
        dest[IGrafMatrix::kTranslateY] -=
            dest[IGrafMatrix::kShearY]*fromRect.fLeft
            + dest[IGrafMatrix::kScaleY]*fromRect.fTop;
        dest[IGrafMatrix::kPerspectiveX] /= du;
        dest[IGrafMatrix::kPerspectiveY] /= dv;
        dest[IGrafMatrix::kHomogeneous] -=
            dest[IGrafMatrix::kPerspectiveX]*fromRect.fLeft
            + dest[IGrafMatrix::kPerspectiveY]*fromRect.fTop;
    }
    else
        TransformException(IGraphicException::kSingularTransform);
}

IGrafMatrix::~IGrafMatrix()
{
    delete [] fMatrix;
    fMatrix = 0 /*NIL*/;
}

/*================
||
|| transformPoint and transformPoint
||
*/

IGRPoint2D IGrafMatrix::transformPoint(const IGRPoint2D& source) const
{
    IGRPoint2D destination;

    switch (fAccelerator) {
    case kPrIdentity:
        // == Identity Transform Rational Points ===
        destination.fX = source.fX;
        destination.fY = source.fY;
        destination.fW = source.fW;
        break;

    case kPrTranslate:
        // == Translate Rational Points ===
        destination.fX = source.fX + fTranslateX*source.fW;
        destination.fY = source.fY + fTranslateY*source.fW;
        destination.fW = source.fW;
        break;

    case kPrRotate:
        // == Rotate Rational Points ===
        switch (fRotateAccelerator) {
        case kRotate0:
            destination.fX = source.fX;
            destination.fY = source.fY;
            break;

        case kRotate90:
            destination.fX = -source.fY + source.fW*(fMatrix[kCenterX] + fMatrix[kCenterY]);
            destination.fY = source.fX + source.fW*(-fMatrix[kCenterX] + fMatrix[kCenterY]);
            break;

        case kRotate180:
            destination.fX = -source.fX + 2.0*source.fW*fMatrix[kCenterX];
            destination.fY = -source.fY + 2.0*source.fW*fMatrix[kCenterY];
            break;

        case kRotate270:
            destination.fX = source.fY + source.fW*(fMatrix[kCenterX] - fMatrix[kCenterY]);
            destination.fY = -source.fX + source.fW*(fMatrix[kCenterX] + fMatrix[kCenterY]);
            break;

        case kSlowRotate:
            // == Arbitrary Rotate Rational Points ===
            GCoordinate oneMinus = 1.0 - fMatrix[kCos];
            GCoordinate translateX = fMatrix[kCenterX]*oneMinus + fMatrix[kCenterY]*fMatrix[kSin];
            GCoordinate translateY = fMatrix[kCenterY]*oneMinus - fMatrix[kCenterX]*fMatrix[kSin];
            destination.fX = fMatrix[kCos]*source.fX - fMatrix[kSin]*source.fY + source.fW*translateX;
            destination.fY = fMatrix[kSin]*source.fX + fMatrix[kCos]*source.fY + source.fW*translateY;
            break;
        }
        destination.fW = source.fW;
        break;

    case kPrScale:
        // == scale Rational Points ===
        destination.fX = source.fX*fMatrix[kScaleX] + source.fW*fMatrix[kTranslateX];
        destination.fY = source.fY*fMatrix[kScaleY] + source.fW*fMatrix[kTranslateY];
        destination.fW = source.fW;
        break;

    case kPrAffine: {
        // == Affine Transform Rational Points ===
        const GCoordinate* mat = fMatrix;
        destination.fX = *mat++*source.fX;
        destination.fY = *mat++*source.fX;
        mat++;
        destination.fX += *mat++*source.fY;
        destination.fY += *mat++*source.fY;
        mat++;
        destination.fX += *mat++*source.fW;
        destination.fY += *mat*source.fW;
        destination.fW = source.fW;
        }
        break;

    case kPrGeneric: {
        // == Generic Transform Rational Points ===
        // Full matrix multiplication
        const GCoordinate* mat = fMatrix;
        destination.fX  = *mat++*source.fX;
        destination.fY  = *mat++*source.fX;
        destination.fW  = *mat++*source.fX;
        destination.fX += *mat++*source.fY;
        destination.fY += *mat++*source.fY;
        destination.fW += *mat++*source.fY;
        destination.fX += *mat++*source.fW;
        destination.fY += *mat++*source.fW;
        destination.fW += *mat++*source.fW;
        }
        break;
    }

    return destination;
}

/*================
||
|| IGrafMatrix::untransformPoint
||
*/

IGRPoint2D IGrafMatrix::untransformPoint(const IGRPoint2D& source) const
{
    IGRPoint2D destination;

    switch (fAccelerator) {
    case kPrIdentity:
        // == Identity UnTransform Rational Point ===
        {
            destination.fX = source.fX;
            destination.fY = source.fY;
            destination.fW = source.fW;
        }
        break;
    case kPrTranslate:
        {
            // == UnTranslate Rational Point ===
            destination.fX = source.fX - source.fW*fTranslateX;
            destination.fY = source.fY - source.fW*fTranslateY;
            destination.fW = source.fW;
        }
        break;
    case kPrScale:
        // == UnScale Rational Point ===
        if (fMatrix[kScaleX] != 0.0 || fMatrix[kScaleY] != 0.0) {
            destination.fX = (source.fX - source.fW*fMatrix[kTranslateX])/fMatrix[kScaleX];
            destination.fY = (source.fY - source.fW*fMatrix[kTranslateY])/fMatrix[kScaleY];
            destination.fW = source.fW;
        }
        else
            TransformException(IGraphicException::kSingularTransform);
        break;
    case kPrRotate: {
        // == Unrotate Rational Point ===
        destination.fW = source.fW;
        switch (fRotateAccelerator) {
            case kRotate0:
                {
                    destination.fX = source.fX;
                    destination.fY = source.fY;
                }
                break;
            case kRotate90:
                {
                    destination.fX =  source.fY + source.fW*(fMatrix[kCenterX] - fMatrix[kCenterY]);
                    destination.fY = -source.fX + source.fW*(fMatrix[kCenterX] + fMatrix[kCenterY]);
                }
                break;
            case kRotate180:
                destination.fX = -source.fX + 2.0*destination.fW*fMatrix[kCenterX];
                destination.fY = -source.fY + 2.0*destination.fW*fMatrix[kCenterY];
                break;
            case kRotate270:
                {
                    destination.fX = -source.fY + destination.fW*(fMatrix[kCenterX] + fMatrix[kCenterY]);
                    destination.fY = source.fX + destination.fW*(-fMatrix[kCenterX] + fMatrix[kCenterY]);
                }
                break;
            case kSlowRotate:
                // == Arbitrary Unrotate Rational Point ===
                {
                    GCoordinate cos = fMatrix[kCos];
                    GCoordinate sin = fMatrix[kSin];
                    GCoordinate oneMinus = 1.0 - cos;
                    GCoordinate oldX = source.fX;
                    destination.fX = cos*oldX + sin*source.fY +
                        source.fW*(fMatrix[kCenterX]*oneMinus - fMatrix[kCenterY]*sin);
                    destination.fY = -sin*oldX + cos*source.fY +
                        source.fW*(fMatrix[kCenterY]*oneMinus + fMatrix[kCenterX]*sin);
                }
                break;
        }
        break;
        }
    case kPrAffine: {
        // == Affine UnTransform Rational Point ===
        GCoordinate det =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];
        if (det != 0.0) {
            GCoordinate x = source.fX;
            destination.fX =
               (x*fMatrix[kScaleY] - source.fY*fMatrix[kShearX]
                + source.fW*(-fMatrix[kTranslateX]*fMatrix[kScaleY] + fMatrix[kTranslateY]*fMatrix[kShearX]))/det;
            destination.fY =
               (-x*fMatrix[kShearY] + source.fY*fMatrix[kScaleX]
                + source.fW*(fMatrix[kTranslateX]*fMatrix[kShearY] - fMatrix[kTranslateY]*fMatrix[kScaleX]))/det;
            destination.fW = source.fW;
        }
        else
            TransformException(IGraphicException::kSingularTransform);
        break;
        }
    case kPrGeneric: {
        // == Generic UnTransform Rational Point ===
        // kPrGeneric inverse = classical adjoint

        GCoordinate newTranslateX =
            fMatrix[kShearX]*fMatrix[kTranslateY] -
            fMatrix[kScaleY]*fMatrix[kTranslateX];

        GCoordinate newTranslateY =
            fMatrix[kShearY]*fMatrix[kTranslateX] -
            fMatrix[kScaleX]*fMatrix[kTranslateY];

        GCoordinate newHomogeneous =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];

        GCoordinate determinant =
            fMatrix[kPerspectiveX]*newTranslateX +
            fMatrix[kPerspectiveY]*newTranslateY +
            fMatrix[kHomogeneous]*newHomogeneous;

        if (!determinant)
            TransformException(IGraphicException::kSingularTransform);

        GCoordinate x =
            source.fX*(
                fMatrix[kScaleY]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveY]*fMatrix[kTranslateY]) +
            source.fY*(
                fMatrix[kPerspectiveY]*fMatrix[kTranslateX] -
                fMatrix[kShearX]*fMatrix[kHomogeneous]) +
            source.fW*newTranslateX;

        GCoordinate y =
            source.fX*(
                fMatrix[kPerspectiveX]*fMatrix[kTranslateY] -
                fMatrix[kShearY]*fMatrix[kHomogeneous]) +
            source.fY*(
                fMatrix[kScaleX]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveX]*fMatrix[kTranslateX]) +
            source.fW*newTranslateY;

        GCoordinate w =
            source.fX*(
                fMatrix[kShearY]*fMatrix[kPerspectiveY] -
                fMatrix[kPerspectiveX]*fMatrix[kScaleY]) +
            source.fY*(
                fMatrix[kPerspectiveX]*fMatrix[kShearX] -
                fMatrix[kScaleX]*fMatrix[kPerspectiveY]) +
            source.fW*newHomogeneous;

            destination.fX = x/determinant;
            destination.fY = y/determinant;
            destination.fW = w/determinant;
    }
    }
    return destination;
}

/*================
||
|| IGrafMatrix::untransformPoint
||
*/

IGPoint2D IGrafMatrix::untransformPoint(const IGPoint2D& source) const
{
    IGPoint2D destination;
    switch (fAccelerator) {
    case kPrIdentity:
        {
            destination.fX = source.fX;
            destination.fY = source.fY;
        }
        break;
    case kPrTranslate:
        destination.fX = source.fX - fTranslateX;
        destination.fY = source.fY - fTranslateY;
        break;
    case kPrScale:
        if (fMatrix[kScaleX] != 0.0 || fMatrix[kScaleY] != 0.0) {
            destination.fX = (source.fX - fMatrix[kTranslateX])/fMatrix[kScaleX];
            destination.fY = (source.fY - fMatrix[kTranslateY])/fMatrix[kScaleY];
        }
        else
            TransformException(IGraphicException::kSingularTransform);
        break;
    case kPrRotate: {
        switch (fRotateAccelerator) {
            case kRotate0:
                {
                    destination.fX = source.fX;
                    destination.fY = source.fY;
                }
                break;
            case kRotate90:
                {
                    destination.fX = source.fY + fMatrix[kCenterX] - fMatrix[kCenterY];
                    destination.fY = -source.fX + fMatrix[kCenterX] + fMatrix[kCenterY];
                }
                break;
            case kRotate180:
                destination.fX = -source.fX + 2.0*fMatrix[kCenterX];
                destination.fY = -source.fY + 2.0*fMatrix[kCenterY];
                break;
            case kRotate270:
                {
                    destination.fX = -source.fY + fMatrix[kCenterX] + fMatrix[kCenterY];
                    destination.fY = source.fX - fMatrix[kCenterX] + fMatrix[kCenterY];
                }
                break;
            case kSlowRotate:
                {
                    // == Arbitrary Unrotate Point ===
                    GCoordinate cos = fMatrix[kCos];
                    GCoordinate sin = fMatrix[kSin];
                    GCoordinate oneMinus = 1.0 - cos;
                    GCoordinate oldX = source.fX;
                    destination.fX = cos*oldX + sin*source.fY + fMatrix[kCenterX]*oneMinus - fMatrix[kCenterY]*sin;
                    destination.fY = -sin*oldX + cos*source.fY + fMatrix[kCenterY]*oneMinus + fMatrix[kCenterX]*sin;
                }
                break;
        }
        break;
        }
    case kPrAffine: {
        GCoordinate det =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];
        if (det != 0.0) {
            GCoordinate x = source.fX;
            destination.fX = (fMatrix[kScaleY]*(x-fMatrix[kTranslateX])
                    - fMatrix[kShearX]*(source.fY-fMatrix[kTranslateY]))/det;
            destination.fY = (-fMatrix[kShearY]*(x-fMatrix[kTranslateX])
                + fMatrix[kScaleX]*(source.fY-fMatrix[kTranslateY]))/det;
        }
        else
            TransformException(IGraphicException::kSingularTransform);
        break;
        }
    case kPrGeneric: {
        // == Generic UnTransform Point ===
        // kPrGeneric inverse = classical adjoint
        GCoordinate newTranslateX =
            fMatrix[kShearX]*fMatrix[kTranslateY] -
            fMatrix[kScaleY]*fMatrix[kTranslateX];

        GCoordinate newTranslateY =
            fMatrix[kShearY]*fMatrix[kTranslateX] -
            fMatrix[kScaleX]*fMatrix[kTranslateY];

        GCoordinate newHomogeneous =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];

        GCoordinate x =
            source.fX*(
                fMatrix[kScaleY]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveY]*fMatrix[kTranslateY]) +
            source.fY*(
                fMatrix[kPerspectiveY]*fMatrix[kTranslateX] -
                fMatrix[kShearX]*fMatrix[kHomogeneous]) +
            newTranslateX;

        GCoordinate y =
            source.fX*(
                fMatrix[kPerspectiveX]*fMatrix[kTranslateY] -
                fMatrix[kShearY]*fMatrix[kHomogeneous]) +
            source.fY*(
                fMatrix[kScaleX]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveX]*fMatrix[kTranslateX]) +
            newTranslateY;

        GCoordinate w =
            source.fX*(
                fMatrix[kShearY]*fMatrix[kPerspectiveY] -
                fMatrix[kPerspectiveX]*fMatrix[kScaleY]) +
            source.fY*(
                fMatrix[kPerspectiveX]*fMatrix[kShearX] -
                fMatrix[kScaleX]*fMatrix[kPerspectiveY]) +
            newHomogeneous;

        if (w != 0.0) {
            destination.fX = x/w;
            destination.fY = y/w;
        }
        else
            TransformException(IGraphicException::kSingularTransform);
        }
    }
    return destination;
}

//------------------
// Matrix Inversion
//-------------------

/*================
||
|| IGrafMatrix::invert
||
*/

void IGrafMatrix::invert()
{
    switch (fAccelerator) {
    case kPrIdentity:
        break;
    case kPrTranslate:
        fTranslateX = -fTranslateX;
        fTranslateY = -fTranslateY;
        updateTimeStamp();
        break;
    case kPrScale:
        if ((fMatrix[kScaleX]==0.0) ||(fMatrix[kScaleY]==0.0))
            TransformException(IGraphicException::kSingularTransform);

        fMatrix[kTranslateX] /= -fMatrix[kScaleX];
        fMatrix[kTranslateY] /= -fMatrix[kScaleY];
        fMatrix[kScaleX] = 1.0/fMatrix[kScaleX];
        fMatrix[kScaleY] = 1.0/fMatrix[kScaleY];
        updateTimeStamp();
        break;
    case kPrRotate:
        fMatrix[kDegrees] = 360 - fMatrix[kDegrees];
        fMatrix[kSin] = -fMatrix[kSin];
        if (fRotateAccelerator == kRotate90)
            // Change to kRotate270
            fRotateAccelerator = kRotate270;
        else if (fRotateAccelerator == kRotate270)
            // Change to kRotate90
            fRotateAccelerator = kRotate90;
        updateTimeStamp();
        break;
    case kPrAffine: {
        GCoordinate ad = fMatrix[kScaleX]*fMatrix[kScaleY];
        GCoordinate bc = fMatrix[kShearX]*fMatrix[kShearY];
        if (ad == bc)
            TransformException(IGraphicException::kSingularTransform);
        GCoordinate determinant = ((Abs(ad) > Abs(bc))
            ? ad*(1.0 - bc/ad)
            : bc*(ad/bc - 1.0));
        // affine simple case: inverse = classical adjoint/ determinant
        GCoordinate temp = fMatrix[kScaleX]/determinant;
        fMatrix[kScaleX] = fMatrix[kScaleY]/determinant;
        fMatrix[kScaleY] = temp;
        fMatrix[kShearX] /= -determinant;
        fMatrix[kShearY] /= -determinant;
        GCoordinate tx = fMatrix[kTranslateX];
        GCoordinate ty = fMatrix[kTranslateY];
        fMatrix[kTranslateX] =
            -tx*fMatrix[kScaleX]-ty*fMatrix[kShearX];
        fMatrix[kTranslateY] =
            -tx*fMatrix[kShearY]-ty*fMatrix[kScaleY];
        updateTimeStamp();
        break;
        }
    case kPrGeneric: {
        // == Generic Matrix Inversion ==
        // kPrGeneric inverse = (classical adjoint)/determinant

        GCoordinate newTranslateX =
            fMatrix[kShearX]*fMatrix[kTranslateY] -
            fMatrix[kScaleY]*fMatrix[kTranslateX];

        GCoordinate newTranslateY =
            fMatrix[kShearY]*fMatrix[kTranslateX] -
            fMatrix[kScaleX]*fMatrix[kTranslateY];

        GCoordinate newHomogeneous =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];

        GCoordinate determinant =
            fMatrix[kPerspectiveX]*newTranslateX +
            fMatrix[kPerspectiveY]*newTranslateY +
            fMatrix[kHomogeneous]*newHomogeneous;

        if (!determinant)
            TransformException(IGraphicException::kSingularTransform);

        GCoordinate newScaleX =
                fMatrix[kScaleY]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveY]*fMatrix[kTranslateY];
        GCoordinate newShearX =
                fMatrix[kPerspectiveY]*fMatrix[kTranslateX] -
                fMatrix[kShearX]*fMatrix[kHomogeneous];
        GCoordinate newShearY =
                fMatrix[kPerspectiveX]*fMatrix[kTranslateY] -
                fMatrix[kShearY]*fMatrix[kHomogeneous];
        GCoordinate newScaleY =
                fMatrix[kScaleX]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveX]*fMatrix[kTranslateX];
        GCoordinate newPerspectiveX =
                fMatrix[kShearY]*fMatrix[kPerspectiveY] -
                fMatrix[kPerspectiveX]*fMatrix[kScaleY];
        GCoordinate newPerspectiveY =
                fMatrix[kPerspectiveX]*fMatrix[kShearX] -
                fMatrix[kScaleX]*fMatrix[kPerspectiveY];

        fMatrix[kScaleX]=newScaleX/determinant;
        fMatrix[kScaleY]=newScaleY/determinant;

        fMatrix[kShearX]=newShearX/determinant;
        fMatrix[kShearY]=newShearY/determinant;

        fMatrix[kTranslateX]=newTranslateX/determinant;
        fMatrix[kTranslateY]=newTranslateY/determinant;

        fMatrix[kPerspectiveX]=newPerspectiveX/determinant;
        fMatrix[kPerspectiveY]=newPerspectiveY/determinant;

        fMatrix[kHomogeneous]=newHomogeneous/determinant;

        updateTimeStamp();
        break;
        }
    }
}

/*================
||
|| IGrafMatrix::transformPoints
||
*/

void IGrafMatrix::transformPoints(
    const IGRPoint2D source[], IGRPoint2D destination[], unsigned long count) const
{
    bool overlap = (source == destination);
    IGRPoint2D* dest = destination;
    const IGRPoint2D* src = source;

    switch (fAccelerator) {
    case kPrIdentity:
        // == Identity Transform Rational Points ===
        if (!overlap) {
            while (count--) {
                dest->fX = src->fX;
                dest->fY = src->fY;
                dest++->fW = src++->fW;
            }
        }
        break;

    case kPrTranslate:
        // == Translate Rational Points ===
        while (count--) {
            dest->fX = src->fX + fTranslateX*src->fW;
            dest->fY = src->fY + fTranslateY*src->fW;
            dest++->fW = src++->fW;
        }
        break;

    case kPrRotate:
        // == Rotate Rational Points ===
        switch (fRotateAccelerator) {
        case kRotate0:
            if (!overlap) {
                while (count--) {
                    dest->fX = src->fX;
                    dest->fY = src->fY;
                    dest++->fW = src++->fW;
                }
            }
            break;

        case kRotate90: {
            GCoordinate translateX = fMatrix[kCenterX] + fMatrix[kCenterY];
            GCoordinate translateY = -fMatrix[kCenterX] + fMatrix[kCenterY];
            if (overlap) {
                while (count--) {
                    GCoordinate oldX = src->fX;
                    dest->fX = -src->fY + src->fW*translateX;
                    dest++->fY = oldX + src++->fW*translateY;
                }
            } else {
                while (count--) {
                    dest->fX = -src->fY + src->fW*translateX;
                    dest->fY = src->fX + src->fW*translateY;
                    dest++->fW = src++->fW;
                }
            }
            }
            break;

        case kRotate180:
            while (count--) {
                dest->fX = -src->fX + 2.0*src->fW*fMatrix[kCenterX];
                dest->fY = -src->fY + 2.0*src->fW*fMatrix[kCenterY];
                dest++->fW = src++->fW;
            }
            break;

        case kRotate270: {
            GCoordinate translateX = fMatrix[kCenterX] - fMatrix[kCenterY];
            GCoordinate translateY = fMatrix[kCenterX] + fMatrix[kCenterY];
            if (overlap) {
                while (count--) {
                    GCoordinate oldX = src->fX;
                    dest->fX = src->fY + src->fW*translateX;
                    dest++->fY = -oldX + src++->fW*translateY;
                }
            } else {
                while (count--) {
                    dest->fX = src->fY + src->fW*translateX;
                    dest->fY = -src->fX + src->fW*translateY;
                    dest++->fW = src++->fW;
                }
            }
            }
            break;

        case kSlowRotate: {
            // == Arbitrary Rotate Rational Points ===
            GCoordinate cos = fMatrix[kCos];
            GCoordinate sin = fMatrix[kSin];
            GCoordinate oneMinus = 1.0 - cos;
            GCoordinate translateX = fMatrix[kCenterX]*oneMinus + fMatrix[kCenterY]*sin;
            GCoordinate translateY = fMatrix[kCenterY]*oneMinus - fMatrix[kCenterX]*sin;
            if (overlap) {
                while (count--) {
                    GCoordinate oldX = src->fX;
                    dest->fX = cos*oldX - sin*src->fY + src->fW*translateX;
                    dest->fY = sin*oldX + cos*src->fY + src->fW*translateY;
                    dest++;
                    src++;
                }
            } else {
                while (count--) {
                    dest->fX = cos*src->fX - sin*src->fY + src->fW*translateX;
                    dest->fY = sin*src->fX + cos*src->fY + src->fW*translateY;
                    dest++->fW = src++->fW;
                }
            }
            }
            break;
        }
        break;

    case kPrScale:
        // == scale Rational Points ===
        while (count--) {
            dest->fX = src->fX*fMatrix[kScaleX] + src->fW*fMatrix[kTranslateX];
            dest->fY = src->fY*fMatrix[kScaleY] + src->fW*fMatrix[kTranslateY];
            dest++->fW = src++->fW;
        }
        break;

    case kPrAffine:
        // == Affine Transform Rational Points ===
        if (overlap) {
            while (count--) {
                const GCoordinate* mat = fMatrix;
                GCoordinate oldX = src->fX;
                GCoordinate oldY = src->fY;
                dest->fX = *mat++*oldX;
                dest->fY = *mat++*oldX;
                mat++;
                dest->fX += *mat++*oldY;
                dest->fY += *mat++*oldY;
                mat++;
                dest->fX += *mat++*src->fW;
                dest->fY += *mat*src->fW;
                dest++;
                src++;
            }
        } else {
            while (count--) {
                const GCoordinate* mat = fMatrix;
                dest->fX = *mat++*src->fX;
                dest->fY = *mat++*src->fX;
                mat++;
                dest->fX += *mat++*src->fY;
                dest->fY += *mat++*src->fY;
                mat++;
                dest->fX += *mat++*src->fW;
                dest->fY += *mat*src->fW;
                dest->fW = src->fW;
                dest++;
                src++;
            }
        }
        break;

    case kPrGeneric:
        // == Generic Transform Rational Points ===
        // Full matrix multiplication
        const GCoordinate* startMat = fMatrix;
        if (overlap) {
            while (count--) {
                const GCoordinate* mat = startMat;
                GCoordinate oldX = src->fX;
                GCoordinate oldY = src->fY;
                GCoordinate oldW = src->fW;
                dest->fX = *mat++*oldX;
                dest->fY = *mat++*oldX;
                dest->fW = *mat++*oldX;
                dest->fX += *mat++*oldY;
                dest->fY += *mat++*oldY;
                dest->fW += *mat++*oldY;
                dest->fX += *mat++*oldW;
                dest->fY += *mat++*oldW;
                dest->fW += *mat++*oldW;
                dest++;
                src++;
            }
        } else {
            while (count--) {
                const GCoordinate* mat = startMat;
                dest->fX = *mat++*src->fX;
                dest->fY = *mat++*src->fX;
                dest->fW = *mat++*src->fX;
                dest->fX += *mat++*src->fY;
                dest->fY += *mat++*src->fY;
                dest->fW += *mat++*src->fY;
                dest->fX += *mat++*src->fW;
                dest->fY += *mat++*src->fW;
                dest->fW += *mat++*src->fW;
                dest++;
                src++;
            }
        }
    }
}

void IGrafMatrix::transformPoints(IGRPoint2DArray& points) const
{
    IGRPoint2D * src, * dest;
    GCoordinate translateX, translateY;

    src = dest = &points[0];
    long count = points.numberOfPoints();

    switch (fAccelerator) {
    case kPrIdentity:
        break;          // Nothing to do...

    case kPrTranslate:
        // == Translate Rational Points ===
        while (count--)
        {
            dest->fX = src->fX + fTranslateX*src->fW;
            dest->fY = src->fY + fTranslateY*src->fW;
            dest++->fW = src++->fW;
        }
        break;

    case kPrRotate:
        // == Rotate Rational Points ===
        switch (fRotateAccelerator) {
        case kRotate0:  // Nothing to do
            break;

        case kRotate90:
            translateX = fMatrix[kCenterX] + fMatrix[kCenterY];
            translateY = -fMatrix[kCenterX] + fMatrix[kCenterY];
            while (count--)
            {
                GCoordinate oldX = src->fX;
                dest->fX = -src->fY + src->fW*translateX;
                dest++->fY = oldX + src++->fW*translateY;
            }
            break;

        case kRotate180:
            while (count--)
            {
                dest->fX = -src->fX + 2.0*src->fW*fMatrix[kCenterX];
                dest->fY = -src->fY + 2.0*src->fW*fMatrix[kCenterY];
                dest++->fW = src++->fW;
            }
            break;

        case kRotate270:
            translateX = fMatrix[kCenterX] - fMatrix[kCenterY];
            translateY = fMatrix[kCenterX] + fMatrix[kCenterY];
            while (count--)
            {
                GCoordinate oldX = src->fX;
                dest->fX = src->fY + src->fW*translateX;
                dest++->fY = -oldX + src++->fW*translateY;
            }
            break;

        case kSlowRotate:
            // == Arbitrary Rotate Rational Points ===
            GCoordinate cos = fMatrix[kCos];
            GCoordinate sin = fMatrix[kSin];
            GCoordinate oneMinus = 1.0 - cos;
            translateX = fMatrix[kCenterX]*oneMinus + fMatrix[kCenterY]*sin;
            translateY = fMatrix[kCenterY]*oneMinus - fMatrix[kCenterX]*sin;
            while (count--)
            {
                GCoordinate oldX = src->fX;
                dest->fX = cos*oldX - sin*src->fY + src->fW*translateX;
                dest->fY = sin*oldX + cos*src->fY + src->fW*translateY;
                dest++;
                src++;
            }
            break;
        }
        break;

    case kPrScale:
        // == scale Rational Points ===
        while (count--)
        {
            dest->fX = src->fX*fMatrix[kScaleX] + src->fW*fMatrix[kTranslateX];
            dest->fY = src->fY*fMatrix[kScaleY] + src->fW*fMatrix[kTranslateY];
            dest++->fW = src++->fW;
        }
        break;

    case kPrAffine:
        // == Affine Transform Rational Points ===
        while (count--) {
            const GCoordinate* mat = fMatrix;
            GCoordinate oldX = src->fX;
            GCoordinate oldY = src->fY;
            dest->fX = *mat++*oldX;
            dest->fY = *mat++*oldX;
            mat++;
            dest->fX += *mat++*oldY;
            dest->fY += *mat++*oldY;
            mat++;
            dest->fX += *mat++*src->fW;
            dest->fY += *mat*src->fW;
            dest++;
            src++;
        }
        break;

    case kPrGeneric:
        // == Generic Transform Rational Points ===
        // Full matrix multiplication
        const GCoordinate* startMat = fMatrix;
        while (count--)
        {
            const GCoordinate* mat = startMat;
            GCoordinate oldX = src->fX;
            GCoordinate oldY = src->fY;
            GCoordinate oldW = src->fW;
            dest->fX = *mat++*oldX;
            dest->fY = *mat++*oldX;
            dest->fW = *mat++*oldX;
            dest->fX += *mat++*oldY;
            dest->fY += *mat++*oldY;
            dest->fW += *mat++*oldY;
            dest->fX += *mat++*oldW;
            dest->fY += *mat++*oldW;
            dest->fW += *mat++*oldW;
            dest++;
            src++;
        }
    }
}


/*================
||
|| IGrafMatrix::transformPoint
||
*/

IGPoint2D IGrafMatrix::transformPoint(const IGPoint2D& source) const
{
    IGPoint2D destination;

    switch (fAccelerator) {
    case kPrIdentity:
        destination.fX = source.fX;
        destination.fY = source.fY;
        break;
    case kPrTranslate:
        {
            destination.fX = source.fX + fTranslateX;
            destination.fY = source.fY + fTranslateY;
        }
        break;
    case kPrRotate: {
        if (fMatrix[kCenterX]==0.0 && fMatrix[kCenterY]==0.0)
        {
            switch (fRotateAccelerator) {
                case kRotate0:
                    destination.fX = source.fX;
                    destination.fY = source.fY;
                    break;
                case kRotate90:
                    destination.fX = -source.fY;
                    destination.fY = source.fX;
                    break;
                case kRotate180:
                    destination.fX = -source.fX;
                    destination.fY = -source.fY;
                    break;
                case kRotate270:
                    destination.fX = source.fY;
                    destination.fY = -source.fX;
                    break;
                case kSlowRotate:
                    // == Arbitrary Rotate Points ===
                    destination.fX = fMatrix[kCos]*source.fX - fMatrix[kSin]*source.fY;
                    destination.fY = fMatrix[kSin]*source.fX + fMatrix[kCos]*source.fY;
                    break;
            }
        }
        else
        {
            switch (fRotateAccelerator) {
                case kRotate0:
                    destination.fX = source.fX;
                    destination.fY = source.fY;
                    break;
                case kRotate90:
                    destination.fX = -source.fY + fMatrix[kCenterX] + fMatrix[kCenterY];
                    destination.fY = source.fX  - fMatrix[kCenterX] + fMatrix[kCenterY];
                    break;
                case kRotate180:
                    destination.fX = -source.fX + 2*fMatrix[kCenterX];
                    destination.fY = -source.fY + 2*fMatrix[kCenterY];
                    break;
                case kRotate270:
                    destination.fX = source.fY + fMatrix[kCenterX] - fMatrix[kCenterY];
                    destination.fY = -source.fX + fMatrix[kCenterX] + fMatrix[kCenterY];
                    break;
                case kSlowRotate:
                    // == Arbitrary Rotate Points ===
                    destination.fX = fMatrix[kCos]*source.fX - fMatrix[kSin]*source.fY +(1.0 - fMatrix[kCos])*fMatrix[kCenterX] + fMatrix[kSin]*fMatrix[kCenterY];
                    destination.fY = fMatrix[kSin]*source.fX + fMatrix[kCos]*source.fY +(1.0 - fMatrix[kCos])*fMatrix[kCenterY] - fMatrix[kSin]*fMatrix[kCenterX];

                    break;
            }
        }
        break;
        }
    case kPrScale:
        destination.fX = source.fX*fMatrix[kScaleX] + fMatrix[kTranslateX];
        destination.fY = source.fY*fMatrix[kScaleY] + fMatrix[kTranslateY];
        break;
    case kPrAffine: {
        const GCoordinate* mat = fMatrix;

        destination.fX = *mat++*source.fX;
        destination.fY = *mat++*source.fX;
        mat++;
        destination.fX += *mat++*source.fY;
        destination.fY += *mat++*source.fY;
        mat++;
        destination.fX += *mat++;
        destination.fY += *mat;
        }
        break;
    case kPrGeneric: {
            const GCoordinate* mat = fMatrix;
            destination.fX = *mat++*source.fX;
            destination.fY = *mat++*source.fX;
            GCoordinate h = *mat++*source.fX;
            destination.fX += *mat++*source.fY;
            destination.fY += *mat++*source.fY;
            h += *mat++*source.fY;
            destination.fX += *mat++;
            destination.fY += *mat++;
            h += *mat;
            if ((h != 1.0) &&(h != 0.0)) {
                destination.fX /= h;
                destination.fY /= h;
            }
        }
        break;
    }
    return destination;
}

/*================
||
|| IGrafMatrix::transformPoints
||
*/

void IGrafMatrix::transformPoints(
    const IGPoint2D source[], IGPoint2D destination[], unsigned long count) const
{
    bool overlap = (source == destination);
    IGPoint2D* dest = destination;
    const IGPoint2D* src = source;

    switch (fAccelerator) {
    case kPrIdentity:
        if (!overlap)
        {
            while (count--) {
                dest->fX = src->fX;
                dest++->fY = src++->fY;
            }
        }
        break;
    case kPrTranslate:
        {
            while (count--) {
                dest->fX = src->fX + fTranslateX;
                dest++->fY = src++->fY + fTranslateY;
            }
        }
        break;
    case kPrRotate:
        {
        if (fMatrix[kCenterX]==0.0 && fMatrix[kCenterY]==0.0)
        {
            switch (fRotateAccelerator) {
                case kRotate0:
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX = src->fX;
                            dest++->fY = src++->fY;
                        }
                    }
                    break;
                case kRotate90:
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX = -src->fY;
                            dest++->fY = src++->fX;
                        }
                    }
                    else
                    {
                        while (count--) {
                            GCoordinate temp = dest->fX;
                            dest->fX = -src++->fY;
                            dest++->fY = temp;
                        }
                    }
                    break;
                case kRotate180:
                        while (count--) {
                        dest->fX = -src->fX;
                        dest++->fY = -src++->fY;
                    }
                    break;
                case kRotate270:
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX = src->fY;
                            dest++->fY = -src++->fX;
                        }
                    }
                    else
                    {
                        while (count--) {
                            GCoordinate temp = dest->fX;
                            dest->fX = src++->fY;
                            dest++->fY = -temp;
                        }
                    }
                    break;
                case kSlowRotate: {
                    // == Arbitrary Rotate Points ===
                    GCoordinate cos = fMatrix[kCos];
                    GCoordinate sin = fMatrix[kSin];
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX = cos*src->fX - sin*src->fY;
                            dest++->fY = sin*src->fX + cos*src->fY;
                            src++;
                        }
                    }
                    else
                    {
                        while (count--) {
                            GCoordinate oldX = dest->fX;
                            dest->fX = cos*oldX - sin*dest->fY;
                            dest++->fY = sin*oldX + cos*src++->fY;
                        }
                    }
                    }
                    break;
            }
        }
        else
        {
            switch (fRotateAccelerator) {
                case kRotate0:
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX = src->fX;
                            dest++->fY = src++->fY;
                        }
                    }
                    break;
                case kRotate90: {
                    GCoordinate translateX = fMatrix[kCenterX] + fMatrix[kCenterY];
                    GCoordinate translateY = -fMatrix[kCenterX] + fMatrix[kCenterY];
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX = -src->fY + translateX;
                            dest++->fY = src++->fX + translateY;
                        }
                    }
                    else
                    {
                        while (count--) {
                            GCoordinate temp = dest->fX;
                            dest->fX = -src++->fY + translateX;
                            dest++->fY = temp + translateY;
                        }
                    }
                    }
                    break;
                case kRotate180: {
                    GCoordinate translateX = 2*fMatrix[kCenterX];
                    GCoordinate translateY = 2*fMatrix[kCenterY];
                    while (count--) {
                        dest->fX = -src->fX + translateX;
                        dest++->fY = -src++->fY + translateY;
                    }
                    }
                    break;
                case kRotate270:
                    {
                        GCoordinate translateX = fMatrix[kCenterX] - fMatrix[kCenterY];
                        GCoordinate translateY = fMatrix[kCenterX] + fMatrix[kCenterY];
                        if (!overlap)
                        {
                            while (count--) {
                                dest->fX = src->fY + translateX;
                                dest++->fY = -src++->fX + translateY;
                            }
                        }
                        else
                        {
                            while (count--) {
                                GCoordinate temp = dest->fX;
                                dest->fX = src++->fY + translateX;
                                dest++->fY = -temp + translateY;
                            }
                        }
                    }
                    break;
                case kSlowRotate: {
                    // == Arbitrary Rotate Points ===
                    GCoordinate cos = fMatrix[kCos];
                    GCoordinate sin = fMatrix[kSin];
                    GCoordinate oneMinus = 1.0 - cos;
                    GCoordinate translateX = oneMinus*fMatrix[kCenterX] + sin*fMatrix[kCenterY];
                    GCoordinate translateY = oneMinus*fMatrix[kCenterY] - sin*fMatrix[kCenterX];
                    if (!overlap)
                    {
                        while (count--) {
                            dest->fX   = cos*src->fX - sin*src->fY + translateX;
                            dest++->fY = sin*src->fX + cos*src->fY + translateY;
                            src++;
                        }
                    }
                    else
                    {
                        while (count--) {
                            GCoordinate oldX = dest->fX;
                            dest->fX   = cos*oldX - sin*src->fY + translateX;
                            dest++->fY = sin*oldX + cos*src++->fY + translateY;
                        }
                    }
                    }
                    break;
            }
        }
        break;
        }
    case kPrScale:  {
            GCoordinate scaleX = fMatrix[kScaleX];
            GCoordinate scaleY = fMatrix[kScaleY];
            GCoordinate translateX = fMatrix[kTranslateX];
            GCoordinate translateY = fMatrix[kTranslateY];
            while (count--) {
                dest->fX = src->fX * scaleX + translateX;
                dest++->fY = src++->fY * scaleY + translateY;
            }
        }
        break;
    case kPrAffine: {
            const GCoordinate* matStart = fMatrix;
            IGPoint2D* dest = destination;
            const IGPoint2D* src = source;
            if (!overlap)
            {
                   while (count--) {
                    const GCoordinate* mat = matStart;
                    dest->fX = *mat++*src->fX;
                    dest->fY = *mat++*src->fX;
                    mat++;
                    dest->fX += *mat++*src->fY;
                    dest->fY += *mat++*src++->fY;
                    mat++;
                    dest->fX += *mat++;
                    dest++->fY += *mat;
                }
            }
            else
            {
                while (count--) {
                    const GCoordinate* mat = matStart;
                    GCoordinate oldX = src->fX;
                    GCoordinate oldY = src++->fY;

                    dest->fX = *mat++*oldX;
                    dest->fY = *mat++*oldX;
                    mat++;
                    dest->fX += *mat++*oldY;
                    dest->fY += *mat++*oldY;
                    mat++;
                    dest->fX += *mat++;
                    dest++->fY += *mat;
                }
            }
        }
        break;
    case kPrGeneric: {
            const GCoordinate* matStart = fMatrix;
            IGPoint2D* dest = destination;
            const IGPoint2D* src = source;
            if (!overlap)
            {
                while (count--) {
                    const GCoordinate* mat = matStart;
                    dest->fX = *mat++*src->fX;
                    dest->fY = *mat++*src->fX;
                    GCoordinate h = *mat++*src->fX;
                    dest->fX += *mat++*src->fY;
                    dest->fY += *mat++*src->fY;
                    h += *mat++*src->fY;
                    dest->fX += *mat++;
                    dest->fY += *mat++;
                    h += *mat;
                    if ((h != 1.0) &&(h != 0.0)) {
                        dest->fX /= h;
                        dest->fY /= h;
                    }
                    dest++;
                    src++;
                }
            }
            else
            {
                while (count--) {
                    const GCoordinate* mat = matStart;
                    GCoordinate oldX = src->fX;
                    GCoordinate oldY = src->fY;

                    dest->fX = *mat++*oldX;
                    dest->fY = *mat++*oldX;
                    GCoordinate h = *mat++*oldX;
                    dest->fX += *mat++*oldY;
                    dest->fY += *mat++*oldY;
                    h += *mat++*oldY;
                    dest->fX += *mat++;
                    dest->fY += *mat++;
                    h += *mat;
                    if ((h != 1.0) &&(h != 0.0)) {
                        dest->fX /= h;
                        dest->fY /= h;
                    }
                    dest++;
                    src++;
                }
            }
        }
        break;
    }
}

void IGrafMatrix::transformPoints(IGPoint2DArray& points) const
{
    GCoordinate cos, sin;
    IGPoint2D trans, scale;
    long count = points.numberOfPoints();
    IGPoint2D* ptr = &points[0];
    const GCoordinate* matStart;

    switch (fAccelerator) {
    case kPrIdentity:               // Nothing to do
        break;
    case kPrTranslate:
            trans = IGPoint2D(fTranslateX, fTranslateY);
            while (count--) {
                *ptr += trans;
                ptr++;
            }
        break;
    case kPrRotate:
        if (fMatrix[kCenterX]==0.0 && fMatrix[kCenterY]==0.0)
        {
            switch (fRotateAccelerator) {
                case kRotate0:      // Nothing to do...
                    break;
                case kRotate90:
                    while (count--) {
                        GCoordinate temp = ptr->fX;
                        ptr->fX = -ptr->fY;
                        ptr->fY = temp;
                        ptr++;
                    }
                    break;
                case kRotate180:
                    while (count--) {
                        *ptr = - *ptr;
                        ptr++;
                    }
                    break;
                case kRotate270:
                    while (count--) {
                        GCoordinate temp = ptr->fX;
                        ptr->fX = ptr->fY;
                        ptr->fY = -temp;
                        ptr++;
                    }
                    break;
                case kSlowRotate:
                    // == Arbitrary Rotate Points ===
                    cos = fMatrix[kCos];
                    sin = fMatrix[kSin];
                    while (count--) {
                        GCoordinate oldX = ptr->fX;
                        ptr->fX = cos*oldX - sin*ptr->fY;
                        ptr->fY = sin*oldX + cos*ptr->fY;
                        ptr++;
                    }
                    break;
            }
        }
        else
        {
            switch (fRotateAccelerator) {
                case kRotate0:
                    break;
                case kRotate90:
                    trans = IGPoint2D(fMatrix[kCenterX] + fMatrix[kCenterY],
                                    -fMatrix[kCenterX] + fMatrix[kCenterY]);
                    while (count--) {
                        GCoordinate temp = ptr->fX;
                        ptr->fX = -ptr->fY + trans.fX;
                        ptr->fY = temp + trans.fY;
                        ptr++;
                    }
                    break;
                case kRotate180:
                    trans = IGPoint2D(2*fMatrix[kCenterX],
                                     2*fMatrix[kCenterY]);
                    while (count--) {
                        ptr->fX = -ptr->fX + trans.fX;
                        ptr->fY = -ptr->fY + trans.fY;
                        ptr++;
                    }
                    break;
                case kRotate270:
                    trans = IGPoint2D(fMatrix[kCenterX] - fMatrix[kCenterY],
                                     fMatrix[kCenterX] + fMatrix[kCenterY]);
                    while (count--) {
                        GCoordinate temp = ptr->fX;
                        ptr->fX = ptr->fY + trans.fX;
                        ptr->fY = -temp + trans.fY;
                        ptr++;
                    }
                    break;
                case kSlowRotate:
                    // == Arbitrary Rotate Points ===
                    cos = fMatrix[kCos];
                    sin = fMatrix[kSin];
                    GCoordinate oneMinus = 1.0 - cos;
                    trans = IGPoint2D(oneMinus*fMatrix[kCenterX] + sin*fMatrix[kCenterY],
                                     oneMinus*fMatrix[kCenterY] - sin*fMatrix[kCenterX]);
                    while (count--) {
                        GCoordinate oldX = ptr->fX;
                        ptr->fX   = cos*oldX - sin*ptr->fY + trans.fX;
                        ptr->fY = sin*oldX + cos*ptr->fY + trans.fY;
                        ptr++;
                    }
                    break;
            }
        }
        break;
    case kPrScale:
        scale = IGPoint2D(fMatrix[kScaleX],
                         fMatrix[kScaleY]);
        trans = IGPoint2D(fMatrix[kTranslateX],
                         fMatrix[kTranslateY]);
           while (count--) {
            *ptr = *ptr * scale + trans;
            ptr++;
        }
        break;
    case kPrAffine:
        matStart = fMatrix;
        while (count--) {
            const GCoordinate* mat = matStart;
            GCoordinate oldX = ptr->fX;
            GCoordinate oldY = ptr->fY;

            ptr->fX = *mat++*oldX;
            ptr->fY = *mat++*oldX;
            mat++;
            ptr->fX += *mat++*oldY;
            ptr->fY += *mat++*oldY;
            mat++;
            ptr->fX += *mat++;
            ptr->fY += *mat;
            ptr++;
        }
        break;
    case kPrGeneric:
        matStart = fMatrix;
        while (count--) {
            const GCoordinate* mat = matStart;
            GCoordinate oldX = ptr->fX;
            GCoordinate oldY = ptr->fY;

            ptr->fX = *mat++*oldX;
            ptr->fY = *mat++*oldX;
            GCoordinate h = *mat++*oldX;
            ptr->fX += *mat++*oldY;
            ptr->fY += *mat++*oldY;
            h += *mat++*oldY;
            ptr->fX += *mat++;
            ptr->fY += *mat++;
            h += *mat;
            if ((h != 1.0) &&(h != 0.0)) {
                ptr->fX /= h;
                ptr->fY /= h;
            }
            ptr++;
        }
        break;
    }
}


/*================
||
|| IGrafMatrix::transformVector
||
*/

IGPoint2D IGrafMatrix::transformVector(const IGPoint2D& source) const
{
    IGPoint2D destination;

    IGPoint2D sourceCopy;
    switch (fAccelerator) {
    case kPrIdentity:
    case kPrTranslate:
        if (&source != &destination) destination = source;
        break;
    case kPrRotate: {
        const IGPoint2D* old = &source;
        if (&source == &destination) {
            sourceCopy = source;
            old = &sourceCopy;
        }
        switch (fRotateAccelerator) {
            case kRotate0:
                destination.fX = old->fX;
                destination.fY = old->fY;
                break;
            case kRotate90:
                destination.fX = -old->fY;
                destination.fY = old->fX;
                break;
            case kRotate180:
                destination.fX = -old->fX;
                destination.fY = -old->fY;
                break;
            case kRotate270:
                destination.fX = old->fY;
                destination.fY = -old->fX;
                break;
            case kSlowRotate: {
                destination.fX = fMatrix[kCos]*old->fX - fMatrix[kSin]*old->fY;
                destination.fY = fMatrix[kSin]*old->fX + fMatrix[kCos]*old->fY;
                break;
                }
        }
        break;
        }
    case kPrScale:
        destination.fX = source.fX*fMatrix[kScaleX];
        destination.fY = source.fY*fMatrix[kScaleY];
        break;
    case kPrGeneric:
    case kPrAffine: {
        const IGPoint2D* old = &source;
        if (&source == &destination) {
            sourceCopy = source;
            old = &sourceCopy;
        }
        const GCoordinate* mat = fMatrix;
        destination.fX = *mat++*old->fX;
        destination.fY = *mat++*old->fX;
        mat++;
        destination.fX += *mat++*old->fY;
        destination.fY += *mat*old->fY;
        break;
        }
    }
    return destination;
}

/*================
||
|| IGrafMatrix::untransformVector
||
*/

IGPoint2D IGrafMatrix::untransformVector(const IGPoint2D& source) const
{
    IGPoint2D destination;
    IGPoint2D sourceCopy;
    switch (fAccelerator) {
    case kPrIdentity:
    case kPrTranslate:
        if (&source != &destination) destination = source;
        break;
    case kPrRotate: {
        const IGPoint2D* old = &source;
        if (&source == &destination) {
            sourceCopy = source;
            old = &sourceCopy;
        }

        switch (fRotateAccelerator) {
            case kRotate0:
                if (&source != &destination) destination = source;
                break;
            case kRotate90:
                destination.fX = old->fY;
                destination.fY = -old->fX;
                break;
            case kRotate180:
                destination.fX = -old->fX;
                destination.fY = -old->fY;
                break;
            case kRotate270:
                destination.fX = -old->fY;
                destination.fY = old->fX;
                break;
            case kSlowRotate:
                destination.fX = fMatrix[kCos]*old->fX - fMatrix[kSin]*old->fY;
                destination.fY = fMatrix[kSin]*old->fX + fMatrix[kCos]*old->fY;
            }
        break;
        }
    case kPrScale:
        if (fMatrix[kScaleX]!=0.0 && fMatrix[kScaleY]!=0.0)
        {
            destination.fX = source.fX/fMatrix[kScaleX];
            destination.fY = source.fY/fMatrix[kScaleY];
        }
        else
        {
            TransformException(IGraphicException::kSingularTransform);
        }
        break;
    case kPrGeneric:
    case kPrAffine: {
        const IGPoint2D* old = &source;
        if (&source == &destination) {
            sourceCopy = source;
            old = &sourceCopy;
        }
        GCoordinate det =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];
        if (det != 0.0) {
            destination.fX = (fMatrix[kScaleY]*old->fX - fMatrix[kShearX]*old->fY)/det;
            destination.fY = (-fMatrix[kShearY]*old->fX + fMatrix[kScaleX]*old->fY)/det;
        }
        else
        {
            TransformException(IGraphicException::kSingularTransform);
        }
        break;
        }
    }
    return destination;
}

/*================
||
|| IGrafMatrix::transformBounds
||
*/

IGRect2D IGrafMatrix::transformBounds(const IGRect2D& r) const
{
    if (fAccelerator == kPrIdentity) {
        return r;
    } else {
        if (r == IGRect2D::zeroRect() || r == IGRect2D::infiniteRect()) return r;

        IGRect2D newBounds;

        switch (fAccelerator) {
        case kPrTranslate:
            newBounds.fLeft = r.fLeft + fTranslateX;
            newBounds.fTop  = r.fTop + fTranslateY;
            newBounds.fRight = r.fRight + fTranslateX;
            newBounds.fBottom  = r.fBottom + fTranslateY;
            break;
        case kPrScale:
            newBounds.fLeft = r.fLeft*fMatrix[kScaleX] + fMatrix[kTranslateX];
            newBounds.fTop  = r.fTop *fMatrix[kScaleY] + fMatrix[kTranslateY];
            newBounds.fRight   = r.fRight *fMatrix[kScaleX] + fMatrix[kTranslateX];
            newBounds.fBottom  = r.fBottom*fMatrix[kScaleY] + fMatrix[kTranslateY];
            break;
        case kPrRotate:
        case kPrAffine:
        case kPrGeneric: {
            newBounds.setToPoint(transformPoint(r.topLeft() ));
            newBounds.extendTo(transformPoint(r.topRight()));
            newBounds.extendTo(transformPoint(r.bottomLeft()));
            newBounds.extendTo(transformPoint(r.bottomRight()));
            break;
        }
        default:
            InternalAssert(false, "Unknown matrix type");
            break;
        }

        newBounds.orderPoints();
        return newBounds;
    }
}

/*================
||
|| IGrafMatrix::untransformBounds
||
*/

IGRect2D IGrafMatrix::untransformBounds(const IGRect2D& r) const
{
    if (r == IGRect2D::zeroRect() || r == IGRect2D::infiniteRect()) return r;

    IGRect2D newBounds;
    switch (fAccelerator) {
        case kPrIdentity:
            return r;
        case kPrTranslate:
            newBounds.fLeft = r.fLeft - fTranslateX;
            newBounds.fTop  = r.fTop - fTranslateY;
            newBounds.fRight = r.fRight - fTranslateX;
            newBounds.fBottom  = r.fBottom - fTranslateY;
            break;
        case kPrScale:
            newBounds.fLeft = (r.fLeft - fMatrix[kTranslateX])/fMatrix[kScaleX];
            newBounds.fTop  = (r.fTop - fMatrix[kTranslateY])/fMatrix[kScaleY];
            newBounds.fRight   = (r.fRight - fMatrix[kTranslateX])/fMatrix[kScaleX];
            newBounds.fBottom  = (r.fBottom - fMatrix[kTranslateY])/fMatrix[kScaleY];
            break;
        case kPrRotate:
        case kPrAffine:
        case kPrGeneric: {
            newBounds.setToPoint(untransformPoint(r.topLeft() ));
            newBounds.extendTo(untransformPoint(r.topRight()));
            newBounds.extendTo(untransformPoint(r.bottomLeft()));
            newBounds.extendTo(untransformPoint(r.bottomRight()));
            break;
        }
        default:
            InternalAssert(false, "Unknown matrix type");
            break;
    }
    newBounds.orderPoints();
    return newBounds;
}

/*================
||
|| IGrafMatrix::concatWith
||
*/

void IGrafMatrix::concatWith(const IGrafMatrix& matrix)
{
    if (matrix.fAccelerator == kPrIdentity)
        // OPERATOR IS Identity
        return;
        //---- DONE ---

    updateTimeStamp();

    if (fAccelerator == kPrIdentity) {
        // OPERAND IS Identity
        fAccelerator = matrix.fAccelerator;
        setTimeStamp(matrix.timeStamp());
        if (fAccelerator == kPrTranslate) {
            fTranslateX = matrix.fTranslateX;
            fTranslateY = matrix.fTranslateY;
        }
        else {
            const GCoordinate* src = matrix.fMatrix;
            GCoordinate* dest = fMatrix = new (::kSameHeap, this) GCoordinate[9];
            for(long i=9; i--;)
                *dest++ = *src++;
        }
        fRotateAccelerator = matrix.fRotateAccelerator;
        return;
        //---- DONE ---
    }
    if (fAccelerator == kPrTranslate) {
        if (matrix.fAccelerator == kPrTranslate) {
            fTranslateX += matrix.fTranslateX;
            fTranslateY += matrix.fTranslateY;
            return;
        }
        else {
            SetInternalMatrixToIdentity();
            fMatrix[kTranslateX] = fTranslateX;
            fMatrix[kTranslateY] = fTranslateY;
        }
    }

    if (matrix.fAccelerator == kPrRotate) {
        // OPERATOR IS ROTATE
        if (fAccelerator == kPrRotate) {
            // both the same type: try to rotate!!
            if ((fMatrix[kCenterX] == matrix.fMatrix[kCenterX]) &&
               (fMatrix[kCenterY] == matrix.fMatrix[kCenterY])) {
                // Success!! do the rotate
                fMatrix[kDegrees] += matrix.fMatrix[kDegrees];
                FirstCycle(fMatrix[kDegrees]);
                fRotateAccelerator = GetFastRotate(fMatrix[kDegrees]);
                if (fRotateAccelerator != kRotate0) {
                    ComputeCosAndSin();
                } else {
                    delete [] fMatrix;
                    fMatrix = 0 /*NIL*/;
                    fAccelerator = kPrIdentity;
                    resetTimeStamp();
                }
                return;
            }
            // failure: convert 'this' to an kAffine matrix
            RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        }

        // The operator is still an accelerated rotate matrix;
        // convert it to an kAffine matrix before multiplying.

        GCoordinate tempAffine[9];
        RotateToAffine(matrix.fRotateAccelerator, matrix.fMatrix, tempAffine);
        if (fAccelerator != kPrGeneric) fAccelerator = kPrAffine;
        PostMatrixMult((fAccelerator == kPrGeneric ? kGenericGeneric : kAffineAffine),
                         tempAffine);
        return;
        //---- DONE ---
    }

    if (fAccelerator == kPrRotate) {
        // THIS IS ROTATE, OPERATOR IS NOT
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        if (matrix.fAccelerator == kPrTranslate) {
            fMatrix[kTranslateX] += matrix.fTranslateX;
            fMatrix[kTranslateY] += matrix.fTranslateY;
            fAccelerator = kPrAffine;
        }
        else {
            fAccelerator = (matrix.fAccelerator == kPrGeneric ? kPrGeneric : kPrAffine);
            PostMatrixMult((fAccelerator == kPrGeneric ? kGenericGeneric : kAffineAffine),
                             matrix.fMatrix);
        }
        return;
        //---- DONE ---
    }

    if (matrix.fAccelerator == kPrTranslate) {
        // OPERATOR IS TRANSLATE
        if (fAccelerator == kPrTranslate) {
            fTranslateX += matrix.fTranslateX;
            fTranslateY += matrix.fTranslateY;
        }
        else {
            if (fAccelerator != kPrGeneric)
            {
                fMatrix[kTranslateX] += matrix.fTranslateX;
                fMatrix[kTranslateY] += matrix.fTranslateY;
            }
            else {
                fMatrix[kScaleX] += fMatrix[kPerspectiveX]*matrix.fTranslateX;
                fMatrix[kShearX] += fMatrix[kPerspectiveY]*matrix.fTranslateX;
                fMatrix[kTranslateX] += fMatrix[kHomogeneous]*matrix.fTranslateX;
                fMatrix[kShearY] += fMatrix[kPerspectiveX]*matrix.fTranslateY;
                fMatrix[kScaleY] += fMatrix[kPerspectiveY]*matrix.fTranslateY;
                fMatrix[kTranslateY] += fMatrix[kHomogeneous]*matrix.fTranslateY;
            }
            fAccelerator = GetAccelerator(fMatrix);
            if (fAccelerator==kPrIdentity) {
                resetTimeStamp();
                delete [] fMatrix; fMatrix = 0 /*NIL*/;
            }
            if (fAccelerator == kPrTranslate) {
                fTranslateX = fMatrix[kTranslateX];
                fTranslateY = fMatrix[kTranslateY];
                delete [] fMatrix; fMatrix = 0 /*NIL*/;
            }
        }
        return;
        //---- DONE ---
    }

    // Wiped out the special cases: brute force matrix multiplication.

    PostMatrixMult((fAccelerator == kPrGeneric || matrix.fAccelerator == kPrGeneric ?
                        kGenericGeneric :  kAffineAffine),
                     matrix.fMatrix);
}

/*================
||
|| IGrafMatrix::translateBy
||
*/

void IGrafMatrix::translateBy(const IGPoint2D& delta)
{
    if (delta.fX == 0.0 && delta.fY == 0.0) return;
    updateTimeStamp();
    if (fAccelerator == kPrIdentity) {
        fTranslateX = delta.fX;
        fTranslateY = delta.fY;
        fAccelerator = kPrTranslate;
        return;
    }
    else if (fAccelerator == kPrTranslate) {
        fTranslateX += delta.fX;
        fTranslateY += delta.fY;
        return;
    }
    else if (fAccelerator == kPrGeneric) {
        fMatrix[kScaleX] += delta.fX*fMatrix[kPerspectiveX];
        fMatrix[kShearY] += delta.fY*fMatrix[kPerspectiveX];
        fMatrix[kShearX] += delta.fX*fMatrix[kPerspectiveY];
        fMatrix[kScaleY] += delta.fY*fMatrix[kPerspectiveY];
        fMatrix[kTranslateX] += delta.fX*fMatrix[kHomogeneous];
        fMatrix[kTranslateY] += delta.fY*fMatrix[kHomogeneous];
        return;
    }
    else if (fAccelerator == kPrRotate) {
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        fAccelerator = kPrAffine;
    }
    fMatrix[kTranslateX] += delta.fX;
    fMatrix[kTranslateY] += delta.fY;
}

/*================
||
|| IGrafMatrix::scaleBy
||
*/

void IGrafMatrix::scaleBy(
    const IGPoint2D& scale,
    const IGPoint2D& centerOfScale)
{
    GCoordinate scaleX = scale.fX;
    GCoordinate scaleY = scale.fY;
    GCoordinate tx =
       ((centerOfScale.fX == 0.0)? 0.0: centerOfScale.fX*(1.0-scaleX));
    GCoordinate ty =
       ((centerOfScale.fY == 0.0)? 0.0: centerOfScale.fY*(1.0-scaleY));
    updateTimeStamp();
    if (fAccelerator == kPrIdentity)
    {
        SetInternalMatrixToIdentity();
        fAccelerator = kPrScale;
        fMatrix[kScaleX] = scaleX;
        fMatrix[kScaleY] = scaleY;
        fMatrix[kTranslateX] = tx;
        fMatrix[kTranslateY] = ty;
        return;
    }
    if (fAccelerator==kPrTranslate)
    {
        SetInternalMatrixToIdentity();
        fAccelerator = kPrScale;
        fMatrix[kScaleX] = scaleX;
        fMatrix[kScaleY] = scaleY;
        fMatrix[kTranslateX] = scaleX*fTranslateX+tx;
        fMatrix[kTranslateY] = scaleY*fTranslateY+ty;
        return;
    }
    if (fAccelerator==kPrRotate)
    {
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        fAccelerator = kPrAffine;
    }
    if (fAccelerator==kPrGeneric) {
        fMatrix[kTranslateX] =
            fMatrix[kTranslateX]*scaleX +
            fMatrix[kHomogeneous]*tx;
        fMatrix[kTranslateY] =
            fMatrix[kTranslateY]*scaleY +
            fMatrix[kHomogeneous]*ty;
    }
    else {
        fMatrix[kTranslateX] = scaleX*fMatrix[kTranslateX]+tx;
        fMatrix[kTranslateY] = scaleY*fMatrix[kTranslateY]+ty;
    }
    fMatrix[kScaleX] *= scaleX;
    fMatrix[kScaleY] *= scaleY;
    if (fAccelerator==kPrScale) return;
    fMatrix[kShearX] *= scaleX;
    fMatrix[kShearY] *= scaleY;
    if (fAccelerator==kPrAffine) return;
    fMatrix[kScaleX] += fMatrix[kPerspectiveX]*tx;
    fMatrix[kShearY] += fMatrix[kPerspectiveX]*ty;
    fMatrix[kShearX] += fMatrix[kPerspectiveY]*tx;
    fMatrix[kScaleY] += fMatrix[kPerspectiveY]*ty;
}

/*================
||
|| IGrafMatrix::rotateBy
||
*/

void IGrafMatrix::rotateBy(
    GDegrees degrees,
    const IGPoint2D& centerOfRotate)
{
    FirstCycle(degrees);
    if (degrees == 0.0 || degrees == 360.0)
        return;

    switch (fAccelerator) {
    case kPrIdentity: {
        fAccelerator = kPrRotate;
        fMatrix = new (::kSameHeap, this) GCoordinate[9];
        fMatrix[kDegrees] = degrees;
        fMatrix[kCenterX] = centerOfRotate.fX;
        fMatrix[kCenterY] = centerOfRotate.fY;
        fRotateAccelerator = GetFastRotate(degrees);
        ComputeCosAndSin();
        updateTimeStamp();
        return;
        }
    case kPrRotate:
        if ((fMatrix[kCenterX] == centerOfRotate.fX) &&
           (fMatrix[kCenterY] == centerOfRotate.fY))
        {
            fMatrix[kDegrees] += degrees;
            FirstCycle(fMatrix[kDegrees]);
            fRotateAccelerator = GetFastRotate(fMatrix[kDegrees]);
            if (fRotateAccelerator != kRotate0) {
                ComputeCosAndSin();
                updateTimeStamp();
            } else {
                delete [] fMatrix;
                fMatrix = 0 /*NIL*/;
                fAccelerator = kPrIdentity;
                resetTimeStamp();
            }
            return;
        }
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
    case kPrTranslate:
        if (fAccelerator==kPrTranslate) {
            SetInternalMatrixToIdentity();
            fMatrix[kTranslateX] = fTranslateX;
            fMatrix[kTranslateY] = fTranslateY;
        }
    case kPrScale:
        fAccelerator = kPrAffine;
    case kPrAffine:
    case kPrGeneric: {
        GCoordinate nCos = Cos(degrees * kDegreesToRadians);
        GCoordinate nSin = Sin(degrees * kDegreesToRadians);
        GCoordinate temp [9];
        GCoordinate* tempPtr = temp;
        *tempPtr++ = nCos;
        *tempPtr++ = nSin;
        *tempPtr++ = 0.0;
        *tempPtr++ = -nSin;
        *tempPtr++ = nCos;
        *tempPtr++ = 0.0;
        if (centerOfRotate.fX || centerOfRotate.fY) {
            *tempPtr++ = -centerOfRotate.fX*(nCos-1.0) + centerOfRotate.fY*nSin ;
            *tempPtr++ = -centerOfRotate.fX*nSin - centerOfRotate.fY*(nCos-1);
        }
        else {
            *tempPtr++ = 0.0;
            *tempPtr++ = 0.0;
        }
        *(tempPtr) = 1;
        if (fAccelerator == kPrAffine)
            PostMatrixMult(kAffineAffine, temp);
        else PostMatrixMult(kGenericGeneric, temp);
        updateTimeStamp();
        return;
        }
    }
    updateTimeStamp();
}

/*================
||
|| IGrafMatrix::preConcatWith
||
*/

void IGrafMatrix::preConcatWith(const IGrafMatrix& matrix)
{
    if (matrix.fAccelerator == kPrIdentity)
        // OPERATOR IS IDENTITY
        return;
        //---- DONE ---

    updateTimeStamp();

    if (fAccelerator == kPrIdentity) {
        // OPERAND IS IDENTITY
        fAccelerator = matrix.fAccelerator;
        setTimeStamp(matrix.timeStamp());
        if (fAccelerator == kPrTranslate) {
            fTranslateX = matrix.fTranslateX;
            fTranslateY = matrix.fTranslateY;
        }
        else {
            const GCoordinate* src = matrix.fMatrix;
            GCoordinate* dest = fMatrix = new (::kSameHeap, this) GCoordinate[9];
            for(long i=9; i--;)
                *dest++ = *src++;
        }
        fRotateAccelerator = matrix.fRotateAccelerator;
        return;
        //---- DONE ---
    }
    if (fAccelerator == kPrTranslate) {
        if (matrix.fAccelerator == kPrTranslate) {
            fTranslateX += matrix.fTranslateX;
            fTranslateY += matrix.fTranslateY;
            return;
        }
        else {
            SetInternalMatrixToIdentity();
            fMatrix[kTranslateX] = fTranslateX;
            fMatrix[kTranslateY] = fTranslateY;
        }
    }

    if (matrix.fAccelerator == kPrRotate) {
        // OPERATOR IS ROTATE
        if (fAccelerator == kPrRotate) {
            // both the same type: try to rotate!!
            if ((fMatrix[kCenterX] == matrix.fMatrix[kCenterX]) &&
               (fMatrix[kCenterY] == matrix.fMatrix[kCenterY])) {
                // Success!! do the rotate
                fMatrix[kDegrees] += matrix.fMatrix[kDegrees];
                FirstCycle(fMatrix[kDegrees]);
                fRotateAccelerator = GetFastRotate(fMatrix[kDegrees]);
                if (fRotateAccelerator != kRotate0) {
                    ComputeCosAndSin();
                } else {
                    delete [] fMatrix;
                    fMatrix = 0 /*NIL*/;
                    fAccelerator = kPrIdentity;
                    resetTimeStamp();
                }
                return;
            }
            // failure: convert 'this' to an kAffine matrix
            RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        }

        // The operator is still an accelerated rotate matrix;
        // convert it to an kAffine matrix before multiplying.

        GCoordinate tempAffine[9];
        RotateToAffine(matrix.fRotateAccelerator, matrix.fMatrix, tempAffine);
        if (fAccelerator != kPrGeneric) fAccelerator = kPrAffine;
        PreMatrixMult((fAccelerator == kPrGeneric ? kGenericGeneric : kAffineAffine),
                         tempAffine);
        return;
        //---- DONE ---
    }

    if (matrix.fAccelerator == kPrTranslate) {
        preTranslateBy(IGPoint2D(matrix.fTranslateX, matrix.fTranslateY));
        return;
        //---- DONE ---
    }

    if (fAccelerator == kPrRotate) {
        // THIS IS ROTATE, OPERATOR IS NOT
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        fAccelerator = (matrix.fAccelerator == kPrGeneric ? kPrGeneric : kPrAffine);
        PreMatrixMult((fAccelerator == kPrGeneric ? kGenericGeneric : kAffineAffine),
                         matrix.fMatrix);
        return;
        //---- DONE ---
    }

    // Wiped out the special cases: brute force matrix multiply.
    PreMatrixMult((fAccelerator == kPrGeneric || matrix.fAccelerator == kPrGeneric ?
                        kGenericGeneric : kAffineAffine),
                     matrix.fMatrix);
}

/*================
||
|| IGrafMatrix::preTranslateBy
||
*/

void IGrafMatrix::preTranslateBy(const IGPoint2D& delta)
{
    GCoordinate tx = delta.fX;
    GCoordinate ty = delta.fY;
    updateTimeStamp();
    if (tx==0.0 && ty == 0.0) return;
    switch (fAccelerator) {
    case kPrIdentity:
        fTranslateX = delta.fX;
        fTranslateY = delta.fY;
        fAccelerator = kPrTranslate;
        break;
    case kPrTranslate:
        fTranslateX += delta.fX;
        fTranslateY += delta.fY;
        break;
    case kPrRotate:
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        fAccelerator = kPrAffine;
        //fall through
    case kPrAffine:
        fMatrix[kTranslateX] += delta.fY*fMatrix[kShearX];
        fMatrix[kTranslateY] += delta.fX*fMatrix[kShearY];
        //fall through
    case kPrScale:
        tx *= fMatrix[kScaleX];
        ty *= fMatrix[kScaleY];
        fMatrix[kTranslateX] += tx;
        fMatrix[kTranslateY] += ty;
        break;
    case kPrGeneric:
        fMatrix[kHomogeneous] +=
            delta.fX*fMatrix[kPerspectiveX] +
            delta.fY*fMatrix[kPerspectiveY];
        fMatrix[kTranslateX] += delta.fY*fMatrix[kShearX] + tx*fMatrix[kScaleX];
        fMatrix[kTranslateY] += delta.fX*fMatrix[kShearY] + ty*fMatrix[kScaleY];
        break;
    }
}

/*================
||
|| IGrafMatrix::preScaleBy
||
*/

void IGrafMatrix::preScaleBy(
    const IGPoint2D& scale,
    const IGPoint2D& centerOfScale)
{
    GCoordinate tx =
       ((centerOfScale.fX == 0.0)? 0.0: centerOfScale.fX*(1.0-scale.fX));
    GCoordinate ty =
       ((centerOfScale.fY == 0.0)? 0.0: centerOfScale.fY*(1.0-scale.fY));
    updateTimeStamp();
    switch (fAccelerator) {
    case kPrIdentity:
        SetInternalMatrixToIdentity();
        fAccelerator = kPrScale;
        fMatrix[kScaleX] = scale.fX;
        fMatrix[kScaleY] = scale.fY;
        fMatrix[kTranslateX] = tx;
        fMatrix[kTranslateY] = ty;
        return;
    case kPrTranslate:
        SetInternalMatrixToIdentity();
        fAccelerator = kPrScale;
        fMatrix[kScaleX] = scale.fX;
        fMatrix[kScaleY] = scale.fY;
        fMatrix[kTranslateX] = fTranslateX + tx;
        fMatrix[kTranslateY] = fTranslateY + ty;
        updateTimeStamp();
        return;
    case kPrScale:
        fMatrix[kTranslateX] += tx*fMatrix[kScaleX];
        fMatrix[kTranslateY] += ty*fMatrix[kScaleY];
        fMatrix[kScaleX] *= scale.fX;
        fMatrix[kScaleY] *= scale.fY;
        break;
    case kPrRotate:
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        fAccelerator = kPrAffine;
        // fall through
    case kPrGeneric:
        fMatrix[kHomogeneous] += (tx*fMatrix[kPerspectiveX] + ty*fMatrix[kPerspectiveY]);
        fMatrix[kPerspectiveX] *= scale.fX;
        fMatrix[kPerspectiveY] *= scale.fY;
        // fall through
    case kPrAffine:
        fMatrix[kTranslateX] += (tx*fMatrix[kScaleX] + ty*fMatrix[kShearX]);
        fMatrix[kTranslateY] += (tx*fMatrix[kShearY] + ty*fMatrix[kScaleY]);
        fMatrix[kScaleX] *= scale.fX;
        fMatrix[kShearX] *= scale.fY;
        fMatrix[kShearY] *= scale.fX;
        fMatrix[kScaleY] *= scale.fY;
        break;
    }
}

/*================
||
|| IGrafMatrix::preRotateBy
||
*/

void IGrafMatrix::preRotateBy(
    GDegrees degrees,
    const IGPoint2D& centerOfRotate)
{
    FirstCycle(degrees);
    if (degrees == 0.0 || degrees == 360.0)
        return;

    updateTimeStamp();
    switch (fAccelerator) {
    case kPrIdentity:
        fAccelerator = kPrRotate;
        fMatrix = new (::kSameHeap, this) GCoordinate[9];
        fMatrix[kDegrees] = degrees;
        fMatrix[kCenterX] = centerOfRotate.fX;
        fMatrix[kCenterY] = centerOfRotate.fY;
        fRotateAccelerator = GetFastRotate(degrees);
        ComputeCosAndSin();
        return;
    case kPrRotate:
        if ((fMatrix[kCenterX] == centerOfRotate.fX) &&
           (fMatrix[kCenterY] == centerOfRotate.fY))
        {
            fMatrix[kDegrees] += degrees;
            FirstCycle(fMatrix[kDegrees]);
            fRotateAccelerator = GetFastRotate(fMatrix[kDegrees]);
            if (fRotateAccelerator != kRotate0) {
                ComputeCosAndSin();
            } else {
                delete [] fMatrix;
                fMatrix = 0 /*NIL*/;
                fAccelerator = kPrIdentity;
                resetTimeStamp();
            }
            return;
        }
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
        fAccelerator = GetAccelerator(fMatrix);
        if (fAccelerator==kPrIdentity) {
            resetTimeStamp();
            delete [] fMatrix; fMatrix = 0 /*NIL*/;
        }
        if (fAccelerator==kPrTranslate) {
            fTranslateX = fMatrix[kTranslateX];
            fTranslateY = fMatrix[kTranslateY];
            delete [] fMatrix; fMatrix = 0 /*NIL*/;
        }
    case kPrTranslate:
        SetInternalMatrixToIdentity();
        fMatrix[kTranslateX] = fTranslateX;
        fMatrix[kTranslateY] = fTranslateY;
    case kPrScale:
        fAccelerator = kPrAffine;
    case kPrAffine:
    case kPrGeneric: {
        GCoordinate nCos = Cos(degrees * kDegreesToRadians);
        GCoordinate nSin = Sin(degrees * kDegreesToRadians);
        GCoordinate temp [9];
        GCoordinate* tempPtr = temp;
        *tempPtr++ = nCos;
        *tempPtr++ = nSin;
        *tempPtr++ = 0.0;
        *tempPtr++ = -nSin;
        *tempPtr++ = nCos;
        *tempPtr++ = 0.0;
        if (centerOfRotate.fX || centerOfRotate.fY) {
            *tempPtr++ = -centerOfRotate.fX*(nCos-1.0) + centerOfRotate.fY*nSin ;
            *tempPtr++ = -centerOfRotate.fX*nSin - centerOfRotate.fY*(nCos-1);
        }
        else {
            *tempPtr++ = 0.0;
            *tempPtr++ = 0.0;
        }
        *(tempPtr) = 1.0;
        if (fAccelerator == kPrAffine)
            PreMatrixMult(kAffineAffine, temp);
        else PreMatrixMult(kGenericGeneric, temp);
        return;
        }
    }
}

/*================
||
//  Private function
|| IGrafMatrix::SetInternalMatrixToIdentity
||
*/

void IGrafMatrix::SetInternalMatrixToIdentity()
{
    if (fMatrix == 0 /*NIL*/)
        fMatrix = new (::kSameHeap, this) GCoordinate[9];

    register GCoordinate* dest = fMatrix;

    *dest++ = 1.0;
    *dest++ = 0.0;
    *dest++ = 0.0;

    *dest++ = 0.0;
    *dest++ = 1.0;
    *dest++ = 0.0;

    *dest++ = 0.0;
    *dest++ = 0.0;
    *dest = 1.0;
}

/*================
||
|| IGrafMatrix::setToIdentity
||
*/

void IGrafMatrix::setToIdentity()
{
    resetTimeStamp();
    fAccelerator = kPrIdentity;
    delete [] fMatrix;
    fMatrix = 0 /*NIL*/;
}

/*================
||
|| IGrafMatrix::setToTranslate
||
*/

void IGrafMatrix::setToTranslate(const IGPoint2D& delta)
{
    updateTimeStamp();
    delete [] fMatrix;
    fMatrix = 0 /*NIL*/;
    if (delta.fX != 0.0 || delta.fY != 0.0)
    {
        fAccelerator = kPrTranslate;
        fTranslateX = delta.fX;
        fTranslateY = delta.fY;
    }
    else fAccelerator = kPrIdentity;
}

/*================
||
|| IGrafMatrix::setToScale
||
*/

void IGrafMatrix::setToScale(
    const IGPoint2D& scale,
    const IGPoint2D& centerOfScale)
{
    updateTimeStamp();
    if (scale.fX != 1.0 || scale.fY != 1.0)
    {
        SetInternalMatrixToIdentity();
        fAccelerator = kPrScale;
        fMatrix[kScaleX] = scale.fX;
        fMatrix[kScaleY] = scale.fY;
        fMatrix[kTranslateX] = centerOfScale.fX*(1.0-scale.fX);
        fMatrix[kTranslateY] = centerOfScale.fY*(1.0-scale.fY);
    }
    else {
        delete [] fMatrix;
        fMatrix = 0 /*NIL*/;
        fAccelerator = kPrIdentity;
    }
}

/*================
||
|| IGrafMatrix::setToRotate
||
*/

void IGrafMatrix::setToRotate(
    GDegrees degrees,
    const IGPoint2D& centerOfRotate)
{
    updateTimeStamp();
    fAccelerator = kPrRotate;
    GCoordinate fcDegrees = degrees;
    FirstCycle(fcDegrees);
    fRotateAccelerator = GetFastRotate(fcDegrees);
    if (fRotateAccelerator != kRotate0) {
        fMatrix = new (::kSameHeap, this) GCoordinate[9];
        fMatrix[kDegrees] = fcDegrees;
        fMatrix[kCenterX] = centerOfRotate.fX;
        fMatrix[kCenterY] = centerOfRotate.fY;
        ComputeCosAndSin();
    } else {
        delete [] fMatrix;
        fMatrix = 0 /*NIL*/;
        fAccelerator = kPrIdentity;
        resetTimeStamp();
    }
}

/*================
||
|| IGrafMatrix::setToPerspectiveMap
||
*/

void IGrafMatrix::setToPerspectiveMap(
    const IGQuadrilateral& fromQuadrilateral,
    const IGQuadrilateral& toQuadrilateral )
{
    ComputePerspectiveMap(fromQuadrilateral,toQuadrilateral);
}

/*================
||
|| IGrafMatrix::matrix
||
*/

// Queries
GCoordinate* IGrafMatrix::matrix(GCoordinate matrix []) const
{
    if (fAccelerator == kPrIdentity) {
        GCoordinate* dest = matrix;
        *dest++ = 1.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = 1.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest = 1.0;
    }
    else if (fAccelerator == kPrTranslate) {
        GCoordinate* dest = matrix;
        *dest++ = 1.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = 0.0;
        *dest++ = 1.0;
        *dest++ = 0.0;
        *dest++ = fTranslateX;
        *dest++ = fTranslateY;
        *dest = 1.0;
    }
    else if (fAccelerator == kPrRotate) {
        GCoordinate centerX = fMatrix[kCenterX];
        GCoordinate centerY = fMatrix[kCenterY];
        GCoordinate nCos = fMatrix[kCos];
        GCoordinate nSin = fMatrix[kSin];
        GCoordinate tx = (centerX==0.0 ? 0.0 : -centerX*(nCos-1.0) + centerY*nSin);
        GCoordinate ty = (centerY==0.0 ? 0.0 : -centerX*nSin - centerY*(nCos-1.0));
        GCoordinate* dest = matrix;
        *dest++ = nCos;
        *dest++ = nSin;
        *dest++ = 0.0;
        *dest++ = -nSin;
        *dest++ = nCos;
        *dest++ = 0.0;
        *dest++ = tx;
        *dest++ = ty;
        *dest = 1.0;
    }
    else {
        register const GCoordinate* source = fMatrix;
        register GCoordinate* dest = matrix;

        *dest++ = *source++;
        *dest++ = *source++;
        *dest++ = *source++;

        *dest++ = *source++;
        *dest++ = *source++;
        *dest++ = *source++;

        *dest++ = *source++;
        *dest++ = *source++;
        *dest = *source;
    }
    return (matrix);
}

/*================
||
|| IGrafMatrix::rotate
||
*/

bool IGrafMatrix::rotate(GDegrees& degrees, IGPoint2D& centerOfRotate) const
{
    if (fAccelerator == kPrRotate) {
        degrees = fMatrix[kDegrees];
        centerOfRotate.fX = fMatrix[kCenterX];
        centerOfRotate.fY = fMatrix[kCenterY];
        return (true);
    }
    else return (false);
}

/*================
||
|| IGrafMatrix::type
||
*/

IGrafMatrix::EMatrixType IGrafMatrix::type() const
{
    EMatrixType publicType = kPerspective;
    switch (fAccelerator) {
        case kPrIdentity: publicType = kIdentity; break;
        case kPrTranslate: publicType = kTranslate; break;
        case kPrRotate: publicType = kRotate; break;
        case kPrScale: publicType = kScale; break;
        case kPrAffine: publicType = kAffine; break;
        case kPrGeneric: publicType = kPerspective; break;
    }
    return publicType;
}

/*================
||
|| IGrafMatrix::isIdentity
||
*/

bool IGrafMatrix::isIdentity() const
{
    return ((fAccelerator == kPrIdentity) ||
           ((fAccelerator == kPrRotate) &&(fRotateAccelerator == kRotate0)));
}

/*================
||
|| IGrafMatrix::isTranslate
||
*/

bool IGrafMatrix::isTranslate() const
{
    return ((fAccelerator == kPrIdentity) ||
           (fAccelerator == kPrTranslate) ||
           ((fAccelerator == kPrRotate) &&(fRotateAccelerator == kRotate0)));
}

/*================
||
|| IGrafMatrix::isAffine
||
*/

bool IGrafMatrix::isAffine() const
{
    return (fAccelerator!=kPrGeneric);
}

/*================
||
|| IGrafMatrix::isRectilinear
||
*/

bool IGrafMatrix::isRectilinear() const
{
    return ((fAccelerator == kPrIdentity) ||
           (fAccelerator == kPrTranslate) ||
           (fAccelerator == kPrScale) ||
           ((fAccelerator == kPrRotate) &&(fRotateAccelerator != kSlowRotate)) ||
           ((fAccelerator == kPrAffine &&
             (((fMatrix[kShearX] == 0.0) &&(fMatrix[kShearY] == 0.0)) ||
              ((fMatrix[kScaleX] == 0.0) &&(fMatrix[kScaleY] == 0.0))))));
}


//---------------
// Expert calls
//---------------

/*================
||
|| IGrafMatrix::element
||
*/

GCoordinate IGrafMatrix::element(EMatrixIndex index) const
{
    GCoordinate element = 0.;

    if (fAccelerator==kPrRotate) {
        if (index == kScaleX)               element = fMatrix[kCos];
        else if (index == kShearY)          element = fMatrix[kSin];
        else if (index == kPerspectiveX)    element = 0.0;
        else if (index == kShearX)          element = -fMatrix[kSin];
        else if (index == kScaleY)          element = fMatrix[kCos];
        else if (index == kPerspectiveY)    element = 0.0;
        else if (index == kTranslateX)      element =
           (fMatrix[kCenterX]==0.0 ? 0.0
            : -fMatrix[kCenterX]*(fMatrix[kCos]-1.0) + fMatrix[kCenterY]*fMatrix[kSin]);
        else if (index == kTranslateY)      element =
           (fMatrix[kCenterY]==0.0 ? 0.0
            : -fMatrix[kCenterX]*fMatrix[kSin] - fMatrix[kCenterY]*(fMatrix[kCos]-1.0));
        else if (index == kHomogeneous)     element = 1.0;
    }
    else if (fAccelerator == kPrIdentity) {
        if (index == kScaleX || index == kScaleY || index == kHomogeneous) element = 1.0;
        else element = 0.0;
    }
    else if (fAccelerator == kPrTranslate) {
        if (index == kTranslateX)
            element = fTranslateX;
        else if (index == kTranslateY)
            element = fTranslateY;
        else if (index == kScaleX || index == kScaleY || index == kHomogeneous)
            element = 1.0;
        else element = 0.0;
    }
    else  element = fMatrix[index];

    return element;
}

/*================
||
|| IGrafMatrix::setElement
||
*/

void IGrafMatrix::setElement(EMatrixIndex index, GCoordinate element)
{
    updateTimeStamp();
    if (fMatrix == 0 /*NIL*/) {
        SetInternalMatrixToIdentity();
    }
    if (fAccelerator == kPrTranslate) {
        fMatrix[kTranslateX] = fTranslateX;
        fMatrix[kTranslateY] = fTranslateY;
    }
    if (fAccelerator==kPrRotate)
        RotateToAffine(fRotateAccelerator, fMatrix, fMatrix);
    fMatrix[index] = element;
    fAccelerator = GetAccelerator(fMatrix);
    if (fAccelerator==kPrIdentity) {
        resetTimeStamp();
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
    if (fAccelerator==kPrTranslate) {
        fTranslateX = fMatrix[kTranslateX];
        fTranslateY = fMatrix[kTranslateY];
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
}

/*================
||
|| IGrafMatrix::determinant
||
*/

// Math Operations
GCoordinate IGrafMatrix::determinant() const
{
    GCoordinate determinant = 1.0;

    switch (fAccelerator) {
    case kPrIdentity:
    case kPrTranslate:
    case kPrRotate:
        determinant = 1.0;
        break;
    case kPrScale:
        determinant = fMatrix[kScaleX]*fMatrix[kScaleY];
        break;
    case kPrAffine:
    case kPrGeneric:
        determinant =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];
        if (fAccelerator==kPrGeneric) {
            determinant *= fMatrix[kHomogeneous];
            determinant += (
                fMatrix[kPerspectiveX]*(
                    fMatrix[kShearX]*fMatrix[kTranslateY] -
                    fMatrix[kScaleY]*fMatrix[kTranslateX]) -
                fMatrix[kPerspectiveY]*(
                    fMatrix[kScaleX]*fMatrix[kTranslateY] -
                    fMatrix[kShearY]*fMatrix[kTranslateX]));
        }
        break;
    }
    return determinant;
}

/*================
||
|| IGrafMatrix::transpose
||
*/

void IGrafMatrix::transpose()
{
    if (fAccelerator==kPrIdentity)
        return;
    updateTimeStamp();
    if (fAccelerator == kPrTranslate) {
        SetInternalMatrixToIdentity();
        fMatrix[kPerspectiveX] = fTranslateX;
        fMatrix[kPerspectiveY] = fTranslateY;
        fAccelerator = kPrGeneric;
    }
    else {
        if (fAccelerator==kPrRotate)
            RotateToAffine(
                fRotateAccelerator, fMatrix, fMatrix);
        GCoordinate temp = fMatrix[kShearX];
        fMatrix[kShearX]=fMatrix[kShearY];
        fMatrix[kShearY]=temp;

        temp = fMatrix[kPerspectiveX];
        fMatrix[kPerspectiveX]=fMatrix[kTranslateX];
        fMatrix[kTranslateX]=temp;

        temp = fMatrix[kPerspectiveY];
        fMatrix[kPerspectiveY]=fMatrix[kTranslateY];
        fMatrix[kTranslateY]=temp;
        fAccelerator = GetAccelerator(fMatrix);
        if (fAccelerator==kPrIdentity) {
            resetTimeStamp();
            delete [] fMatrix; fMatrix = 0 /*NIL*/;
        }
        if (fAccelerator == kPrTranslate) {
            fTranslateX = fMatrix[kTranslateX];
            fTranslateY = fMatrix[kTranslateY];
            delete [] fMatrix; fMatrix = 0 /*NIL*/;
        }
    }
}

/*================
||
|| IGrafMatrix::makeAdjoint
||
*/

void IGrafMatrix::makeAdjoint()
{
    GCoordinate temp;
    GCoordinate newTranslateX;
    GCoordinate newTranslateY;
    switch (fAccelerator) {
    case kPrIdentity:
        break;
    case kPrTranslate:
        fTranslateX = -fTranslateX;
        fTranslateY = -fTranslateY;
        break;
    case kPrScale:
        fMatrix[kTranslateX] *= -fMatrix[kScaleY];
        fMatrix[kTranslateY] *= -fMatrix[kScaleX];
        temp = fMatrix[kScaleX];
        fMatrix[kScaleX] = fMatrix[kScaleY];
        fMatrix[kScaleY] = temp;
        break;
    case kPrRotate:
        fMatrix[kDegrees] = 360 - fMatrix[kDegrees];
        fMatrix[kSin] = -fMatrix[kSin];
        if (fRotateAccelerator == kRotate90)
            // Change to kRotate270
            fRotateAccelerator = kRotate270;
        else if (fRotateAccelerator == kRotate270)
            // Change to kRotate90
            fRotateAccelerator = kRotate90;
        break;
    case kPrAffine:
        temp = fMatrix[kScaleX];
        fMatrix[kScaleX] = fMatrix[kScaleY];
        fMatrix[kScaleY] = temp;
        fMatrix[kShearX] = -fMatrix[kShearX];
        fMatrix[kShearY] = -fMatrix[kShearY];
        newTranslateX =
            fMatrix[kShearX]*fMatrix[kTranslateY] -
            fMatrix[kScaleY]*fMatrix[kTranslateX];
        newTranslateY =
            fMatrix[kScaleX]*fMatrix[kTranslateY] -
            fMatrix[kShearY]*fMatrix[kTranslateX];
        fMatrix[kTranslateX] = newTranslateX;
        fMatrix[kTranslateY] = newTranslateY;
        break;
    case kPrGeneric: {
        // kPrGeneric inverse = (classical adjoint)
        // == Generic Matrix Inversion ==
        // kPrGeneric inverse = (classical adjoint)/determinant

        GCoordinate newTranslateX =
            fMatrix[kShearX]*fMatrix[kTranslateY] -
            fMatrix[kScaleY]*fMatrix[kTranslateX];

        GCoordinate newTranslateY =
            fMatrix[kShearY]*fMatrix[kTranslateX] -
            fMatrix[kScaleX]*fMatrix[kTranslateY];

        GCoordinate newHomogeneous =
            fMatrix[kScaleX]*fMatrix[kScaleY] -
            fMatrix[kShearX]*fMatrix[kShearY];

        GCoordinate newScaleX =
                fMatrix[kScaleY]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveY]*fMatrix[kTranslateY];
        GCoordinate newShearX =
                fMatrix[kPerspectiveY]*fMatrix[kTranslateX] -
                fMatrix[kShearX]*fMatrix[kHomogeneous];
        GCoordinate newShearY =
                fMatrix[kPerspectiveX]*fMatrix[kTranslateY] -
                fMatrix[kShearY]*fMatrix[kHomogeneous];
        GCoordinate newScaleY =
                fMatrix[kScaleX]*fMatrix[kHomogeneous] -
                fMatrix[kPerspectiveX]*fMatrix[kTranslateX];
        GCoordinate newPerspectiveX =
                fMatrix[kShearY]*fMatrix[kPerspectiveY] -
                fMatrix[kPerspectiveX]*fMatrix[kScaleY];
        GCoordinate newPerspectiveY =
                fMatrix[kPerspectiveX]*fMatrix[kShearX] -
                fMatrix[kScaleX]*fMatrix[kPerspectiveY];

        fMatrix[kScaleX]=newScaleX;
        fMatrix[kScaleY]=newScaleY;

        fMatrix[kShearX]=newShearX;
        fMatrix[kShearY]=newShearY;

        fMatrix[kTranslateX]=newTranslateX;
        fMatrix[kTranslateY]=newTranslateY;

        fMatrix[kPerspectiveX]=newPerspectiveX;
        fMatrix[kPerspectiveY]=newPerspectiveY;

        fMatrix[kHomogeneous]=newHomogeneous;
        break;
        }
    }
    if (fAccelerator != kPrIdentity) updateTimeStamp();
}

/*================
||
|| IGrafMatrix::normalize
||
*/

bool IGrafMatrix::normalize()
{
    if (fAccelerator != kPrGeneric) return (true);
    GCoordinate h = fMatrix[kHomogeneous];
    if (!h) return (false);
    updateTimeStamp();
    GCoordinate* mat = fMatrix;
    *mat++ /= h;
    *mat++ /= h;
    *mat++ /= h;
    *mat++ /= h;
    *mat++ /= h;
    *mat++ /= h;
    *mat++ /= h;
    *mat++ /= h;
    *mat = 1.0;
    return (true);
}

/*================
||
|| IGrafMatrix::operator=
||
*/

IGrafMatrix& IGrafMatrix::operator= (const IGrafMatrix& matrix)
{
    fAccelerator = matrix.fAccelerator;
    setTimeStamp(matrix.timeStamp());
    if (fAccelerator == kPrTranslate) {
        fTranslateX = matrix.fTranslateX;
        fTranslateY = matrix.fTranslateY;
    }
    else if (fAccelerator == kPrIdentity) {
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
    else
    {
        if (fMatrix == 0 /*NIL*/)
            fMatrix = new (::kSameHeap, this) GCoordinate[9];
        const GCoordinate* src = matrix.fMatrix;
        GCoordinate* dest = fMatrix;
        for(long i=9; i--;)
            *dest++ = *src++;
        fRotateAccelerator = matrix.fRotateAccelerator;
    }
    return (*this);
}



bool IGrafMatrix::operator!= (const IGrafMatrix& obj) const
{
    return !operator== (obj);
}

bool IGrafMatrix::operator== (const IGrafMatrix& obj) const
{
    IGrafMatrix* matrix = (IGrafMatrix*) &obj;
    if (fAccelerator == kPrIdentity && matrix->fAccelerator == kPrIdentity)
         return true;
    if (fAccelerator == kPrTranslate &&
        matrix->fAccelerator == kPrTranslate) {
        return (fTranslateX == matrix->fTranslateX &&
                fTranslateY == matrix->fTranslateY);
    }
    if (fMatrix == 0 /*NIL*/ || matrix->fMatrix == 0 /*NIL*/)
        return false;
    if (fAccelerator == kPrRotate)
        return ((
           (fMatrix[kDegrees] == matrix->fMatrix[kDegrees]) &&
           (fMatrix[kCenterX] == matrix->fMatrix[kCenterX]) &&
           (fMatrix[kCenterY] == matrix->fMatrix[kCenterY])));
    register const GCoordinate* source = fMatrix;
    register const GCoordinate* dest = matrix->fMatrix;
    if (*dest++ != *source++) return (false);
    if (*dest++ != *source++) return (false);
    if (*dest++ != *source++) return (false);

    if (*dest++ != *source++) return (false);
    if (*dest++ != *source++) return (false);
    if (*dest++ != *source++) return (false);

    if (*dest++ != *source++) return (false);
    if (*dest++ != *source++) return (false);
    if (*dest != *source) return (false);
    return (true);
}

IDataStream&    IGrafMatrix::operator>>= (IDataStream& toWhere) const
{
    writeToStream(toWhere);
    return toWhere;
}

IDataStream&    IGrafMatrix::operator<<= (IDataStream& fromWhere)
{
    readFromStream(fromWhere);
    return fromWhere;
}

void IGrafMatrix::writeToStream(IDataStream& toWhere) const
{
    long(fAccelerator) >>= toWhere;
    switch (fAccelerator) {
        case kPrIdentity: break;
        case kPrTranslate:
            fTranslateX >>= toWhere;
            fTranslateY >>= toWhere;
            break;
        case kPrRotate: {
        long(fRotateAccelerator) >>= toWhere;
            fMatrix[kDegrees] >>= toWhere;
            fMatrix[kCenterX] >>= toWhere;
            fMatrix[kCenterY] >>= toWhere;
            fMatrix[kSin] >>= toWhere;
            fMatrix[kCos] >>= toWhere;
            }
            break;
        case kPrGeneric:
            fMatrix[kPerspectiveX] >>= toWhere;
            fMatrix[kPerspectiveY] >>= toWhere;
            fMatrix[kHomogeneous] >>= toWhere;
            // fall through
        case kPrAffine:
            fMatrix[kShearX] >>= toWhere;
            fMatrix[kShearY] >>= toWhere;
            // fall through
        case kPrScale:
            fMatrix[kScaleX] >>= toWhere;
            fMatrix[kScaleY] >>= toWhere;
            fMatrix[kTranslateX] >>= toWhere;
            fMatrix[kTranslateY] >>= toWhere;
            break;
        default:
            break;
    }
}

void IGrafMatrix::readFromStream(IDataStream& fromWhere)
{
    unsigned long  temp;
    temp <<= fromWhere;

    fAccelerator = (EPrMatrixType) temp;

    if (fAccelerator==kPrIdentity || fAccelerator==kPrTranslate) {
        delete [] fMatrix;
        fMatrix = 0 /*NIL*/;
    }
    else {
        SetInternalMatrixToIdentity();
    }

    switch (fAccelerator) {
        case kPrIdentity:
            break;
        case kPrTranslate:
            fTranslateX <<= fromWhere;
            fTranslateY <<= fromWhere;
            break;
        case kPrRotate: {
            long  temp;
        temp <<= fromWhere;
            fRotateAccelerator = (ERotateType) temp;
            fMatrix[kDegrees] <<= fromWhere;
            fMatrix[kCenterX] <<= fromWhere;
            fMatrix[kCenterY] <<= fromWhere;
            fMatrix[kSin] <<= fromWhere;
            fMatrix[kCos] <<= fromWhere;
            }
            break;
        case kPrGeneric:
            fMatrix[kPerspectiveX] <<= fromWhere;
            fMatrix[kPerspectiveY] <<= fromWhere;
            fMatrix[kHomogeneous] <<= fromWhere;
            // fall through
        case kPrAffine:
            fMatrix[kShearX] <<= fromWhere;
            fMatrix[kShearY] <<= fromWhere;
            // fall through
        case kPrScale:
            fMatrix[kScaleX] <<= fromWhere;
            fMatrix[kScaleY] <<= fromWhere;
            fMatrix[kTranslateX] <<= fromWhere;
            fMatrix[kTranslateY] <<= fromWhere;
            break;
        default:
            break;
    }

    if (fAccelerator==kPrIdentity) resetTimeStamp();
    else updateTimeStamp();
}

/*================
||
|| IGrafMatrix::GetAccelerator
||
*/

IGrafMatrix::EPrMatrixType IGrafMatrix::GetAccelerator(const GCoordinate matrix []) const
{
    // ----- perspective -----
    if (matrix[kPerspectiveX] != 0.0 ||
        matrix[kPerspectiveY] != 0.0 ||
        matrix[kHomogeneous]  != 1.0)
        return (kPrGeneric);
    // ----- shear -----
    if (matrix[kShearX] != 0.0 || matrix[kShearY] != 0.0)
        return (kPrAffine);
    // ----- scale -----
    if (matrix[kScaleX] != 1.0 || matrix[kScaleY] != 1.0)
        return (kPrScale);
    // ----- translate -----
    if (matrix[kTranslateX] != 0.0 || matrix[kTranslateY] != 0.0)
        return (kPrTranslate);
    // ----- identity -----
    return (kPrIdentity);
}


/*================
||
|| IGrafMatrix::ComputeCosAndSin
||
*/

void IGrafMatrix::ComputeCosAndSin()
{
    GDegrees angle = fMatrix[kDegrees];
    GCoordinate c = 0, s = 0;

    // Special-case rectilinear rotations because Cos and Sin don't return
    // the exact values(0, -1, or 1) for these cases

    if (angle == 0)
        c = 1;
    else if (angle == 90)
        s = 1;
    else if (angle == 180)
        c = -1;
    else if (angle == 270)
        s = -1;
    else    // general case
    {
        GDegrees angleInRadians = angle * kDegreesToRadians;
        c = Cos(angleInRadians);
        s = Sin(angleInRadians);
    }

    fMatrix[kCos] = c;
    fMatrix[kSin] = s;
}


/*================
||
||  RotateToAffine
||
||  rotate = | cos  sin  0 |
||           | -sin cos  0 |
||           | Tx   Ty   1 |
|| where Tx = -Cx*(cos-1.0) + Cy*(sin)
||   and Ty = -Cx*(sin) - Cy*(cos-1.0)
|| * If need be: I can optimize the following
||
*/

void IGrafMatrix::RotateToAffine(
    IGrafMatrix::ERotateType rotateAccelerator,
    const GCoordinate rotateMatrix [],
    GCoordinate genericMatrix []) const
{
    GCoordinate centerX = rotateMatrix[kCenterX];
    GCoordinate centerY = rotateMatrix[kCenterY];
    GCoordinate nCos = 1.0;
    GCoordinate nSin = 0.0;
    switch (rotateAccelerator) {
        case kRotate0:
            break;
        case kRotate90:
            nCos = 0.0;
            nSin = 1.0;
            break;
        case kRotate180:
            nCos = -1.0;
            nSin = 0.0;
            break;
        case kRotate270:
            nCos = 0.0;
            nSin = -1.0;
            break;
        case kSlowRotate:
            nCos = rotateMatrix[kCos];
            nSin = rotateMatrix[kSin];
    }
    GCoordinate tx = -centerX*(nCos-1.0) + centerY*nSin;
    GCoordinate ty = -centerX*nSin - centerY*(nCos-1.0);
    GCoordinate* dest = genericMatrix;
    *dest++ = nCos;
    *dest++ = nSin;
    *dest++ = 0.0;
    *dest++ = -nSin;
    *dest++ = nCos;
    *dest++ = 0.0;
    *dest++ = tx;
    *dest++ = ty;
    *dest = 1.0;
}

/*================
||
||  PreMatrixMult
||  matrix 2 = (matrix 1)*(matrix 2)
||  Note: very brute force multiplication:
||      This code will ## DEFINITELY ## be revisited
*/

void IGrafMatrix::PreMatrixMult(
    long matrixType,
    const GCoordinate matrix1 [])
{
    // Make a pointer to the array matrices to reduce dereferencing
    long cols = ((matrixType==kAffineAffine) ? 2: 3);
    const GCoordinate * mat1 = matrix1;
    GCoordinate * mat2 = fMatrix;
    long i;
    for(i=0; i<cols; i++) {
                                //  i = column of mat2
        GCoordinate a = *mat2;      //  a = mat2[0][i]
        GCoordinate b = *(mat2+3);  //  b = mat2[1][i]
        GCoordinate c = *(mat2+6);  //  c = mat2[2][i]
        GCoordinate * colMat = mat2;    //  colMat = mat2 [0][i]
        const GCoordinate * rowMat = mat1; //
        for(long j=0; j<3; j++) {
            // mat2[j][i] = mat1[0][j]*a + mat1[1][j]*b + mat1[2][j]*c
            *colMat  = *(rowMat++)*a;
            *colMat += *(rowMat++)*b;
            if (matrixType==kAffineAffine) {
                // the c is only useful in the last row
                if (j==2) *colMat += c;
                rowMat++;
            }
            else *colMat += *(rowMat++)*c;
            colMat += 3;
        }
        mat2++;
    }
    fAccelerator = GetAccelerator(fMatrix);
    if (fAccelerator==kPrIdentity) {
        resetTimeStamp();
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
    if (fAccelerator==kPrTranslate) {
        fTranslateX = fMatrix[kTranslateX];
        fTranslateY = fMatrix[kTranslateY];
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
}

/*================
||
||  PostMatrixMult
||  matrix 2 = (matrix 2)*(matrix 1)
||  Note: very brute force multiplication:
||      This code will ## DEFINITELY ## be revisited
*/

void IGrafMatrix::PostMatrixMult(
    long matType,
    const GCoordinate matrix1 [])
{
    // Make a pointer to the array matrices to reduce dereferencing
    long numCols = ((matType==kAffineAffine) ? 2: 3);
    const GCoordinate * mat1 = matrix1;
    GCoordinate * mat2 = fMatrix;
    for(long row=0; row<3; row++) {
        GCoordinate a = *mat2;              //  a = mat2[row][0]
        GCoordinate b = *(mat2+1);          //  b = mat2[row][1]
        GCoordinate c = *(mat2+2);          //  c = mat2[row][2]
        const GCoordinate * colMat = mat1;  //  colMat = mat1 [0][0]
        GCoordinate * rowMat = mat2;        //  rowMat = fMatrix [0][0]
        for(long col=0; col<numCols; col++) {
            // mat2[row][col] = mat1[0][col]*a + mat1[1][col]*b + mat1[2][col]*c
            *rowMat  = *(colMat)*a;
            colMat+=3;
            *rowMat += *(colMat)*b;
            colMat+=3;
            if (matType==kAffineAffine) {
                if (row==2) *rowMat += *(colMat);
            }
            else *rowMat += *(colMat)*c;
            rowMat++;
            colMat -= 5;
        }
        mat2 += 3;
    }
    fAccelerator = GetAccelerator(fMatrix);
    if (fAccelerator==kPrIdentity) {
        resetTimeStamp();
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
    if (fAccelerator==kPrTranslate) {
        fTranslateX = fMatrix[kTranslateX];
        fTranslateY = fMatrix[kTranslateY];
        delete [] fMatrix; fMatrix = 0 /*NIL*/;
    }
}

// Note: assumes that 0 <= degrees <= 360 (FirstCycle used)
IGrafMatrix::ERotateType IGrafMatrix::GetFastRotate(GCoordinate degrees) const
{
    if (degrees == 0.0) return (kRotate0);
    if (degrees == 90.0) return (kRotate90);
    if (degrees == 180) return (kRotate180);
    if (degrees == 270) return (kRotate270);
    return (kSlowRotate);
}

//=================//
// IGQuadrilateral //
//=================//

// private

IGQuadrilateral::EVertexOrientation IGQuadrilateral::GetVertexOrientation(
    const IGPoint2D& a, const IGPoint2D& b, const IGPoint2D& c) const
{
    EVertexOrientation result;
    GCoordinate x1 = b.fX-a.fX;
    GCoordinate y1 = b.fY-a.fY;
    GCoordinate x2 = c.fX-b.fX;
    GCoordinate y2 = c.fY-b.fY;
    GCoordinate sign = x1*y2-y1*x2;
    if (sign > 0.0)
    result = kClockWise;
    else if (sign < 0.0)
    result = kCounterClockwise;
    else result = kDegenerate;
    return result;
}

/*================
||
||  IGQuadrilateral::isDegenerate
||
*/

bool IGQuadrilateral::isDegenerate() const
{
    bool result = false;
    IGPoint2D points[4];

    points[0] = point(0);
    points[1] = point(1);
    points[2] = point(2);
    points[3] = point(3);

    const IGPoint2D* thisPt = (&points[0]);

    // first handle the boundary vertices that require
    // wrapping around fPoints

    EVertexOrientation primaryOrientation = GetVertexOrientation(points[2], points[3], points[0]);
    bool isCollinear = (primaryOrientation == kDegenerate);
    EVertexOrientation currentOrientation = GetVertexOrientation(points[3], points[0], points[1]);

    // if the first vertex was collinear and this one isn't
    // set up the primaryOrientation and turn the flag off

    if ((isCollinear) &&(currentOrientation != kDegenerate))
    {
        isCollinear = false;
        primaryOrientation = currentOrientation;
    }
    else // if the currentOrientation is collinear then substitute
        // in the primaryOrientation since it is not
        if (currentOrientation == kDegenerate)
        currentOrientation = primaryOrientation;

    if (currentOrientation == primaryOrientation)
    {
        long numPoints = 2;
        // loop over all the vertices up to the ones
        // covered by the above tests which required
        // wrapping around the point list
        while ((numPoints-- > 0) &&(result == false))
        {
            currentOrientation = GetVertexOrientation(*thisPt, *(thisPt+1), *(thisPt+2));
            if (currentOrientation != kDegenerate)
            {
                // We have something other than collinear points.
                // then turn the flag off and set the primaryOrientation
                isCollinear = false;

                // otherwise if the currentOrientation is not the
                // same as the primaryOrientation we have a concave
                // polygon so set result to true
                if (currentOrientation != primaryOrientation)
                        result = true;

                primaryOrientation = currentOrientation;
            }
            thisPt++;
        }
        if (isCollinear) result = true; // all points were collinear
    }
    else result = true;

    return (result);
}

IGQuadrilateral::IGQuadrilateral() : fPoints(4)
{
    fPoints[0] = IGPoint2D::origin();
    fPoints[1] = IGPoint2D::origin();
    fPoints[2] = IGPoint2D::origin();
    fPoints[3] = IGPoint2D::origin();
}
IGQuadrilateral::IGQuadrilateral(const IGQuadrilateral& quad) : fPoints(4)
{
    fPoints = quad.fPoints;
}

IGQuadrilateral::IGQuadrilateral(
            const IGPoint2D& p0,
            const IGPoint2D& p1,
            const IGPoint2D& p2,
            const IGPoint2D& p3) : fPoints(4)
{
    fPoints[0] = p0;
    fPoints[1] = p1;
    fPoints[2] = p2;
    fPoints[3] = p3;
}

IGQuadrilateral::IGQuadrilateral(
            const IGRect2D& r) : fPoints(4)
{
    fPoints[0] = r.topLeft();
    fPoints[1] = r.topRight();
    fPoints[2] = r.bottomRight();
    fPoints[3] = r.bottomLeft();
}

void IGQuadrilateral::points(IGPoint2D& p0, IGPoint2D& p1, IGPoint2D& p2, IGPoint2D& p3) const
{
    p0 = fPoints[0];
    p1 = fPoints[1];
    p2 = fPoints[2];
    p3 = fPoints[3];
}

IGPoint2D IGQuadrilateral::point(unsigned long index) const
{
    ParameterException((index > 3), IGraphicException::kIndexAboveRange);

    return fPoints[index];
}

void IGQuadrilateral::setPoint(unsigned long index, const IGPoint2D& p)
{
    ParameterException((index > 3), IGraphicException::kIndexAboveRange);

    fPoints[index] = p;
}

void IGQuadrilateral::transformBy(const IGrafMatrix& mat)
{
    mat.transformPoints(fPoints);
}

IGRect2D IGQuadrilateral::bounds() const
{
    IGRect2D g;
    g.setToPoint(fPoints[0]);
    g.extendTo(fPoints[1]);
    g.extendTo(fPoints[2]);
    g.extendTo(fPoints[3]);
    return g;
}

IGQuadrilateral::~IGQuadrilateral()
{
}

IDataStream& IGQuadrilateral::operator>>=(IDataStream& toWhere) const
{
    writeToStream(toWhere);
    return toWhere;
}

IDataStream& IGQuadrilateral::operator<<=(IDataStream& fromWhere)
{
    readFromStream(fromWhere);
    return fromWhere;
}

void IGQuadrilateral::writeToStream(IDataStream& toWhere) const
{
    fPoints >>= toWhere;
}

void IGQuadrilateral::readFromStream(IDataStream& fromWhere)
{
    fPoints <<= fromWhere;
}

IGQuadrilateral& IGQuadrilateral::operator=(const IGQuadrilateral& Src)
{
    if (&Src != this)
        fPoints = Src.fPoints;

    return (*this);
}

bool IGQuadrilateral::operator==(const IGQuadrilateral& Src) const
{
    return (fPoints == Src.fPoints);
}

bool IGQuadrilateral::operator!=(const IGQuadrilateral& Src) const
{
    return (!(operator== (Src)));
}
