/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 14 1.29.1.6 source/albert/graph2d/ipaints.cpp, 2d, ioc.v400, 990114 
/*================
  ||
  ||  File:   Paints.C
  ||
  ||  What:   Albert Paint Attribute
  ||
  ||
  ||          All Rights Reserved.
||
*/

#include <ipaints.hpp>
#include <ibundles.hpp>
#include <igimage.hpp>

#include <stdio.h>

#include <inumeric.hpp>
#include <grtype.hpp>
#include <idatstrm.hpp>
#include <grstrmod.hpp>
#include <grassert.hpp>
#include <grafdevs.hpp>
#include <iprimlck.hpp>
#include <itrace.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
#define _IPAINTS_CPP_
#include <ipagetun.h>
#endif

//==============//
// IMaskPattern //
//==============//


IMaskPattern::IMaskPattern(
			   unsigned char row0,
			   unsigned char row1,
			   unsigned char row2,
			   unsigned char row3,
			   unsigned char row4,
			   unsigned char row5,
			   unsigned char row6,
			   unsigned char row7) :

  fPattern(IGrafDevice::kCustomPattern),
#ifndef IC_MOTIF
  fBits(new unsigned short[8])
#else
  fBits(new unsigned char[8])
#endif
{
  fBits[0] = row0;
  fBits[1] = row1;
  fBits[2] = row2;
  fBits[3] = row3;
  fBits[4] = row4;
  fBits[5] = row5;
  fBits[6] = row6;
  fBits[7] = row7;
}

IMaskPattern::IMaskPattern(const IMaskPattern& source) :
  fPattern(source.fPattern),
  fBits(0 /*NIL*/)
{
  if (fPattern == IGrafDevice::kCustomPattern)
    {
#ifndef IC_MOTIF
			fBits = new unsigned short[8];
#else
 			fBits = new unsigned char[8];
#endif

      fBits[0] = source.fBits[0];
      fBits[1] = source.fBits[1];
      fBits[2] = source.fBits[2];
      fBits[3] = source.fBits[3];
      fBits[4] = source.fBits[4];
      fBits[5] = source.fBits[5];
      fBits[6] = source.fBits[6];
      fBits[7] = source.fBits[7];
    }
}

IMaskPattern::~IMaskPattern()
{
  delete [] fBits;
}

IMaskPattern& IMaskPattern::operator=(const IMaskPattern& source)
{
  if (this != &source)
    {
      if (source.fPattern == IGrafDevice::kCustomPattern)
	{
	  if (fPattern != IGrafDevice::kCustomPattern)
#ifndef IC_MOTIF
			fBits = new unsigned short[8];
#else
 			fBits = new unsigned char[8];
#endif

	  fBits[0] = source.fBits[0];
	  fBits[1] = source.fBits[1];
	  fBits[2] = source.fBits[2];
	  fBits[3] = source.fBits[3];
	  fBits[4] = source.fBits[4];
	  fBits[5] = source.fBits[5];
	  fBits[6] = source.fBits[6];
	  fBits[7] = source.fBits[7];

#ifdef IC_MOTIF
	  fPattern = source.fPattern;
#endif
	}

      else
	{
	  if (fPattern == IGrafDevice::kCustomPattern)
	    {
	      delete [] fBits;
	      fBits = 0; // NIL
	    }

	  fPattern = source.fPattern;

	} // else source pattern was one of the standard ones

    } // only if source and destn are different

  return *this;

} // IMaskPattern::operator=



bool IMaskPattern::operator==(const IMaskPattern& rhs) const
{
  if (fPattern != IGrafDevice::kCustomPattern)
    {
      return (fPattern == rhs.fPattern);
    }
  else
    {
      return (
	      rhs.fPattern == IGrafDevice::kCustomPattern &&
	      fBits[0] == rhs.fBits[0] &&
	      fBits[1] == rhs.fBits[1] &&
	      fBits[2] == rhs.fBits[2] &&
	      fBits[3] == rhs.fBits[3] &&
	      fBits[4] == rhs.fBits[4] &&
	      fBits[5] == rhs.fBits[5] &&
	      fBits[6] == rhs.fBits[6] &&
	      fBits[7] == rhs.fBits[7]);
    }
}

bool IMaskPattern::operator!=(const IMaskPattern& rhs) const
{
  return !(operator==(rhs));
}

IMaskPattern::IMaskPattern(int pattern) :
  fPattern(pattern),
  fBits(0 /*NIL*/)
{
  ParameterAssert( (pattern != IGrafDevice::kCustomPattern),
		   "Custom pattern requires pattern bits." );
}



int IMaskPattern::pattern() const
{
  return fPattern;
}

const unsigned char* IMaskPattern::bits() const
{
  return (const unsigned char*)fBits;
}

IDataStream& IMaskPattern::operator>>=(IDataStream& toWhere) const
{
  IStreamOutFrame streamOutFrame(toWhere);
  long(fPattern) >>= toWhere;
  bool noBits = true;
  if (fBits != 0 /*NIL*/)
    noBits = false;
  long(noBits) >>= toWhere;
  if (! noBits)
    {
      for (long i = 0; i < 8;  i++)
	fBits[i] >>= toWhere;
    }
  return toWhere;
}

IDataStream& IMaskPattern::operator<<=(IDataStream& fromWhere)
{
  IStreamInFrame streamInFrame(fromWhere);
  long temp;
  temp <<= fromWhere;
  fPattern = int(temp);

  //
  //  Here we try and avoid the 'new' and 'delete' operations
  //  for efficiency reasons as the amount of memory being
  //  handled is only 8 bytes.
  //

  bool noBits;
  temp <<= fromWhere;
  noBits = bool(temp);
  if (! noBits)
    {
      if ( fBits == 0  /* NIL */ )
	{
#ifndef IC_MOTIF
			fBits = new unsigned short[8];
#else
 			fBits = new unsigned char[8];
#endif
	}

      for (long i = 0; i < 8;  i++)
	fBits[i] <<= fromWhere;

    } // if there is bit pattern in the stream

  else
    {
      delete [] fBits;
      fBits = 0 /*NIL*/;
    }

  return fromWhere;

} // IMaskPattern::operator<<=

/*================
//
// Lazy evaluation support for the IMaskPattern constants
//
*/
struct IMaskPatternStatics {
    IMaskPatternStatics();

    IMaskPattern fSolid;
    IMaskPattern fDiagonalDown;
    IMaskPattern fCross;
    IMaskPattern fDiagonalCross;
    IMaskPattern fDiagonalUp;
    IMaskPattern fHorizontal;
    IMaskPattern fVertical;
};

IMaskPatternStatics::IMaskPatternStatics()
    : fSolid        (IGrafDevice::kSolidPattern)
#ifndef IC_MOTIF
    , fDiagonalDown (IGrafDevice::kDiagonalDownPattern)
    , fCross        (IGrafDevice::kCrossPattern)
    , fDiagonalCross(IGrafDevice::kDiagonalCrossPattern)
    , fDiagonalUp   (IGrafDevice::kDiagonalUpPattern)
    , fHorizontal   (IGrafDevice::kHorizontalPattern)
    , fVertical     (IGrafDevice::kVerticalPattern)
#else
    , fDiagonalDown (0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x01)
    , fCross        (0x08,0x08,0x08,0x08,0xff,0x08,0x08,0x08)
    , fDiagonalCross(0x80,0x41,0x22,0x14,0x08,0x14,0x22,0x41)
    , fDiagonalUp   (0x01,0x02,0x04,0x08,0x10,0x20,0x40,0x80)
    , fHorizontal   (0x00,0x00,0x00,0xff,0x00,0x00,0x00,0xff)
    , fVertical     (0x11,0x11,0x11,0x11,0x11,0x11,0x11,0x11)
#endif
{
}

static IMaskPatternStatics* gStatics = 0;

static IMaskPatternStatics *statics()
{
    if (!gStatics) {
        IPrimalLock lock;
        if (!gStatics) {
            gStatics = new IMaskPatternStatics;
        }
    }
    return gStatics;
}

const IMaskPattern& IMaskPattern::solid()           { return statics()->fSolid; }
const IMaskPattern& IMaskPattern::diagonalDown()    { return statics()->fDiagonalDown; }
const IMaskPattern& IMaskPattern::cross()           { return statics()->fCross; }
const IMaskPattern& IMaskPattern::diagonalCross()   { return statics()->fDiagonalCross; }
const IMaskPattern& IMaskPattern::diagonalUp()      { return statics()->fDiagonalUp; }
const IMaskPattern& IMaskPattern::horizontal()      { return statics()->fHorizontal; }
const IMaskPattern& IMaskPattern::vertical()        { return statics()->fVertical; }

//========//
// IPaint //
//========//

/*================
||
||  IPaint destructor and constructors
||
*/

StreamableDefinitionsMacro(IPaint, gGraphicsStreamModule);

IPaint::~IPaint()
{
  IFUNCTRACE_DEVELOP();
  if (fColor != &fBaseColor)
    delete fColor;
  if (fMaskPattern->pattern() == IGrafDevice::kCustomPattern)
    delete fMaskPatStorage;
  delete fImage;
}

IPaint::IPaint() :
  fBaseColor(),
  fMaskPattern(&IMaskPattern::solid()),
  fMaskPatStorage(0 /*NIL*/),
  fPatternPhase(IGPoint2D::origin()),
  fImage(0 /*NIL*/)
{
  IFUNCTRACE_DEVELOP();
  fColor = &fBaseColor;   // always initialise this to the default color...
}

IPaint::IPaint(
	       const IBaseColor& aColor,
	       const IMaskPattern& maskPattern,
	       const IGPoint2D& patternPhase) :

  fBaseColor(aColor),
  fMaskPattern(&maskPattern),
  fMaskPatStorage(0 /*NIL*/),
  fPatternPhase(patternPhase),
  fImage(0 /*NIL*/)
{
  IFUNCTRACE_DEVELOP();
  if (GraphicTypeid(aColor) == GraphicTypeid(fBaseColor))
    {
      fColor = &fBaseColor;   // always initialise this to the default color...
    }
  else
    {
      fColor = GraphicCopy(aColor);
    }

  if (maskPattern.pattern() == IGrafDevice::kCustomPattern)
    fMaskPattern = fMaskPatStorage = new IMaskPattern(maskPattern);
}

IPaint::IPaint(
	       const IGImage& imagePattern,
	       const IGPoint2D& patternPhase) :

  fBaseColor(),
  fMaskPattern(&IMaskPattern::solid()),
  fMaskPatStorage(0 /*NIL*/),
  fPatternPhase(patternPhase),
  fImage(new IGImage(imagePattern))
{
  IFUNCTRACE_DEVELOP();
  fColor = &fBaseColor;   // always initialise this to the default color...
}

/*================
||
||  IPaint copy constructor
||
*/

IPaint::IPaint(const IPaint& source) :
  fMaskPatStorage(0 /*NIL*/)
{
  IFUNCTRACE_DEVELOP();
  if (GraphicTypeid(*source.fColor) == GraphicTypeid(fBaseColor))
    {
      fBaseColor = *source.fColor;
      fColor = &fBaseColor;
    }
  else
    {
      fColor = GraphicCopy(*source.fColor);
    }

  fPatternPhase = source.fPatternPhase;

  if (source.fMaskPattern->pattern() != IGrafDevice::kCustomPattern)
    {
      fMaskPattern = source.fMaskPattern;
    }
  else
    {
      fMaskPattern = fMaskPatStorage = new IMaskPattern(*source.fMaskPattern);
    }

  if (!source.fImage)
    fImage = 0 /*NIL*/;
  else
    // PTOM fImage = GraphicCopy(*source.fImage);
    fImage = new IGImage(*source.fImage);
}

// getters ...

const IBaseColor* IPaint::color() const
{
  return fColor;
}


const IGPoint2D* IPaint::patternPhase() const
{
  return &fPatternPhase;
}

const IMaskPattern* IPaint::maskPattern() const
{
  return fMaskPattern;
}

const IGImage* IPaint::imagePattern() const
{
  return fImage;
}

// setters...

void IPaint::setColor(const IBaseColor& color)
{
  if (GraphicTypeid(color) != GraphicTypeid(*fColor))
    {
      if (fColor != &fBaseColor)
	delete fColor;

      if (GraphicTypeid(color) == GraphicTypeid(fBaseColor))
	{       // copy it into the fBaseColor
	  fBaseColor = color;
	  fColor = &fBaseColor;
	}
      else
	{       // clone the source
	  fColor = GraphicCopy(color);
	}
    }
  else
    {       // reuse the storage...
      *fColor = color;
    }
}

void IPaint::setMaskPattern(const IMaskPattern& maskPattern)
{
  IFUNCTRACE_DEVELOP();
  if (maskPattern.pattern() != IGrafDevice::kCustomPattern)
    {
      if (fMaskPattern->pattern() == IGrafDevice::kCustomPattern)
	{
	  delete fMaskPatStorage;
	  fMaskPatStorage = 0 /*NIL*/;
	}
      fMaskPattern = &maskPattern;
    }
  else
    {
      if (fMaskPattern->pattern() != IGrafDevice::kCustomPattern)
	fMaskPattern = fMaskPatStorage = new IMaskPattern(maskPattern);
      else    // storage is already allocated, fMaskPattern is already pointing to the storage
	*fMaskPatStorage = maskPattern;
    }
}

void IPaint::setPatternPhase(const IGPoint2D& patternPhase)
{
  fPatternPhase = patternPhase;
}

void IPaint::setImagePattern(const IGImage& imagePattern)
{
  IFUNCTRACE_DEVELOP();
  if (fImage)
    delete fImage;

  // PTOM fImage = GraphicCopy(imagePattern);
  fImage = new IGImage (imagePattern);
}

void IPaint::removeImagePattern()
{
  if (fImage)
    {
      delete fImage;
      fImage = 0 /*NIL*/;
    }
}


// === Base Functionality ===

IPaint& IPaint::operator=(const IPaint& source)
{
  IFUNCTRACE_DEVELOP();
  if (this == &source)
    return *this;

  setColor(*source.fColor);
  setPatternPhase(source.fPatternPhase);
  setMaskPattern(*source.fMaskPattern);

  if (fImage)
    delete fImage;
  if (source.fImage)
    // PTOM fImage = GraphicCopy(*(source.fImage));
    fImage = new IGImage (*(source.fImage));
  else
    fImage = 0 /*NIL*/;

  return (*this);

} // IPaint::operator=



bool IPaint::operator!=(const IPaint& obj) const
{
  return !operator==(obj);
}
bool IPaint::operator==(const IPaint& source) const
{
  if (GraphicTypeid(*this) != GraphicTypeid(source) ||
      *fColor != *(source.fColor) ||
      fPatternPhase != source.fPatternPhase ||
      *fMaskPattern != *source.fMaskPattern ||
      (fImage && !source.fImage) ||
      (!fImage && source.fImage))
    /******* don't compare images till images can be streamed ****
      || (fImage && (*fImage != *source.fImage)))
      ***************************************************************/
    return false;
  else
    return true;
}


void IPaint::writeToStream( IDataStream& toWhere ) const
{
  IFUNCTRACE_DEVELOP();
  IStreamOutFrame streamOutFrame(toWhere);

  bool streamPtr=true;
  if (fColor == &fBaseColor)
    streamPtr=false;
  long(streamPtr) >>= toWhere;
  if (streamPtr)
    ::writeObject(fColor, toWhere);
  else
    fBaseColor >>= toWhere;

  streamPtr=true;
  if (fMaskPattern->pattern() == IGrafDevice::kCustomPattern)
    streamPtr = false;
  long(streamPtr) >>= toWhere;
  if (streamPtr)
    {
      *fMaskPattern >>= toWhere;
    }else
      {
	*fMaskPatStorage >>= toWhere;
      }

  fPatternPhase >>= toWhere;
  bool nilPtr = true;
  if (fImage != 0 /*NIL*/)
    nilPtr = false;
  long(nilPtr) >>= toWhere;
  if (!nilPtr)
    *fImage >>= toWhere; // does nothing; see operator==
}

void IPaint::readFromStream( IDataStream& fromWhere )
{
  IFUNCTRACE_DEVELOP();
  IStreamInFrame streamInFrame(fromWhere);

  bool streamPtr;
  long temp;
  temp <<= fromWhere;
  streamPtr=bool(temp);

  if (fColor != &fBaseColor)
    delete fColor;
  if (streamPtr)
    {
      ::readObject(fColor, fromWhere);
    }
  else
    {
      fBaseColor <<= fromWhere;
      fColor = &fBaseColor;
    }

  temp <<= fromWhere;
  streamPtr=bool(temp);

  if (fMaskPattern->pattern() == IGrafDevice::kCustomPattern)
    {
      delete fMaskPatStorage;
      fMaskPatStorage = 0 /*NIL*/;
    }

  IMaskPattern maskPat(IMaskPattern::solid());
  maskPat <<= fromWhere;

  if (streamPtr)
    {
      IGrafDevice::EMaskPattern  pat = (IGrafDevice::EMaskPattern) maskPat.pattern ();
      switch (pat)
	{
	case IGrafDevice::kSolidPattern:
	  fMaskPattern = 	 &IMaskPattern::solid();
	  break;
	
	case IGrafDevice::kDiagonalDownPattern:
	  fMaskPattern = 	 &IMaskPattern::diagonalDown();
	  break;
	
	case IGrafDevice::kCrossPattern:
	  fMaskPattern = 	 &IMaskPattern::cross();
	  break;
	
	case IGrafDevice::kDiagonalCrossPattern:
	  fMaskPattern = 	 &IMaskPattern::diagonalCross();
	  break;
	
	case IGrafDevice::kDiagonalUpPattern:
	  fMaskPattern = 	 &IMaskPattern::diagonalUp();
	  break;
	
	case IGrafDevice::kHorizontalPattern:
	  fMaskPattern = 	 &IMaskPattern::horizontal();
	  break;
	
	case IGrafDevice::kVerticalPattern:
	  fMaskPattern = 	 &IMaskPattern::vertical();
	  break;
	
	default:
	  ParameterAssert (false, "");
	  break;

	} // switch

    } // if streamed out a standard pattern

  else
    {
      fMaskPattern = fMaskPatStorage = new IMaskPattern(maskPat);
    }

  fPatternPhase <<= fromWhere;

  bool nilPtr;
  temp <<= fromWhere;
  nilPtr = bool(temp);

  if(!nilPtr)
  {
    if (! fImage)
      fImage = new IGImage;

    *fImage <<= fromWhere; // does nothing; see operator==
  }
  else
  {
    delete fImage;
    fImage = 0 /* NIL */;
  }

} // IPaint::readFromStream


