// Revision: 88 1.19.1.20 source/ui/baseapp/ithread.cpp, thread, ioc.v400, 990114 
/*----------------------------------------------------------------------------*/
/* FILE NAME: ithread.cpp                                                     */
/*                                                                            */
/* DESCRIPTION:                                                               */
/*   Implementation of the class(es):                                         */
/*     IThread                                                                */
/*     ICurrentThread                                                         */
/*                                                                            */
/*                                                                            */
/* COPYRIGHT:                                                                 */
/*   IBM Open Class Library                                                   */
/*   Licensed Materials - Property of IBM                                     */
/*                                                                            */
/*   5645-001                                                                 */
/*   (C) Copyright IBM Corporation 1992, 1997                                 */
/*                                                                            */
/*   US Government Users Restricted Rights - Use, duplication, or             */
/*   disclosure restricted by GSA ADP Schedule Contract with IBM              */
/*   Corp.                                                                    */
/*                                                                            */
/*----------------------------------------------------------------------------*/
// Priority INT_MIN (-2147483647 - 1) + 1024 + 512

#ifdef IC_WIN
#pragma priority( -2147481624 )
#else
#pragma priority( -2147482112 )
#endif

#include <ibase.hpp>

extern "C" {
#ifdef IC_MOTIF
  #include <stdio.h>
  #include <unistd.h>
  #include <pthread.h>
  #include <sys/types.h>
  #include <sys/errno.h>
  #include <X11/Xmu/Editres.h>
  #ifdef IC_SUN
    #include <X11/Sunkeysym.h>       // Replace aix_keysym.h
    #include <sys/procfs.h>          // Needed for prpsinfo_t
    #include <sys/stat.h>            //
    #include <fcntl.h>               //
  #else
    #include <X11/aix_keysym.h>
    #include <procinfo.h>            // procsinfo structure definition
  #endif
  #include <X11/keysym.h>
  #include <Xm/Xm.h>
  #include <Xm/ScrollBar.h>
  #include <ipfx.h>
#else // ! IC_MOTIF
  #include <stdlib.h>
  #define INCL_DOSPROCESS
  #define INCL_DOSSESMGR
  #define INCL_DOSERRORS
  #define INCL_WINMESSAGEMGR
  #define INCL_WINFRAMEMGR
  #define INCL_WINHOOKS
  #include <iwindefs.h>
#endif // ! IC_MOTIF
}

#include <ithread.hpp>
#include <iapp.hpp>
#include <icconst.h>
#include <icolmap.hpp>
#include <iexcept.hpp>
#include <iframe.hpp>
#include <iframprv.hpp>
#include <ikeyevt.hpp>
#include <ingthrdp.hpp>
#include <ireslock.hpp>
#include <istring.hpp>
#include <ithreadp.hpp>
#include <itrace.hpp>
#include <iwindow.hpp>

#ifdef IC_MOTIF
  #include <ixdc.hpp>
  #include <ixconst.h>
  #include <iwinpriv.hpp>
  #include <imenuprv.hpp>
#endif // IC_MOTIF

#ifdef IC_WIN
  #include <iplatfrm.hpp>
#endif  // IC_WIN


// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ITHREAD_CPP_
  #include <ipagetun.h>
#endif


#ifdef IC_MOTIF

// Prototype for getprocs subroutine.  Does not appear in system headers.
extern "C" {
 int getprocs ( procsinfo*, int, fdsinfo*, int, pid_t*, int );
}

// Save copy of last button pressed event... initialize type field to 0
// Used in IPopUpMenu::show.
XButtonPressedEvent  ithreadLastButtonEvent = { 0 };

// Prototype for our XmColorProc function.
static void IThread__xmColorProc(XColor *bg_color, XColor *fg_color,
             XColor *sel_color, XColor *ts_color, XColor *bs_color);

// Pointer to default XmColorProc used by Motif.
static XmColorProc IThread__defaultColorProc = 0;

/*------------------------------------------------------------------------------
| XLibErrorHandler                                                             |
------------------------------------------------------------------------------*/
int XLibErrorHandler( Display *display, XErrorEvent *xevent )
{
  IThread::current().setXerrorCode ( xevent->error_code  );
  return 0;
}

/*------------------------------------------------------------------------------
| ICLUIWarningHandler                                                          |
------------------------------------------------------------------------------*/
static void ICLUIWarningHandler ( String warning )
{
  // This replaces the default XtErrorHandler which prints warnings to
  // the display. This annoyed our users so we are going to remove the
  // warnings.
  ITRACE_DEVELOP( IString( warning ) );
}

// currently we always call XtDispatch with ButtonPress and ButtonRelease messages
// regardless what the mouse handler returns. This is causing Montana problems so
// we agreed to work around it with this global variable.
bool fAlwaysDispatchMouseEvents = true;
#endif   // IC_MOTIF


/*------------------------------------------------------------------------------
                        S T A T I C    O B J E C T S
------------------------------------------------------------------------------*/

// Keys for IThread data stored using INonGUIThread::variable and setVariable.
// The fgKeyBase is used as a prefix for all of the variables.
const char* IThreadData::fgHMQKey               = "hmq";
const char* IThreadData::fgHABKey               = "hab";
const char* IThreadData::fgQueueSizeKey         = "queueSize";
const char* IThreadData::fgAutoInitGUIKey       = "autoInitGUI";
const char* IThreadData::fgCurrentThreadDataKey = "currentThreadData";
const char* IThreadData::fgMsgLoopSerialKey     = "msgLoopSerial";
const char* IThreadData::fgAppShellKey          = "appShell";
const char* IThreadData::fgXerrorCode           = "XErrorCode";
const char* IThreadData::fgIsTopLevelShell      = "isTopLevelShell";
const char* IThreadData::fgKeyBase              = "IThreadData::";


bool
  IThread::dfltAutoInitGUI = -1;

#ifdef IC_PM
long
  IThread::dfltQueueSize = 30;
#else
long
  IThread::dfltQueueSize = 0;
#endif

ICurrentThread
 *IThread::pCurrent = 0;


/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
|                                                                              |
| The role of the IThread constructors is twofold:                             |
|   1. They must locate the appropriate IStartedThread instance,               |
|      creating one if necessary, set the "thread" member to point             |
|      to this instance, and bump the IStartedThread use count.                |
|   2. Start the OS/2 thread (if necessary).                                   |
------------------------------------------------------------------------------*/
IThread::IThread ( const IThreadId& tid,
                   const IThreadHandle& handle  )
  : INonGUIThread( tid, handle )
  , fThreadData( 0 )
{
  IMODTRACE_ALL("IThread::IThread");
}


/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
------------------------------------------------------------------------------*/
IThread::IThread ( )
  : INonGUIThread( )
  , fThreadData( 0 )
{
}

/*------------------------------------------------------------------------------
| IThread::IThread                                                             |
------------------------------------------------------------------------------*/
IThread::IThread ( const IThread &aThread )
  : INonGUIThread( aThread )
  , fThreadData( 0 )
{
}

/*------------------------------------------------------------------------------
| IThread::newStartedThread                                                    |
------------------------------------------------------------------------------*/
IStartedThread* IThread::newStartedThread ( )
{
  return( Inherited::newStartedThread() );
}

/*------------------------------------------------------------------------------
| IThread::~IThread                                                            |
|                                                                              |
| The destructor must remove the reference to the associated IStartedThread.   |
------------------------------------------------------------------------------*/
IThread::~IThread ( )
{
}

/*------------------------------------------------------------------------------
| IThread::operator =                                                          |
|                                                                              |
| Attach this thread to aThread's associated started thread.                   |
------------------------------------------------------------------------------*/
IThread& IThread::operator = ( const IThread &aThread )
{
  Inherited::operator = ( aThread );
  return *this;
}

/*------------------------------------------------------------------------------
| IThread::startedThread                                                       |
|                                                                              |
| Returns the thread member.                                                   |
------------------------------------------------------------------------------*/
IStartedThread* IThread::startedThread ( ) const
{
  return( Inherited::startedThread() );
}

/*------------------------------------------------------------------------------
| IThread::stop                                                                |
|                                                                              |
| Issue DosKillThread.  If it fails, throw an exception.                       |
------------------------------------------------------------------------------*/
void IThread::stop ( )
{
  IMODTRACE_DEVELOP("IThread::stop");
  Inherited::stop();
}

/*------------------------------------------------------------------------------
| ICurrentThread::~ICurrentThread                                              |
|                                                                              |
| Empty dtor to prevent generated static version.                              |
------------------------------------------------------------------------------*/
ICurrentThread::~ICurrentThread ( )
{
}

/*------------------------------------------------------------------------------
| ICurrentThread::suspend                                                      |
|                                                                              |
| Issue DosSuspendThread.  If it fails, throw an exception.                    |
|                                                                              |
| For MOTIF:                                                                   |
| Suspend thread unless GUI was initialized by this thread. In that case       |
| throw an exception.                                                          |
------------------------------------------------------------------------------*/
void ICurrentThread::suspend ( )
{
  if ( this->isGUIInitialized() )
  {
    ITHROWLIBRARYERROR( IC_SUSPEND_PM_THREAD,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }
  else {
      INonGUIThread::current().suspend();
  }
}

/*------------------------------------------------------------------------------
| IThread::autoInitGUI                                                         |
|                                                                              |
| Return the autoInitPM member of the referenced IStartedThread.               |
------------------------------------------------------------------------------*/
bool IThread::autoInitGUI ( ) const
{
  return IThreadData::autoInitGUI( *this );
}

/*------------------------------------------------------------------------------
| IThread::setAutoInitGUI                                                      |
|                                                                              |
| Set the autoInitPM member of the referenced IStartedThread.                  |
------------------------------------------------------------------------------*/
IThread &IThread::setAutoInitGUI ( bool newFlag )
{
  IThreadData::setAutoInitGUI( *this, newFlag );
  return *this;
}

/*------------------------------------------------------------------------------
| IThread::stopProcessingMsgs                                                  |
|                                                                              |
| If the thread is in a message polling loop, post a WM_QUIT, else, throw an   |
| invalid request exception.                                                   |
------------------------------------------------------------------------------*/
IThread &IThread::stopProcessingMsgs ( )
{
  IMODTRACE_DEVELOP( "ICurrentThread::stopProcessingMsgs" );

  if ( this->isProcessingMsgs() )
  {
#if 0  //  Causes sporatic traps in Extra - HT
    // Allow some amount of clean up on the windows.  These windows
    // otherwise would not do WM_DESTROY processing, since WM_QUIT
    // would shut down any further messages.
    IWindowList* winList = this->windowList();
    if ( winList )
    {
       IWindowList::Cursor cursor( *winList );
       for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
       {         // Send a message to all windows.
          IWindow* window = winList->elementAt( cursor );
          window->sendEvent( IC_UM_CLOSE );
       }
    }
#endif

    this->postEvent( WM_QUIT );

  }
  else
  {
    ITHROWLIBRARYERROR( IC_THREAD_NOT_PROCESSING_MSGS,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  }

  return *this;
}

/*------------------------------------------------------------------------------
| IThread::isProcessingMsgs                                                    |
|                                                                              |
| Returns true if the thread is processing messages.                           |
------------------------------------------------------------------------------*/
bool IThread::isProcessingMsgs ( ) const
{
  // If there is a non-0 pointer for CurrentThreadData, it means an
  // event loop is active.
  return IThreadData::currentThreadData(*this) ? true : false;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
   Return the main application shell for the current thread.
   This method is Motif-specific.
------------------------------------------------------------------------------*/
IWindowHandle IThread::applicationShell ( ) const
{
  // Get the stored value.
  return IThreadData::applicationShell( *this );
}
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| IThread::queueSize                                                           |
|                                                                              |
| Returns the queue size from the associated started thread                    |
------------------------------------------------------------------------------*/
long IThread::queueSize ( ) const
{
  ITRACE_WIN_NOP();
  ITRACE_MOTIF_NOP();
  return IThreadData::queueSize(*this);
}

/*------------------------------------------------------------------------------
| IThread::setQueueSize                                                        |
|                                                                              |
| Set the queue size of the associated started thread                          |
------------------------------------------------------------------------------*/
IThread &IThread::setQueueSize ( long newQSize )
{
  ITRACE_WIN_NOP();
  ITRACE_MOTIF_NOP();
  IThreadData::setQueueSize( *this, newQSize );
  return *this;
}


/*------------------------------------------------------------------------------
| IThread::messageQueue                                                        |
|                                                                              |
| Return the hmq. First try the stored hmq (if there is one), else return 0 .  |
| This non-const member function will adjust the stored HMQ if it is           |
| found to be invalid.                                                         |
------------------------------------------------------------------------------*/
IMessageQueueHandle IThread::messageQueue ( )
{
  // Get the stored value.
  IMessageQueueHandle hmq( IThreadData::messageQueue( *this ) );
#ifdef IC_PM
  if ( hmq )
  {
    MQINFO info;
    if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
    {
      // No longer valid.  Reset the stored value.
      IThreadData::setMessageQueue(*this, 0);
      hmq    = 0;
    }
  }
#endif
  return hmq;
}

/*------------------------------------------------------------------------------
| IThread::messageQueue                                                        |
|                                                                              |
| Return the hmq. First try the stored hmq (if there is one), else return 0 .  |
------------------------------------------------------------------------------*/
IMessageQueueHandle IThread::messageQueue ( ) const
{
  // Get the stored value
  IMessageQueueHandle hmq( IThreadData::messageQueue( *this ) );

#ifdef IC_PM
  if ( hmq )
  {
    MQINFO info;
    if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
    {
      // No longer valid.
      hmq    = 0;
    }
  }
#endif
  return hmq;
}

/*------------------------------------------------------------------------------
| IThread::postEvent                                                           |
------------------------------------------------------------------------------*/
void IThread::postEvent ( unsigned long eventId,
                          const IEventParameter1 &parm1,
                          const IEventParameter2 &parm2 ) const
{
  IThread::postEvent( *this, eventId, parm1, parm2 );
}

/*------------------------------------------------------------------------------
| IThread::postEvent                                                           |
|  post (asynchronously)  a window system message/event to the                 |
|  specified thread using the specified parameters.                            |
------------------------------------------------------------------------------*/
void IThread::postEvent( const IThread& thread, unsigned long eventId,
                         const IEventParameter1 &parm1,
                         const IEventParameter2 &parm2 )
{
#ifdef IC_PMWIN
   if ( !IPOSTQUEUEMESSAGE( thread.messageQueue().asUnsigned(),
                            eventId, parm1, parm2 ) )
      ITHROWGUIERROR2( "WinPostQueueMsg",
                       IBaseErrorInfo::invalidParameter,
                       IException::recoverable );
#endif
#ifdef IC_MOTIF
  // The Motif implementation posts the message to the application shell
  // widget for the specified thread.
  IWindowHandle shell( thread.applicationShell() );
  if ( shell )
     shell.postEvent( eventId, parm1, parm2 );
#endif
}

/*------------------------------------------------------------------------------
| IThread::postEvents                                                          |
|  postEvent - post (asynchronously)  a window system message/event to         |
|  all threads that have a message queue, using the specified parameters.      |
------------------------------------------------------------------------------*/
void IThread::postEvents ( unsigned long eventId,
                           const IEventParameter1 &parm1,
                           const IEventParameter2 &parm2 )
{
#ifdef IC_PM
   if ( !WinBroadcastMsg( 0, eventId, parm1, parm2, BMSG_POSTQUEUE ) )
      ITHROWGUIERROR2( "WinBroadcastMsg",
                       IBaseErrorInfo::invalidParameter,
                       IException::recoverable );
#endif
#ifdef IC_WIN
   if ( !IPOSTMESSAGE( HWND_BROADCAST, eventId, parm1, parm2 ) )
      ITHROWGUIERROR2( "PostMessage",
                       IBaseErrorInfo::invalidParameter,
                       IException::recoverable );
#endif
  ITRACE_MOTIF_NOP();
#ifdef IC_MOTIF
  // Currently, Motif / Xt only support one thread.  We implement this function
  // by posting the event to the current thread.
  IThread::postEvent( IThread::current(), eventId, parm1, parm2 );
#endif
}


/*------------------------------------------------------------------------------
| IThreadData::variableValue                                                   |
------------------------------------------------------------------------------*/
void* IThreadData::variableValue ( const INonGUIThread& thread,
                                   const char* key )
{
  IString fullKey(IThreadData::fgKeyBase);
  fullKey += key;
  IString value = thread.variable(fullKey);
  if ( value.length() == sizeof( void* ) )
     return *( (void**)(char*) value );
  else
     return 0;
}

/*------------------------------------------------------------------------------
| IThreadData::setVariableValue                                                |
------------------------------------------------------------------------------*/
void IThreadData::setVariableValue ( INonGUIThread& thread,
                                     const char* key,
                                     void* value )
{
  IString fullKey( IThreadData::fgKeyBase );
  fullKey += key;
  void* tvalue = value;
  IString stringValue( (void*)(&tvalue), sizeof( void* ) );
  thread.setVariable( fullKey, stringValue );
}


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IThreadData::getArgs                                                         |
------------------------------------------------------------------------------*/
IString IThreadData::getArgs ( )
{
  // The option -F for ps command is not supported by SUN
  // use 'ioctl' command to get the process name          -- SY
#ifdef IC_SUN
  const int BUFSIZE = 1024;
  char buf[ BUFSIZE ];
  int fd;
  prpsinfo_t psbuff;

  sprintf( buf, "/proc/%d", getpid() );
  if ( ( fd = open( buf, O_RDONLY ) ) == -1 )
    return IString( "" );

  if ( ioctl( fd, PIOCPSINFO, &psbuff ) == -1 )
  {
     close( fd );
     return IString( "" );
  }

  close( fd );

  return IString( psbuff.pr_fname );
#else // ! IC_SUN
  // The following is the AIX version implementation  -- SY
  // Use the getprocs subroutine to obtain info about the current
  // process.  The procsinfo.pi_comm field contains the name of
  // the command which started the process.

  pid_t curPid = getpid();
  procsinfo processbuffer;
  IString result;
  // Ask for the process info about the current process.  Don't care
  // about file handle info.
  if ( 1 == getprocs( &processbuffer, sizeof( procsinfo ),
                      0, sizeof( fdsinfo ),
                      &curPid, 1 ) )
  {
     result = processbuffer.pi_comm;
  }
  return result;
#endif // ! IC_SUN
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IMouseEventDetect::eventId                                                   |
------------------------------------------------------------------------------*/
int IMouseEventDetect::eventId ( const XEvent& event )
{
  int messageEventId;
  bClSynthesizeEvent = false;
  if ( event.type == ButtonPress )
  {
    if ( sameButton( event.xbutton.button )  &&
         withinMultiClickTime( event )  &&
         !anyButtonEdge()  &&
         ( ( iClPreviousMouseState == WM_BUTTON1CLICK  ||
             iClPreviousMouseState == WM_BUTTON2CLICK  ||
             iClPreviousMouseState == WM_BUTTON3CLICK )  ||
           ( anyButtonLevel()  &&
             ( iClPreviousMouseDownState == WM_BUTTON1DOWN  ||
               iClPreviousMouseDownState == WM_BUTTON2DOWN  ||
               iClPreviousMouseDownState == WM_BUTTON3DOWN ) ) ) )
    {
      switch ( event.xbutton.button )
      {
        case Button1:
          messageEventId = WM_BUTTON1DBLCLK;
          break;
        case Button2:
          messageEventId = WM_BUTTON2DBLCLK;
          break;
        case Button3:
          messageEventId = WM_BUTTON3DBLCLK;
          break;
        default:
          // Use button 1 for now.
          messageEventId = WM_BUTTON1DBLCLK;
          break;
      }
    }
    else
    {
      switch ( event.xbutton.button )
      {
        case Button1:
          messageEventId = WM_BUTTON1DOWN;
          break;
        case Button2:
          messageEventId = WM_BUTTON2DOWN;
          break;
        case Button3:
          messageEventId = WM_BUTTON3DOWN;
          break;
        default:
          messageEventId = WM_BUTTON1DOWN;
          break;
      }
      if ( anyButtonEdge( ) )
      {
        resetEdge( );
        bClSynthesizeEvent = true;
        iClSynthesizeEventId = WM_CHORD;
      }
    }
    iClPreviousMouseDownState = messageEventId;
    setButton( event.xbutton.button );
  }
  else if ( event.type == ButtonRelease )
  {
    switch ( event.xbutton.button )
    {
      case Button1:
        messageEventId = WM_BUTTON1UP;
        break;
      case Button2:
        messageEventId = WM_BUTTON2UP;
        break;
      case Button3:
        messageEventId = WM_BUTTON3UP;
        break;
      default:
        messageEventId = WM_BUTTON1UP;
        break;
    }
    resetLevel( event.xbutton.button );

    Widget tempWinToW = NULL;
    tempWinToW =
       XtWindowToWidget( event.xany.display, event.xany.window );

    if ( tempWinToW && XtIsSubclass( tempWinToW, xmScrollBarWidgetClass )
           &&  ( event.xbutton.button == 1 ) )
    {
      bClSynthesizeEvent = true;
      iClSynthesizeEventId = SB_ENDSCROLL;
    }
    else if ( ( iClPreviousMouseState == WM_BUTTON1DOWN  ||
                iClPreviousMouseState == WM_BUTTON2DOWN  ||
                iClPreviousMouseState == WM_BUTTON3DOWN )  &&
              withinMultiClickTime( event )  &&
              !anyButtonLevel( ) )
    {
      bClSynthesizeEvent = true;
      switch ( event.xbutton.button )
      {
        case Button1:
          iClSynthesizeEventId = WM_BUTTON1CLICK;
          break;
        case Button2:
          iClSynthesizeEventId = WM_BUTTON2CLICK;
          break;
        case Button3:
          iClSynthesizeEventId = WM_BUTTON3CLICK;
          break;
        default:
          iClSynthesizeEventId = WM_BUTTON1CLICK;
          break;
      }
    }
  }
  ulClLastButtonTime = event.xbutton.time;
  iClPreviousButton = event.xbutton.button;
  iClPreviousMouseState = messageEventId;
  return messageEventId;
}
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| ICurrentThread::ICurrentThread                                               |
|                                                                              |
| Initialize referenced IStartedThread to zero.                                |
|                                                                              |
| Notes:                                                                       |
|   1. During the course of constructing this object, an IStartedThread        |
|      corresponding to thread 1 (the main application thread) is              |
|      created.  This IStartedThread is never used directly, however.          |
|   2. This constructor is protected to prevent creation of additional         |
|      objects of this class.  The only instance is the static                 |
|      currentThread object within this module (see above).                    |
------------------------------------------------------------------------------*/
ICurrentThread :: ICurrentThread ( )
#ifdef IC_WIN
  : IThread( GetCurrentThreadId() )
#endif
#ifdef IC_PM
  : IThread( IStartedThread::fgPrimaryTID )
#endif
#ifdef IC_MOTIF
  : IThread( currentId() )
#endif
{
  IMODTRACE_ALL( "ICurrentThread::ICurrentThread" );
}

/*------------------------------------------------------------------------------
| ICurrentThread::anchorBlock                                                  |
|                                                                              |
| Return an IAnchorBlockHandle constructed from the current thread's HAB.      |
------------------------------------------------------------------------------*/
IAnchorBlockHandle ICurrentThread::anchorBlock ( ) const
{
  ITRACE_WIN_NOP();
  IAnchorBlockHandle::Value
    storedHAB = IThreadData::anchorBlock( *this );

  if ( !storedHAB )
  {
#ifdef IC_PM
     if ( ( storedHAB = WinInitialize( 0 ) ) == 0 )
     {
        ITHROWGUIERROR2( "WinInitialize",
                         IBaseErrorInfo::accessError,
                         IException::unrecoverable );
     }
#endif
#ifdef IC_WIN
     // Unused value, but set to non-zero to avoid breaking existing code
     storedHAB = (IAnchorBlockHandle::Value)1;
#endif
#ifdef IC_MOTIF
     // Install our function as the Motif XmColorProc.
     if (IThread__defaultColorProc != IThread__xmColorProc)
     {
         ITRACE_DEVELOP("Installing IThread__xmColorProc");
         // Install our procedure as the Motif color calculation procedure.
         IThread__defaultColorProc =
            XmSetColorCalculation( IThread__xmColorProc );
     }

     // Set up the default locale for X
     XtSetLanguageProc( NULL, NULL, NULL );

     int argC = IApplication::current().argc();
     char **argV;
     bool needToFreeArgs = false;
     if ( argC )
     {
        argV = new char*[ argC ];
        for ( int i = 0; i < argC; i++ )
           argV[ i ] = IApplication::current().argv( i );
     }
     else
     {
        needToFreeArgs = true;
        IString args = IThreadData::getArgs();
        argC = args.numWords();
        argV = new char * [ argC ];
        for ( int index = 0; index < argC; index++ )
        {
           IString arg = args.word( index + 1 ); // words are 1 based
           argV[ index ] = new char[ arg.length() + 1 ];
           strcpy( argV[index], arg );
        }
     }
     // Set the default class name to the name of the program.
     // X will look for defaults info under this name.   If not
     // available use a default name.
     IString appName = DEFAULTAPPCLASS;
     if ( argC > 0 )
         appName = IString( argV[ 0 ] );

     unsigned indexOfPathDelimiter = appName.lastIndexOf( '/' );
     if ( indexOfPathDelimiter )
        appName = appName.subString( indexOfPathDelimiter + 1,
                                     appName.length() - indexOfPathDelimiter );
     // The X format for resource states that the first character should
     // be upper case. If the first character is X then the first two
     // characters should be upper case.
     if ( appName.subString( 1, 1 ).isLowerCase() )
        appName = appName.subString( 1, 1 ).upperCase() +
                  appName.subString( 2, appName.length() - 1 );
     if ( appName[1] == 'X' )
          appName = appName.subString( 1, 2 ).upperCase() +
                    appName.subString( 3, appName.length() - 2 );

     ITRACE_DEVELOP( "Application name is " + appName );
     // make a local copy of the argv list.  Xt may modify it
     Widget shell =
        XtAppInitialize( &storedHAB,   /* application context */
                         appName,      /* class name */
                         NULL,         /* command line options table */
                         0,            /* number of entries in options */
                         &argC,        /* argument count */
                         argV,         /* arguments */
                         NULL,         /* fallback resources */
                         NULL,         /* resource ArgList */
                         0 );          /* count of resource args */

     if ( shell == 0 )
     {
        ITHROWLIBRARYERROR( IC_APPLICATION_INITIALIZE,
                            IBaseErrorInfo::accessError,
                            IException::unrecoverable );
     }

     // Insure that startedThread is retained until terminateGUI so
     // that we cleanup properly.
     IStartedThread
       *p = this->startedThread();
     p->addRef();

     // Store the display obtained from XtAppInitialize in the
     // graphics library for use by graphics code.
     IXDisplay display( XtDisplay( shell ) );

     XtVaSetValues( shell,
                    XmNmappedWhenManaged, false,
                    NULL );

     // Set up XSetErrorHandler
     XSetErrorHandler( XLibErrorHandler );
     XtAppSetWarningHandler( storedHAB, ICLUIWarningHandler );
#endif //IC_MOTIF

     ICurrentThread* localThis = (ICurrentThread*) this;
     IThreadData::setAnchorBlock( *localThis, storedHAB );

#ifdef IC_MOTIF
     // Store the shell widget for later reference.
     IThreadData::setApplicationShell( *localThis, shell );
     if ( needToFreeArgs )
     {
       // Replace the argc / argv stuff in IApplication::current if the args
       // were never set.
       IApplication::current().setArgs( argC, argV );
        for ( int index=0; index < argC; index++ )
        {
           delete [] argV[ index ];
        }
     }
     delete [] argV;
#endif
  }  // not already initialized
  return IAnchorBlockHandle( storedHAB );
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThread::setAnchorBlock                                               |
|                                                                              |
------------------------------------------------------------------------------*/
void ICurrentThread::setAnchorBlock (  const IWindowHandle& appShell,
                                       const IAnchorBlockHandle& appContext )

{
  IAnchorBlockHandle::Value
    storedHAB = IThreadData::anchorBlock( *this );

  Widget fAppShell = appShell;
//  if ( !storedHAB )
//  {
     // save the anchor block.
//     XtAppContext fAppContext;
//     fAppContext = XtWidgetToApplicationContext( (Widget ) appShell );

//     storedHAB = fAppContext;
     storedHAB = (XtAppContext) appContext;
     // Insure that startedThread is retained until terminateGUI so
     // that we cleanup properly.
     //IStartedThread
     //  *p = this->startedThread();
     //p->addRef();


     // Store the display obtained from XtAppInitialize in the
     // graphics library for use by graphics code.
     IXDisplay display( XtDisplay( (Widget) fAppShell ) );

     //XtVaSetValues( fAppShell,
     //               XmNmappedWhenManaged, false,
     //               NULL );

     // Replace the argc / argv stuff in IApplication::current
     //IApplication::current().setArgs( argC, argV );

     // Set up XSetErrorHandler
     //XSetErrorHandler( XLibErrorHandler );
     //XtAppSetWarningHandler( storedHAB, ICLUIWarningHandler );

     ICurrentThread* localThis = (ICurrentThread*) this;
     IThreadData::setAnchorBlock( *localThis, storedHAB );
     // Store the shell widget for later reference.
     IThreadData::setApplicationShell( *localThis, fAppShell );
//  }  // not already initialized
}
#endif

/*------------------------------------------------------------------------------
| ICurrentThread::isGUIInitialized                                             |
|                                                                              |
| Return true if GUI has been initialized for this thread.                     |
------------------------------------------------------------------------------*/
bool ICurrentThread::isGUIInitialized ( ) const
{
  // Use the stored value in IThreadData instead of the anchorBlock() member
  // function because anchorBlock has the side effect of initializing
  // the GUI.
  if ( ( IThreadData::anchorBlock( *this ) != 0 )
       &&
       ( this->messageQueue() != 0 ) )
    return true;
  else
    return false;
}

/*------------------------------------------------------------------------------
| ICurrentThread::messageQueue                                                 |
|                                                                              |
| Return the hmq. First try the stored hmq ( IThread handles this)             |
| (if there is one), else try HMQ_CURRENT.                                     |
| Return 0 if no message queue is stored and HMQ_CURRENT is invalid.           |
------------------------------------------------------------------------------*/
IMessageQueueHandle ICurrentThread::messageQueue ( )
{
  IMessageQueueHandle hmq = this->Inherited::messageQueue();
  if ( !hmq )
  {
#ifdef IC_PM
     MQINFO info;
     hmq = HMQ_CURRENT;
     if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
        hmq = 0;
#endif
#ifdef IC_WIN
                           // message queue id == thread id
     hmq = (IMessageQueueHandle::Value)GetCurrentThreadId();
#endif
#ifdef IC_MOTIF
                           // message queue id == thread id
     hmq = (IMessageQueueHandle::Value)pthread_self();
#endif
  }
  return hmq;
}

/*------------------------------------------------------------------------------
| ICurrentThread::messageQueue                                                 |
|                                                                              |
| Return the hmq. First try the stored hmq ( IThread handles this)             |
| (if there is one), else try HMQ_CURRENT.                                     |
| Return 0 if no message queue is stored and HMQ_CURRENT is invalid.           |
------------------------------------------------------------------------------*/
IMessageQueueHandle ICurrentThread::messageQueue ( ) const
{
  IMessageQueueHandle hmq = this->Inherited::messageQueue();
  if ( !hmq )
  {
#ifdef IC_PM
     MQINFO info;
     hmq = HMQ_CURRENT;
     if ( !WinQueryQueueInfo( hmq, &info, sizeof info ) )
        hmq = 0;
#endif
#ifdef IC_WIN
                          // message queue id == thread id
     hmq = (IHandle::Value)GetCurrentThreadId();
#endif
#ifdef IC_MOTIF
                          // message queue id == thread id
     hmq = (IMessageQueueHandle::Value)pthread_self();
#endif
  }
  return hmq;
}


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThread::isTopLevelShell                                              |
|                                                                              |
| Return true if application window has been completed.                        |
------------------------------------------------------------------------------*/
bool ICurrentThread::isTopLevelShell ( ) const
{
  return IThreadData::isTopLevelShell( *this );
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThread::setTopLevelShell                                             |
|                                                                              |
| Set the application level indicator to true.                                 |
------------------------------------------------------------------------------*/
void  ICurrentThread::setTopLevelShell ( ) const
{
  IThreadData::setTopLevelShell( *( (ICurrentThread*)this ), true );
}
#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| ICurrentThread::processMsgs                                                  |
|                                                                              |
| Loop, retrieving and dispatching events, until a WM_QUIT is received.        |
------------------------------------------------------------------------------*/
void ICurrentThread::processMsgs ( )
{
  IMODTRACE_DEVELOP( "ICurrentThread::processMsgs" );

  //-------------------------------------------------------------------
  // Gui initialization
  //-------------------------------------------------------------------
  if ( ! this->isGUIInitialized() )
     this->initializeGUI();


  //-------------------------------------------------------------------
  // Preparation to enter the event loop.  A per thread ICurrentThreadData
  // object is allocated, and event loop invariant data set up.
  // The loopData object is a smart pointer to the ICurrentThreadData object
  // for this thread with a constructor and destructor.  The constructor
  // finds the instance for the current thread (allocating if needed)
  // and increments the fMsgLoop counter.  The destructor decrements
  // the counter and deletes the ICurrentThreadData object if this
  // counter is 0.  Note that the destructor is also needed in case an
  // exception occurs while the event is being processed.
  //-------------------------------------------------------------------
  ICurrentThreadDataRef loopData;


  IAnchorBlockHandle
    hab = this->anchorBlock();
#ifdef  IC_PMWIN
  QMSG
    msg;
#endif // IC_PMWIN
#ifdef  IC_MOTIF
  XEvent event;
  // the following event pointers need to be here because this method is
  // reentrant.
  IEvent *lastEvent=0, *lastSynthesizedEvent=0;

  // Montana needs a way to stop the button messages from always being dispatched,
  // even in the case where the mouse handler returns true. We agreed to try an
  // environment variable so this behavior could be controlled.
#endif // IC_MOTIF

  //-------------------------------------------------------------------
  // Main event loop.
  // The essential logic is
  //    - get next event
  //    - inspect it for special IOC or system required processing
  //    - dispatch it.
  // There are dependencies between this code and modal display code
  // in IFrameWindow::showModally, and in some case in IMessageBox
  // to correctly process dismissal of modal dialogs.
  //-------------------------------------------------------------------
#ifdef IC_MOTIF
  // post an event so that all pending messages can be flushed. This is to
  // support post messages before the widgets have windows.
  IThread::applicationShell().postEvent( IC_UM_FLUSH_POSTS );
  unsigned long lastKeyPress = 0;
#endif
  bool timeToQuit = false;
  while ( !timeToQuit )
  {
#ifdef IC_WIN
     if ( (IGETMSG( hab, &msg, 0, 0, 0 ) > 0) && (msg.message != IC_UM_QUIT) )
     {
       ITRACE_ALL(
          IString("Event loop  hwnd=") +
          IString((unsigned long)msg.hwnd).d2x()+
          IString(", Msg=") + IString(msg.message).d2x()+
          IString(", wParam=") + IString(msg.wParam).d2x()+
          IString(", lParam=") + IString(msg.lParam).d2x() );

       // Check for accelerators.   If the translateAcclerator function
       // returns true we have handled this event.
         if (!( ((msg.message >= WM_KEYFIRST) && (msg.message <= WM_KEYLAST)) &&
                ICurrentThreadData::translateAccelerator( msg ) ) )
         {
             // Not accelerator.  Call TranslateMessage to create WM_CHARs if
             // appropriate and then dispatch the event.
             TranslateMessage( &msg );
             IDISPATCHMSG( hab, &msg );
         }
     }
     else
     {
         ITRACE_DEVELOP(
            IString("WM_QUIT hwnd=") +
            IString((unsigned long)msg.hwnd).d2x()+
            IString(", Msg=") + IString(msg.message).d2x()+
            IString(", wParam=") + IString(msg.wParam).d2x()+
            IString(", lParam=") + IString(msg.lParam).d2x() );

         // Check for WM_QUIT (from Windows) or IC_UM_QUIT and a value
         // in lParam equal to the current inMsgLoop value.  If either is
         // true exit the loop.
         if ( (msg.message == WM_QUIT) ||
              ((msg.message == IC_UM_QUIT) &&
               (msg.lParam == loopData->fMsgLoop)))
         {
             timeToQuit = true;
         }
         else if (msg.lParam < loopData->fMsgLoop)
         {
             // We have received an IC_UM_QUIT which is not the top of the
             // nesting stack.  We have already hidden the modal dialog and
             // reenabled the owner, so at this point we just have to remember
             // the fact that we were asked to exit.  We stack up the requests
             // and check the stack after each exit from this routine.  See
             // IFrameWindow::showModally.
             if (!loopData->fquitStack)
                loopData->fquitStack = new IWMQuitStack(10);
             loopData->fquitStack->add( msg.lParam );
         }
     }
#endif //IC_WIN
#ifdef IC_PM
     if ( IGETMSG( hab, &msg, 0, 0, 0 ) )
     {
         IDISPATCHMSG( hab, &msg );
     }
     else
     {
         timeToQuit = true;
     } // else getmsg
#endif // IC_PM
#ifdef IC_MOTIF
     XtAppNextEvent( hab, &event );
     // Uncomment the following trace to see events
     //if ( (event.type == ButtonPress) || (event.type == ButtonRelease) )
     //{
     //    ITRACE_DEVELOP( IString("IThread::processMsgs ") +
     //                    IString(p->inMsgLoop) +
     //                    IString(" eventID=") + IString(event.type).d2x() +
     //                    IString(" window=") + IString(event.xany.window).d2x() );
     //}

     bool eventHandled = false;

     // call the ipfx event dispatcher.  Returns false if we should not
     // call XtDispatchEvent.  We currently extend this to apply to all
     // IOC event handling as well.
     if ( !XhDispatchEvent( hab, &event ) )
        eventHandled = true;
     else
     {
        // Examine the event for special processing.
        switch (event.type)
        {
           case ButtonPress:
           case ButtonRelease:
               // Button events are always dispatched to XtDispatch
               // Don't assign eventHandled.
               eventHandled = loopData->handleMouseButtonEvent( event, lastEvent,
                                                  lastSynthesizedEvent );
               if (::fAlwaysDispatchMouseEvents)
               {
                 eventHandled = false;
               }
               break;
           case KeyPress:
               // Check for accelerators (ICurrentThreadData::translateAcclerator
               // returns true if it processed the key press as an accelerator
               // key).
               eventHandled = ICurrentThreadData::translateAccelerator( event );
               if ( eventHandled )
               {
                  break;
               }
               // Check for duplicate KeyPress events in certain situations
               // See defect 27786
               if ( (lastKeyPress == event.xkey.serial) && 
                    (event.xkey.type == 2) ) {
                  break;
               }
               lastKeyPress = event.xkey.serial;
               // Else fall into the key release case below.
           case KeyRelease:
               eventHandled = loopData->handleKeyboardEvent( event, lastEvent);
               break;
           case MappingNotify:
               if ( event.xmapping.request == MappingModifier )
                   loopData->getModifierMappings();
               break;
           case ClientMessage:
               if ( event.xclient.data.l[0] == WM_QUIT )
               {
                  eventHandled = true;
                  if ( !event.xclient.data.l[2]  ||
                       ( event.xclient.data.l[2] == loopData->fMsgLoop ))
                  {
                     timeToQuit = true;
                  }
                  else if (event.xclient.data.l[2] < loopData->fMsgLoop)
                  {
                     // We have received an WM_QUIT which is not the top of the
                     // nesting stack.  We have already hidden the modal dialog and
                     // reenabled the owner, so at this point we just have to remember
                     // the fact that we were asked to exit.  We stack up the requests
                     // and check the stack after each exit from this routine.  See
                     // IFrameWindow::showModally.
                     if ( !loopData->fquitStack )
                        loopData->fquitStack = new IWMQuitStack( 10 );
                     loopData->fquitStack->add( event.xclient.data.l[ 2 ] );
                  }
               }     // WM_QUIT
               break;
        }   //switch
     }  // else

     if ( !eventHandled )
     {
        // Dispatch the event to X
        XtDispatchEvent( &event);
        //if ( (event.type == ButtonPress) || (event.type == ButtonRelease) )
        //{
        //   ITRACE_DEVELOP( IString( "IThread::processMsgs - event dispatched" ) );
        //}

        // Dispatch certain events up the owner chain.
        switch ( event.type )
        {
            case ButtonPress:
            case ButtonRelease:
                loopData->sendMouseEventsUpOwnerChain( lastEvent,
                                                        lastSynthesizedEvent );
                break;
            case KeyPress:
            case KeyRelease:
                loopData->sendKeyEventsUpOwnerChain( lastEvent );
                break;
        }
     }  // if event not handled

     // Clean up mouse and keyboard event stuff.
     if ( lastEvent )
     {
        delete lastEvent;
        lastEvent = 0;
     }
     if ( lastSynthesizedEvent )
     {
        delete lastSynthesizedEvent;
        lastSynthesizedEvent = 0;
     }
#endif //IC_MOTIF

  } // end while


  //-------------------------------------------------------------------
  // Exit from the event loop.
  // -  Handle any cleanup needed for modal dialogs.
  // -  clean up loop invariant data.
  // At this point the loopData object is destructed.  This decrements
  // the fMsgLoop counter, handles any pending modal loop exits,
  // and deletes the ICurrentThreadData object if it is not needed anymore.
  //-------------------------------------------------------------------

}


/*------------------------------------------------------------------------------
| ICurrentThread::initializeGUI                                                |
|                                                                              |
| Initialize the thread for PM by calling WinInitialize and WinCreateMsgQueue. |
| If an error occurs, throw an exception.                                      |
|  Motif - Perform an XtAppInitialize() to extablish the application           |
|  context and main shell.                                                     |
|                                                                              |
| Notes:                                                                       |
|   1. Increment the use count for the referenced IStartedThread (so the       |
|      thread-related hab/hmq are retained till terminateGUI).                 |
------------------------------------------------------------------------------*/
void ICurrentThread::initializeGUI ( long queueSize )
{
  IMODTRACE_DEVELOP( "ICurrentThread::initializeGUI" );

  IAnchorBlockHandle hab = this->anchorBlock();
  // Moved WinInitialize/XtAppInitialize logic to anchorBlock() function
  IMessageQueueHandle hmq = this->messageQueue();

#ifdef IC_PM
  if ( !hmq )
  {
     if ( !queueSize )
        queueSize = this->queueSize();

     if ( ( hmq = WinCreateMsgQueue( hab, queueSize ) ) != 0 )
     {
        // store the value
        IThreadData::setMessageQueue( *this, hmq );

        IStartedThread
          *p = this->startedThread();
        p->addRef();
     }
     else
       // WinCreateMsqQueue failed, throw exception...
     {
       ITHROWGUIERROR2( "WinCreateMsgQueue",
                        IBaseErrorInfo::accessError,
                        IException::unrecoverable );
     }
  }
#endif //IC_PM

#ifdef IC_MOTIFWIN
   if ( Inherited::messageQueue() != hmq )
   {
      // Insure the hmq is stored so IThread can access it.
      IThreadData::setMessageQueue( *this, hmq );

      // Windows: do not need to worry about addRef,
      // since we don't need to cleanup.  Queue size cannot be set.
      // Motif: anchorBlock does the addRef because it is done in
      // XtAppInitialize.
   }

#ifdef IC_WIN
   // Call a function in USER.LIB to insure GUI initialized.  This is
   // automatic but certain functions can fail without it (like
   // PostThreadMessage).
   GetDesktopWindow();
#endif //IC_WIN

#endif //IC_MOTIFWIN
}

/*------------------------------------------------------------------------------
| ICurrentThread::terminateGUI                                                 |
|                                                                              |
| Issue WinDestroyMsgQueue and WinTerminte, as necessary.                      |
| Motif: destroy the application context                                       |
------------------------------------------------------------------------------*/
void ICurrentThread::terminateGUI ( )
{
  IMODTRACE_DEVELOP( "ICurrentThread::terminateGUI" );

#ifdef IC_PM
  if ( this->messageQueue() )
  {
     WinDestroyMsgQueue( this->messageQueue() );
     if ( IThreadData::messageQueue( *this ) )
     {             // Created and stored in initializeGUI.
        IThreadData::setMessageQueue( *this, 0 );
        IStartedThread
         *p = this->startedThread();
        p->removeRef();
     }
  }

  // Use the IThreadData stored value to test, because anchorBlock will
  // call WinInitialize.
  if ( IThreadData::anchorBlock( *this ) )
     WinTerminate( this->anchorBlock(  ) );
#endif //IC_PM

#ifdef IC_WIN
  // Message queue deleted automatically in Windows so no-op
#endif //IC_WIN

#ifdef IC_MOTIF
  // Use the IThreadData stored value to test, because anchorBlock will
  // reinitialize.
  IAnchorBlockHandle::Value context = IThreadData::anchorBlock( *this );
  if ( context != 0 )
  {
     // Invalidate the display stored in the graphics library by
     // initializeGUI.
     IXDisplay::closeDisplay( );

     // Invalidate the appcontext and shell we have stored.
     IThreadData::setAnchorBlock( *this, 0 );
     IThreadData::setApplicationShell( *this, 0 );

     // Destroy the application context and close the display.
     XtDestroyApplicationContext( context );

     IStartedThread *p = startedThread();
     p->removeRef();
  }
#endif //IC_MOTIF
}

/*------------------------------------------------------------------------------
| ICurrentThread::startedThread                                                |
|                                                                              |
| Returns the result of looking up the current TID.                            |
------------------------------------------------------------------------------*/
IStartedThread* ICurrentThread::startedThread ( ) const
{
  return( INonGUIThread::current().startedThread() );
}

/*------------------------------------------------------------------------------
| IThread::inGUISession                                                        |
|                                                                              |
| Query the current session type and return whether it is PM.                  |
------------------------------------------------------------------------------*/
bool IThread::inGUISession ( )
{
  IMODTRACE_DEVELOP( "IThread::inGUISession" );

#ifdef IC_PM
  PTIB
    ptib;
  PPIB
    ppib;
  DosGetInfoBlocks( &ptib, &ppib );
  if ( ppib->pib_ultype != SSF_TYPE_PM )
     return( false );
#endif

  // Always true for any flavor of Windows or Motif.
  return true;
}

/*------------------------------------------------------------------------------
| IThread::defaultAutoInitGUI                                                  |
|                                                                              |
| Return static default value.                                                 |
------------------------------------------------------------------------------*/
bool IThread::defaultAutoInitGUI ( )
{
  if ( IThread::dfltAutoInitGUI == -1 )
     IThread::dfltAutoInitGUI = IThread::inGUISession();
  return IThread::dfltAutoInitGUI;
}

/*------------------------------------------------------------------------------
| IThread::current                                                             |
|                                                                              |
| Returns the current thread                                                   |
------------------------------------------------------------------------------*/
ICurrentThread& IThread::current ( )
{
  IMODTRACE_ALL( "ICurrentThread::current()" );

  if ( !IThread::pCurrent )
  {
     IResourceLock
       lock( IStartedThread::lock() );
     // check again with resources locked to insure we don't get 2 instances
     if ( !IThread::pCurrent )
        IThread::pCurrent = new ICurrentThread;
  }

  return *IThread::pCurrent;
}

/*------------------------------------------------------------------------------
| ICurrentThreadDataRef::ICurrentThreadDataRef                                 |
------------------------------------------------------------------------------*/
ICurrentThreadDataRef::ICurrentThreadDataRef( )
  : fLoopData(0)
{
  IMODTRACE_DEVELOP("ICurrentThreadDataRef::ICurrentThreadDataRef");
  // Try to get the pointer to the ICurrentThreadData for the event loop
  // on this thread.  If there is not one currently allocated, it means
  // that this thread is entering the event loop at nesting level 1.  If
  // one is already allocated, it means that we have nested event loops.
  ICurrentThread& thread( IThread::current() );
  fLoopData = IThreadData::currentThreadData( thread );
  if ( !fLoopData )
  {
     // Allocate the ICurrentThreadData object for this thread.
     fLoopData = new ICurrentThreadData;
     IThreadData::setCurrentThreadData( thread, fLoopData );
#ifdef IC_MOTIF
     // Get double click time for this app. This is used to generate a double
     // click event.
     fLoopData->getMultiClickTime(
         XtDisplay( (Widget)thread.applicationShell() ) );
     fLoopData->getModifierMappings();
     //  p->msgLoopSerial++;
#endif //IC_MOTIF
  }
  // Increment the counter indicating number of active processMsgs loops
  fLoopData->fMsgLoop++;
  ITRACE_DEVELOP( "Starting message loop - Message loop nesting = " +
                  IString( fLoopData->fMsgLoop ) );
}

/*------------------------------------------------------------------------------
| ICurrentThreadDataRef::~ICurrentThreadDataRef                                |
------------------------------------------------------------------------------*/
ICurrentThreadDataRef::~ICurrentThreadDataRef( )
{
  IMODTRACE_DEVELOP("ICurrentThreadDataRef::~ICurrentThreadDataRef");
  ITRACE_DEVELOP( "Ending message loop - Message loop nesting = " +
                  IString( fLoopData->fMsgLoop ) );
  fLoopData->fMsgLoop--;          //signal end of the loop
#ifdef IC_MOTIFWIN
  // If we have a stack of unsatisfied requests to exit, inspect it
  // to see if we can satisfy a request now.
  if ( ( fLoopData->fquitStack )  &&
       ( fLoopData->fquitStack->lastElement() >= fLoopData->fMsgLoop ) )
  {
     // Ready to exit.  Remove the request and post an IC_UM_QUIT message.
     // If the stack is now empty delete it.
     fLoopData->fquitStack->removeLast();
#ifdef IC_WIN
     IThread::current().postEvent( IC_UM_QUIT, 0, fLoopData->fMsgLoop );
#endif //IC_WIN
#ifdef IC_MOTIF
     IThread::current().postEvent( WM_QUIT, 0, fLoopData->fMsgLoop );
#endif //IC_MOTIF
     if (fLoopData->fquitStack->isEmpty() )
     {
        delete fLoopData->fquitStack;
        fLoopData->fquitStack = 0;
     }
  }
#endif

  if ( fLoopData->fMsgLoop == 0 )
  {
     IThreadData::setCurrentThreadData( IThread::current(), 0 );
     delete fLoopData;
  }
}

#ifdef IC_MOTIFWIN
/*------------------------------------------------------------------------------
| ICurrentThreadData::translateAccelerator                                     |
|                                                                              |
| Emulate the traversal of WM_TRANSLATEACCEL messages up the parent chain      |
| in PM, by chasing up the parent chain for an accelerator table.              |
------------------------------------------------------------------------------*/
#ifdef IC_WIN
bool ICurrentThreadData::translateAccelerator ( MSG& message )
#endif
#ifdef IC_MOTIF
bool ICurrentThreadData::translateAccelerator ( const XEvent& event )
#endif
{
  bool
    translated = false;
  IWindowHandle
    desktopHandle = IWindow::desktopWindow()->handle(),
    objectHandle = IWindow::objectWindow()->handle();

#ifdef IC_WIN
  for ( IWindowHandle hwnd = message.hwnd;
#endif
#ifdef IC_MOTIF
  for ( IWindowHandle hwnd = XtWindowToWidget( event.xkey.display,
                                               event.xkey.window );
#endif
        translated == false  &&  hwnd  &&
          hwnd != desktopHandle  &&  hwnd != objectHandle;
        hwnd = IPARENTOF( hwnd ) )
  {  // Traverse up the parent chain to see if any window wants to
     // translate the message.
     IWindow* window = IWindow::windowWithHandle( hwnd );
     if ( window )
     {  // See if window has an accelerator table.
        // IC_NOTYET  Should really skip notebook page windows
        //            for compatibility with PM.
        IAccelTblHandle accelHandle = window->acceleratorHandle();
        if ( accelHandle )
        {
#ifdef IC_WIN
           translated = TranslateAccelerator( hwnd, accelHandle, &message );
#endif
#ifdef IC_MOTIF
           translated =
             ICurrentThreadData::translateAccelerator( hwnd, accelHandle, event );
#endif
        }
     }
  }
  return translated;
}
#endif

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThreadData::translateAccelerator                                     |
|                                                                              |
| Function emulating the Windows accelerator table API, TranslateAccelerator.  |
------------------------------------------------------------------------------*/
bool
  ICurrentThreadData:: translateAccelerator( const IWindowHandle& hwnd,
                                             const IAccelTblHandle& haccel,
                                             const XEvent& event )
{
  bool
    accelFound = false;
  ACCELTABLE
   *accelTable = (ACCELTABLE*)(IAccelTblHandle::Value) haccel;
  unsigned long
    keyCount = accelTable->accelNumber;
  KeyCode
    keyCode = event.xkey.keycode;
  Modifiers
    modifiers = event.xkey.state,
    comparedModifiers = 0;
  unsigned int
    modifierKeys = modifiers &
                     ( ShiftMask | ControlMask | Mod1Mask |
                       Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask );
  KeySym
    keysym = NoSymbol;
  XtTranslateKeycode( XtDisplay( (Widget) hwnd ), keyCode, modifiers,
                      &comparedModifiers, &keysym );
  bool
    done = false;
  ACCEL
   *accelKey = accelTable->accelerators;
  while ( ! done )
  {
     if ( accelKey->key == keysym  &&
          accelKey->modifiers == modifierKeys )
     {  // Entry found for the key in this accelerator table.
        // Generate a command event for the window with this
        // accelerator table.

        // Find parent frame window who ownes this accelerator table
        // to ensure the accelerated command is not disabled in the
        // frame's menu
        IWindow* parentFrame = IWindow::windowWithHandle( hwnd );
        while ( parentFrame && parentFrame->isFrameWindow() == false )
        {
          // Go up parent chain until a frame is found
          parentFrame = parentFrame->parent();
        }
        Widget frameMBar = 0;
        if ( parentFrame && parentFrame->isFrameWindow() )
        {
          // Obtain owners "MainWindow" child as the Menubar is created
          // on the MainWindow and not the BullitenBoard
          IWindowHandle mainWindow =
            IXmMainWindow::getMainWindowChild( parentFrame->handle() );
          IASSERTPARM( mainWindow != 0 );
          XtVaGetValues( mainWindow, XmNmenuBar, &frameMBar, NULL );
        }
        // Avoid exception if MainWindow has no Menubar
        bool generateCmd = true;
        if ( frameMBar )
        {
          // Retrieve the IWindow associated with the handle
          MenuWindow* menuWindow( IMenuPrivate::menuWindowForHandle(
            frameMBar ));
          if (menuWindow)
          {
            IMenu* mBar( menuWindow->menu() );
            if ( mBar )
            {
              try
              {
                if ( mBar->menuItem(accelKey->commandId).isDisabled() )
                  generateCmd = false;
              }
              catch (...)
              {
                // accelKey->commandId not found in menu so continue
              }
            }
          }
        }
        if ( generateCmd )
        {
          hwnd
           .sendEvent( WM_COMMAND,
               IEventParameter1( accelKey->commandId ),
               IEventParameter2( ICommandEvent::accelerator, 0 ) );
          accelFound = true;
        }
        done = true;
     }
     else
     {
        accelKey++;
        keyCount--;
        if ( keyCount == 0 )
        {  // No match found in this accelerator table.
           done = true;
        }
     }
  }
  return accelFound;
}
#endif

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThreadData::handleMouseButtonEvent                                   |
|                                                                              |
| Create the appropriate Mouse button event and dispatch it to the registered  |
| handlers.                                                                    |
|                                                                              |
| Notes:                                                                       |
|   1) This method assumes that the event.type is buttonPress or ButtonRelease |
------------------------------------------------------------------------------*/
bool ICurrentThreadData::handleMouseButtonEvent ( const XEvent &event,
                                                  IEvent* &lastEvent,
                                                  IEvent* &lastSynthesizedEvent )
{
  bool eventHandled( false );

  // Save a copy of the event in ithreadLastButtonEvent.   This is
  // used in the IPopUpMenu::show function to get a valid XButtonPressedEvent
  // structure.
  ithreadLastButtonEvent = event.xbutton;

  IWindowHandle widget = XtWindowToWidget( event.xany.display, event.xany.window );
  IWindow *win = IWindow::windowWithHandle( widget );

  ITRACE_ALL    ( IString("ButtonEvent ") +
                  IString(" type=") + IString(event.type) +
                  IString(" serial=")     + IString(event.xbutton.serial) +
                  IString(" send_event=") + IString(event.xbutton.send_event  ) +
                  IString(" display="   ) + IString((unsigned long)event.xbutton.display).d2x() +
                  IString(" window="    ) + IString(event.xbutton.window      ).d2x() +
                  IString(" root="      ) + IString(event.xbutton.root        ).d2x() +
                  IString(" subwindow=" ) + IString(event.xbutton.subwindow   ).d2x() +
                  IString(" time="      ) + IString(event.xbutton.time        ) );
  ITRACE_ALL    ( IString("ButtonEvent ") +
                  IString(" x="         ) + IString(event.xbutton.x           ) +
                  IString(" y="         ) + IString(event.xbutton.y           ) +
                  IString(" x_root="    ) + IString(event.xbutton.x_root      ) +
                  IString(" y_root="    ) + IString(event.xbutton.y_root      ) +
                  IString(" state="     ) + IString(event.xbutton.state       ) +
                  IString(" button="    ) + IString(event.xbutton.button      ) +
                  IString(" same_screen=")+ IString(event.xbutton.same_screen ) +
                  IString(" widget=") + IString(widget.asUnsigned()).d2x() );


  // Try to find an IWindow to dispatch the event to. Walk up the parent chain
  // until an IWindow is found. We need this check because of our aggregate
  // controls
  while ( !win  &&  widget )
  {
     widget = XtParent( (Widget) widget );
     win = IWindow::windowWithHandle( widget );
  }

  if ( win )
  {
     int eventId = medClMouseEvent.eventId( event ) ;
     IEvent evt( win->handle(), eventId,
                 IEventParameter1( 0 ), IEventParameter2( (void*) &event ) );
     lastEvent = new IEvent( evt );

     // go up the owner chain before we get a chance to dispatch this event
     // to the widget
     evt.setPassToOwner( false );

     eventHandled = win->dispatch( evt );
     // get the synthesized event ready to dispatch during
     // sendMouseEventsUpOwner chain. The synthesized events are dispatched
     // after the event goes up the owner chain.
     if ( medClMouseEvent.synthesizeEvent( ) )
     {
        int synthesizeEventId = medClMouseEvent.synthesizeEventId( );
        IEvent evt( win->handle(), synthesizeEventId,
                    IEventParameter1( 0 ), IEventParameter2( (void*) &event) );
        lastSynthesizedEvent = new IEvent( evt );
     }
  }
  return eventHandled;
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThreadData::handleKeyboardEvent                                      |
|                                                                              |
| Create the appropriate keyboard event and dispatch it to the registered      |
| handlers.                                                                    |
|                                                                              |
| Notes:                                                                       |
|   1) This method assumes that the event.type is keyPress or keyRelease.      |
------------------------------------------------------------------------------*/
bool ICurrentThreadData::handleKeyboardEvent ( const XEvent &event,
                                               IEvent* &lastEvent )
{
  bool eventHandled( false );

  // First try to use the widget that has focus.
  Widget widget =
     XmGetFocusWidget( XtWindowToWidget( event.xany.display,
                                         event.xany.window ) );

  // if nobody has keyboard focus then try the window that the event was
  // dispatched to.
  if ( widget == 0 )
     widget = XtWindowToWidget( event.xany.display, event.xany.window );

  IWindow *win;
  // Get the IWindow* for the widget.
  win = IWindow::windowWithHandle( widget );

  // Try to find an IWindow to dispatch the event to. Walk up the parent chain
  // until an IWindow is found.
  while ( !win  &&  widget )
  {
     widget = XtParent( (Widget) widget );
     win = IWindow::windowWithHandle( widget );
  }

  // if we found an IWindow* then dispatch the event.
  if ( win )
  {
     IEvent evt( win->handle(), event.type + xEventOffset,
                 IEventParameter1( 0 ), IEventParameter2( (void*) &event ) );
     lastEvent = new IEvent( evt );
     // set the pass to owner flag in IEvent to false so this event does not
     // go up the owner chain before we get a chance to dispatch this event
     // to the widget
     evt.setPassToOwner( false );
     eventHandled = win->dispatch( evt );
  }
  return eventHandled;
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThreadData::getModifierMappings                                      |
|                                                                              |
| When a MappingNotify event is detected, find which modifier state mask       |
| should be used to find the Alt key.                                          |
|                                                                              |
| Notes:                                                                       |
|   1) Throw an exception if the left and right alt keys map to a different    |
|      modifier mask.                                                          |
|   2) If super are hyper keys are to be detected add getting the mask here.   |
------------------------------------------------------------------------------*/
ICurrentThreadData& ICurrentThreadData::getModifierMappings ( )
{
  IMODTRACE_ALL( "IThreadData::getModifierMappings" );

  XModifierKeymap *modifiers;
  KeyCode *keycodes;
  Display *display = XtDisplay( (Widget) IThread::current().applicationShell() );

  modifiers = XGetModifierMapping( display );
  keycodes = modifiers->modifiermap;

  int stateMask = 1;
  unsigned long altMask = 0;
  for ( int i = 0; i < 8 * modifiers->max_keypermod; i++ )
  {
     if ( ( i % modifiers->max_keypermod == 0 )  &&  ( i != 0 ) )
        stateMask = stateMask << 1;
     if ( keycodes[ i ] != 0 )
     {
        KeySym ks;
        ks = XKeycodeToKeysym( display, keycodes[i], 0 );
        if ( ( ks == XK_Alt_L )  ||  ( ks == XK_Alt_R ) )
        {
           if ( ( altMask != 0 )  &&  ( altMask != stateMask ) )
              ITHROWLIBRARYERROR( IC_MODIFIER_MISMATCH,
                                  IBaseErrorInfo::invalidRequest,
                                  IException::recoverable );
           altMask = stateMask;
           ITRACE_ALL( "Alt Mask is " + IString( stateMask ) );
           IKeyboardEvent::ulAltMask = altMask;
        }
//      if ( ( ks == XK_Super_L )  ||  ( ks == XK_Super_R ) )
//      {
//         ulClSuperMask = stateMask;
//         ITRACE_DEVELOP( "Super Mask is " + IString( stateMask ) );
//      }
//      if ( ( ks == XK_Hyper_L )  ||  ( ks == XK_Hyper_R ) )
//      {
//         ulClHyperMask = stateMask;
//         ITRACE_DEVELOP( "Hyper Mask is " + IString( stateMask ) );
//      }
     }
  }
  return *this;
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThreadData::sendMouseEventsUpOwnerChain                              |
|                                                                              |
| If the event created for the button press or button release in               |
| handleMouseButtonEvents is not handled by the handlers for the IWindow then  |
| this method sends the event to the owner to start its way up the owner chain |
------------------------------------------------------------------------------*/
ICurrentThreadData&
  ICurrentThreadData::sendMouseEventsUpOwnerChain ( IEvent *lastEvent,
                                                    IEvent* lastSynthesizedEvent )
{
  if ( lastEvent )
  {
     // We need to relook up the window associated with this event in case
     // the window was deleted as a result of dispatching the event to the
     // original IWindow. Originally we saved the IWindow from handleMouseEvents
     // but this cased a segmentation fault is this routine. We tried to
     // dispatch the event to the owner of a window that had already been
     // deleted . This method is slower but safer.
     XEvent *event = (XEvent*) lastEvent->parameter2().asUnsignedLong();
     Widget widget = XtWindowToWidget( event->xany.display, event->xany.window );
     IWindow *win = IWindow::windowWithHandle( widget );

     // Try to find an IWindow to dispatch the event to. Walk up the parent chain
     // until an IWindow is found.
     while ( !win  &&  widget )
     {
        widget = XtParent( (Widget) widget );
        win = IWindow::windowWithHandle( widget );
     }

     if ( win )
     {
        bool eventHandled( false );
        IWindow *owner = win->owner();
        if ( owner )
        {
           if ( win->passEventToOwner( *lastEvent ) )
           {
              // before we pass the event up the chain we need to set the
              // handle
              lastEvent->setHandle( owner->handle() );
              eventHandled = owner->dispatch( *lastEvent );
           }
        }
        // now the event has gone up the owner chain dispatch any synthesized
        // events.
        if ( lastSynthesizedEvent )
           bool synEventHandled = win->dispatch( *lastSynthesizedEvent );
     }
  }
  return *this;
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThreadData::sendKeyEventsUpOwnerChain                                |
|                                                                              |
| If the event created for the Key press or Key release in                     |
| handleKeyboardEvents is not handled by the handlers for the IWindow then     |
| this method sends the event to the owner to start its way up the owner chain |
------------------------------------------------------------------------------*/
ICurrentThreadData&
  ICurrentThreadData::sendKeyEventsUpOwnerChain ( IEvent *lastEvent )
{
  if ( lastEvent )
  {
     // see comment in sendMouseEventsUpOwnerChain.
     XEvent *event = (XEvent*) lastEvent->parameter2().asUnsignedLong();

     // try to dispatch the event to the widget with focus.
     Widget fWidget =
        XmGetFocusWidget( XtWindowToWidget( event->xany.display,
                                            event->xany.window ) );
     IWindow *win;
     if ( fWidget )
     {
        win = IWindow::windowWithHandle( fWidget );
     }
     else
     {
        fWidget = XtWindowToWidget( event->xany.display,
                                    event->xany.window );
        win = IWindow::windowWithHandle( fWidget );
     }

     // Try to find an IWindow to dispatch the event to. Walk up the parent
     // chain until an IWindow is found. We need this check because of our
     // aggregate widgets.
     while ( !win  &&  fWidget )
     {
        fWidget = XtParent( (Widget) fWidget );
        win = IWindow::windowWithHandle( fWidget );
     }

     if ( win )
     {
        bool eventHandled( false );
        IWindow *owner = win->owner();
        if ( owner )
        {
           if ( win->passEventToOwner( *lastEvent ) )
           {
              // change the handle to the owner's before dispatching the
              // event.
              lastEvent->setHandle( owner->handle() );
              eventHandled = owner->dispatch( *lastEvent );
           }
        }
     }
  }
  return *this;
}
#endif //IC_MOTIF

#ifdef IC_WIN
// Note: These functions are inlined on all platforms except Windows due to
//       variables they reference not being exported.
/*------------------------------------------------------------------------------
| IThread::setDefaultAutoInitGUI                                               |
------------------------------------------------------------------------------*/
void IThread::setDefaultAutoInitGUI ( bool newFlag )
{
  IThread::dfltAutoInitGUI = newFlag;
}

/*------------------------------------------------------------------------------
| IThread::defaultQueueSize                                                    |
------------------------------------------------------------------------------*/
long IThread::defaultQueueSize ( )
{
  ITRACE_MOTIF_NOP();
  ITRACE_WIN_NOP();
  return IThread::dfltQueueSize;
}

/*------------------------------------------------------------------------------
| IThread::setDefaultQueueSize                                                 |
------------------------------------------------------------------------------*/
void IThread::setDefaultQueueSize ( long newQSize )
{
  ITRACE_MOTIF_NOP();
  ITRACE_WIN_NOP();
  IThread::dfltQueueSize = newQSize;
}
#endif // IC_WIN


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThread::setXerrorCode                                                |
------------------------------------------------------------------------------*/
void ICurrentThread::setXerrorCode ( int errorCode )
{
  IThreadData::setXerrorCode( *this, errorCode );
}

/*------------------------------------------------------------------------------
| ICurrentThread::isXerrorCodeAvailable                                        |
------------------------------------------------------------------------------*/
bool ICurrentThread::isXerrorCodeAvailable ( ) const
{
  return IThreadData::xerrorCode( *this ) ? true : false;
}

/*------------------------------------------------------------------------------
| ICurrentThread::XerrorCode                                                   |
------------------------------------------------------------------------------*/
int ICurrentThread::XerrorCode ( ) const
{
  return IThreadData::xerrorCode( *this );
}
#endif // IC_MOTIF

/*------------------------------------------------------------------------------
| ICurrentThread::remainingStack                                               |
------------------------------------------------------------------------------*/
unsigned long ICurrentThread::remainingStack ( ) const
{
  return INonGUIThread::current().remainingStack( );
}

/*------------------------------------------------------------------------------
| ICurrentThread::sleep                                                        |
------------------------------------------------------------------------------*/
ICurrentThread& ICurrentThread::sleep ( unsigned long milliseconds )
{
  INonGUIThread::current().sleep( milliseconds );
  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentThread::waitFor                                                      |
------------------------------------------------------------------------------*/
ICurrentThread& ICurrentThread::waitFor( const INonGUIThread& anotherThread,
                                         long                 timeout )
{
  INonGUIThread::current().waitFor( anotherThread, timeout );
  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentThread::waitForAllThreads                                            |
------------------------------------------------------------------------------*/
ICurrentThread& ICurrentThread::waitForAllThreads ( long timeout,
                                                   bool *timedOut)
{
  INonGUIThread::current().waitForAllThreads( timeout, timedOut );
  return *this;
}

/*------------------------------------------------------------------------------
| ICurrentThread::waitForAnyThread                                             |
------------------------------------------------------------------------------*/
IThreadId ICurrentThread::waitForAnyThread ( long timeout )
{
  return INonGUIThread::current().waitForAnyThread( timeout );
}

/*------------------------------------------------------------------------------
| ICurrentThread::exit                                                         |
------------------------------------------------------------------------------*/
void ICurrentThread::exit ( unsigned long returnCode )
{
  INonGUIThread::current().exit( returnCode );
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| IThread__matchXColor                                                         |
| Utility function used by IThread__xmColorProc                                |
------------------------------------------------------------------------------*/
static void IThread__matchXColor( IColorMap& colormap, XColor& xcolor)
{
   ITRACE_DEVELOP(IString("matchXColor requested" ) +
                  IString(" red=") + IString( xcolor.red ).d2x() +
                  IString(" green=") + IString( xcolor.green ).d2x() +
                  IString(" blue=") + IString( xcolor.blue ).d2x() );
   IColor color( (unsigned char)(xcolor.red >> 8),
                 (unsigned char)(xcolor.green >> 8),
                 (unsigned char)(xcolor.blue >> 8) );
   // Find the closest match using index() and then construct an new IColor
   // object with these RGB values.
   IColor matchedColor( color.index(), &colormap );
   xcolor.red   = matchedColor.redMix() << 8;
   xcolor.green = matchedColor.greenMix() << 8;
   xcolor.blue  = matchedColor.blueMix() << 8;
   ITRACE_DEVELOP(IString("matchXColor allocated" ) +
                  IString(" red=") + IString( xcolor.red ).d2x() +
                  IString(" green=") + IString( xcolor.green ).d2x() +
                  IString(" blue=") + IString( xcolor.blue ).d2x() +
                  IString(" oldpixel=") + IString( color.index() ) +
                  IString(" newpixel=") + IString( matchedColor.index() ) );
   // Note that the pixel value in the XColor is not used in this case.
}

/*------------------------------------------------------------------------------
| IThread__xmColorProc                                                         |
| Function called by Motif to generate colors.  We intercept this call         |
| to force Motif to use colors from our color table.                           |
------------------------------------------------------------------------------*/
static void IThread__xmColorProc(XColor *bg_color, XColor *fg_color,
        XColor *sel_color, XColor *ts_color, XColor *bs_color)
{
   IFUNCTRACE_DEVELOP();
   // Call the default procedure provided by Motif.
   (*IThread__defaultColorProc)(bg_color, fg_color, sel_color,
                                ts_color, bs_color);

   // Now, match the colors Motif want's with closest matches in our
   // color table, and return the ones from our table.  XAllocColor is
   // attempted with these colors only.  bg_color is already allocated, so
   // we only need to worry about the other 4.  Note that 0 is valid for
   // any of the 4 colors being generated.
   IColorMap& colormap( IApplication::current().colorMap() );
   if (fg_color)  IThread__matchXColor( colormap, *fg_color );
   if (sel_color) IThread__matchXColor( colormap, *sel_color );
   if (ts_color)  IThread__matchXColor( colormap, *ts_color );
   if (bs_color)  IThread__matchXColor( colormap, *bs_color );
}
#endif

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
   Return the application context for the current thread.
   This method is Motif-specific.
------------------------------------------------------------------------------*/
IContextHandle ICurrentThread::appContext ( ) const
{
  return this->anchorBlock();
}
#endif //IC_MOTIF

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICurrentThread::appShell                                                     |
------------------------------------------------------------------------------*/
IWindowHandle  ICurrentThread::appShell ( ) const
{
  return Inherited::applicationShell();
}
#endif //IC_MOTIF
#endif // IC_OBSOLETE

class IThreadStaticDestructor
{
public:
    ~IThreadStaticDestructor() {
        IFUNCTRACE_DEVELOP();
        if (IThread::pCurrent) {
            delete IThread::pCurrent;
            IThread::pCurrent = 0;
        }
    }
};

IThreadStaticDestructor privateDestructor;

