// Revision: 03 1.34 source/ui/basectl/imcelcv.cpp, canvas, ioc.v400, 990114 
/*******************************************************************************
* FILE NAME: imcelcv.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imcelcv.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C" {
  #define INCL_GPIPRIMITIVES
  #include <iwindefs.h>
}

#include <imcelcv.hpp>

#include <ialgnatt.hpp>
#include <ibidiset.hpp>
#include <icconst.h>
#include <icvobsrv.hpp>
#include <iexcept.hpp>
#include <iexgrprt.hpp>
#include <igbase2d.hpp>
#include <igraph2d.hpp>
#include <igrport.hpp>
#include <imcelcvh.hpp>
#include <imcelcvl.hpp>
#include <imcelprv.hpp>
#include <imcelrch.hpp>
#include <inotifev.hpp>
#include <ipoint.hpp>
#include <irect.hpp>
#include <itrace.hpp>
#include <iwposbuf.hpp>


// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IMCELCV_CPP_
  #include <ipagetun.h>
#endif

const IMultiCellCanvas::Style
  IMultiCellCanvas::gridLines       ( 0, ICNV_GRID ),
  IMultiCellCanvas::dragLines       ( 0, ICNV_DRAG ),
  IMultiCellCanvas::spaceAddedToLast( 0, ICNV_ADDTOLAST ),
  IMultiCellCanvas::classDefaultStyle =  WS_VISIBLE;

#ifdef IC_MOTIF
const char
  IMultiCellCanvasData::className[21] = "IMultiCellCanvas    ";
#endif

unsigned long
  IMultiCellCanvas::ulDefaultWidth  = 10,
  IMultiCellCanvas::ulDefaultHeight = 10;


/*------------------------------------------------------------------------------
  IMultiCellCanvasData ctor
------------------------------------------------------------------------------*/
IMultiCellCanvasData::IMultiCellCanvasData ( )
  : layoutInProgress( 0 ),
    sizeForLayout( ISize( 0, 0 ) )
  , pMcelcvObserver( 0 )
#ifdef IC_PMWIN
  , fBidiLayout( IBidiSettings::layoutLeftToRight )
#endif
  { }

/*------------------------------------------------------------------------------
  IMultiCellCanvasData dtor
------------------------------------------------------------------------------*/
IMultiCellCanvasData::~IMultiCellCanvasData ( )
{
  if ( pMcelcvObserver )
     delete pMcelcvObserver;
}

/*------------------------------------------------------------------------------
  IMultiCellCanvas :: currentDefaultStyle
  Default style for new objects (initial value).
------------------------------------------------------------------------------*/
IMultiCellCanvas::Style IMultiCellCanvas::currentDefaultStyle = WS_VISIBLE;


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: defaultStyle
------------------------------------------------------------------------------*/
IMultiCellCanvas::Style IMultiCellCanvas::defaultStyle ( )
{
  return currentDefaultStyle;
}

/*------------------------------------------------------------------------------
  IMultiCellCanvas :: setDefaultStyle
------------------------------------------------------------------------------*/
void IMultiCellCanvas::setDefaultStyle( const IMultiCellCanvas::Style& style )
{
  currentDefaultStyle = style;
}

/*------------------------------------------------------------------------------
| IMultiCellCanvas::convertToGUIStyle                                          |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long IMultiCellCanvas::convertToGUIStyle ( const IBitFlag& guiStyle,
                                                    bool bExtOnly ) const
{
  // Obtain the style from the class (ICanvas) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if ( bExtOnly )
  {
    // Use mask to only return extended styles in the user defined range.
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
    // Replace canvas window style with multicell canvas window style.
#ifdef IC_PMWIN
    ulStyle |= ( guiStyle.asUnsignedLong() & ~IC_CVS_CANVAS ) |
                                              IC_CVS_MULTICELLCANVAS;
#else
    // No styles are defined for the base canvas, so we do not mask.
    ulStyle |= guiStyle.asUnsignedLong();
#endif

    // No additional processing required since ICanvas::convertToGUIStyle is
    // doing all the required work ...
  }

  return ulStyle;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvasDeleter - Static function used as input to allElementsDo(),
   resulting in all members of the collection being deleted.
------------------------------------------------------------------------------*/
bool IMultiCellCanvasDeleter ( IMCCvCell* const& cell, void* )
{
  if ( cell->isInUse() )
  {
     delete (IMCCvCell*)cell;
  }
  return true;
}


/*------------------------------------------------------------------------------
   IRowColumnHeaderDeleter - Static function used as input to allElementsDo(),
   resulting in all members of the collection being deleted.
------------------------------------------------------------------------------*/
bool IRowColumnHeaderDeleter ( IRCHeader* const& header, void* )
{
  delete (IRCHeader*)header;
  return true;
}


/*------------------------------------------------------------------------------
   RowColHeaderInitializer - Static function used as input to allElementsDo(),
   resulting in all headers of the collection having a default width/height
   before affected by minimum sizes of child windows.
------------------------------------------------------------------------------*/
bool IRowColHeaderInitializer ( IRCHeader* const& header, void* )
{
  unsigned long width = 0;
  if ( !header->isDefaultValue() )
  {                     // Use user's value.
     width = header->requestedWidth();
  }
  header->setWidth( width );

  return true;
}


/*------------------------------------------------------------------------------
|  Constructor. Window created in ICanvas class. Create the collections.       |
------------------------------------------------------------------------------*/
IMultiCellCanvas::IMultiCellCanvas ( unsigned long windowIdentifier,
                                     IWindow* parent,
                                     IWindow* owner,
                                     const IRectangle &initialSize,
                                     const Style &style )
  : IMultiCellCanvas::Inherited( ),
    fMultiCol( false ),
    fMultiRow( false ),
    fExpandCol( false ),
    fExpandRow( false ),
    pCanvasList( new IMultiCellCanvasList ),
    pRowColHeaderList( new IRowColumnHeaderList ),
    pCanvasHandler( new IMultiCellCanvasHandler ),
    fMultiCellCanvasData( new IMultiCellCanvasData )
{
  // Save the extended style to make sure we have a copy of it stored
  setExtendedStyle( extendedStyle() | style.asExtendedUnsignedLong() );

  // Create the presentation system window.
  this->initialize( windowIdentifier,
                    parent,
                    owner,
                    initialSize,
                    this->convertToGUIStyle( style ),
                    this->convertToGUIStyle( style, true ) );

  pCanvasHandler->handleEventsFor( this );
  fMultiCellCanvasData->pMcelcvObserver = new ICanvasObserver(this);
#ifdef IC_MOTIF
  XtVaSetValues( this->handle(),
                 XmNuserData, (XtPointer) fMultiCellCanvasData->className,
                 NULL );
#endif
}


/*------------------------------------------------------------------------------
|  Destructor. Remove and delete collection objects, and delete collections.   |
------------------------------------------------------------------------------*/
IMultiCellCanvas::~IMultiCellCanvas ( )
{
  /*---------------------------------------------------------------------
  |  NOTE REGARDING ATTRIBUTES:                                         |
  |  This would be the appropriate place to interate child windows,     |
  |  issuing stopHandlingNotificationsFor() for those remaining which   |
  |  were not removed via the removeFromCell method.  However, at this  |
  |  point the canvas window itself is likely destroyed so we do not    |
  |  issue stopHandlingNotificationsFor for the remaining children as   |
  |  this canvas and its observer are in the process of being destroyed.|
  ---------------------------------------------------------------------*/

  if ( pCanvasList )
  {
     pCanvasList->removeAll( &IMultiCellCanvasDeleter );
     delete pCanvasList;
  }

  if ( pRowColHeaderList )
  {
     pRowColHeaderList->removeAll( &IRowColumnHeaderDeleter );
     delete pRowColHeaderList;
  }

  if ( pCanvasHandler )
  {
     pCanvasHandler->stopHandlingEventsFor( this );
     delete pCanvasHandler;
  }

  if ( fMultiCellCanvasData )
  {
     delete fMultiCellCanvasData;
  }
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: hasDragLines
   Returns true if the multi-cell canvas has draggable lines.
------------------------------------------------------------------------------*/
bool IMultiCellCanvas::hasDragLines ( ) const
{
  return( (extendedStyle() & dragLines.asExtendedUnsignedLong() ? true
                                                                : false) );
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: enableDragLines
   Either turn on or turn off the dragLines style bit.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::enableDragLines ( bool bEnable )
{
#ifdef IC_PMWIN
  unsigned long ulExtStyles = this->extendedStyle();
  bool attrChanged = false;

  if ( bEnable )
  {
    if ( !(ulExtStyles & dragLines.asExtendedUnsignedLong()) )
    {
      ulExtStyles |= dragLines.asExtendedUnsignedLong();
      attrChanged = true;
    }
  }
  else
  {
    if (ulExtStyles & dragLines.asExtendedUnsignedLong())
    {
      ulExtStyles &= ~dragLines.asExtendedUnsignedLong();
      attrChanged = true;
    }
  }

  if ( attrChanged )
  {
    this->setExtendedStyle( ulExtStyles );
    this->refresh();
  }
#endif // IC_PMWIN
#ifdef IC_MOTIF
  // When this is implemented, code to add/remove bit gravity
  // (similar to that in IMultiCellCanvas::enableGridLines) should
  // be added here.
#endif
  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: disableDragLines
   Turn off the dragLines style bit.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::disableDragLines ( )
{
  return this->enableDragLines( false );
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: hasGridLines
   Returns true if the multi-cell canvas has grid lines.
------------------------------------------------------------------------------*/
bool IMultiCellCanvas::hasGridLines ( ) const
{
  return( (this->extendedStyle() & gridLines.asExtendedUnsignedLong() ? true
                                                                      : false) );
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: enableGridLines
   Either turn on or turn off the gridLines style bit.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::enableGridLines ( bool bEnable )
{
  unsigned long ulExtStyles = this->extendedStyle();
  bool attrChanged = false;

  if ( bEnable )
  {
    if ( !(ulExtStyles & gridLines.asExtendedUnsignedLong()) )
    {
      ulExtStyles |= gridLines.asExtendedUnsignedLong();
      attrChanged = true;
    }
  }
  else
  {
    if ( ulExtStyles & gridLines.asExtendedUnsignedLong() )
    {
      ulExtStyles &= ~gridLines.asExtendedUnsignedLong();
      attrChanged = true;
    }
  }

  if ( attrChanged )
  {
#ifdef IC_MOTIF
     XSetWindowAttributes
       attributes;
     if ( bEnable )
     {
        // Remove bit gravity, so the canvas paints its entire window
        // whenever it is sized, including resized smaller. This keeps
        // the grid lines from a previous paint from being left on the
        // screen after a partial paint.
        attributes.bit_gravity = ForgetGravity;
     }
     else
     {
        // Add bit gravity, so the canvas repaints only invalidated
        // areas when it is resized. This is a performance feature
        // to reduce the number of paints that occur during resizing.
        attributes.bit_gravity = StaticGravity;
     }
     XChangeWindowAttributes( XtDisplay( (Widget) this->handle() ),
                              XtWindow( (Widget) this->handle() ),
                              CWBitGravity,
                              &attributes );
#endif // IC_MOTIF

    this->setExtendedStyle( ulExtStyles );
    this->refresh();
  }
  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: disableGridLines
   Turn off the gridLines style bit.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::disableGridLines ( )
{
  return this->enableGridLines( false );
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: addToCell
   Add a window to a cell of the multi-cell canvas.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::addToCell ( IWindow*      childWindow,
                                                unsigned long startingColumn,
                                                unsigned long startingRow,
                                                unsigned long numberOfColumns,
                                                unsigned long numberOfRows )
{
  IASSERTPARM(childWindow != 0);
  IASSERTPARM(childWindow->parent()== this );
  IASSERTPARM(startingColumn != 0 && numberOfColumns != 0);
  IASSERTPARM(startingRow != 0 && numberOfRows != 0);
  IMCCvCell* pCell = new IMCCvCell( startingColumn,
                                    startingRow,
                                    childWindow,
                                    numberOfColumns,
                                    numberOfRows );

  /* Let's make sure that we have no overlaps. */
  IMCCvCell& cell = (*pCanvasList)(startingColumn, startingRow);
  if (cell.isInUse())
  {
     delete pCell;
     ITHROWLIBRARYERROR1(IC_MULTICELLCANVAS_CELL_OVERLAP,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable,
                         IString( "(" + IString(startingColumn) +
                                  "," + IString(startingRow) + ")" ) );
  }

  /* O.K. to add the cell to the canvas. */
  pCanvasList->addOrReplaceElementWithKey(pCell);

  /* Ensure Column/Row header records exist. */
  pRowColHeaderList->updateHeaders(startingColumn + numberOfColumns - 1,
                                   startingRow + numberOfRows - 1);

  if (numberOfColumns > 1)
  {                     // Child window spans multiple columns.
     fMultiCol++;       // Track expandable columns.
  }
  if (numberOfRows > 1)
  {                     // Child window spans multiple columns.
     fMultiRow++;       // Track expandable rows.
  }

  unsigned long layoutAttributeOn ( IWindow::layoutChanged );
  if ( this->invalidatedRect() == IRectangle() )
  {  // Request a refresh.  Most windows, when created and then added to
     // the canvas, automatically invalidate the parent (this canvas).  There
     // are a few that do not, however, so this is here to force consistency.

     #ifdef IC_TRACE_DEVELOP
     // This test is redundant, but it is useful for debugging to see when
     // this code is causing a refresh.
     if ( !this->isLayoutDistorted( IWindow::immediateUpdate) )
        ITRACE_DEVELOP("addToCell - refresh");
     #endif

     layoutAttributeOn |= IWindow::immediateUpdate;
  }

  setLayoutDistorted(layoutAttributeOn, 0);

  if ( fMultiCellCanvasData->pMcelcvObserver )
  {
     ICanvasObserver* cObsrv = fMultiCellCanvasData->pMcelcvObserver;
     cObsrv->handleNotificationsFor(*childWindow,
                               IWindow::attributeAddReplaceId);
     cObsrv->handleNotificationsFor(*childWindow,
                               IWindow::attributeRemoveId);
  }

  return *this;
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: removeFromCell
  Remove a window from a cell if found and delete the cell object.
------------------------------------------------------------------------------*/
IWindow* IMultiCellCanvas::removeFromCell ( IWindow* childWindow )
{
  IWindow* pWindow = 0;
  IMCCvCell& cell = (*pCanvasList)(childWindow);
  if (cell.isInUse())
  {
     if (cell.columnRange() > 1)
     {                  // Child window spans multiple columns.
        fMultiCol--;    // Track expandable columns.
     }
     if (cell.rowRange() > 1)
     {                  // Child window spans multiple rows.
        fMultiRow--;    // Track expandable rows.
     }
     if ( fMultiCellCanvasData->pMcelcvObserver )
     {
        ICanvasObserver* cObsrv = fMultiCellCanvasData->pMcelcvObserver;
        cObsrv->stopHandlingNotificationsFor(*childWindow,
                                    IWindow::attributeAddReplaceId);
        cObsrv->stopHandlingNotificationsFor(*childWindow,
                                    IWindow::attributeRemoveId);
     }
     pCanvasList->removeElementWithKey(cell.key());
     delete &cell;
     setLayoutDistorted(IWindow::layoutChanged, 0);
     pWindow = (IWindow*)childWindow;
  }
  return pWindow;
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: removeFromCell
  Remove a window from a cell if found and delete the cell object.
------------------------------------------------------------------------------*/
IWindow* IMultiCellCanvas::removeFromCell ( unsigned long column,
                                            unsigned long row )
{
  return removeFromCell( windowInCell( column, row ));
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: windowInCell
  Return a pointer to the IWindow for the cell at the specified column and row.
------------------------------------------------------------------------------*/
IWindow* IMultiCellCanvas::windowInCell ( unsigned long column,
                                          unsigned long row ) const
{
  IMCCvCell& cell = (*pCanvasList)(column, row);
  return cell.window();
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: setColumnWidth
  Set the minimum width of a column.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::setColumnWidth ( unsigned long column,
                                                     unsigned long widthInPixels,
                                                     bool          expandable )
{
  /* Ensure Column/Row header records exist. */
  if (column > pRowColHeaderList->columns())
     pRowColHeaderList->updateHeaders(column, 0);

  /* Get the header record for this column. */
  IRCHeader& colHeader = (*pRowColHeaderList)(column, 0);
  if (expandable  &&  !colHeader.isExpandable())
  {                     // New expandable column.
     fExpandCol++;      // Track it.
  }
  else if (!expandable  &&  colHeader.isExpandable())
  {                     // Column no longer expandable.
     fExpandCol--;      // Track it.
  }
  colHeader.setExpandable(expandable);
  colHeader.setRequestedWidth(widthInPixels);

  setLayoutDistorted(IWindow::layoutChanged, 0);
  return *this;
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: setRowHeight
  Set the minimum height of a row.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::setRowHeight( unsigned long row,
                                                  unsigned long heightInPixels,
                                                  bool          expandable )
{
  /* Ensure Column/Row header records exist. */
  if (row > pRowColHeaderList->rows())
     pRowColHeaderList->updateHeaders(0, row);

  /* Get the header record for this row. */
  IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
  if (expandable  &&  !rowHeader.isExpandable())
  {                     // New expandable row.
     fExpandRow++;      // Track it.
  }
  else if (!expandable  &&  rowHeader.isExpandable())
  {                     // Row no longer expandable.
     fExpandRow--;      // Track it.
  }
  rowHeader.setExpandable(expandable);
  rowHeader.setRequestedHeight(heightInPixels);

  setLayoutDistorted(IWindow::layoutChanged, 0);
  return *this;
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: columnWidth
  Query the current width of a column.
------------------------------------------------------------------------------*/
unsigned long IMultiCellCanvas::columnWidth ( unsigned long column ) const
{
  /* Get the header record for this column. */
  IRCHeader& colHeader = (*pRowColHeaderList)(column, 0);
  return colHeader.width();
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: rowHeight
  Query the current height of a row.
------------------------------------------------------------------------------*/
unsigned long IMultiCellCanvas::rowHeight ( unsigned long row ) const
{
  /* Get the header record for this row. */
  IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
  return rowHeader.height();
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: setDefaultCell
   Set default cell size (width and height).
------------------------------------------------------------------------------*/
void IMultiCellCanvas::setDefaultCell ( const ISize &columnAndHeight )
{
  IMultiCellCanvas::ulDefaultWidth  = columnAndHeight.width();
  IMultiCellCanvas::ulDefaultHeight = columnAndHeight.height();
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: defaultCell
   Query default cell size, used to size empty rows and columns.
------------------------------------------------------------------------------*/
ISize IMultiCellCanvas::defaultCell ( )
{
  return ISize (IMultiCellCanvas::ulDefaultWidth,
                IMultiCellCanvas::ulDefaultHeight);
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: isColumnExpandable
   Return true if the column is expandable.
------------------------------------------------------------------------------*/
bool IMultiCellCanvas::isColumnExpandable ( unsigned long column ) const
{
  /* Get the header record for this column. */
  IRCHeader& colHeader = (*pRowColHeaderList)(column, 0);
  return colHeader.isExpandable();
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: isRowExpandable
   Return true if the row is expandable.
------------------------------------------------------------------------------*/
bool IMultiCellCanvas::isRowExpandable ( unsigned long row ) const
{
  IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
  return rowHeader.isExpandable();
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas::hasChildrenToLayout
  Identify whether the canvas contains any child windows to manage.  The
  application must have passed these child windows to the addToCell function.
------------------------------------------------------------------------------*/
bool IMultiCellCanvas::hasChildrenToLayout ( ) const
{
  // Search for each child window in the list of managed children.
  bool
    childToManage = false;
  IMultiCellCanvas
   *nonConstThis = (IMultiCellCanvas*)this;
  ChildCursor
    cursor( *nonConstThis );
  for ( cursor.setToFirst();
        !childToManage  &&  cursor.isValid();
        cursor.setToNext() )
  {
     IWindow
      *childWindow = IWindow::childWindowAt( cursor );
     IMCCvCell
      &cell = (*pCanvasList)( childWindow );
     if ( cell.isInUse() )
     {
        childToManage = true;
     }
  }
  return childToManage;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: setLayoutDistorted
   Tell the canvas to determine whether or not its layout is still accurate,
   so it can make corrections (later) if needed.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::setLayoutDistorted ( unsigned long layoutAttributeOn,
                                                         unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = (layoutAttributeOn &
                               (unsigned long)
                                  (~(IWindow::sizeChanged |
                                    IWindow::childMinimumSizeChanged |
                                    IWindow::childWindowCreated |
                                    IWindow::childWindowDestroyed)));
         // (Ignore child changes until the application calls addToCell
         // or removeFromCell.)

  // Ignore child minimum size changes during a layout (during a layout,
  // assume these are a by-product of calling calcMinimumSize, such is
  // the case for a child set or multicell canvas).
  if ( ( layoutAttributeOn & IWindow::childMinimumSizeChanged )  &&
       fMultiCellCanvasData->layoutInProgress == 0 )
  {                     // Change to a cell.
     ulFlagsOn |= IWindow::layoutChanged;  // Flag it.
     if ( ! ( layoutAttributeOn & IWindow::fontPropogated )  &&
          ! this->isLayoutDistorted( IWindow::fontPropogated ) )
     {                  // Change buffered til now?
        ulFlagsOn |= IWindow::immediateUpdate;
     }
  }

  Inherited::setLayoutDistorted( ulFlagsOn, layoutAttributeOff );

  return *this;
}


/*------------------------------------------------------------------------------
  IMultiCellCanvas :: layout
  Implements the layout alogrithm for this canvas.  It computes the position
  and size of each window in the canvas window, by using the member functions
  pass1(), pass2(), pass3() and pass4().
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::layout ( )
{
  IMODTRACE_DEVELOP("MultiCellCv::layout");

  if ( this->isLayoutDistorted( IWindow::layoutChanged )  &&
       this->isUpdateEnabled() )
  {
     fMultiCellCanvasData->layoutInProgress++;

     this->pass1();
     if ( fMultiCol || fMultiRow )
     {
        this->pass2();
     }
     this->pass3();

#ifdef IC_PMWIN
     unsigned long style = this->style();
     if ( (style & WS_VISIBLE ))
       this->enableUpdate( false );      // Surpress screen updates.
#endif

     this->pass4();

     ITRACE_DEVELOP( IString( "Canvas size = (" ) + layoutSize().asString() );
     this->setLayoutDistorted( 0, IWindow::layoutChanged );

#ifdef IC_PMWIN
     if ( ( style & WS_VISIBLE ) )
     {
       // enable update will show the window so only call it if the window is
       // visible
       this->enableUpdate( true );       // Show screen updates.
     }
#endif

     fMultiCellCanvasData->layoutInProgress--;
  }

#ifdef IC_MOTIF
  if ( hasGridLines() )
  {
     postEvent(IC_UM_DRAWGRIDLINES, 0, 0);
  }
#endif // IC_MOTIF

  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: computeExpandedRowsCols
   Compute the available space within which expandable rows and columns may
   grow, and grow them as appropriate.  Note that this routine must update
   the offsets stored by the row and column headers.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::computeExpandedRowsCols ( )
{
  IMODTRACE_DEVELOP("McellCv::computeExpandedRowsCols");
  unsigned long row;
  unsigned long col;

  unsigned long ulNonExpandWidth = 0;  // Minimum width of columns.
  unsigned long ulMinExpandWidth = 0;
  unsigned long ulExpandableCols = fExpandCol;
                        // Count of expandable columns.
  if (ulExpandableCols)
  {                     // Have an expandable column(s).
     for (col = 1; col <= pRowColHeaderList->columns(); col++)
     {                  // Loop through all columns.
        IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
        if (colHeader.isExpandable())
        {               // Found an expandable column.
           unsigned long ulWidth = colHeader.width();
           if (ulWidth == 0)
           {       // (Only if setColumnWidth and default cell are 0.)
              ulWidth = 1;   // Prevent from never expanding.
           }
           ulMinExpandWidth += ulWidth;
        }
        ulNonExpandWidth += colHeader.width();
     }
  }                     // End have an expandable column(s).

  unsigned long ulNonExpandHeight = 0; // Minimum height of rows.
  unsigned long ulMinExpandHeight = 0;
  unsigned long ulExpandableRows  = fExpandRow;
                        // Count of expandable rows.
  if (ulExpandableRows)
  {                     // Have an expandable row(s).
     for (row = 1; row <= pRowColHeaderList->rows(); row++)
     {                  // Loop through all rows.
        IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
        if (rowHeader.isExpandable())
        {               // Found an expandable row.
           unsigned long ulHeight = rowHeader.height();
           if (ulHeight == 0)
           {       // (Only if setRowHeight and default cell are 0.)
              ulHeight = 1;  // Prevent from never expanding.
           }
           ulMinExpandHeight += ulHeight;
        }
        ulNonExpandHeight += rowHeader.height();
     }
  }                     // End have an expandable row(s).

  // Calculate space that expandable rows/columns can take up.
  IRectangle
    rectForLayout = this->rectangleInsideBorder( this->size() );
  ISize
    sizeForLayout = rectForLayout.size();
  long lExpandWidth = sizeForLayout.width() - ulNonExpandWidth;
  long lExpandHeight = sizeForLayout.height() - ulNonExpandHeight;
  unsigned long
    initXOffset = rectForLayout.minX(),
    offset = initXOffset;

  // Take care of any extra horizontal space.
  if (lExpandWidth > 0  &&  ulExpandableCols)
  {                     // Have extra width for columns.
     long lExpandLeft = lExpandWidth;
     for (col = 1; col <= pRowColHeaderList->columns(); col++)
     {                  // Disperse expandable width proporionately.
        IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
        if (colHeader.isExpandable())
        {
           unsigned long ulWidth = colHeader.width();
           if (ulWidth == 0)
           {        // (Only if setColumnWidth and default cell are 0.)
              ulWidth = 1;             // Prevent from never expanding.
           }
           unsigned long ulPercent = (ulWidth << 7) / ulMinExpandWidth;
           unsigned long ulToAdd = (lExpandWidth * ulPercent) >> 7;
           ulWidth += ulToAdd;         // Grow proportionately.
           colHeader.setWidth(ulWidth);   // Save new size.
           lExpandLeft -= ulToAdd;
        }               // End expandable column.
        colHeader.setOffset( offset );
        offset += colHeader.width();
     }                  // End for all columns.
     if (lExpandLeft > 0)
     {                  // Have pels left over (from rounding errors).
        unsigned long ulExtraChunk = lExpandLeft / ulExpandableCols;
        unsigned long ulSinglePels = lExpandLeft % ulExpandableCols;
        offset = initXOffset;
        for (col = 1; col <= pRowColHeaderList->columns(); col++)
        {               // Disperse left-overs equally.
           IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
           if (colHeader.isExpandable())
           {
              unsigned long ulWidth = colHeader.width();
              ulWidth += ulExtraChunk; // Grow evenly.
              if (ulSinglePels)
              {                        // Left-over single pels.
                 ulWidth++;            // First-come, first-serve.
                 ulSinglePels--;
              }
              colHeader.setWidth(ulWidth);
           }            // End expandable column.
           colHeader.setOffset( offset );
           offset += colHeader.width();
        }               // End for all columns.
     }                  // End pels left over.
  }                     // End need to expand columns.

  if (lExpandHeight > 0  &&  ulExpandableRows)
  {                     // Have extra height for rows.
     unsigned long
       initYOffset = ( ICoordinateSystem::applicationOrientation() ==
                            ICoordinateSystem::kOriginLowerLeft ) ?
                       this->size().height() - rectForLayout.maxY() :
                       rectForLayout.minY();
     offset = initYOffset;
     long lExpandLeft = lExpandHeight;
     for (row = 1; row <= pRowColHeaderList->rows(); row++)
     {                  // Disperse expandable width proporionately.
        IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
        if (rowHeader.isExpandable())
        {
           unsigned long ulHeight = rowHeader.height();
           if (ulHeight == 0)
           {          // (Only if setRowHeight and default cell are 0.)
              ulHeight = 1;            // Prevent from never expanding
           }
           unsigned long ulPercent = (ulHeight << 7) / ulMinExpandHeight;
           unsigned long ulToAdd = (lExpandHeight * ulPercent) >> 7;
           ulHeight += ulToAdd;        // Grow proportionately
           rowHeader.setHeight(ulHeight); // Save new size
           lExpandLeft -= ulToAdd;
        }               // End expandable row.
        rowHeader.setOffset( offset );
        offset += rowHeader.height();
     }                  // End for all rows.
     if (lExpandLeft > 0)
     {                      // Have pels left over (from rounding errors)
        unsigned long ulExtraChunk = lExpandLeft / ulExpandableRows;
        unsigned long ulSinglePels = lExpandLeft % ulExpandableRows;
        offset = initYOffset;
        for (row = 1; row <= pRowColHeaderList->rows(); row++)
        {               // Disperse left-overs equally.
           IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
           if (rowHeader.isExpandable())
           {
              unsigned long ulHeight = rowHeader.height();
              ulHeight += ulExtraChunk;  // Grow evenly.
              if (ulSinglePels)
              {                        // Left-over single pels.
                 ulHeight++;           // First-come, first-serve.
                 ulSinglePels--;
              }
              rowHeader.setHeight(ulHeight);
           }            // End expandable row.
           rowHeader.setOffset( offset );
           offset += rowHeader.height();
        }               // End for all rows.
     }                  // End pels left over.
  }                     // End need to expand rows.

  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: pass1
   Get minimum sizes for all child windows, update the size of rows and
   columns that contain windows that don't span multiple rows and columns,
   and update the size of empty rows and columns.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::pass1 ( )
{
  IMODTRACE_DEVELOP("MCellCv::pass1");
  IMultiCellCanvasList::Cursor cursor(*pCanvasList);
  IMCCvCell* pCell;
  unsigned long row;
  unsigned long col;
  ISize siz;

  ITRACE_DEVELOP(IString("Start Pass 1: Rows: ") +
                 IString(pRowColHeaderList->rows()) +
                 IString(" Cols: ") + IString(pRowColHeaderList->columns()));
  pRowColHeaderList->allElementsDo ( &IRowColHeaderInitializer );
  /*------------------------------------------------------------------
  |  Iterate all the occupied cells in the collection:               |
  |    1) Call the child window to find its minimum size.            |
  |    2) Store its minimum size in the cell.                        |
  |    3) If the window occupies a single row/column, then update    |
  |       the size of the row/column (use the larger of the          |
  |       row/column size and the minimum size of the child window). |
  ------------------------------------------------------------------*/
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     pCell = pCanvasList->elementAt(cursor);
     (*pCell)
      .setWidth(0)
      .setHeight(0);

     if (pCell->isInUse())
     {
        /* Set the occupied cells to their min size */
        siz = pCell->window()->minimumSize();
        pCell->setMinimumWidth(siz.width());
        pCell->setMinimumHeight(siz.height());

        if (pCell->columnRange() == 1)
        {
           IRCHeader& colHeader = (*pRowColHeaderList)(pCell->column(), 0);
           if (colHeader.width() < pCell->minimumWidth())
           {
              colHeader.setWidth(pCell->minimumWidth());
           }
        }

        if (pCell->rowRange() == 1)
        {
           IRCHeader& rowHeader = (*pRowColHeaderList)(0, pCell->row());
           if (rowHeader.height() < pCell->minimumHeight())
           {
              rowHeader.setHeight(pCell->minimumHeight());
           }
        }
      } /* endif window */
  } /* endfor ALL */

  /*------------------------------------------------------------------
  |  Iterate all columns to update the size of empty columns with    |
  |  the default cell size.                                          |
  ------------------------------------------------------------------*/
  for (col = 1; col <= pRowColHeaderList->columns(); col++)
  {
     IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
     ITRACE_DEVELOP(IString ("ColHdr = ") + colHeader.asDebugInfo());
     if (colHeader.width() == 0  &&  colHeader.isDefaultValue())
     {
        colHeader.setWidth(IMultiCellCanvas::defaultCell().width());
     }
  }

  /*------------------------------------------------------------------
  |  Iterate all rows to update the size of empty rows with          |
  |  the default cell size.                                          |
  ------------------------------------------------------------------*/
  for (row = 1; row <= pRowColHeaderList->rows(); row++)
  {
     IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
     ITRACE_DEVELOP(IString ("RowHdr = ") + rowHeader.asDebugInfo());
     if (rowHeader.height() == 0  &&  rowHeader.isDefaultValue())
     {
        rowHeader.setHeight(IMultiCellCanvas::defaultCell().height());
     }
  }

#ifdef  IC_DEVELOP
  ITRACE_DEVELOP("After minimumSize");
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     pCell = pCanvasList->elementAt(cursor);
     ITRACE_DEVELOP(pCell->asDebugInfo());
  }
#endif

  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: pass2
   Update the size of all rows and columns that have windows that span
   multiple rows or columns, or are expandable, so that the multi-cell
   canvas is now at its minimum size.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::pass2 ( )
{
  IMODTRACE_DEVELOP("McellCv::pass2");
  IMultiCellCanvasList::Cursor cursor(*pCanvasList);
  IMCCvCell* pCell;

  /*------------------------------------------------------------------
  |  Iterate all occupied cells:                                     |
  |    1) Find all child windows that span multiple columns, where   |
  |       none of the columns are expandable.                        |
  |       If the window is wider than the sum of the columns, grow   |
  |       the first column that the window occupies.                 |
  |    2) Find all child windows that span multiple rows, where      |
  |       none of the rows are expandable.                           |
  |       If the window is taller than the sum of the rows, grow     |
  |       the first row that the window occupies.                    |
  ------------------------------------------------------------------*/
  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     pCell = pCanvasList->elementAt(cursor);

     // Window spans multiple columns.
     if (pCell->isInUse()  &&  pCell->columnRange() > 1)
     {
        // Determine the summation of the columns for this cell.
        unsigned long
          ulCellSum = 0,
          firstCol  = pCell->column(),
          col       = firstCol,
          lastCol   = firstCol + pCell->columnRange() - 1;
        bool expandable = false;
        for ( ; col <= lastCol  &&  !expandable; col++ )
        {
           IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
           if (colHeader.isExpandable())
           {            // Lies in an expandable column.
              expandable = true;  // Skip this window for now.
           }
           else
           {
              ulCellSum = ulCellSum + colHeader.width();
           }
        } /* endfor column headers */

        if (!expandable)
        {               // Window is not expandable.
           if (ulCellSum < pCell->minimumWidth())
           {
              // This child window needs more space than the sum of the widths
              // of its cells, so increase the width of its first or last column
              // (based on the style of the canvas) to make up the difference.
              unsigned long
                columnToGrow =
                  ( this->extendedStyle()
                      & spaceAddedToLast.asExtendedUnsignedLong() ) ?
                                                       lastCol : firstCol;
              IRCHeader &colHeader = (*pRowColHeaderList)( columnToGrow, 0 );
              ulCellSum -= colHeader.width();
              colHeader.setWidth(pCell->minimumWidth() - ulCellSum);
              ulCellSum += colHeader.width();
           }
           pCell->setWidth(ulCellSum);
        }
     }

     // Window spans multiple rows.
     if (pCell->isInUse()  &&  pCell->rowRange() > 1)
     {
        // Determine the summation of the rows for this cell.
        unsigned long
          ulCellSum = 0,
          firstRow  = pCell->row(),
          row       = firstRow,
          lastRow   = firstRow + pCell->rowRange() - 1;
        bool expandable = false;
        for ( ; row <= lastRow  &&  !expandable; row++ )
        {
           IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
           if (rowHeader.isExpandable())
           {            // Lies in an expandable row.
              expandable = true;  // Skip this window for now.
           }
           else
           {
              ulCellSum = ulCellSum + rowHeader.height();
           }
        }

        if (!expandable)
        {               // Window is not expandable.
           if (ulCellSum < pCell->minimumHeight())
           {
              // This child window needs more space than the sum of the heights
              // of its cells, so increase the height of its first or last row
              // (based on the style of the canvas) to make up the difference.
              unsigned long
                rowToGrow =
                  ( this->extendedStyle()
                      & spaceAddedToLast.asExtendedUnsignedLong() ) ?
                                                       lastRow : firstRow;
              IRCHeader &rowHeader = (*pRowColHeaderList)( 0, rowToGrow );
              ulCellSum -= rowHeader.height();
              rowHeader.setHeight(pCell->minimumHeight() - ulCellSum);
              ulCellSum += rowHeader.height();
           } /* endif */

           pCell->setHeight(ulCellSum);
        }
     } /* end pWindow & rowRange > 1 */
  } /* endfor  ALL */

  /*------------------------------------------------------------------
  |  Iterate all occupied cells to calculate how much to minimally   |
  |  grow expandable rows and columns:                               |
  |    1) Find all child windows that span multiple columns, where   |
  |       one or more of the columns are expandable.                 |
  |       Keep largest fraction of how much the existing widths      |
  |       must be grown to give the window its minimum width.        |
  |    2) Find all child windows that span multiple rows, where      |
  |       one or more of the rows are expandable.                    |
  |       Keep the largest fraction of how much the existing heights |
  |       must be grown to give the window its minimum height.       |
  ------------------------------------------------------------------*/
  unsigned long
    ulMinExpandColNumerator = 0,
    ulMinExpandColDenominator = 1,
    ulMinExpandRowNumerator = 0,
    ulMinExpandRowDenominator = 1;
  if (fExpandCol || fExpandRow)
  {                     // Canvas has expandable row/column.
     for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
     {                  // Repeat for all child windows.
        pCell = pCanvasList->elementAt(cursor);

        // Window spans multiple columns.
        if (pCell->isInUse()  &&  pCell->columnRange() > 1
              &&  pCell->width() == 0)
        {               // Assume no width means may be expandable.
           unsigned long
             ulCellSum = 0,
             col       = pCell->column(),
             lastCol   = col + pCell->columnRange();
           bool expandable = false;
           unsigned long ulExpandableSum = 0;
           for ( ; col < lastCol; col++ )
           {            // Sum the columns.
              IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
              unsigned long width = colHeader.width();
              ulCellSum += width;
              if (colHeader.isExpandable())
              {         // Lies in an expandable column.
                 if (width == 0)
                 {
                    width = 1;
                 }
                 ulExpandableSum += width;
                 expandable = true;  // Flag it.
              }
           }

           if (expandable  &&  ulCellSum < pCell->minimumWidth())
           {            // Need to expand columns to match.
              unsigned long toGrow = pCell->minimumWidth() - ulCellSum;
              if (toGrow * ulMinExpandColDenominator >
                    ulExpandableSum * ulMinExpandColNumerator)
              {         // Keep biggest ratio to grow expandable cols.
                 ulMinExpandColNumerator = toGrow;
                 ulMinExpandColDenominator = ulExpandableSum;
              }
           }
        }

        // Window spans multiple rows.
        if (pCell->isInUse()  &&  pCell->rowRange() > 1
              &&  pCell->height() == 0)
        {               // Assume no height means may be expandable.
           // Determine the summation of the rows for this cell.
           unsigned long
             ulCellSum = 0,
             row       = pCell->row(),
             lastRow   = row + pCell->rowRange();
           bool expandable = false;
           unsigned long ulExpandableSum = 0;
           for ( ; row < lastRow; row++ )
           {            // Sum the rows.
              IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
              unsigned long height = rowHeader.height();
              ulCellSum += height;
              if (rowHeader.isExpandable())
              {            // Lies in an expandable row.
                 if (height == 0)
                 {
                    height = 1;
                 }
                 ulExpandableSum += height;
                 expandable = true;  // Flag it.
              }
           }

           if (expandable  &&  ulCellSum < pCell->minimumHeight())
           {            // Need to expand rows to match.
              unsigned long toGrow = pCell->minimumHeight() - ulCellSum;
              if (toGrow * ulMinExpandRowDenominator >
                    ulExpandableSum * ulMinExpandRowNumerator)
              {         // Keep biggest ratio to grow expandable cols.
                 ulMinExpandRowNumerator = toGrow;
                 ulMinExpandRowDenominator = ulExpandableSum;
              }
           }
        }
     } /* end for loop */
  }

  /*------------------------------------------------------------------
  |  Iterate all rows and columns to minimally grow all expandable   |
  |  ones:                                                           |
  |    1) Find all columns that are expandable, and multiply their   |
  |       width by the fraction calculated above.                    |
  |    2) Find all rows that are expandable, and multiply their      |
  |       height by the fraction calculated above.                   |
  ------------------------------------------------------------------*/
  if (ulMinExpandColNumerator)
  {
     ulMinExpandColNumerator += ulMinExpandColDenominator;
     unsigned long
       col = 1,
       lastCol = col + pRowColHeaderList->columns();
     for ( ; col <= lastCol; col++ )
     {            // Sum the columns.
        IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
        if (colHeader.isExpandable())
        {         // Grow it by the calculated ratio.
           unsigned long origWidth = colHeader.width();
           if (origWidth == 0)
           {
              origWidth = 1;
           }
           unsigned long newWidth = origWidth * ulMinExpandColNumerator;
           bool roundUp =
             (newWidth % ulMinExpandColDenominator) > (origWidth / 2);
           newWidth /= ulMinExpandColDenominator;
           if (roundUp)
           {
              newWidth++;
           }
           colHeader.setWidth( newWidth );
        }
     }
  }

  // Grow all expandable rows by ratio calculated above.
  if (ulMinExpandRowNumerator)
  {
     ulMinExpandRowNumerator += ulMinExpandRowDenominator;
     unsigned long
       row = 1,
       lastRow = row + pRowColHeaderList->rows();
     for ( ; row <= lastRow; row++ )
     {            // Sum the rows.
        IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
        if (rowHeader.isExpandable())
        {         // Grow it by the calculated ratio.
           unsigned long origHeight = rowHeader.height();
           if (origHeight == 0)
           {
              origHeight = 1;
           }
           unsigned long newHeight = origHeight * ulMinExpandRowNumerator;
           bool roundUp =
             (newHeight % ulMinExpandRowDenominator) > (origHeight / 2);
           newHeight /= ulMinExpandRowDenominator;
           if (roundUp)
           {
              newHeight++;
           }
           rowHeader.setHeight( newHeight );
        }
     }
  }

#ifdef IC_DEVELOP
  ITRACE_DEVELOP("After multi-adjust");
  unsigned long col;
  for (col = 1; col <= pRowColHeaderList->columns(); col++)
  {
     IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
     ITRACE_DEVELOP(IString ("ColHdr = ") + colHeader.asDebugInfo());
  }
  unsigned long row;
  for (row = 1; row <= pRowColHeaderList->rows(); row++)
  {
     IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
     ITRACE_DEVELOP(IString ("RowHdr = ") + rowHeader.asDebugInfo());
  }
#endif

  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: pass3
   Store the minimum size of the canvas, then finish growing all expandable
   rows and columns, if necessary to fill the size of the multi-cell canvas.
   All row and column headers will be finalized with their size and offsets.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::pass3 ( )
{
  IMODTRACE_DEVELOP("McellCv::pass3");

  // Find the starting offset for positioning child windows.
  IPoint
    topLeftOffset( this->topLeftLayoutOffset() );
  unsigned long
    initialX = topLeftOffset.x();
#ifdef IC_ORIGINLOWERLEFT
  unsigned long
    initialY = this->bottomRightLayoutOffset().y();
#else
  unsigned long
    initialY = topLeftOffset.y();
#endif
  long
    lColumnOffset = initialX,
    minCanvasHeight = initialY;

  /*------------------------------------------------------------------
  | Now set the real offsets into the header records.                |
  | Note that rows are done bottom up so that we are in screen       |
  | coordinates (bottom left).                                       |
  ------------------------------------------------------------------*/
  unsigned long col;
  for (col = 1; col <= pRowColHeaderList->columns(); col++)
  {
     IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
     colHeader.setOffset(lColumnOffset);
     lColumnOffset += colHeader.width();
     ITRACE_DEVELOP(colHeader.asDebugInfo());
  }

  unsigned long row;
  for (row = 1; row <= pRowColHeaderList->rows(); row++)
  {
     IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
#ifndef IC_ORIGINLOWERLEFT
     rowHeader.setOffset( minCanvasHeight );
     ITRACE_DEVELOP( rowHeader.asDebugInfo() );
#endif
     minCanvasHeight += rowHeader.height();
  }

  // Calculate minimum size before expanding any more rows/cols.
  lColumnOffset -= initialX;
  minCanvasHeight -= initialY;
      // (ICanvas::sizeWithBorder will add these border sizes back in.)
  ISize
    sizLayout( this->sizeWithBorder( ISize( lColumnOffset,
                                            minCanvasHeight ) ) );
  lColumnOffset = sizLayout.width();
  ITRACE_DEVELOP(IString ("Min size = ") + sizLayout.asDebugInfo());
  if (sizLayout != layoutSize())
  {                     // New minimum size for canvas.
     setLayoutSize(sizLayout);    // Store the new size.
     setLayoutDistorted(IWindow::minimumSizeChanged, 0);
  }                     // Cause layout in parent canvas.

  if (fExpandCol || fExpandRow)
  {                     // Grow expandable rows/cols if possible.
     computeExpandedRowsCols();
  }                     // computeExpandedRowsCols must update offsets.

  // Position child windows based on the current height of the canvas,
  // unless it has no height.  In this case, position them based on the
  // calculated minimum height to optimize the case when/if the canvas
  // is next sized to its minimum height.
  // Also store the width used to layout child windows.  This is the sum
  // of the minimum widths of the canvas' columns, unless the canvas has
  // expandable columns, in which case it may be the width of the canvas.
  long lRowOffset = size().height();
  if ( lRowOffset == 0 )
  {
     lRowOffset = sizLayout.height();
  }
  if ( fExpandCol  &&
       this->size().width() > lColumnOffset )
  {
     lColumnOffset = this->size().width();
  }
  else if ( IBidiSettings::isBidiSupported() )
  {
     IBidiSettings
       bidiSettings( *this );
#ifdef IC_PMWIN
     this->fMultiCellCanvasData->fBidiLayout = bidiSettings.windowLayout();
#endif
     if ( bidiSettings.windowLayout() == IBidiSettings::layoutRightToLeft )
     {  // Layout based on right edge of canvas.
        lColumnOffset = this->size().width();
     }
  }
  (fMultiCellCanvasData->sizeForLayout)
   .setWidth( lColumnOffset )
   .setHeight( lRowOffset );

#ifdef IC_ORIGINLOWERLEFT
  // Convert offsets from top-relative to native bottom-relative.
  lRowOffset -= topLeftOffset.y();
  for (row = 1; row <= pRowColHeaderList->rows(); row++)
  {
     IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
     lRowOffset -= rowHeader.height();
     rowHeader.setOffset(lRowOffset);
     ITRACE_DEVELOP(rowHeader.asDebugInfo());
  }
#endif

  return *this;
}


/*------------------------------------------------------------------------------
   IMultiCellCanvas :: pass4
   Size and position all child windows based on the row and column headers.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas::pass4 ( )
{
  IMODTRACE_DEVELOP("McellCv::pass4");
  /*----------------------------------------------------------
  | Update the window pos buffer.                             |
  -----------------------------------------------------------*/
  IMultiCellCanvasList::Cursor cursor(*pCanvasList);
  IMCCvCell* pCell;
  unsigned long ulCellWidth;
  unsigned long ulCellHeight;

  ISize sizLayout = layoutSize();      // Save original size.
  IWindowPosBuffer wposbuf = fixupChildren();
  setLayoutSize(sizLayout);     // Fix value saved by fixupChildren.

  for (cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     pCell = pCanvasList->elementAt(cursor);
     if (pCell->isInUse())
     {
        unsigned long col = pCell->column();
        IRCHeader& colHeader = (*pRowColHeaderList)(col, 0);
        ulCellWidth = colHeader.width();
        long colOffset = colHeader.offset();
        if (pCell->columnRange() > 1)
        {
           IRCHeader& lastColHeader =
             (*pRowColHeaderList)(col + pCell->columnRange() - 1, 0);
           ulCellWidth = lastColHeader.offset() - colOffset
                           + lastColHeader.width();
        }

        unsigned long row = pCell->row();
        IRCHeader& rowHeader = (*pRowColHeaderList)(0, row);
        ulCellHeight = rowHeader.height();
        long rowOffset = rowHeader.offset();
        if (pCell->rowRange() > 1)
        {
           IRCHeader& lastRowHeader =
             (*pRowColHeaderList)(0, row + pCell->rowRange() - 1);
           unsigned long lastRowOffset = lastRowHeader.offset();
#ifdef IC_ORIGINLOWERLEFT
           // Calculate using bottom-relative offsets.
           ulCellHeight += (rowOffset - lastRowOffset);
           rowOffset = lastRowOffset;
#else
           // Calculate using top-relative offsets.
           ulCellHeight = lastRowOffset - rowOffset
                             + lastRowHeader.height();
#endif
        }

        IRectangle rect(colOffset, rowOffset,
                        colOffset + ulCellWidth,
                        rowOffset + ulCellHeight);

        // Check window for alignment attributes.

        const IAttribute* attr;
        const IAlignmentAttribute* algnAttr;
        bool algnAttrFound = false;
        IWindow* win = pCell->window();
        IWindow::AttributeCursor cursor (*win);
        algnAttr = (const IAlignmentAttribute*)NULL;
        for (cursor.setToFirst();
             !algnAttrFound && cursor.isValid();
             cursor.setToNext())
        {
          IAttributeName attrName=win->attributeNameAt(cursor);
          attr=win->attributeWithName(attrName);
          algnAttr = dynamic_cast<const IAlignmentAttribute*>(attr);
          if (algnAttr)
             algnAttrFound=true;
        }
        if (algnAttr)    // Process window alignment attribute
        {
          unsigned long ulXDivisor, ulYDivisor;
          long attColSiz, attRowSiz;
          bool bXYDefPos;
          ulXDivisor = ulYDivisor = 0;
          attColSiz = attRowSiz = 0;
          bXYDefPos = false;

          // Flag default pos.
#ifdef IC_ORIGINLOWERLEFT
          if ( (algnAttr->horizontalAlignment() ==
                 IAlignmentAttribute::kLeft) ||
               (algnAttr->verticalAlignment() ==
                 IAlignmentAttribute::kBottom) )
#else
          if ( (algnAttr->horizontalAlignment() ==
                 IAlignmentAttribute::kLeft) ||
               (algnAttr->verticalAlignment() ==
                 IAlignmentAttribute::kTop) )
#endif
            bXYDefPos = true;
          // Default use cell width and height
          if (algnAttr->horizontalAlignment() ==
               IAlignmentAttribute::kStretch)
            attColSiz=ulCellWidth;
          if (algnAttr->verticalAlignment() ==
               IAlignmentAttribute::kStretchVertical)
            attRowSiz=ulCellHeight;
          // Pos. non-default
          if (algnAttr->horizontalAlignment() ==
               IAlignmentAttribute::kRight)
            ulXDivisor=1;
          else if (algnAttr->horizontalAlignment() ==
               IAlignmentAttribute::kCenter)
            ulXDivisor=2;
#ifdef IC_ORIGINLOWERLEFT
          if (algnAttr->verticalAlignment() ==
               IAlignmentAttribute::kTop)
#else
          if (algnAttr->verticalAlignment() ==
               IAlignmentAttribute::kBottom)
#endif
            ulYDivisor=1;
          else if (algnAttr->verticalAlignment() ==
               IAlignmentAttribute::kCenterVertical)
            ulYDivisor=2;
          // Default is stretch, centerStretch, otherwise
          if ( (ulXDivisor) || (ulYDivisor) || (bXYDefPos) )
          {
            long attColOffset = 0;
            long attRowOffset = 0;
            unsigned long ulXDelta = 0;
            unsigned long ulYDelta = 0;
            // calc. right, center
            ulXDelta =
               (ulXDivisor ? ((ulCellWidth - pCell->minimumWidth())
                                             / ulXDivisor) : 0);
            attColOffset = ulXDelta + rect.minX();
            // calc. bottom, centerVer
            ulYDelta =
               (ulYDivisor ? ((ulCellHeight - pCell->minimumHeight())
                                              / ulYDivisor) : 0);
            attRowOffset = ulYDelta + rect.minY();
            // stretch or minimumSize
            if (attColSiz)
               attColSiz += attColOffset;
            else
               attColSiz = attColOffset + pCell->minimumWidth();
            if (attRowSiz)
               attRowSiz += attRowOffset;
            else
               attRowSiz = attRowOffset + pCell->minimumHeight();

            rect = IRectangle(attColOffset,
                              attRowOffset,
                              attColSiz,
                              attRowSiz);
          }
        }

        // Adjust the window size and position for controls that have
        // special layout adjustment needs (like combo box).

        IRectangle adjustRect = pCell->window()->layoutAdjustment();
        if ( adjustRect != IRectangle() )
        {
          // Since the adjust rectangle is non-zero, this is a special
          // layout control.  We now need to calculate the suggested
          // size for the control by adding the minimum size to the
          // adjusted size.  If the suggested size is bigger than the
          // size calculated by layout, the control is re-sized to the
          // suggested size.

          ISize adjustSize = pCell->window()->minimumSize() +
                             adjustRect.size();
          if ( adjustSize.width() > rect.width() )
          {
            if (adjustRect.left() != 0 )
              rect = IRectangle(rect.right()-adjustSize.width(),
                                rect.bottom(),
                                rect.right(),
                                rect.top());
            else
              rect = IRectangle(rect.left(),
                                rect.bottom(),
                                rect.left()+adjustSize.width(),
                                rect.top());
          }
          if ( adjustSize.height() > rect.height() )
          {
#ifdef IC_ORIGINLOWERLEFT
            if (adjustRect.bottom() != 0 )
#else
            // layoutAdjustment assumes a bottom-left coordinate system.
            if ( adjustRect.maxY() != 0 )
#endif
              rect = IRectangle(rect.left(),
                                rect.top()-adjustSize.height(),
                                rect.right(),
                                rect.top());
            else
              rect = IRectangle(rect.left(),
                                rect.bottom(),
                                rect.right(),
                                rect.bottom()+adjustSize.height());
          }
        }

        ITRACE_DEVELOP(IString("-->Window size = (") + rect.asString());

        wposbuf.moveSizeTo(pCell->window(), rect);
     } /* endif InUse */
  }

  wposbuf.apply();      // Reposition/size child windows.

  return *this;
}


#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
   IMultiCellCanvas :: drawGridLines
   Draw grid lines around all child windows based on the row and column headers.
------------------------------------------------------------------------------*/
IMultiCellCanvas& IMultiCellCanvas :: drawGridLines()
{
  IMODTRACE_DEVELOP("McellCv::drawGridLines");
  if ( hasGridLines() )
  {
    IPresSpaceHandle hps = presSpace();
    ISize sizeCanvas = size();
    unsigned long canvasWidth = sizeCanvas.width();
    IPoint topLeftOffset(topLeftLayoutOffset() ),
           botRightOffset(bottomRightLayoutOffset() );

#ifdef IC_ORIGINUPPERLEFT
    IRootGrafPort
       nativeRootPort( hps );
    GCoordinate lineTop = topLeftOffset.y(),
                lineBottom = sizeCanvas.height() - botRightOffset.y();
#else
    IExtendedRootGrafPort
       nativeRootPort( hps, kRightHand );
    GCoordinate lineTop = sizeCanvas.height() - topLeftOffset.y(),
                lineBottom = botRightOffset.y();
#endif
    IGrafBundle gb ( IBaseColor::kBlack, IGrafBundle::kFill);
    ILinkedGrafPort drawPort ( &nativeRootPort, &gb );

    // Draw vertical lines first.
    GCoordinate xPos = 0;
    unsigned long i;
    bool leftToRight = true;  // Orientation of grid lines.
    if ( IBidiSettings::isBidiSupported() )
    {  // Orientation of gridlines must match layout.
       IBidiSettings
         bidiSettings( *this );
       if ( bidiSettings.windowLayout() == IBidiSettings::layoutRightToLeft )
       {  // Layout based on right edge of canvas.
          // Draw gridlines from right side of canvas to left.
          leftToRight = false;
       }
    }
    for ( i = 1; i <= pRowColHeaderList->columns(); i++ )
    {
      IRCHeader& colHeader = (*pRowColHeaderList)(i, 0);
      xPos = colHeader.offset() + colHeader.width();
      if ( ! leftToRight )
      {
         xPos = canvasWidth - xPos;
      }
      drawPort.draw( IGLine2D( IGPoint2D( xPos, lineTop ),
                               IGPoint2D( xPos, lineBottom ) ) );
    }

    // Draw horizontal lines next.
    GCoordinate lineLeft = topLeftOffset.x(),
                lineRight = canvasWidth - botRightOffset.y(),
                yPos = 0;

    for ( i = 1; i <= pRowColHeaderList->rows(); i++ )
    {
      IRCHeader& rowHeader = (*pRowColHeaderList)(0, i);
#ifdef IC_ORIGINLOWERLEFT
      yPos = rowHeader.offset() - 1;
#else
      yPos = rowHeader.offset() + rowHeader.height();
#endif
      drawPort.draw( IGLine2D( IGPoint2D( lineLeft, yPos ),
                               IGPoint2D( lineRight, yPos ) ) );
    }

    releasePresSpace( hps );
  }
  return *this;
}
#endif // IC_MOTIF


/*------------------------------------------------------------------------------
| IMultiCellCanvasList functions                                               |
------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
   Constructor
------------------------------------------------------------------------------*/
IMultiCellCanvasList::IMultiCellCanvasList ( )
  : IVPtrKeySet<IMCCvCell*, unsigned long>()
{ }

/*------------------------------------------------------------------------------
   Destructor - So the compiler doesn't generate one.
------------------------------------------------------------------------------*/
IMultiCellCanvasList::~IMultiCellCanvasList ( )
{ }

/*------------------------------------------------------------------------------
   operator - return the cell at the specified column and row
------------------------------------------------------------------------------*/
IMCCvCell& IMultiCellCanvasList::operator ( ) ( unsigned long column,
                                                unsigned long row )
{
  IMCCvCell* pCell = 0;
  IMultiCellCanvasList::Cursor cursor( *this );

  bool fFound = this->locateElementWithKey( CellFromCoords( column, row ),
                                            cursor);
  if ( fFound )
  {
     pCell = this->elementAt( cursor );
     return *pCell;
  }
  return IMCCvCell::nullCell();
}

/*------------------------------------------------------------------------------
   operator - Return the cell that contains the specified window.
------------------------------------------------------------------------------*/
IMCCvCell& IMultiCellCanvasList::operator ( ) ( const IWindow* childWindow )
{
  IMultiCellCanvasList::Cursor cursor( *this );
  IMCCvCell* pCell;
  bool fFound = false;
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
     pCell = this->elementAt( cursor );
     if ( pCell->window() == childWindow )
     {
        fFound = true;
        break;
     }
  } /* endfor */

  if ( fFound )
  {
     pCell = this->elementAt( cursor );
     return *pCell;
  }
  return IMCCvCell::nullCell();
}


/*------------------------------------------------------------------------------
| IRowColumnHeaderList functions                                               |
------------------------------------------------------------------------------*/
/*------------------------------------------------------------------------------
   Constructor
------------------------------------------------------------------------------*/
IRowColumnHeaderList::IRowColumnHeaderList ( )
  : IVPtrKeySet<IRCHeader*, unsigned long>(),
    ulColumnCount( 0 ), ulRowCount( 0 )
{ }

/*------------------------------------------------------------------------------
   Destructor - So the compiler doesn't generate one.
------------------------------------------------------------------------------*/
IRowColumnHeaderList::~IRowColumnHeaderList ( )
{ }

/*------------------------------------------------------------------------------
   operator - Return the header at the specified column and row.
------------------------------------------------------------------------------*/
IRCHeader& IRowColumnHeaderList::operator ( ) ( unsigned long column,
                                                unsigned long row )
{
  IRCHeader* pHeader = 0;
  IRowColumnHeaderList::Cursor cursor( *this );

  bool fFound = this->locateElementWithKey( CellFromCoords( column, row ),
                                            cursor);
  if ( fFound )
  {
     pHeader = this->elementAt( cursor );
     return *pHeader;
  }
  return IRCHeader::nullCell();
}

/*------------------------------------------------------------------------------
  IRowColumnHeaderList :: updateHeaders
  Make sure header records exist for the necessary columns.  Also add a
  pointer to the null record for any empty cells.
------------------------------------------------------------------------------*/
IRowColumnHeaderList&
  IRowColumnHeaderList::updateHeaders( unsigned long column,
                                       unsigned long row )
{
  unsigned long aKey;
  IRowColumnHeaderList::Cursor cursor( *this );

  for ( int c = ulColumnCount; c <= column; c++ )
  {  /* See if the row/column header records exists. */
     aKey = CellFromCoords( c, 0 );
     bool fFound = this->locateElementWithKey( aKey, cursor );
     if ( ! fFound )
     {
        IRCHeader* pHeader = new IRCHeader( c, 0 );
        this->addOrReplaceElementWithKey( pHeader );
     }
  }     // endfor

  for ( int r = ulRowCount; r <= row; r++ )
  {
     /* See if the row/column header records exists  */
     aKey = CellFromCoords( 0, r );
     bool fFound = this->locateElementWithKey( aKey, cursor );
     if ( ! fFound )
     {
        IRCHeader* pHeader = new IRCHeader( 0, r );
        this->addOrReplaceElementWithKey( pHeader );
     }
  }     // endfor

  if ( column > ulColumnCount )
     ulColumnCount = column;
  if ( row > ulRowCount )
     ulRowCount = row;

  return *this;
}


/*------------------------------------------------------------------------------
| IRCHeader functions                                                          |
------------------------------------------------------------------------------*/
IRCHeader IRCHeader::fgNullCell(0,0);

/*------------------------------------------------------------------------------
   Destructor - So the compiler doesn't generate one.
------------------------------------------------------------------------------*/
IRCHeader::~IRCHeader ( )
{ }

/*------------------------------------------------------------------------------
   IRCHeader :: nullCell
   Return a null cell.
------------------------------------------------------------------------------*/
IRCHeader& IRCHeader::nullCell ( )
{
  return fgNullCell;
}

/*------------------------------------------------------------------------------
   IRCHeader :: asString
   Return selected information about this object.
------------------------------------------------------------------------------*/
IString IRCHeader::asString ( ) const
{
  IString str = IString( "IRCHeader(" ) + IString( this->column() ) +
                IString( "," ) + IString( this->row() ) + IString( ")," );
  return str;
}

/*------------------------------------------------------------------------------
   IRCHeader :: asDebugInfo
   Return debug information about this object.
------------------------------------------------------------------------------*/
IString IRCHeader::asDebugInfo ( ) const
{
  IString str = asString() + IString( " Width/Height=" )
                           + IString( ulWidthHeight )
                           + IString( "," ) + IString( "ReqWidthHeight=" )
                           + IString( ulRequestedWidthHeight )
                           + IString( " Offset=" ) + IString( lOffset )
                           + IString( ", Expand=" )
                           + IString( fFlags & IRCHeader::expandable )
                           + IString( ", Default=" )
                           + IString( fFlags & IRCHeader::usingDefault );
  return str;
}
