// Revision: 10 1.13.1.4 source/ui/cnr/icnrhdr.cpp, container, ioc.v400, 990114 
/*******************************************************************************
* FILE NAME: icnrhdr.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in icnrhdr.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#define INCL_WINSTDCNR
#define INCL_WINWINDOWMGR

extern "C" {
  #include <iwindefs.h>
  #include <stdlib.h>
  #include <icconst.h>
}

#include <icnrrec.hpp>  // Must be first for OS flags

#include <ibase.hpp>
#include <icnrctl.hpp>
#include <icnrcol.hpp>
#include <icnrhdr.hpp>
#include <icnrevt.hpp>
#include <icnrobj.hpp>
#include <ihandle.hpp>
#include <itrace.hpp>
#include <iexcept.hpp>
#include <ihelp.hpp>
#include <iwcname.hpp>
#include <ireslib.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ICNRHDR_CPP_
  #include <ipagetun.h>
#endif


/*------------------------------------------------------------------------------
| ICnrHandler::ICnrHandler                                                     |
|
| Constructor.
------------------------------------------------------------------------------*/
ICnrHandler::ICnrHandler()
{
   IMODTRACE_DEVELOP("CnrHdr::CnrHdr");
}

/*------------------------------------------------------------------------------
| ICnrHandler::~ICnrHandler                                                    |
------------------------------------------------------------------------------*/
ICnrHandler:: ~ICnrHandler()
{
  IMODTRACE_DEVELOP("CnrHdr::~CnrHdr");
}


/*------------------------------------------------------------------------------
| ICnrHandler::dispatchHandlerEvent                                            |
|                                                                              |
| Routes the container events to the the appropriate event functions.          |
| Currently supports the following 3 callbacks that are all that's available   |
| for the Motif 1.2 Container widget:                                          |
|    XmNdefaultActionCallback --> enterEvent                                   |
|    XmNselectionCallback     --> emphasisEvent                                |
|    XmNvalueChangedCallback  --> collapse/expand of tree                      |
|                                                                              |
| Additional work is needed to support the other virtual functions related     |
| to PM specific events such as scrolling, delta, in-use and cursored          |
| emphasis, etc.                                                               |
------------------------------------------------------------------------------*/
bool ICnrHandler::dispatchHandlerEvent(IEvent& evt)
{
   bool fStopDispatch(false) ;
   if (evt.eventId() == WM_CONTROL)
   {                              // Allocate class name string.
      IControlEvent ctlevt(evt);
      IWindow* pwndControl = ctlevt.controlWindow();

      if(pwndControl==0)
        return false;

      if ( pwndControl->handle() )
      {
        IWindowClassName className( pwndControl->handle() );
        unsigned long ulMsg = evt.parameter1().number2();

#ifdef IC_WIN
        if (className == WC_CONTAINER || className == WC_NATIVECONTAINER)
#endif
#ifdef IC_PM
        if (className == WC_CONTAINER)
#endif
        {                        // container
          switch(ulMsg)
          {
            case CN_QUERYDELTA:
            {
              ICnrQueryDeltaEvent cqdevt(ctlevt);
              fStopDispatch = deltaReached(cqdevt);
              evt.setResult(cqdevt.result());
              break;
            }
            case CN_ENTER:
            {
              ICnrEnterEvent cenevt(ctlevt);
              fStopDispatch = enter(cenevt);
              evt.setResult(cenevt.result());
              break;
            }
            case CN_EMPHASIS:
            {
              ICnrEmphasisEvent cemevt(ctlevt);
              if(cemevt.changed(IContainerObject::cursored))
                  cursoredChanged(cemevt);
              if(cemevt.changed(IContainerObject::inuse))
                  inuseChanged(cemevt);
              if(cemevt.changed(IContainerObject::selected))
                  selectedChanged(cemevt);
              evt.setResult(cemevt.result());
              break;
            }
            case CN_COLLAPSETREE:
            {
              IContainerObject* pcnrobj =
                       IObjFromRec((IMiniCnrRecord*)(void*)evt.parameter2());
              return treeCollapsed(pcnrobj,
                                   (IContainerControl*)ctlevt.controlWindow());
            }
            case CN_EXPANDTREE:
            {
              IContainerObject* pcnrobj =
                       IObjFromRec((IMiniCnrRecord*)(void*)evt.parameter2());
              return treeExpanded(pcnrobj,
                                   (IContainerControl*)ctlevt.controlWindow());
            }
            case CN_HELP:
            {
              ICnrHelpEvent chelpevt(ctlevt);
              fStopDispatch = help(chelpevt);
              evt.setResult(chelpevt.result());
              break;
            }
            case CN_SCROLL:
            {
              ICnrScrollEvent cscrevt(ctlevt);
              fStopDispatch = windowScrolled(cscrevt);
              evt.setResult(cscrevt.result());
              break;
            }
          } /* end switch ulMsg */
        } /* endif */
      } /* endif */
   } /* endif */

return fStopDispatch;
}

/*------------------------------------------------------------------------------
| ICnrHandler::deltaReached                                                    |
|                                                                              |
| Provide default behavior.  deltaReached is called when the container is      |
| scrolled to the predefined area.                                             |
------------------------------------------------------------------------------*/
bool ICnrHandler :: deltaReached(ICnrQueryDeltaEvent& evt)
{
  ITRACE_MOTIF_NOP();
  return false;
}


/*------------------------------------------------------------------------------
| ICnrHandler::treeCollapsed                                                   |
|                                                                              |
| Called when an object is being collapsed in the container.                   |
------------------------------------------------------------------------------*/
bool ICnrHandler :: treeCollapsed(IContainerObject* pcnrobj,
                                     IContainerControl* pcnrctl)
{
   pcnrobj->handleTreeCollapse(pcnrctl);
   return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::treeExpanded                                                    |
|                                                                              |
| Called when an object is being expanded in the container.                    |
------------------------------------------------------------------------------*/
bool ICnrHandler :: treeExpanded(IContainerObject* pcnrobj,
                                     IContainerControl* pcnrctl)
{
   pcnrobj->handleTreeExpand(pcnrctl);
   return false;
}


/*------------------------------------------------------------------------------
| ICnrHandler::enter                                                           |
|                                                                              |
| Called when the Enter key or the mouse select button is used on an object.   |
| Notes: If the Enter occurs over an object, the default behavior is to call   |
|          the IContainerObject::handleOpen function.                          |
|        Otherwise, the Enter is ignored.                                      |
------------------------------------------------------------------------------*/
bool ICnrHandler :: enter(ICnrEnterEvent& evt)
{
  if(evt.validObject())
     evt.object()->handleOpen((IContainerControl*)evt.controlWindow());
  return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::selectedChanged                                                 |
|                                                                              |
| Provide default behavior for a change in an object's selection emphasis.     |
------------------------------------------------------------------------------*/
bool ICnrHandler :: selectedChanged(ICnrEmphasisEvent& evt)
{
    IContainerControl* pcnrctl = (IContainerControl*)evt.controlWindow();
    bool fAcquired;

#ifdef IC_MOTIF
   fAcquired = evt.isAcquired();
#endif //IC_MOTIF

#ifdef IC_PMWIN
    /******************************************************************/
    /* The WC_CONTAINER sends us a CN_EMPHASIS on adding an object to */
    /* the container and yet will not handle a CM_QUERYRECORDINFO     */
    /* on the record.  As a work-around until this is fixed, we will  */
    /* treat a failure on isSelected as if the object is acquiring    */
    /* selected emphasis for the first time.                          */
    /******************************************************************/
    try
    {
       fAcquired = pcnrctl->isSelected(evt.object());
    }
    catch(IException&)
    {
       fAcquired = true;
    }
#endif //IC_PMWIN
    evt.object()->handleSelectedChange(pcnrctl, fAcquired);
    return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::inuseChanged                                                    |
|                                                                              |
| Provide default behavior for a change in an object's in-use emphasis.        |
------------------------------------------------------------------------------*/
bool ICnrHandler :: inuseChanged(ICnrEmphasisEvent& evt)
{
    IContainerControl* pcnrctl = (IContainerControl*)evt.controlWindow();
    bool fAcquired = pcnrctl->isInUse(evt.object());
    evt.object()->handleInuseChange(pcnrctl, fAcquired);
    return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::cursoredChanged                                                 |
|                                                                              |
| Provide default behavior for a change in an object's cursored emphasis.      |
------------------------------------------------------------------------------*/
bool ICnrHandler :: cursoredChanged(ICnrEmphasisEvent& evt)
{
    bool fAcquired;
    IContainerControl* pcnrctl = (IContainerControl*)evt.controlWindow();

#ifdef IC_MOTIF
    fAcquired = evt.isAcquired() ;
#endif // IC_MOTIF

#ifdef IC_PMWIN
    /********************************************************************/
    /* The WC_CONTAINER sends us a CN_EMPHASIS on adding an object to   */
    /* the container and yet will not handle a CM_QUERYRECORDINFO       */
    /* on the record.  As a work-around until this is fixed, we will    */
    /* treat a failure on isSelected as if the object is acquiring      */
    /* selected emphasis for the first time.                            */
    /********************************************************************/
    try
    {
       fAcquired = pcnrctl->isCursored(evt.object());
    }
    catch(IException&)
    {
       fAcquired = true;
    }
#endif //IC_PMWIN
    evt.object()->handleCursoredChange(pcnrctl, fAcquired);
    return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::windowScrolled                                                  |
|                                                                              |
| Provide default behavior for scrolling a window.  Default behavior is do     |
| nothing.                                                                     |
------------------------------------------------------------------------------*/
bool ICnrHandler :: windowScrolled(ICnrScrollEvent& evt)
{
  return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::help                                                            |
|                                                                              |
| Provide default behavior when help is requested in a container               |
------------------------------------------------------------------------------*/
bool ICnrHandler :: help(ICnrHelpEvent& evt)
{
   IContainerControl* pcnrctl = (IContainerControl*)evt.controlWindow();
   IContainerColumn* pcnrcol = 0;
   IContainerObject* pcnrobj = 0;

   IHelpWindow* pHelp = IHelpWindow::helpWindow(pcnrctl);
   if(pHelp!=0)
   {
      unsigned long ulHelpId = 0;
      if(pcnrctl!=0)
      {
         pcnrcol = pcnrctl->currentEditColumn();
         if (pcnrcol)
            ulHelpId = pcnrcol->helpId();
         else
         {
            pcnrobj = evt.object();
            if(pcnrobj)
               ulHelpId = pcnrobj->helpId();
         }

         // last resort, use container control help id.
         if (ulHelpId==0)
           ulHelpId = pcnrctl->helpId();
      }

      if (ulHelpId)
      {
         pHelp->show(IResourceId(ulHelpId));
         return true;
      } /* endif */

   } /* endif */
   return false;
}

/*------------------------------------------------------------------------------
| ICnrHandler::handleEventsFor                                                 |
|
------------------------------------------------------------------------------*/
IHandler& ICnrHandler :: handleEventsFor( IWindow *window )
{
    ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
    return *this;
}

/*------------------------------------------------------------------------------
| ICnrHandler::handleEventsFor                                                 |
|                                                                              |
| Attaches the handler to the container control passed in the argument.        |
------------------------------------------------------------------------------*/
ICnrHandler& ICnrHandler :: handleEventsFor( IContainerControl *container )
{
  IASSERTPARM(container != 0);
  IHandler::handleEventsFor(container);
  return *this;
}

/*------------------------------------------------------------------------------
| ICnrHandler::stopHandlingEventsFor                                           |
------------------------------------------------------------------------------*/
IHandler& ICnrHandler :: stopHandlingEventsFor( IWindow *window )
{
    ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
    return *this;
}

/*------------------------------------------------------------------------------
| ICnrHandler::stopHandlingEventsFor                                           |
|                                                                              |
| Detaches the handler from the container control passed in the argument.      |
------------------------------------------------------------------------------*/
ICnrHandler& ICnrHandler :: stopHandlingEventsFor( IContainerControl *container)
{
  IASSERTPARM(container != 0);
  IHandler::stopHandlingEventsFor(container);
  return *this;
}
