/*
-------------------------------------------------------------------------------
Copyright (c) IBM Corporation 1997.
All Rights Reserved.

Permission is granted to copy, use, modify, and merge this software into your
applications and to permit others to do any of the foregoing. You must include
this permission and copyright notice in all copies and modified versions of
this software. THIS SOFTWARE IS PROVIDED IN ITS 'AS IS' CONDITION.
IBM DISCLAIMS ANY LIABILITY OF ANY KIND FOR DAMAGES WHATSOEVER RESULTING FROM
THE USE OF THIS SOFTWARE.
-------------------------------------------------------------------------------
*/
// Revision: 13 1.4.2.2 samples/testfw/tmultest.cpp, ocsamples-L1, ioc.v400, 981106 
#include <stdio.h>

#include "tmultest.hpp"
#include "iargdict.hpp"
#include <iseq.h>

//-----------------------------------------------------------------------------
// TMultiplexerTest: PUBLIC Special Member Functions
//-----------------------------------------------------------------------------

IMultiplexerTest::IMultiplexerTest(): ITestMultiplexer()
{
        fNeedToParseInput = true;
}

IMultiplexerTest::IMultiplexerTest(const IMultiplexerTest& source): ITestMultiplexer(source)
{
        fNeedToParseInput = source.fNeedToParseInput;
        fProgrammatic = source.fProgrammatic;
        fTestInputsAsKeys = source.fTestInputsAsKeys;
        for (unsigned short i = 0; i < kNumberOfKeys; i++) fWasCalled[i] = source.fWasCalled[i];
}

IMultiplexerTest& IMultiplexerTest::operator=(const IMultiplexerTest& rhs)
{
        if (this != &rhs) {
                ITestMultiplexer::operator=(rhs);
                fNeedToParseInput = rhs.fNeedToParseInput;
                fProgrammatic = rhs.fProgrammatic;
                fTestInputsAsKeys = rhs.fTestInputsAsKeys;
                for (unsigned short i = 0; i < kNumberOfKeys; i++) fWasCalled[i] = rhs.fWasCalled[i];
        }
        return *this;
}


IMultiplexerTest::~IMultiplexerTest()
{
}


//------------------------------------------------------------------------------
// IMultiplexerTest: PUBLIC TTest Functions
//------------------------------------------------------------------------------

void IMultiplexerTest::copyInfo()
{
        addInfo(kDescriptionKey, IString("A test for the TTesIMultiplexer::Test method."));
        addInfo(kInputSyntaxKey, IString("IMultiplexerTest [-p <keyword> | -i <keyword...>]"));
        addInfo(kTargetClassKey, IString("IMultiplexerTest"));
}

//-----------------------------------------------------------------------------
// IMultiplexerTest: PUBLIC Decision Functions
//-----------------------------------------------------------------------------

bool IMultiplexerTest::appleFunction()
{
        fWasCalled[kApple] = true;
        ITHROW(IException("Testing handleDecision : apple function is called"));
}

bool IMultiplexerTest::bearFunction()
{
        fWasCalled[kBear] = true;
        return true;
}

bool IMultiplexerTest::cartFunction()
{
        fWasCalled[kCart] = true;
        return true;
}

bool IMultiplexerTest::doorFunction()
{
        fWasCalled[kDoor] = true;
        return true;
}

bool IMultiplexerTest::elbowFunction()
{
        fWasCalled[kElbow] = true;
        return true;
}

//------------------------------------------------------------------------------
// IMultiplexerTest: PROTECTED TTest Overrides
//------------------------------------------------------------------------------

void IMultiplexerTest::setup()
{
        bool everythingOkay = true;

        if (fNeedToParseInput) {
                bool validInput = parseInput(); // Returns false if input bad
                if (validInput) fNeedToParseInput = false;
                else {
                        setSuccess(false);
                        everythingOkay = false;
                }
        }
        if (everythingOkay) {
                for (short i = 0; i < kNumberOfKeys; i++) fWasCalled[i] = false;
        }

}

void IMultiplexerTest::cleanup()
/*
        Cleanup interprets the results to see if the correct decision functions
        were called.
*/
{
        if (!fNeedToParseInput) { // fNeedToParseInput = input was not parsed successfully
                for (short i=0; i<kNumberOfKeys; i++) {
                        if (fWasCalled[i] && !fExpectCall[i]) {
                                outputTextStream() << "IMultiplexerTest - Got unexpected call of "
                                        << keyword(i) << ".\n";
                                setSuccess(false);
                        }
                        else if (!fWasCalled[i] && fExpectCall[i]) {
                                outputTextStream() << "IMultiplexerTest - Didn't get expected call of "
                                        << keyword(i) << ".\n";
                                setSuccess(false);
                        }
                }
                if (isSuccessUndecided()) setSuccess(true);
        }
}

//-----------------------------------------------------------------------------
// IMultiplexerTest: PROTECTED TTesIMultiplexer Overrides
//-----------------------------------------------------------------------------

void IMultiplexerTest::loadDecisions()
{
/*
for i in Apple Bear Cart Door Elbow
echo "  AddDecision(s, GetKeyword(k{i}), (TTestDecisionFn)&IMultiplexerTest::{i}Function);"
end
*/
        ITestMultiplexer::loadDecisions();
        addDecision(keyword(kApple),  (ITestDecisionFn)appleFunction);
        addDecision(keyword(kBear),  ITestDecisionFn(&IMultiplexerTest::bearFunction));
        addDecision(keyword(kCart),  ITestDecisionFn(&IMultiplexerTest::cartFunction));
        addDecision(keyword(kDoor),  ITestDecisionFn(&IMultiplexerTest::doorFunction));
        addDecision(keyword(kElbow), ITestDecisionFn(&IMultiplexerTest::elbowFunction));
}

bool IMultiplexerTest::handleDecision(ITestDecisionFn fun, const IString& key)
{
        outputTextStream().setIndent(outputTextStream().indent() + 1);
        bool result;
        try {
                result = (this->*fun)();
        }
        catch (const IException& ie){
                outputTextStream() << ie.text() <<"\n";
                result = true;
        }

        outputTextStream().setIndent(outputTextStream().indent() - (unsigned short)1);
        return result;
}

//-----------------------------------------------------------------------------
// IMultiplexerTest: PRIVATE Methods
//-----------------------------------------------------------------------------

bool IMultiplexerTest::parseInput()
/*
        ParseInput interprets the TTest input arguments. Usage is:

                (-p <keyword> | -i <keyword>) <remainder of line ignored>

        The -p and -i options are exclusive. If -p is given, then only one keyword
        can be specified, and the test will set it programatically by calling
        SetDecisionKey. If the -i option is given, then all inputs will be
        used as keywords. If neither -p nor -i is given, then no special keyword
        is set, so the default of all decisions should be called.

        ParseInput sets data members:

                fProgrammatic: false initially, set to true on -p option
                fTestInputsAsKeys: 0 initially, set to # of keys on -i option
                fExpectCall: Set to true for only those functions which should be called

        ParseInput also calls SetDecisionKey or SetInputsAsKeys as appropriate.
*/
{
        //static
         const IString kMinusP("-p"); // -programmatic
        //static
         const IString kMinusI("-i"); // -inputs

        //static
         const IString kNakedOptions("-p -i");

        bool goodParse = true, allDecisions = false, minusI = false;
        ISequence<IString> keysForMinusI; // TTexts to be used as inputs

        IArgumentDictionary arguments(*this, kNakedOptions);

        short howManyKeys = 0, i;

        outputTextStream().pushTier(ITieredTextBuffer::kHeadline);
        fProgrammatic = false;
        fTestInputsAsKeys = 0;
        for (i=0; i<kNumberOfKeys; i++) fExpectCall[i] = false;

        // Look for kMinusP & kMinusI arguments
        IString value;
        if (arguments.textAt(kMinusP, value))
                 fProgrammatic = true;
        if (arguments.textAt(kMinusI, value)) minusI = true;
        if (fProgrammatic && minusI) {
                outputTextStream() << "IMultiplexerTest - Do not supply both -p and -i options.\n";
                goodParse = false;
        }

        // If neither kMinusP nor kMinusI is given, then call all decisions
        if (!fProgrammatic && !minusI) allDecisions = true;
        else {
                // Scan for input keywords for kMinusP or kMinusI
                for (howManyKeys=1; goodParse; howManyKeys++) {
                        short keyIndex;
                        IString key;
                        if (!arguments.nthValue(howManyKeys, key)) {
                                if (howManyKeys == 1) {
                                        outputTextStream() <<
                                                "IMultiplexerTest - No keywords given.\n";
                                        goodParse = false;
                                }
                                break; // No more keys
                        }
                        else {
                                bool validKey, allDecisionsKey;

                                if (fProgrammatic && howManyKeys > 1) {
                                        outputTextStream() <<
                                                "IMultiplexerTest - Only one programmatic key allowed.\n";
                                        goodParse = false;
                                        break;
                                }

                                if (key == ITestMultiplexer::kAllDecisions)
                                        allDecisionsKey = allDecisions = true; // Local & global vars
                                else
                                        validKey = textToKeywordIndex(key, keyIndex);

                                if (validKey || allDecisionsKey) {
                                        if (fProgrammatic) {
                                                fProgrammaticKeyword = key;
                                        setDecisionKey(fProgrammaticKeyword);

                                        }
                                        else keysForMinusI.addAsLast(key);

                                        if (validKey) fExpectCall[keyIndex] = true;
                                }
                                else {
                                        outputTextStream() <<
                                                "IMultiplexerTest - Bad keyword \"" << key << "\".\n";
                                        goodParse = false;
                                        break;
                                }
                        }
                }
        }

        // For kMinusI copy the keywords in as the test inputs for TTestMultiplexer
        if (goodParse && minusI) {
                setInputs(keysForMinusI);
                fTestInputsAsKeys = howManyKeys - 1;
                setInputsAsKeys(fTestInputsAsKeys);
        }

        // Handle all decisions case
        if (allDecisions) for (i=0; i<kNumberOfKeys; i++) fExpectCall[i] = true;

        outputTextStream().popTier();
        return goodParse;
}

bool IMultiplexerTest::textToKeywordIndex(const IString& theText, short& theIndex) const
/*
        TextToKeywordIndex attempt to map theText to a keyword. If successful it
        puts the result in theIndex and returns true. Otherwise it leaves
        theIndex untouched and returns false.
*/
{
        for (short i=0; i < kNumberOfKeys; i++) {
                if (theText == keyword(i)) {
                        theIndex = i;
                        return true;
                }
        }
        return false;
}


const IString& IMultiplexerTest::keyword(const short keyNumber) const
{
        static  IString fgKeyword[kNumberOfKeys];

        if (fgKeyword[0].length() == 0) {
                fgKeyword[kApple] = IString("Apple");
                fgKeyword[kBear] = IString("Bear");
                fgKeyword[kCart] = IString("Cart");
                fgKeyword[kDoor] = IString("Door");
                fgKeyword[kElbow] = IString("Elbow");
        }
        return fgKeyword[keyNumber];
}


//eof

