/*
-------------------------------------------------------------------------------
Copyright (c) IBM Corporation 1997.
All Rights Reserved.

Permission is granted to copy, use, modify, and merge this software into your
applications and to permit others to do any of the foregoing. You must include
this permission and copyright notice in all copies and modified versions of
this software. THIS SOFTWARE IS PROVIDED IN ITS 'AS IS' CONDITION.
IBM DISCLAIMS ANY LIABILITY OF ANY KIND FOR DAMAGES WHATSOEVER RESULTING FROM
THE USE OF THIS SOFTWARE.
-------------------------------------------------------------------------------
*/
// Revision: 25 1.5.2.2 samples/testfw/ttestrun.cpp, ocsamples-L1, ioc.v400, 981106 
#include "ttestrun.hpp"
#include "stgexcep.hpp"
#include "iargdict.hpp"

//------------------------------------------------------------------------------
// ITestTestRun: PUBLIC Special Member Functions
//------------------------------------------------------------------------------

ITestTestRun::ITestTestRun(): ITest()
{
        fExceptionStage = ITest::kNone;
        fNextExpectedStage = ITest::kSetup;
        fNeedToParseInput = true;
        fLocalSuccess = true;
}

ITestTestRun::ITestTestRun(const ITestTestRun& source): ITest(source)
{
        fExceptionStage = source.fExceptionStage;
        fNextExpectedStage = source.fNextExpectedStage;
        fNeedToParseInput = source.fNeedToParseInput;
        fLocalSuccess = source.fLocalSuccess;
}

ITestTestRun& ITestTestRun::operator=(const ITestTestRun& rhs)
{
        if (this != &rhs) {
                ITest::operator=(rhs);
                fExceptionStage = rhs.fExceptionStage;
                fNextExpectedStage = rhs.fNextExpectedStage;
                fNeedToParseInput = rhs.fNeedToParseInput;
                fLocalSuccess = rhs.fLocalSuccess;
        }
        return *this;
}

ITestTestRun::~ITestTestRun()
{
}

//------------------------------------------------------------------------------
// ITestTestRun: PUBLIC ITest Overrides
//------------------------------------------------------------------------------

void ITestTestRun::run()
/*
        run overrides the ITest::run method to do the following: 1. Parse the input
        arguments. 2. Preset the variables for the run. These steps cannot be done
        in setup because setup is one of the methods being tested, and potentially
        may never get called (if ITest::run is broken) and also because we need
        everything to be ready in reset, which is called before setup.
*/
{
        bool everythingOkay = true;

        fNextExpectedStage = ITest::kSetup;
        if (runCount() > 0) fNextExpectedStage = ITest::kReset;
        if (fNeedToParseInput) {
                everythingOkay = parseInput(); // Returns false if input bad
                if (everythingOkay) fNeedToParseInput = false;
                else {
                        ITest::reset();
                        setSuccess(false);
                }
        }
        if (everythingOkay) ITest::run();
}

//------------------------------------------------------------------------------
// ITestTestRun: PROTECTED ITest Overrides
//------------------------------------------------------------------------------

void ITestTestRun::reset()
{
        if (fLocalSuccess) setSuccess(true);
        if (fNextExpectedStage != ITest::kReset) {
                outputTextStream() << "ITestTestRun: reset method run out of order.\n";
                setSuccess(false);      fLocalSuccess = false;
        }
        if (fExceptionStage == ITest::kReset) {
                fNextExpectedStage = ITest::kCleanup;
                ITHROW(IException("exception in reset()"));
        }
        fNextExpectedStage = ITest::kSetup;
}

void ITestTestRun::setup()
{
        if (fLocalSuccess) setSuccess(true);
        if (fNextExpectedStage != ITest::kSetup) {
                outputTextStream() << "ITestTestRun: setup method run out of order.\n";
                setSuccess(false);      fLocalSuccess = false;
        }
        if (fExceptionStage == ITest::kSetup) {
                fNextExpectedStage = ITest::kCleanup;
                ITHROW(IException("exception in setup()"));
        }
        fNextExpectedStage = ITest::kTest;
}

void ITestTestRun::test()
{
        if (fLocalSuccess) setSuccess(true);
        if (fNextExpectedStage != ITest::kTest) {
                outputTextStream() << "ITestTestRun: test method run out of order.\n";
                setSuccess(false);      fLocalSuccess = false;
        }
        if (fExceptionStage == ITest::kTest) {
                fNextExpectedStage = ITest::kCleanup;
                ITHROW(IException("exception in test()"));
        }
        fNextExpectedStage = ITest::kCleanup;
}

void ITestTestRun::cleanup()
{
        // Check for exceptions in previous stages and set success accordingly
        ITest::EStage expectedExceptionStage = fExceptionStage;

        // Handle the special case of an exception to be thrown in the Reset
        // stage during the first run (when Reset is not called).
        if (runCount() == 1 && fExceptionStage == ITest::kReset)
                expectedExceptionStage = ITest::kNone;

        ITest::EStage aExceptionStage = exceptionStage();
        if (aExceptionStage != expectedExceptionStage) {
                outputTextStream() << "ITestTestRun: Exception reported to be in wrong stage.\n";
                setSuccess(false);      fLocalSuccess = false;
        }

        // If an exception was thrown correctly, then we should mark this test
        // as successful. We first check fLocalSuccess to maintain the previous
        // failure if there was one. We cannot use ITest::GetSuccess because
        // throwing an exception triggers an automatic failure.
        if (fLocalSuccess) setSuccess(true);

        if (fNextExpectedStage != ITest::kCleanup) {
                outputTextStream() << "ITestTestRun: Cleanup method run out of order.\n";
                setSuccess(false);      fLocalSuccess = false;
        }
        fNextExpectedStage = ITest::kNone;
}

//------------------------------------------------------------------------------
// ITestTestRun: PRIVATE Methods
//------------------------------------------------------------------------------

bool ITestTestRun::parseInput()
/*
        parseInput accepts:

                [-exception reset | test | setup]

        It sets up <fExceptionStage>, and returns true if the input is parsed well.
*/
{
        static const IString kExceptionKey("-exception"); // -exception <stage>
        static const IString kReset("reset");
        static const IString kTest("test");
        static const IString kSetup("setup");

        bool validInput = true;
        IArgumentDictionary args(*this);
        IString stage;
        if (args.textAt(kExceptionKey, stage)){
                if (stage == kReset) fExceptionStage = ITest::kReset;
                else if (stage == kTest) fExceptionStage = ITest::kTest;
                else if (stage == kSetup) fExceptionStage = ITest::kSetup;
                else {
                        validInput = false;
                        outputTextStream() << "ITestTestRun: Unknown stage \"" << stage << "\".\n";
                }
        }
        return validInput;
}

//eof

