/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/
// Revision: 41 1.36.1.17 source/albert/graph2d/igrport.cpp, 2d, ioc.v400, 990114 
/*================
||
||  File:   GrafPort.c
||
||  What:   Definitions for the Albert IGrafPort
||
||  Change History:
||  18 Apr 96   tt      First version in OpenClass
||
*/

#include <igrport.hpp>
#include <grstate.hpp>
#include <grafdevs.hpp>
#include <iexgrprt.hpp>
#include <irect.hpp>
#include <itrace.hpp>

#if defined(IC_WIN)
#include <gdidev.hpp>
#elif defined(IC_PM)
#include <gpidev.hpp>
#elif defined(IC_MOTIF)
#include <xdevice.hpp>
#endif

// Segment definitions
#ifdef IC_PAGETUNE
#define _IGRPORT_CPP_
#include <ipagetun.h>
#endif

class IGLine2D;
class IGPolyline2D;
class IGCurve2D;
class IGRect2D;
class IGEllipse2D;
class IGPolygon2D;
class IGLoop2D;
class IGArea;
class IGImage;
class IGTextRun;

//================
// IGrafPortContext: This is replacement for IGrafStateAccessor
// use it on the stack to be exception safe.  It will semaphore IGrafPort::useState
// and IGrafPort::doneWithState a thread safe manner

#pragma pack(push,4)

extern bool fLowerLeftCoordOriginHack;

class IGrafPortContext
{
  public:
    IGrafPortContext(const IGrafPort* grafPort);
    ~IGrafPortContext();
  private:
    IGrafPortContext() {}
    const IGrafPort* fGrafPort;
};

#pragma pack(pop)

inline IGrafPortContext::IGrafPortContext(const IGrafPort* grafPort)
{
    fGrafPort = grafPort;
    grafPort->useState();
}

inline IGrafPortContext::~IGrafPortContext()
{
    fGrafPort->doneWithState();
}

#pragma pack(push,4)

// inlined helper class for saving the orphaned chilren states like growable array
class IGrafStateOrphanage {
  public:
	IGrafStateOrphanage();
	~IGrafStateOrphanage();

	IGrafState* operator[](unsigned long i);
	void adopt(IGrafState* obj);

  private:
	IGrafState** fState;
	unsigned long fCount;
	unsigned long fLength;
};

#pragma pack(pop)

inline IGrafStateOrphanage::IGrafStateOrphanage() :
	fCount(0), fLength(0), fState(0 /*NIL*/)
{}

inline IGrafStateOrphanage::~IGrafStateOrphanage() {
	if (fCount > 0) {	// delete all the adopted orphan states...
		while (fCount--)
			delete fState[fCount];
		free(fState);
	}
}

inline IGrafState* IGrafStateOrphanage::operator[](unsigned long i) {
	return fState[i];
}

inline void IGrafStateOrphanage::adopt(IGrafState* obj) {
	unsigned long newSize = fCount + 1;
	if (fLength < newSize) { // need to grow...
		fLength = (newSize + 3) >> 2 << 2;	// increments of 4
		newSize = sizeof(obj) * fLength;
		if (fCount == 0)
			fState = (IGrafState**)malloc(newSize);
		else
			fState = (IGrafState**)realloc((void*)fState, newSize);
	}
	fState[fCount++] = obj;
}


//================
// IGrafPort routines
//================

#ifdef IC_MOTIF
bool IGrafPort::restoreGC = true;
#endif

/*================
||
||  IGrafPort destructor
||
*/

IGrafPort::~IGrafPort()
{
	// clean up the orphaned child grafstates, if any...
	delete fGrafStateOrphanage;
	if(fColormapContext){
		delete fColormapContext;
		fColormapContext = 0;
	}
}


/*================
||
||  IGrafPort draw functions
||
*/

void IGrafPort::draw(const IGLine2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderLine(geometry, *state());
}

void IGrafPort::draw(const IGPolyline2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderPolyline(geometry, *state());
}

void IGrafPort::draw(const IGCurve2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderCurve(geometry, *state());
}

void IGrafPort::draw(const IGRect2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderRect(geometry, *state());
}

void IGrafPort::draw(const IGEllipse2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderEllipse(geometry, *state());
}

void IGrafPort::draw(const IGPolygon2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderPolygon(geometry, *state());
}

void IGrafPort::draw(const IGLoop2D& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderLoop(geometry, *state());
}

void IGrafPort::draw(const IGArea& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderArea(geometry, *state());
}

void IGrafPort::draw(const IGImage& geometry)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderImage(geometry, *state());
}

void IGrafPort::draw(const IGImage& geometry, const IGImage& mask)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderMaskedImage(geometry, mask, *state());
}

void IGrafPort::draw(const IGImage& image, const IGRect2D& source)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderImage(image, source, *state());
}

void IGrafPort::draw(const IGImage& image, const IGImage& mask, const IGRect2D& source)
{
    IFUNCTRACE_DEVELOP();
    IGrafPortContext tempContext(this);
    if (fDevice->isValid())
    	fDevice->renderMaskedImage(image, mask, source, *state());
}

IGPoint2D IGrafPort::draw(const IGTextRun& geometry)
{
    IFUNCTRACE_DEVELOP();
	IGrafPortContext tempContext(this);
	if (fDevice->isValid())
		return fDevice->renderTextRun(geometry, *state());
	else
		return IGPoint2D::origin();
}


/*================
||
||  IGrafPort draw functions with overriding attribute state & model matrix
||
*/

void IGrafPort::draw(
    const IGLine2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
    {
        bundlePort.draw(geometry);
    }
    else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGPolyline2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGCurve2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGRect2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGEllipse2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGPolygon2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGLoop2D& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGArea& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGImage& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(geometry);
    }
}

void IGrafPort::draw(
    const IGImage& image,
    const IGImage& mask,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(image, mask);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(image, mask);
    }
}

void IGrafPort::draw(
    const IGImage& image,
    const IGRect2D& source,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(image, source);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(image, source);
    }
}

void IGrafPort::draw(
    const IGImage& image,
    const IGImage& mask,
    const IGRect2D& source,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        bundlePort.draw(image, mask, source);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        port.draw(image, mask, source);
    }
}

IGPoint2D IGrafPort::draw(
	const IGTextRun& geometry,
    const IAttributeState& attributeState,
    const IGrafMatrix& modelMatrix)
{
    IFUNCTRACE_DEVELOP();
    ILinkedGrafPort bundlePort(this, &attributeState);
    if (modelMatrix.isIdentity())
	{
        return bundlePort.draw(geometry);
	}
	else
    {
        ILinkedGrafPort port(&bundlePort, kModelMatrix, &modelMatrix);
        return port.draw(geometry);
    }
}


/*================
||
||  IGrafPort substate inquiries
||
*/

const IGrafMatrix* IGrafPort::matrix(EMatrixKind kind) const
{
    IFUNCTRACE_DEVELOP();
	if (kind == kModelMatrix)
		return fGrafState->modelMatrix();
	else
		return fGrafState->viewMatrix();
}

const IGArea* IGrafPort::clipArea() const
{
    IFUNCTRACE_DEVELOP();
	return fGrafState->clipArea();
}

const IAttributeState* IGrafPort::attributes() const
{
    IFUNCTRACE_DEVELOP();
	return fGrafState->attributeState();
}

/*================
||
||  IGrafPort device transfer onwership:
||
*/

IPresSpaceHandle IGrafPort::orphanDeviceContext()
{
    IFUNCTRACE_DEVELOP();
	return fDevice->orphanDeviceContext();
}

void IGrafPort::adoptDeviceContext(IPresSpaceHandle deviceContext)
{
    IFUNCTRACE_DEVELOP();
#ifndef IC_MOTIF
	IColorMap* colmap = 0;
	if(fColormapContext)
		colmap = &(fColormapContext->colormap());
#endif //IC_MOTIF
	fDevice->adoptDeviceContext(deviceContext);
#ifndef IC_MOTIF
	if(fColormapContext){
		delete fColormapContext;
		fColormapContext = new IColorMapContext(deviceContext, colmap);
	}
#endif //IC_MOTIF
}


/*================
||
||  IGrafPort device-type inquiry
||
*/

const IString* IGrafPort::portType() const
{
    IFUNCTRACE_DEVELOP();
#if IC_RTTI_ENABLED
	return &fDeviceType;
#else
	return fDevice->deviceType();
#endif // IC_RTTI_ENABLED
}

const IGrafPort* IGrafPort::portForStateUsage() const
{
    IFUNCTRACE_DEVELOP();
	return this;
}

const IGrafMatrix& IGrafPort::invertedDeviceTransform() const
{
    IFUNCTRACE_DEVELOP();
	return IGrafMatrix::identity();
}


const IGrafMatrix& IGrafPort::pixelToPointSize() const
{
    IFUNCTRACE_DEVELOP();
	return IGrafMatrix::identity();
}

/*================
||
||  IGrafPort constructors
||
*/

IGrafPort::IGrafPort(IGrafDevice* referencedDevice, IColorMap *colormap) :
	fDevice(referencedDevice),
	fGrafStateOrphanage(0 /*NIL*/),
	fGrafState(0 /*NIL*/), // is always set later by the subclass...
	fColormapContext(0)
{
    IFUNCTRACE_DEVELOP();
#if IC_RTTI_ENABLED
	if (fDevice)
		fDeviceType = IString(typeid(*fDevice).name());
#endif // IC_RTTI_ENABLED
#ifndef IC_MOTIF
        if(colormap && IColorMap::hasColorMapSupport())
		fColormapContext = new IColorMapContext(fDevice->deviceContext(), colormap);
#endif //IC_MOTIF
}

// following three functions are disabled at the API:
IGrafPort::IGrafPort() :
	fDevice(0 /*NIL*/),
	fGrafStateOrphanage(0 /*NIL*/),
	fGrafState(0 /*NIL*/),
	fColormapContext(0)
{
    IFUNCTRACE_DEVELOP();
}

IGrafPort::IGrafPort(const IGrafPort& source) :
	fDevice(0 /*NIL*/),
	fGrafStateOrphanage(0 /*NIL*/),
	fGrafState(0 /*NIL*/)
{
    IFUNCTRACE_DEVELOP();
}

IGrafPort& IGrafPort::operator=(const IGrafPort& source)
{
    IFUNCTRACE_DEVELOP();
    return *this;
}



/*================
||
||  IGrafPort getters and setters (used only by the IBaseRootGrafPort
||
*/

IGrafDevice* IGrafPort::device()
{
    IFUNCTRACE_DEVELOP();
   	return fDevice;
}

const IGrafState* IGrafPort::state() const
{
    IFUNCTRACE_DEVELOP();
    return fGrafState;
}

const IPresSpaceHandle& IGrafPort::deviceContext()
{
    IFUNCTRACE_DEVELOP();
	return fDevice->deviceContext();
}

void IGrafPort::flush()
{
    IFUNCTRACE_DEVELOP();
	fDevice->flush();
}


#ifdef IC_MOTIF

Display* IGrafPort::display() const
{
    IFUNCTRACE_DEVELOP();
    if (fDevice){
        IXDC *xdc = fDevice->deviceContext();
        return (xdc->xDisplay);
    }
    return NULL;
}

Drawable IGrafPort::drawable() const
{
    IFUNCTRACE_DEVELOP();
    if (fDevice){
        IXDC *xdc = fDevice->deviceContext();
        return (xdc->xDrawable);
    }
    return 0;
}

GC IGrafPort::gc() const
{
    IFUNCTRACE_DEVELOP();
    if (fDevice){
        IXDC *xdc = fDevice->deviceContext();
        return (xdc->gc());
    }
    return NULL;
}

#endif //IC_MOTIF

void IGrafPort::setDevice(IGrafDevice* referencedDevice)
{
    IFUNCTRACE_DEVELOP();
    fDevice = referencedDevice;
#if IC_RTTI_ENABLED
    if (fDevice)
        fDeviceType = IString(typeid(*fDevice).name());
#endif // IC_RTTI_ENABLED
}

void IGrafPort::setState(IGrafState* referencedState)
{
    IFUNCTRACE_DEVELOP();
    fGrafState = referencedState;
}

void IGrafPort::adoptOrphanedGrafState(IGrafState* orphan)
{
    IFUNCTRACE_DEVELOP();
	if (!fGrafStateOrphanage)
		fGrafStateOrphanage = new IGrafStateOrphanage;

	fGrafStateOrphanage->adopt(orphan);
}


IGRect2D IGrafPort::convertRect(const IRectangle theRect) const
{
    IFUNCTRACE_DEVELOP();

    IGPoint2D r1, r2;
    IPoint pts[2];

    // get the rectangle in world space
    pts[0].setX(theRect.left());
    pts[0].setY(theRect.bottom());
    pts[1].setX(theRect.right());
    pts[1].setY(theRect.top());

    // ******** THE CONVERSION *****************************	
    r1 = invertedDeviceTransform().transformPoint(IGPoint2D(pts[0]));
    r2 = invertedDeviceTransform().transformPoint(IGPoint2D(pts[1]));

		IGRect2D    theResult(r1, r2);
    theResult.orderPoints();
		
		// IRectangle is inclusive, but IGRect2D is exclusive
    theResult.set(theResult.topLeft(), theResult.bottomRight() + IGPoint2D(1,1));

    return theResult;
}


//================
// IBaseRootGrafPort routines
//================

/*================
||
||  IBaseRootGrafPort constructors
||
*/

IBaseRootGrafPort::~IBaseRootGrafPort()
{
    IFUNCTRACE_DEVELOP();
	delete fInvertedDeviceMatrix;
	delete fPixelToPointSizeMatrix;
	delete fRootGrafState;
	// it owns the device as the RootGrafPort
	delete fDevice;
}

// implementation for disabled default constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IBaseRootGrafPort::IBaseRootGrafPort()
{}

// implementation for disabled copy constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IBaseRootGrafPort::IBaseRootGrafPort(const IBaseRootGrafPort&)
{}

IBaseRootGrafPort::IBaseRootGrafPort(const IPresSpaceHandle& deviceContext, IColorMap* colormap) :
	IGrafPort(IGrafDeviceMaker::makeDevice(deviceContext, IGrafPort::kIBaseRootGrafPort), colormap),
	fRootGrafState(new IRootGrafState()),
	fInvertedDeviceMatrix(0 /*NIL*/),
	fPixelToPointSizeMatrix(0 /*NIL*/)
{
    IFUNCTRACE_DEVELOP();
	setDevice(fDevice);
	setState(fRootGrafState);
	setupCoordinateSystem(kLeftHand);	// default to left hand coordinate system
}

IBaseRootGrafPort::IBaseRootGrafPort(IGrafDevice* adoptDevice,
			IColorMap* colormap,
			ICoordinateSystem::EOrientation orientation) :

	IGrafPort(adoptDevice, colormap),
	fRootGrafState(new IRootGrafState()),
	fInvertedDeviceMatrix(0 /*NIL*/),
	fPixelToPointSizeMatrix(0 /*NIL*/)
{
    IFUNCTRACE_DEVELOP();
	setState(fRootGrafState);
	setupCoordinateSystem(orientation);	// default to left hand coordinate system
}

void IBaseRootGrafPort::setupCoordinateSystem(int coordinateSystem)
{
    IFUNCTRACE_DEVELOP();
	IGrafDevice* dev = device();
	// wBounds is in device coordinate(pixels) when the device is initilised.
	IGRect2D wBounds = worldBounds();
	if (dev)
	{
		if ((int)dev->coordinateSystem() != coordinateSystem)
		{	// if IExtendedRootGrafPort should be removed, EDeviceCoordinateSystem should be in grafdev.hpp...
			IGrafMatrix *coordinateMap;
#ifndef IC_MOTIF
                        coordinateMap = new IGrafMatrix(IGPoint2D(0.0, wBounds.height() - 1));
#else
                        if (::fLowerLeftCoordOriginHack )
                                coordinateMap = new IGrafMatrix(IGPoint2D(0.0, wBounds.height() - 1));
                        else
                                coordinateMap = new IGrafMatrix(IGPoint2D(0.0, wBounds.height()));
#endif
                        coordinateMap->preScaleBy(IGPoint2D(1.0, -1.0));
                        fRootGrafState->setCoordinateFlip(*coordinateMap);
                        coordinateMap->preConcatWith(fDevice->deviceResolution());
                        fRootGrafState->setDefaultViewMatrix(*coordinateMap);
                        dev->setWorldBounds(coordinateMap->untransformBounds(wBounds));
                        delete coordinateMap;

		}
		else
		{
			fRootGrafState->setCoordinateFlip(IGrafMatrix::identity());
			fRootGrafState->setDefaultViewMatrix(dev->deviceResolution());
			// the worldBounds of device should be in world coordinate
			if (!(dev->deviceResolution().isIdentity()))
				dev->setWorldBounds(dev->deviceResolution().untransformBounds(wBounds));
		}
	}
}


/*================
||
||  IBaseRootGrafPort state access
||
*/

void IBaseRootGrafPort::useState() const
{
	// currently it dosn't do anything
}

void IBaseRootGrafPort::doneWithState() const
{
	// currently it dosn't do anything
}

// world bounds
IGRect2D IBaseRootGrafPort::worldBounds() const
{
    IFUNCTRACE_DEVELOP();
	return fDevice->worldBounds();
}

const IGrafMatrix& IBaseRootGrafPort::invertedDeviceTransform() const
{
    IFUNCTRACE_DEVELOP();
	if (fInvertedDeviceMatrix)
		return *fInvertedDeviceMatrix;
	else {
		((IBaseRootGrafPort*)this)->fInvertedDeviceMatrix = new IGrafMatrix(*fRootGrafState->modelMatrix());
		((IGrafMatrix*)fInvertedDeviceMatrix)->invert();
		return *fInvertedDeviceMatrix;
	}
}

#ifdef IC_MOTIF

Display* IBaseRootGrafPort::display() const
{
    IFUNCTRACE_DEVELOP();
    if (fDevice){
        IXDC *xdc = fDevice->deviceContext();
        return (xdc->xDisplay);
    }
    return NULL;
}

Drawable IBaseRootGrafPort::drawable() const
{
    IFUNCTRACE_DEVELOP();
    if (fDevice){
        IXDC *xdc = fDevice->deviceContext();
        return (xdc->xDrawable);
    }
    return 0;
}

GC IBaseRootGrafPort::gc() const
{
    IFUNCTRACE_DEVELOP();
    if (fDevice){
        IXDC *xdc = fDevice->deviceContext();
        return (xdc->gc());
    }
    return NULL;
}

#endif //IC_MOTIF


const IGrafMatrix& IBaseRootGrafPort::pixelToPointSize() const
{
    IFUNCTRACE_DEVELOP();
	if (fPixelToPointSizeMatrix)
			return *fPixelToPointSizeMatrix;
	else{	
	    const IGrafMatrix* coordFlip = (fRootGrafState->root())->coordinateFlip();
			
		  if (ICoordinateSystem::isConversionNeeded() && coordFlip) // case for flip
				if (! coordFlip->isIdentity())
				{
			    ((IBaseRootGrafPort*)this)->fPixelToPointSizeMatrix = new IGrafMatrix(*coordFlip);
						fPixelToPointSizeMatrix->invert();
				    fPixelToPointSizeMatrix->preConcatWith(*fGrafState->modelMatrix());// undo the coordinate flip from the model matrix.
						fPixelToPointSizeMatrix->invert();
			      return  *fPixelToPointSizeMatrix ;
				 }
				
			// case for no flip
			 ((IBaseRootGrafPort*)this)->fPixelToPointSizeMatrix = new IGrafMatrix(*fGrafState->modelMatrix());
	      fPixelToPointSizeMatrix->invert();
				return  *fPixelToPointSizeMatrix;
	}	
}




//================
// IExtendedRootGrafPort routines
//================

/*================
||
||  IExtendedRootGrafPort constructors
||
*/

IExtendedRootGrafPort::~IExtendedRootGrafPort()
{}

// implementation for disabled default constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IExtendedRootGrafPort::IExtendedRootGrafPort()
{}

// implementation for disabled copy constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IExtendedRootGrafPort::IExtendedRootGrafPort(const IExtendedRootGrafPort&)
{}

IExtendedRootGrafPort::IExtendedRootGrafPort(
	IPresSpaceHandle deviceContext,
	ICoordinateSystem::EOrientation coordinateSystem,
	EDeviceMappingMode mappingMode) :

	IBaseRootGrafPort(IGrafDeviceMaker::makeDevice(deviceContext,IGrafPort::kIExtendedRootGrafPort, mappingMode))
{
    IFUNCTRACE_DEVELOP();
	setupCoordinateSystem(coordinateSystem);
}

IExtendedRootGrafPort::IExtendedRootGrafPort(
	IPresSpaceHandle deviceContext,
	EDeviceMappingMode mappingMode) :

	IBaseRootGrafPort(IGrafDeviceMaker::makeDevice(deviceContext,IGrafPort::kIExtendedRootGrafPort, mappingMode))
{
    IFUNCTRACE_DEVELOP();
	setupCoordinateSystem(ICoordinateSystem::applicationOrientation());
}

IExtendedRootGrafPort::IExtendedRootGrafPort(IGrafDevice* referencedDevice) :
	IBaseRootGrafPort(referencedDevice)
{}

//================
// ILinkedGrafPort routines
//================


/*================
||
||  ILinkedGrafPort state access
||
*/

void ILinkedGrafPort::useState() const
{
    IFUNCTRACE_DEVELOP();
    fPortForStateUsage->useState();
}

void ILinkedGrafPort::doneWithState() const
{
    IFUNCTRACE_DEVELOP();
    fPortForStateUsage->doneWithState();
}

const IGrafPort* ILinkedGrafPort::portForStateUsage() const
{
    IFUNCTRACE_DEVELOP();
	return fPortForStateUsage;
}

const IGrafMatrix& ILinkedGrafPort::invertedDeviceTransform() const
{
    IFUNCTRACE_DEVELOP();
	return fParent->invertedDeviceTransform();
}

const IGrafMatrix& ILinkedGrafPort::pixelToPointSize() const
{
    IFUNCTRACE_DEVELOP();
	return fParent->pixelToPointSize();
}

/*================
||
||  ILinkedGrafPort destructor
||
*/

ILinkedGrafPort::~ILinkedGrafPort()
{
    IFUNCTRACE_DEVELOP();
	if (fLinkedGrafState && fLinkedGrafState->referenceCount() != 0)
		fParent->adoptOrphanedGrafState(fGrafState);
	else
    	delete fLinkedGrafState;
}

// implementation for disabled default constructor... this does not have to be implemented if the compiler extended RTTI does not require it
ILinkedGrafPort::ILinkedGrafPort()
{}

// implementation for disabled copy constructor... this does not have to be implemented if the compiler extended RTTI does not require it
ILinkedGrafPort::ILinkedGrafPort(const ILinkedGrafPort&)
{}

// the allocation of linked-grafstates can be managed by using a pool of linked-grafstates
// objects from a cache as Peter suggested. Thus the heap events can be reduced.

//================
// Linked Bundle Port constructor
//================

ILinkedGrafPort::ILinkedGrafPort(
    IGrafPort* referencedParent,
    const IAttributeState* referencedAttributeState) :

	IGrafPort(referencedParent->device(), (IColorMap *)0),
	fPortForStateUsage(referencedParent->portForStateUsage()),
	fParent(referencedParent),
    fLinkedGrafState(new ILinkedAttributeGrafState(referencedParent->state(), referencedAttributeState))
{
    IFUNCTRACE_DEVELOP();
	setState(fLinkedGrafState);
}


//================
// Linked Matrix Port constructor
//================

ILinkedGrafPort::ILinkedGrafPort(
    IGrafPort* referencedParent,
    EMatrixKind matrixKind,
    const IGrafMatrix* referencedMatrix) :

	IGrafPort(referencedParent->device(), (IColorMap *)0),
	fPortForStateUsage(referencedParent->portForStateUsage()),
	fParent(referencedParent),
    fLinkedGrafState(0 /*NIL*/)
{
    IFUNCTRACE_DEVELOP();
	if (matrixKind == kViewMatrix)
    	fLinkedGrafState = new ILinkedViewMatrixGrafState(referencedParent->state(), referencedMatrix);
    else
		fLinkedGrafState = new ILinkedModelMatrixGrafState(referencedParent->state(), referencedMatrix);
	setState(fLinkedGrafState);
}


//================
// Linked Clipping Port constructor
//================

ILinkedGrafPort::ILinkedGrafPort(
    IGrafPort* referencedParent,
    const IGArea* referencedClipArea) :

	IGrafPort(referencedParent->device(), (IColorMap *)0),
	fPortForStateUsage(referencedParent->portForStateUsage()),
	fParent(referencedParent),
    fLinkedGrafState(new ILinkedClipGrafState(referencedParent->state(), referencedClipArea))
{
    IFUNCTRACE_DEVELOP();
	setState(fLinkedGrafState);
}


//================
// IStatefulGrafPort routines
//================

IStatefulGrafPort::IStatefulGrafPort(IBaseRootGrafPort* root) :
	IGrafPort(root->device(), (IColorMap *)0),
	fParent(root),
	fStatefulGrafState(new IStatefulGrafState(root->state()))
{
    IFUNCTRACE_DEVELOP();
	setState(fStatefulGrafState);
}

// implementation for disabled default constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IStatefulGrafPort::IStatefulGrafPort()
{}

// implementation for disabled copy constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IStatefulGrafPort::IStatefulGrafPort(const IStatefulGrafPort&)
{}

IStatefulGrafPort::~IStatefulGrafPort()
{
    IFUNCTRACE_DEVELOP();
	if (fStatefulGrafState && fStatefulGrafState->referenceCount() != 0)
		fParent->adoptOrphanedGrafState(fStatefulGrafState);
	else
		delete fStatefulGrafState;
}

void IStatefulGrafPort::useState() const
{
	// currently it dosn't do anything
}

void IStatefulGrafPort::doneWithState() const
{
	// currently it dosn't do anything
}

const IGrafPort* IStatefulGrafPort::portForStateUsage() const
{
	// someone is trying to link to this!!!
	// I could through an exception here to prevent the linking ...
    IFUNCTRACE_DEVELOP();
	return this;
}

const IGrafMatrix& IStatefulGrafPort::invertedDeviceTransform() const
{
    IFUNCTRACE_DEVELOP();
	return fParent->invertedDeviceTransform();
}

const IGrafMatrix& IStatefulGrafPort::pixelToPointSize() const
{
    IFUNCTRACE_DEVELOP();
	return fParent->pixelToPointSize();
}


void IStatefulGrafPort::setBundle(const IGrafBundle& bundle)
{
    IFUNCTRACE_DEVELOP();
	fStatefulGrafState->setBundle(bundle);
}

void IStatefulGrafPort::setMatrix(EMatrixKind matrixKind, const IGrafMatrix& matrix)
{
    IFUNCTRACE_DEVELOP();
	if (matrixKind == kViewMatrix)
		fStatefulGrafState->setViewMatrix(matrix);
	else
		fStatefulGrafState->setModelMatrix(matrix);
}

void IStatefulGrafPort::setClipArea(const IGArea& clipArea)
{
    IFUNCTRACE_DEVELOP();
	fStatefulGrafState->setClipArea(clipArea);
}


#if (IC_OBSOLETE <= IC_OBSOLETE_3)

// IRootGrafPort is a subclass of IExtendedRootGrafPort

IRootGrafPort::~IRootGrafPort()
{}

// implementation for disabled default constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IRootGrafPort::IRootGrafPort()
{}

// implementation for disabled copy constructor... this does not have to be implemented if the compiler extended RTTI does not require it
IRootGrafPort::IRootGrafPort(const IRootGrafPort&)
{}

IRootGrafPort::IRootGrafPort(IPresSpaceHandle deviceContext)
  :IExtendedRootGrafPort(deviceContext,ICoordinateSystem::kOriginUpperLeft)
{}


IRootGrafPort::IRootGrafPort(IGrafDevice* referencedDevice)
  :IExtendedRootGrafPort(referencedDevice)
{}

#endif // IC_OBSOLETE
