// Revision: 69 1.53.2.2 source/core/text/itext/itext.cpp, text, ioc.v400, 990114 
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
//
//      IText.cpp
//
//      Function: The new variable-length styled string object, based on TStandardText
//              from CommonPoint, and obeying the ANSI basic_string protocol.
//
/*
*****************************************************************************************
*                                                                                       *
* COPYRIGHT:                                                                            *
*   IBM Open Class Library                                                              *
*   (C) Copyright International Business Machines Corporation,  1997                    *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.                  *
*   US Government Users Restricted Rights - Use, duplication, or disclosure             *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                              *
*                                                                                       *
*****************************************************************************************
*/

//
//  Change History:
//       G. Anton 3/8/96 Fixed code to be compliant with Open Class coding guidelines.
//
//~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

#ifndef IC_ITEXT
#include "itext.hpp"
#endif

#ifndef _IUGNRL_HPP
#include "iugnrl.hpp"
#endif

#ifndef _ISTRING_
#include "istring.hpp"
#endif

#ifndef IC_ITXTITER
#include "itxtiter.hpp"
#endif

#ifndef IC_ITXTSTOR
#include "itxtstor.hpp"
#endif

#ifndef IC_ISTXSTOR
#include "istxstor.hpp"
#endif

#ifndef IC_ISTYLSET
#include "istylset.hpp"
#endif

#ifndef IC_ISTYLSTO
#include "istylsto.hpp"
#endif

#ifndef _IDATSTRM_
#include "idatstrm.hpp"
#endif

#ifndef _ISTMFND_
#include "istrmfnd.hpp"
#endif

#ifndef _IEXCEPT_
#include <iexcept.hpp>
#endif

#ifndef _ITRACE_
#include <itrace.hpp>
#endif

#ifndef _IPRIMLCK_
#include <iprimlck.hpp>
#endif

#ifdef IC_PAGETUNE
        #define _ITEXT_CPP
        #include <ipagetun.h>
#endif

#ifndef _ITRANCOD_
#include <itrancod.hpp>
#endif

//--------------------------------------------------------------
// private implementation-utility functions
// (these two functions are not published in the header file)
//--------------------------------------------------------------
IText::length_type
find_length(    const UniChar*  x)
{
        const UniChar*  temp = x;

        while (*temp != UUnicodeSpecial::kNull)
                temp++;
        return temp - x;
}

inline void
pin(    IText::length_type&             value,
                IText::length_type              pinTo)
{
        if (value > pinTo)
                value = pinTo;
}


//--------------------------------------------------------------
//  Lazy evaluation functions
//--------------------------------------------------------------
const IText& IText::emptyText()
{
    static const IText* gEmptyText = 0;

    if (!gEmptyText)
    {
        IPrimalLock lockInit;

        if (!gEmptyText)
            gEmptyText = new IText;
    }
    return *gEmptyText;
}

// Lazily-constructed default (code-page sensitive) transcoder
ITranscoder *gDefaultTranscoder = 0;
bool gInitDefaultTranscoderDone = false;

// this function should be called once to create the default transcoder
// A default transcoder is only created if MB_CUR_MAX > 1
void
gInitDefaultTranscoder()
{
    if (!gInitDefaultTranscoderDone) {
        IPrimalLock lockInit;
        
        if (!gInitDefaultTranscoderDone) {
            try {
                if (MB_CUR_MAX > 1)
                    gDefaultTranscoder = ITranscoder::createTranscoder();
                else
                    gDefaultTranscoder = 0;
            }
            // may fail, so must check if gDefaultTranscoder != 0 before using
            catch (IException& e) {
                gDefaultTranscoder = 0;
            }
            gInitDefaultTranscoderDone = true;
        }
    }
}

//========================================================================================
// CLASS IText: A variable-length string class capable of storing any arbitrary amount
// of Unicode text information (and, eventually, style information).
//========================================================================================

#pragma info(nodcl)
const IText::length_type _EXPORT IText::npos = (IText::length_type)(-1);
#pragma info(restore)

//--------------------------------------------------------------
// constructors and destructor
//--------------------------------------------------------------

IText::IText()
: fStorage(0)
{
        fStorage = ITextStorage::emptyTextStorage();
        fStorage->addRef();
}

IText::IText(   const IText&    that)
: fStorage(that.fStorage)
{
        fStorage->addRef();
}

IText::IText(   const IText&    that,
                                offset_type             thatPos,
                                length_type             thatLength)
: fStorage(0)
{
        // (if we're initializing ourselves to the empty string, just point ourselves to
        // fgEmptyTextStorage.
        if (thatLength == 0) {
                fStorage = ITextStorage::emptyTextStorage();
                fStorage->addRef();
        }
        else if (thatPos == 0 && thatLength == npos) {
                fStorage = that.fStorage;
                fStorage->addRef();
        }
        else {
                pin(thatLength, that.length() - thatPos);

                fStorage = ITextStorage::createNew(thatLength);
                fStorage = fStorage->replace(0, 0, that.fStorage, thatPos, thatLength);
        }
}

IText::IText(   const UniChar   that[],
                                length_type             thatLength)
: fStorage(0)
{
        // (if we're initializing ourselves to the empty string, just point ourselves to
        // fgEmptyTextStorage.
        if (thatLength == 0) {
                fStorage = ITextStorage::emptyTextStorage();
                fStorage->addRef();
        }
        else {
                fStorage = ITextStorage::createNew(thatLength);
                fStorage = fStorage->replace(0, 0, that, thatLength, 0, false);
        }
}

IText::IText(   const UniChar*  that)
: fStorage(0)
{
        length_type             thatLength = find_length(that);

        // (if we're initializing ourselves to the empty string, just point ourselves to
        // fgEmptyTextStorage.
        if (thatLength == 0) {
                fStorage = ITextStorage::emptyTextStorage();
                fStorage->addRef();
        }
        else {
                fStorage = ITextStorage::createNew(thatLength);
                fStorage = fStorage->replace(0, 0, that, thatLength, 0, false);
        }
}

IText::IText(   length_type     thatCopies,
                                UniChar         that)
: fStorage(0)
{
        // (if we're initializing ourselves to the empty string, just point ourselves to
        // fgEmptyTextStorage.
        if (thatCopies == 0) {
                fStorage = ITextStorage::emptyTextStorage();
                fStorage->addRef();
        }
        else {
                fStorage = ITextStorage::createNew(thatCopies);
                fStorage = fStorage->replace(0, 0, &that, 1, thatCopies - 1, true);
        }
}

IText::IText(UniChar    that)
: fStorage(0)
{
        fStorage = ITextStorage::createNew(1);
        fStorage = fStorage->replace(0, 0, &that, 1, 0, false);
}

IText::IText(   ITextIterator   thatFrom,
                                ITextIterator   thatTo)
: fStorage(0)
{
        IASSERTPARM(thatFrom.text() == thatTo.text() && thatFrom.offset() <= thatTo.offset())

        length_type             thatLength = thatTo.offset() - thatFrom.offset();

        // (if we're initializing ourselves to the empty string, just point ourselves to
        // fgEmptyTextStorage.
        if (thatLength == 0) {
                fStorage = ITextStorage::emptyTextStorage();
                fStorage->addRef();
        }
        else {
                fStorage = ITextStorage::createNew(thatLength);
                fStorage = fStorage->replace(0, 0, thatFrom.text()->fStorage, thatFrom.offset(),
                                                        thatTo.offset() - thatFrom.offset());
        }
}

IText::IText(   const char*     that)
: fStorage(0)
{
        length_type thatLength = strlen(that);

        // we know that the process of transcoding creates a brand-new ITextStorage
        // all the time anyway, so don't bother creating a new one here that's just
        // going to get thrown away again... just use fgEmptyTextStorage

        fStorage = ITextStorage::emptyTextStorage();
        fStorage->addRef();

        // create a default transcoder to do the conversion
        if (!gInitDefaultTranscoderDone) {
           gInitDefaultTranscoder();
        }

        if (thatLength != 0) { // if we're being initialized to the empty string, just
                               // keep pointing to fgEmptyTextStorage; don't transcode
           if (gDefaultTranscoder) {
              const char* next = NULL;
              length_type uniTextLength = gDefaultTranscoder->uniCharBufferSize(that, that+thatLength);
              UniChar* uniText = new UniChar[uniTextLength];
              UniChar* uniText_end = uniText + uniTextLength;
              UniChar* uniText_next = NULL;
              ITranscoder::result result;

              result = gDefaultTranscoder->toUnicode(that, that+thatLength, next,
                                                     uniText, uniText_end, uniText_next);

              if (result == codecvt_base::ok) {
                 fStorage = fStorage->replace(0, 0, uniText, uniText_next - uniText, 0, false);
              }
              // transcoder failed - use the non-transcoder algorithm
              else {
                 fStorage = fStorage->replace(0, 0, that, thatLength);
              }
              delete uniText;
           }
           // transcoder not created - use the non-transcoder algorithm
           else {
              fStorage = fStorage->replace(0, 0, that, thatLength);
           }
        }
}

IText::IText(   const IString&  that)
: fStorage(0)
{
        length_type thatLength = that.length();
        // we know that the process of transcoding creates a brand-new ITextStorage
        // all the time anyway, so don't bother creating a new one here that's just
        // going to get thrown away again... just use fgEmptyTextStorage

        fStorage = ITextStorage::emptyTextStorage();
        fStorage->addRef();

        // create the default transcoder to do the conversion
        if (!gInitDefaultTranscoderDone) {
           gInitDefaultTranscoder();
        }

        if (thatLength != 0) { // if we're being initialized to the empty string, just
                                  // keep pointing to fgEmptyTextStorage; don't transcode
           if (gDefaultTranscoder) {
              const char* next = NULL;
              length_type uniTextLength = gDefaultTranscoder->uniCharBufferSize(that);
              UniChar* uniText = new UniChar[uniTextLength];
              UniChar* uniText_end = uniText + uniTextLength;
              UniChar* uniText_next = NULL;
              ITranscoder::result result;

              result = gDefaultTranscoder->toUnicode((const char*)that, (const char*)that+thatLength, next,
                                                     uniText, uniText_end, uniText_next);

              if (result == codecvt_base::ok) {
                 fStorage = fStorage->replace(0, 0, uniText, uniText_next - uniText, 0, false);
              }
              // transcoder failed - use the non-transcoder algorithm
              else {
                 fStorage = fStorage->replace(0, 0, that, thatLength);
              }
              delete uniText;
           }
           // transcoder not created - use the non-transcoder algorithm
           else {
              fStorage = fStorage->replace(0, 0, (const char*)that, thatLength);
           }
        }
}


// IText::IText(        const basic_string<char>&       that)
// : fStorage(0)
// {
//      fStorage = ITextStorage::createNew(that.length());
//      fStorage = fStorage->replace(0, 0, that.data(), that.length());
// }
//
// IText::IText(        const basic_string<UniChar>&    that)
// : fStorage(0)
// {
//      fStorage = ITextStorage::createNew(that.length());
//      fStorage = fStorage->replace(0, 0, that.data(), that.length(), 0, false);
// }

IText::IText(ITextStorage*      storage)
: fStorage(0)
{
        // this is an internal constructor that is only used by ITextStorage
        storage->addRef();
        fStorage = storage;
}

IText::~IText()
{
        fStorage->drop();
}

//--------------------------------------------------------------
// assignment
//--------------------------------------------------------------

IText&
IText::operator=(       const IText&    that)
{
        that.fStorage->addRef();
        fStorage->drop();
        fStorage = that.fStorage;
        return *this;
}

IText&
IText::operator=(       const UniChar*  that)
{
        fStorage = fStorage->replace(0, length(), that, find_length(that), 0, false);
        return *this;
}

IText&
IText::operator=(       UniChar that)
{
        fStorage = fStorage->replace(0, length(), &that, 1, 0, false);
        return *this;
}

IText&
IText::assign(  const IText&    that)
{
        fStorage = fStorage->replace(0, length(), that.fStorage, 0, that.length());
        return *this;
}

IText&
IText::assign(  const IText&    that,
                                offset_type             thatPos,
                                length_type             thatLength)
{
        pin(thatLength, that.length() - thatPos);
        fStorage = fStorage->replace(0, length(), that.fStorage, thatPos, thatLength);
        return *this;
}

IText&
IText::assign(  const UniChar   that[],
                                length_type             thatLength)
{
        fStorage = fStorage->replace(0, length(), that, thatLength, 0, false);
        return *this;
}

IText&
IText::assign(  const UniChar*  that)
{
        fStorage = fStorage->replace(0, length(), that, find_length(that), 0, false);
        return *this;
}

IText&
IText::assign(  length_type     thatCopies,
                                UniChar         that)
{
        if (thatCopies == 0)
                return erase(0, length());
        else
                fStorage = fStorage->replace(0, length(), &that, 1, thatCopies - 1, true);
        return *this;
}

IText&
IText::assign(  ITextIterator   thatFrom,
                                ITextIterator   thatTo)
{
        IASSERTPARM(thatFrom.text() == thatTo.text() && thatFrom.offset() <= thatTo.offset())
        fStorage = fStorage->replace(0, length(), thatFrom.text()->fStorage, thatFrom.offset(),
                                                thatTo.offset() - thatFrom.offset());
        return *this;
}

//--------------------------------------------------------------
// conversion
//--------------------------------------------------------------

IText::operator const char*() const
{
        return (const char*)(fStorage->asIString());
}

//IText::operator IString() const
//{
//        return fStorage->asIString();
//}

IText::operator const UniChar*() const
{
        return fStorage->c_str();
}

// IText::operator basic_string<char>() const
// {
//      // FILL IN LATER!
// }
//
// IText::operator basic_string<UniChar>() const
// {
//      // FILL IN LATER!
// }

//--------------------------------------------------------------
// streaming
//--------------------------------------------------------------

void
IText::writeToStream(IDataStream&       toWhere) const
{
        // The call to typeRepresentation() basically prevents writeAliasedObject from writing
        // any type information onto the stream for this ITextStorage.  This allows us to stream
        // back in using an IStreamInTextStorage [for more on why we want to do this, see the
        // block comment for IStreamInTextStorage].  This only works if all subclasses of
        // ITextStorage have the same stream format, which they're supposed to anyway.
        ::writeAliasedObject(fStorage, fStorage->typeRepresentation(), toWhere);
}

void
IText::readFromStream(IDataStream&      fromWhere)
{
        // If there's no type information on the stream (and we suppressed it above), then the call
        // below will instantiate and stream in an IStreamInTextStorage instead of whatever class
        // the original streamed ITextStorage is.  We want this because the streaming mechanism
        // doesn't know about our overridden new and delete operators and can't allocate an
        // ITextStorage itself that's big enough to hold the characters.  We use the extra level
        // of indirection provided by ITextStorage to get around this limitation.  This works as
        // long as all ITextStorage suclasses have the same stream format, but they're supposed to
        // anyway.

        ITextStorage*   tempStorage;
        ::readAliasedObject(tempStorage, IStreamInTextStorage::staticTypeRepresentation(), fromWhere);
        fStorage->drop();
        fStorage = tempStorage;
}

void
IText::operator>>=(IDataStream& toWhere) const
{
        writeToStream(toWhere);
}

void
IText::operator<<=(IDataStream& fromWhere)
{
        readFromStream(fromWhere);
}

//--------------------------------------------------------------
// size/capacity/internal storage
//--------------------------------------------------------------

IText::length_type
IText::size() const
{
        return fStorage->length();
}

IText::length_type
IText::length() const
{
        return fStorage->length();
}

IText::length_type
IText::max_size() const
{
        return fStorage->maxSize();
}

void
IText::resize(  length_type     numChars,
                                UniChar         fillChar)
{
        if (numChars > length())
                fStorage = fStorage->replace(length(), 0, &fillChar, 1, numChars - 1 - length(), true);
        else if (numChars < length())
                fStorage = fStorage->replace(numChars, length() - numChars, ITextStorage::
                                                        emptyTextStorage(), 0, 0);
}

void
IText::resize(  length_type     numChars)
{
        resize(numChars, UUnicodeSpecial::kNull);
}

IText::length_type
IText::capacity() const
{
        return fStorage->capacity();
}

void
IText::reserve( length_type     numChars)
{
        fStorage = fStorage->reserve(numChars);
}

bool
IText::empty() const
{
        return fStorage->length() == 0;
}

const UniChar*
IText::c_str() const
{
        return fStorage->c_str();
}

const UniChar*
IText::data() const
{
        return c_str();
}

const UniChar*
IText::storage_chunk(   offset_type             searchPos,
                                                offset_type&    chunkPos,
                                                length_type&    chunkLength) const
{
        return fStorage->storageChunk(searchPos, chunkPos, chunkLength);
}

// const allocator_type&
// IText::get_allocator() const
// {
//      // FILL IN LATER!
// }

//--------------------------------------------------------------
// iterators
//--------------------------------------------------------------

ITextIterator
IText::begin()
{
        return ITextIterator(this, 0);
}

IConstTextIterator
IText::begin() const
{
        return IConstTextIterator(this, 0);
}

ITextIterator
IText::end()
{
        return ITextIterator(this, length());
}

IConstTextIterator
IText::end() const
{
        return IConstTextIterator(this, length());
}

IReverseTextIterator
IText::rbegin()
{
        return IReverseIterator<ITextIterator, ICharacterReference>(ITextIterator(this, length() - 1));
}

IReverseConstTextIterator
IText::rbegin() const
{
        return IReverseIterator<IConstTextIterator, UniChar>(IConstTextIterator(this, length() - 1));
}

IReverseTextIterator
IText::rend()
{
        // I'm using a negative number in an unsigned variable to signify the past-the-beginning
        // value, thus depending on things wrapping around correctly.  This probably isn't cool.
        return IReverseIterator<ITextIterator, ICharacterReference>(ITextIterator(this, IText::npos));
}

IReverseConstTextIterator
IText::rend() const
{
        // I'm using a negative number in an unsigned variable to signify the past-the-beginning
        // value, thus depending on things wrapping around correctly.  This probably isn't cool.
        return IReverseIterator<IConstTextIterator, UniChar>(IConstTextIterator(this, IText::npos));
}

//--------------------------------------------------------------
// character access
//--------------------------------------------------------------

UniChar
IText::operator[](      offset_type     pos) const
{
        return fStorage->at(pos);
}

ICharacterReference
IText::operator[](      offset_type     pos)
{
        if (!isStyled() && fStorage->count() == 1)
                return ICharacterReference(&fStorage->at(pos));
        else
                return ICharacterReference(this, pos);
}

UniChar
IText::at(      offset_type     pos) const
{
        return fStorage->at(pos);
}

ICharacterReference
IText::at(      offset_type     pos)
{
        return ICharacterReference(this, pos);
}

void
IText::at_put(  offset_type     pos,
                                UniChar         newChar)
{
        fStorage = fStorage->atPut(pos, newChar);
}

//--------------------------------------------------------------
// substring extraction
//--------------------------------------------------------------

IText
IText::substr(  offset_type     pos,
                                length_type     length) const
{
        return IText(*this, pos, length);
}

//--------------------------------------------------------------
// appending
//--------------------------------------------------------------

IText&
IText::operator+=(      const IText&    that)
{
        fStorage = fStorage->replace(length(), 0, that.fStorage, 0, that.length());
        return *this;
}

IText&
IText::operator+=(      const UniChar*  that)
{
        fStorage = fStorage->replace(length(), 0, that, find_length(that), 0, false);
        return *this;
}

IText&
IText::operator+=(      UniChar that)
{
        fStorage = fStorage->replace(length(), 0, &that, 1, 0, false);
        return *this;
}

IText&
IText::append(  const IText&    that)
{
        fStorage = fStorage->replace(length(), 0, that.fStorage, 0, that.length());
        return *this;
}

IText&
IText::append(  const IText&    that,
                                offset_type             thatPos,
                                length_type             thatLength)
{
        pin(thatLength, that.length() - thatPos);
        fStorage = fStorage->replace(length(), 0, that.fStorage, thatPos, thatLength);
        return *this;
}

IText&
IText::append(  const UniChar   that[],
                                length_type             thatLength)
{
        fStorage = fStorage->replace(length(), 0, that, thatLength, 0, false);
        return *this;
}

IText&
IText::append(  const UniChar*  that)
{
        fStorage = fStorage->replace(length(), 0, that, find_length(that), 0, false);
        return *this;
}

IText&
IText::append(  length_type     thatCopies,
                                UniChar         that)
{
        if (thatCopies != 0)
                fStorage = fStorage->replace(length(), 0, &that, 1, thatCopies - 1, true);
        return *this;
}

IText&
IText::append(  ITextIterator   thatFrom,
                                ITextIterator   thatTo)
{
        IASSERTPARM(thatFrom.text() == thatTo.text() && thatFrom.offset() <= thatTo.offset())
        fStorage = fStorage->replace(length(), 0, thatFrom.text()->fStorage, thatFrom.offset(),
                                                thatTo.offset() - thatFrom.offset());
        return *this;
}

//--------------------------------------------------------------
// insertion
//--------------------------------------------------------------

IText&
IText::insert(  offset_type             insertPos,
                                const IText&    that)
{
        fStorage = fStorage->replace(insertPos, 0, that.fStorage, 0, that.length());
        return *this;
}

IText&
IText::insert(  offset_type             insertPos,
                                const IText&    that,
                                offset_type             thatPos,
                                length_type             thatLength)
{
        pin(thatLength, that.length() - thatPos);
        fStorage = fStorage->replace(insertPos, 0, that.fStorage, thatPos, thatLength);
        return *this;
}

IText&
IText::insert(  offset_type             insertPos,
                                const UniChar   that[],
                                length_type             thatLength)
{
        fStorage = fStorage->replace(insertPos, 0, that, thatLength, 0, false);
        return *this;
}

IText&
IText::insert(  offset_type             insertPos,
                                const UniChar*  that)
{
        fStorage = fStorage->replace(insertPos, 0, that, find_length(that), 0, false);
        return *this;
}

IText&
IText::insert(  offset_type             insertPos,
                                length_type             thatCopies,
                                UniChar                 that)
{
        if (thatCopies !=0)
                fStorage = fStorage->replace(insertPos, 0, &that, 1, thatCopies - 1, true);
        return *this;
}

ITextIterator
IText::insert(  ITextIterator   insertPos,
                                UniChar                 that)
{
        IASSERTPARM(insertPos.text() == this)
        fStorage = fStorage->replace(insertPos.offset(), 0, &that, 1, 0, false);
        return insertPos;
}

void
IText::insert(  ITextIterator   insertPos,
                                length_type             thatCopies,
                                UniChar                 that)
{
        IASSERTPARM(insertPos.text() == this)
        if (thatCopies != 0)
                fStorage = fStorage->replace(insertPos.offset(), 0, &that, 1, thatCopies - 1, true);
}

void
IText::insert(  ITextIterator   insertPos,
                                ITextIterator   thatFrom,
                                ITextIterator   thatTo)
{
        IASSERTPARM(insertPos.text() == this && thatFrom.text() == thatTo.text())
        fStorage = fStorage->replace(insertPos.offset(), 0, thatFrom.text()->fStorage,
                                                thatFrom.offset(), thatTo.offset() - thatFrom.offset());
}

IText&
IText::insert_and_propagate_styles(     offset_type             insertPos,
                                                                        const UniChar   that[],
                                                                        length_type             thatLength)
{
        fStorage = fStorage->replace(insertPos, 0, that, thatLength, 0, false, true);
        return *this;
}

IText&
IText::insert_and_propagate_styles(     offset_type             insertPos,
                                                                        length_type             thatCopies,
                                                                        UniChar                 that)
{
        if (thatCopies != 0)
                fStorage = fStorage->replace(insertPos, 0, &that, 1, thatCopies - 1, true, true);
        return *this;
}

//--------------------------------------------------------------
// deletion
//--------------------------------------------------------------

IText&
IText::erase(   offset_type     pos,
                                length_type     numChars)
{
        pin(numChars, length() - pos);
        if (pos < length() && numChars != 0)
                fStorage = fStorage->replace(pos, numChars, ITextStorage::emptyTextStorage(), 0, 0);
        return *this;
}

ITextIterator
IText::erase(   ITextIterator   pos)
{
        IASSERTPARM(pos.text() == this)
        if (pos.offset() < length())
                fStorage = fStorage->replace(pos.offset(), 1, ITextStorage::emptyTextStorage(), 0, 0);
        return pos;
}

ITextIterator
IText::erase(   ITextIterator   from,
                                ITextIterator   to)
{
        IASSERTPARM(from.text() == this && to.text() == this && from.offset() <= to.offset())
        if (from.offset() < length() && from != to)
                fStorage = fStorage->replace(from.offset(), to.offset() - from.offset(),
                                                        ITextStorage::emptyTextStorage(), 0, 0);
        return from;
}

//--------------------------------------------------------------
// replacement
//--------------------------------------------------------------

IText&
IText::replace( offset_type             thisPos,
                                length_type             thisLength,
                                const IText&    that)
{
        pin(thisLength, length() - thisPos);
        fStorage = fStorage->replace(thisPos, thisLength, that.fStorage, 0, that.length());
        return *this;
}

IText&
IText::replace( offset_type             thisPos,
                                length_type             thisLength,
                                const IText&    that,
                                offset_type             thatPos,
                                length_type             thatLength)
{
        pin(thisLength, length() - thisPos);
        pin(thatLength, that.length() - thatPos);
        fStorage = fStorage->replace(thisPos, thisLength, that.fStorage, thatPos, thatLength);
        return *this;
}

IText&
IText::replace( offset_type             thisPos,
                                length_type             thisLength,
                                const UniChar   that[],
                                length_type             thatLength)
{
        pin(thisLength, length() - thisPos);
        fStorage = fStorage->replace(thisPos, thisLength, that, thatLength, 0, false);
        return *this;
}

IText&
IText::replace( offset_type             thisPos,
                                length_type             thisLength,
                                const UniChar*  that)
{
        pin(thisLength, length() - thisPos);
        fStorage = fStorage->replace(thisPos, thisLength, that, find_length(that), 0, false);
        return *this;
}

IText&
IText::replace( offset_type             thisPos,
                                length_type             thisLength,
                                length_type             thatCopies,
                                UniChar                 that)
{
        pin(thisLength, length() - thisPos);
        if (thatCopies == 0)
                return erase(thisPos, thisLength);
        else
                fStorage = fStorage->replace(thisPos, thisLength, &that, 1, thatCopies - 1, true);
        return *this;
}

IText&
IText::replace( ITextIterator   thisFrom,
                                ITextIterator   thisTo,
                                const IText&    that)
{
        IASSERTPARM(thisFrom.text() == this && thisTo.text() == this && thisFrom.offset() <=
                                                thisTo.offset())
        fStorage = fStorage->replace(thisFrom.offset(), thisTo.offset() - thisFrom.offset(),
                                                that.fStorage, 0, that.length());
        return *this;
}

IText&
IText::replace( ITextIterator   thisFrom,
                                ITextIterator   thisTo,
                                const UniChar   that[],
                                length_type             thatLength)
{
        IASSERTPARM(thisFrom.text() == this && thisTo.text() == this && thisFrom.offset() <=
                                                thisTo.offset())
        fStorage = fStorage->replace(thisFrom.offset(), thisTo.offset() - thisFrom.offset(),
                                                that, thatLength, 0, false);
        return *this;
}

IText&
IText::replace( ITextIterator   thisFrom,
                                ITextIterator   thisTo,
                                const UniChar*  that)
{
        IASSERTPARM(thisFrom.text() == this && thisTo.text() == this && thisFrom.offset() <=
                                                thisTo.offset())
        fStorage = fStorage->replace(thisFrom.offset(), thisTo.offset() - thisFrom.offset(),
                                                that, find_length(that), 0, false);
        return *this;
}

IText&
IText::replace( ITextIterator   thisFrom,
                                ITextIterator   thisTo,
                                length_type             thatCopies,
                                UniChar                 that)
{
        IASSERTPARM(thisFrom.text() == this && thisTo.text() == this && thisFrom.offset() <=
                                                thisTo.offset())
        if (thatCopies == 0)
                erase(thisFrom, thisTo);
        else
                fStorage = fStorage->replace(thisFrom.offset(), thisTo.offset() - thisFrom.offset(),
                                                &that, 1, thatCopies - 1, true);
        return *this;
}

IText&
IText::replace( ITextIterator   thisFrom,
                                ITextIterator   thisTo,
                                ITextIterator   thatFrom,
                                ITextIterator   thatTo)
{
        IASSERTPARM(thisFrom.text() == this && thisTo.text() == this && thisFrom.offset() <=
                                                thisTo.offset())
        IASSERTPARM(thatFrom.text() == thatTo.text() && thatFrom.offset() <= thatTo.offset())
        fStorage = fStorage->replace(thisFrom.offset(), thisTo.offset() - thisFrom.offset(),
                                                thatFrom.text()->fStorage, thatFrom.offset(), thatTo.offset()
                                                - thatFrom.offset());
        return *this;
}

//--------------------------------------------------------------
// comparison
//--------------------------------------------------------------

int
IText::compare( const IText&    that) const
{
        // before going through the trouble of looping through the characters,
        // check to see whether these two ITexts share the same storage.  If so,
        // then they're obviously equal.
        if (fStorage == that.fStorage)
                return 0;

        IFastTextIterator       thisIter = fStorage->beginAt(0);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());
        IFastTextIterator       thatIter = that.fStorage->beginAt(0);
        IFastTextIterator       thatEnd = that.fStorage->beginAt(that.length());

        while (thisIter < thisEnd && thatIter < thatEnd && *thisIter == *thatIter) {
                ++thisIter;
                ++thatIter;
        }
        if (thisIter == thisEnd && thatIter == thatEnd)
                return 0;
        else if (thisIter == thisEnd)
                return -1;
        else if (thatIter == thatEnd)
                return 1;
        else if (*thisIter < *thatIter)
                return -1;
        else
                return 1;
}

int
IText::compare( offset_type             thisPos,
                                length_type             thisLength,
                                const IText&    that) const
{
        IASSERTPARM(thisPos < length())

        IFastTextIterator       thisIter = fStorage->beginAt(thisPos);
        IFastTextIterator       thisEnd = fStorage->beginAt((thisPos + thisLength <= length()) ?
                                                        thisPos + thisLength : length());
        IFastTextIterator       thatIter = that.fStorage->beginAt(0);
        IFastTextIterator       thatEnd = that.fStorage->beginAt(that.length());

        while (thisIter < thisEnd && thatIter < thatEnd && *thisIter == *thatIter) {
                ++thisIter;
                ++thatIter;
        }
        if (thisIter == thisEnd && thatIter == thatEnd)
                return 0;
        else if (thisIter == thisEnd)
                return -1;
        else if (thatIter == thatEnd)
                return 1;
        else if (*thisIter < *thatIter)
                return -1;
        else
                return 1;
}

int
IText::compare( offset_type             thisPos,
                                length_type             thisLength,
                                const IText&    that,
                                offset_type             thatPos,
                                length_type             thatLength) const
{
        IASSERTPARM(thisPos < length() && thatPos < that.length())

        IFastTextIterator       thisIter = fStorage->beginAt(thisPos);
        IFastTextIterator       thisEnd = fStorage->beginAt((thisPos + thisLength <= length()) ?
                                                        thisPos + thisLength : length());
        IFastTextIterator       thatIter = that.fStorage->beginAt(thatPos);
        IFastTextIterator       thatEnd = that.fStorage->beginAt((thatPos + thatLength <= that.length()) ?
                                                        thatPos + thatLength : that.length());

        while (thisIter < thisEnd && thatIter < thatEnd && *thisIter == *thatIter) {
                ++thisIter;
                ++thatIter;
        }
        if (thisIter == thisEnd && thatIter == thatEnd)
                return 0;
        else if (thisIter == thisEnd)
                return -1;
        else if (thatIter == thatEnd)
                return 1;
        else if (*thisIter < *thatIter)
                return -1;
        else
                return 1;
}

int
IText::compare( const UniChar*          that) const
{
        IFastTextIterator       thisIter = fStorage->beginAt(0);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());

        while (thisIter < thisEnd && *that != UUnicodeSpecial::kNull && *thisIter == *that) {
                ++thisIter;
                ++that;
        }
        if (thisIter == thisEnd && *that == UUnicodeSpecial::kNull)
                return 0;
        else if (thisIter == thisEnd)
                return -1;
        else if (*that == UUnicodeSpecial::kNull)
                return 1;
        else if (*thisIter < *that)
                return -1;
        else
                return 1;
}

int
IText::compare( offset_type             thisPos,
                                length_type             thisLength,
                                const UniChar   athat[],

                                length_type             thatLength) const
{
        const UniChar *that = athat;
        IASSERTPARM(thisPos < length())

        IFastTextIterator       thisIter = fStorage->beginAt(thisPos);
        IFastTextIterator       thisEnd = fStorage->beginAt((thisPos + thisLength <= length()) ?
                                                        thisPos + thisLength : length());
        const UniChar*          thatEnd = that + (long)thatLength;

        while (thisIter < thisEnd && that < thatEnd && *thisIter == *that) {
                ++thisIter;
                ++that;
        }
        if (thisIter == thisEnd && that == thatEnd)
                return 0;
        else if (thisIter == thisEnd)
                return -1;
        else if (that == thatEnd)
                return 1;
        else if (*thisIter < *that)
                return -1;
        else
                return 1;
}

long
IText::hash() const
{
        return fStorage->hash();
}

//--------------------------------------------------------------
// miscellaneous
//--------------------------------------------------------------

IText::length_type
IText::copy(    UniChar                 that[],
                                length_type             maxChars,
                                offset_type             startPos) const
{
        IASSERTPARM(startPos <= length())

        if (empty())
                return 0;

        length_type             charsToCopy;

        if (maxChars > length() - startPos)
                charsToCopy = length() - startPos;
        else
                charsToCopy = maxChars;
        fStorage->copyOut(that, startPos, charsToCopy);
        return charsToCopy;
}

void
IText::swap(    IText&          that)
{
        fStorage->addRef();
        that.fStorage->addRef();

        ITextStorage*   temp = fStorage;
        fStorage = that.fStorage;
        that.fStorage = temp;

        fStorage->removeRef();
        that.fStorage->removeRef();
}

//--------------------------------------------------------------
// searching
//--------------------------------------------------------------

IText::offset_type
IText::find(    const IText&    key,
                                offset_type             startPos) const
{
        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());
        IFastTextIterator       keyBegin = key.fStorage->beginAt(0);
        IFastTextIterator       keyIter = keyBegin;
        IFastTextIterator       keyEnd = key.fStorage->beginAt(key.length());

        while (thisIter != thisEnd && keyIter != keyEnd) {
                if (*thisIter == *keyIter) {
                        ++keyIter;
                        ++thisIter;
                }
                else {
                        if (keyIter != keyBegin)
                                keyIter = keyBegin;
                        else
                                ++thisIter;
                }
        }
        if (keyIter == keyEnd)
                return (offset_type)(thisIter - thisBegin - key.length());
        else
                return npos;
}

IText::offset_type
IText::find(    const UniChar   key[],
                                offset_type             startPos,
                                length_type             keyLength) const
{
        return find(IText(key, keyLength), startPos);
}

IText::offset_type
IText::find(    const UniChar*  key,
                                offset_type             startPos) const
{
        return find(IText(key), startPos);
}

IText::offset_type
IText::find(    UniChar                 key,
                                offset_type             startPos) const
{
        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());

        while (thisIter != thisEnd && *thisIter != key)
                ++thisIter;
        if (thisIter == thisEnd)
                return npos;
        else
                return thisIter - thisBegin;
}

IText::offset_type
IText::rfind(   const IText&    key,
                                offset_type             startPos) const
{
        if (startPos == 0)
                return npos;

        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisEnd = fStorage->beginAt(length() - 1);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos - 1);
        IFastTextIterator       keyBegin = key.fStorage->beginAt(0);
        IFastTextIterator       keyEnd = key.fStorage->beginAt(key.length() - 1);
        IFastTextIterator       keyIter = keyEnd;

        --thisBegin;
        --keyBegin;             // we really want past-the-beginning values here

        while (thisIter != thisBegin && keyIter != keyBegin) {
                if (*thisIter == *keyIter) {
                        --keyIter;
                        --thisIter;
                }
                else {
                        if (keyIter != keyEnd)
                                keyIter = keyEnd;
                        else
                                --thisIter;
                }
        }
        if (keyIter == keyBegin)
                return thisIter - thisBegin;
        else
                return npos;
}

IText::offset_type
IText::rfind(   const UniChar   key[],
                                offset_type             startPos,
                                length_type             keyLength) const
{
        return rfind(IText(key, keyLength), startPos);
}

IText::offset_type
IText::rfind(   const UniChar*  key,
                                offset_type             startPos) const
{
        return rfind(IText(key), startPos);
}

IText::offset_type
IText::rfind(   UniChar                 key,
                                offset_type             startPos) const
{
        if (startPos == 0)
                return npos;

        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisEnd = fStorage->beginAt(length() - 1);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos - 1);

        --thisBegin;

        while (thisIter != thisBegin && *thisIter != key)
                --thisIter;
        if (thisIter == thisBegin)
                return npos;
        else
                return thisIter - (++thisBegin);
}

IText::offset_type
IText::find_first_of(   const IText&    keys,
                                                offset_type             startPos) const
{
        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());
        IFastTextIterator       keysBegin = keys.fStorage->beginAt(0);
        IFastTextIterator       keysIter = keysBegin;
        IFastTextIterator       keysEnd = keys.fStorage->beginAt(keys.length());

        while (thisIter != thisEnd && keysIter == keysBegin) {
                while (keysIter != keysEnd && *thisIter != *keysIter)
                        ++keysIter;
                if (keysIter == keysEnd) {
                        keysIter = keysBegin;
                        ++thisIter;
                }
                else
                        return thisIter - thisBegin;
        }
        return npos;
}

IText::offset_type
IText::find_first_of(   const UniChar   keys[],
                                                offset_type             startPos,
                                                length_type             numKeys) const
{
        return find_first_of(IText(keys, numKeys), startPos);
}

IText::offset_type
IText::find_first_of(   const UniChar*  keys,
                                                offset_type             startPos) const
{
        return find_first_of(IText(keys), startPos);
}

IText::offset_type
IText::find_first_of(   UniChar                 key,
                                                offset_type             startPos) const
{
        return find(key, startPos);
}

IText::offset_type
IText::find_last_of(    const IText&    keys,
                                                offset_type             startPos) const
{
        if (startPos == 0)
                return npos;

        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos - 1);
        IFastTextIterator       keysBegin = keys.fStorage->beginAt(0);
        IFastTextIterator       keysIter = keysBegin;
        IFastTextIterator       keysEnd = keys.fStorage->beginAt(keys.length());

        --thisBegin;

        while (thisIter != thisBegin && keysIter == keysBegin) {
                while (keysIter != keysEnd && *thisIter != *keysIter)
                        ++keysIter;
                if (keysIter == keysEnd) {
                        keysIter = keysBegin;
                        --thisIter;
                }
                else
                        return thisIter - (++thisBegin);
        }
        return npos;
}

IText::offset_type
IText::find_last_of(    const UniChar   keys[],
                                                offset_type             startPos,
                                                length_type             numKeys) const
{
        return find_last_of(IText(keys, numKeys), startPos);
}

IText::offset_type
IText::find_last_of(    const UniChar*  keys,
                                                offset_type             startPos) const
{
        return find_last_of(IText(keys), startPos);
}

IText::offset_type
IText::find_last_of(    UniChar                 key,
                                                offset_type             startPos) const
{
        return rfind(key, startPos);
}

IText::offset_type
IText::find_first_not_of(       const IText&    keys,
                                                        offset_type             startPos) const
{
        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());
        IFastTextIterator       keysBegin = keys.fStorage->beginAt(0);
        IFastTextIterator       keysIter = keysBegin;
        IFastTextIterator       keysEnd = keys.fStorage->beginAt(keys.length());

        while (thisIter != thisEnd && keysIter == keysBegin) {
                while (keysIter != keysEnd && *thisIter != *keysIter)
                        ++keysIter;
                if (keysIter != keysEnd) {
                        keysIter = keysBegin;
                        ++thisIter;
                }
                else
                        return thisIter - thisBegin;
        }
        return npos;
}

IText::offset_type
IText::find_first_not_of(       const UniChar   keys[],
                                                        offset_type             startPos,
                                                        length_type             numKeys) const
{
        return find_first_not_of(IText(keys, numKeys), startPos);
}

IText::offset_type
IText::find_first_not_of(       const UniChar*  keys,
                                                        offset_type             startPos) const
{
        return find_first_not_of(IText(keys), startPos);
}

IText::offset_type
IText::find_first_not_of(       UniChar                 key,
                                                        offset_type             startPos) const
{
        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos);
        IFastTextIterator       thisEnd = fStorage->beginAt(length());

        while (thisIter != thisEnd && *thisIter == key)
                ++thisIter;
        if (thisIter == thisEnd)
                return npos;
        else
                return thisIter - thisBegin;
}

IText::offset_type
IText::find_last_not_of(        const IText&    keys,
                                                        offset_type             startPos) const
{
        if (startPos == 0)
                return npos;

        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos - 1);
        IFastTextIterator       keysBegin = keys.fStorage->beginAt(0);
        IFastTextIterator       keysIter = keysBegin;
        IFastTextIterator       keysEnd = keys.fStorage->beginAt(keys.length());

        --thisBegin;

        while (thisIter != thisBegin && keysIter == keysBegin) {
                while (keysIter != keysEnd && *thisIter != *keysIter)
                        ++keysIter;
                if (keysIter != keysEnd) {
                        keysIter = keysBegin;
                        --thisIter;
                }
                else
                        return thisIter - (++thisBegin);
        }
        return npos;
}

IText::offset_type
IText::find_last_not_of(        const UniChar   keys[],
                                                        offset_type             startPos,
                                                        length_type             numKeys) const
{
        return find_last_not_of(IText(keys, numKeys), startPos);
}

IText::offset_type
IText::find_last_not_of(        const UniChar*  keys,
                                                        offset_type             startPos) const
{
        return find_last_not_of(IText(keys), startPos);
}

IText::offset_type
IText::find_last_not_of(        UniChar                 key,
                                                        offset_type             startPos) const
{
        if (startPos == 0)
                return npos;

        if (startPos > length())
                startPos = length();

        IFastTextIterator       thisBegin = fStorage->beginAt(0);
        IFastTextIterator       thisIter = fStorage->beginAt(startPos - 1);

        --thisBegin;

        while (thisIter != thisBegin && *thisIter == key)
                --thisIter;
        if (thisIter == thisBegin)
                return npos;
        else
                return thisIter - (++thisBegin);
}

//--------------------------------------------------------------
// adding and removing styles
//--------------------------------------------------------------

void
IText::addStyles(       const ITextStyle&       style,
                                        offset_type                     starting,
                                        length_type                     count)
{
        fStorage = fStorage->cloneIfNecessary();
        pin(count, length() - starting);
        fStorage->styles()->modifyStyles(starting, count, &ITextStyleSet::add, style);
}

void
IText::addStyles(       const ITextStyleSet&    styles,
                                        offset_type                             starting,
                                        length_type                             count)
{
        fStorage = fStorage->cloneIfNecessary();
        pin(count, length() - starting);
        fStorage->styles()->modifyStyles(starting, count, &ITextStyleSet::add, styles);
}

void
IText::removeStyles(    const ITextStyle&       style,
                                                offset_type                     starting,
                                                length_type                     count)
{
        if (isStyled()) {
                fStorage = fStorage->cloneIfNecessary();
                pin(count, length() - starting);
                fStorage->styles()->modifyStyles(starting, count, &ITextStyleSet::remove, style);
        }
}

void
IText::removeStyles(    const ITextStyleSet&    styles,
                                                offset_type                             starting,
                                                length_type                             count)
{
        if (isStyled()) {
                fStorage = fStorage->cloneIfNecessary();
                pin(count, length() - starting);
                fStorage->styles()->modifyStyles(starting, count,  &ITextStyleSet::remove, styles);
        }
}

void
IText::removeStyles(    offset_type     starting,
                                                length_type     count)
{
        if (isStyled()) {
                fStorage = fStorage->cloneIfNecessary();
                pin(count, length() - starting);
                if (starting == 0 && count == length())
                        fStorage->removeAllStyles();
                else {
                        fStorage->styles()->modifyStyles(starting, count, IPmfOperationOn<ITextStyleSet>(&ITextStyleSet::removeAll));
                        fStorage->styles()->extendRangeToParagraphBounds(starting, count);
                        fStorage->styles()->modifyStyles(starting, count, IPmf1OperationOn<ITextStyleSet, ITextStyle::EStylePropagation>
                                                                (&ITextStyleSet::filterByType, ITextStyle::kPropagateByCharacter));
                }
        }
}

//--------------------------------------------------------------
// getting style information
//--------------------------------------------------------------

const ITextStyleSet*
IText::stylesAt(        offset_type             position,
                                        offset_type&    starting,
                                        length_type&    count) const
{
        const IStyleStorage*    styles = fStorage->styles();

        if (styles == 0) {
                starting = 0;
                count = fStorage->length();
                return &(ITextStyleSet::emptyStyleSet());
        }
        else
                return styles->stylesAt(position, starting, count);
}

void
IText::stylesAt(        offset_type                                             position,
                                        ITextStyleSet&                                  set,
                                        offset_type&                                    starting,
                                        length_type&                                    count,
                                        ITextStyle::EStylePropagation   propagationToMatch) const
{
        const IStyleStorage*    styles = fStorage->styles();

        if (styles == 0) {
                starting = 0;
                count = fStorage->length();
                set.removeAll();
        }
        else
                styles->stylesAt(position, set, starting, count, propagationToMatch);
}

bool
IText::maximumStyleSpan(        offset_type                     position,
                                                        const ITextStyle&       style,
                                                        offset_type&            starting,
                                                        length_type&            count) const
{
        const IStyleStorage*    styles = fStorage->styles();

        if (styles == 0) {
                starting = 0;
                count = fStorage->length();
                return false;
        }
        else
                return styles->maximumStyleSpan(position, style, starting, count);
}

bool
IText::maximumStyleSpan(        offset_type                             position,
                                                        const ITextStyleSet&    set,
                                                        offset_type&                    starting,
                                                        length_type&                    count) const
{
        const IStyleStorage*    styles = fStorage->styles();

        if (styles == 0) {
                starting = 0;
                count = fStorage->length();
                return set.empty();
        }
        else
                return styles->maximumStyleSpan(position, set, starting, count);
}

bool
IText::continuousStylesOver(    offset_type             starting,
                                                                length_type             count,
                                                                ITextStyleSet&  styles) const
{
        const IStyleStorage*    myStyles = fStorage->styles();

        if (myStyles == 0) {
                styles.removeAll();
                return false;
        }
        else
                return myStyles->continuousStylesOver(starting, count, styles);
}

bool
IText::continuousStylesOver(    offset_type                                             starting,
                                                                length_type                                             count,
                                                                ITextStyleSet&                                  styles,
                                                                ITextStyle::EStylePropagation   propagationToMatch) const
{
        const IStyleStorage*    myStyles = fStorage->styles();

        if (myStyles == 0) {
                styles.removeAll();
                return false;
        }
        else
                return myStyles->continuousStylesOver(starting, count, styles, propagationToMatch);
}

bool
IText::isStyled() const
{
        return fStorage->isStyled();
}

bool
IText::sameStyledText(const IText& other) const
{
        // if the characters are the same, don't bother looking at the styles
        if (*this != other)
                return false;

        // if one of us is styled and the other one isn't, that's all we care about
        if (isStyled() != other.isStyled())
                return false;

        // if neither of us is styles, that's also enough to dump out
        if (!isStyled())
                return true;

        // but if we're both styled and our characters are equal, then compare the styles
        return *fStorage->styles() == *other.fStorage->styles();
}

bool
IText::sameCharacters(const IText& other) const
{
        // this is the same as operator==(), so let's not implement it twice
        return *this == other;
}

//--------------------------------------------------------------
// debugging
//--------------------------------------------------------------

void
IText::printDebugInfo() const
{
    ITRACE_RUNTIME( IString("Characters:" + IString(*this)));

        if (!isStyled())
    {
        ITRACE_RUNTIME(IString("\nUnstyled\n"));
    }
        else
    {
        ITRACE_RUNTIME(IString("\nStyle info:\n"));
                fStorage->styles()->printDebugInfo();
        }
        fStorage->printDebugInfo();
}

//========================================================================================
// NON-MEMBER FUNCTIONS ON IText
//========================================================================================

//--------------------------------------------------------------
// concatenation
//--------------------------------------------------------------

IText
operator+(      const IText&    a,
                        const IText&    b)
{
        IText           result(a);
        return result += b;
}

IText
operator+(      const UniChar*  a,
                        const IText&    b)
{
        IText           result(a);
        return result += b;
}

IText
operator+(      UniChar                 a,
                        const IText&    b)
{
        IText           result(1, a);
        return result += b;
}

IText
operator+(      const IText&    a,
                        const UniChar*  b)
{
        IText           result(a);
        return result += b;
}

IText
operator+(      const IText&    a,
                        UniChar                 b)
{
        IText           result(a);
        return result += b;
}

//--------------------------------------------------------------
// comparison
//--------------------------------------------------------------

bool
operator==(     const IText&    a,
                        const IText&    b)
{
        return a.compare(b) == 0;
}

bool
operator==(     const UniChar*  a,
                        const IText&    b)
{
        return b.compare(a) == 0;
}

bool
operator==(     const IText&    a,
                        const UniChar*  b)
{
        return a.compare(b) == 0;
}

bool
operator!=(     const IText&    a,
                        const IText&    b)
{
        return a.compare(b) != 0;
}

bool
operator!=(     const UniChar*  a,
                        const IText&    b)
{
        return b.compare(a) != 0;
}

bool
operator!=(     const IText&    a,
                        const UniChar*  b)
{
        return a.compare(b) != 0;
}

bool
operator<(      const IText&    a,
                        const IText&    b)
{
        return a.compare(b) < 0;
}

bool
operator<(      const UniChar*  a,
                        const IText&    b)
{
        return b.compare(a) > 0;
}

bool
operator<(      const IText&    a,
                        const UniChar*  b)
{
        return a.compare(b) < 0;
}

bool
operator>(      const IText&    a,
                        const IText&    b)
{
        return a.compare(b) > 0;
}

bool
operator>(      const UniChar*  a,
                        const IText&    b)
{
        return b.compare(a) < 0;
}

bool
operator>(      const IText&    a,
                        const UniChar*  b)
{
        return a.compare(b) > 0;
}

bool
operator<=(     const IText&    a,
                        const IText&    b)
{
        return a.compare(b) <= 0;
}

bool
operator<=(     const UniChar*  a,
                        const IText&    b)
{
        return b.compare(a) >= 0;
}

bool
operator<=(     const IText&    a,
                        const UniChar*  b)
{
        return a.compare(b) <= 0;
}

bool
operator>=(     const IText&    a,
                        const IText&    b)
{
        return a.compare(b) >= 0;
}

bool
operator>=(     const UniChar*  a,
                        const IText&    b)
{
        return b.compare(a) <= 0;
}

bool
operator>=(     const IText&    a,
                        const UniChar*  b)
{
        return a.compare(b) >= 0;
}

//--------------------------------------------------------------
// miscellaneous
//--------------------------------------------------------------

void
swap(   IText&  a,
                IText&  b)
{
        a.swap(b);
}


// basic_ostream<UniChar>&
// operator<<(  basic_ostream<UniChar>& stream,
//                      const IText&                    text)
// {
//      // FILL IN LATER!
// }
// basic_istream<UniChar>&
// operator>>(  basic_istream<UniChar>& stream,
//                      IText&                                  text)
// {
//      // FILL IN LATER!
// }
//
// basic_istream<UniChar>&
// getline(     basic_istream<UniChar>& stream,
//                      IText&                                  text,
//                      UniChar                                 delimiter)
// {
//      // FILL IN LATER!
// }
