// Revision: 81 1.7.1.25 source/ui/baseapp/isubmenu.cpp, menu, ioc.v400, 990114 
/*******************************************************************************
* FILE NAME: isubmenu.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in isubmenu.cpp                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

// #define IC_TRACE_ALL

extern "C"
{
  #define INCL_WINMENUS
  #define INCL_WINWINDOWMGR
  #include <iwindefs.h>

  #ifdef IC_MOTIF
    #include <Xm/RowColumn.h>
    #include <Xm/MenuShell.h>
    #include <Xm/Separator.h>
    #ifdef IC_TRACE_ALL
       #include <X11/IntrinsicP.h>
       #include <X11/CoreP.h>
    #endif
  #endif //IC_MOTIF
}

#include <icconst.h>
#include <iexcept.hpp>
#include <iframe.hpp>
#include <iframprv.hpp>
#include <imenuprv.hpp>
#include <imnitem.hpp>
#include <ithread.hpp>
#include <ireslib.hpp>
#include <isubmenu.hpp>
#include <iplatfrm.hpp>
#include <iwindow.hpp>

// Segment definitions
#ifdef IC_PAGETUNE
  #define _ISUBMENU_CPP_
  #include <ipagetun.h>
#endif

// enum and ISubmenuData used for maintaining changes made
// so that they can be undone

#pragma enum(4)
#pragma pack(push,4)

enum RecordType { removeRecord, addRecord, changeRecord };

class ISubmenuData {
public:
 RecordType     recordType;
 IMenuItem*     pMenuItem;
 ISubmenuData*  pNext;
} ;

#pragma pack(pop)
#pragma enum(pop)

/*------------------------------------------------------------------------------
  Constructor. submenuHandle can be a root menu (e.g., menubar) or a dropdown
------------------------------------------------------------------------------*/
ISubmenu::ISubmenu ( const IMenuHandle& submenuHandle )
  : hwndHandle(0),
    fSubmenuData(0),
    pItemList(0),
    fMenuWindowNewdInCtor(false),
    fMenuFromPriorMenuAttribute(0)
{
  IMODTRACE_DEVELOP("ISubmenu::ISubmenu");
  ITRACE_DEVELOP(IString(" Menu handle is: ") +
                 IString(submenuHandle.asUnsigned()).d2x());
  IASSERTPARM(submenuHandle);

#ifdef IC_WIN
  hwndHandle = submenuHandle;

  /*------------------------------------------------------------------*/
  /* For either menu type, need to find owner of menu                 */
  /*------------------------------------------------------------------*/
  FRAMEINFO frameInfo;
  frameInfo.hmenu  = submenuHandle;
  frameInfo.owner  = 0;
  frameInfo.frame  = 0;

  /*------------------------------------------------------------------*/
  /* Enumerate all of the top level windows (to find frame w/ table,  */
  /*  or, more specifically, the frame which owns submenuHandle)      */
  /*------------------------------------------------------------------*/
  EnumWindows( (WNDENUMPROC)IMenuPrivate::EnumAllFrames,
               (LPARAM)&frameInfo );

  /*------------------------------------------------------------------*/
  /* If owner parameter set (meaning found), then use it              */
  /*------------------------------------------------------------------*/
  if (frameInfo.owner)
  {
    fowner     = frameInfo.owner;
    fmenuFrame = frameInfo.frame;
  }
  else
    ITRACE_DEVELOP( IString("menu handle=") +
                    IString(submenuHandle.asUnsigned()).d2x() +
                    IString(" must be for a dropdown. Thus, fowner=0.") );
#endif
#ifdef IC_MOTIF
  Widget rcHandle = submenuHandle;
  // The handle passed in may be that of an IPopUpMenu's MenuShell.
  // If so, we get the (only) child of the MenuShell - the popup RowColumn widget.
  // Obtain the rowcolumn widget for this popup menu (window()->handle() is its XmMenuShell parent)
  if (XmIsMenuShell(rcHandle)) {
    ITRACE_DEVELOP(IString("Found a menu shell, switching to child"));
    WidgetList children = NULL;
    Cardinal numChildren = 0;
    XtVaGetValues (submenuHandle,
                   XmNchildren, &children,
                   XmNnumChildren, &numChildren,
                   NULL);
    if (numChildren==1)
       rcHandle = children[0];
    else
      ITHROWGUIERROR("InternalErrorInvalidMenu");
  }
  // In case this is a button on the menu, walk up until we find SOME RowColumn
  while (!XmIsRowColumn(rcHandle)) {
     rcHandle = XtParent(rcHandle);
  }
  hwndHandle = IMenuHandle(rcHandle);
  IWindow* rootMenuWindow = initialize(hwndHandle);         //26329
#endif
#ifdef IC_PM
  hwndHandle = submenuHandle;
  IWindow* rootMenuWindow = 0;     //26329 - OS/2 doesn't need event re-dispatching
#endif

#ifdef IC_PM
  // Find or create a IWindow wrapper for submenuHandle & make it our
  // non-portable "menu" window (we must take the small risk that client code
  // could delete it out from under us).
  MenuWindow* submenuWindow = dynamic_cast<MenuWindow*>(
     IWindow::windowWithHandle((IWindowHandle)hwndHandle));
  if (!submenuWindow) {
    submenuWindow = new MenuWindow((IWindowHandle)hwndHandle, rootMenuWindow);    //26329
    submenuWindow->setAutoDestroyWindow(false);
    submenuWindow->reserveUserWindowWord( false );
  }
  //Save any existing menu attr (like when fixedSubmenuHandle is our menubar!)
  fMenuFromPriorMenuAttribute = submenuWindow->menu();
  //Replace existing with our menu attribute value (existing IMenuAttribute is deleted)
  submenuWindow->setMenu( this );
  setWindow(submenuWindow);
#endif // IC_PM


#ifdef IC_MOTIF
  // Find or create a IWindow wrapper for submenuHandle & make it our
  // non-portable "menu" window (we must take the small risk that client code
  // could delete it out from under us).
  MenuWindow* submenuWindow = dynamic_cast<MenuWindow*>(
     IWindow::windowWithHandle((IWindowHandle)hwndHandle));
  if (!submenuWindow) {
    submenuWindow = new MenuWindow((IWindowHandle)hwndHandle, rootMenuWindow);    //26329
    fMenuWindowNewdInCtor = true;
    submenuWindow->setAutoDestroyWindow(false);
    submenuWindow->reserveUserWindowWord( false );
  }
  //Save any existing menu attr (like when fixedSubmenuHandle is our menubar!)
  fMenuFromPriorMenuAttribute = submenuWindow->menu();
  //Replace existing with our menu attribute value (existing IMenuAttribute is deleted)
  submenuWindow->setMenu( this );
  setWindow(submenuWindow);
#endif // IC_MOTIF
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ISubmenu::initialize                                                         |
------------------------------------------------------------------------------*/
IWindow* ISubmenu::initialize(const IMenuHandle& submenuHandle)
{
  MenuWindow* pwin( IMenuPrivate::rootMenuWindowForHandle( submenuHandle ) );
  if ( pwin )
  {
     IMenu* pmenu( pwin->menu() );
     if (pmenu)
     {
        delete this->fMenuData;
        this->fMenuData = &(pmenu->fMenuData->getPData());
     }
  }
  else
    ITRACE_DEVELOP(IString("Don't have a root IMenuBar/IPopUpMenu from IMenuAttribute"));

  return pwin;
}

#endif //IC_MOTIF

/*------------------------------------------------------------------------------
| ISubmenu::~ISubmenu                                                          |
------------------------------------------------------------------------------*/
ISubmenu::~ISubmenu()
{
  IMODTRACE_DEVELOP("ISubmenu::~ISubmenu");
  ITRACE_DEVELOP(IString(" Menu handle is: ") +
                 IString(hwndHandle.asUnsigned()).d2x());

  if (pItemList) {
     ISubmenuData* pList = pItemList;
     ISubmenuData* pRec  = pList->pNext;
     while (pRec) {
        ISubmenuData* pTemp = pRec;
        pRec = pRec->pNext;
        delete pTemp;
     } /* endwhile */
     delete pList;
  } /* endif */

#ifdef IC_MOTIFPM
  // Restore the previous menu contents.
  MenuWindow* menuWindow = dynamic_cast<MenuWindow*>( window() );
  if (menuWindow)
     menuWindow->setMenu( fMenuFromPriorMenuAttribute );
  fMenuFromPriorMenuAttribute = 0;
#endif // IC_MOTIFPM

  // delete fSubmenuData;
}

/*------------------------------------------------------------------------------
| ISubmenu::handle                                                             |
------------------------------------------------------------------------------*/
IMenuHandle ISubmenu::handle ( ) const
{
   return hwndHandle;
}

/*------------------------------------------------------------------------------
| ISubmenu::menuHandle                                                         |
------------------------------------------------------------------------------*/
IMenuHandle ISubmenu::menuHandle ( ) const
{
   return hwndHandle;
}

//
// For all ISubmenu member functions the basic approach is:
// Call the corresponding function in IMenu to visually change the
// Menu, then journal the change so that it can be undone.
//

/*------------------------------------------------------------------------------
| ISubmenu::checkItem                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::checkItem(unsigned long itemId, bool checked)
{
  IMODTRACE_DEVELOP("ISubmenu::checkItem");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::checkItem(itemId, checked);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::enableItem                                                         |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::enableItem(unsigned long itemId, bool enabled)
{
  IMODTRACE_DEVELOP("ISubmenu::enableItem");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::enableItem(itemId, enabled);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::uncheckItem                                                        |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::uncheckItem(unsigned long itemId)
{
  IMODTRACE_DEVELOP("ISubmenu::uncheckItem");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::checkItem(itemId, false);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::disableItem                                                        |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::disableItem(unsigned long itemId)
{
  IMODTRACE_DEVELOP("ISubmenu::disableItem");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::enableItem(itemId, false);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::setItem                                                            |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::setItem   ( const IMenuItem      &item )
{
  IMODTRACE_DEVELOP("ISubmenu::setItem");

  IMenuItem origItem = Inherited::menuItem(item.id());
  Inherited::setItem(item);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::setText                                                            |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::setText( unsigned long        itemId,
                             const char*          newText )
{
  IMODTRACE_DEVELOP("ISubmenu::setText");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::setText(itemId, newText);
  setUndoChangeItem(origItem);
  return *this;
}


/*------------------------------------------------------------------------------
| ISubmenu::setText                                                            |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::setText   ( unsigned long itemId,
                                const IResourceId    &newTextResId )
{
  IMODTRACE_DEVELOP("ISubmenu::setText");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::setText(itemId, newTextResId);
  setUndoChangeItem(origItem);
  return *this;
}


/*------------------------------------------------------------------------------
| ISubmenu::setBitmap                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::setBitmap ( unsigned long        itemId,
                                const IResourceId    &newBitmapResId )
{
  IMODTRACE_DEVELOP("ISubmenu::setBitmap");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::setBitmap(itemId, newBitmapResId);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::setBitmap                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::setBitmap ( unsigned long        itemId,
                                unsigned long        newBitmapResId )
{
  IMODTRACE_DEVELOP("ISubmenu::setBitmap");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::setBitmap(itemId, IResourceId(newBitmapResId));
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::setBitmap                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::setBitmap ( unsigned long        itemId,
                                const IBitmapHandle  &bitmapHandle )
{
  IMODTRACE_DEVELOP("ISubmenu::setBitmap");

  IMenuItem origItem = Inherited::menuItem(itemId);
  Inherited::setBitmap(itemId, bitmapHandle);
  setUndoChangeItem(origItem);
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::deleteItem                                                         |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::deleteItem(unsigned long itemId)
{
  IMODTRACE_DEVELOP("ISubmenu::deleteItem");

  // See if the item exists in the menu, if not throw exception
  IMenuItem* pNewItem = 0;
#ifdef IC_WIN
  /********************************************************************/
  /* Get the menu state. If invalid, query will throw exception       */
  /********************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    MENUITEMINFO miItem;
    miItem.cbSize = sizeof(MENUITEMINFO);
    miItem.fMask = MIIM_STATE | MIIM_ID;

    // See if the item exists in the menu, if not throw exception.
    if (!(GetMenuItemInfo( menuHandle(), (UINT)itemId, FALSE, &miItem )))
    {
      ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable );
    }

    // Save the item while we still got it
    // We're using the default copy constructor
    pNewItem = new IMenuItem(Inherited::menuItem(itemId));

    // Remove menu (not DeleteMenu) since we may want to restore this
    RemoveMenu( menuHandle(), (UINT)itemId, MF_BYCOMMAND );

  }
  else
  {
    IMenuHandle tempHMenu;
    long itemPos=0;
    bool located = IMenuPrivate::locateMenuItem( fmenuFrame,
                            menuHandle(), itemId, &itemPos, &tempHMenu );

    // See if the item exists in the menu, if not throw exception.
    if (!located || (GetMenuState( tempHMenu, (UINT)itemPos,
                                            MF_BYPOSITION ) == 0xFFFFFFFF ))
    {
      ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable );
    }

    // Save the item while we still got it
    // We're using the default copy constructor
    pNewItem = new IMenuItem(Inherited::menuItem(itemId));

    // Remove menu (not DeleteMenu) since we may want to restore this
    RemoveMenu( tempHMenu, (UINT)itemPos, MF_BYPOSITION );
  }
#endif //IC_WIN

#ifdef IC_PM
  MENUITEM miItem;

  if(!(window()->sendEvent(MM_QUERYITEM,
                           IEventParameter1(itemId, false),
                           IEventParameter2(&miItem)).asUnsignedLong()))
  {
     ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */

  // Save the item while we still got it
  // We're using the default copy constructor
  pNewItem = new IMenuItem(Inherited::menuItem(itemId));

  // Must REMOVEITEM rather than DELETEITEM as in IMenu::deleteItem().
  // DELETEITEM could destroy a bitmap that we want to be able to restore.
  window()->sendEvent(MM_REMOVEITEM, IEventData(itemId, false), 0);
#endif //IC_PM

#ifdef IC_MOTIF
  Widget itemHwnd = 0;

  ITRACE_ALL( IString("deleting itemId=") + IString(itemId) +
              IString(" from submenu handle=") +
              IString(this->window()->handle().asUnsigned()).d2x() );

  // Have to handle separators specially, because we change the id under the
  // covers for uniqueness.  Some code (notably the VAC++ IDE) assumes the
  // resource compiler ID.
  if (IMenuPrivate::isDefaultSeparatorId( itemId ))
  {   // An itemId > 65535 means the itemId is really the widget handle
      // (our adjusted ID), so nothing special is needed for compatibilty.
      if (itemId <= 65535)
      {  // Find first widget matching the ID.  Adjust the ID if it is
         // truly a separator.
         itemHwnd = XtNameToWidget((Widget)handle(),
            (char *)((IString("*")) + (IString(itemId))));
         ITRACE_DEVELOP(IString("Adjusting default separatorId itemId=") +
                        IString(itemId) +
                        IString(" actual ID=") +
                        IString((unsigned long)itemHwnd));
         if (itemHwnd && XmIsSeparator( itemHwnd ))
            itemId = fMenuData->itemWithHandle( itemHwnd );
      }
  }

  // Save the item while we still got it.  Invalid items cause an exception.
  pNewItem = new IMenuItem(Inherited::menuItem(itemId));

  // Must REMOVEITEM rather than DELETEITEM as in IMenu::deleteItem().
  // DELETEITEM could destroy a bitmap that we want to be able to restore.
  if (!itemHwnd)
     itemHwnd = XtNameToWidget((Widget)handle(),
                  (char *)((IString("*")) + (IString(itemId))));

  XtUnmanageChild(itemHwnd);
#endif //IC_MOTIF

  // Save the item in the list
  if (pItemList) {
     ISubmenuData* pNew = new ISubmenuData;
     pNew->recordType   = removeRecord;
     pNew->pMenuItem    = pNewItem;
     pNew->pNext        = pItemList;
     pItemList          = pNew;
  } else {
     ISubmenuData* pList = pItemList;
     pList              = new ISubmenuData;
     pList->recordType  = removeRecord;
     pList->pMenuItem   = pNewItem;
     pList->pNext       = 0;
     pItemList          = pList;
  } /* endif */

  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::removeSubmenu                                                      |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::removeSubmenu ( unsigned long itemWithSubmenuId )
{
  IMODTRACE_DEVELOP("ISubmenu::removeSubmenu");

   // Create two copies of the item, one for archiving, one for changing
   IMenuItem origItem = Inherited::menuItem(itemWithSubmenuId);
   IMenuItem mn  = Inherited::menuItem(itemWithSubmenuId);

#ifdef IC_MOTIF
   unsigned long afStyle = mn.style();
   afStyle  &= ~MIS_SUBMENU;
   mn.setStyle(afStyle);
#endif //IC_MOTIF

  // Reset the submenu handle in one item and then change the menu
   mn.setSubmenuHandle(0);
   Inherited::setItem(mn);

   // Finally archive the original item removed and exit
   setUndoChangeItem(origItem);
   return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::Cursor::Cursor                                                     |
| Cursor ctor.                                                                 |
------------------------------------------------------------------------------*/
ISubmenu::Cursor::Cursor(const ISubmenu&  menu)
  : lCurrent(0), pSubmenu((ISubmenu*)&menu), fCursorData(0)
{
  IMODTRACE_DEVELOP("ISubmenu::Cursor::Cursor");
  IASSERTSTATE(menu.isValid());    //27292

#ifdef IC_WIN
  IASSERTSTATE( menu.menuHandle() != 0 );
#endif //IC_WIN
#ifdef IC_PM
  IASSERTSTATE(WinIsWindow(IThread::current().anchorBlock(), menu.handle() ));
#endif //IC_PM
#ifdef IC_MOTIF
  IASSERTSTATE(XmIsRowColumn(menu.handle()));
  setToFirst();
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
| ISubmenu::Cursor::Cursor                                                     |
| Copy ctor.                                                                   |
------------------------------------------------------------------------------*/
ISubmenu::Cursor::Cursor(const ISubmenu::Cursor& cursor )
  : lCurrent(cursor.lCurrent), pSubmenu(cursor.pSubmenu),
    fCursorData(cursor.fCursorData)
{
}

/*------------------------------------------------------------------------------
| ISubmenu::Cursor::~Cursor                                                  |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
ISubmenu::Cursor::~Cursor()
{
  // delete fCursorData;
}

/*------------------------------------------------------------------------------
| ISubmenu::menuItemId                                                         |
|  Return menu-item ID of item at cursor position                              |
------------------------------------------------------------------------------*/
unsigned long ISubmenu::menuItemId ( const Cursor& cursor ) const
{
  IASSERTSTATE( cursor.isValid());
  unsigned long ulId = 0;
  bool rc = false;
#ifdef IC_WIN
  if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
  {
    // Query for the id of the item at the cursor position
    ulId = GetMenuItemID( menuHandle(), (int)cursor.lCurrent );

    // If error, the item might be a submenu so query for a menu handle
    if ( ulId == 0xFFFFFFFFul )
    {
      HMENU hPop = GetSubMenu( menuHandle(), (int)cursor.lCurrent );

      // If a menu handle is returned, then get the id from the lookup table
      if (hPop)
        ulId = IMenuPrivate::getItemIdFromLookUpTable( fmenuFrame, hPop );
    }

    // If -1, then either it wasn't a popup, or it wasn't in the lookup table
    rc = ( ulId != 0xFFFFFFFFul );
  }
  else
  {
    MENUITEMINFO miItem;
    miItem.cbSize = sizeof(MENUITEMINFO);
    miItem.fMask = MIIM_STATE | MIIM_ID;

    rc = GetMenuItemInfo( menuHandle(), (int)cursor.lCurrent, TRUE,
                          &miItem );
    ulId = (unsigned long)miItem.wID;
  }

#endif //IC_WIN

#ifdef IC_PM
  IWindowHandle hwnd = (cursor.pSubmenu->handle());
  ulId = (unsigned long)hwnd.sendEvent( MM_ITEMIDFROMPOSITION,
                                        (int)cursor.lCurrent, 0);
  if (ulId != MIT_ERROR)
     rc = true;
#endif //IC_PM

#ifdef IC_MOTIF
  // The menu item ID is obtained from the item located by position; there could be duplicates (e.g., Separators)
  long i = 0;
  WidgetList submenuList = 0;
  XtVaGetValues(cursor.pSubmenu->hwndHandle,
                 XmNnumChildren, &i,
                 XmNchildren, &submenuList,
                 NULL);
//  if (cursor.lCurrent <= i)    //27292
  if (cursor.lCurrent < i)
  {
     ulId = fMenuData->itemWithHandle( submenuList[cursor.lCurrent] );
     rc = true;
  }
#endif //IC_MOTIF
  if (!rc)
  {
     ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */
  return ulId;
}

/*------------------------------------------------------------------------------
| ISubmenu::Cursor ISubmenu::cursor                                            |
------------------------------------------------------------------------------*/
ISubmenu::Cursor ISubmenu::cursor ( unsigned long itemId ) const
{
#ifdef IC_WIN
   long ulId;
   if (!IMenuPrivate::locateMenuItem( fmenuFrame, menuHandle(), itemId,
                                      &ulId, NULL ))
   {
     ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   } /* endif */
#endif //IC_WIN

#ifdef IC_PM
   unsigned long ulId = window()->handle().sendEvent(MM_ITEMPOSITIONFROMID,
                                                     IEventData(itemId, false), 0);
   if (ulId == MIT_NONE)

   {
     ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
   } /* endif */
#endif //IC_PM


#ifdef IC_MOTIF
   long  ulId=MIT_NONE;
   if (XmIsRowColumn(handle()))
   {
      IWindowHandle hwnd = IWindow::handleWithParent(itemId, handle());
      Cardinal numChildren = 0;
      WidgetList submenuList = 0;
      XtVaGetValues(handle(),
                    XmNnumChildren, &numChildren,
                    XmNchildren, &submenuList,
                    NULL);
      ulId = numChildren - 1;
      while (ulId > MIT_NONE)
      {
         ITRACE_ALL(IString("ISubmenu::cursor hwnd=") +
                    IString(hwnd.asUnsigned()).d2x() +
                    IString(" index=") + IString(ulId) );
         if (submenuList[ulId] == hwnd)
            break;
         else
            ulId--;
      }
   }
   if (ulId <= MIT_NONE)
   {
      ITHROWLIBRARYERROR1(IC_INVALID_MENUITEM,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable,
                          "ISubmenu");
   } /* endif */
#endif //IC_MOTIF

   ISubmenu::Cursor cursor(*this);
   cursor.lCurrent = ulId;
   ITRACE_ALL( IString("ISubmenu::cursor lCurrent=") + IString(ulId) );
   return cursor;
}

/*------------------------------------------------------------------------------
| ISubmenu::undo                                                               |
| Undo changes. Restores previous menu-items and state                         |
------------------------------------------------------------------------------*/
ISubmenu&  ISubmenu::undo()
{
  IMODTRACE_DEVELOP("ISubmenu::undo");

  if (pItemList) {
     ISubmenuData* pList = pItemList;
     ISubmenuData* pNextRec = pList;
     while(pNextRec) {

        switch (pNextRec->recordType) {
        case addRecord:
           Inherited::deleteItem(pNextRec->pMenuItem->id());
           break;
        case changeRecord:
           Inherited::setItem(*pNextRec->pMenuItem);
           break;
        case removeRecord:
#ifdef IC_PMWIN
           Inherited::addItem(*pNextRec->pMenuItem);
#endif //IC_PMWIN
#ifdef IC_MOTIF
          ITRACE_ALL( IString("undo: managing itemId=") +
                      IString(pNextRec->pMenuItem->id()) +
                      IString(" for submenu handle=") +
                      IString(this->window()->handle().asUnsigned()).d2x() );
           Widget itemHwnd = XtNameToWidget((Widget)handle(),
              (char *)((IString("*")) + (IString(pNextRec->pMenuItem->id()))));
           XtManageChild(itemHwnd);
#endif //IC_MOTIF
        } /* endswitch */

        ISubmenuData* pTemp = pNextRec;
        pNextRec = pNextRec->pNext;
        delete pTemp->pMenuItem;
        delete pTemp;
     } /* endwhile */
     pItemList = 0;
  } /* endif */
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::setUndoChangeItem                                                  |
|  Save a change so that it can be undone                                      |
------------------------------------------------------------------------------*/
void ISubmenu::setUndoChangeItem(const IMenuItem& origItem)
{
  IMODTRACE_DEVELOP("ISubmenu::setUndoChangeItem");

  if (pItemList) {
     ISubmenuData* pList = pItemList;
     ISubmenuData* pNextRec = pList;
     while(pNextRec) {
        if (pNextRec->pMenuItem->id() == origItem.id() ) {
           if (pNextRec->recordType == changeRecord ||
               pNextRec->recordType == addRecord)
               return;
        } /* endif */
        pNextRec = pNextRec->pNext;
     } /* endif */

     IMenuItem* pNewItem = new IMenuItem(origItem);
     ISubmenuData* pNew = new ISubmenuData;
     pNew->recordType  = changeRecord;
     pNew->pMenuItem   = pNewItem;
     pNew->pNext       = pItemList;
     pItemList         = pNew;
  } else {
     IMenuItem* pNewItem = new IMenuItem(origItem);
     ISubmenuData* pList    = new ISubmenuData;
     pList->recordType  = changeRecord;
     pList->pMenuItem   = pNewItem;
     pList->pNext       = 0;
     pItemList          = pList;
  } /* endif */
}

/*------------------------------------------------------------------------------
| ISubmenu::setUndoAddItem                                                     |
|  Save an add so that it can be undone                                        |
------------------------------------------------------------------------------*/
void ISubmenu::setUndoAddItem(unsigned long itemId)
{
  IMODTRACE_DEVELOP("ISubmenu::setUndoAddItem");

  IMenuItem* pNewItem = new IMenuItem(itemId);

  if (pItemList) {
     ISubmenuData* pNew  = new ISubmenuData;
     pNew->recordType  = addRecord;
     pNew->pMenuItem   = pNewItem;
     pNew->pNext       = pItemList;
     pItemList         = pNew;
  } else {
     ISubmenuData* pList  = new ISubmenuData;
     pList->recordType  = addRecord;
     pList->pMenuItem   = pNewItem;
     pList->pNext       = 0;
     pItemList          = pList;
  } /* endif */
}

/*------------------------------------------------------------------------------
| ISubmenu::addItem                                                            |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addItem(IMenuItem&          menuItem,
                            unsigned long       intoSubmenuId )
{
  IFUNCTRACE_ALL();
#ifdef IC_PMWIN
  Inherited::addItem(menuItem);
#endif //IC_PMWIN
#ifdef IC_MOTIF
  ITRACE_ALL( IString("adding itemId=") + IString(menuItem.id()) +
              IString(" to submenu handle=") +
              IString(this->window()->handle().asUnsigned()).d2x() );
  Inherited::addItem(menuItem, this->id());
#endif //IC_MOTIF

  setUndoAddItem(menuItem.id());
  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::addSeparator                                                       |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addSeparator(unsigned long itemId,
                                 unsigned long intoSubmenuId)
{
  IFUNCTRACE_ALL();
  Inherited::addSeparator(itemId, 0);

  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::addSeparator                                                       |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addSeparator(unsigned long itemId)
{
  IFUNCTRACE_ALL();
  // This derived function changes the meaning of the single argument
  // from the base class IMenu.  In IMenu, its the intoSubmenuId.  In
  // this class, its the itemId.   This is to preserve compatibility
  // with prior releases.
  Inherited::addSeparator(itemId, 0);

  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::addText                                                            |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addText ( unsigned long       itemId,
                              const char*         itemText,
                              unsigned long       intoSubmenuId )
{
  IFUNCTRACE_ALL();
  Inherited::addText(itemId, itemText);

  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::addText                                                            |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addText( unsigned long       itemId,
                             const IResourceId  &newItemTextResId ,
                             unsigned long       intoSubmenuId )
{
  IFUNCTRACE_ALL();
  Inherited::addText(itemId, newItemTextResId);

  return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::addBitmap                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addBitmap    ( unsigned long       itemId,
                                   const IResourceId   &newItemBitmapResId,
                                   unsigned long       intoSubmenuId )
{
  IFUNCTRACE_ALL();
  Inherited::addBitmap(itemId, newItemBitmapResId);

  return *this;
}

/*------------------------------------------------------------------------------
  Add bitmap menu-item from .rc
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addBitmap    ( unsigned long itemId,
                                   unsigned long newItemBitmapResId,
                                   unsigned long intoSubmenuId )
{
  IFUNCTRACE_ALL();
  return addBitmap(itemId, IResourceId(newItemBitmapResId));
}


/*------------------------------------------------------------------------------
| ISubmenu::addBitmap                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addBitmap    ( unsigned long       itemId,
                                   const IBitmapHandle &itemBitmap ,
                                   unsigned long       intoSubmenuId )
{
  Inherited::addBitmap(itemId, itemBitmap);

  return *this;
}

/*------------------------------------------------------------------------------
| bool ISubmenu::Cursor::setToNext                                             |
------------------------------------------------------------------------------*/
bool ISubmenu::Cursor::setToNext ()
{
#ifdef IC_PMWIN
  if (lCurrent >= 0 && lCurrent < pSubmenu->numberOfItems() - 1 ) {
     ++lCurrent;
     skipRemoved();   //27045
     return (lCurrent < 0) ? false : true;   //27292
  } else {
     lCurrent = -1L;
     return false;
  } /* endif */
#endif //IC_PMWIN
#ifdef IC_MOTIF
  lCurrent = IMenuData::positionCursor(
     this->pSubmenu->window()->handleForChildCreation(),
     lCurrent,
     IMenuData::kNext);
  skipRemoved();   //27045
  return isValid();
#endif //IC_MOTIF
}


/*------------------------------------------------------------------------------
| ISubmenu::Cursor::skipRemoved                                                |
| Starts at the current position in the cursor and increments it (lCurrent)    |
| passed any "removed" items until it finds the first valid item at or beyond  |
| the original cursor position.  PRIVATE METHOD                                |
------------------------------------------------------------------------------*/
void ISubmenu::Cursor::skipRemoved (bool skipForward )
{
#ifdef IC_MOTIF
  IFUNCTRACE_ALL();
  ITRACE_ALL( IString(" lCurrent=") + IString(lCurrent) +
              IString(" skipForward=") + IString(skipForward ? "true":"false") );

  //27045 - skip over any items marked "removed" in submenu's pItemList chain of temp changes
  if (pSubmenu->pItemList)
  {            // only check if we have made changes
    bool currentMayBeRemovedItem = true;
    while (currentMayBeRemovedItem && isValid())
    {
      IMenuItem possibleItem = pSubmenu->elementAt(*this);
      ISubmenuData* current = pSubmenu->pItemList;
      int gone = 0;       //counter to see if repeated add/removes end up with net remove
      while (current)
      {
        if ( possibleItem.index() == current->pMenuItem->index() )
        {
          if ( current->recordType == removeRecord )
            ++gone;
          else if ( current->recordType == addRecord )
            --gone;
        }
        current = current->pNext;
      } //endwhile

      if (gone)
      {     //more removeRecords than addRecords (should only be 1 more max)
        ITRACE_ALL(IString("Skipping id=") + IString(possibleItem.id()) +
                   IString(" index=") + IString(possibleItem.index()) +
                   IString(" pItemList id=") + IString(current->pMenuItem->id()));
        lCurrent = IMenuData::positionCursor(
           this->pSubmenu->window()->handleForChildCreation(),
           lCurrent,
           skipForward ? IMenuData::kNext : IMenuData::kPrevious);
      }
      else
        currentMayBeRemovedItem = false;
    } //endWhile currentMayBeRemovedItem
  } //endif pSubmenu->pItemList
#endif // endof 27045 ===================================================

  return;
}


/*------------------------------------------------------------------------------
| ISubmenu::add                                                                |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::add( IMenuItem&  menuItem, Cursor&   cursor)
{
   IASSERTPARM(cursor.isValid());              //27292
   menuItem.setIndex(cursor.lCurrent);
   return cursor.pSubmenu->addItem(menuItem);
}

/*------------------------------------------------------------------------------
| ISubmenu::addAsNext                                                          |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addAsNext ( IMenuItem&  menuItem, Cursor&   cursor)
{
   IASSERTPARM(cursor.isValid());              //27292
   menuItem.setIndex(cursor.lCurrent + 1);     //27292
//   menuItem.setIndex(cursor.lCurrent);
   return cursor.pSubmenu->addItem(menuItem);
}

/*------------------------------------------------------------------------------
| ISubmenu::elementAt                                                          |
------------------------------------------------------------------------------*/
IMenuItem ISubmenu::elementAt ( const IMenu::Cursor&   cursor) const
{
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   IMenuItem mn(0);
   return mn;
}

/*------------------------------------------------------------------------------
| ISubmenu::deleteAt                                                           |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::deleteAt ( IMenu::Cursor&   cursor)
{
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::removeSubmenuAt                                                    |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::removeSubmenuAt ( IMenu::Cursor&   cursor)
{
   ITHROWLIBRARYERROR(IC_MEMBER_ACCESS_ERROR,
                      IBaseErrorInfo::invalidRequest,
                      IException::recoverable);
   return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::add  (hidden IMenu::Cursor version)                                |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::add ( IMenuItem&  menuItem, IMenu::Cursor&   cursor)
{
   ITHROWLIBRARYERROR1(IC_MEMBER_ACCESS_ERROR,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable,
                       "ISubmenu");
   return *this;
}

/*------------------------------------------------------------------------------
| ISubmenu::addAsNext (hidden IMenu::Cursor version)                           |
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::addAsNext ( IMenuItem&  menuItem, IMenu::Cursor&   cursor)
{
   ITHROWLIBRARYERROR1(IC_MEMBER_ACCESS_ERROR,
                       IBaseErrorInfo::invalidRequest,
                       IException::recoverable,
                       "ISubmenu");
   return *this;
}

/*------------------------------------------------------------------------------
  Have the cursor point to the first element
------------------------------------------------------------------------------*/
bool ISubmenu::Cursor::setToFirst ()
{
#ifdef IC_PMWIN
  lCurrent = 0;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  lCurrent = IMenuData::positionCursor(
     this->pSubmenu->window()->handleForChildCreation(), 0, IMenuData::kFirst);
#endif //IC_MOTIF
  skipRemoved();   //27045
  return isValid();
}

/*------------------------------------------------------------------------------
  Have the cursor point to the previous element
------------------------------------------------------------------------------*/
bool ISubmenu::Cursor::setToPrevious ()
{
#ifdef IC_PMWIN
  --lCurrent;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  lCurrent = IMenuData::positionCursor(
     this->pSubmenu->window()->handleForChildCreation(),
     lCurrent,
     IMenuData::kPrevious);
#endif //IC_MOTIF
  skipRemoved(false);   //27045 - false => skip backward
  return isValid();
}

/*------------------------------------------------------------------------------
  Have the cursor point to the last element
------------------------------------------------------------------------------*/
bool ISubmenu::Cursor::setToLast ()
{
#ifdef IC_PMWIN
  lCurrent = pSubmenu->numberOfItems() - 1;
#endif //IC_PMWIN
#ifdef IC_MOTIF
  lCurrent = IMenuData::positionCursor(
     this->pSubmenu->window()->handleForChildCreation(),
     lCurrent,
     IMenuData::kLast);
#endif //IC_MOTIF
  skipRemoved(false);   //27045 - false => skip backward
  return isValid();
}

/*------------------------------------------------------------------------------
  Return true if the cursor is valid, false if it is not
------------------------------------------------------------------------------*/
bool ISubmenu::Cursor::isValid () const
{
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
  Invalidate the cursor
------------------------------------------------------------------------------*/
void ISubmenu::Cursor::invalidate ()
{
  lCurrent = -1L;
}

/*------------------------------------------------------------------------------
  Get menu item at cursor position.
------------------------------------------------------------------------------*/
IMenuItem ISubmenu::elementAt ( const Cursor& cursor ) const
{
   IASSERTPARM(cursor.isValid());
#ifdef IC_PMWIN
   return cursor.pSubmenu->menuItem(cursor.pSubmenu->menuItemId(cursor));
#endif
#ifdef IC_MOTIF    //27045, 27292
//   IMenuItem item =  cursor.pSubmenu->menuItem(cursor.pSubmenu->menuItemId(cursor));
//   item.setIndex(cursor.lCurrent);    //be sure we point to the correct positional item
   IWindowHandle widget;
   WidgetList children = NULL;
   Cardinal numChildren = 0;
   XtVaGetValues (cursor.pSubmenu->hwndHandle,
                  XmNchildren, &children,
                  XmNnumChildren, &numChildren,
                  NULL);
   if (numChildren <= cursor.lCurrent)
     ITHROWLIBRARYERROR1(IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable,
                        "elementAt(ISubmenu::Cursor)");
   widget = children[cursor.lCurrent];
#ifdef IC_TRACE_ALL
   {
      int i = cursor.lCurrent;
      short positionIndex = 0;
      IString str("SUBMENUDUMP");
      str += " child ";
      str += IString(i);
      str += " (" + IString(XtName(children[i])) + ")";
      XtVaGetValues( children[i],
                     XmNpositionIndex, &positionIndex, 0);
      str += IString(" positionIndex=") + IString(positionIndex);
      str += IString(" handle=") + IString((unsigned long)children[i]).d2x();
      str += XtIsManaged( children[i] ) ? "    Managed " : " NotManaged ";
      str +=  children[i]->core.being_destroyed ? "  DELETING " : " ";

      str += " " + IString(XtClass( children[i] )->core_class.class_name );
      Dimension x, y, h, w;
      XtVaGetValues( children[i],
                     XmNx, &x, XmNy, &y, XmNheight, &h, XmNwidth, &w,
                     0);
      str += " pos=(" + IString(x) + "," + IString(y) + ")";
      str += " size=(" + IString(w) + "," + IString(h) + ")";
      ITRACE_ALL(str);
   }
#endif
   IMenuItem item = cursor.pSubmenu->Inherited::menuItemWithHandle(widget);
   return item;
#endif
}

/*------------------------------------------------------------------------------
  Delete menu item at cursor position.
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::deleteAt      ( Cursor&           cursor   )
{
   IASSERTPARM(cursor.isValid());
#ifdef IC_PMWIN
   return cursor.pSubmenu->deleteItem(cursor.pSubmenu->menuItemId(cursor));
#endif
#ifdef IC_MOTIF   //27045
  ITRACE_ALL( IString("deleteAt lCurrent=") + IString(cursor.lCurrent) +
              IString(" from submenu handle=") +
              IString(this->window()->handle().asUnsigned()).d2x() );
  // Must REMOVEITEM rather than DELETEITEM as in IMenu::deleteItem().
  // DELETEITEM could destroy a bitmap that we want to be able to restore.
  IWindowHandle itemHwnd;
  WidgetList children = NULL;
  Cardinal numChildren = 0;
  XtVaGetValues (window()->handle(),
                 XmNchildren, &children,
                 XmNnumChildren, &numChildren,
                 NULL);
  if ( cursor.lCurrent < numChildren )
    itemHwnd = children[cursor.lCurrent];
  else
    ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  // Save the item while we still got it
  // We're using the default copy constructor
  IMenuItem* pNewItem = new IMenuItem(Inherited::menuItemWithHandle(itemHwnd));
  XtUnmanageChild(itemHwnd);
  // Save the item in the list
  if (pItemList) {
     ISubmenuData* pNew = new ISubmenuData;
     pNew->recordType   = removeRecord;
     pNew->pMenuItem    = pNewItem;
     pNew->pNext        = pItemList;
     pItemList          = pNew;
  } else {
     ISubmenuData* pList = pItemList;
     pList              = new ISubmenuData;
     pList->recordType  = removeRecord;
     pList->pMenuItem   = pNewItem;
     pList->pNext       = 0;
     pItemList          = pList;
  } /* endif */

  cursor.setToPrevious();       //27406
  return *this;
#endif //IC_MOTIF

}

/*------------------------------------------------------------------------------
  Remove submenu at cursor position.
------------------------------------------------------------------------------*/
ISubmenu& ISubmenu::removeSubmenuAt ( Cursor&           cursor   )
{
   IASSERTPARM(cursor.isValid());
#ifdef IC_PMWIN
   return cursor.pSubmenu->removeSubmenu(menuItemId(cursor));
#endif
#ifdef IC_MOTIF
  ITRACE_ALL( IString("removeSubmenuAt lCurrent=") + IString(cursor.lCurrent) +
              IString(" from submenu handle=") +
              IString(this->window()->handle().asUnsigned()).d2x() );
  IWindowHandle itemHwnd;
  WidgetList children = NULL;
  Cardinal numChildren = 0;
  XtVaGetValues (window()->handle(),
                 XmNchildren, &children,
                 XmNnumChildren, &numChildren,
                 NULL);
  if ( cursor.lCurrent < numChildren )
    itemHwnd = children[cursor.lCurrent];
  else
    ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable );
  // Save the item while we still got it (one to archive, one to change)
  IMenuItem pNewItem(Inherited::menuItemWithHandle(itemHwnd));
  IMenuItem origItem(pNewItem);

   unsigned long afStyle = pNewItem.style();
   afStyle  &= ~MIS_SUBMENU;
   pNewItem.setStyle(afStyle);
  // Reset the submenu handle in one item and then change the menu
   pNewItem.setSubmenuHandle(0);
   Inherited::setItem(pNewItem);

   // Finally archive the original item removed and exit
   setUndoChangeItem(origItem);
   return *this;
#endif
}

