/*******************************************************************************
* CLASS NAME:  ITextChangeEvent                                                *
*                                                                              *
* DESCRIPTION   This provides the implementation of the class                  *
*             ITextChangeEvent class                                           *
*                                                                              *
*                                                                              *
* COPYRIGHT:                                                                   *
*   Licensed Materials - Property of IBM                                       *
*   (C) Copyright IBM Corporation 1994, 1995                                   *
*   All Rights Reserved                                                        *
*   US Government Users Restricted Rights - Use, duplication, or               *
*   disclosure                                                                 *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

extern "C" {
  #include <iwindefs.h>
}

#ifdef IC_MOTIF
  #include <Xm/Xm.h>
#endif //IC_MOTIF

#include <itextevt.hpp>
#include <iexcept.hpp>
#include <ihandle.hpp>
#include <iwindow.hpp>
#include <istring.hpp>
#include <ientryfd.hpp>
#include <imle.hpp>
#include <ispinbas.hpp>
#include <ispinszh.hpp>
#include <ikeyevt.hpp>

/*-----------------------------------------------------------------------------
| ITextChangeEvent::ITextChangeEvent                                           |
|                                                                              |
| Constructor to create an instance from another ITextChangeEvent              |
------------------------------------------------------------------------------*/
ITextChangeEvent :: ITextChangeEvent (const ITextChangeEvent& evt )
                 : IControlEvent(evt)
                 , verifyText(0)
#ifdef IC_WIN
                 , fAllowChange(true)
                 , startPos(0)
                 , endPos(0)  
                 , cursorLinePosition(0)
                 , originalText(0)
#endif //IC_WIN
{
  // Create an IString object containing the proposed text as modified so far
  if (evt.verifyText)            //if it has text
    verifyText = new IString ( evt.proposedText().asString() );
}

/*-----------------------------------------------------------------------------
| ITextChangeEvent::ITextChangeEvent                                           |
|                                                                              |
| Constructor to create an instance from an IEvent                             |
------------------------------------------------------------------------------*/
ITextChangeEvent :: ITextChangeEvent ( IEvent& evt )
                 : IControlEvent(evt)
                 , verifyText(0)
#ifdef IC_WIN
                 , fAllowChange(true)
                 , startPos(0)
                 , endPos(0)  
                 , cursorLinePosition(0)
                 , originalText(0)
#endif //IC_WIN

{
#ifdef IC_WIN
  IKeyboardEvent keyEvent(evt);
  if (keyEvent.isCharacter())
  {
    verifyText = new IString ((char*)evt.parameter1());
  }
  else
  {
    verifyText = new IString();
  }

  IEntryField* ef = dynamic_cast<IEntryField *>(evt.window());

  // if valid entryfield
  if (ef)
    {
      //cache original text
      originalText = ef->text();

      //determine start/end position
      if (ef->hasSelectedText())
      {
        startPos = ef->selectedRange().lowerBound();
        endPos = startPos + ef->selectedTextLength();
      }
      else
      {
        startPos = endPos = ef->cursorPosition();
      }

    } 

  IMultiLineEdit* mle = dynamic_cast<IMultiLineEdit *>(evt.window());

  // if valid mle
  if (mle)
    {
      //cache original text
      originalText = mle->text();

      //determine start/end position
      if (mle->hasSelectedText())
      {
        startPos = mle->selectedRange().lowerBound();
        endPos = startPos + mle->selectedTextLength();
      }
      else
      {
        startPos = endPos = mle->cursorPosition();
      }

    } 

  IBaseSpinButton* sb = dynamic_cast<IBaseSpinButton *>(evt.window());

  // if valid spinbutton
  if (sb)
    {
      IWindowHandle efHandle = ISpinButtonStatics::buddyHandle( sb->handle() );
      IEntryField spinEf = IEntryField(efHandle);

      //cache original text
      originalText = spinEf.text();

      //determine start/end position
      if (spinEf.hasSelectedText())
      {
        startPos = spinEf.selectedRange().lowerBound();
        endPos = startPos + spinEf.selectedTextLength();
      }
      else
      {
        startPos = endPos = spinEf.cursorPosition();
      }

    } 


#endif //IC_WIN
#ifdef IC_MOTIF
  // Create an IString object containing the proposed text, for ease of use
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  if (tcbs && (tcbs->reason == XmCR_MODIFYING_TEXT_VALUE))       // if there is something and only if right kind of IEvent --defect 3495
    verifyText = new IString (tcbs->text->ptr, tcbs->text->length);
#endif //IC_MOTIF

}

/*-----------------------------------------------------------------------------
| ITextChangeEvent::~ITextChangeEvent                                          |
|                                                                              |
| Destructor                                                                   |
------------------------------------------------------------------------------*/
ITextChangeEvent :: ~ITextChangeEvent ( )
{
  if (verifyText)
    delete verifyText;
}


bool ITextChangeEvent :: isChangeAllowed ( ) const
  {
#ifdef IC_MOTIF
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  return tcbs->doit;
#endif //IC_MOTIF
#ifdef IC_WIN
  return fAllowChange;
#endif //IC_WIN
  }

unsigned long ITextChangeEvent :: startPosition ( ) const
  {
#ifdef IC_MOTIF
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  return (unsigned long) tcbs->startPos;
#endif //IC_MOTIF
#ifdef IC_WIN
  return startPos;
#endif //IC_WIN
  }

unsigned long ITextChangeEvent :: endPosition ( ) const
  {
#ifdef IC_MOTIF
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  return (unsigned long) tcbs->endPos;
#endif //IC_MOTIF
#ifdef IC_WIN
  return endPos;
#endif //IC_WIN

  }

IString ITextChangeEvent :: proposedText ( ) const
  {
  return *verifyText;
  }

ITextChangeEvent& ITextChangeEvent :: setProposedText ( IString text )
  {
  *verifyText = text;
  return *this;
  }

IString ITextChangeEvent :: resultingText ( ) const
{
#ifdef IC_WIN
  return
    IString( originalText.subString( 1, startPosition() + cursorLinePosition ) ) +
    proposedText() +
    originalText.subString( endPosition()+ cursorLinePosition + 1 );
#endif //IC_WIN

#ifdef IC_MOTIF
  IString originalText;

  ITextControl* textControl = dynamic_cast<ITextControl*>( window() );
  if( textControl )
	originalText = textControl->text();

  return
    IString( originalText.subString( 1, startPosition() ) ) +
    proposedText() +
    originalText.subString( endPosition() + 1 );
#endif //IC_MOTIF
}

ITextChangeEvent& ITextChangeEvent :: allowChange ( bool allow )
  {
#ifdef IC_WIN
  fAllowChange = allow;
#endif //IC_WIN
#ifdef IC_MOTIF
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  tcbs->doit = allow;
#endif //IC_MOTIF
  return *this;
  }

ITextChangeEvent& ITextChangeEvent :: setStartPosition ( unsigned long pos )
  {
#ifdef IC_WIN
  startPos = pos;
#endif //_WIN
#ifdef IC_MOTIF
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  tcbs->startPos =  pos;
#endif //IC_MOTIF
  return *this;
  }

ITextChangeEvent&  ITextChangeEvent :: setEndPosition  ( unsigned long pos )
  {
#ifdef IC_WIN
  endPos = pos;
#endif //IC_WIN
#ifdef IC_MOTIF
  XmTextVerifyCallbackStruct *tcbs = (XmTextVerifyCallbackStruct *)parameter1().asUnsignedLong();
  tcbs->endPos =  pos;
#endif //IC_MOTIF
  return *this;
  }

