// Revision: 22 1.10.1.19 source/ui/baseapp/pmwin/imenu.cpp, menu, ioc.v400, 990114 
/*******************************************************************************
* FILE NAME: imenu.cpp                                                         *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in imenu.cpp                                                               *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/

/*******************************************************************************
|                                                                              |
|  IMPORTANT NOTE:                                                             |
|  Any call within this compiliation unit to an IMenu virtual function such    |
|  as setText, setItem, etc. should be qualified as such:                      |
|     IMenu::setText, IMenu::setItem                                           |
|  This must be done because the corresponding versions of these virtual       |
|  functions in ISubmenu, will call their IMenu counterparts to change a menu  |
|  item, then save the change made in a list so that it can be undone.         |
|  If we don't qualify the calls here with IMenu we can get ISubmenu::setText  |
|  rather than IMenu::setText and we'll needlessly save that change in the list|
|  This can cause unwarranted exceptions at undo time.                         |
|                                                                              |
*******************************************************************************/
#pragma priority( -2147481424 )

extern "C"
{
  #define INCL_WINFRAMEMGR
  #define INCL_WINWINDOWMGR
  #define INCL_WINMENUS
  #define INCL_GPIPRIMITIVES
  #define INCL_WINSYS
  #define INCL_GPIBITMAPS
  #include <iwindefs.h>
  #include <string.h>
}

#include <imenu.hpp>
#include <ibidiset.hpp>
#include <icconst.h>
#include <icolor.hpp>
#include <iexcept.hpp>
#include <iframe.hpp>
#include <imenuprv.hpp>
#include <imnitem.hpp>
#include <inotifev.hpp>
#include <iplatfrm.hpp>
#include <irect.hpp>
#include <ireslib.hpp>
#include <isubmenu.hpp>
#include <itrace.hpp>
#include <iwindow.hpp>

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _IMENU_CPP_
  #include <ipagetun.h>
#endif

const IMenu::Style
  IMenu::noStyle            = 0,
  IMenu::classDefaultStyle  = 0,
  IMenu::visible            = 0;

  IMenu::Style IMenu::currentDefaultStyle = 0;

// maintain list of bitmaps so that they can be freed
typedef struct _BITMAPLIST {

 IBitmapHandle* pBitmap;
 _BITMAPLIST*   pNext;
} BITMAPLIST;

/*------------------------------------------------------------------------------
| IMenuData::IMenuData                                                         |
|                                                                              |
| This class contains menu data for the class                                  |
------------------------------------------------------------------------------*/
IMenuData::IMenuData ( )
  : fUpdateAllowed(true)
{ }

/*------------------------------------------------------------------------------
| IMenuData::~IMenuData                                                        |
|                                                                              |
| Destructor here for page tuning.                                             |
------------------------------------------------------------------------------*/
IMenuData::~IMenuData ( )
{ }


#ifdef IC_PM
/*------------------------------------------------------------------------------
| queryItem                                                                    |
|  Local function used to query menu-item. Also used to check for valid        |
|  menu item and throw exception.                                              |
------------------------------------------------------------------------------*/
MENUITEM queryItem( const IWindowHandle& menuHandle,
                    unsigned long        itemId )
{
  IMODTRACE_DEVELOP("IMenu::queryItem");
  MENUITEM miItem;
  if(!menuHandle.sendEvent( MM_QUERYITEM,
                            IEventParameter1((unsigned short)itemId,
                                             true),
                            IEventParameter2(&miItem)).asUnsignedLong())
  {
     ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
  } /* endif */
  return miItem;
}
#endif
#ifdef IC_WIN
/*------------------------------------------------------------------------------
| queryItem                                                                    |
|  Local function used to query menu-item. Also used to check for valid        |
|  menu item and throw exception.                                              |
------------------------------------------------------------------------------*/
MENUITEMINFO queryItem( const IMenuHandle& menuHandle,
                        unsigned long      itemId,
                        bool               position,
                        char*              text )
{
  IMODTRACE_DEVELOP("IMenu::queryItem");
  char         menuText[IC_MENUTEXTLENGTH];
  MENUITEMINFO miItem;
  miItem.cbSize = sizeof(MENUITEMINFO);
  miItem.fMask = MIIM_STATE | MIIM_ID | MIIM_SUBMENU | MIIM_TYPE | MIIM_DATA;
  if (text)
    miItem.dwTypeData = text;                   // returning text
  else
    miItem.dwTypeData = menuText;               // throwing away text
  miItem.cch = IC_MENUTEXTLENGTH;

  if (!(GetMenuItemInfo( menuHandle, (unsigned int)itemId, position, &miItem )))
  {
    ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                        IBaseErrorInfo::invalidRequest,
                        IException::recoverable);
  }
  return miItem;
}

/*------------------------------------------------------------------------------
| queryItemState                                                               |
|  Local function used to query menu-item. Also used to check for valid        |
|  menu item and throw exception.                                              |
------------------------------------------------------------------------------*/
unsigned int queryItemState( IFrameWindow*      owner,
                             const IMenuHandle& baseMenuHandle,
                             unsigned long      itemId,
                             long*              itemPosition,
                             IMenuHandle*       menuHandle )
{
  IMODTRACE_DEVELOP("IMenu::queryItem");
  IMenuHandle tempHMenu;
  unsigned int menuState = GetMenuState( baseMenuHandle,
                                         (unsigned int)itemId, MF_BYCOMMAND );
  if ( menuState == 0xFFFFFFFF )
  {
    long itemPos;
    if (IMenuPrivate::locateMenuItem( owner, baseMenuHandle, itemId,
                                      &itemPos, &tempHMenu ))
    {
      menuState = GetMenuState( tempHMenu,
                                (unsigned int)itemPos, MF_BYPOSITION );

      // If pointer passed to get position, then return it
      if (itemPosition)
        *itemPosition = itemPos;
      if (menuHandle)
        *menuHandle = tempHMenu;

    }
    else
    {
      ITHROWLIBRARYERROR( IC_INVALID_MENUITEM,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable );
    }
  }
  // If popup item, then mask out count value in upper byte
  if (menuState & MF_POPUP)
    menuState &= 0x000000FF;

  return menuState;
}
#endif

/*------------------------------------------------------------------------------
| IMenu::IMenu                                                                 |
------------------------------------------------------------------------------*/
IMenu::IMenu ( )
  : pBitmapList(0),
#ifdef IC_WIN
    fMenuData(new IMenuData()), fhMenu(0), fowner(0), fmenuFrame(0)
#else
    fMenuData(0), fWindow(0), fMenuWindowNewdInCtor(false)
#endif
{ IMODTRACE_DEVELOP("IMenu::IMenu" ); }

/*------------------------------------------------------------------------------
  IMenu constructor for an existing menu handle
------------------------------------------------------------------------------*/
IMenu::IMenu ( const IMenuHandle& menuHandle )
  : pBitmapList(0),
#ifdef IC_WIN
    fMenuData(new IMenuData()), fhMenu(0), fowner(0), fmenuFrame(0)
#else
    fMenuData(0), fWindow(0), fMenuWindowNewdInCtor(false)
#endif
{
  IMODTRACE_DEVELOP("IMenu::IMenu(menuHandle)");
  ITRACE_ALL(IString(" Menu handle is: ")+IString((unsigned long)menuHandle).d2x());
  IASSERTPARM(menuHandle!=0);

#ifdef IC_PM
  IWindow* hndWindow = IWindow::windowWithHandle((IWindowHandle)menuHandle);
  if (!hndWindow) {
     hndWindow = new MenuWindow((IWindowHandle)menuHandle,NULL);
     hndWindow->setAutoDestroyWindow(false);
  }

  const IAttributeName menuName(IMenuAttribute::menuAttrName);
  IMenuAttribute menuAttr(this);
  hndWindow->addOrReplaceAttribute(menuName, menuAttr);
  
  setWindow(hndWindow);
#endif

#ifdef IC_WIN
  /*------------------------------------------------------------------*/
  /* Save menu handle and search for owner of menu                    */
  /*------------------------------------------------------------------*/
  fhMenu = menuHandle;
  FRAMEINFO frameInfo;
  frameInfo.hmenu  = menuHandle;
  frameInfo.owner  = 0;
  frameInfo.frame  = 0;

  /*------------------------------------------------------------------*/
  /* Enumerate all of the top level windows (to find frame w/ table)  */
  /*------------------------------------------------------------------*/
  EnumWindows( (WNDENUMPROC)IMenuPrivate::EnumAllFrames,
               (LPARAM)&frameInfo );

  /*------------------------------------------------------------------*/
  /* If owner parameter set (meaning found), then use it              */
  /*------------------------------------------------------------------*/
  if (frameInfo.owner)
  {
    fowner     = frameInfo.owner;
    fmenuFrame = frameInfo.frame;
//AJ: debug/verification
    HMENU menu = GetMenu(fowner);
    bool ans = false;
    if (menu == fhMenu)
       ans = true;
//AJ: end debug/verif
  }
#endif
}


/*------------------------------------------------------------------------------
| IMenu::~IMenu                                                                |
| Definition of pure virtual destructor so linker won't complain               |
------------------------------------------------------------------------------*/
IMenu :: ~IMenu()
{
   IMODTRACE_DEVELOP("IMenu::dtor");
   if (pBitmapList)
   {
      BITMAPLIST* pList = (BITMAPLIST*)pBitmapList;
      BITMAPLIST* pNextRec;
      pNextRec = pList->pNext;
      while (pNextRec) {
        BITMAPLIST* pTempRec = pNextRec;
        delete pNextRec->pBitmap;
        pNextRec = pNextRec->pNext;
        delete pTempRec;
      } /* endwhile */
      delete pList->pBitmap;
      delete pList;
   } /* endif */
   if (fMenuData)
     delete fMenuData;
#ifndef IC_WIN
	// This portion of code is only called for PM
  // Checking to see if we are using a MenuWindow or an IWindow
  // If we havce a MenuWindow then the window will be cleaned up with the 
  // IMRefCounted::removeRef
  // If fWindow is not a MenuWindow then IMenu did NOT allocate the memory therefore 
  // the releasing of the memory is up to the owner

  setWindow(0);

#endif
}


/*------------------------------------------------------------------------------
| IMenu::addItem                                                               |
| Add new menu item from IMenuItem structure.                                  |
------------------------------------------------------------------------------*/
IMenu& IMenu::addItem( IMenuItem&          newItem,
                       unsigned long       intoSubmenuId)
{
   IMODTRACE_DEVELOP("IMenu::addItem (1)");
#ifdef IC_WIN
   IMenuHandle hMenu = (intoSubmenuId) ?
                         submenuHandle(intoSubmenuId) : menuHandle();

   if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
   {
     // Depending on menu type, place identity into menu
     UINT  menu1 = (UINT)newItem.id();
     if (newItem.isSubmenu())
       menu1 = (UINT)(newItem.submenuHandle().asUnsigned());

     // Depending on menu type, place information into menu
     LPSTR menu2 = (LPSTR)(char*)newItem.text();
     if (newItem.isBitmap())
       menu2 = (LPSTR)(void*)newItem.bitmap();

     InsertMenu( hMenu,
                 (UINT)newItem.index(),
                 (UINT)(newItem.style() | newItem.attribute() | MF_BYPOSITION),
                 menu1, menu2 );
     // Refresh menu if item inserted
     if ( fMenuData->fUpdateAllowed )
       DrawMenuBar( fowner );

     // Ref count the bitmap if one passed in
     if (newItem.isBitmap() && newItem.bitmap() )
       refCountBitmap( menu2 );
   }
   else
   {
     // Construct a menu item info block and initialize header
     MENUITEMINFO mn;
     memset( &mn, 0, sizeof(MENUITEMINFO));
     mn.cbSize = sizeof(MENUITEMINFO);

     // Now fill out the block, taking information from the menu item passed in
     mn.fMask = MIIM_STATE | MIIM_ID | MIIM_SUBMENU | MIIM_TYPE;
     mn.wID           = (UINT)newItem.id();
     mn.fType         = (UINT)newItem.style();
     mn.fState        = (UINT)newItem.attribute();
     mn.hSubMenu      = newItem.submenuHandle();

     if (IPlatform::isWin9x())
     {
        // If Windows 95, reset the popup flag since that platform
        // does not allow it.
        mn.fType &= (unsigned long)(~MF_POPUP);
     }

     // Check for unavailable style on Windows and convert
     if ( newItem.extendedStyle() &
                       IMenuItem::unavailable.asExtendedUnsignedLong() )
       mn.fState |= MF_DISABLED;

     // In the data field, get the bitmap handle if item is a bitmap
     // otherwise get the text string for the item
     if (newItem.isBitmap())
       mn.dwTypeData    = (char *)((HBITMAP)newItem.bitmap());
     else
       mn.dwTypeData    = (char *)newItem.text();

     // Finally insert the item at the specified position
     if (InsertMenuItem( hMenu, (UINT)newItem.index(), TRUE, &mn ))
     {
       if ( IBidiSettings::isBidiSupported() )
       {
          if ( fowner )
          {
             // Give the menu the bidi attributes of the owner window.
             IBidiSettings
               bidiSettings( fowner );
             IMenuPrivate::setBidiAttributes( hMenu, bidiSettings );
          }
          else
          {
             // Give the menu default bidi attributes.
             IBidiSettings
               bidiSettings( IBidiSettings::applicationDefaults() );
             IMenuPrivate::setBidiAttributes( hMenu, bidiSettings );
          }
       }

       // Refresh menu if item inserted without error
       if ( fMenuData->fUpdateAllowed )
         DrawMenuBar( fowner );

       // If item was a bitmap, the update ref count for bitmap
       if (mn.fType & MFT_BITMAP)
         refCountBitmap( mn.dwTypeData );
     }
   }
   /*******************************************************************/
   /* Save any help id set in the item into the lookup table          */
   /*******************************************************************/
   if ( newItem.helpId() && newItem.id() )
     IMenuPrivate::setMenuItemHelpId( fmenuFrame, newItem.id(), newItem.helpId() );
#else
   MENUITEM mn;
   mn.iPosition    = (unsigned short)(newItem.index());
   mn.afStyle      = (unsigned short)(newItem.style());
   mn.afAttribute  = (unsigned short)(newItem.attribute());
   mn.id           = (unsigned short)(newItem.id());
   mn.hItem        = newItem.bitmap();
   mn.hwndSubMenu  = newItem.submenuHandle();

   IMenuHandle hwnd = (intoSubmenuId) ?
                         submenuHandle(intoSubmenuId) : menuHandle();

   long reply = hwnd.sendEvent( MM_INSERTITEM,
                                IEventData(&mn),
                                (newItem.text().length()) ?
                                  IEventData((char*)newItem.text()) :
                                  IEventData());

   if (reply == MIT_MEMERROR || reply == MIT_ERROR)
     ITHROWGUIERROR("MM_INSERTITEM");
   else
     newItem.setIndex(reply);

   if (mn.afStyle & MIS_BITMAP) {   //ref. count bitmap.
      if (mn.hItem) {
         if(!hwnd.sendEvent( MM_SETITEMHANDLE,
                             IEventData(mn.id),
                             IEventData(mn.hItem)).asUnsignedLong())
         {
            ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                               IBaseErrorInfo::invalidRequest,
                               IException::recoverable);
         } /* endif */
         refCountBitmap(mn.hItem);
      } else {
         ITHROWLIBRARYERROR(IC_NO_BITMAP_HANDLE,
                            IBaseErrorInfo::invalidRequest,
                            IException::recoverable);
      } /* endif */
   } else {
      if (mn.afStyle & MIS_TEXT)
         // In case we were called by ISubmenu
         IMenu::setText(mn.id, newItem.text());
   } /* endif */
#endif
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::addSubmenu                                                            |
| Create an empty menu and add it to the menu-item as submenu                  |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSubmenu( unsigned long itemId )
{
   IMODTRACE_DEVELOP("IMenu::addSubmenu(1)");
   IASSERTSTATE(itemId != FID_MENU); //menu item cannot be menu-bar ID

   IMenuItem mn = menuItem(itemId);
   if (mn.submenuHandle()) {
      ITHROWLIBRARYERROR(IC_ADD_SUBMENU_FAIL,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */
#ifdef IC_WIN
   HMENU hmenuMenu = CreatePopupMenu();
   if (hmenuMenu)
   {
     mn.setSubmenuHandle(hmenuMenu);
     if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
       IMenuPrivate::addToLookUpTable( fmenuFrame, hmenuMenu, itemId );
   }
#else
   HWND hwndMenu =   window()->create( mn.id(),
                      0,
                      0,
                      WC_MENU,
                      HWND_OBJECT,   /* 9732C */
                      window()->handle(),
                      IRectangle(),
                      0,
                      0,
                      IWindow::onTopOfSiblings);
   if (hwndMenu) {
       mn.setSubmenuHandle(hwndMenu);

       unsigned long afStyle = WinQueryWindowULong(mn.submenuHandle(),
                                                   QWL_STYLE);
       afStyle &= (unsigned long)
                    ~(MS_CONDITIONALCASCADE |  //Menu is create as menu-bar
                      MS_VERTICALFLIP |        //Change it.
                      MS_ACTIONBAR);
       WinSetWindowULong(mn.submenuHandle(), QWL_STYLE, afStyle);
   }
#endif
   IMenu::setItem(mn);  //set it.

   return *this;
}


/*------------------------------------------------------------------------------
| IMenu::addSubmenu                                                            |
| Load a menu from resource fileand add it to the menu-item as submenu         |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSubmenu( unsigned long      itemId,
                          const IResourceId& submenuResId )
{
   IMODTRACE_DEVELOP("IMenu::addSubmenu(2)");
   IASSERTPARM(itemId != FID_MENU); //menu item cannot be menu-bar ID

   IMenuItem mn = menuItem(itemId);
   if (mn.submenuHandle()) {        //cannot add if already has submenu.
      ITHROWLIBRARYERROR(IC_ADD_SUBMENU_FAIL,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */

   mn.setSubmenuHandle(submenuResId.resourceLibrary().loadMenu
                                             (submenuResId, owner()));
#ifdef IC_WIN
   // Add handle to lookup table to create mapping to id for Windows
   if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
     IMenuPrivate::buildLookUpTableFromResourceLoad( fmenuFrame,
                    (void*)mn.submenuHandle().asUnsigned(), submenuResId );
#else
   WinSetParent(mn.submenuHandle(), HWND_OBJECT, false);   /* 9732C */

   unsigned long afStyle = WinQueryWindowULong(mn.submenuHandle(), QWL_STYLE);
   WinSetWindowUShort(mn.submenuHandle(),  //Menu window ID is create with
                      QWS_ID,              //FID_MENU. Change it.
                      (unsigned short)itemId);
   afStyle &= (unsigned long)
                ~(MS_CONDITIONALCASCADE |  //Menu is create as menu-bar
                  MS_VERTICALFLIP |        //Change it.
                  MS_ACTIONBAR);
   WinSetWindowULong(mn.submenuHandle(), QWL_STYLE, afStyle);
#endif
   IMenu::setItem(mn);  //set it.
   return *this;
}


/*------------------------------------------------------------------------------
| IMenu::addSeparator                                                          |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSeparator( unsigned long intoSubmenuId )
{
   return IMenu::addSeparator( 0, intoSubmenuId );
}


/*------------------------------------------------------------------------------
| IMenu::setSubmenu                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::setSubmenu( unsigned long        itemId,
                          const IResourceId&   submenuResId )
{
   IMODTRACE_DEVELOP("IMenu::setSubmenu(2)");
   IASSERTPARM(itemId != FID_MENU); //menu item cannot be menu-bar ID

   IMenuItem mn = menuItem(itemId);

#ifdef IC_WIN
   if (mn.isSubmenu())
     IMenuPrivate::removeFromLookUpTable( fmenuFrame,
                       (void*)mn.submenuHandle().asUnsigned(), itemId );
#else            // ModifyMenu will automatically destroy old pop-up
                 // and free memory associated with it
   if (mn.submenuHandle()) {
      WinDestroyWindow(mn.submenuHandle());
   } /* endif */
#endif

   mn.setSubmenuHandle( submenuResId.resourceLibrary().loadMenu
                                               (submenuResId, owner()));

#ifdef IC_WIN
   // Add handle to lookup table to create mapping to id for Windows
   if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
     IMenuPrivate::buildLookUpTableFromResourceLoad( fmenuFrame,
                 (void*)mn.submenuHandle().asUnsigned(), submenuResId );
#else
   unsigned long afStyle = WinQueryWindowULong(mn.submenuHandle(), QWL_STYLE);

   WinSetWindowUShort(mn.submenuHandle(),  //Menu window ID is create with
                      QWS_ID,              //FID_MENU. Change it.
                      (unsigned short)itemId);
   afStyle &= (unsigned long)
                ~(MS_CONDITIONALCASCADE |  //Menu is create as menu-bar
                  MS_VERTICALFLIP |        //Change it.
                  MS_ACTIONBAR);
   WinSetWindowULong(mn.submenuHandle(), QWL_STYLE, afStyle);
#endif
   IMenu::setItem(mn);  //set it.
   return *this;
}


/*------------------------------------------------------------------------------
| IMenu::setItem                                                               |
| Replace an existing menu-item data with IMenuItem                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::setItem( const IMenuItem&  menuItem )
{
   IMODTRACE_DEVELOP("IMenu::setItem(2)");
#ifdef IC_WIN
   if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
   {
     UINT fuflags = MF_BYCOMMAND;
     long itemPos = -1;
     IMenuHandle nestedMenu;

     // Depending on menu type, place identity into menu
     UINT  menu1 = (UINT)menuItem.id();
     if (menuItem.isSubmenu())
       menu1 = (UINT)(menuItem.submenuHandle().asUnsigned());

     // Depending on menu type, place information into menu
     LPSTR menu2 = (LPSTR)(char*)menuItem.text();
     if (menuItem.isBitmap())
       menu2 = (LPSTR)(void*)menuItem.bitmap();

     /******************************************************************/
     /* Query the state of the previous item to be set - if submenu,   */
     /*  then modify by position returned instead of by id             */
     /******************************************************************/
     queryItemState( fmenuFrame, menuHandle(), menuItem.id(), &itemPos,
                     &nestedMenu );
     if (itemPos >= 0)
       fuflags = MF_BYPOSITION;

     /******************************************************************/
     /* Change the item, indexing based on previous item type          */
     /******************************************************************/
     if (!(bool)ModifyMenu(
           (fuflags & MF_BYPOSITION) ? nestedMenu : menuHandle(),
           (fuflags & MF_BYPOSITION) ? (UINT)itemPos : (UINT)menuItem.id(),
           (UINT)(menuItem.style() | menuItem.attribute() | fuflags),
           menu1, menu2 ))
     {
        ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                           IBaseErrorInfo::invalidRequest,
                           IException::recoverable);
     }
     // Refresh menu if item modified without error
     if ( fMenuData->fUpdateAllowed )
       DrawMenuBar( fowner );

     // Ref count the bitmap if one passed in
     if (menuItem.isBitmap() && menuItem.bitmap() )
       refCountBitmap( menu2 );
   }
   else
   {
     MENUITEMINFO mn;
     mn.cbSize = sizeof(MENUITEMINFO);
     mn.fMask = MIIM_STATE | MIIM_ID | MIIM_SUBMENU | MIIM_TYPE;
     mn.wID           = (UINT)menuItem.id();
     mn.fType         = (UINT)menuItem.style();
     mn.fState        = (UINT)menuItem.attribute();
     mn.hSubMenu      = menuItem.submenuHandle();

     if (IPlatform::isWin9x())
     {
        // If Windows 95, reset the popup flag since that platform
        // does not allow it.
        mn.fType &= (unsigned long)(~MF_POPUP);
     }

     // Check for unavailable style on Windows and convert
     if ( menuItem.extendedStyle() &
                       IMenuItem::unavailable.asExtendedUnsignedLong() )
       mn.fState |= MF_DISABLED;

     // In the data field, get the bitmap handle if item is a bitmap
     // otherwise get the text string for the item
     if (menuItem.isBitmap())
       mn.dwTypeData    = (char *)((HBITMAP)menuItem.bitmap());
     else
       mn.dwTypeData    = (char *)menuItem.text();

     if (!SetMenuItemInfo( menuHandle(), mn.wID, FALSE, &mn ))
        ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                           IBaseErrorInfo::invalidRequest,
                           IException::recoverable);

     if ( IBidiSettings::isBidiSupported() )
     {
        if ( fowner )
        {
           // Give the menu the bidi attributes of the owner window.
           IBidiSettings
             bidiSettings( fowner );
           IMenuPrivate::setBidiAttributes( fhMenu, bidiSettings );
        }
        else
        {
           // Give the menu default bidi attributes.
           IBidiSettings
             bidiSettings( IBidiSettings::applicationDefaults() );
           IMenuPrivate::setBidiAttributes( fhMenu, bidiSettings );
        }
     }

     // Refresh menu if item modified without error
     if ( fMenuData->fUpdateAllowed )
       DrawMenuBar( fowner );

     // If item was a bitmap, the update ref count for bitmap
     if (mn.fType & MFT_BITMAP)
       refCountBitmap( mn.dwTypeData );
   }
   /*******************************************************************/
   /* Save any help id set in the item into the lookup table          */
   /*******************************************************************/
   if ( menuItem.helpId() && menuItem.id() )
     IMenuPrivate::setMenuItemHelpId( fmenuFrame, menuItem.id(), menuItem.helpId() );
#else
   MENUITEM mn;
   mn.iPosition    = (unsigned short)(menuItem.index());
   mn.afStyle      = (unsigned short)(menuItem.style());
   mn.afAttribute  = (unsigned short)(menuItem.attribute());
   mn.id           = (unsigned short)(menuItem.id());
   mn.hwndSubMenu  = menuItem.submenuHandle();
   mn.hItem        = menuItem.bitmap();

   if (!(window()->sendEvent( MM_SETITEM,
                   IEventData(0, true),
                   IEventData(&mn)).asUnsignedLong()))
   {
      ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */

   if (mn.afStyle & MIS_BITMAP) {
      if (mn.hItem) {
         if(!(window()->sendEvent( MM_SETITEMHANDLE,
                        IEventData(mn.id),
                        IEventData(mn.hItem)).asUnsignedLong()))
         {
            ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                               IBaseErrorInfo::invalidRequest,
                               IException::recoverable);
         } /* endif */
         refCountBitmap(mn.hItem);
      } else {
            ITHROWLIBRARYERROR(IC_NO_BITMAP_HANDLE,
                               IBaseErrorInfo::invalidRequest,
                               IException::recoverable);
      } /* endif */
   } else {
      if (mn.afStyle & MIS_TEXT)
         IMenu::setText(mn.id, menuItem.text());
   } /* endif */
#endif
   return *this;
}


/*------------------------------------------------------------------------------
| IMenu::setText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::setText( unsigned long  menuItemId,
                       const char*    newText )
{
   IMODTRACE_DEVELOP("IMenu::setText");
#ifdef IC_WIN
   // Get the menuitem for the id passed in
   IMenuItem mn = menuItem(menuItemId);

   // Check and make sure that the item is a text item
   if ( mn.isText() )
   {
     // If so, change the item text and update the menu
     mn.setText( newText );
     setItem( mn );
   }
   else
   {
     ITHROWLIBRARYERROR( IC_SETMENUITEM_FAIL,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
   }
#else
   //No need to ensure menu item has textItem style.
   //WinSendMsg will fail, if style is not textItem.

   if(!(window()->sendEvent( MM_SETITEMTEXT,
                  IEventData((short)menuItemId),
                  IEventData((char*)newText)).asUnsignedLong()))
   {
      ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable);
   } /* endif */
#endif
   return *this;
}


/*------------------------------------------------------------------------------
| IMenu::setText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::setText( unsigned long        menuItemId,
                       const IResourceId&   newTextResId )
{
  return IMenu::setText(menuItemId,
                        newTextResId.resourceLibrary().loadString(newTextResId));
}


/*------------------------------------------------------------------------------
| IMenu::setBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBitmap( unsigned long        menuItemId,
                         const IBitmapHandle& bitmapHandle )
{
   IMODTRACE_ALL("IMenu::setBitmap");
   //No need to ensure menu item has textItem style.
   //WinSendMsg will fail, if style is not textItem.
   IMenuItem mItem = menuItem(menuItemId);
   mItem.setBitmap(bitmapHandle);
   return IMenu::setItem(mItem);
}


/*------------------------------------------------------------------------------
| IMenu::setBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBitmap( unsigned long        menuItemId,
                         const IResourceId&   newBitmapResId )
{

   return IMenu::setBitmap(menuItemId,
                   newBitmapResId.resourceLibrary().loadBitmap(newBitmapResId));
}


/*------------------------------------------------------------------------------
| IMenu::setBitmap                                                             |
| This version needed to resolved ambiguity                                    |
------------------------------------------------------------------------------*/
IMenu& IMenu::setBitmap( unsigned long        menuItemId,
                         unsigned long        newBitmapResId )
{
   return IMenu::setBitmap(menuItemId, IResourceId(newBitmapResId));
}


/*------------------------------------------------------------------------------
| IMenu::conditionalCascade                                                    |
| Return the id of the default item within the given submenu.                  |
------------------------------------------------------------------------------*/
unsigned long IMenu::conditionalCascade( unsigned long itemWithSubmenuId)
{
  unsigned long defaultItemId = 0;
  /*******************************************************************/
  /* Insure id is valid. If invalid, query will throw exception      */
  /*******************************************************************/
#ifdef IC_PM
  queryItem(window()->handle(), itemWithSubmenuId);
  /*******************************************************************/
  /* If valid, then construct a menu item to query                   */
  /*******************************************************************/
  IMenuItem mn = menuItem(itemWithSubmenuId);
  if (mn.submenuHandle() == 0) {
     ITHROWLIBRARYERROR( IC_NO_SUBMENU,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
  }
  else {
     defaultItemId = mn.submenuHandle().sendEvent(MM_QUERYDEFAULTITEMID, 0, 0);
     if (defaultItemId == 0)
        defaultItemId = -1UL;   //signal "there is none" in same way that WIN does it
  }
#endif
#ifdef IC_WIN
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell()) {
    queryItem( menuHandle(), itemWithSubmenuId, false, 0 );
    /*******************************************************************/
    /* If valid, then construct a menu item to query                   */
    /*******************************************************************/
    IMenuItem mn = menuItem(itemWithSubmenuId);
    if (mn.submenuHandle() == 0)
       ITHROWLIBRARYERROR( IC_NO_SUBMENU,
                           IBaseErrorInfo::invalidRequest,
                           IException::recoverable );
    else
    defaultItemId = GetMenuDefaultItem( mn.submenuHandle(),
                                        FALSE,
                                        GMDI_GOINTOPOPUPS | GMDI_USEDISABLED  );
  }
#endif

  return defaultItemId;
}


/*------------------------------------------------------------------------------
| IMenu::setConditionalCascade                                                 |
------------------------------------------------------------------------------*/
IMenu& IMenu::setConditionalCascade( unsigned long itemWithSubmenuId,
                                     unsigned long defaultItemId )
{
   IMODTRACE_ALL("IMenu::setConditionalCascade");
  /*******************************************************************/
  /* Insure ids are valid. If invalid, query will throw exception    */
  /*******************************************************************/
#ifdef IC_PM
  queryItem(window()->handle(), itemWithSubmenuId);
  queryItem(window()->handle(), defaultItemId);
#endif
#ifdef IC_WIN
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell()) {
    queryItem( menuHandle(), itemWithSubmenuId, false, 0 );
    queryItem( menuHandle(), defaultItemId, false, 0 );
#endif

    /*******************************************************************/
    /* If valid, then construct a menu item to use                     */
    /*******************************************************************/
    IMenuItem mn = menuItem(itemWithSubmenuId);
    if (mn.submenuHandle() == 0) {
       ITHROWLIBRARYERROR( IC_NO_SUBMENU,
                           IBaseErrorInfo::invalidRequest,
                           IException::recoverable );
    }
#ifdef IC_PM
    else {
       WinSetWindowBits(mn.submenuHandle(), QWL_STYLE,
                        MS_CONDITIONALCASCADE,
                        MS_CONDITIONALCASCADE);
       mn.submenuHandle()
         .sendEvent(MM_SETDEFAULTITEMID,
                    IEventData((unsigned short)defaultItemId,0), 0);
    } /* endif */
#endif
#ifdef IC_WIN
    else
      SetMenuDefaultItem( mn.submenuHandle(), (UINT)defaultItemId, FALSE );
  } /* endif for Windows platform checking (for extended menu support)*/
#endif
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::removeConditionalCascade                                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::removeConditionalCascade( unsigned long itemWithSubmenuId )
{
   IMODTRACE_ALL("IMenu::removeConditionalCascade");
#ifdef IC_PM
  queryItem(window()->handle(), itemWithSubmenuId);   //Check is valid ID to cause exception.
#endif
#ifdef IC_WIN
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    /******************************************************************/
    /* Check for valid id. If invalid, query will throw exception     */
    /******************************************************************/
    queryItem( menuHandle(), itemWithSubmenuId, false, 0 );
#endif

  /*******************************************************************/
  /* If valid, then construct a menu item to use                     */
  /*******************************************************************/
  IMenuItem mn = menuItem(itemWithSubmenuId);
  if (mn.submenuHandle() == 0)
  {
     ITHROWLIBRARYERROR( IC_NO_SUBMENU,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
  }
#ifdef IC_PM
  else {
     // determine the default item and remove its checkmark
     long defaultItemId =
            mn.submenuHandle().sendEvent( MM_QUERYDEFAULTITEMID, 0, 0);
     if (defaultItemId != MIT_ERROR &&
         defaultItemId != MIT_NONE &&
         defaultItemId != 0)
       uncheckItem(defaultItemId);

     // remove the conditional cascade style
     WinSetWindowBits(mn.submenuHandle(), QWL_STYLE,
                      (unsigned long)~MS_CONDITIONALCASCADE,
                      MS_CONDITIONALCASCADE);
  } /* endif */
#endif
#ifdef IC_WIN
    else
      // Remove default item by specifying -1
      SetMenuDefaultItem( mn.submenuHandle(), (UINT)-1, FALSE );
  } /* endif for Windows platform checking (for extended menu support)*/
#endif
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::deleteItem                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::deleteItem( unsigned long ulItemId )
{
  IMODTRACE_ALL("IMenu::deleteItem");
#ifdef IC_WIN
  UINT fuflags = MF_BYCOMMAND;
  long itemPos = -1;
  IMenuHandle nestedMenu;

  /********************************************************************/
  /* Check for valid id. If invalid, query will throw exception       */
  /********************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    queryItem( menuHandle(), ulItemId, false, 0 );
  }
  else
  {
    /******************************************************************/
    /* Query the state (to check validity) - if submenu, the remove   */
    /*  by position returned instead of by id                         */
    /******************************************************************/
    queryItemState( fmenuFrame, menuHandle(), ulItemId, &itemPos,
                    &nestedMenu );
    if (itemPos >= 0)
      fuflags = MF_BYPOSITION;
  }
  DeleteMenu( (fuflags & MF_BYPOSITION) ? nestedMenu : menuHandle(),
              (fuflags & MF_BYPOSITION) ? (UINT)itemPos : (UINT)ulItemId,
              fuflags );

  // Refresh menu if item removed
  if ( fMenuData->fUpdateAllowed )
    DrawMenuBar( fowner );
#else
  queryItem(window()->handle(), ulItemId);   //Check is valid ID to cause exception.
  window()->sendEvent(MM_DELETEITEM, IEventData((unsigned short)ulItemId, true), 0);
#endif
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::removeSubmenu                                                         |
------------------------------------------------------------------------------*/
IMenu& IMenu::removeSubmenu( unsigned long itemWithSubmenuId )
{
   IMODTRACE_ALL("IMenu::removeSubmenu");
   IMenuItem mn  = menuItem(itemWithSubmenuId);
   IMenuHandle hMenu = mn.submenuHandle();
   if ( hMenu )
   {
      /****************************************************************/
      /* Set submenu handle to NULL - This also resets the popup flag */
      /****************************************************************/
      mn.setSubmenuHandle(0);
#ifdef IC_PM
      IMenu::setItem(mn);
#endif
#ifdef IC_WIN
      /****************************************************************/
      /* If new-shell, use normal logic to replace item               */
      /****************************************************************/
      if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
      {
        IMenu::setItem(mn);
      }
      /****************************************************************/
      /* Modify Menu (used in setItem) does not seem to work when     */
      /* you actually try and replace a popup item with a non-popup   */
      /* item on NT old-shell, so need to work around this            */
      /****************************************************************/
      else
      {
        long itemPos = -1;
        IMenuHandle nestedMenu;

        /**************************************************************/
        /* Get position & parent menu for this submenu (by-product of)*/
        /**************************************************************/
        queryItemState( fmenuFrame, menuHandle(), itemWithSubmenuId,
                        &itemPos, &nestedMenu );
        if (itemPos >= 0)
        {
          /************************************************************/
          /* Remove the popup menu                                    */
          /************************************************************/
          RemoveMenu( nestedMenu, (UINT)itemPos, MF_BYPOSITION );

          /************************************************************/
          /* Depending on menu type, place information into menu      */
          /************************************************************/
          LPSTR menu2 = (LPSTR)(char*)mn.text();
          if (mn.isBitmap())
            menu2 = (LPSTR)(void*)mn.bitmap();

          /************************************************************/
          /* Then re-add it as just an item                           */
          /************************************************************/
          InsertMenu( nestedMenu, (UINT)itemPos,
                      (UINT)(mn.style() | mn.attribute() | MF_BYPOSITION),
                      (UINT)itemWithSubmenuId, menu2 );
        }
      }
      /****************************************************************/
      /* Remove this item from lookup table, since the id or handle   */
      /* may be reused in the future                                  */
      /****************************************************************/
      IMenuPrivate::removeFromLookUpTable( fmenuFrame, hMenu, itemWithSubmenuId );
#endif
   } /* endif */
   return *this;
}


/*------------------------------------------------------------------------------
| IMenu::checkItem                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::checkItem( unsigned long ulItemId, bool check )
{
   IMODTRACE_ALL("IMenu::checkItem");
  unsigned long menuState;
#ifdef IC_WIN
  MENUITEMINFO miItem;
  UINT fuflags = MF_BYCOMMAND | MF_UNCHECKED;
  long itemPos = -1;
  IMenuHandle nestedMenu;

  /********************************************************************/
  /* Get the item state for the item                                  */
  /********************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    miItem = queryItem( menuHandle(), ulItemId, false, 0 );
    menuState = miItem.fState;
    miItem.fMask = MIIM_STATE;
  }
  else
  {
    /******************************************************************/
    /* Query the state (to check validity) - if submenu, the remove   */
    /*  by position returned instead of by id                         */
    /******************************************************************/
    menuState = queryItemState( fmenuFrame, menuHandle(), ulItemId,
                                &itemPos, &nestedMenu );
    if (itemPos >= 0)
      fuflags |= MF_BYPOSITION;
  }
#endif
#ifdef IC_PM
  MENUITEM miItem = queryItem(window()->handle(), ulItemId);
  menuState = miItem.afAttribute;
#endif
  bool bYes = false;

  if (check) {
     if (!(menuState & MIA_CHECKED))
     {
#ifdef IC_WIN
        if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
          fuflags |= MF_CHECKED;
        else
          miItem.fState |= MFS_CHECKED;
#endif
#ifdef IC_PM
        menuState |= MIA_CHECKED;
#endif
        bYes = true;
     } /* endif */
  } else {
     if (menuState & MIA_CHECKED) {
#ifdef IC_WIN
        if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
          miItem.fState &= (unsigned long)~MFS_CHECKED;
#else
        menuState &= (unsigned long)~MIA_CHECKED;
#endif
        bYes = true;
     } /* endif */
  } /* endif */

  /********************************************************************/
  /* If state is to be updated, then make the system call to do so    */
  /*  and then check the return code for any errors (and flag if any) */
  /********************************************************************/
  if (bYes)
  {
#ifdef IC_WIN
     if (((!(IPlatform::isWin9x() || IPlatform::isNTNewShell())) &&
           (CheckMenuItem( (fuflags & MF_BYPOSITION) ? nestedMenu :
                           menuHandle(), (fuflags & MF_BYPOSITION) ?
            (UINT)itemPos : (UINT)ulItemId, fuflags ) == 0xFFFFFFFFul)) ||
           ((IPlatform::isWin9x() || IPlatform::isNTNewShell()) &&
           (!SetMenuItemInfo( menuHandle(), (UINT)ulItemId, FALSE, &miItem ))))
#else
     if(!(window()->sendEvent( MM_SETITEMATTR,
                    IEventData((unsigned short)ulItemId, true),
                    IEventData(MIA_CHECKED,
                              (unsigned short)menuState )).asUnsignedLong()))
#endif
     {
       ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable);
     }
  }

  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::uncheckItem                                                           |
------------------------------------------------------------------------------*/
IMenu& IMenu::uncheckItem(unsigned long ulItemId)
{
  return checkItem(ulItemId, false);
}


/*------------------------------------------------------------------------------
| IMenu::isItemChecked                                                         |
------------------------------------------------------------------------------*/
bool IMenu::isItemChecked(unsigned long ulItemId) const
{
#ifdef IC_WIN
  /*******************************************************************/
  /* Get the item state for the item                                 */
  /*******************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    MENUITEMINFO miItem = queryItem( menuHandle(), ulItemId, false, 0 );
    return (bool)( miItem.fState & MFS_CHECKED );
  }
  else
  {
    unsigned long menuState = queryItemState( fmenuFrame, menuHandle(),
                                              ulItemId, 0, NULL );
    return (bool)(menuState & MF_CHECKED);
  }
#endif
#ifdef IC_PM
   //no exception required. PM returns false if ID is not found.
   return window()->sendEvent(MM_QUERYITEMATTR,
                              IEventData((unsigned short)ulItemId, true),
                              IEventData(MIA_CHECKED)).asUnsignedLong();
#endif
}

/*------------------------------------------------------------------------------
| IMenu::enableItem                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::enableItem(unsigned long ulItemId, bool enable)
{
   IMODTRACE_ALL("IMenu::enableItem");
  unsigned long menuState;
#ifdef IC_WIN
  MENUITEMINFO miItem;
  UINT fuflags = MF_BYCOMMAND | MF_ENABLED;
  long itemPos = -1;
  IMenuHandle nestedMenu;

  /*******************************************************************/
  /* Get the item state for the item                                 */
  /*******************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    miItem = queryItem( menuHandle(), ulItemId, false, 0 );
    menuState = miItem.fState;
    miItem.fMask = MIIM_STATE;
  }
  else
  {
    /******************************************************************/
    /* Query the state (to check validity) - if submenu, the remove   */
    /*  by position returned instead of by id                         */
    /******************************************************************/
    menuState = queryItemState( fmenuFrame, menuHandle(), ulItemId,
                                &itemPos, &nestedMenu );
    if (itemPos >= 0)
      fuflags |= MF_BYPOSITION;
  }
#else
  MENUITEM miItem = queryItem(window()->handle(), ulItemId);
  menuState = miItem.afAttribute;
#endif
  bool bYes = false;

  if (enable)
  {
#ifdef IC_WIN
     if (menuState & (MF_DISABLED | MF_GRAYED) )
     {
       if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
       {
         miItem.fState &= (unsigned long)~MFS_DISABLED;
       }
       bYes = true;
     } /* endif */
#else
     if (menuState & MIA_DISABLED)
     {
       menuState &= (unsigned long)~MIA_DISABLED;
       bYes = true;
     } /* endif */
#endif
  }
  else
  {
#ifdef IC_WIN
     if ( !(menuState & (MF_DISABLED | MF_GRAYED)) )
     {
        if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
          fuflags |= MF_GRAYED;    // not MF_DISABLED to get visual too
        else
          miItem.fState |= MFS_DISABLED;

        bYes = true;
     }
#else
     if (!(menuState & MIA_DISABLED))
     {
        menuState |= MIA_DISABLED;
        bYes = true;
     } /* endif */
#endif
  } /* endif */

  /********************************************************************/
  /* If state is to be updated, then make the system call to do so    */
  /*  and then check the return code for any errors (and flag if any) */
  /********************************************************************/
  if (bYes)
  {
#ifdef IC_WIN
     if (((!(IPlatform::isWin9x() || IPlatform::isNTNewShell())) &&
           (EnableMenuItem( (fuflags & MF_BYPOSITION) ? nestedMenu :
                            menuHandle(), (fuflags & MF_BYPOSITION) ?
            (UINT)itemPos : (UINT)ulItemId, fuflags ) == 0xFFFFFFFFul)) ||
           ((IPlatform::isWin9x() || IPlatform::isNTNewShell()) &&
           (!SetMenuItemInfo( menuHandle(), (UINT)ulItemId, FALSE, &miItem ))))
#else
     if (!(window()->sendEvent( MM_SETITEMATTR,
                                IEventData((unsigned short)ulItemId, true),
                                IEventData(MIA_DISABLED,
                                     (unsigned short)menuState)).asUnsignedLong()))
#endif
     {
       ITHROWLIBRARYERROR(IC_SETMENUITEM_FAIL,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable);
     } /* endif */
#ifdef IC_WIN
     /*****************************************************************/
     /* Refresh menu bar (in case item being enabled/disabled is on   */
     /* the menubar itself)                                           */
     /*****************************************************************/
     if ( fMenuData->fUpdateAllowed )
       DrawMenuBar( fowner );
#endif
  } /* endif */
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::disableItem                                                           |
------------------------------------------------------------------------------*/
IMenu& IMenu::disableItem(unsigned long ulItemId)
{
  return enableItem(ulItemId, false);
}

/*------------------------------------------------------------------------------
| IMenu::isItemEnabled                                                         |
------------------------------------------------------------------------------*/
bool IMenu::isItemEnabled(unsigned long ulItemId) const
{
#ifdef IC_WIN
  /*******************************************************************/
  /* Get the item state for the item                                 */
  /*******************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    MENUITEMINFO miItem = queryItem( menuHandle(), ulItemId, false, 0 );
    return (bool)( !(miItem.fState & MFS_DISABLED) );
  }
  else
  {
    unsigned long menuState = queryItemState( fmenuFrame, menuHandle(),
                                              ulItemId, 0, NULL );
    return (bool)( !(menuState & (MF_DISABLED | MF_GRAYED)) );
  }
#else
  //no exception required. PM returns false if ID is not found.
  return (!(window()->sendEvent(MM_QUERYITEMATTR,
                                IEventData((unsigned short)ulItemId, true),
                                IEventData(MIA_DISABLED)).asUnsignedLong()));
#endif
}

/*------------------------------------------------------------------------------
| IMenu::selectItem                                                            |
------------------------------------------------------------------------------*/
IMenu& IMenu::selectItem( unsigned long itemId )
{
  IMODTRACE_ALL("IMenu::selectItem");
#ifdef IC_WIN
  UINT fuflags = MF_BYCOMMAND | MF_HILITE;
  long itemPos = -1;
  IMenuHandle nestedMenu;

  /*******************************************************************/
  /* Get the item state for the item                                 */
  /*******************************************************************/
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
    queryItem( menuHandle(), itemId, false, 0 );
  else
  {
    /******************************************************************/
    /* Query the state (to check validity) - if submenu, then hilite  */
    /*  by position returned instead of by id                         */
    /******************************************************************/
    queryItemState( fmenuFrame, menuHandle(), itemId, &itemPos,
                    &nestedMenu );
    if (itemPos >= 0)
      fuflags |= MF_BYPOSITION;
  }
  HiliteMenuItem( fowner, (fuflags & MF_BYPOSITION) ? nestedMenu : menuHandle(),
                  (fuflags & MF_BYPOSITION) ? (UINT)itemPos : (UINT)itemId,
                  fuflags );
#else
  MENUITEM miItem = queryItem(window()->handle(), itemId);
  window()->sendEvent(MM_SELECTITEM,
                      IEventData((unsigned short)itemId, true),
                      IEventData(0, (unsigned short)
                                       (!(miItem.afAttribute & MIA_NODISMISS))));
#endif
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::menuItem                                                              |
|  Query menu-item and return as IMenuItem object.                             |
------------------------------------------------------------------------------*/
IMenuItem IMenu::menuItem(unsigned long ulItemId) const
{
   IMODTRACE_ALL("IMenu::menuItem");
#ifdef IC_WIN
  unsigned long menuState = 0;
  MENUITEMINFO miItem;
  char         pszItemText[IC_MENUTEXTLENGTH];
  UINT fuflags = MF_BYCOMMAND;
  long itemPos = -1;
  IMenuHandle  nestedMenu;

  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    /******************************************************************/
    /* Query for all of the information about this menu item.  Make   */
    /*  sure we pass a temp string in (not just a local) to save text */
    /******************************************************************/
    miItem = queryItem( menuHandle(), ulItemId, false, pszItemText );
  }
  else
  {
    menuState = queryItemState( fmenuFrame, menuHandle(), ulItemId,
                                &itemPos, &nestedMenu );
    if (itemPos >= 0)
      fuflags |= MF_BYPOSITION;
  }
#else
  MENUITEM miItem;
//  const ISubmenu* thisAsSubmenu = dynamic_cast<const ISubmenu*>(this);
//  if (thisAsSubmenu)     //ISubMenu do not have a wrapper window (on Motif and OS2)
//    miItem = queryItem(thisAsSubmenu->handle(), ulItemId);
//  else                   // All other IMenus do, so use it
    miItem = queryItem(window()->handle(), ulItemId);
#endif
  IMenuItem mn(ulItemId);

#ifdef IC_WIN
  if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
  {
    mn.setStyle( menuState & IMS_STYLEMASK );
    mn.setAttribute( menuState & IMS_ATTRMASK );

    if (!(menuState & (MF_BITMAP | MF_OWNERDRAW | MF_SEPARATOR)))
    {
      mn.setExtendedStyle( IMIS_TEXT );
      GetMenuString( (fuflags & MF_BYPOSITION) ? nestedMenu : menuHandle(),
                     (fuflags & MF_BYPOSITION) ? (UINT)itemPos : (UINT)ulItemId,
                     pszItemText, IC_MENUTEXTLENGTH, fuflags );
      mn.setText(pszItemText);
    }
    /******************************************************************/
    /* If submenu, get the submenu handle and set the menuitem data   */
    /******************************************************************/
    if (menuState & MF_POPUP)
    {
      /****************************************************************/
      /* Get position of popup in menu                                */
      /****************************************************************/
      IMenuPrivate::locateMenuItem( fmenuFrame, menuHandle(), ulItemId,
                                    &itemPos, NULL );

      /****************************************************************/
      /* Query the submenu handle, either from base menu or nested one*/
      /****************************************************************/
      HMENU hPop = GetSubMenu( (fuflags & MF_BYPOSITION) ? nestedMenu :
                               menuHandle(), (int)itemPos );
      mn.setSubmenuHandle( hPop );
    }
  }
  else
  {
    /*******************************************************************/
    /* Update menu item state, attribute and any submenu info          */
    /*******************************************************************/
    mn.setStyle(miItem.fType);
    mn.setAttribute(miItem.fState);
    mn.setSubmenuHandle(miItem.hSubMenu);

    /*******************************************************************/
    /* Check special case of unavailable state for Windows             */
    /*******************************************************************/
    if ((miItem.fState & MF_DISABLED) && !(miItem.fState & MF_GRAYED))
      mn.setSelectable(false);

    /*******************************************************************/
    /* If bitmap menu item, then data to set is the bitmap handle      */
    /*******************************************************************/
    if ( miItem.fType & MFT_BITMAP )
      mn.hBitmapHandle = miItem.dwTypeData;

    /*******************************************************************/
    /* otherwise it is a text menu item, so data to set is text string */
    /*******************************************************************/
    else if (!( miItem.fType & (MFT_OWNERDRAW | MFT_SEPARATOR)))
      mn.setText( miItem.dwTypeData );          // == pszItemText
  }

  /********************************************************************/
  /* Set the index of this menuitem (if found in table)               */
  /********************************************************************/
  itemPos = -1;
  IMenuPrivate::locateMenuItem( fmenuFrame, menuHandle(), ulItemId,
                                &itemPos, NULL );
  mn.setIndex(itemPos);

  /********************************************************************/
  /* Set the help id of this menuitem (if one exists in table)        */
  /********************************************************************/
  mn.setHelpId( IMenuPrivate::queryMenuItemHelpId( fmenuFrame, ulItemId ) );
#else
  mn.setIndex(miItem.iPosition);
  mn.setStyle(miItem.afStyle);
  mn.setAttribute(miItem.afAttribute);
  mn.setSubmenuHandle(miItem.hwndSubMenu);

  if (miItem.afStyle & MIS_TEXT) {
    /* Shouldn't need to throw exceptions here */
    IWindowHandle theHandle;
//    if (thisAsSubmenu)    //ISubMenu do not have a wrapper window (on Motif and OS2)
//      theHandle = thisAsSubmenu->handle();
//    else                  // All other IMenus do, so use it
      theHandle = window()->handle();
    unsigned long ulLen = theHandle.sendEvent(MM_QUERYITEMTEXTLENGTH,
                                               IEventData(ulItemId),
                                               0);
     ++ulLen;
     char* pszItemText = new char[ulLen];
     theHandle.sendEvent(MM_QUERYITEMTEXT,
                        IEventData((unsigned short)ulItemId,
                                   (unsigned short)ulLen),
                        IEventData(pszItemText));
     mn.setText(pszItemText);
     delete pszItemText;
  }
  else {
    if (miItem.afStyle & MIS_BITMAP)
      mn.hBitmapHandle = miItem.hItem;
  } /* endif */
#endif
   return mn;
}

/*------------------------------------------------------------------------------
| IMenu::numberOfItems                                                         |
------------------------------------------------------------------------------*/
unsigned long IMenu::numberOfItems ( unsigned long forSubmenuId) const
{
#ifdef IC_WIN
  IMenuHandle hMenu;
  /********************************************************************/
  /* If id is 0, use base handle, otherwise get submenu handle for id */
  /********************************************************************/
  if (forSubmenuId == 0)
    hMenu = menuHandle();
  else
    hMenu = submenuHandle(forSubmenuId);

  /********************************************************************/
  /* Return the count of the items for the handle retrieved above     */
  /********************************************************************/
  return (unsigned long)GetMenuItemCount( hMenu );
#endif
#ifdef IC_PM
  IMenuHandle hwnd = (forSubmenuId == 0) ?
                         menuHandle() : submenuHandle(forSubmenuId);
  return (unsigned long) hwnd.sendEvent(MM_QUERYITEMCOUNT, 0, 0);
#endif
}

/*------------------------------------------------------------------------------
| IMenu::itemRect                                                              |
|  Returns the bounding rectangle of a menu item                               |
------------------------------------------------------------------------------*/
IRectangle IMenu::itemRect( unsigned long itemId ) const
{
#ifdef IC_WIN
  if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
  {
    // No support for item rect under old menu API, so return NULL rect
    return IRectangle();
  }
  else
  {
    RECT rect;
    long id;

    IMenuPrivate::locateMenuItem( fmenuFrame, menuHandle(), itemId,
                                  &id, NULL );
    GetMenuItemRect( fowner, menuHandle(), (UINT)id, &rect );
    return IRectangle(rect);
  }
#endif
#ifdef IC_PM
  PRECTL prectl = new RECTL;
  window()->sendEvent( MM_QUERYITEMRECT,
                       IEventData((unsigned short)itemId, true),
                       IEventData(prectl));

  return IRectangle( prectl->xLeft, prectl->yBottom,
                     prectl->xRight, prectl->yTop );
#endif
}

/*------------------------------------------------------------------------------
| IMenu::menuItemId                                                            |
|  Return menu-item ID of item at cursor position                              |
------------------------------------------------------------------------------*/
unsigned long IMenu::menuItemId ( const Cursor& cursor ) const
{
  IMODTRACE_ALL("IMenu::menuItemId");
  IASSERTSTATE( cursor.isValid());
  long id;
#ifdef IC_WIN
  bool rc = false;
  IMenuHandle hMenu = (cursor.ulSubmenu == 0) ?
                          cursor.pMenu->menuHandle() :
                          cursor.pMenu->submenuHandle(cursor.ulSubmenu);

  if (!(IPlatform::isWin9x() || IPlatform::isNTNewShell()))
  {
    // Query for the id of the item at the cursor position
    id = GetMenuItemID( hMenu, (int)cursor.lCurrent );

    // If error, the item might be a submenu so query for a menu handle
    if ( id == 0xFFFFFFFFul )
    {
      HMENU hPop = GetSubMenu( hMenu, (int)cursor.lCurrent );

      // If a menu handle is returned, then get the id from the lookup table
      if (hPop)
        id = IMenuPrivate::getItemIdFromLookUpTable( fmenuFrame, hPop );
    }

    // If -1, then either it wasn't a popup, or it wasn't in the lookup table
    rc = ( id != 0xFFFFFFFFul );
  }
  else
  {
    MENUITEMINFO miItem;
    miItem.cbSize = sizeof(MENUITEMINFO);
    miItem.fMask = MIIM_STATE | MIIM_ID;

    rc = GetMenuItemInfo( hMenu, (int)cursor.lCurrent, TRUE,
                          &miItem );
    id = miItem.wID;
  }

  if (!rc)
  {
     if ((cursor.lCurrent < 0) ||
         (cursor.lCurrent >= GetMenuItemCount( menuHandle() )))
#else
  IMenuHandle hwnd = (cursor.ulSubmenu == 0) ?
                          cursor.pMenu->menuHandle() :
                          cursor.pMenu->submenuHandle(cursor.ulSubmenu);
  id = (long)hwnd.sendEvent( MM_ITEMIDFROMPOSITION,
                             (int)cursor.lCurrent, 0 );
  if (id == MIT_ERROR)
  {
     /*-------------------------------------------------------------------------
     | If menu item number is bad, then throw exception.  Otherwise the item   |
     | must be a MENUITEM SEPARATOR and the dummy value is valid               |
     -------------------------------------------------------------------------*/
     if ((cursor.lCurrent < 0) ||
         (cursor.lCurrent >= hwnd.sendEvent(MM_QUERYITEMCOUNT,0,0)))
#endif
     {
        ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                           IBaseErrorInfo::invalidRequest,
                           IException::recoverable);
     }
  } /* endif */
  return id;
}


/*------------------------------------------------------------------------------
| IMenu::defaultStyle                                                          |
|                                                                              |
| Return the default style for new menu objects.                               |
------------------------------------------------------------------------------*/
IMenu::Style  IMenu:: defaultStyle()
{
  return currentDefaultStyle;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| IMenu::setDefaultStyle                                                       |
|                                                                              |
| Set the default style for new menu objects.                                  |
------------------------------------------------------------------------------*/
void IMenu::setDefaultStyle( const IMenu::Style& newDefault )
{
  currentDefaultStyle = newDefault;
}
#endif


/*------------------------------------------------------------------------------
| IMenu::setItemHelpId                                                         |
|                                                                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::setItemHelpId( unsigned long  menuItemId,
                             unsigned long  helpTopicId )
{
  IMODTRACE_ALL("IMenu::setItemHelpId");
#ifdef IC_PM
  // IC_NOTYET  help stuff needs testing
  IFrameWindow* fmenuFrame( (IFrameWindow*)this->owner() );
  if ( fmenuFrame && !fmenuFrame->isFrameWindow() )
     fmenuFrame = 0;
#endif
  IMenuPrivate::setMenuItemHelpId( fmenuFrame, menuItemId, helpTopicId );
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::itemHelpId                                                            |
|                                                                              |
------------------------------------------------------------------------------*/
unsigned long IMenu::itemHelpId( unsigned long  menuItemId ) const
{
#ifdef IC_PM
  // IC_NOTYET  help stuff needs testing
  IFrameWindow* fmenuFrame( (IFrameWindow*)this->owner() );
  if ( fmenuFrame && !fmenuFrame->isFrameWindow() )
     fmenuFrame = 0;
#endif
  return IMenuPrivate::queryMenuItemHelpId( fmenuFrame, menuItemId );
}


/*------------------------------------------------------------------------------
| IMenu::Cursor::Cursor                                                        |
| Cursor ctor.                                                                 |
------------------------------------------------------------------------------*/
IMenu::Cursor::Cursor(const IMenu&  menu, unsigned long forSubmenuId )
  : lCurrent(0), pMenu((IMenu*)&menu), ulSubmenu(forSubmenuId), fCursorData(0)
{
  IASSERTSTATE(menu.isValid());
#ifdef IC_PM
  // If we have a forSubmenuId ensure it exists in menu
  // queryItem will throw if forSubmenuId not found in menu
  if (forSubmenuId)
    MENUITEM tempItem = queryItem(menu.window()->handle(), forSubmenuId);
#else
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    // If we have a forSubmenuId ensure it exists in menu
    // queryItem will throw if forSubmenuId not found in menu
    if (forSubmenuId)
      MENUITEMINFO tempItem = queryItem( menu.menuHandle(), forSubmenuId,
                                         false, 0 );
  }
#endif
}

/*------------------------------------------------------------------------------
| IMenu::Cursor::Cursor                                                        |
| Copy ctor.                                                                   |
------------------------------------------------------------------------------*/
IMenu::Cursor::Cursor(const IMenu::Cursor& cursor )
  : lCurrent(cursor.lCurrent), pMenu(cursor.pMenu), ulSubmenu(cursor.ulSubmenu),
    fCursorData(0)
{
}

/*------------------------------------------------------------------------------
| IMenu::Cursor :: ~Cursor                                                     |
|                                                                              |
| Empty destructor here for page tuning.                                       |
------------------------------------------------------------------------------*/
IMenu::Cursor :: ~Cursor() { }

/*------------------------------------------------------------------------------
| IMenu::Cursor IMenu::cursor                                                  |
------------------------------------------------------------------------------*/
IMenu::Cursor IMenu::cursor ( unsigned long itemId,
                              unsigned long inSubmenuId ) const
{
#ifdef IC_WIN
   IMenuHandle hwnd = (inSubmenuId == 0) ?
                         menuHandle() : submenuHandle(inSubmenuId);

   long id;
   if (!IMenuPrivate::locateMenuItem( fmenuFrame, hwnd, itemId,
                                      &id, NULL ))
#else
   IMenuHandle hwnd = (inSubmenuId == 0) ?
                         menuHandle() : submenuHandle(inSubmenuId);
   long id = hwnd.sendEvent(MM_ITEMPOSITIONFROMID,
                            IEventData((unsigned short)itemId, false),
                            0);
   if (id == MIT_NONE)
#endif
   {
       ITHROWLIBRARYERROR(IC_INVALID_MENUITEM,
                          IBaseErrorInfo::invalidRequest,
                          IException::recoverable);
   } /* endif */
   IMenu::Cursor cursor(*this, inSubmenuId);
   cursor.lCurrent = id;
   return cursor;
}

/*------------------------------------------------------------------------------
| IMenu::Cursor :: setToLast                                                   |
------------------------------------------------------------------------------*/
bool IMenu::Cursor::setToLast()
{
  try {
     lCurrent = pMenu->numberOfItems(ulSubmenu) - 1;
  }
  // Want to return false like the other cursor functions if
  // ulSubmenu doesn't exist
  catch(...) {
     lCurrent = -1;
  }
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
| IMenu::refCountBitmap                                                        |
| Add reference count on bitmap handle.                                        |
------------------------------------------------------------------------------*/
void IMenu::refCountBitmap(const IBitmapHandle& bitmapHandle)
{
   IBitmapHandle* pBmp = new IBitmapHandle(bitmapHandle);

   if (pBitmapList) {
      BITMAPLIST* pNew = new BITMAPLIST;
      pNew->pBitmap = pBmp;
      pNew->pNext   = (BITMAPLIST*)pBitmapList;
      pBitmapList   = pNew;
   } else {
      BITMAPLIST* pList = new BITMAPLIST;          //Create new record.
      pList->pBitmap = pBmp;
      pList->pNext   = 0;
      pBitmapList    = (void*)pList;
   } /* endif */
}

/*------------------------------------------------------------------------------
| IMenu::submenuHandle                                                         |
------------------------------------------------------------------------------*/
IMenuHandle IMenu::submenuHandle ( unsigned long ulInSubmenuId ) const
{
  IMODTRACE_ALL("IMenu::submenuHandle");
#ifdef IC_WIN
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
  {
    MENUITEMINFO miItem = queryItem( menuHandle(), ulInSubmenuId, false,
                                     0 );
    return IMenuHandle( miItem.hSubMenu );
  }
  else
  {
    // Use lookup table to retrieve id for mapping
    return IMenuHandle( IMenuPrivate::getHMenuFromLookUpTable( fmenuFrame,
                                                        ulInSubmenuId ));
  }
#else
  MENUITEM miItem = queryItem(window()->handle(), ulInSubmenuId);
  return IMenuHandle( miItem.hwndSubMenu );
#endif
}

/*------------------------------------------------------------------------------
| IMenu::addText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::addText( unsigned long      newItemId,
                       const char*        itemText,
                       unsigned long      intoSubmenuId )
{
   IMODTRACE_ALL("IMenu::addText");
   IMenuItem mn(newItemId);
   mn.setText(itemText);
   return addItem(mn, intoSubmenuId);
}


/*------------------------------------------------------------------------------
| IMenu::addText                                                               |
------------------------------------------------------------------------------*/
IMenu& IMenu::addText( unsigned long      newItemId,
                       const IResourceId& textResId,
                       unsigned long      intoSubmenuId )
{
   IMODTRACE_ALL("IMenu::addText");
   IMenuItem mn(newItemId);
   mn.setText(textResId);
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::addBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addBitmap( unsigned long        newItemId,
                         const IResourceId&   bitmapResId,
                         unsigned long        intoSubmenuId )
{
   IMODTRACE_ALL("IMenu::addBitmap");
   IMenuItem mn(newItemId);
   mn.setBitmap(bitmapResId);
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::addBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addBitmap( unsigned long        newItemId,
                         unsigned long        bitmapResId,
                         unsigned long        intoSubmenuId )
{
   IMODTRACE_ALL("IMenu::addBitmap");
   IMenuItem mn(newItemId);
   mn.setBitmap(IResourceId(bitmapResId));
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::addBitmap                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addBitmap( unsigned long        newItemId,
                         const IBitmapHandle& bitmapHandle,
                         unsigned long        intoSubmenuId )
{
   IMODTRACE_ALL("IMenu::addBitmap");
   IMenuItem mn(newItemId);
   mn.setBitmap(bitmapHandle);
   return addItem(mn, intoSubmenuId);
}


/*------------------------------------------------------------------------------
| IMenu::addSeparator                                                          |
------------------------------------------------------------------------------*/
IMenu& IMenu::addSeparator( unsigned long  newItemId,
                            unsigned long  intoSubmenuId )
{
   IMODTRACE_ALL("IMenu::addSeparator");
   IMenuItem mn(newItemId, IMenuItem::separator);
   return addItem(mn, intoSubmenuId);
}

/*------------------------------------------------------------------------------
| IMenu::Cursor::setToNext                                                     |
------------------------------------------------------------------------------*/
bool IMenu::Cursor::setToNext()
{
  if (lCurrent >= 0 && lCurrent < pMenu->numberOfItems(ulSubmenu) - 1 ) {
     ++lCurrent;
     return true;
  } else {
     lCurrent = -1L;
     return false;
  } /* endif */
}

/*------------------------------------------------------------------------------
| IMenu::deleteAt                                                              |
------------------------------------------------------------------------------*/
IMenu& IMenu::deleteAt( Cursor& cursor )
{
   cursor.pMenu->deleteItem(cursor.pMenu->menuItemId(cursor));
   if (cursor.lCurrent > 0)
     --cursor.lCurrent;
   return *this;
}

/*------------------------------------------------------------------------------
| IMenu::add                                                                   |
------------------------------------------------------------------------------*/
IMenu& IMenu::add( IMenuItem& menuItem, Cursor& cursor )
{
   menuItem.setIndex(cursor.lCurrent);
   return cursor.pMenu->addItem(menuItem, cursor.ulSubmenu);
}

/*------------------------------------------------------------------------------
| IMenu::addAsNext                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::addAsNext( IMenuItem& menuItem, Cursor& cursor )
{
   menuItem.setIndex(cursor.lCurrent + 1);
   cursor.pMenu->addItem(menuItem, cursor.ulSubmenu);
   cursor.setToNext();
   return *this;
}

/*------------------------------------------------------------------------------
  Have the cursor point to the first element
------------------------------------------------------------------------------*/
bool IMenu::Cursor::setToFirst()
{
  lCurrent = 0;
  return true;
}

/*------------------------------------------------------------------------------
  Have the cursor point to the previous element
------------------------------------------------------------------------------*/
bool IMenu::Cursor::setToPrevious()
{
  --lCurrent;
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
  Invalidate the cursor
------------------------------------------------------------------------------*/
bool IMenu::Cursor::isValid() const
{
  return (lCurrent < 0) ? false : true;
}

/*------------------------------------------------------------------------------
  Invalidate the cursor
------------------------------------------------------------------------------*/
void IMenu::Cursor::invalidate()
{
  lCurrent = -1L;
}

/*------------------------------------------------------------------------------
  Get menu item at cursor position.
------------------------------------------------------------------------------*/
IMenuItem IMenu::elementAt( const Cursor& cursor ) const
{
   return cursor.pMenu->menuItem(cursor.pMenu->menuItemId(cursor));
}

/*------------------------------------------------------------------------------
  Remove submenu at cursor position.
------------------------------------------------------------------------------*/
IMenu& IMenu::removeSubmenuAt( Cursor& cursor )
{
   return cursor.pMenu->removeSubmenu(menuItemId(cursor));
}

/*------------------------------------------------------------------------------
  Function to query menu handle for owner window
------------------------------------------------------------------------------*/
IMenuHandle IMenu::menuHandle() const
{
#ifdef IC_WIN
   return fhMenu;
#endif
#ifdef IC_PM
   return (IMenuHandle)(fWindow->handle());
#endif
}

/*------------------------------------------------------------------------------
| IMenu::isValid                                                               |
|                                                                              |
| Is the menu a valid one?                                                     |
------------------------------------------------------------------------------*/
bool IMenu::isValid() const
{
#ifdef IC_WIN
  if ( menuHandle() != 0 )
     return true;
  else
     return false;
#else
  return fWindow->isValid();
#endif  // IC_WIN
}

/*------------------------------------------------------------------------------
| IMenu::owner                                                                 |
|                                                                              |
| Get the owner of the menu                                                    |
------------------------------------------------------------------------------*/
IWindow* IMenu::owner() const
{
#ifdef IC_WIN
  return IWindow::windowWithHandle( fowner, false );
#else
  return window()->owner();
#endif  // IC_WIN
}

/*------------------------------------------------------------------------------
| IMenu::isFrameWindow                                                         |
|                                                                              |
| Is menu a frame window?  Override here since menus are not windows           |
------------------------------------------------------------------------------*/
bool IMenu::isFrameWindow() const
{
  return false;
}

/*------------------------------------------------------------------------------
| IMenu::id                                                                    |
|                                                                              |
| Get the menu identity.                                                       |
------------------------------------------------------------------------------*/
unsigned long IMenu::id() const
{
  unsigned long ulId = 0;
#ifdef IC_WIN
  if (IPlatform::isWin9x() || IPlatform::isNTNewShell())
    ulId = IMenuPrivate::findIdForPopup( GetMenu(fowner), menuHandle() );
  else
    ulId = IMenuPrivate::getItemIdFromLookUpTable( fmenuFrame, menuHandle() );
#else
  ulId = window()->id();
#endif  // IC_WIN

  return ulId;
}

/*------------------------------------------------------------------------------
| IMenu::enableUpdate                                                          |
|                                                                              |
| Allow further updates to the menu to be painted to the display.              |
------------------------------------------------------------------------------*/
IMenu& IMenu::enableUpdate( bool enableWindow )
{
  IMODTRACE_ALL("IMenu::enableUpdate");
#ifdef IC_WIN
  fMenuData->fUpdateAllowed = enableWindow;
//  IWindow* owner = IWindow::windowWithHandle((IWindowHandle)fowner);
//  owner->enableUpdate( enableWindow ); //AJ: blows for submenus, when fowner=0
  if (enableWindow)
    DrawMenuBar( fowner );
#else
  window()->enableUpdate( enableWindow );
#endif  // IC_WIN
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::disableUpdate                                                         |
|                                                                              |
| Delay updates to the menu to not display                                     |
------------------------------------------------------------------------------*/
IMenu& IMenu::disableUpdate()
{
  return IMenu::enableUpdate( false );
}

/*------------------------------------------------------------------------------
| IMenu::refresh                                                               |
|                                                                              |
| Refresh the menu                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::refresh( IWindow::RefreshType type )
{
  IMODTRACE_ALL("IMenu::refresh");
#ifdef IC_WIN
  ITRACE_ALL(IString(" fUpdateAllowed is: ")+IString(fMenuData->fUpdateAllowed));
  // Only refresh menubar if updates are enabled
  if ( fMenuData->fUpdateAllowed )
    DrawMenuBar( fowner );
#else
  window()->refresh( type );   //IWindow method checks update flag
#endif  // IC_WIN
  return *this;
}

/*------------------------------------------------------------------------------
| IMenu::refresh                                                               |
|                                                                              |
| Refresh the menu                                                             |
------------------------------------------------------------------------------*/
IMenu& IMenu::refresh( const IRectangle& rectArea, bool immediate )
{
  IMODTRACE_ALL("IMenu::refresh(rect, bool)");
#ifdef IC_WIN
  ITRACE_ALL(IString(" fUpdateAllowed is: ")+IString(fMenuData->fUpdateAllowed));
  // Only refresh menubar if updates are enabled
  if ( fMenuData->fUpdateAllowed )
    DrawMenuBar( fowner );
#else
  window()->refresh( rectArea, immediate );
#endif  // IC_WIN
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::menuWithParent                                                        |
|                                                                              |
| Return the handle of the menu associated with the given window.              |
------------------------------------------------------------------------------*/
IMenuHandle IMenu::menuWithParent(IWindow& parent)
{
  IMODTRACE_ALL("IMenu::menuWithParent");
#ifdef IC_WIN
  return GetMenu(parent.handle());
#else
  IMenuHandle menuHandle = 0;
  const IAttributeName menuName(IMenuAttribute::menuAttrName);

  // First, assume this parent has a IMenuBar. Try to find it's handle.
  IFrameWindow* parentFrame = dynamic_cast<IFrameWindow*>(&parent);
  if (parentFrame) {
     HWND frameMenuBar = IWindow::handleWithParent(FID_MENU, parentFrame->handle());
     IWindow* menuBarWrapper =
       IWindow::windowWithHandle(IWindowHandle(frameMenuBar));
     if (menuBarWrapper) {
       const IMenuAttribute* menuAttr =     // get the IMenuBar for this frame (if any)
             dynamic_cast<const IMenuAttribute*>(menuBarWrapper->attributeWithName(menuName));
       if (menuAttr)
          menuHandle = menuAttr->fMenu->menuHandle();
     } //endif menuBarWrapper
  } //endif parentFrame

  if (!menuHandle) {
    // If we didn't find a menubar, iterate thru all parent's children looking for menuAttr
    IWindow::ChildCursor cursor(parent);
    // Enumerate all child windows.
    for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
    {
       IWindow* pwndChild = parent.childWindowAt( cursor );
       const IMenuAttribute* menuAttr =     // get the IMenu for this window (if any)
             dynamic_cast<const IMenuAttribute*>(pwndChild->attributeWithName(menuName));
       if (menuAttr)
          menuHandle = menuAttr->fMenu->menuHandle();
    } //endfor
  } //endif !menuHandle

  return menuHandle;
#endif  // IC_WIN
}


//AJ*************** New methods promoted from IWindow *************************
/*------------------------------------------------------------------------------
| IMenu::show                                                                  |
|                                                                              |
| Show the menu                                                                |
------------------------------------------------------------------------------*/
IMenu& IMenu::show( bool showMenu )
{
  IMODTRACE_ALL("IMenu::show");
#ifdef IC_WIN
  if (fowner)       //in case someone calls this on a non-root ISubmenu
    if (showMenu)
      ShowWindow(fowner, SW_SHOW);
    else
      ShowWindow(fowner, SW_HIDE);
#else
  window()->show(showMenu);
#endif  // IC_WIN
  return *this;
}


/*------------------------------------------------------------------------------
| IMenu::hide                                                                  |
|                                                                              |
| Hide the menu                                                                |
------------------------------------------------------------------------------*/
IMenu& IMenu::hide( )
{
  return show(false);
}


/*------------------------------------------------------------------------------
| IMenu::isVisible                                                             |
|                                                                              |
| Return whether the menu is visible or not                                    |
------------------------------------------------------------------------------*/
bool IMenu::isVisible( ) const
{
#ifdef IC_WIN
  return IsWindowVisible(fowner);
#else
  return window()->isVisible();
#endif  // IC_WIN
}


/*------------------------------------------------------------------------------
| IMenu::isShowing                                                             |
|                                                                              |
| Return whether the menu is unobscured by other windows                       |
------------------------------------------------------------------------------*/
bool IMenu::isShowing( ) const
{
#ifdef IC_WIN
  return isVisible();  //as for IWindow::isShowing, on WIN this isn't always accurate
#else
  return window()->isShowing();
#endif  // IC_WIN
}


/*------------------------------------------------------------------------------
| IMenu::itemProvider, IMenu::setItemProvider                                  |
|                                                                              |
| Can't support it for Win, current design requires an IWindow to associate    |
| with the provider (acts as its "key" in the global provider collection).     |
------------------------------------------------------------------------------*/
#ifndef IC_WIN
IDMItemProvider* IMenu::itemProvider() const
{ return window()->itemProvider(); }


IMenu& IMenu::setItemProvider( IDMItemProvider* dragProvider )
{ window()->setItemProvider(dragProvider); return *this;}
#endif
//*************** end of new methods promoted from IWindow *****************



#ifdef IC_PM
/*------------------------------------------------------------------------------
| IMenu::window                                                                |
|                                                                              |
| Return the underlying IWindow for this menu. Throw if it's invalid.          |
------------------------------------------------------------------------------*/
IWindow* IMenu::window() const
{
  IMODTRACE_ALL("IMenu::window");
// IWindow may be useful even if invalid (e.g., in order to delete it)
//  if ( !fWindow ||
//       !(fWindow->isValid()) )
  if (!fWindow)
     ITHROWLIBRARYERROR( IC_UNEXPECTED_ERROR,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
  return fWindow;
}


/*------------------------------------------------------------------------------
| IMenu::setWindow                                                             |
|                                                                              |
| Set the underlying IWindow for this menu                                     |
------------------------------------------------------------------------------*/
void IMenu::setWindow(IWindow* window)
{
  IMODTRACE_ALL("IMenu::setWindow");

	//must remove the reference if there is an existing fWindow of type MenuWindow
  IMRefCounted* refCnt = dynamic_cast<IMRefCounted*>(fWindow);
  if(refCnt)
  {
		refCnt->removeRef();
  }

	//add a reference if a MenuWindow was passed in
  IMRefCounted* refWin = dynamic_cast<IMRefCounted*>(window);
  if( refWin ) 
  {
    refWin->addRef();
  }
  
  fWindow = window;
  return;
}


/*------------------------------------------------------------------------------
| IMenu::handle                                                                |
|                                                                              |
| Return the handle of the underlying IWindow for this menu, or 0 if none.     |
------------------------------------------------------------------------------*/
IWindowHandle IMenu::handle() const
{
  IMODTRACE_ALL("IMenu::handle");
  if ( !fWindow ||
       !(fWindow->isValid()) )
     ITHROWLIBRARYERROR( IC_INVALIDHANDLE,
                         IBaseErrorInfo::invalidRequest,
                         IException::recoverable );
  return fWindow->handle();
}
#endif


