// Revision: 34 1.72 source/ui/basectl/icanvas.cpp, canvas, ioc.v400, 990114 
/*******************************************************************************
* FILE NAME: icanvas.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   This file contains the implementation of classes/functions declared        *
*   in icanvas.hpp.                                                            *
*                                                                              *
* COPYRIGHT:                                                                   *
*   IBM Open Class Library                                                     *
*   (C) Copyright International Business Machines Corporation 1992, 1997       *
*   Licensed Material - Program-Property of IBM - All Rights Reserved.         *
*   US Government Users Restricted Rights - Use, duplication, or disclosure    *
*   restricted by GSA ADP Schedule Contract with IBM Corp.                     *
*                                                                              *
*******************************************************************************/
#pragma priority( -2147481524 )

#include <ibase.hpp>

extern "C" {
  #ifdef IC_PM
  #define INCL_WINDIALOGS       // DLGC_xxx
  #define INCL_WINFRAMEMGR      // WM_QUERYFOCUSCHAIN
  #define INCL_WININPUT         // WM_MOUSEMOVE, WM_BUTTONxDOWN, WinQueryFocus, etc.
  #define INCL_WINMESSAGEMGR    // WM_COMMAND, etc., CMDSRC_OTHER
  #define INCL_WINPOINTERS      // WinSetPointer
  #define INCL_WINSYS           // PP_BACKGROUNDCOLOR
  #define INCL_WINWINDOWMGR     // QWL_xxx, WinQueryWindowULong
  #endif
  #include <iwindefs.h>
}

#ifdef IC_MOTIF
  #include <canvas.h>
  #include <X11/IntrinsicP.h>
  #include <X11/CoreP.h>        // Widget
  #include <X11/Intrinsic.h>    // XtAddExposureToRegion, etc.
#endif

#include <icanvas.hpp>
#include <icanvprv.hpp>

#include <ibcolor.hpp>
#include <ibidiset.hpp>
#include <ibundles.hpp>
#include <icconst.h>
#include <icolor.hpp>
#include <icoordsy.hpp>
#include <icvhdr.hpp>
#include <iexcept.hpp>
#include <iexgrprt.hpp>
#include <ifont.hpp>
#include <iframe.hpp>
#include <igarea2d.hpp>
#include <igbase2d.hpp>
#include <igline2d.hpp>
#include <igraftxt.hpp>
#include <igrport.hpp>
#include <ipainevt.hpp>
#include <irecohdr.hpp>
#include <ireslib.hpp>
#include <istyles.hpp>
#include <istylset.hpp>
#include <itrace.hpp>
#include <iwposbuf.hpp>
#ifdef IC_PMWIN
#include <icmdevt.hpp>
#include <iplatfrm.hpp>
#include <iwcname.hpp>
#endif
#ifdef IC_PM
#include <ithread.hpp>
#endif
#ifdef IC_MOTIF
#include <ixdc.hpp>
#endif

// Segment definitions.
#ifdef IC_PAGETUNE
  #define _ICANVAS_CPP_
  #include <ipagetun.h>
#endif

#ifdef IC_WIN
extern unsigned long propagationInProgress;
#endif

/*------------------------------------------------------------------------------
| Public canvas styles.                                                        |
------------------------------------------------------------------------------*/
const ICanvas::Style ICanvas::classDefaultStyle = WS_VISIBLE;

#ifdef IC_MOTIF
const char ICanvasData::className[21] = "ICanvas            ";
#endif

/*------------------------------------------------------------------------------
| Default style for new objects (initial value).                               |
------------------------------------------------------------------------------*/
const ICanvas::Style
  ICanvas::border( 0, ICNV_BORDER );
ICanvas::Style ICanvas::currentDefaultStyle     = WS_VISIBLE;

/*------------------------------------------------------------------------------
| ICanvasData::ICanvasData                                                     |
|                                                                              |
| Constructor for the private canvas data class.                               |
------------------------------------------------------------------------------*/
ICanvasData::ICanvasData ( )
  : fShowWindow ( true )
  , fChildCount ( 0 )
  , IAmAnICanvas( false )
  , fHandler( )
  , fClassStyleAllPaints ( false )
#ifdef IC_WIN
  , fSize( )
#endif
#ifdef IC_MOTIF
  , fCanvasHandler( )
  , xlfdString("")
#endif
{ }

/*------------------------------------------------------------------------------
| ICanvasData::~ICanvasData                                                    |
------------------------------------------------------------------------------*/
ICanvasData::~ICanvasData ( )
{
}

/*------------------------------------------------------------------------------
| ICanvas::ICanvas                                                             |
|                                                                              |
| Constructor to create a canvas window.                                       |
------------------------------------------------------------------------------*/
ICanvas::ICanvas ( unsigned long windowIdentifier,
                   IWindow* parent,
                   IWindow* owner,
                   const IRectangle& initial,
                   const Style& style )
    : sizClLayout()
    , bClChildrenReversed( false )
    , bClDefaultHandlerAdded( false )
    , fFont( 0 )
    , fBorderText()
    , fBorderColor( 0 )
    , fCanvasData( new ICanvasData() )
{
  // Save the extended style to make sure we have a copy of it stored.
  this->setExtendedStyle( this->extendedStyle() | style.asExtendedUnsignedLong() );

  fCanvasData->IAmAnICanvas = true;

  // Public ctor must use Window Class that generates WM_PAINT
  // messages even for resizing smaller, allowing a users
  // handler more control over the paint process.
  fCanvasData->fClassStyleAllPaints = true;

  // Create the presentation system window.
  this->initialize( windowIdentifier,
                    parent,
                    owner,
                    initial,
                    this->convertToGUIStyle( style ),
                    this->convertToGUIStyle( style, true ) );

  /*****************************************************************/
  /* Implementation notes:                                         */
  /* - The WS_TABSTOP style is added in later by layout() only if  */
  /*   the canvas has a tab-able child window.  This window style  */
  /*   is needed to allow a frame or canvas parent to identify the */
  /*   canvas as a tab-able child window with WinEnumDlgItem       */
  /*   during Tab key processing (PM does this for a frame window, */
  /*   and the ICanvasHandler class does this for a parent         */
  /*   canvas), and for the ICanvasHandler to give the input focus */
  /*   to the appropriate child window during processing of a      */
  /*   WM_SETFOCUS message.                                        */
  /* - Like all other controls, the initial rectangle is only used */
  /*   if the canvas is not a frame control and not a child of     */
  /*   another canvas window (if it is a child of another canvas,  */
  /*   it will size itself to be able to hold all of its child     */
  /*   windows in calcMinimumSize()).                              */
  /*****************************************************************/

  // Add the default recoordination handler.  This handler is needed
  // for the canvas classes that do not do their own positioning of
  // child windows.   Those that do child window positioning need to
  // later remove the handler.
  IRecoordHandler::defaultHandler()->handleEventsFor( this );
  fCanvasData->fHandler.handleEventsFor( this );
                                            // Add private handler
#ifdef IC_MOTIF
  fCanvasData->fCanvasHandler.handleEventsFor( this );
#endif // IC_MOTIF
  bClDefaultHandlerAdded = true;            // Flag as added

  if ( IWindow::defaultOrdering() == IWindow::behindSiblings )
  {                                    // No need to reverse children
     bClChildrenReversed = true;
  }
}

/*------------------------------------------------------------------------------
| ICanvas::ICanvas                                                             |
------------------------------------------------------------------------------*/
ICanvas::ICanvas ( )
    : sizClLayout()
    , bClChildrenReversed( false )
    , bClDefaultHandlerAdded( false )
    , fFont( 0 )
    , fBorderText()
    , fBorderColor( 0 )
    , fCanvasData( new ICanvasData() )
{
  fCanvasData->IAmAnICanvas = false;

  // Protected ctor must use Window Class that does NOT generate
  // WM_PAINT messages for resizing smaller, thus allowing IOC
  // performance savings during paint processing.
  fCanvasData->fClassStyleAllPaints = false;

  fCanvasData->fHandler.handleEventsFor( this );
#ifdef IC_MOTIF
  fCanvasData->fCanvasHandler.handleEventsFor( this );
#endif // IC_MOTIF
  bClDefaultHandlerAdded = true;            // Flag as added.
  // Assume that derived classes that need the recoordination handler
  // will add one themselves (canvases that do not do their own
  // positioning of child windows).

  if ( IWindow::defaultOrdering() == IWindow::behindSiblings )
  {                                    // No need to reverse children.
     bClChildrenReversed = true;
  }
}

/*------------------------------------------------------------------------------
| ICanvas::initialize                                                          |
|                                                                              |
| Create the presentation system window.                                       |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::initialize ( unsigned long     windowIdentifier,
                               IWindow*          parent,
                               IWindow*          owner,
                               const IRectangle& initialRect,
                               unsigned long     style,
                               unsigned long     extendedStyle )
{
  IASSERTPARM( parent != 0 );

#ifdef IC_PMWIN
  bool bSuc = false;

  // Register the private window class for canvases
  if ( ( (fCanvasData->fClassStyleAllPaints) &&
       (! ICanvasStatics::isWindowClassRegistered(
               ICanvasStatics::kClassStyleMaxPaints) ) ) ||
       ( (!fCanvasData->fClassStyleAllPaints) &&
       (! ICanvasStatics::isWindowClassRegistered(
               ICanvasStatics::kClassStyleMinPaints) ) ) )
  {
#ifdef IC_WIN
    WNDCLASS wndClass;

    wndClass.style = CS_DBLCLKS;
    wndClass.lpszClassName = IC_WC_IOC_CANVAS;
    if (fCanvasData->fClassStyleAllPaints)
    {
      wndClass.style = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
      wndClass.lpszClassName = IC_WC_IOC_CANVAS_AP;
    }
    wndClass.lpfnWndProc = (WNDPROC)_pfnwpICCanProc;
    wndClass.cbClsExtra = 0;
    wndClass.cbWndExtra = 0;
    wndClass.hInstance = GetModuleHandle( 0 );
    wndClass.hIcon = 0;
    wndClass.hCursor = LoadCursor( NULL, IDC_ARROW );
    wndClass.hbrBackground = (HBRUSH)0;
    wndClass.lpszMenuName = 0;

    bSuc = RegisterClass( &wndClass );
#else
    unsigned long classStyle = CS_HITTEST;
    char *className = IC_WC_IOC_CANVAS;
    if (fCanvasData->fClassStyleAllPaints)
    {
      classStyle = CS_HITTEST | CS_SIZEREDRAW;
      className = IC_WC_IOC_CANVAS_AP;
    }
    IThread::current().initializeGUI();    // ensure the GUI is started
    bSuc = (bool) WinRegisterClass( IThread::current().anchorBlock(),
                                    className,
                                    _pfnwpICCanProc,
                                    classStyle,   // Class style
                                    4 );          // Need QWL_USER only.
#endif // IC_WIN

    if ( bSuc )
    {
      if (fCanvasData->fClassStyleAllPaints)
      {
        ICanvasStatics::setWindowClassRegistered(
               ICanvasStatics::kClassStyleMaxPaints);
      }
      else
      {
        ICanvasStatics::setWindowClassRegistered(
               ICanvasStatics::kClassStyleMinPaints);
      }
    }
    else
    {
       ITHROWGUIERROR( "RegisterClass" );
    }
  }

  IWindowHandle parentHandle(0);
  IWindowHandle ownerHandle(0);
  if (parent)
    parentHandle = parent->handleForChildCreation();
  else
    parentHandle = IWindow::desktopWindow()->handle();
  if (owner)
    ownerHandle = owner->handle();

  char *className = IC_WC_IOC_CANVAS;
  if (fCanvasData->fClassStyleAllPaints)
    className = IC_WC_IOC_CANVAS_AP;
  IWindowHandle canvasHandle =
                   this->create( windowIdentifier,
                                 0,
                                 style,
                                 className,
                                 parentHandle,
                                 ownerHandle,
                                 initialRect,
                                 0,
                                 0,
                                 IWindow::defaultOrdering(),
                                 extendedStyle );
  this->startHandlingEventsFor( canvasHandle );

#ifdef IC_WIN
  // Replace this with a call to IWindow::isParentFrameADialog()
  IWindow *fParentWindow = parent;
  while (fParentWindow && !fParentWindow->isFrameWindow() )
  {
    fParentWindow = fParentWindow->parent();
  }
  if (fParentWindow)
  {
    IFrameWindow *frameWindow =
      dynamic_cast< IFrameWindow *>(fParentWindow);
    if (frameWindow && frameWindow->usesDialogBackground())
    {
      setColor( PP_BACKGROUNDCOLOR, IColor(IColor::kDialogBgnd));
    }
    else
    {
      setColor( PP_BACKGROUNDCOLOR, IColor(IColor::kWindowBgnd));
      setColor( PP_FOREGROUNDCOLOR, IColor(IColor::kWindowStaticText));
    }
  }
#endif // IC_WIN

#endif // IC_PMWIN
#ifdef IC_MOTIF
  Arg args[10];
  int n( 0 );
  IWindowHandle whCanvas;

  // Temporary change to avoid creeping controls bug: hardcode margins to 0
  XtSetArg(args[n], XmNmarginHeight, 0 ); n++;
  XtSetArg(args[n], XmNmarginWidth, 0 ); n++;
  // d6953 - explicitly set the borderWidth to zero, since IMultiCellCanvas
  // assumes it in its layout calculations.
  XtSetArg(args[n], XmNborderWidth, 0 ); n++;

  XtSetArg( args[n], XmNuserData, (XtPointer) fCanvasData->className );
  IWindowHandle parentHandle(0);
  IWindowHandle ownerHandle(0);
  if (parent)
    parentHandle = parent->handleForChildCreation();
  else
    parentHandle = IWindow::desktopWindow()->handle();
  if (owner)
    ownerHandle = owner->handle();

  whCanvas = Inherited::create( windowIdentifier,
                                NULL,
                                style,
                                (IXmCreateFunction) XiclCreateCanvas,
                                parentHandle,
                                ownerHandle,
                                initialRect,
                                args,
                                n );
  this->startHandlingEventsFor( whCanvas );
  this->registerCallbacks();
#endif // IC_MOTIF

  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::~ICanvas                                                            |
------------------------------------------------------------------------------*/
ICanvas::~ICanvas ( )
{
  // Remove the recoordination handler.  May be a noop if never added,
  // or a derived class already removed it in its ctor.
  IRecoordHandler::defaultHandler()->stopHandlingEventsFor( this );

  if ( bClDefaultHandlerAdded )
  {                               // Remove the private handler.
     fCanvasData->fHandler.stopHandlingEventsFor( this );
#ifdef IC_MOTIF
     fCanvasData->fCanvasHandler.stopHandlingEventsFor( this );
#endif
     bClDefaultHandlerAdded = false;        // Flag as removed.
  }

#ifdef IC_MOTIF
  unregisterCallbacks();
#endif

  if ( fFont )
  {
     delete fFont;
  }
  if ( fBorderColor )
  {
     delete fBorderColor;
  }
  if ( fCanvasData )
  {
     delete fCanvasData;
  }
}

/*------------------------------------------------------------------------------
| ICanvas::defaultStyle                                                        |
|                                                                              |
| Return the current default style for the class.                              |
------------------------------------------------------------------------------*/
ICanvas::Style ICanvas::defaultStyle ( )
{
  return currentDefaultStyle;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ICanvas::setDefaultStyle                                                     |
|                                                                              |
| Sets the current default style for the class.                                |
------------------------------------------------------------------------------*/
void ICanvas::setDefaultStyle ( const ICanvas::Style& style )
{
  currentDefaultStyle = style;
}
#endif

/*------------------------------------------------------------------------------
| ICanvas::convertToGUIStyle                                                   |
|                                                                              |
| Returns base style for the control by default, or extended style if          |
| extended flag (bExtOnly) is set.                                             |
------------------------------------------------------------------------------*/
unsigned long ICanvas::convertToGUIStyle ( const IBitFlag& guiStyle,
                                           bool            bExtOnly ) const
{
  // Obtain the style from the class (IControl) that we inherit from.
  unsigned long ulStyle = Inherited::convertToGUIStyle( guiStyle, bExtOnly );

  if ( bExtOnly )
  {
    // Use mask to only return extended styles in the user defined range.
    ulStyle |= guiStyle.asExtendedUnsignedLong() & IS_EXTMASK;
  }
  else
  {
    // No styles are defined for the base canvas, so we do not mask.
    ulStyle |= guiStyle.asUnsignedLong()
               | IWindow::group.asUnsignedLong()
#ifdef IC_WIN
               | WS_CHILD
#endif
#ifdef IC_PMWIN
    // Add the canvas window style.
               | IC_CVS_CANVAS
#endif
      ;
  }

  return ulStyle;
}

/*------------------------------------------------------------------------------
| ICanvas::hasBorder                                                           |
|                                                                              |
| Specifies whether the canvas has the ICanvas::border style to draw a border  |
| around the edges of its window.                                              |
------------------------------------------------------------------------------*/
bool ICanvas::hasBorder ( ) const
{
  return ( this->extendedStyle() & ICanvas::border.asExtendedUnsignedLong()
             ? true : false );
}

/*------------------------------------------------------------------------------
| ICanvas::addBorder                                                           |
|                                                                              |
| Adds the ICanvas::border style to the canvas.  This style causes the canvas  |
| to draw a border, and optionally text, around the edges of its window when   |
| it paints (however, caling this function does not cause it to immediately    |
| paint).  This function may cause a derived class to change the size and/or   |
| position of its child windows.                                               |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::addBorder ( )
{
  if ( ! this->hasBorder() )
  {
#ifdef IC_MOTIF
     if ( ! this->isClassStyleAllPaints() )
     {
        // Remove bit gravity, so the canvas paints its entire window
        // whenever it is sized, including resized smaller. This keeps
        // the borders from a previous paint from being left on the
        // screen after a partial paint.
        XSetWindowAttributes
          attributes;
        attributes.bit_gravity = ForgetGravity;
        XChangeWindowAttributes( XtDisplay( (Widget) this->handle() ),
                                 XtWindow( (Widget) this->handle() ),
                                 CWBitGravity,
                                 &attributes );
     }
#endif // IC_MOTIF

     this->setExtendedStyle( this->extendedStyle()
                              | ICanvas::border.asExtendedUnsignedLong() );
     this->setLayoutDistorted( IWindow::minimumSizeChanged
                                | IWindow::layoutChanged
                                | IWindow::immediateUpdate, 0 );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::removeBorder                                                        |
|                                                                              |
| Removes the ICanvas::border style from the canvas, causing it to no longer   |
| draw a border and text around the edges of its window when it paints         |
| (however, caling this function does not cause it to immediately paint).      |
| This function may cause a derived class to change the size and/or position   |
| of its child windows.                                                        |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::removeBorder ( )
{
  if ( this->hasBorder() )
  {
#ifdef IC_MOTIF
     if ( ! this->isClassStyleAllPaints() )
     {
        // Add bit gravity, so the canvas repaints only invalidated
        // areas when it is resized. This is a performance feature
        // to reduce the number of paints that occur during resizing.
        XSetWindowAttributes
          attributes;
        attributes.bit_gravity = StaticGravity;
        XChangeWindowAttributes( XtDisplay( (Widget) this->handle() ),
                                 XtWindow( (Widget) this->handle() ),
                                 CWBitGravity,
                                 &attributes );
     }
#endif // IC_MOTIF

     this->setExtendedStyle( this->extendedStyle()
                              & ~ICanvas::border.asExtendedUnsignedLong() );
     this->setLayoutDistorted( IWindow::minimumSizeChanged
                                | IWindow::layoutChanged
                                | IWindow::immediateUpdate, 0 );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::borderText                                                          |
|                                                                              |
| Returns the text that the canvas uses to draw part of its top border.  Note  |
| that the canvas must have the ICanvas::border style to draw the text and     |
| border.                                                                      |
------------------------------------------------------------------------------*/
IText ICanvas::borderText ( ) const
{
  return fBorderText;
}

/*------------------------------------------------------------------------------
| ICanvas::setBorderText                                                       |
|                                                                              |
| Assigns border text to the canvas.  If the canvas also has the               |
| ICanvas::border style, it will paint this text as part of its top border     |
| (however, caling this function does not cause it to immediately paint).      |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setBorderText ( const IText& borderText,
                                  bool         showBorder )
{
  // Make a copy of the text.
  fBorderText = borderText;

  if ( showBorder  &&  ! this->hasBorder() )
  {
     this->addBorder();
  }
  else
  {  // Change in the text being displayed.
     // Duplicate the notification in ICanvas::addBorder
     // (assume this changes the minimum size).
     this->setLayoutDistorted( IWindow::minimumSizeChanged
                                | IWindow::layoutChanged
                                | IWindow::immediateUpdate, 0 );
  }

  // Assume there is a change in the text.
  this->notifyObservers( INotificationEvent( *this, ICanvas::textId ) );

  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::setBorderText                                                       |
|                                                                              |
| Assigns border text to the canvas.  If the canvas also has the               |
| ICanvas::border style, it wil paint this text as part of its top border      |
| (however, caling this function does not cause it to immediately paint).      |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setBorderText ( const IResourceId& borderText,
                                  bool               showBorder )
{
  IString text = borderText.resourceLibrary().loadString( borderText );
  return this->setBorderText( text, showBorder );
}

/*------------------------------------------------------------------------------
| ICanvas::borderColor                                                         |
|                                                                              |
| Returns the color that the canvas uses to paint its border.                  |
------------------------------------------------------------------------------*/
IColor ICanvas::borderColor ( ) const
{
  if ( fBorderColor )
  {
     return *fBorderColor;
  }
  else
  {
     return IColor( IColor::kButtonDark );
  }
}

/*------------------------------------------------------------------------------
| ICanvas::setBorderColor                                                      |
|                                                                              |
| Assigns the color that the canvas uses to paint its border.                  |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setBorderColor ( const IColor& borderColor )
{
  if ( fBorderColor )
  {
     *fBorderColor = borderColor;
  }
  else
  {
     fBorderColor = new IColor( borderColor );
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::resetBorderColor                                                    |
|                                                                              |
| Sets the color that the canvas uses to paint its border back to the default. |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::resetBorderColor ( )
{
  if ( fBorderColor )
  {
     delete fBorderColor;
     fBorderColor = 0;
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::defaultBorderTextColor                                              |
|                                                                              |
| Returns the color that the canvas uses to paint its border text.             |
------------------------------------------------------------------------------*/
IColor ICanvas::defaultBorderTextColor ( ) const
{
#ifdef IC_PMWIN
  return IWindow::color( PP_FOREGROUNDCOLOR,
                         IColor( IColor::kWindowStaticText ) );
#else
  return this->foregroundColor();
#endif
}

/*------------------------------------------------------------------------------
| ICanvas::isClassStyleAllPaints                                               |
|                                                                              |
| Determines if the canvas was created using the window class which generates  |
| all WM_PAINT messages or not.                                                |
------------------------------------------------------------------------------*/
bool ICanvas::isClassStyleAllPaints ( ) const
{
  return ( this->fCanvasData->fClassStyleAllPaints ? true : false );
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ICanvas::backgroundColor                                                     |
|                                                                              |
| Returns the background color of the canvas.                                  |
------------------------------------------------------------------------------*/
IColor ICanvas::backgroundColor ( ) const
{
  return Inherited::backgroundColor();
}
#endif // IC_PMWIN

#ifdef IC_MOTIFWIN
/*------------------------------------------------------------------------------
| ICanvas::setFont                                                             |
|                                                                              |
| Sets the font for the canvas.                                                |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setFont ( const IFont& fm )
{
#ifdef IC_WIN
  // First save this font in our window data before propagating.
  // Note that storing the font in the ICanvas::fFont member data is needed
  // to support the separate requirement for constructing an IFont from the
  // window handle of a canvas.  Calling IWindow::setFont is needed to
  // propagate the font to owned windows.  This order is important, so that
  // the font in ICanvas::fFont is stored before IWindow::setLayoutDistorted
  // is called with IWindow::fontChanged.
  if ( this->fFont )
  {                // Clean up the prior font.
     delete this->fFont;
  }
  this->fFont = new IFont( fm );
  Inherited::setFont( fm );

  // Bump in propagation static by 1 (since nested calls could occur).
  propagationInProgress++;

  // Now lets propagate this font to all of the children of the canvas.
  // This is to give it similar behavior to OS/2 pres-params support.
  IWindowHandle hwndChild = 0;
  IWindow::ChildCursor cursor( *(IWindow*)this );

  // Enumerate all child windows.
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
    // Get the IWindow object for each child found
    bool bNewChild = false;
    IWindowHandle hwndChild = this->childAt( cursor );
    IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );

    // If no IWindow object exists, create one temporarily
    if ( !pwndChild )
    {
       pwndChild = new IWindow( hwndChild );
       bNewChild = true;
    }

    // Set the font for that window
    pwndChild->setFont( fm );

    // Delete any temporary IWindow object created (if one was needed)
    if ( bNewChild )
       delete pwndChild;
  }
  propagationInProgress--;
#endif //IC_WIN
#ifdef IC_MOTIF
  // cache X Windows Logical Font Descriptor (XLFD) string
  fCanvasData->xlfdString = fm.qualifiedName();
  Inherited::setFont( fm );
#endif //IC_MOTIF
  return *this;
}

#ifdef IC_WIN
/*------------------------------------------------------------------------------
| ICanvas::resetFont                                                           |
|                                                                              |
| Resets the font for the canvas.                                              |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::resetFont ( )
{
  // First delete any existing font in our private data.
  if ( this->fFont )
  {                // Clean up the prior font.
     delete this->fFont;
     this->fFont = 0;
  }
  Inherited::resetFont();

  // Now lets propagate this font to all of the children of the canvas.
  // This is to give it similar behavior to OS/2 pres-params support.
  IWindowHandle hwndChild = 0;
  IWindow::ChildCursor cursor( *(IWindow*)this );

  // Enumerate all child windows.
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {
    // Get the IWindow object for each child found
    bool bNewChild = false;
    IWindowHandle hwndChild = this->childAt( cursor );
    IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );

    // If no IWindow object exists, create one temporarily
    if ( !pwndChild )
    {
       pwndChild = new IWindow( hwndChild );
       bNewChild = true;
    }

    // Set the font for that window
    pwndChild->resetFont();

    // Delete any temporary IWindow object created (if one was needed)
    if ( bNewChild )
       delete pwndChild;
  }

  return *this;
}
#endif //IC_WIN

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICanvas::font                                                                |
|                                                                              |
| Returns the font used for the canvas.                                        |
------------------------------------------------------------------------------*/
IFont ICanvas::font () const
{
  Inherited::font();
  if (fCanvasData->xlfdString.length() > 0)
    // construct font based on cached XLFD string
    return IFont(fCanvasData->xlfdString);
  else
    return IFont(this);
}
#endif //IC_MOTIF
#endif // IC_MOTIFWIN

/*------------------------------------------------------------------------------
| ICanvas::isTabStop                                                           |
|                                                                              |
| Returns true if the canvas can be tabbed to based on whether any of its      |
| child windows can be tabbed to.                                              |
------------------------------------------------------------------------------*/
bool ICanvas::isTabStop ( ) const
{
  bool bTabStop = false;       // Assume no child windows are tab stops.
  IWindowHandle hwndChild = 0;
  IWindow::ChildCursor cursor( *(IWindow*)this );

  for ( cursor.setToFirst();
        cursor.isValid()  &&  bTabStop == false;
        cursor.setToNext() )
  {                               // Enumerate all child windows.
     IWindowHandle hwndChild = this->childAt( cursor );
     IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );
     if ( !pwndChild )
     {                            // Child window object not found.
#ifdef IC_PMWIN
              // Note: Can just check style directly, since any
              // non-IWindow child cannot be another canvas, could
              // not call isTabStop() function anyway unless we wrapper.
        if ( hwndChild )
        {
           if ( ISTYLEOF(hwndChild) & WS_TABSTOP )
           {
              bTabStop = true;
           }
        }
#endif // IC_PMWIN
     }
     else
     {                            // Child window object found.
        bTabStop = pwndChild->isTabStop();
              // Note: Can't just check style directly, since if the
              // child is another canvas, its isTabStop() function
              // needs to be called.
     }
  }                               // End for all child windows.

#ifdef IC_PMWIN
  ((ICanvas*)this)->enableTabStop( bTabStop );
                 // Make non-const to store appropriate tabStop style.
#endif // IC_PMWIN

  return bTabStop;
}

/*------------------------------------------------------------------------------
| ICanvas::matchForMnemonic                                                    |
|                                                                              |
| Returns the first child window that uses the specified character as a        |
| mnemonic.                                                                    |
------------------------------------------------------------------------------*/
IWindowHandle ICanvas::matchForMnemonic ( unsigned short character ) const
{
  return ICanvasStatics::matchForMnemonic( this, character );
}

/*------------------------------------------------------------------------------
| ICanvas::calcMinimumSize                                                     |
|                                                                              |
| Calculate the minimum screen size needed by the canvas.  This is the size    |
| needed to display all child windows (as determined by the layout routine).   |
------------------------------------------------------------------------------*/
ISize ICanvas::calcMinimumSize ( ) const
{
  ((ICanvas*)this)->layout();        // Calculate minimum size.
  ISize sizMin = layoutSize();       // Use size from layout().
  return sizMin;
}

/*------------------------------------------------------------------------------
| ICanvas::hasChildrenToLayout                                                 |
|                                                                              |
| Identify whether the canvas contains any child windows to manage.            |
------------------------------------------------------------------------------*/
bool ICanvas::hasChildrenToLayout ( ) const
{
  ICanvas
   *nonConstThis = (ICanvas*)this;
  ChildCursor
    cursor( *nonConstThis );
  return cursor.setToFirst();
}

/*------------------------------------------------------------------------------
| ICanvas::setLayoutDistorted                                                  |
|                                                                              |
| Provide common canvas processing for various distortion notifications.       |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setLayoutDistorted( unsigned long layoutAttributeOn,
                                      unsigned long layoutAttributeOff )
{
  unsigned long ulFlagsOn = layoutAttributeOn;

  if ( layoutAttributeOn & IWindow::childWindowCreated )
  {                                    // New child window.
     ulFlagsOn &= (unsigned long)(~IWindow::childWindowCreated);
     ulFlagsOn |= IWindow::layoutChanged;
  }                                    // Need new layout.
  if ( layoutAttributeOn & IWindow::childWindowDestroyed )
  {                                    // Lost a child window.
     ulFlagsOn &= (unsigned long)(~IWindow::childWindowDestroyed);
     ulFlagsOn |= IWindow::layoutChanged;
  }                                    // Need new layout.
  if ( layoutAttributeOn & IWindow::fontChanged )
  {           // Assume the border font is changing.
     if ( this->hasBorder() )
     {        // Using a border and maybe border text.
        if ( ! this->borderText().isStyled() )
        {  // Border text has no explicit font, so inherits the font
           // of the canvas.
          ulFlagsOn |= ( IWindow::layoutChanged |
                        IWindow::immediateUpdate );
        }
     }
  }
  if ( layoutAttributeOn & IWindow::colorChanged )
  {  // Canvas needs to repaint with the new color.
     this->refresh();
  }

  Inherited::setLayoutDistorted( ulFlagsOn, layoutAttributeOff );

  if ( layoutAttributeOn  &&
       this->isLayoutDistorted( IWindow::immediateUpdate )  &&
       this->isVisible() )
  {                                    // Need to update canvas appearance.
     Inherited::setLayoutDistorted( 0, IWindow::immediateUpdate );
     if ( this->size() > ISize() )
     {                                 // Truly visible window.
        this->refresh();               // Don't bypass paint handlers.
     }
     else                              // Height or width is 0.
     {                                 // Will never get a paint.
        this->layout();                // Circumvent WM_PAINT.
     }
  }
  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::layout                                                              |
|                                                                              |
| This method should be overridden by derived classes to position/size child   |
| windows.                                                                     |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::layout ( )
{
  IMODTRACE_DEVELOP( "ICanvas::layout()" );

  if ( this->isLayoutDistorted( IWindow::layoutChanged ))
  {
     ITRACE_DEVELOP( "isLayoutDistorted(layoutChanged)==true" );
     /**************************************************************/
     /* The layout is distorted so see if this canvas needs        */
     /* updating.                                                  */
     /*                                                            */
     /* The user is responsible for positioning child windows and  */
     /* sizing the canvas large enough so that all child windows   */
     /* are contained by the canvas.                               */
     /*                                                            */
     /* Some of this responsibility will be easied if the canvas   */
     /* is a child of a specialized canvas (one that sizes and     */
     /* positions children according to minimumSize()).            */
     /* minimumSize() calls ICanvas::calcMinimumSize() if the user */
     /* has not specified a minimum size himself.  This function   */
     /* will return the size stored by the layout() routine, one   */
     /* large enough to contain all the child windows.             */
     /**************************************************************/
     ISize sizLayout = this->layoutSize(); // Save original size for children.

     IWindowPosBuffer wposbuf = this->fixupChildren();
     wposbuf.apply( false );           // Reorder z-order of children.

     this->setLayoutDistorted( 0, IWindow::layoutChanged );

     // Update the size needed to contain all child windows, set by
     // ICanvas::fixupChildren.
     ISize newLayoutSize( this->layoutSize() );
     newLayoutSize = this->sizeWithBorder( newLayoutSize );

     if ( sizLayout != newLayoutSize ) // New minimum size for children.
     {                                 // Cause layout in parent canvas.
        this->setLayoutDistorted( IWindow::minimumSizeChanged, 0 );
     }
  }   /* end isDistorted */

  return *this;
}

/*------------------------------------------------------------------------------
| ICanvas::fixupChildren                                                       |
|                                                                              |
| Generates a list of all child windows, determines whether any can be tabbed  |
| to, and allows for reversing the order in which they are iterated.           |
------------------------------------------------------------------------------*/
IWindowPosBuffer ICanvas::fixupChildren ( )
{
  IMODTRACE_DEVELOP( "ICanvas::fixupChildren()" );
  IWindowPosBuffer wposbuf( this );
#ifdef IC_PMWIN
  /*****************************************************************/
  /* Dynamically set the WS_TABSTOP style based on whether the     */
  /* canvas has any tab-able children (WS_TABSTOP is needed so     */
  /* PM's frame window procedure and the ICanvasHandler will       */
  /* consider giving the input focus to the canvas during Tab key  */
  /* processing, and so the canvas can receive WM_SETFOCUS and     */
  /* give the input focus to an appropriate child window).         */
  /*                                                               */
  /* Additionally, reverse the z-order of the canvas' child        */
  /* windows, since the order that windows are constructed and     */
  /* expected to be layed out is typically the opposite of the     */
  /* z-order.  Because of complications involving tabbing and      */
  /* cursor movement with autoselect radio buttons, it became far  */
  /* easier to reverse the z-order of the child windows than to    */
  /* compensate in the tab and cursor movement code to fix this.   */
  /*****************************************************************/
  unsigned long childCount = 0;
  bool bTabControl = false;         // Assume no tab-able children.
  IPoint ptMax;
  IWindowHandle hwndChild = 0;
  IWindowHandle hwndLastChild = 0;
  bool
    defaultPBFound = ( this->defaultPushButton() == 0 ) ?
                          false : true;

  IWindowHandle canvasHwnd = this->handle();
  IWindow::ChildCursor cursor( *(IWindow*)this );
  for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
  {                                    // Enumerate all child controls.
     IPoint ptChild;
     childCount++;
     hwndChild = this->childAt( cursor );
     IWindow* pwndChild = IWindow::windowWithHandle( hwndChild );
     if ( ! pwndChild )
     {                                 // A non-wrappered window.
        if ( hwndChild )
        {
#ifdef IC_PM
           SWP swp;
           WinQueryWindowPos( hwndChild, &swp );
           ptChild = IPoint (swp.x+swp.cx, swp.y+swp.cy);
#endif
#ifdef IC_WIN
           RECT rect;
           IQUERYWINDOWRECT( hwndChild, &rect );
           MapWindowPoints( HWND_DESKTOP,
                            canvasHwnd,
                            (POINT*) &rect,
                            2);
           IRectangle aRect(rect);
           ptChild = aRect.maxXMaxY();
#endif // IC_WIN
           if ( bTabControl == false )
           {
              if ( ISTYLEOF(hwndChild) & WS_TABSTOP )
              {
                 bTabControl = true;
              }
           }
        }
     }
     else
     {
        ptChild = pwndChild->rect().maxXMaxY();
        if ( bTabControl == false )
        {                                 // No tab-able children found yet.
           if ( pwndChild->isTabStop() )
           {                         // Child window that can be tabbed to.
              bTabControl = true;         // Flag as such.
           }
           /*******************************************************/
           /* Note: if child is a canvas, isTabStop() will allow  */
           /* it to check all its children for a tab-able control)*/
           /*******************************************************/
        }
     }
     ptMax = ptMax.maximum( ptChild );

     if ( bClChildrenReversed == false )
     {                              // Need to reverse z-order of children.
        if ( hwndLastChild )
        {                           // Have a preceding child.
           wposbuf.orderAfter( hwndLastChild, hwndChild );
        }                           // Push preceding towards bottom.
        hwndLastChild = hwndChild;
     }                              // End reverse children.

     if ( ! defaultPBFound )
     {             // Frame doesn't have a default push button yet.
        if ( hwndChild.sendEvent( WM_QUERYDLGCODE, 0, 0 ).asUnsignedLong() &
             DLGC_DEFAULT)
        {                           // Found a default push button.
           defaultPBFound = true;
        }                           // End found a default push button.
     }                              // End default push button search.
  }                                 // End enumerate all children.

  fCanvasData->fChildCount = childCount;
  if ( bClChildrenReversed == false  &&  hwndLastChild )
  {                                 // Need to reverse z-order of children.
     wposbuf.orderAsFirst( hwndLastChild );  // Re-order last one.
     bClChildrenReversed = true;
  }                                 // Make topmost child.
  this->enableTabStop( bTabControl );
  this->setLayoutSize( ISize( ptMax ));  // Store size used by layout.
#endif // IC_PMWIN

  return wposbuf;
}

/*------------------------------------------------------------------------------
| ICanvas::topLeftLayoutOffset                                                 |
|                                                                              |
| Returns the offset from the top-left corner of the canvas where the canvas   |
| can display its child windows.  The canvas displays its border and border    |
| text above and to the left of this point.  If the canvas does not have the   |
| ICanvas::border style, this function returns a zero offset.                  |
------------------------------------------------------------------------------*/
IPoint ICanvas::topLeftLayoutOffset ( ) const
{
  IPoint
    layoutOffset;

  if ( this->hasBorder() )
  {
     // Use the border font height to center the top line of the border,
     // even if the canvas does not have border text.
     IPresSpaceHandle
       hps( this->presSpace() );
     {  // Scope the grafport inside the lifetime of IPresSpaceHandle.
        IExtendedRootGrafPort
          borderPort( hps, ICoordinateSystem::kOriginUpperLeft );
        IGraphicText
          borderLabel( this->stylizedBorderText(), IGraphicText::kSingleLine );
        IGRect2D
          textRect( borderLabel.looseFitBounds( &borderPort ) );
        layoutOffset += IPoint( this->borderThickness(),
                                (unsigned long) textRect.height() );
     }
     this->releasePresSpace( hps );

     // Also add in the margin between the border and canvas contents.
     layoutOffset += this->borderMargin();
  }

  return layoutOffset;
}

/*------------------------------------------------------------------------------
| ICanvas::bottomRightLayoutOffset                                             |
|                                                                              |
| Returns the offset from the bottom-right corner of the canvas where the      |
| canvas can display its child windows.  The canvas displays its border below  |
| and to the right of this point.  If the canvas does not have the             |
| ICanvas::border style, this function returns a zero offset.                  |
------------------------------------------------------------------------------*/
IPoint ICanvas::bottomRightLayoutOffset ( ) const
{
  IPoint
    layoutOffset;

  if ( this->hasBorder() )
  {
     layoutOffset += IPoint( this->borderThickness() );
     layoutOffset += this->borderMargin();
  }

  return layoutOffset;
}

/*------------------------------------------------------------------------------
| ICanvas::rectangleInsideBorder                                               |
|                                                                              |
| Calculates and returns the portion of the canvas not used for the border,    |
| relative to the canvas window.  This is the portion of the canvas used to    |
| display child windows.  If the canvas does not use the ICanvas::border       |
| style, this function returns the rectangle for the canvas.                   |
------------------------------------------------------------------------------*/
IRectangle ICanvas::rectangleInsideBorder ( const ISize& sizeWithBorder ) const
{
  IRectangle
    rectWithinBorder( IPoint( 0, 0 ), sizeWithBorder );

  if ( this->hasBorder() )
  {
     // Canvas has borders to consider.
     IPoint
       topLeft( this->topLeftLayoutOffset() ),
       bottomRight( this->bottomRightLayoutOffset() );
     ISize
       totalBorder( topLeft + bottomRight );

     if ( ( rectWithinBorder.height() <= totalBorder.height() )  ||
          ( rectWithinBorder.width() <= totalBorder.width() ) )
     {    // Borders consume entire rectangle.
        rectWithinBorder
         .sizeTo( ISize( 0, 0 ) );
     }
     else
     {  // Find rectangle inside the border.
        rectWithinBorder
         .sizeTo( rectWithinBorder.size() - totalBorder );
        if ( ICoordinateSystem::applicationOrientation() ==
                            ICoordinateSystem::kOriginLowerLeft )
        {
           rectWithinBorder
            .moveBy( IPoint( topLeft.x(), bottomRight.y() ) );
        }
        else
        {
           rectWithinBorder
            .moveBy( topLeft );
        }
     }
  }

  return rectWithinBorder;
}

/*------------------------------------------------------------------------------
| ICanvas::sizeWithBorder                                                      |
|                                                                              |
| Calculates and returns the size needed for the canvas to include a border    |
| and border text around the specified contents size.  If the canvas does not  |
| use the ICanvas::border style, the input size is returned.  Otherwise this   |
| function considers the size added by the border, border margin, and border   |
| text given the border font.                                                  |
------------------------------------------------------------------------------*/
ISize ICanvas::sizeWithBorder ( const ISize& sizeWithoutBorder ) const
{
  ISize
    adjustedSize( sizeWithoutBorder );

  if ( this->hasBorder() )
  {  // The canvas displays its contents within a border.
     // Add in the border thickness and space between the border and
     // the normal canvas contents.
     adjustedSize += ( this->topLeftLayoutOffset() +
                       this->bottomRightLayoutOffset() );

     if ( this->borderText().length() )
     {  // Need to consider the width of the border text (the height has already
        // been added through ICanvas::topLeftLayoutOffset).
        unsigned long
          drawnTextWidth = 0;
        IPresSpaceHandle
          hps( this->presSpace() );
        {  // Scope the grafport inside the lifetime of IPresSpaceHandle.
           IExtendedRootGrafPort
             grafPort( hps, ICoordinateSystem::kOriginUpperLeft );
           IGraphicText
             theLabel( this->stylizedBorderText(), IGraphicText::kSingleLine );
           drawnTextWidth = (unsigned long)
                              theLabel.looseFitBounds( &grafPort ).width();
        }
        this->releasePresSpace( hps );

        // Add in the margins around the text (11 pels on the left; 7 pels
        // plus the width of the right border on the right).
        drawnTextWidth += ( 11 + 7 + this->borderThickness() );

        if ( drawnTextWidth > adjustedSize.width() )
        {
           adjustedSize.setWidth( drawnTextWidth );
        }
     }
  }

  return adjustedSize;
}

/*------------------------------------------------------------------------------
| ICanvas::borderThickness                                                     |
|                                                                              |
| Returns the thickness of the border that the canvas draws when the           |
| application sets the ICanvas::border style or calls ICanvas::addBorder.      |
------------------------------------------------------------------------------*/
unsigned long ICanvas::borderThickness ( ) const
{
  #define BORDER_THICKNESS  2     // Main border + hilight/shadow.
  return BORDER_THICKNESS;
}

/*------------------------------------------------------------------------------
| ICanvas::borderMargin                                                        |
|                                                                              |
| Returns the width and height of the empty space that the canvas should use   |
| between its border and child windows.                                        |
------------------------------------------------------------------------------*/
ISize ICanvas::borderMargin ( ) const
{
  #define BORDER_MARGIN  3
  return ISize( BORDER_MARGIN );
}

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| _pfnwpICCanProc                                                              |
|                                                                              |
| Common window procedure used in ICLUI Private Canvas Control.                |
|                                                                              |
| Router and default event processing for Canvas events.                       |
| Notes: Canvas event processing includes:                                     |
|    -- WM_COMMAND, WM_SYSCOMMAND                                              |
|       (propagates these messages to the owner window, unless doing so would  |
|       cause an infinite loop between the frame and client area canvas)       |
|    -- WM_MOUSEMOVE (IC_PM)                                                   |
|       (generates a WM_CONTROLPOINTER message and sets the mouse pointer      |
|       appropriately)                                                         |
|    -- WM_CONTROLPOINTER / WM_SETCURSOR                                       |
|       (propagates this message to the owner window)                          |
|    -- WM_BUTTON1DOWN, WM_BUTTON1DBLCLK, WM_BUTTON2DOWN                       |
|       (don't pass event to owner window if canvas is a notebook page window) |
|    -- WM_QUERYDLGCODE                                                        |
|       (helps define the events handled by this window procedure)             |
|    -- WM_PAINT                                                               |
|       (paints the background and optional border and border text of the      |
|       canvas; also with IC_UM_CANVAS_PAINT, calls the layout routine to      |
|       provide specialized sizing and positioning of child controls)          |
|    -- IC_UM_CANVAS_PAINT                                                     |
|       (with WM_PAINT, calls the layout routine to provide specialized sizing |
|       and positioning of child controls)                                     |
|    -- WM_SHOW / WM_SHOWWINDOW                                                |
|       (with IC_UM_CANVAS_PAINT, calls the layout routine to provide          |
|       specialized sizing and positioning of child controls)                  |
|    -- WM_ERASEBKGND (IC_WIN)                                                 |
|       (defers background painting to WM_PAINT code)                          |
|    -- WM_SETREDRAW (IC_WIN)                                                  |
|       ()                                                                     |
|    -- WM_IC_UM_REQUEST_CANVASFONT (IC_WIN)                                   |
|       (supports the font function)                                           |
|    -- WM_CALCVALIDRECTS (IC_PM)                                              |
|       (identifies bit blting information on window size changes)             |
------------------------------------------------------------------------------*/
#ifdef IC_WIN
void* __stdcall _pfnwpICCanProc(
                               void* hwnd,
#endif
#ifdef IC_PM
void* _System _pfnwpICCanProc(
                               unsigned long hwnd,
#endif
                               unsigned long ulMsg,
                               void* mp1, void* mp2 )
{
   IMODTRACE_ALL( "_pfnwpICCanProc" );
   IEvent event( IWindowHandle( hwnd ),
                 ulMsg,
                 IEventParameter1( mp1 ),
                 IEventParameter2( mp2 ) );

   IWindow* pwin = 0;

   ITRACE_ALL(IString("WinP HWND=") + IString((unsigned long)hwnd).d2x()+
              IString(", Msg=") + IString(ulMsg).d2x()+
              IString(", MP1=") + IString((unsigned long)mp1).d2x()+
              IString(", MP2=") + IString((unsigned long)mp2).d2x());

   pwin = IWindow::windowWithHandle(hwnd, false);
   if ( pwin )
   {
      event.setDispatchingWindow( pwin );
   }
   else
   {
#ifdef IC_WIN
      return (void*)(IDEFWINDOWPROC( hwnd, (unsigned)ulMsg, (unsigned)mp1, (long)mp2 ));
#else
      return IDEFWINDOWPROC( hwnd, ulMsg, mp1, mp2 );
#endif
   }

   /*-----------------------------------------------------------------*/
   /* Process ICanvas messages as they were previously processed by   */
   /* the default handler for PM and Windows.  This WinProc replaces  */
   /* the former default canvas handler for PM and Windows only,      */
   /* ICanvasHandler::dispatchHandlerEvent.  As a result only the     */
   /* Motif dispatchHandlerEvent exists in the current ICanvasHandler */
   /* class.                                                          */
   /*                                                                 */
   /* This WinProc will get called for each unprocessed message after */
   /* all handlers in the handler list have had an opportunity to     */
   /* process it.                                                     */
   /*-----------------------------------------------------------------*/

   bool bProcessed = false;

   switch ( event.eventId() )
   {
      case WM_COMMAND:
      case WM_SYSCOMMAND:
      {                        // These aren't normally sent to owner.
#ifdef IC_WIN
         // If the message is a control notification message, then do
         // not send it to the owner.  Routing of such messages (known as
         // WM_CONTROL messages in PM) is handled by _pfnwpICWinProc in IWindow.
         // Control notification messages are identified by a non-0
         // value in parameter 2.
         if ((event.eventId() == WM_COMMAND) && (event.parameter2() != 0))
             break;
#endif
         /**********************************************************/
         /* Must ensure that WM_COMMAND and WM_SYSCOMMAND are not  */
         /* sent to the owner window if this canvas is the client  */
         /* area, in the cases where doing so would cause an       */
         /* endless loop between the frame and canvas handlers.    */
         /* The underlying problem is that the frame handler sends */
         /* WM_COMMAND and WM_SYSCOMMAND/SC_CLOSE events, and PM's */
         /* default frame procedure sends WM_COMMAND messages back */
         /* to the client area (FID_CLIENT), which the canvas      */
         /* would then send back to the owner frame, ad infinitum. */
         /* To remedy this problem, the canvas handler sets a bit  */
         /* reserved by the class library in mp2 of the event      */
         /* before sending it on to the owner frame, but doesn't   */
         /* send on such events if the bit is already set (the bit */
         /* is used to signal whether the event has already been   */
         /* routed to the frame, and doesn't need to be sent again */
         /* since it would just come back from the frame).         */
         /**********************************************************/
         ICanvas* pcvCanvas = (ICanvas*)pwin;
         const IWindow* pwndOwner = pcvCanvas->owner();

         if ( pwndOwner )
         {          // Have an owner window to route the event to.

            ICommandEvent cmdevt( event );

            bool bClient = ( pcvCanvas->id() == FID_CLIENT  &&
                               pwndOwner->isFrameWindow() );

            if (!bClient  ||  cmdevt.isFromFrame() == false)
            {       // Canvas is not client or event never sent to frame.
               IWindowHandle whOwner = pwndOwner->handle();

               if (whOwner != IWindow::desktopWindow()->handle()  &&
                   whOwner != IWindow::objectWindow()->handle())
               {                          // Valid owner window.
#ifndef IC_WIN
                  unsigned long ul2 = (unsigned long)event.parameter2();

                  if (bClient)
                  {                       // Canvas is client window.
                     ul2 |= 0x8000;       // Set reserved bit in source value.
                  }
                  event.setResult(pwndOwner->handle().
                                    sendEvent(event.eventId(),
                                              event.parameter1(),
                                              IEventParameter2(ul2)));
                             // Forward command event to owner window.
#endif
#ifdef IC_WIN
                  unsigned long ul1 = (unsigned long)event.parameter1();

                  if ( bClient  &&
                       ( event.eventId() == WM_COMMAND  ||
                         event.parameter1().number1() != SC_KEYMENU ) )
                  {                       // Canvas is client window.
                     // (Don't modify wparam for a WM_SYSCOMMAND/SC_KEYMENU
                     // message, since this causes menu bar mnemonics to not
                     // work from a canvas.)
                     ul1 |= 0x80000000ul;   // Set reserved bit in source value.
                  }
                  event.setResult(pwndOwner->handle().
                                     sendEvent(event.eventId(),
                                               IEventParameter1(ul1),
                                               event.parameter2() ));
                             // Forward command event to owner window.
#endif
                  bProcessed = true;
               }
            } /* endif */
         } /* endif */
         break;
      } /* endcase */

#ifdef IC_PM
      case WM_MOUSEMOVE:
      {
         /**********************************************************/
         /* Generate the WM_CONTROLPOINTER message to allow the    */
         /* application to control the cursor used for the mouse   */
         /* pointer, and set the mouse pointer as appropriate      */
         /* (OS/2's WinDefWindowProc does not do this by default). */
         /**********************************************************/
         ICanvas
          *canvas = (ICanvas*)pwin;
         // Typically send WM_CONTROLPOINTER to the owner window.
         const IWindow
          *ptrChangeWindow = canvas->owner();
         if ( ! ptrChangeWindow )
         {  // No owner window or it isn't wrappered.
             ptrChangeWindow = canvas;
         }

         ISystemPointerHandle
           defaultCanvasPtr( ISystemPointerHandle::arrow );
         IEventResult
           result( ptrChangeWindow
                    ->sendEvent( WM_CONTROLPOINTER,
                                 IEventParameter1( canvas->id() ),
                                 IEventParameter2( defaultCanvasPtr ) ) );

         ISETPOINTER( IPointerHandle( result ) );
         event.setResult( true );
         bProcessed = true;
         break;
      } /* endcase */
#endif // IC_PM

      case WM_CONTROLPOINTER:
      {                          // This isn't normally sent to owner.
         /**********************************************************/
         /* Pass the WM_CONTROLPOINTER to the owner.  This message */
         /* will make its way to the frame handler (if another     */
         /* handler does not process the message) where the mouse  */
         /* pointer can be set for a frame window and all of its   */
         /* child windows.  Even if the canvas is the client, the  */
         /* control pointer message will not be rerouted by the    */
         /* frame back to the client (the default PM behavior)     */
         /* since the frame handler will always return true for    */
         /* this message.                                          */
         /**********************************************************/
         ICanvas* pcvCanvas = (ICanvas*)pwin;
         const IWindow* pwndOwner = pcvCanvas->owner();

         if ( pwndOwner != 0 )
         {
            IWindowHandle whOwner = pwndOwner->handle();
            if ( whOwner != IWindow::desktopWindow()->handle()  &&
                 whOwner != IWindow::objectWindow()->handle() )
            {      // Valid owner window.
               event.setResult( whOwner.sendEvent( event.eventId(),
                                                   event.parameter1(),
                                                   event.parameter2() ) );
               bProcessed = true;
            }
         }

         break;
      } /* endcase */

      case WM_BUTTON1DOWN:
      case WM_BUTTON1DBLCLK:
      case WM_BUTTON2DOWN:
      {                                // Notebook page window case?
         /**********************************************************/
         /* These mouse events are processed to provide special    */
         /* processing in the case where the canvas is a notebook  */
         /* page.  Without this, clicking on an entry field child  */
         /* window of a canvas, or in white space of the canvas    */
         /* itself will result in the notebook eventually getting  */
         /* the button event and changing the input focus back to  */
         /* itself.                                                */
         /**********************************************************/
         HWND parent = IPARENTOF( event.handle() );
         HWND grandParent = IPARENTOF( parent );
                // (Can't use parent(), since notebook's intermediate
                // window isn't known to the class library.)
         if (grandParent)
         {                             // Notebook was orig parent?
            IWindowClassName className( grandParent );

            if (className == WC_NOTEBOOK)
            {                          // Parent was notebook.
               IWindowHandle canvasWindow = event.handle();
               IWindowHandle focusWindow =
                  IQUERYFOCUS(IWindow::desktopWindow()->handle());
               IEventResult mr =
                  focusWindow.sendEvent(WM_QUERYFOCUSCHAIN,
                                        IEventParameter1(QFC_PARTOFCHAIN),
                                        IEventParameter2(canvasWindow));
               if (mr.asUnsignedLong() == false)
               {                       // Focus not on canvas child
                  event.window()->setFocus();
               }
               bProcessed = true;      // Don't let notebook get event
                                       //   since it will only steal
                                       //   away the input focus.
            }                          // End parent was notebook.
         }                             // End know who grand parent is.
         break;
      } /* endcase */

      case WM_QUERYDLGCODE:
      {
#ifdef IC_WIN
         // we want keys
         event.setResult( DLGC_WANTALLKEYS );
#else
         event.setResult( 0 );
#endif
         bProcessed = true;
         break;
      } /* endcase */


      case WM_PAINT:
      {
         IMODTRACE_DEVELOP("_pfnwpICCanProc(WM_PAINT)");
         ICanvas* pcvCanvas = (ICanvas*)pwin;
#ifdef IC_WIN
         pcvCanvas->setLayoutDistorted(0, IWindow::fontPropogated);
#endif

         if (pcvCanvas->isLayoutDistorted(IWindow::layoutChanged)  &&
             pcvCanvas->isVisible())
         {              // Use layout to show child changes on screen.
            ITRACE_DEVELOP("Send IC_UM_CANVAS_PAINT message");
            pcvCanvas->sendEvent(IC_UM_CANVAS_PAINT, 0, 0);
         }                             // Pass processing on to PM.

         // Now paint the canvas.
         bProcessed = pcvCanvas->paintWindow( event );
         break;
      } /* endcase */

      case IC_UM_CANVAS_PAINT:
      {
         ICanvas* pcvCanvas = (ICanvas*)pwin;
         pcvCanvas->setLayoutDistorted(0, IWindow::fontPropogated);
         if (pcvCanvas->isLayoutDistorted(IWindow::layoutChanged))
         {
            pcvCanvas->layout();
         }
         bProcessed = true;
         break;
      } /* endcase */

#ifdef IC_PM
      case WM_SHOW:
#endif
#ifdef IC_WIN
      case WM_SHOWWINDOW:
#endif
      {
         ICanvas* pcvCanvas = (ICanvas*)pwin;
         if ( ( event.parameter1().number1() ) && (pcvCanvas) )
         {                              // Canvas is being shown.
            pcvCanvas->fCanvasData->fShowWindow = true;

            if (pcvCanvas->isLayoutDistorted(IWindow::layoutChanged) )
            {                   // Window being shown and need new layout.
               IMODTRACE_DEVELOP("ICanvasHandler::dispatchHandlerEvent(WM_SHOW)");
               pcvCanvas->sendEvent(IC_UM_CANVAS_PAINT, 0, 0);
            }                        // Pass processing on to PM.
         }
         break;
      } /* endcase */

#ifdef IC_WIN
      case WM_ERASEBKGND:
      {
         event.setResult( 1 );
         bProcessed = true;  // Handle background painting in WM_PAINT.
         break;
      }

      case WM_SETREDRAW:
      {
         ITRACE_DEVELOP(IString("ICanvasHandler WM_SETREDRAW flag=") +
                      IString(event.parameter1().asUnsignedLong())  );
         if ( event.parameter1().asUnsignedLong() )
         {
            bool wasVisible = pwin->isVisible();
            if (!wasVisible)
            {
               ITRACE_DEVELOP("was not visible");
               IDEFWINDOWPROC(event.handle(),
                              (UINT)event.eventId(),
                              event.parameter1(),
                              event.parameter2() );

#ifdef IC_WU
// IC_WU_NOTYET - Some of the RedrawWindow( ) options are not supported by
//    Wind/U until Wind/U 3.2.  Use InvalidateRect() for now.
               IINVALIDATERECT(
                        event.handle(),         // this window
                        0,                      // entire window (rect)
                        true );
#else
               // RDW_ALLCHILDREN may be too much, but seem to need
               // it to get everything to paint.
               unsigned int rdFlags = RDW_ERASE | RDW_FRAME | RDW_ALLCHILDREN |
                                      RDW_INVALIDATE;
               RedrawWindow( event.handle(),         // this window
                             0,                      // entire window (rect)
                             0,                      // entire window (region)
                             rdFlags );              // flags
#endif // ! IC_WU
               bProcessed = true;
            }
            else
            {
               ITRACE_DEVELOP("was visible");
            }
         }
         break;
      }

      case IC_UM_REQUEST_CANVASFONT:
      {
         ICanvas* pcvCanvas = (ICanvas*)pwin;
         if (pcvCanvas->fFont)
           event.setResult( pcvCanvas->fFont );
         else
           event.setResult( 0 );
         bProcessed = true;
         break;
      }

      case WM_SIZE:
      {
         ICanvas* canvas = (ICanvas*)pwin;
         unsigned long
           prevWidth = canvas->fCanvasData->fSize.width(),
           prevHeight = canvas->fCanvasData->fSize.height(),
           newWidth = event.parameter2().lowNumber(),
           newHeight = event.parameter2().highNumber();

         if ( !canvas->isClassStyleAllPaints() )
         {
           if ( canvas->hasBorder() )
           {  // Need to ensure that the right and bottom edges of the
              // previously-drawn border are removed if the canvas is
              // being grown.  Also that the right and bottom edges of
              // the border are drawn if the canvas is shrunk.
              ISize
                botRightOffset = canvas->bottomRightLayoutOffset();
              unsigned long
                botOffset = botRightOffset.height(),
                rightOffset = botRightOffset.width();
              if ( prevWidth > newWidth )
              {  // Canvas shrunk in width.
                 // Need to paint the new right edge of the border.
                 canvas->refresh( IRectangle( IPoint( newWidth - rightOffset, 0 ),
                                              ISize( rightOffset, newHeight ) ) );
              }
              else if ( prevWidth < newWidth )
              {  // Canvas grown in width.
                 // Need to remove the previous right edge of the border.
                 canvas->refresh( IRectangle( IPoint( prevWidth - rightOffset, 0 ),
                                              ISize( rightOffset, prevHeight ) ) );
              }
              if ( prevHeight > newHeight )
              {  // Canvas shrunk in height.
                 // Need to paint the new bottom edge of the border.
                 canvas->refresh( IRectangle( IPoint( 0, newHeight - botOffset ),
                                              ISize( newWidth, botOffset ) ) );
              }
              else if ( prevHeight < newHeight )
              {  // Canvas grown in height.
                 // Need to remove the previous bottom edge of the border.
                 canvas->refresh( IRectangle( IPoint( 0, prevHeight - botOffset ),
                                              ISize( prevWidth, botOffset ) ) );
              }
           }
         }

         // Update the current height now.
         canvas->fCanvasData->fSize.setWidth( newWidth );
         canvas->fCanvasData->fSize.setHeight( newHeight );

         bProcessed = false;
         break;
      }
#endif // IC_WIN

#ifdef IC_PM
      case WM_CALCVALIDRECTS:
      {
         ICanvas* canvas = (ICanvas*)pwin;
         if ( !canvas->isClassStyleAllPaints() )
         {
           _RECTL *oldRect =
               (_RECTL *)(event.parameter1().asUnsignedLong());
           _RECTL *newRect =
               (_RECTL *)((event.parameter1().asUnsignedLong())+
                           sizeof(_RECTL));
#ifdef IC_DEVELOP
           SWP *newPt = (SWP *)(event.parameter2().asUnsignedLong());
           IRectangle rect1 (oldRect->xLeft, oldRect->yBottom,
                             oldRect->xRight, oldRect->yTop);
           IRectangle rect2 (newRect->xLeft, newRect->yBottom,
                             newRect->xRight, newRect->yTop);
           IPair::Coord newX = (long) newPt->x;
           IPair::Coord newY = (long) newPt->y;
           IPoint locPt (newX, newY);
           ITRACE_DEVELOP(IString("Old rect = ") + rect1.asDebugInfo());
           ITRACE_DEVELOP(IString("New rect = ") + rect2.asDebugInfo());
           ITRACE_DEVELOP(IString("New point = ") + locPt.asDebugInfo());
#endif // IC_DEVELOP
           if ( newRect->yTop != oldRect->yTop  ||
                newRect->yBottom != oldRect->yBottom )
           {  // Height of the canvas has changed.
              // Repaint the entire canvas to force top alignment.
              event.setResult( CVR_REDRAW );
           }
           else
           {  // The width has changed.
              if ( canvas->hasBorder() )
              {  // Force painting of the new right edge of the canvas
                 // border for the case where the width of the window is
                 // made smaller.  Also force a paint to remove the prior
                 // right edge of the border for the case where the width
                 // of the window is made bigger.
                 unsigned long
                   rightOffset = canvas->bottomRightLayoutOffset().x();
                 oldRect->xRight = oldRect->xRight - rightOffset;
                 newRect->xRight = newRect->xRight - rightOffset;
                 event.setResult( 0 );
              }
              else
              {
                 event.setResult( CVR_ALIGNBOTTOM | CVR_ALIGNLEFT );
              }
           }
         }
         bProcessed = true;
         break;
      }
#endif // IC_PM

      default:
         break;
   }  // end switch

   // Route unprocessed messages to the default window procedure.
   if ( bProcessed == false )
   {
#ifdef IC_WIN
      return (void*)(IDEFWINDOWPROC(hwnd, (unsigned)ulMsg, (unsigned)mp1, (long)mp2));
#else
      return IDEFWINDOWPROC(hwnd, ulMsg, mp1, mp2);
#endif
   }
   else
   {
      return event.result();
   }
} /* end win proc */
#endif // IC_PMWIN

/*------------------------------------------------------------------------------
| ICanvas::stylizedBorderText                                                  |
|                                                                              |
| Return the border text, stylized with the font of the canvas if the text has |
| no styles.                                                                   |
------------------------------------------------------------------------------*/
IText ICanvas::stylizedBorderText ( ) const
{
  IText
    stylizedText( this->borderText() );
  if ( ! stylizedText.isStyled() )
  {  // No font information, so give the text the canvas's font.
     IFont
       canvasFont = this->font();
     ITextStyleSet
       textStyles = canvasFont.styleSet();

     // No color information, so use the foreground color for the text.
     IColor
       fgColor = this->foregroundColor();
     textStyles
      .add( ITextColorStyle( fgColor.redMix(),
                             fgColor.greenMix(),
                             fgColor.blueMix() ) );

     stylizedText
      .addStyles( textStyles );
  }
  return stylizedText;
}

/*------------------------------------------------------------------------------
| ICanvas::paintWindow                                                         |
|                                                                              |
| Paint the background and optional border and border text of the canvas.      |
------------------------------------------------------------------------------*/
bool ICanvas::paintWindow ( const IEvent& event )
{
#ifdef IC_PM
  // Use WinBeginPaint because it sets the clip region of the hps
  // to be the invalid/update region of the window.  Otherwise the
  // canvas exhibits painting problems in the composition editor of
  // the Visual Builder (when the Builder removes the grab handles of
  // a child window, the canvas would overpaint the child window).
  // Although IPaintEvent appears to set the clip region similar to
  // WinBeginPaint, the call to GpiSetClipRegion in
  // IPaintEvent::initializeGC seems to result in a clip rectangle
  // set instead.
  _RECTL
    paintStruct;
  IPresSpaceHandle
    hps = IBEGINPAINT( this->handle(), 0, &paintStruct );
  IRectangle
    rectPaint( paintStruct );
#endif // IC_PM
#ifdef IC_MOTIFWIN
  IPaintEvent
    paintEvent( event );
  IRectangle
    rectPaint( paintEvent.rect() );
  IPresSpaceHandle
    hps = paintEvent.presSpaceHandle();
#endif // IC_MOTIFWIN

#ifdef IC_MOTIF
  // this is a temporary workaround to some painting problems we are having.
  IXDC *fXDC = (IXDC *)hps.asUnsigned();
  XSetSubwindowMode(XtDisplay((Widget)this->handle()),
                    fXDC->gc(),
                    ClipByChildren);
#endif

#ifdef IC_PM
  if ( ( rectPaint.top() == rectPaint.bottom() )  ||
       ( rectPaint.left() == rectPaint.right() ) )
  {  // Let the default window proc paint in this case to avoid infinite loops.
     return false;
  }
#endif

  // First paint the background of the canvas.
#ifdef IC_DEVELOP
  ITRACE_DEVELOP( "  @@ Canvas ID=" + IString( this->id() ) +
                  ", HWND=" + this->handle().asString() +
                  ", invalid rect=" + rectPaint.asString() );
#endif

  { // Brackets to scope IGrafPort's before IENDPAINT.
    // rootPort is always in native coordinate system for the OS.
#ifdef IC_ORIGINUPPERLEFT
    // Coordinate system with an upper-left origin (e.g. Windows, Motif).
    IExtendedRootGrafPort
      rootPort( hps, ICoordinateSystem::kOriginUpperLeft );
#else
    // Coordinate system with a lower-left origin (e.g. OS/2).
    IExtendedRootGrafPort
      rootPort( hps, ICoordinateSystem::kOriginLowerLeft );
#endif

    // The rectPaint rectangle is in the native coordinate system.
    // Use exclusive fill painting because this is a window rectangle,
    // rather than a true graphics rectangle.
    IGRect2D box( rectPaint );
    IBaseColor bgndColor( this->backgroundColor() );
    IFillBundle bgndFill( bgndColor );
    ILinkedGrafPort bgndPort( &rootPort, &bgndFill );

    bool clipChildren = true;

#ifdef IC_MOTIFWIN
    if ( this->style() & WS_CLIPCHILDREN )
    {
       // Don't clip out child windows if the canvas is using the child
       // window clipping support provided by the operating system.
       clipChildren = false;
       this->fCanvasData->fShowWindow = false;

       // Note: Do not attempt to optimize painting performance in the
       // case of many child windows by not clipping the child windows
       // from the region to be painted.  This can cause the canvas to
       // overpaint parts of child windows drawn in response to WM_NCPAINT
       // and WM_ERASEBKGND messages.  This overpainting only occurs when
       // the canvas receives a WM_PAINT after its child windows have
       // received these other paint-related messages--such as after
       // cancelling from the Windows NT Security dialog (invoked via
       // Ctrl+Alt+Del) when the canvas has a static text, list box, entry
       // field, or MLE with 3D borders.  The alternative of invoking the
       // canvas paint code from WM_ERASEBKGND is not viable because it
       // would not allow paint handlers (keyed off WM_PAINT) To work
       // effectively when attached to a canvas.
    }
    else if ( this->fCanvasData->fShowWindow )
    {
       this->fCanvasData->fShowWindow = false;
    }
#endif // IC_MOTIFWIN

#ifdef IC_PM
    #define MAX_CHILDREN_FOR_CLIPPING   6

    if ( this->fCanvasData->fShowWindow  ||
         this->fCanvasData->fChildCount > MAX_CHILDREN_FOR_CLIPPING  ||
         this->style() & WS_CLIPCHILDREN )
    {
       // Don't clip on the initial display, or if building the clip
       // region looks too expensive because of the number of child
       // windows, or if the canvas is using child clipping support
       // provided by the presentation system.
       clipChildren = false;
       this->fCanvasData->fShowWindow = false;
    }
#endif // IC_PM

    IGArea clipRegion( box );

    if ( clipChildren )
    {
       // Set up clip regions so that child windows are not overpainted.
       bool
         convertCoords = ICoordinateSystem::isConversionNeeded();
       IWindow::ChildCursor cursor( *this );
       ISize cvSize( this->size() );
       for ( cursor.setToFirst(); cursor.isValid(); cursor.setToNext() )
       {
         IWindowHandle hwndChild = this->childAt( cursor );
         if ( hwndChild )
         {
           IWindow* pChild = IWindow::windowWithHandle( hwndChild );
           if ( pChild  &&  pChild->isVisible() )
           {
             // The rectangle of the child window is returned in
             // application coordinates, but must be in top-left
             // coordinates.
             IRectangle vRect( pChild->visibleRectangle() );
             if ( convertCoords )
             {
                vRect =
                   ICoordinateSystem::convertToNative( vRect, cvSize );
             }

             IGRect2D visibleRect( vRect );
             visibleRect.orderPoints();
             IGArea childRegion( visibleRect );
             clipRegion -= childRegion;
           }
         }
       }
    }

    // Paint the background of the canvas.
    ILinkedGrafPort thePort( &bgndPort, &clipRegion );
    thePort.draw( box );

    ITRACE_DEVELOP( IString( "Paint rectangle = " ) +
                    rectPaint.asDebugInfo() );
    ITRACE_DEVELOP( IString( "Window rectangle = " ) +
                    this->nativeRect().asDebugInfo() );

    // Paint the border and border text if applicable.
    if ( this->hasBorder() )
    {
       // draw the outline box
#ifdef IC_ORIGINLOWERLEFT
       // All drawing of the border and border text will be done
       // using one set of code that assumes an upper-left
       // coordinate system.  Therefore get a grafport with this
       // orientation.
       IExtendedRootGrafPort
         nativePort( hps, ICoordinateSystem::kOriginUpperLeft );
       ILinkedGrafPort
         borderPort( &nativePort, &bgndFill );
#else // ! IC_ORIGINLOWERLEFT
#ifdef IC_MOTIF
       #define borderPort bgndPort
#else
       #define borderPort thePort
#endif // ! IC_MOTIF
#endif // ! IC_ORIGINLOWERLEFT

       IGRect2D rc( IRectangle( this->size() ) );

       IGraphicText
         theLabel( this->stylizedBorderText(), IGraphicText::kSingleLine );
       IGRect2D
         textRect( theLabel.looseFitBounds( &borderPort ) );
       unsigned long
         charHeight = (unsigned long) textRect.height();
       if ( charHeight < this->borderThickness() )
       {
          charHeight = this->borderThickness();
       }

       GCoordinate
         th = this->borderThickness(),
         boxtop = (GCoordinate)( charHeight - th ) * 3 / 4,
         borderLeft = rc.fLeft,
         borderRight = rc.fRight,
         borderBottom = rc.fBottom,
         textTop = rc.fTop,
         textBottom = textTop + charHeight;

       // Adjust the coordinates for the border rectangle, so that the
       // inclusive fill painting done with a frame bundle stays within
       // the window rectangle.
       borderBottom--;
       borderRight--;

       if ( th )
       {
          if ( th > 1 )
          {  // Take off one pel for hilighting/shadowing.
             th--;
          }
          GCoordinate
            th0 = th - 1;

          // Draw a 3D border.  First draw a box using the border color.
          // Add a highlight line inside the upper and left edges of the
          // box.  Add a shadow line inside the lower and right edges.

          // Draw the main part of the border.
          IGrafBundle
            borderBundle( IBaseColor( this->borderColor() ) );
          IPen
            borderPen( IPen::kHairline, 1.0 );
          borderBundle.setFramePen( borderPen );
          borderPort
           .draw( IGRect2D( borderLeft + th0, boxtop + th0,
                            borderRight - th0, borderBottom - th0 ),
                  borderBundle );

          if ( this->borderThickness() > 1 )
          {  // Add hilighting/shadowing.
             // Hilight inside the top and left edges.

             IGPolyline2D line( 3 );
             line.setPoint( 0, IGPoint2D( borderLeft + th,
                                          borderBottom - th ) ); // Lower left
             line.setPoint( 1, IGPoint2D( borderLeft + th,
                                          boxtop + th ) );       // Upper left
             line.setPoint( 2, IGPoint2D( borderRight - th + 1,
                                          boxtop + th ) );       // Upper right

             IColor lightButtonColor( IColor::kButtonLight );
             IGrafBundle buttonBundle( lightButtonColor );
             IPen
               buttonPen( IPen::kHairline, 1.0 );
             buttonBundle.setFramePen( buttonPen );
             borderPort.draw( line, buttonBundle );

             // Draw a shadow inside the bottom and right edges.
             line.setPoint( 0, IGPoint2D( borderLeft + th + 1,
                                          borderBottom - th ) ); // Lower left
             line.setPoint( 1, IGPoint2D( borderRight - th,
                                          borderBottom - th ) ); // Lower right
             line.setPoint( 2, IGPoint2D( borderRight - th,
                                          boxtop + th ) );       // Upper right
             IColor shadowColor( IColor::kButtonDark );
             IGrafBundle shadowBundle( shadowColor );
             IPen
               shadowPen( IPen::kHairline, 1.0 );
             shadowBundle.setFramePen( shadowPen );
             borderPort.draw( line, shadowBundle );
          }
       } // End th != 0

       // Draw the text if any.
       if ( this->borderText().length() )
       {
          // Draw the background for the text.  If the canvas is
          // right-to-left, start 9 pels in from the left edge
          // and go right for the length of the text.  Otherwise,
          // start 9 pels in from the right edge and go left for
          // the length of the text.  In both cases, include two
          // blank pels on each side of the text.
          rc.fBottom = textBottom;
          bool
            leftToRight = true;
          if ( IBidiSettings::isBidiSupported() )
          {
             IBidiSettings
                bidiSettings( *this );
             if ( bidiSettings.windowLayout() ==
                      IBidiSettings::layoutRightToLeft )
             {  // Right justify the text, rather than left justify.
                leftToRight = false;
             }
          }
          if ( leftToRight )
          {  // Left justify the text.
             rc.fLeft += 9;
             rc.fRight = rc.fLeft + textRect.width();
             rc.fRight += 4;
          }
          else
          {  // Right justify the text.
             rc.fRight -= 9;
             rc.fLeft = rc.fRight - textRect.width();
             rc.fLeft -= 4;
          }
          IGRect2D fillRect( rc );
          borderPort.draw( fillRect );

          // Now draw the text itself.
          theLabel.setLocation( IGPoint2D( rc.fLeft + 2, rc.fBottom ) );
                // (Bottom is defined as minY.)
          IFillBundle
            textBundle( IBaseColor( this->defaultBorderTextColor() ) );
          ILinkedGrafPort
            textPort( &borderPort, &textBundle );
          theLabel.draw( textPort );
       }

    }  // if group box
  }  // Brackets to scope IGrafPort's before IENDPAINT.

#ifdef IC_PM
  IENDPAINT( this->handle(), hps, &paintStruct );
#endif
#ifdef IC_WIN
  PAINTSTRUCT
    paintStruct;
  IBEGINPAINT( this->handle(), 0, &paintStruct );
  IENDPAINT( this->handle(), 0, &paintStruct );
#endif
#ifdef IC_MOTIF
  this->releasePresSpace( hps );
#endif

  return true;
}

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICanvas::registerCallbacks                                                   |
|                                                                              |
| Add callbacks and X event handlers for events which simple canvases can      |
| experience.                                                                  |
------------------------------------------------------------------------------*/
void ICanvas::registerCallbacks ( )
{
//  IWindowHandle widget = handle();   // get validated widget handle
//  if ( fCanvasData->IAmAnICanvas )
//  {
//    if ( !XtIsSubclass(widget, xmBulletinBoardWidgetClass) )
//      ITHROWGUIERROR("CantRegisterNonBulletinBoardWidget");
    // Add the callback to handle gaining focus
//    XtAddCallback(
//       (Widget)widget,           // widget
//       XmNfocusCallback,         // callback name
//       iwindowMotifCallback,     // callback routine
//       (XtPointer) this);                    // client data
//  }
}
#endif

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICanvas::unregisterCallbacks                                                 |
|                                                                              |
| Remove callbacks and X event handlers added in registerCallbacks().          |
------------------------------------------------------------------------------*/
void ICanvas::unregisterCallbacks ( )
{
//  if ( this->isValid() )
//  {
//     IWindowHandle widget = handle();   // get validated widget handle
//     if ( fCanvasData->IAmAnICanvas )
//     {
//        if ( !XtIsSubclass(widget, xmBulletinBoardWidgetClass) )
//           ITHROWGUIERROR("CantUnregisterNonBulletinBoardWidget");
//
//        XtRemoveCallback(
//           (Widget)widget,           // widget
//           XmNfocusCallback,         // callback name
//           iwindowMotifCallback,     // callback routine
//           (XtPointer) this );       // client data
//     }
//  }
}
#endif

#ifdef IC_MOTIF
/*------------------------------------------------------------------------------
| ICanvas::passEventToOwner                                                    |
|                                                                              |
| Returns whether or not the event can be passed up to the owner of this       |
| control.                                                                     |
------------------------------------------------------------------------------*/
bool ICanvas::passEventToOwner ( IEvent& event )
{
  return Inherited::passEventToOwner( event );
}
#endif

#ifdef IC_PMWIN
/*------------------------------------------------------------------------------
| ICanvas::setBidiSettings                                                     |
|                                                                              |
| Provide processing specific to a canvas for a change to its bidirectional    |
| attributes.                                                                  |
------------------------------------------------------------------------------*/
ICanvas& ICanvas::setBidiSettings ( const IBidiSettings& bidiSettings,
                                    bool childInherit,
                                    bool refresh )
{
  // Most canvases manage the positions of their child windows.
  // Check if the canvas needs to consider reversing their positions,
  // based on whether the canvas has changed from left-to-right to
  // right-to-left, or vice-versa.                                                               |
  IBidiSettings
    prevBidiSettings( *this );
  if ( prevBidiSettings.windowLayout() != bidiSettings.windowLayout() )
  {
     this->setLayoutDistorted( IWindow::layoutChanged, 0 );
  }

  this->ICanvas::Inherited::setBidiSettings( bidiSettings,
                                             childInherit,
                                             refresh );
  return *this;
}
#endif // IC_PMWIN

#if (IC_OBSOLETE <= IC_OBSOLETE_3)
/*------------------------------------------------------------------------------
| ICanvas::origDefaultButtonHandle                                             |
|                                                                              |
| Return the original default push button for the frame window.                |
------------------------------------------------------------------------------*/
IWindowHandle ICanvas::origDefaultButtonHandle ( ) const
{
  return Inherited::defaultPushButton();
}

/*------------------------------------------------------------------------------
| ICanvas::defaultPushButton                                                   |
|                                                                              |
| Return the first child window that is a default push button, if one exists.  |
------------------------------------------------------------------------------*/
IWindowHandle ICanvas::defaultPushButton ( ) const
{
  return Inherited::defaultPushButton();
}
#endif // IC_OBSOLETE
