/*
    COPYRIGHT:
    ----------
    Copyright (C) International Business Machines Corp., 1998

    DISCLAIMER OF WARRANTIES:
    -------------------------
    The following [enclosed] code is sample code created by IBM
    Corporation. This sample code is not part of any standard IBM
    product and is provided to you solely for the purpose of assisting
    you in the development of your applications.  The code is provided
    "AS IS", without warranty of any kind.  IBM shall not be liable
    for any damages arising out of your use of the sample code, even
    if they have been advised of the possibility of such damages.
*/

//-----------------------------------------------------------------------------
// Class MandelbrotImage
// Purpose:
//      Represents an image of an area on the complex plane.  Pixels
//      are colored according to how "close" the points they represent
//      are to the Mandelbrot set.  Closeness is defined in terms of
//      how many iterations it took to determine the point was not in
//      the Mandelbrot set.
//-----------------------------------------------------------------------------

class MandelbrotImage
{
public:

    //-------------------------------------------------------------------------
    // Nested Class: Observer
    // Purpose:
    //      Objects of this class are notified as to the progress of the
    //      image being generated by a "MandelbrotImage" object.
    //-------------------------------------------------------------------------

    class Observer
    {
        friend class MandelbrotImage;
    protected:
        // PURPOSE: Called to give progress notification.  The
        // parameter "percent" specifies what percentage of the
        // overall image has been generated.
        virtual void imageProgress(unsigned percent) = 0;

        // PURPOSE: Called to discover if image generation should be
        // continued or canceled.
        virtual bool cancelImage() = 0;
    };

    // PURPOSE: Initialize the object. The area of the complex plane
    // belonging to the image is defined by the points "upperLeft" and
    // "lowerRight".  The size (in pixels) of the image to generate is
    // specified by "width" and "height".  The "limit" is the maximum
    // number of iteration to allow before deciding a point is within
    // the Mandelbrot set.
    MandelbrotImage(complex upperLeft, complex lowerRight,
                    unsigned long width, unsigned long height,
		    unsigned limit);

    // PURPOSE: Finalize the object.
    virtual ~MandelbrotImage();

    // PURPOSE: Generate the image.  The "observer" object will be
    // notified of the progress of the image generation.
    void generate(Observer & observer);

    // PURPOSE: Draw the specified portion of the complex plane onto
    // the "port" object.
    void draw(IPresSpaceHandle & presSpace) const;
private:
    complex             fTopLeft;
    complex             fBotRight;
    unsigned long       fWidth;
    unsigned long       fHeight;
    unsigned		fLimit;
    IGImage *           fGImage;
    IImage *  		fIImage;

    //-------------------------------------------------------------------------
    // Nested Struct: ComplexCache
    // Purpose:
    //      Represents a complex number.  It caches the squares of the
    //      real and imaginary components to allow for faster
    //      calculation.
    //-------------------------------------------------------------------------

    struct ComplexCache
    {
        double r, i;
        double rsq, isq;

        ComplexCache(double real, double imag) : r(real), i(imag)
        {
            rsq = r*r; isq = i*i;
        }
    };

};

//-----------------------------------------------------------------------------

MandelbrotImage::MandelbrotImage(complex upperLeft, complex lowerRight,
				 unsigned long width, unsigned long height,
				 unsigned limit) :
    fTopLeft(upperLeft),
    fBotRight(lowerRight),
    fWidth(width),
    fHeight(height),
    fLimit(limit),
    fGImage(0),
    fIImage(0)
{
    fGImage = new IGImage(width, height, IGImage::k256Color8Bit);
    fIImage = new IImage(*fGImage);
}

//-----------------------------------------------------------------------------

MandelbrotImage::~MandelbrotImage()
{
    delete fIImage;

    // Bug in IOC: IImage deletes the IGImage object passed to it, so
    // we can't delete it ourselves.
    // delete fGImage;
}

//-----------------------------------------------------------------------------

void
MandelbrotImage::generate(Observer & observer)
{
    // Create the image of the Mandelbrot set.  For each complex
    // number c within the defined range, iterate over the calculation
    // z = z*z + c, where z starts at complex(0.0, 0.0).  If the
    // magnitude of z ever exceeds 2.0, the point is not in the
    // Mandelbrot set.  If the iteration reaches the predefined limit,
    // decide that it probably is in the set.

    IGImagePixelAccessor pa(*fGImage);
    int percent = 0;    // Percent complete
    double re, im;
    int xi, yi;
    const double STEP_RE = (real(fBotRight) - real(fTopLeft)) / fWidth;
    const double STEP_IM = (imag(fBotRight) - imag(fTopLeft)) / fHeight;
    for (re = real(fTopLeft), xi = 0; xi < fWidth; ++xi, re += STEP_RE) {
        for (im = imag(fTopLeft), yi = 0; yi < fHeight; ++yi, im += STEP_IM) {
            // Every tenth point, check if the observer wants to
            // cancel image generation.
            if (yi % 10 == 0 && observer.cancelImage())
                return;

            // Determine if the current point, complex(re, im) is in
            // the mandelbrot set.
            ComplexCache z(0.0, 0.0);
	    int count;
            for (count = 0; count < fLimit && z.rsq + z.isq < 4.0; ++count) {
                // The folowing code calculates z = z * z + c, where c
                // is complex(re, im).
                double rtmp = z.rsq - z.isq + re;
                z.i = z.r * z.i * 2.0 + im;
                z.isq = z.i * z.i;
                z.r = rtmp;
                z.rsq = rtmp * rtmp;
	    }
            if (count == fLimit) {
                // Point is probably in the Mandelbrot set.
                pa.setPixel(IGPoint2D(xi, yi), IColor::kBlack);
            }
            else {
                // Point is not it the Mandelbrot set.  Set the color
                // of the pixel based on how many iterations it took
                // before deciding this.  The following algorithm
                // produces an interesting looking picture.
                const int MAX = 500;
                const int LIMIT = 64;
                const int MULT = 256 / LIMIT;
                int fac;
                for (fac = 0; fac < LIMIT; ++fac)
                    if (count >= MAX / (fac + 1))
                        break;
                if (fac < LIMIT) {
                    unsigned primary = 255 - fac * MULT;
                    pa.setPixel(IGPoint2D(xi, yi),
                                IBaseColor(primary, primary, primary));
                }
                else {
                    pa.setPixel(IGPoint2D(xi, yi), IColor::kBlack);
                }
            }
        }

        // If the percentage of columns completed has changed by at
        // least one percent, notify the observer.
        int npercent = xi * 100 / fWidth;
        if (npercent > percent) {
            percent = npercent;
            observer.imageProgress(percent);
        }
    }
}

//-----------------------------------------------------------------------------

void
MandelbrotImage::draw(IPresSpaceHandle & presSpace) const
{
    IExtendedRootGrafPort port(presSpace);
    fIImage->draw(port);
}
