/*******************************************************************************
* FILE NAME: IMidiDmp.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Class implementation of the class:                                         *
*     IMidiDumper- MIDI Datastream dumper                                      *
*******************************************************************************/
#ifndef _INOTIFEV_
#include <inotifev.hpp>
#endif

#ifndef _IOBSERVR_
#include <iobservr.hpp>
#endif

#ifndef _ISTDNTFY_
#include <istdntfy.hpp>
#endif

#ifndef _IMIDIDMP_
#include "IMidiDmp.hpp"
#endif

#ifndef _IMIDICLS_
#include "IMidiCls.hpp"
#endif

#ifndef _IVBDEFS_
#include <ivbdefs.h>
#endif

#ifndef _ITRACE_
#include <itrace.hpp>
#endif

#include <iostream.h>

const INotificationId IMidiDumper::readyId = "IMidiDumper::readyId";

//------------------------------------------------------------------------------
// IMidiDumper :: IMidiDumper
//------------------------------------------------------------------------------
IMidiDumper::IMidiDumper()
  : IMidiInstance( NULL ),
    iMLE( NULL )
{
   IFUNCTRACE_DEVELOP();
   // Create application instance
   MINSTANCE AppInstance = IRTMidi::theIRTMidi->addInstance(
                       this,
                       IRTMidi::theIRTMidi->applicationClass()->number(),
                       "MIDI Dumper" );
}


//------------------------------------------------------------------------------
// IMidiDumper :: updateAttributes
//------------------------------------------------------------------------------
void IMidiDumper::updateAttributes()
{
   IFUNCTRACE_DEVELOP();
   theInfo.ulAttributes = MIDI_INST_ATTR_CAN_RECV;
}


//------------------------------------------------------------------------------
// IMidiDumper :: ~IMidiDumper
//------------------------------------------------------------------------------
IMidiDumper::~IMidiDumper()
{
   IFUNCTRACE_DEVELOP();
   stopDumper();
   // If it didn't shut down, kill it
   if ( theThread.isStarted() )
   {
      IThread::current().sleep( 10 );
      theThread.stop();
   }
}

//------------------------------------------------------------------------------
// IMidiDumper :: initializePart
//------------------------------------------------------------------------------
IMidiDumper & IMidiDumper::initializePart()
{
   IFUNCTRACE_DEVELOP();
   makeConnections();
   notifyObservers(INotificationEvent(readyId, *this));
   return *this;
}


//------------------------------------------------------------------------------
// IMidiDumper :: makeConnections
//------------------------------------------------------------------------------
Boolean IMidiDumper::makeConnections()
{
   IFUNCTRACE_DEVELOP();
   this->enableNotification();
   return true;
}


//------------------------------------------------------------------------------
// IMidiDumper :: startDumper
//------------------------------------------------------------------------------
IMidiDumper & IMidiDumper::startDumper( IMultiLineEdit * mle )
{
   IFUNCTRACE_DEVELOP();
   iMLE = mle;

   if ( !theThread.isStarted() )
   {
      dumpThread = new IThreadMemberFn< IMidiDumper >( *this, IMidiDumper::dumper );
      theThread.start( dumpThread );
   }

   return *this;
}


//------------------------------------------------------------------------------
// IMidiDumper :: stopDumper
//------------------------------------------------------------------------------
IMidiDumper & IMidiDumper::stopDumper()
{
   IFUNCTRACE_DEVELOP();
   if ( theThread.isStarted() )
   {
      // send a sysex message, via a short term instance to shut down the thread
      MINSTANCE stopper;
      MIDICreateInstance( IRTMidi::theIRTMidi->applicationClass()->number(),
                          &stopper, "stopper", 0 );
      MIDIEnableInstance( stopper, MIDI_ENABLE_SEND );
      MIDIAddLink( stopper, instance(), 0, 0 );
      MESSAGE msg;
      msg.ulSourceInstance = stopper;
      msg.ulTime           = 0;
      msg.ulTrack          = 0;
      msg.msg.ulMessage    = 0xf7fffff0;
      MIDISendMessages( &msg, 1, 0 );
      MIDIRemoveLink( stopper, instance(), 0, 0 );
      MIDIDisableInstance( stopper, MIDI_DISABLE_SEND );
      MIDIDeleteInstance( stopper, 0 );
   }
   return *this;
}


//------------------------------------------------------------------------------
// IMidiDumper :: dumper
//------------------------------------------------------------------------------
void IMidiDumper::dumper()
{
   IFUNCTRACE_DEVELOP();
   IString line;
   MESSAGE theMessage;
   unsigned long numBytes;
   unsigned long rc = 0;
   iMLE->addAsLast( "Dumper started\n", 0 , IMultiLineEdit::noTran );
   while( rc == 0 )
   {
      numBytes = sizeof( theMessage );
      rc = MIDIRetrieveMessages( instance(), &theMessage, &numBytes, 0 );
      // stop thread on special sysex message
      if ( rc == 0 && theMessage.msg.ulMessage == 0xf7fffff0 )
         rc = 1;
      if ( rc == 0 )
      {
         line = IString( (USHORT)theMessage.msg.abData[0] ).d2x()
              + " "
              + IString( (USHORT)theMessage.msg.abData[1] ).d2x()
              + " "
              + IString( (USHORT)theMessage.msg.abData[2] ).d2x()
              + " "
              + IString( (USHORT)theMessage.msg.abData[3] ).d2x()
              + "\n";
         iMLE->addAsLast( line , 0 , IMultiLineEdit::noTran );
         if( iMLE->numberOfLines() > iMLE->visibleLines() )
            iMLE->setTop( iMLE->top() + 1 );
      }
   }
   iMLE->addAsLast( "Dumper stopped\n", 0 , IMultiLineEdit::noTran );
}
