/*******************************************************************************
* FILE NAME: IMidiIns.cpp                                                      *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Class implementation of the class:                                         *
*     IMidi- Real Time Midi Instance Class                                     *
*******************************************************************************/

#ifndef _IMIDIINS_
#include "IMidiIns.hpp"
#endif

#ifndef _ITRACE_
#include <itrace.hpp>
#endif

#include <string.h>


const INotificationId IMidiInstance::attribStringId = "IMidiInstance::attribStringId";
const INotificationId IMidiInstance::readyId        = "IMidiInstance::readyId";

//------------------------------------------------------------------------------
// IMidiInstance :: IMidiInstance
//------------------------------------------------------------------------------
IMidiInstance::IMidiInstance( PMIDIINSTANCEINFO i )
{
   IFUNCTRACE_DEVELOP();
   if ( i )
   {
      setInfo( i );
   }
   initializePart();
}

//------------------------------------------------------------------------------
// IMidiInstance :: initializePart
//------------------------------------------------------------------------------
IMidiInstance & IMidiInstance::initializePart()
{
   IFUNCTRACE_DEVELOP();
   makeConnections();
   notifyObservers(INotificationEvent(readyId, *this));
   return *this;
}


//------------------------------------------------------------------------------
// IMidiInstance :: makeConnections
//------------------------------------------------------------------------------
Boolean IMidiInstance::makeConnections()
{
   IFUNCTRACE_DEVELOP();
   this->enableNotification();
   return true;
}


//------------------------------------------------------------------------------
// IMidiInstance :: setInfo
//------------------------------------------------------------------------------
void IMidiInstance::setInfo( PMIDIINSTANCEINFO i )
{
   IFUNCTRACE_DEVELOP();
   memcpy( &theInfo, i, sizeof( MIDIINSTANCEINFO ) );
   updateAttributes();
   theName = theInfo.szmInstanceName;
   genAttribString();
}


//------------------------------------------------------------------------------
// IMidiInstance :: updateAttributes  (to be overriden)
//------------------------------------------------------------------------------
void IMidiInstance::updateAttributes()
{
   IFUNCTRACE_DEVELOP();
}


//------------------------------------------------------------------------------
// IMidiInstance :: ~IMidiInstance
//------------------------------------------------------------------------------
IMidiInstance::~IMidiInstance()
{
   IFUNCTRACE_DEVELOP();
   // Disable send and receive if set
   if ( isReceive() )
      enableReceive( false );
   if ( isSend() )
      enableSend( false );
   // Remove from RTMIDI
   IRTMidi::theIRTMidi->delInstance( this );
}


//------------------------------------------------------------------------------
// IMidiInstance :: instance
//------------------------------------------------------------------------------
MINSTANCE IMidiInstance::instance() const
{
   IFUNCTRACE_DEVELOP();
   return theInfo.minstance;
}


//------------------------------------------------------------------------------
// IMidiInstance :: classNum
//------------------------------------------------------------------------------
MINSTANCE IMidiInstance::classNum() const
{
   IFUNCTRACE_DEVELOP();
   return theInfo.ulClassNumber;
}


//------------------------------------------------------------------------------
// IMidiInstance :: name
//------------------------------------------------------------------------------
IString IMidiInstance::name() const
{
   IFUNCTRACE_DEVELOP();
   return theName;
}


//------------------------------------------------------------------------------
// IMidiInstance :: attribString
//------------------------------------------------------------------------------
IString IMidiInstance::attribString() const
{
   IFUNCTRACE_DEVELOP();
   return iAttribString;
}


//------------------------------------------------------------------------------
// IMidiInstance :: genAttribString (private)
//------------------------------------------------------------------------------
void IMidiInstance::genAttribString()
{
   IFUNCTRACE_DEVELOP();
   if( canSend() )
      iAttribString  = "Y  ";
   else
      iAttribString  = "N  ";
   if( canReceive() )
      iAttribString += "Y  ";
   else
      iAttribString += "N  ";
   if( isSend() )
      iAttribString += "Y ";
   else
      iAttribString += "N ";
   if( isReceive() )
      iAttribString += "Y";
   else
      iAttribString += "N";
   notifyObservers(INotificationEvent(attribStringId, *this));
}


//------------------------------------------------------------------------------
// IMidiInstance :: numLinks
//------------------------------------------------------------------------------
unsigned long IMidiInstance::numLinks() const
{
   IFUNCTRACE_DEVELOP();
   return theInfo.ulNumLinks;
}


//------------------------------------------------------------------------------
// IMidiInstance :: attributes
//------------------------------------------------------------------------------
unsigned long IMidiInstance::attributes() const
{
   IFUNCTRACE_DEVELOP();
   return theInfo.ulAttributes;
}


//------------------------------------------------------------------------------
// IMidiInstance :: canReceive
//------------------------------------------------------------------------------
Boolean IMidiInstance::canReceive() const
{
   IFUNCTRACE_DEVELOP();
   return ( theInfo.ulAttributes & MIDI_INST_ATTR_CAN_RECV );
}


//------------------------------------------------------------------------------
// IMidiInstance :: isReceive
//------------------------------------------------------------------------------
Boolean IMidiInstance::isReceive() const
{
   IFUNCTRACE_DEVELOP();
   return ( theInfo.ulAttributes & MIDI_INST_ATTR_ENABLE_R );
}


//------------------------------------------------------------------------------
// IMidiInstance :: enableReceive
//------------------------------------------------------------------------------
IMidiInstance & IMidiInstance::enableReceive( Boolean enable )
{
   IFUNCTRACE_DEVELOP();
   unsigned long rc;
   if ( enable )
   {
      // Enable receive
      ITRACE_DEVELOP("MIDIEnableInstance[RECEIVE] " + name() );
      rc = MIDIEnableInstance( instance(), MIDI_ENABLE_RECEIVE );
      if ( rc == 0 )
         theInfo.ulAttributes |= MIDI_INST_ATTR_ENABLE_R;
   }
   else
   {
      // Disable receive
      ITRACE_DEVELOP("Calling MIDIDisableInstance[RECEIVE] " + name() );
      rc = MIDIDisableInstance( instance(), MIDI_DISABLE_RECEIVE );
      if ( rc == 0 )
         theInfo.ulAttributes &= ~MIDI_INST_ATTR_ENABLE_R;
   }
   IRTMidi::theIRTMidi->setLastRC( rc );
   genAttribString();
   return *this;
}


//------------------------------------------------------------------------------
// IMidiInstance :: canSend
//------------------------------------------------------------------------------
Boolean IMidiInstance::canSend() const
{
   IFUNCTRACE_DEVELOP();
   return ( theInfo.ulAttributes & MIDI_INST_ATTR_CAN_SEND );
}


//------------------------------------------------------------------------------
// IMidiInstance :: isSend
//------------------------------------------------------------------------------
Boolean IMidiInstance::isSend() const
{
   IFUNCTRACE_DEVELOP();
   return ( theInfo.ulAttributes & MIDI_INST_ATTR_ENABLE_S );
}


//------------------------------------------------------------------------------
// IMidiInstance :: enableSend
//------------------------------------------------------------------------------
IMidiInstance & IMidiInstance::enableSend( Boolean enable )
{
   IFUNCTRACE_DEVELOP();
   unsigned long rc;
   if ( enable )
   {
      // Enable send
      ITRACE_DEVELOP("Calling MIDIEnableInstance[SEND] " + name() );
      rc = MIDIEnableInstance( instance(), MIDI_ENABLE_SEND );
      if ( rc == 0 )
         theInfo.ulAttributes |= MIDI_INST_ATTR_ENABLE_S;
   }
   else
   {
      // Disable send
      ITRACE_DEVELOP("Calling MIDIDIsableInstance[SEND] " + name() );
      rc = MIDIDisableInstance( instance(), MIDI_DISABLE_SEND );
      if ( rc == 0 )
         theInfo.ulAttributes &= ~MIDI_INST_ATTR_ENABLE_S;
   }
   IRTMidi::theIRTMidi->setLastRC( rc );
   genAttribString();
   return *this;
}


//------------------------------------------------------------------------------
// IMidiInstance :: addLink
//------------------------------------------------------------------------------
IMidiInstance & IMidiInstance::addLink( IMidiInstance * toLink )
{
   IFUNCTRACE_DEVELOP();
   if ( canSend() && toLink->canReceive() )
   {
      if ( !isSend() )
      {
         enableSend();
      }
      if ( !toLink->isReceive() )
      {
         toLink->enableReceive();
      }
      ITRACE_DEVELOP("Calling MIDIAddLink " + name() + " to " + toLink->name() );
      IRTMidi::theIRTMidi->setLastRC(
         MIDIAddLink( instance(), toLink->instance(),
                      0,       // slot ?
                      0 ) );
   }
   return *this;
}


//------------------------------------------------------------------------------
// IMidiInstance :: removeLink
//------------------------------------------------------------------------------
IMidiInstance & IMidiInstance::removeLink( IMidiInstance * toLink )
{
   IFUNCTRACE_DEVELOP();
   if ( canSend() && toLink->canReceive() )
   {
      ITRACE_DEVELOP("Calling MIDIRemoveLink " + name() + " from " + toLink->name() );
      IRTMidi::theIRTMidi->setLastRC(
         MIDIRemoveLink( instance(), toLink->instance(),
                      0,       // slot?
                      0 ) );
   }
   return *this;
}


//------------------------------------------------------------------------------
// IMidiInstance :: sendMessage (private)
//------------------------------------------------------------------------------
void IMidiInstance::sendMessage( BYTE b1, BYTE b2, BYTE b3, BYTE b4 ) const
{
   IFUNCTRACE_DEVELOP();
   // Build message
   MESSAGE msg;
   msg.ulSourceInstance = instance();
   msg.ulTime           = 0;
   msg.ulTrack          = 0;
   msg.msg.abData[0]    = b1;
   msg.msg.abData[1]    = b2;
   msg.msg.abData[2]    = b3;
   msg.msg.abData[3]    = b4;
   IRTMidi::theIRTMidi->setLastRC( MIDISendMessages( &msg, 1, 0 ) );
}
