/*******************************************************************************
* FILE NAME: IRTMidi.cpp                                                       *
*                                                                              *
* DESCRIPTION:                                                                 *
*   Class implementation of the class:                                         *
*     IRTMidi- Real Time Midi Base Class                                       *
*******************************************************************************/
#ifndef _INOTIFEV_
#include <inotifev.hpp>
#endif

#ifndef _IOBSERVR_
#include <iobservr.hpp>
#endif

#ifndef _ISTDNTFY_
#include <istdntfy.hpp>
#endif

#ifndef _IRTMIDI_
#include "IRTMidi.hpp"
#endif

#ifndef _IMIDICLS_
#include "IMidiCls.hpp"
#endif

#ifndef _IMIDIINS_
#include "IMidiIns.hpp"
#endif

#ifndef _IVBDEFS_
#include <ivbdefs.h>
#endif

#ifndef _ITRACE_
#include <itrace.hpp>
#endif

const INotificationId IRTMidi::readyId     = "IRTMidi::readyId";
const INotificationId IRTMidi::classesId   = "IRTMidi::classes";
const INotificationId IRTMidi::instancesId = "IRTMidi::instances";
const INotificationId IRTMidi::lastRCId    = "IRTMidi::lastRC";
const INotificationId IRTMidi::RCExplanationId = "IRTMidi::RCExplanation";

IRTMidi * IRTMidi::theIRTMidi = NULL;

// used for MIDIQueryVersion
#define  MIDIVER_MAJOR   0xC000
#define  MIDIVER_MINOR   0x3C00
#define  MIDIVER_BUGFIX  0x03C0
#define  MIDIVER_PHASE   0x0030
#define  MIDIVER_BUILD   0x000F


//------------------------------------------------------------------------------
// IRTMidi :: IRTMidi
//------------------------------------------------------------------------------
IRTMidi::IRTMidi()
   : appClass( NULL ),
     hdwClass( NULL ),
     iLastRC( 0 )
{
   IFUNCTRACE_DEVELOP();

   // Exception of more that one created
   // code to come

   // Remember my address
   theIRTMidi = this;


   unsigned long rc = 0;

   // Initialize MIDI Setup structure and call MIDISetup
   theSetup.ulStructLength         = sizeof( MIDISETUP );
   theSetup.pulMaxRTSysexLength    = &maxRTSysexLen;
   theSetup.ppulMIDICurrentTime    = &currentTime;
   theSetup.ulFlags                = 0;
   theSetup.pfnMIDI_NotifyCallback = NULL;
   theSetup.hwndCallback           = 0;
   theSetup.hqueueCallback         = 0;
   rc = MIDISetup( &theSetup, 0 );

   // Query Classes and copy them
   if ( rc == 0 )
   {
      rc = getClasses();
   }

   // Ditto the instances (Hardware Instances)
   if ( rc == 0 )
   {
      rc = getInstances();
   }

   setLastRC( rc );

   // Start up the MIDI timer
   startTimer();

}     //end constructor


//------------------------------------------------------------------------------
// IRTMidi :: ~IRTMidi
//------------------------------------------------------------------------------
IRTMidi::~IRTMidi()
{
   IFUNCTRACE_DEVELOP();
   this->disableNotification();
   stopTimer();
   delInstances();
   delClasses();
}


//------------------------------------------------------------------------------
// IRTMidi :: initializePart
//------------------------------------------------------------------------------
IRTMidi & IRTMidi::initializePart()
{
   IFUNCTRACE_DEVELOP();
   makeConnections();
   notifyObservers(INotificationEvent(readyId, *this));
   return *this;
}


//------------------------------------------------------------------------------
// IRTMidi :: makeConnections
//------------------------------------------------------------------------------
Boolean IRTMidi::makeConnections()
{
   IFUNCTRACE_DEVELOP();
   this->enableNotification();
   return true;
}


//------------------------------------------------------------------------------
// IRTMidi :: version
//------------------------------------------------------------------------------
IString IRTMidi::version() const
{
   IFUNCTRACE_DEVELOP();
   unsigned long ulVersion;
   ((IRTMidi*)this)->setLastRC( MIDIQueryVersion( &ulVersion ) );
   if ( iLastRC == 0 )
   {
      return "Ver "
           + IString((ulVersion & MIDIVER_MAJOR) >> 14)
           + "."
           + IString((ulVersion & MIDIVER_MINOR) >> 10)
           + ", Fix "
           + IString((ulVersion & MIDIVER_BUGFIX) >> 6)
           + ", Phase "
           + IString((ulVersion & MIDIVER_PHASE) >> 4)
           + ", Bld "
           + IString(ulVersion & MIDIVER_BUILD);
   }
   else
      return "MIDIQueryVersion failed, rc=" + IString(iLastRC);
}


//------------------------------------------------------------------------------
// IRTMidi :: lastRC
//------------------------------------------------------------------------------
unsigned long IRTMidi::lastRC() const
{
   IFUNCTRACE_DEVELOP();
   return iLastRC;
}


//------------------------------------------------------------------------------
// IRTMidi :: setLastRC
//------------------------------------------------------------------------------
IRTMidi& IRTMidi::setLastRC(unsigned long aLastRC)
{
   IFUNCTRACE_DEVELOP();
   if ( iLastRC != aLastRC
     && iLastRC == 0 )
   {
      iLastRC = aLastRC;
      notifyObservers(INotificationEvent(IRTMidi::lastRCId, *this));
      notifyObservers(INotificationEvent(IRTMidi::RCExplanationId, *this));
   }
   return *this;
}


//------------------------------------------------------------------------------
// IRTMidi :: setLastRC
//------------------------------------------------------------------------------
IRTMidi& IRTMidi::resetLastRC()
{
   IFUNCTRACE_DEVELOP();
   if ( iLastRC != 0 )
   {
      iLastRC = 0;
      notifyObservers(INotificationEvent(IRTMidi::lastRCId, *this));
      notifyObservers(INotificationEvent(IRTMidi::RCExplanationId, *this));
   }
  return *this;
}


//------------------------------------------------------------------------------
// IRTMidi :: RCExplanation
//------------------------------------------------------------------------------
IString IRTMidi::RCExplanation() const
{
   IFUNCTRACE_DEVELOP();
   IString DebugString;
   switch( iLastRC )
   {
   case 0:
      DebugString = "OK";
      break;

   // Timer error messages
   case TIMERERR_INVALID_PARAMETER:
      DebugString = "MIDI Timer: Invalid Parameter";
      break;
   case TIMERERR_INTERNAL_SYSTEM:
      DebugString = "MIDI Timer: Internal System Error";
      break;

   // RTMIDI messages
   case MIDIERR_DUPLICATE_INSTANCE_NAME:
      DebugString = "MIDIERR_DUPLICATE_INSTANCE_NAME";
      break;
   case MIDIERR_HARDWARE_FAILED:
      DebugString    = "MIDIERR_HARDWARE_FAILED";
      break;
   case MIDIERR_INTERNAL_SYSTEM:
      DebugString = "MIDIERR_INTERNAL_SYSTEM";
      break;
   case MIDIERR_INVALID_BUFFER_LENGTH:
      DebugString = "MIDIERR_INVALID_BUFFER_LENGTH";
      break;
   case MIDIERR_INVALID_CLASS_NUMBER:
      DebugString    = "MIDIERR_INVALID_CLASS_NUMBER";
      break;
   case MIDIERR_INVALID_CONFIG_DATA:
      DebugString    = "MIDIERR_INVALID_CONFIG_DATA";
      break;
   case MIDIERR_INVALID_FLAG:
      DebugString    = "MIDIERR_INVALID_FLAG";
      break;
   case MIDIERR_INVALID_INSTANCE_NAME:
      DebugString    = "MIDIERR_INVALID_INSTANCE_NAME";
      break;
   case MIDIERR_INVALID_INSTANCE_NUMBER:
      DebugString    = "MIDIERR_INVALID_INSTANCE_NUMBER";
      break;
   case MIDIERR_INVALID_PARAMETER:
      DebugString = "MIDIERR_INVALID_PARAMETER";
      break;
   case MIDIERR_INVALID_SETUP:
      DebugString = "MIDIERR_INVALID_SETUP";
      break;
   case MIDIERR_NO_DRIVER:
      DebugString    = "MIDIERR_NO_DRIVER";
      break;
   case MIDIERR_NO_DEFAULT_HW_NODE:
      DebugString    = "MIDIERR_NO_DEFAULT_HW_NODE";
      break;
   case MIDIERR_NOT_ALLOWED:
      DebugString    = "MIDIERR_NOT_ALLOWED";
      break;
   case MIDIERR_NOTIFY_MISSED:
      DebugString    = "MIDIERR_NOTIFY_MISSED";
      break;
   case MIDIERR_RESOURCE_NOT_AVAILABLE:
      DebugString    = "MIDIERR_RESOURCE_NOT_AVAILABLE";
      break;
   case MIDIERR_SENDONLY:
      DebugString    = "MIDIERR_SENDONLY";
      break;
   case MIDIERR_RECEIVEONLY:
      DebugString    = "MIDIERR_RECEIVEONLY";
      break;

   default:
      DebugString    = "Beats Me!";
      break;
   }
   return DebugString;
}


//------------------------------------------------------------------------------
// IRTMidi :: delClasses (private)
//------------------------------------------------------------------------------
void IRTMidi::delClasses()
{
   IFUNCTRACE_DEVELOP();
   IVSequence<IMidiClass*>::Cursor c(iClasses);
   forCursor( c )
   {
      IMidiClass * cls = c.element();
      if ( cls )
         delete c.element();
   }
}


//------------------------------------------------------------------------------
// IRTMidi :: getClasses (private)
//------------------------------------------------------------------------------
unsigned long IRTMidi::getClasses()
{
   IFUNCTRACE_DEVELOP();
   unsigned long rc;
   unsigned long nc = 0;
   PMIDICLASSINFO ci = NULL;

   delClasses();
   // Query classes and post them
   rc = MIDIQueryNumClasses( &nc, 0 );
   if ( rc == 0 && nc > 0 )
   {
      ci = new MIDICLASSINFO[ nc ];
      rc = MIDIQueryClassList( nc, ci, 0 );
   }
   if ( rc == 0 && ci )
   {
      for ( int i = 0; i < nc; i++ )
      {
         IMidiClass * cls = new IMidiClass( &ci[i] );
         if ( cls->name() == "Application" )
            appClass = cls;
         if ( cls->name() == "Hardware" )
            hdwClass = cls;
         iClasses.add( cls );
      }
   }
   if ( ci )
   {
      delete[] ci;
   }
   notifyObservers(INotificationEvent(IRTMidi::classesId, *this));
   return rc;
}


//------------------------------------------------------------------------------
// IRTMidi :: addInstance
//------------------------------------------------------------------------------
MINSTANCE IRTMidi::addInstance( IMidiInstance * theInstance,
                                ULONG classNum, IString instanceName )
{
   IFUNCTRACE_DEVELOP();
   unsigned long rc;
   MINSTANCE newInstance;
   unsigned long ni = 0;
   PMIDIINSTANCEINFO ci = NULL;

   // Add new instance
   rc = MIDICreateInstance( classNum,
                            &newInstance,
                            instanceName,
                            0 );
   // Now find it to initialize the Instance Object
   if ( rc == 0 )
   {
      rc = MIDIQueryNumInstances( &ni, 0 );
      if ( rc == 0 && ni > 0 )
      {
         ci = new MIDIINSTANCEINFO[ ni ];
         rc = MIDIQueryInstanceList( ni, ci, 0 );
      }
   }
   if ( rc == 0 && ci )
   {
      for ( int i = 0; i < ni; i++ )
      {
         if ( ci[i].minstance == newInstance )
         {
            theInstance->setInfo( &ci[i] );
            iInstances.add( theInstance );
         }
      }
   }
   if ( ci )
   {
      delete[] ci;
   }
   setLastRC( rc );
   notifyObservers(INotificationEvent(IRTMidi::instancesId, *this));
   return newInstance;
}


//------------------------------------------------------------------------------
// IRTMidi :: delInstance
//------------------------------------------------------------------------------
void IRTMidi::delInstance( IMidiInstance * theInstance )
{
   IFUNCTRACE_DEVELOP();
   unsigned long rc;

   // Find instance in sequence and remove it
   for( INumber i = 1; i <= iInstances.numberOfElements(); i++ )
   {
     if ( iInstances.elementAtPosition( i ) == theInstance )
     {
        iInstances.removeAtPosition( i );
     }
   }

   // Delete MIDI Instance
   rc = MIDIDeleteInstance( theInstance->instance(), 0 );
   setLastRC( rc );
   notifyObservers(INotificationEvent(IRTMidi::instancesId, *this));
}


//------------------------------------------------------------------------------
// IRTMidi :: delInstances (private)
//------------------------------------------------------------------------------
void IRTMidi::delInstances()
{
   IFUNCTRACE_DEVELOP();
   // Delete all instances this way as delete of an instance removes
   // it from the instance list through the miracle of double
   // dispatching.  See above.
   while( iInstances.numberOfElements() > 0 )
   {
      IMidiInstance * ins = iInstances.elementAtPosition( 1 );
      if ( ins )
         delete ins;
   }
}


//------------------------------------------------------------------------------
// IRTMidi :: getInstances (private)
//------------------------------------------------------------------------------
unsigned long IRTMidi::getInstances()
{
   IFUNCTRACE_DEVELOP();
   unsigned long rc;
   unsigned long ni = 0;
   PMIDIINSTANCEINFO ci = NULL;

   iInstances.removeAll();
   // Query instances and post them
   rc = MIDIQueryNumInstances( &ni, 0 );
   if ( rc == 0 && ni > 0 )
   {
      ci = new MIDIINSTANCEINFO[ ni ];
      rc = MIDIQueryInstanceList( ni, ci, 0 );
   }
   if ( rc == 0 && ci )
   {
      for ( int i = 0; i < ni; i++ )
      {
         iInstances.add( new IMidiInstance( &ci[i] ) );
      }
   }
   if ( ci )
   {
      delete[] ci;
   }
   notifyObservers(INotificationEvent(IRTMidi::instancesId, *this));
   return rc;
}


//------------------------------------------------------------------------------
// IRTMidi :: startTimer
//------------------------------------------------------------------------------
void IRTMidi::startTimer() const
{
   IFUNCTRACE_DEVELOP();
   ((IRTMidi*)this)->setLastRC( MIDITimer( MIDI_START_TIMER, 0 ) );
}


//------------------------------------------------------------------------------
// IRTMidi :: stopTimer
//------------------------------------------------------------------------------
void IRTMidi::stopTimer() const
{
   IFUNCTRACE_DEVELOP();
   ((IRTMidi*)this)->setLastRC( MIDITimer( MIDI_STOP_TIMER, 0 ) );
}


//------------------------------------------------------------------------------
// IRTMidi :: classes (static)
//------------------------------------------------------------------------------
IVSequence<IMidiClass*> * IRTMidi::classes()
{
   IFUNCTRACE_DEVELOP();
   return &iClasses;
}


//------------------------------------------------------------------------------
// IRTMidi :: instances (static)
//------------------------------------------------------------------------------
IVSequence<IMidiInstance*> * IRTMidi::instances()
{
   IFUNCTRACE_DEVELOP();
   return &iInstances;
}


//------------------------------------------------------------------------------
// IRTMidi :: applicationClass
//------------------------------------------------------------------------------
IMidiClass * IRTMidi::applicationClass()
{
   IFUNCTRACE_DEVELOP();
   return appClass;
}


//------------------------------------------------------------------------------
// IRTMidi :: hardwareClass
//------------------------------------------------------------------------------
IMidiClass * IRTMidi::hardwareClass()
{
   IFUNCTRACE_DEVELOP();
   return hdwClass;
}
