/*****************************************************************************\
| Standard skeletal structure for new REXX programs.                          |
|                                                                             |
| Libraries:   REXXSAA                                                        |
|  Category:                                                                  |
|     Class:                                                                  |
|      Type:                                                                  |
|    Author:   Bob Rice <bobrice@ibm.net>                                     |
|                                                                             |
| Copyright (c) 1995, 1966 Empirical Heuristics                               |
\**************************************************************************r4*/
  EH. = ''                                      /* Initialize global stem    */
  /*=========================================================================*\
  | >>>>> NOTE:  You should set the values of the next three variables! <<<<< |
  | The dollar signs in the strings Revision: and Date: may be removed, if    |
  | desired.  They are there to permit automated source code version control. |
  \*=========================================================================*/
  EH.pgm_ver  = '$Revision:   1.2  $'           /* Version number of program */
  EH.pgm_date = '$Date:   20 Oct 1996 22:48:56  $' /* Revision date of pgm   */
  EH.reqd_params = 0                            /* Number of required params */
  parse arg args; call Prolog args; drop args; signal on Error; signal on Halt
  signal on Failure; signal on NoValue; signal on Syntax; signal on NotReady
  if EH.trace \= '' then say '--> Entering' EH.pgm_filespec
  trace value EH.trace; nop
/*------------------------------------------------------End Skeletal Header--*/

  /***************************************************************************\
  |                              START OF PROGRAM                             |
  \***************************************************************************/
  /*>>>>>>>>>>>>>>>>>>>>>>>>>> your code goes here <<<<<<<<<<<<<<<<<<<<<<<<<<*/
  exit

  /***************************************************************************\
  |                            PROGRAM SUBROUTINES                            |
  \***************************************************************************/
  /*>>>>>>>>>>>>>>>>>>>>>> your procedures can go here <<<<<<<<<<<<<<<<<<<<<<*/

  /***************************************************************************\
  |                        GENERAL LIBRARY SUBROUTINES                        |
  \***************************************************************************/
  /*>>>>>>>>>>>>>>>>>>>>>> your procedures can go here <<<<<<<<<<<<<<<<<<<<<<*/

  /*=========================================================================*\
  | This routine will be called when an error is trapped.  It is called after |
  | all error handling is completed.  You may insert your own code here to do |
  | cleanup and whatever else is necessary for your program on abnormal exit. |
  | But DO NOT DELETE THIS ROUTINE; it is required for proper operation.      |
  \*=========================================================================*/
ErrorExit:
  return

/*****************************************************************************\
|                          REQUIRED SKELETAL PROLOG                           |
\*****************************************************************************/
Prolog: procedure expose EH.
                EH.prolog_ver = '1.1'
  if pos(':',EH.pgm_ver)  > 0 then parse var EH.pgm_ver ':' EH.pgm_ver '$'
  EH.pgm_ver = strip(EH.pgm_ver)
  if pos(':',EH.pgm_date) > 0 then do           /* Extract date from string  */
    parse var EH.pgm_date ':' d m y . '$'
    EH.pgm_date = strip(d m y); drop d m y
  end
  parse arg     EH.cmd_line
  parse version EH.REXX EH.REXX_ver EH.REXX_date
  parse source  EH.os EH.calling_environ EH.pgm_filespec
  if EH.os = 'OS/2' then do QueCmd('VER'); pull EH.os_ver
    if EH.os_ver \= '' then EH.os_ver = word(EH.os_ver,words(EH.os_ver))
  end
  EH.environ  = 'OS2ENVIRONMENT'
  EH.pgm_fnfe = filespec('N',EH.pgm_filespec)
  EH.pgm_path = filespec('D',EH.pgm_filespec) || filespec('P',EH.pgm_filespec)
  parse var EH.pgm_fnfe EH.pgm_fn '.' EH.pgm_fe; arg help .
  if wordpos(help,'? ?? ??? ???? /? /H /HELP -H -HELP') > 0 then call Tell help
  if EH.reqd_params > words(EH.cmd_line) then call Tell '??'
  /*--------------------------------------------------------------*\
  |  Get trace value first from the environment, then overwrite it |
  |  if it is also specified on the command line.  Also get debug. |
  \*--------------------------------------------------------------*/
  EH.trace = value('TRACE',,EH.environ)
  t = wordpos('/TRACE',translate(EH.cmd_line))
  if t > 0 then EH.trace = word(EH.cmd_line, t+1)
  EH.debug = value('DEBUG',,EH.environ)
  if EH.debug \= 1 then EH.debug = 0
  d = wordpos('/DEBUG',translate(EH.cmd_line))
  if d > 0 then EH.debug = 1
  /* Remove /DEBUG and /TRACE params from EH.cmd_line */
  tmp = ''
  do i = 1 to words(EH.cmd_line)
    if i = d then iterate
    if i = t then do; i = i + 1; iterate; end
    tmp = tmp word(EH.cmd_line,i)
  end
  EH.cmd_line = strip(tmp); drop d t tmp
  if FuncReg() = 0 then return                  /* Register external funcs   */
                   else exit 255
Tell: procedure expose EH.                      /* Display help information  */
  parse arg help
  call TellHelp help, EH.pgm_filespec, EH.pgm_ver, EH.pgm_date, EH.prolog_ver
  exit 254
/*===========================================================================*\
|                               Trap Routines                                 |
\*===========================================================================*/
Error:    call ProcessTrap SIGL, 'ERROR',    rc
Failure:  call ProcessTrap SIGL, 'FAILURE',  rc
Halt:     call ProcessTrap SIGL, 'HALT',     ''
NotReady: call ProcessTrap SIGL, 'NOTREADY', ''
NoValue:  call ProcessTrap SIGL, 'NOVALUE',  ''
Syntax:   call ProcessTrap SIGL, 'SYNTAX',   rc
ProcessTrap:
  if pos(':',EH.pgm_filespec) > 0 then TRAP.source_line = ,
    strip(sourceline(arg(1)))            /* Get source line if available     */
  else do                                /* Program is running in macrospace */
    TRAP.path_and_program = value('TEMP',,EH.environ) || TRAP.path_and_program
    TRAP.source_line = 'Source line is not available.'
  end
  EH.dump_file = ErhTrap(arg(1),arg(2),arg(3),EH.pgm_filespec,TRAP.source_line)
  if EH.dump_file  \= '' then do      /* Create DMP file if appropriate      */
    call SysFileDelete EH.dump_file
    drop TRAP. RC RESULT SIGL         /* Remove meaningless vars for clarity */
    call VarDump EH.dump_file         /* Write variables to program.DMP file */
  end
  erc = ErhTrap('FINISH_UP',EH.dump_file,EH.pgm_filespec,arg(1)) /* Finish up*/
  call ErrorExit
  exit erc

  /***************************************************************************\
  |                              DEBUG ROUTINES                               |
  |                                                                           |
  | The routines below this point in the code are not part of the skeleton    |
  | or the error handler and are not necessary for its proper operation.      |
  | These routines may prove useful for debugging during program development  |
  | and as such are included here for convenience.  You may delete them after |
  | program development is completed and before program release.              |
  \***************************************************************************/

  /*=========================================================================*\
  | Debug routine to dump accessible vars to a sorted file.  The filename is  |
  | the line number from which the call is made and the file extent is .V.    |
  | The directory is the current directory.                                   |
  |                                                                           |
  | Params: [ var-list ] [, { I | E }]                                        |
  |                                                                           |
  |   var-list    is a quoted list of variable names separated by blanks.     |
  |                                                                           |
  |   [ I | E ]   determines whether the listed variables are to be included  |
  |               or excluded.  Included is the default.                      |
  |                                                                           |
  | Example: 'i j k lines.', 'e'                                              |
  |                                                                           |
  | 06/01/95 - REXXSAA, REXXLIB                                               |
  | (c) 1995 Empirical Heuristics                                             |
  \*=========================================================================*/
DumpV:
  parse arg _dumpv.str, _dumpv.ie
  _dumpv.file = sigl'.V'
  call SysFileDelete _dumpv.file
  if _dumpv.str \= '' then do
    _dumpv.list = ''''word(_dumpv.str,1)''''
    do _dumpvi = 2 to words(_dumpv.str)
      _dumpv.list = _dumpv.list || ','''word(_dumpv.str,_dumpvi)''''
    end
  end
  else _dumpv.list = ''
  if _dumpv.ie = '' then _dumpv.ie = 'I'
  if _dumpv.list = ''
    then call VarDump _dumpv.file
    else interpret 'call VarDump _dumpv.file, _dumpv.ie,' _dumpv.list
  call FileRead _dumpv.file, '_dumpv.',, 'E','CRANDLF'
  call ArraySort '_dumpv.'
  call FileWrite _dumpv.file,'_dumpv.','R'
  say 'Variable data dumped to file' _dumpv.file'.'
  drop _dumpv. _dumpvi
  return
/*--Begin Help-----------------------------------------------------------------
This section is the help text.

Params:

where:

________________
Alternate Params: [ ? | ?? | ??? | ???? ]

where:

  ?     Displays up to the "Syntax:" or "Params:" portion of this help text.

  ??    Displays this entire help text except for the technical information.

  ???   Displays this entire help text.

  ????  Puts this help text into a file whose name is the same as the name of
        this program and whose extent is .ABS.  The file is written to the same
        directory as that in which this program resides.

_______________
Technical Notes

>>>>>  IMPORTANT - IMPORTANT - I M P O R T A N T - IMPORTANT - IMPORTANT  <<<<<

Several lines in this skeletal structure contain strings which will be used by
an update program to update the skeleton in your program to a newer version
should one become available.  It is extremely important that these lines be
present in their original relative locations and that the strings within these
lines remain intact.  Any disturbance may cause the update program to fail when
upgrading the skeletal structure of a program you have developed based on the
Skeleton.  The strings which must not be disturbed are:

             "--End Skeletal Header--"
             "REQUIRED SKELETAL PROLOG" and the line above it
             "--Begin Help--"
             "--End Help--"

In addition, except for changing the values of three variables, as noted below
in the section "Global" Variables, do not change any of the lines between the
header comment box and the "--End Skeletal Header--" line.

Also, do not delete the ErrorExit procedure.  It is okay to leave it empty, but
it must remain in your code.

Finally, do not delete the REQUIRED SKELETAL PROLOG comment box, or any of the
code following it.  It is okay, however, to delete the DumpV procedure, its
comment header, and the DEBUG ROUTINES comment box, if you wish to do so.

____________________
Debugging Assistance

The keyword parameter /DEBUG may be added to the invocation line.  This will
set the variable EH.debug to 1. You can then use this variable in your code to
selectively execute code when /DEBUG is specified as a parameter.  If /DEBUG is
not specified, EH.debug is set to 0. You may also set an environment variable
called DEBUG to either 0 or 1 to accomplish the same thing.  The /DEBUG
parameter will override the DEBUG environment variable.

The keyword parameter /TRACE followed by a valid trace argument may be added to
the end of the invocation line to initiate tracing as, for example:  /TRACE ?r
You may also set an environment variable called TRACE to a valid REXX trace
parameter to accomplish the same thing.  The /TRACE parameter will override the
TRACE environment variable.

____________
Help Display

The four parameters ?, ??, ???, and ????  display various levels of help
extracted from the embedded help information.  See the help in TellHelp.cmd for
a fuller explanation of what these do.  Knowing what these four parameter
values do will assist you in preparing your own embedded help information.

Note that when help is requested by the user, none of your code is executed.


Help Content

The section of this embedded help entitled Technical Notes is quite extensive
and intended for the programmer rather than the user.  I suggest that you
delete the entire Technical Information section from your own embedded help to
reduce the bulk of your programs.

What you write for embedded help is, of course, up to you.  But I suggest you
leave in it the section called Alternate Params, since a users of your programs
could not guess that additional help is available for the asking.  Of course,
if you don't include any additional help after the "Params:" or "Syntax:"
line, then by all means delete the Alternate Params section.

Whatever you decide to do, be sure not to delete the "--End Help--" marker
line!

__________________
"Global" Variables

Several variables are initialized by Skeleton before your code begins to
execute.  Some of these provide information to the Skeleton for use in handling
errors and processing help requests; others do not.  All can be referred to in
your own code to save you from having to set similar variables yourself.  You
should not change the value of any of them.  Here is the list of variables:

  EH.calling_environ    can be 'COMMAND', 'SUBROUTINE', or 'FUNCTION'
  EH.cmd_line           everything after program name except: the /DEBUG para-
                        meter, the /TRACE parameter and the token following it,
                        and any extraneous spaces before, after and between
                        parameters
  EH.debug              can be 0 or 1
  EH.environ            'OS2ENVIRONMENT' for OS/2
  EH.os                 'OS/2' for OS/2
  EH.os_ver             '3.00' for OS/2 Warp
  EH.pgm_fe             program file extent - 'cmd' normally
  EH.pgm_filespec       complete drive, path, filename and extent as one token
  EH.pgm_fn             the name of the program excluding the '.' and extent
  EH.pgm_fnfe           the filename and extent with '.' as one token
  EH.pgm_path           the drive and path including trailing '\'
  EH.prolog_ver         the version of the prolog (Skeleton) code
  EH.REXX               the REXX interpreter used - 'REXXSAA' for OS/2 REXX
  EH.REXX_date          the date of the REXX interpreter version as dd mmm yyyy
  EH.REXX_ver           the version of the REXX interpreter - '4.00' for Warp
  EH.trace              null or a valid REXX trace parameter if set correctly

  EH.pgm_ver            the version number of your program up to six characters
  EH.pgm_date           the date of your program version as dd mmm yyyy
  EH.reqd_params        the number of parameters your program requires,
                        excluding optional parameters (see below)

Note that you must set the values of the last three variables yourself.  You'll
find these just above the START OF PROGRAM comment box.


EH.reqd_params

The purpose of this variable is to force the display of help information, at
the ? level, in the event the program is invoked with fewer than the number of
required parameters, presumably because the user has made an error and needs
help.  It has no other effect.  If you don't want help for your program to be
forced, then leave this value set to 0.


Handling EH. Variables

The REXX language makes no provision for true global variables, so we must
simulate this need.  All the variables required for the proper operation of the
Skeleton are available as elements of the EH. stem array.  The values of these
variables must be available to every part of your code.  This presents a
problem inside the code of your procedures when the REXX keyword instruction
PROCEDURE is used.  This instruction prevents your procedure code from seeing
the EH. variables.  To circumvent this, YOU MUST USE THE EXPOSE OPTION TO MAKE
THESE VARIABLES AVAILABLE TO YOUR PROCEDURE CODE.  For example, if you have a
procedure called MyProc which uses the PROCEDURE instruction, then do this:

  MyProc: procedure expose EH.
    /* ...your procedure code goes here... */
    return

This ensures the availability of the required variables.  If you don't expose
EH. to your procedure code, then Skeleton cannot process any errors occuring
within the code.

Procedures not using the PROCEDURE instruction need not use EXPOSE; indeed,
that would be a syntax error.

_______________________
Predefined return codes:

  Code  Meaning
  ----  -------------------------------------------------
   254  Display of internal help was requested or forced.
   255  Failure in external function registration.

___________________
Development History

$Log:   G:/rxdv/skeleton/vcs/skeleton.cm!  $
  
     Rev 1.2   20 Oct 1996 22:48:56
  *  Added GENERAL LIBRARY SUBROUTINES section.
  *  Changed author's address.

     Rev 1.1   11 Nov 1995 16:10:50
  *  Now correctly set EH.cmd_line when /TRACE or /DEBUG used.
  *  Added NotReady trapping.

     Rev 1.0   03 Aug 1995 20:10:24
  Initial revision.
--End Help-------------------------------------------------------------------*/
