/*
 * JAccel.cpp
 *
 * Accelerator table classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JAccel.hpp"

#include "JPM.hpp"
#include "JResID.hpp"
#include <string.h>

// "Invalid accelerator table in method %s"
#define accelError( s) jlib_throw( new JException( err_accel, JExn::base, s))

// JAccel class - an entry in an accelerator table

JAccel::JAccel() : fs( 0), key( 0), cmd( 0)
{}

JAccel::JAccel( void *pAccel)
{
   assertParms( pAccel, "JAccel::JAccel");
   PACCEL p = (PACCEL) pAccel;
   fs = p->fs; key = p->key; cmd = p->cmd;
}

JAccel::JAccel( JAccel::options opts, ushort k, ushort v)
       : fs( opts), key( k), cmd( v)
{}

JAccel::JAccel( JAccel::options opts, JKeyEvent::vkeys k, ushort v)
       : fs( opts), key( k), cmd( v)
{}

// JAccelTable class - /slightly/ confused, but not overly so

void JAccelTable::get()
{
   if( hAccel) {
      ulong size = WinCopyAccelTable( hAccel, NULL, 0);

      if( !size)
         accelError( "JAccelTable::get");
      else {
         pTable = malloc( size);

         if( !WinCopyAccelTable( hAccel, (PACCELTABLE) pTable, size))
            accelError( "JAccelTable::get");
      }
   } else { // make empty table
      pTable = malloc( sizeof( ACCELTABLE) - sizeof( ACCEL));
      ((PACCELTABLE) pTable)->cAccel = 0;
      if( (((PACCELTABLE) pTable)->codepage =
          WinQueryCp( JPM::current()->hmq())) == 0);
         pmError( 3, "WinQueryCp");
   }
}

JAccelTable::JAccelTable() : hAccel( 0), autodelete( 0)
{
   get();
}

JAccelTable::JAccelTable( const JAccelTable &copy)
{
   hAccel = copy.handle();
   pTable = 0;
   autodelete = 0;
}

JAccelTable::JAccelTable( const JResID &id, const JModule &mod)
            : pTable( 0), autodelete( true)
{
   hAccel = WinLoadAccelTable( JPM::current()->hab(),
                               mod,
                               id.value());
   if( !hAccel)
      resError( "accelerator table");
}

JAccelTable::~JAccelTable()
{
   if( pTable) free( pTable);
   if( autodelete)
      if( !WinDestroyAccelTable( hAccel))
         accelError( "JAccelTable::~JAccelTable");
}

JAccelTable &JAccelTable::addAccel( const JAccel &entry)
{
   if( !pTable) get();
   PACCELTABLE pAccelTable = (PACCELTABLE) pTable;
   pAccelTable->cAccel++;
   pTable = realloc( pTable,
              sizeof( ACCELTABLE) + (pAccelTable->cAccel - 1) * sizeof( ACCEL));
   memcpy( &pAccelTable->aaccel[ pAccelTable->cAccel - 1],
           &entry,
           sizeof( ACCEL));

   return self;
}

ushort JAccelTable::query( ushort key, JAccel::options options)
{
   if( !pTable) get();

   for( int i = 0; i < ((PACCELTABLE) pTable)->cAccel; i++)
      if( ((PACCELTABLE) pTable)->aaccel[ i].key == key &&
          ((PACCELTABLE) pTable)->aaccel[ i].fs == options)
             return ((PACCELTABLE) pTable)->aaccel[ i].cmd;

   notFound( "accelerator table entry");
   return 0;
}
