/*
 * JAtoms.cpp
 *
 * Atom tabels, atoms
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "JOs2.h"
#include "JAtoms.hpp"
#include "JStr.hpp"

// We only support string atoms.
// Atom table ----------------------------------------------------------------
JAtomTable::JAtomTable( ulong h) : hTable( h)
{
   assertParms( h, "JAtomTable::ctor");
}

JAtomTable::JAtomTable() : hTable( 0)
{
   hTable = WinCreateAtomTable( 0, 0);
   if( !hTable)
      pmError( 4, "WinCreateAtomTable");
}

JAtomTable::~JAtomTable()
{
   if( hTable != WinQuerySystemAtomTable()) {
      ulong rc = WinDestroyAtomTable( hTable);
      if( rc)
         pmError( 1, "WinDestroyAtomTable");
   }
}

// Atom insertion, deletion, querying, usage count ---------------------------
JAtom JAtomTable::add( const char *s) const
{
   ulong rc = WinAddAtom( hTable, s);
   if( !rc)
      pmError( 3, "WinAddAtom");

   return JAtom( rc);
}

JAtomTable &JAtomTable::remove( const JAtom &atom)
{
   ulong rc = WinDeleteAtom( hTable, atom.handle());
   if( rc)
      pmError( 3, "WinDeleteAtom");
   return self;
}

JStr JAtomTable::nameOf( const JAtom &atom) const
{
   ulong rc = WinQueryAtomLength( hTable, atom.handle());
   if( !rc)
      pmError( 4, "WinQueryAtomLength");

   JStr str;
   str.setsize( rc + 1);

   rc = WinQueryAtomName( hTable, atom.handle(), str.buffer(), rc + 1);
   if( !rc)
      pmError( 3, "WinQueryAtomName");

   return str;
}

ulong JAtomTable::usageCountOf( const JAtom &atom) const
{
   ulong rc = WinQueryAtomUsage( hTable, atom.handle());
   if( !rc)
      pmError( 2, "WinQueryAtomUsage");
   return rc;
}

// System atom table ---------------------------------------------------------
JAtomTable JAtomTable::system( WinQuerySystemAtomTable());

// Message IDs ---------------------------------------------------------------
JMsgID::JMsgID( const char *name) : atom( JAtomTable::system.add( name))
{}

JMsgID::~JMsgID()
{
   JAtomTable::system.remove( atom);
}
