/*
 * JBFile.cpp
 *
 * Base disk file class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JBFile.hpp"
#include "JBuffer.hpp"
#include "JStr.hpp"
#include "JIcon.hpp"
#include "JDtTime.hpp"
#include <string.h>

// static functions
static void _gpathinfo( const char *name, void *buff, ulong lvl = 1,
                        ulong sz = sizeof( FILESTATUS3))
{
   ulong rc = DosQueryPathInfo( name, lvl, buff, sz);
   dosError( rc, "DosQueryPathInfo");
}

static void _gfileinfo( ulong hf, void *buff, ulong lvl = 1,
                        ulong sz = sizeof( FILESTATUS3))
{
   ulong rc = DosQueryFileInfo( hf, lvl, buff, sz);
   dosError( rc, "DosQueryPathInfo");
}

// Exceptions
JEANotString::JEANotString() : JException( err_badea, JExn::eaMismatch)
{}

// ctor & dtor
JFile::JFile( const char *nme, BOOL failIfNew, BOOL append, BOOL isRead,
              BOOL isWrite, JStream::sharing s)
      : JClientStream( nme, failIfNew, append, isRead, isWrite, s), name( nme)
{}

// file pointer manipulation
JFile &JFile::seek( JFile::seekType how, long count)
{
   ulong where;
   ulong rc = DosSetFilePtr( handle(), count, (ulong) how, &where);
   dosError( rc, "DosSetFilePtr");
   return self;
}

ulong JFile::tell() const
{
   ulong where;
   ulong rc = DosSetFilePtr( handle(), 0, JFile::current, &where);
   dosError( rc, "DosSetFilePtr");
   return where;
}

// file-size getting & setting
ulong JFile::size() const
{
   FILESTATUS3 stats;
   _gfileinfo( handle(), &stats);
   return stats.cbFile;
}

JFile &JFile::setSize( ulong sz)
{
   ulong rc = DosSetFileSize( handle(), sz);
   dosError( rc, "DosSetFileSize");
   return self;
}

ulong JFile::sizeOf( const char *f)
{
   FILESTATUS3 stats;
   _gpathinfo( f, &stats);
   return stats.cbFile;
}

// get full path of a file
JStr JFile::path() const
{
   JStr nme;
   nme.setsize( 260);
   _gpathinfo( name, nme.buffer(), 5, 260);
   return nme;
}

JStr JFile::pathOf( const char *who)
{
   JStr nme;
   nme.setsize( 260);
   _gpathinfo( who, nme.buffer(), 5, 260);
   return nme;
}

// delete functions
void JFile::erase( BOOL force)
{
   ulong rc = DosDelete( name);
   if( rc && force) {
      takeAttribute( JFile::readOnly);
      rc = DosDelete( name);
   }
   dosError( rc, "DosDelete");
}

void JFile::erase( const char *nme)
{
   ulong rc = DosDelete( nme);
   dosError( rc, "DosDelete");
}

// does a file exist?
BOOL JFile::exists( const char *which)
{
   // use finder?
   FILESTATUS3 fileStatus;
   ulong rc = DosQueryPathInfo( which, 1, &fileStatus, sizeof( FILESTATUS3));
   return (rc != 2 && rc != 3);
}

// extended attribute manipulation. EAs - a candidate for the worst documented
// part of the control program. The file class allows manipulation of string
// type eas - .TYPE is specifically implemented - using the packed structure
// below, and a lot of faffing with pointers.

// !! erm, this is really broken, bork bork bork.
// !! the ea-string writing thing works, but .TYPE is meant to be a set of
// !! string attributes; it's just a coincidence that this works...

#pragma pack(1)
struct stringea
{
   ushort type;   // == EAT_ASCII == 0xfffd
   ushort length;
   char   str[ 1];
};
#pragma pack()

JFile &JFile::setEA( const char *type, const char *value)
{
   JBuffer   buff1( 400);     // buffer alloc's memory on the heap
   JBuffer   buff2( 400);     // use daft buffers 'cos of exceptions
   FEA2LIST *fea2 = (FEA2LIST *) buff1.pvAddr();
   stringea *pString = ( stringea *) buff2.pvAddr();
   EAOP2     eaop2 = { 0, fea2, 0 };

   // set up the ea value structure
   long cbEAvalue = strlen( value) + 4;
   pString->type = EAT_ASCII; // 0xfffd
   pString->length = cbEAvalue - 4;
   strcpy( pString->str, value);

   // we're going to set the ea 'type' to 'value'
   long cbName = strlen( type);
   fea2->list[ 0].oNextEntryOffset = 0;
   fea2->list[ 0].fEA              = 0;
   fea2->list[ 0].cbName           = cbName;
   fea2->list[ 0].cbValue          = cbEAvalue;

   strcpy( fea2->list[ 0].szName, type);
   memcpy( fea2->list[ 0].szName + cbName + 1, pString, cbEAvalue);

   fea2->cbList = sizeof( FEA2LIST) + cbName + cbEAvalue;

   ulong rc = DosSetFileInfo( handle(), 2, &eaop2, sizeof( EAOP2));
   dosError( rc, "DosSetFileInfo");

   return self;
}

// get the value of a single ea
JStr JFile::EAValue( const char *type) const
{
   JBuffer   buff1( 300);
   JBuffer   buff2( 300);

   GEA2LIST *gea2 = (GEA2LIST *) buff1.pvAddr();
   FEA2LIST *fea2 = (FEA2LIST *) buff2.pvAddr();
   EAOP2     eaop = { gea2, fea2, 0 };

   // set up gea2list (what we want from the ea-set)
   ulong cbName = strlen( type);
   gea2->list[ 0].oNextEntryOffset = 0;
   gea2->list[ 0].cbName           = cbName;
   strcpy( gea2->list[ 0].szName, type);
   gea2->cbList = sizeof( GEA2) + cbName;

   // set up fea2 list (buffer where cp puts the eas)
   fea2->cbList = 300;

   // call api
   ulong rc = DosQueryFileInfo( handle(), 3, &eaop, sizeof( EAOP2));
   dosError( rc, "DosQueryFileInfo");
   if( rc) return JStr();

   // set up string
   JStr      ret;
   stringea *ea = (stringea *) ( ((char *) &(fea2->list[ 0].szName)) + 1 +
                                  fea2->list[ 0].cbName);
   if( ea->type != EAT_ASCII)
      jlib_throw( new JEANotString);
   else {
      ret.setsize( ea->length + 1);
      strncpy( ret.buffer(), ea->str, ea->length);
      ret[ ea->length] = 0;
   }

   return ret;
}

JFile &JFile::setType( const char *type)
{
   return setEA( ".TYPE", type);
}

JStr JFile::fileType() const
{
   return EAValue( ".TYPE");
}

// icons... (statics)
JIcon JFile::getIcon( const char *fname, BOOL s)
{
   ulong rc = WinLoadFileIcon( fname, s);
   if( !rc) pmError( 3, "WinLoadFileIcon");
   return JIcon( rc);
}

void JFile::setIcon( const char *fname, const JResID &id, const JModule &mod)
{
   ICONINFO info = { sizeof( ICONINFO), ICON_RESOURCE, 0,
                     mod.handle(), id.value(), 0, 0 };
   BOOL rc = WinSetFileIcon( fname, &info);
   if( !rc) pmError( 2, "WinSetFileIcon");
}

// attributes
ulong JFile::attributes() const
{
   FILESTATUS3 fileStatus;
   _gfileinfo( handle(), &fileStatus);
   return fileStatus.attrFile;
}

BOOL JFile::hasAttribute( JFile::attribute a) const
{
   return attributes() & (ulong) a;
}

JFile &JFile::setAttributes( ulong attrs)
{
   FILESTATUS3 fS;
   _gfileinfo( handle(), &fS);
   memset( &fS, 0, 3 * ( sizeof( FDATE) + sizeof( FTIME)));
   fS.attrFile = attrs;
   ulong rc = DosSetFileInfo( handle(), 1, &fS, sizeof( FILESTATUS3));
   dosError( rc, "DosSetFileInfo");
   return self;
}

JFile &JFile::giveAttribute( JFile::attribute a)
{
   ulong attr = attributes();
   attr |= (ulong) a;
   return setAttributes( attr);
}

JFile &JFile::takeAttribute( JFile::attribute a)
{
   ulong attr = attributes();
   attr &= ~((ulong) a);
   return setAttributes( attr);
}

BOOL JFile::hasAttribute( const char *file, JFile::attribute a)
{
   FILESTATUS3 fs;
   _gpathinfo( file, &fs);
   return fs.attrFile & (ulong) a;
}

// date & time
JDateTime JFile::whenCreated() const
{
   FILESTATUS3 fs;
   _gfileinfo( handle(), &fs);
   return JDateTime( fs.ftimeCreation, fs.fdateCreation);
}

JDateTime JFile::whenLastAccessed() const
{
   FILESTATUS3 fs;
   _gfileinfo( handle(), &fs);
   return JDateTime( fs.ftimeLastAccess, fs.fdateLastAccess);
}

JDateTime JFile::whenLastWritten() const
{
   FILESTATUS3 fs;
   _gfileinfo( handle(), &fs);
   return JDateTime( fs.ftimeLastWrite, fs.fdateLastWrite);
}
