/*
 * JBNPipe.cpp
 *
 * Base named-pipe stream classes
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "Jos2.h"
#include "JPM.hpp"
#include "JBNPipe.hpp"
#include "JStr.hpp"

// static functions to get & set states
static ulong _gstate( ulong hPipe)
{
   ulong state;
   ulong rc = DosQueryNPHState( hPipe, &state);
   dosError( rc, "DosQueryNPHState");
   return state;
}

static void _sstate( ulong hPipe, ulong state)
{
   ulong rc = DosSetNPHState( hPipe, state);
   dosError( rc, "DosSetNPHState");
}

// base named pipe class -----------------------------------------------------
BOOL JNamedPipe::isBlocking() const
{
   return !(_gstate( handle()) & NP_NOWAIT);
}

JNamedPipe &JNamedPipe::setBlock( BOOL set)
{
   ulong state = 0;
   if( isMsgRead()) state = NP_READMODE_MESSAGE;
   if( !set) state |= NP_NOWAIT;
   _sstate( handle(), state);
   return self;
}

BOOL JNamedPipe::isMsgPipe() const
{
   return _gstate( handle()) & NP_TYPE_MESSAGE;
}

BOOL JNamedPipe::isMsgRead() const
{
   return _gstate( handle()) & NP_READMODE_MESSAGE;
}

JNamedPipe &JNamedPipe::setMsgRead( BOOL set)
{
   ulong state = 0;
   if( !isBlocking()) state = NP_NOWAIT;
   if( set) state |= NP_READMODE_MESSAGE;
   _sstate( handle(), state);
   return self;
}

JNamedPipe::pstate JNamedPipe::state() const
{
   ulong d1, d2, state;
   ulong rc = DosPeekNPipe( handle(), NULL, 0, &d1, (PAVAILDATA) &d2, &state);
   dosError( rc, "DosPeekNPipe");
   return (JNamedPipe::pstate) state;
}

ulong JNamedPipe::peek( JVBuffer &buff) const
{
   ulong state, actual, availdata;
   ulong rc = DosPeekNPipe( handle(), buff.pvAddr(), buff.cbSize(), &actual,
                            (PAVAILDATA) &availdata, &state);
   dosError( rc, "DosPeekNPipe");
   return actual;
}

void JNamedPipe::addSemaphore( const JEventSem &s, ulong id) const
{
   ulong rc = DosSetNPipeSem( handle(), (void *) s.handle(), id);
   dosError( rc, "DosSetNPipeSem");
}

ulong JNamedPipe::transact( const JVBuffer &write, JVBuffer &read) const
{
   ulong cbRead;
   ulong rc = DosTransactNPipe( handle(), write.pvAddr(), write.cbSize(),
                                read.pvAddr(), read.cbSize(), &cbRead);
   dosError( rc, "DosTransactNPipe");
   return cbRead;
}

ulong JNamedPipe::call( const char *nme, const JVBuffer &input,
                        JVBuffer &output, ulong wait, const char *servername)
{
   JStr name( "\\");
   // double backslash for unc
   if( servername) name = name + "\\" + servername + "\\PIPE\\" + nme;
   else name = name + "PIPE\\" + nme;

   ulong cbRead;
   ulong rc = DosCallNPipe( name, input.pvAddr(), input.cbSize(),
                            output.pvAddr(), output.cbSize(), &cbRead, wait);
   dosError( rc, "DosCallNPipe");
   return cbRead;
}

// pipeserver object; this creates & manages the server end ------------------
JPipeServer::JPipeServer( const char *nme, BOOL isRead, BOOL isWrite,
                          BOOL isBytePipe, uchar count, BOOL block,
                          BOOL openAsByte, ulong inBuffer, ulong outBuffer,
                          ulong wait) : JNamedPipe()
{
   assertParms( nme, "JPipeServer::JPipeServer");

   JVStr name( "\\PIPE\\%s", nme);

   ulong openmode = NP_NOWRITEBEHIND;
   if( isRead && isWrite)
      openmode |= NP_ACCESS_DUPLEX;
   else if( isRead)
      openmode |= NP_ACCESS_INBOUND;
   else if( isWrite)
      openmode |= NP_ACCESS_OUTBOUND;

   ulong pipemode = 0;
   if( !block) pipemode |= NP_NOWAIT;
   if( !isBytePipe) pipemode |= NP_TYPE_MESSAGE;
   if( !openAsByte && !isBytePipe) pipemode |= NP_READMODE_MESSAGE;
   if( count == (uchar)-1) pipemode |= 0xff;
   else pipemode |= count;

   ulong rc = DosCreateNPipe( name, &hStream, openmode, pipemode, outBuffer,
                              inBuffer, wait);
   dosError( rc, "DosCreateNPipe");
}

// the handle gets close'd by JStream, which is enough to kill the pipe.
JPipeServer::connection::connection( JPipeServer &sv) : srv( sv)
{
   ulong rc = DosConnectNPipe( srv.handle());
   dosError( rc, "DosConnectNPipe");
}

JPipeServer::connection::~connection()
{
   ulong rc = DosDisConnectNPipe( srv.handle());
   dosError( rc, "DosDisConnectNPipe");
}

// pipeclient object ---------------------------------------------------------
JPipeClient::JPipeClient( const char *nme, const char *srv, ulong timeToWait,
                          BOOL isR, BOOL isW) : JNamedPipe(), JClientStream()
{
   JStr name( "\\");

   if( srv) name = name + "\\" + srv + "\\PIPE\\" + nme;
   else name = name + "PIPE\\" + nme;

#ifdef REALLY_THROW_EXCEPTIONS
   try {
#endif
      open( name, true, true, isR, isW, JStream::denyAll);

#ifdef REALLY_THROW_EXCEPTIONS
   } catch( JDosExcep *d) {
      if( d->error() == ERROR_PIPE_BUSY) {
#else
   JPM *p( JPM::current());
   if( p->type == JExn::dos && ((JDosExcep*) p->exn)->error() == ERROR_PIPE_BUSY) {
      jlib_catch();
#endif
      ulong rc = DosWaitNPipe( name, timeToWait);
      dosError( rc, "DosWaitNPipe");
      if( !rc) open( name, true, isR, isW, JStream::denyAll);
   }
#ifdef REALLY_THROW_EXCEPTIONS
   }
#endif
}
