/*
 * JBitmap.cpp
 *
 * Bitmap class
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#include "Jos2.h"
#include "JBitmap.hpp"
#include "JCoord.hpp"
#include "JWindow.hpp"
#include "JResID.hpp"

#include "JFile.hpp"
#include "JBuffer.hpp"
#include "JPSpace.hpp"
#include "JDC.hpp"
#include "JGLPrim.hpp"
#include "JColour.hpp"
#include <string.h>
#include <math.h>

// Data class & stuff ---------------------------------------------------------
static ulong autodel  = 0x01;
static ulong hasTrans = 0x02;

struct JBitmapData
{
   ulong   hBm;
   ulong   flags;
   JColour transp;

   JBitmapData() : hBm( 0), flags( autodel), transp( 0, 0, 0) {}
  ~JBitmapData() {
      if( (flags & autodel) && (false == GpiDeleteBitmap( hBm)))
            pmError( 1, "GpiDeleteBitmap");
   }
};

// Constructors ---------------------------------------------------------------

// wrap an existing handle
JBitmap::JBitmap( ulong hBmp) : data( new JBitmapData)
{
   assertParms( hBmp, "JBitmap::ctor");
   data->hBm = hBmp;
}

// copy another bitmap object (erm)
JBitmap::JBitmap( const JBitmap &copy) : data( new JBitmapData)
{
   data->hBm    = copy.data->hBm;
   data->transp = copy.data->transp;
   data->flags  = copy.data->flags;
   copy.data->flags &= ~autodel;
}

// load from a resource file
JBitmap::JBitmap( const JResID &id, const JModule &mod, const JSize &s)
        : data( new JBitmapData)
{
   JWindowPS ps( &JWindow::theDesktopWindow);

   data->hBm = GpiLoadBitmap( ps.handle(), mod, id.value(), s.x, s.y);
   if( !data->hBm)
      pmError( 2, "GpiLoadBitmap");
}

// create a new bitmap to a given size & colour depth
JBitmap::JBitmap( const JSize &sz, ulong bpp) : data( new JBitmapData)
{
   BITMAPINFOHEADER2 hdr;

   memset( &hdr, 0, sizeof( BITMAPINFOHEADER2));
   hdr.cbFix = sizeof( BITMAPINFOHEADER2);
   hdr.cx = sz.x;
   hdr.cy = sz.y;
   hdr.cPlanes = 1;
   hdr.cBitCount = bpp;

   JWindowPS deskPS( &JWindow::theDesktopWindow);
   data->hBm = GpiCreateBitmap( deskPS.handle(), &hdr, 0, NULL, NULL);
   if( !data->hBm)
      pmError( 3, "GpiCreateBitmap");
}

JBitmap::~JBitmap()
{
   delete data;
}

// Attributes -----------------------------------------------------------------
ulong JBitmap::handle() const
{ return data->hBm; }

static PBITMAPINFOHEADER hdr( PBITMAPINFOHEADER h, ulong hBm)
{
   h->cbFix = sizeof( BITMAPINFOHEADER);
   BOOL rc = GpiQueryBitmapParameters( hBm, h);
   if( !rc)
      pmError( 1, "GpiQueryBitmapParameters");
   return h;
}

JSize JBitmap::size() const
{
   BITMAPINFOHEADER bmpInfo;
   hdr( &bmpInfo, data->hBm);
   return JSize( bmpInfo.cx, bmpInfo.cy);
}

ulong JBitmap::bpp() const
{
   BITMAPINFOHEADER bmpInfo;
   hdr( &bmpInfo, data->hBm);
   return bmpInfo.cBitCount;
}

// Saving ---------------------------------------------------------------------
#define BIF (*((PBITMAPFILEHEADER2) buffer.pvAddr()))

// !! really need to add compound file support, pointers, icons...

JBitmap &JBitmap::saveAs( const char *filename)
{
   // Get some gen on the bmp
   BITMAPINFOHEADER info;
   hdr( &info, data->hBm);

   // work out the size of the bmp data, colour table, and whole file
   ulong cbBits = (((info.cBitCount * info.cx) + 31) / 32) * info.cy * 4;
   ulong cbClrTable = (info.cBitCount == 24 || info.cBitCount == 0) ?
                                          0                         :
                               (1 << info.cBitCount) * sizeof( RGB2);
   ulong cbTotal = sizeof( BITMAPFILEHEADER2) + cbClrTable + cbBits;

   // create a single buffer to hold the pieces
   JBuffer buffer( cbTotal);

   // fill in the fields of the fileheader and infoheader needed by the api
   BIF.usType = BFT_BMAP;
   BIF.cbSize = sizeof( BITMAPFILEHEADER2);
   BIF.offBits = sizeof( BITMAPFILEHEADER2) + cbClrTable;
   BIF.bmp2.cbFix = sizeof( BITMAPINFOHEADER2); // 16;
   BIF.bmp2.cx = info.cx;
   BIF.bmp2.cy = info.cy;
   BIF.bmp2.cPlanes = 1;
   BIF.bmp2.cBitCount = info.cBitCount;
   // allegedly these are needed too...
   BIF.bmp2.ulCompression = BCA_UNCOMP;
   BIF.bmp2.usRecording = BRA_BOTTOMUP;
   BIF.bmp2.usRendering = BRH_NOTHALFTONED;
   BIF.bmp2.ulColorEncoding = BCE_RGB;

   // create a memory ps and select the bitmap into it
   JMemoryDC dc;
   JPSpace   ps( dc);
   ps.select( self);

   // get the data
   long rc = GpiQueryBitmapBits( ps.handle(), 0, info.cy,
                                 ((char *) buffer.pvAddr()) + BIF.offBits,
                                 (PBITMAPINFO2) &BIF.bmp2);
   if( rc == GPI_ALTERROR)
      pmError( 3, "GpiQueryBitmapBits");

   // blat it out to a file
   JWFile file( (char *) filename);
   file.write( buffer);
   file.setType( "Bitmap");

   ps.deselectBitmap();

   return self;
}

// Transparency ---------------------------------------------------------------
JColour JBitmap::transparentColour() const
{
   return data->transp;
}

BOOL JBitmap::hasTransparency() const
{
   return data->flags & hasTrans;
}

JBitmap &JBitmap::setTransparentColour( const JColour &c)
{
   data->flags |= hasTrans;
   data->transp = c;
   return self;
}

// Rendering to a ps ----------------------------------------------------------
BOOL JBitmap::renderIn( JBPSpace &ps) const
{
   JSize  sz = size();
   JPoint currPos = ps.position();
   JPoint pts[ 4] = { currPos, currPos + sz, JPoint(), sz };

   JGSettings oldSetts( ps.settings());

   if( hasTransparency()) {
      JGSettings sets;
      sets.setBGMix( JGfx::bgMix::transparent1);
      sets.setBGColour( transparentColour());
      ps.render( sets);
   }

   long rc = GpiWCBitBlt( ps.handle(), data->hBm, 4, (PPOINTL) pts,
                          ROP_SRCCOPY, BBO_IGNORE);
   if( hasTransparency())
      ps.render( oldSetts);

   if( rc == GPI_ERROR)
      pmError( 2, "GpiWCBitBlt");

   return rc == GPI_HITS;
}

// BitmapReader ---------------------------------------------------------------

// Written apart from JLib, so doesn't use things like JRFile etc.
// Could be made to do so; effort probably better spent elsewhere...
//
// Exceptions hacked out very painfully...

class Reader
{
   ulong hFile;
   ulong cbRead;

 public:
   Reader( const char *fname) : hFile( 0) {
      DosOpen( fname, &hFile, &cbRead, 0, 0, OPEN_ACTION_OPEN_IF_EXISTS,
               OPEN_SHARE_DENYNONE | OPEN_ACCESS_READONLY, 0);
      cbRead = 0;
   }
  ~Reader() {
     DosClose( hFile);
   }

   BOOL isok() { return !!hFile; }

   ulong read( ulong cb, PVOID buff)
   {
      ulong dummy = 0, rc;
      rc = DosRead( hFile, buff, cb, &dummy);
      if( dummy != cb) rc = 1;// printf( "Yowzers - not enough file!");
      cbRead += cb;
      return rc;
   }

   ulong ensure( ulong offset) {
      ulong rc = 0;
      if( offset < cbRead) {
         rc = 1;// printf( "Yargh - reverse seek request encountered!\n");
      } else if( offset > cbRead) {
         char *buffer = new char [ offset - cbRead];
         rc = read( offset - cbRead, buffer);
         delete [] buffer;
      }
      return rc;
   }
};

#pragma pack(1)
struct ARF // arrayfileheader
{
   ushort usType;
   ulong  cbSize;
   ulong  offNext;
   ushort cxDisplay;
   ushort cyDisplay;
};

struct BFH // bmpfh
{
   short  xHotspot;
   short  yHotspot;
   ulong  offbits;
   ulong  cbInfo2;
};
#pragma pack()

// argh - it's another bitmap structure...
class JBmpInfo
{
   char  *pelData;
   char  *bmpInfo;

   ushort compression;
   ulong  cbImage;

   BFH    bfh;

 public:
   ushort bpp;
   ulong  cx, cy;

   JBmpInfo( Reader &file, BITMAPINFOHEADER2 &, BFH &);
   JBmpInfo( Reader &file, BITMAPFILEHEADER &);

   ulong readColourTable( Reader &file, ushort cbCTEntry, ulong cCTEntry,
                         void *hdr, ulong cbHdr);
   ulong readPelData( Reader &file);
   ulong createBitmap( JWindowPS &ps)
   {
      ulong hbm = GpiCreateBitmap( ps.handle(),
                                   (PBITMAPINFOHEADER2) bmpInfo,
                                   CBM_INIT, pelData,
                                   (PBITMAPINFO2) bmpInfo);
      delete [] pelData; delete [] bmpInfo;
      return hbm;
   }

   void report() {
      //printf( "Found image, %d bpp, %d x %d\n", bpp, cx, cy);
      //printf( "Compression = %d, cbImage = %d\n", compression, cbImage);
   }
};

JBmpInfo::JBmpInfo( Reader &file, BITMAPINFOHEADER2 &iH2, BFH &bfhd)
         : compression( iH2.ulCompression), cbImage( iH2.cbImage),
           bfh( bfhd), bpp( iH2.cBitCount), cx( iH2.cx), cy( iH2.cy)
{
   ulong cColours = 0;

   if( iH2.cbFix >= 36 && iH2.cclrUsed) cColours = iH2.cclrUsed;
   else cColours = ( bpp == 24 ? 0 : (1 << bpp));

   readColourTable( file, sizeof( RGB2), cColours, &iH2, iH2.cbFix);

   report();
}

JBmpInfo::JBmpInfo( Reader &file, BITMAPFILEHEADER &fH)
         : compression( 0), cbImage( 0),
           bpp( fH.bmp.cBitCount), cx( fH.bmp.cx), cy( fH.bmp.cy)
{
   bfh.xHotspot = fH.xHotspot;
   bfh.yHotspot = fH.yHotspot;
   bfh.offbits  = fH.offBits;

   readColourTable( file, sizeof( RGB),
                    bpp == 24 ? 0 : (1 << bpp), &fH.bmp, sizeof fH.bmp);
   report();
}

// build bmpinfo = header plus colour table from stream
ulong JBmpInfo::readColourTable( Reader &file, ushort cbCTEntry,
                                 ulong cCTEntry, void *hdr, ulong cbHdr)
{
   ulong cbColourTable = cbCTEntry * cCTEntry;
   bmpInfo = new char [ cbHdr + cbColourTable];
   memcpy( bmpInfo, hdr, cbHdr);
   return file.read( cbColourTable, bmpInfo + cbHdr);
}

// read pel data from stream
ulong JBmpInfo::readPelData( Reader &file)
{
   ulong bytesToRead;

   if( compression)
      bytesToRead = cbImage;
   else {
      ulong bitspl = bpp * cx;
      ulong dwordspl = bitspl / 32 + ((bitspl % 32) ? 1 : 0);
      bytesToRead = dwordspl * 4 * cy;
   }
   pelData = new char [ bytesToRead];

   file.ensure( bfh.offbits);
   return file.read( bytesToRead, pelData);
}

typedef JSequence<JBmpInfo> JBmpInfos;

static ulong readFileHeader( Reader &file, JBmpInfos &infos, ushort type = 0);
static ulong readBlock( Reader &file, JBmpInfos &infos);

JBitmapReader &JBitmapReader::read( const char *filename)
{
   empty();

   JBmpInfos infos;

   Reader file( filename);

   JWindowPS ps( &JWindow::theDesktopWindow);

   if( file.isok()) {
      // read a block - hopefully the entire set of fileheaders
      ulong rc;

      rc = readBlock( file, infos);

      if( !rc) {
         // go through headers and create bitmaps
         JSequence<JBmpInfo>::cursor c( infos);

         for_cursor( c) {
            rc = c.current()->readPelData( file);
            if( rc) break; // error

            ulong hBmp = c.current()->createBitmap( ps);
            if( !hBmp) pmError( 1, "GpiCreateBitmap"); // but keep going...
            else append( new JBitmap( hBmp));
         }
      }

//    printf( "Ought to be at the end of the file...\n");
//    char a; file.read( 1, &a);
   }

   return self;
}

// hmm...
static ulong override;

ulong readBlock( Reader &file, JBmpInfos &infos)
{
   ushort fType;
   ulong  rc;
   rc = file.read( 2, &fType);
   switch( fType) {
      case BFT_BMAP:
      case BFT_ICON:
      case BFT_POINTER:
      case BFT_BITMAPARRAY:
      case BFT_COLORICON:
      case BFT_COLORPOINTER:
         switch( fType) {
            case BFT_BITMAPARRAY:
            {
               BOOL first = TRUE;

               ARF arf;

               for( ;;) {
                  if( first) {
                     file.read( sizeof arf - 2, &arf.cbSize);
                     first = FALSE;
                  } else
                     file.read( sizeof arf, &arf);

                  if( arf.cbSize == sizeof( BITMAPARRAYFILEHEADER2))
                     override = 2;
                  else if( arf.cbSize == sizeof( BITMAPARRAYFILEHEADER))
                     override = 1;

                  rc = readBlock( file, infos);

                  if( arf.offNext) file.ensure( arf.offNext);
                  else break;
               }
               break;
            }
            case BFT_BMAP:
            case BFT_ICON:
            case BFT_POINTER:
               rc = readFileHeader( file, infos, fType);
               break;
            case BFT_COLORICON:
            case BFT_COLORPOINTER:
               rc = readFileHeader( file, infos, fType);
               rc |= readFileHeader( file, infos);
               break;
         }
         break;
      default:
         rc = 1; // !! throw weird filetype
   }
   return rc;
}

// add a JBmpInfo to the collection from the stream. Position is /after/
// the `usType' field if `type' is set.
// return 0 on success, !0 on error
ulong readFileHeader( Reader &file, JBmpInfos &infos, ushort type)
{
   ulong cbHeader, rc;

   if( !type) file.read( 2, &type);
   rc = file.read( 4, &cbHeader);

   if( !rc) {
      switch( override ? override : cbHeader) {
         case 1:
         case sizeof( BITMAPFILEHEADER):
         {
            BITMAPFILEHEADER fileHeader;

            rc = file.read( cbHeader - 6, &fileHeader.xHotspot);
            infos.append( JBmpInfo( file, fileHeader));
            break;
         }
         default:
//          printf( "Guessing at file format...\n");
            // MS DIBs appear to use a truncated BITMAPINFOHEADER2-type thing.
         case 2:
         case sizeof( BITMAPFILEHEADER2):
         {
            // read past fileheader
            BFH bfh;
            file.read( sizeof bfh, &bfh);

            // get bitmapinfo2 - variable length misery.
            BITMAPINFOHEADER2 infoHeader2;
            rc = file.read( bfh.cbInfo2 - 4, &infoHeader2.cx);
            infoHeader2.cbFix = bfh.cbInfo2;

            infos.append( JBmpInfo( file, infoHeader2, bfh));
            break;
         }
      }
   }
   return rc;
}
