/*
 * JCnr.hpp
 *
 * Container control
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JCnr.hpp"
#include "JCnrRec.hpp"
#include "JWindow.hpp"
#include "JCtlH.hpp"
#include "JCnrH.hpp"
#include "JMenu.hpp"
#include <string.h>
#include "JDrag.hpp"
#include "JDrgItem.hpp"
#include "JDragSet.hpp"
#include "JIntern.hpp"

// handy macros --------------------------------------------------------------
#define cnrError( s) jlib_throw( new JException( err_cnr, JExn::base, s))

#define rcFromObj( s) ( (RECORDCORE *) (((char *)s) - sizeof( MINIRECORDCORE)))
#define objFromRc( s) ( (JCnrRecord *) (((char *)s) + sizeof( MINIRECORDCORE)))

// style ---------------------------------------------------------------------
const unsigned long JContainer::readOnly     = CCS_READONLY;
const unsigned long JContainer::autoPosition = CCS_AUTOPOSITION;
const unsigned long JContainer::normal       = 0;

static const ushort nodrags     = 0x01;
static const ushort nodrops     = 0x02;
static const ushort nolazydrags = 0x04;

// private functions for setting & getting cnrInfo ---------------------------
void JContainer::setInfo( long l, void *cnrInfo) const
{
   BOOL rc = sendEvent( CM_SETCNRINFO, cnrInfo, l);
   if( !rc)
      cnrError( "set the cnrinfo");
}

void JContainer::getInfo( void *cnrInfo) const
{
   BOOL rc = sendEvent( CM_QUERYCNRINFO, cnrInfo, sizeof( CNRINFO));
   if( !rc)
      cnrError( "get the cnrinfo");
}

// Ctors ---------------------------------------------------------------------
JContainer::JContainer( ulong hwnd) : JControl( hwnd)
{ setup(); }

JContainer::JContainer( JWindow *w, ulong id) : JControl( w, id)
{ setup(); }

JContainer::JContainer( JWindow *parent, const JPoint &pos, const JSize &size,
                        ulong ID, ulong style, JCnr::selType sel) : JControl()
{
   assertParms( parent, "JCnr::JCnr");
   style |= ( CCS_MINIRECORDCORE | CCS_MINIICONS | (ulong) sel);

   JCreateWBlock b( parent->handle(), WC_CONTAINER, 0, JWindow::visible | style,
                    pos, size, parent->handle(), HWND_TOP, ID, 0, 0);

   setHwnd( JInternal::window::create( &b));

   setup();
}

static JCnrHandler _JLCnrHandler;
static JCnrKeyHandler _JLCnrKeyHandler;

void JContainer::setup()
{
   setView( JCnr::icon);
   showTreeLines();
   attach( &_JLCnrHandler);
   attach( &_JLCnrKeyHandler);
   menuRec = menuCnr = 0;
   flStatus = nolazydrags; // !! lazy drags go badly wrong atm...
   currentOp = 0;
}

// Dtor ----------------------------------------------------------------------
// Things are cleaned up by the cnrhandler when we get a wm_destroy

// view - I churlishly fail to track the detailstitles fudge -----------------
JContainer &JContainer::setView( JCnr::view_type v)
{
   CNRINFO cnrInfo;
   // preserve other flags
   getInfo( &cnrInfo);
   // mask out present view
   cnrInfo.flWindowAttr &= 0xffffff00;
   // add in requested view
   cnrInfo.flWindowAttr |= (long) v;
   setInfo( CMA_FLWINDOWATTR, &cnrInfo);
   return self;
}

JCnr::view_type JContainer::view() const
{
   CNRINFO         cnrInfo;
   JCnr::view_type rc;

   getInfo( &cnrInfo);

   rc = (JCnr::view_type) (cnrInfo.flWindowAttr & 0xff);
   if( cnrInfo.flWindowAttr & 0x8000 && rc == JCnr::details)
      rc = JCnr::detailsTitles;

   return rc;
}

// title - only used when the container isn't the only window in the frame ---
JContainer &JContainer::addTitle( const char *text, JCnr::title_align a,
                                  BOOL sep, BOOL readonly)
{
   assertParms( text, "JContainer::addTitle");
   CNRINFO cnrInfo;

   // make sure we preserve current flWindowAttr...
   getInfo( &cnrInfo);
   cnrInfo.flWindowAttr |= ( CA_CONTAINERTITLE | (ulong) a |
                            ( sep ? CA_TITLESEPARATOR : 0) |
                            ( readonly ? CA_TITLEREADONLY : 0));

   // when do we free() this, then?
   // j- how 'bout ~Cnr...
   cnrInfo.pszCnrTitle = strdup( text);
   setInfo( CMA_FLWINDOWATTR | CMA_CNRTITLE, &cnrInfo);

   return self;
}

JContainer &JContainer::setTitle( const char *text)
{
   assertParms( text, "JContainer::setTitle");

   // check there is a title. If not, make one.
   CNRINFO cnrInfo;
   getInfo( &cnrInfo);
   if( !(cnrInfo.flWindowAttr & CA_CONTAINERTITLE)) addTitle( text);
   else {
      cnrInfo.pszCnrTitle = strdup( text);
      setInfo( CMA_CNRTITLE, &cnrInfo);
   }
   return self;
}

JStr JContainer::title() const
{
   CNRINFO cnrInfo;
   getInfo( &cnrInfo);
   return JStr( cnrInfo.pszCnrTitle);
}

JContainer &JContainer::editTitle()
{
   CNREDITDATA data = { sizeof( CNREDITDATA), handle(), NULL, NULL, NULL,
                        CID_CNRTITLEWND };
   BOOL rc = sendEvent( CM_OPENEDIT, &data);
   if( !rc)
      cnrError( "open the cnr title mle");
   return self;
}

// turning ownerdraw on & off ------------------------------------------------
JContainer &JContainer::setCustomBackground( BOOL set)
{
   CNRINFO cnrInfo;
   getInfo( &cnrInfo);
   if( set) cnrInfo.flWindowAttr |= CA_OWNERPAINTBACKGROUND;
   else cnrInfo.flWindowAttr &= ~CA_OWNERPAINTBACKGROUND;
   setInfo( CMA_FLWINDOWATTR, &cnrInfo);
   return self;
}

JContainer &JContainer::setCustomRecord( BOOL set)
{
   CNRINFO cnrInfo;
   getInfo( &cnrInfo);
   if( set) cnrInfo.flWindowAttr |= CA_OWNERDRAW;
   else cnrInfo.flWindowAttr &= ~CA_OWNERDRAW;
   setInfo( CMA_FLWINDOWATTR, &cnrInfo);
   return self;
}

// showing/hiding cnr record text --------------------------------------------
JContainer &JContainer::showText( BOOL show)
{
   BOOL rc = sendEvent( CM_SETTEXTVISIBILITY, show);
   if( !rc)
      cnrError( "set the text visibility");
   return self;
}

JContainer &JContainer::hideText()
{ return showText( false); }

// scroll container ----------------------------------------------------------
JContainer &JContainer::scroll( JCnr::scroll_dir d, long ds)
{
   BOOL rc = sendEvent( CM_SCROLLWINDOW, (ushort)d, ds);
   if( !rc)
      cnrError( "scroll the window");
   return self;
}

// set source-menu emphasis to container -------------------------------------
JContainer &JContainer::setSource( BOOL on)
{
   BOOL rc = sendEvent( CM_SETRECORDEMPHASIS, JMP(), JMP( on, CRA_SOURCE));
   if( !rc)
      cnrError( "set the cnr source emphasis");
   return self;
}

// Tree-view settings --------------------------------------------------------
JContainer &JContainer::showTreeLines( BOOL show)
{
   CNRINFO info;
   getInfo( &info);
   if( show) info.flWindowAttr |= CA_TREELINE;
   else      info.flWindowAttr &= ~CA_TREELINE;
   setInfo( CMA_FLWINDOWATTR, &info);
   return self;
}

JContainer &JContainer::setTreeIndent( ulong howfar)
{
   CNRINFO info;
   info.cxTreeIndent = howfar;
   setInfo( CMA_CXTREELINE, &info);
   return self;
}

JContainer &JContainer::setTreeLinesWidth( ulong pels)
{
   CNRINFO info;
   info.cxTreeIndent = pels;
   setInfo( CMA_CXTREEINDENT, &info);
   return self;
}

// Miscellaneous record methods ----------------------------------------------

JCnrRecord *JContainer::initialise( JCnrRecord *rec) const
{
   rec->cnr = (JContainer *)this;
   return rec;
}

ulong JContainer::records() const
{
   CNRINFO cnrInfo;
   getInfo( &cnrInfo);
   return cnrInfo.cRecords;
}

JContainer &JContainer::arrange()
{
   BOOL rc = sendEvent( CM_ARRANGE);
   if( !rc)
      cnrError( "arrange the cnr");
   return self;
}

// Inserting records ---------------------------------------------------------

JContainer &JContainer::insert( JCnrRecord *rec, JCnr::rec_insert i)
{
   RECORDINSERT inserter = { sizeof( RECORDINSERT), (PRECORDCORE) i,
                             NULL, true, CMA_TOP, 1 };
   ulong rc = sendEvent( CM_INSERTRECORD, rcFromObj( rec), &inserter);
   if( !rc)
      pmError( 1, "cnr::insert");
   return self;
}

JContainer &JContainer::insert( JCnrRecord *rec, JCnrRecord *after)
{
   RECORDINSERT inserter = { sizeof( RECORDINSERT), rcFromObj( after),
                             NULL, true, CMA_TOP, 1 };
   ulong rc = sendEvent( CM_INSERTRECORD, rcFromObj( rec), &inserter);
   if( !rc)
      cnrError( "insert a rec");
   return self;
}

JContainer &JContainer::insertAsChild( JCnrRecord *rec, JCnrRecord *mom)
{
   RECORDINSERT inserter = { sizeof( RECORDINSERT), (PRECORDCORE) CMA_FIRST,
                             rcFromObj( mom), true, CMA_TOP, 1 };
   ulong rc = sendEvent( CM_INSERTRECORD, rcFromObj( rec), &inserter);
   if( !rc)
      cnrError( "insert a rec");
   return self;
}

JContainer &JContainer::insert( JCnrRecordList &list, JCnr::rec_insert r)
{
   ulong        count = list.elements(), i;
   if( count) {
      RECORDINSERT inserter = { sizeof( RECORDINSERT), (PRECORDCORE) r,
                                NULL, true, CMA_TOP, count };

      PRECORDCORE *recs = (PRECORDCORE *) malloc( count * sizeof( PRECORDCORE));

      JCnrRecordList::cursor c( list);

      for( c.toFirst(), i = 0; c.isValid(); i++, c.toNext())
         recs[ i] = rcFromObj( *c.current());

      ulong rc = sendEvent( CM_INSERTRECORDARRAY, recs, &inserter);
      if( !rc) {
         JPMExcep e("");
         cnrError( JVStr( "rc = %d", e.error()));
      }
   }

   return self;
}

JContainer &JContainer::insert( JCnrRecordList &list, JCnrRecord *after)
{
   ulong        count = list.elements(), i;
   if( count) {
      RECORDINSERT inserter = { sizeof( RECORDINSERT), rcFromObj( after),
                                NULL, true, CMA_TOP, count };

      PRECORDCORE *recs = (PRECORDCORE *) malloc( count * sizeof( PRECORDCORE));

      JCnrRecordList::cursor c( list);

      for( c.toFirst(), i = 0; c.isValid(); i++, c.toNext())
         recs[ i] = rcFromObj( *c.current());

      ulong rc = sendEvent( CM_INSERTRECORDARRAY, recs, &inserter);
      if( !rc) {
         JPMExcep e("");
         cnrError( JVStr( "rc = %d", e.error()));
      }
   }

   return self;
}

JContainer &JContainer::insertAsChildren( JCnrRecordList &list, JCnrRecord *mom)
{
   ulong        count = list.elements(), i;
   RECORDINSERT inserter = { sizeof( RECORDINSERT), (PRECORDCORE) CMA_FIRST,
                             rcFromObj( mom), true, CMA_TOP, count };

   PRECORDCORE *recs = (PRECORDCORE *) malloc( count * sizeof( PRECORDCORE));

   JCnrRecordList::cursor c( list);

   for( c.toFirst(), i = 0; c.isValid(); i++, c.toNext())
      recs[ i] = rcFromObj( *c.current());

   ulong rc = sendEvent( CM_INSERTRECORDARRAY, recs, &inserter);
   if( !rc) {
      JPMExcep e("");
      cnrError( JVStr( "rc = %d", e.error()));
   }

   return self;
}

// Removing records ----------------------------------------------------------
// mechanics for this is in the record object.
JContainer &JContainer::remove( JCnrRecordList &recs, BOOL freeRecs)
{
   JCnrRecordList::cursor c( recs);

   for_cursor( c)
      (*c.current())->remove( freeRecs);

   return self;
}

JContainer &JContainer::empty( BOOL freeEm)
{
   JCnrRecordList list;
   allRecords( list);
   return remove( list, freeEm);
}

// Querying records with certain emphases ------------------------------------
JCnrRecord *JContainer::nextRecord( JCnr::emphasis e, JCnrRecord *after) const
{
   void *pRC = sendEvent( CM_QUERYRECORDEMPHASIS,
                          after ? rcFromObj( after) : (PRECORDCORE) CMA_FIRST,
                          (ushort) e);
   if( pRC) {
      JCnrRecord *rec = objFromRc( pRC);
      initialise( rec);
      return rec;
   } else
      return NULL;
}

// selected
JCnrRecordList &JContainer::selectedRecords( JCnrRecordList &list) const
{
   JCnrRecord *rec = 0;

   while( ( rec = nextRecord( JCnr::selected, rec)) != 0)
      list.append( rec);

   return list;
}

// sourced
JCnrRecordList &JContainer::sourcedRecords( JCnrRecordList &list) const
{
   JCnrRecord *rec = 0;

   while( ( rec = nextRecord( JCnr::sourced, rec)) != 0)
      list.append( rec);

   return list;
}

// picked
JCnrRecordList &JContainer::pickedRecords( JCnrRecordList &list) const
{
   JCnrRecord *rec = 0;

   while( ( rec = nextRecord( JCnr::picked, rec)) != 0)
      list.append( rec);

   return list;
}

// open
JCnrRecordList &JContainer::openRecords( JCnrRecordList &list) const
{
   JCnrRecord *rec = 0;

   while( ( rec = nextRecord( JCnr::open, rec)) != 0)
      list.append( rec);

   return list;
}

// filtered
JCnrRecordList &JContainer::filteredRecords( JCnrRecordList &list) const
{
   JCnrRecord *rec = 0;

   while( ( rec = nextRecord( JCnr::filtered, rec)) != 0)
      list.append( rec);

   return list;
}

// container record iterator things ------------------------------------------

JCnrRecord *JContainer::firstRecord() const
{
   JCnrRecord *rc = 0;

   void *v = sendEvent( CM_QUERYRECORD, JMP(), JMP( CMA_FIRST, CMA_ITEMORDER));

   if( (long) v == -1 )
      cnrError( "query a record");
   else if( v) rc = initialise( objFromRc( v));

   return rc;
}

JCnrRecord *JContainer::lastRecord() const
{
   JCnrRecord *rc = 0;

   void *v = sendEvent( CM_QUERYRECORD, JMP(), JMP( CMA_LAST, CMA_ITEMORDER));

   if( (long) v == -1 )
      cnrError( "query a record");
   else if( v) rc = initialise( objFromRc( v));

   return rc;
}

JContainer &JContainer::runIterator( JCnrIterator &iterator, JCnrRecord *top)
{
   JCnrRecord *first = top ? top : firstRecord();

   if( first)
      iterate( first, iterator);
   return self;
}

BOOL JContainer::iterate( JCnrRecord *rec, JCnrIterator &func)
{
   JCnrRecord *next = rec->firstChild();

   if( next)
      // has children
      if( iterate( next, func))
         return true;

   next = rec->next();
   if( next)
      // has siblings
      if( iterate( next, func))
         return true;

   // do the item here
   return func( rec);
}

// querying all records; uses an iterator ------------------------------------
class JCnrRecIterator : public JCnrIterator
{
   JCnrRecordList &l;
 public:
   JCnrRecIterator( JCnrRecordList &list) : l(list) {}
   BOOL operator () ( JCnrRecord *rec)
   {
      // called once for each rec in the cnr.
      l.append( rec);
      return false;
   }
};

JCnrRecordList &JContainer::allRecords( JCnrRecordList &list)
{
   JCnrRecIterator recIterator( list);

   runIterator( recIterator);

   return list;
}

// Filters -------------------------------------------------------------------
struct cnrfilterdata
{
   JCnrFilter &filter;
   JContainer *cnr;
   cnrfilterdata( JCnrFilter &f, JContainer *c) : filter( f), cnr( c) {}
};

static BOOL _System fnFilter( RECORDCORE *rc, void *data);

JContainer &JContainer::filter( JCnrFilter &filter)
{
   cnrfilterdata data( filter, this);
   BOOL rc = sendEvent( CM_FILTER, JMP((void*)&fnFilter), &data);
   if( !rc)
      cnrError( "do a filter");
   return self;
}

static BOOL _System fnFilter( RECORDCORE *rc, void *d)
{
   cnrfilterdata *data = (cnrfilterdata *) d;
   JCnrRecord    *rec = objFromRc( rc);

   data->cnr->initialise( rec);
   return data->filter( rec);
}

JContainer &JContainer::unfilter()
{
   BOOL rc = sendEvent( CM_FILTER);
   if( !rc)
      cnrError( "do a filter");
   return self;
}

// Searching -----------------------------------------------------------------
JCnrRecord *JContainer::matchingRecord( const char *str, JCnrRecord *after,
                                        BOOL start, BOOL ins)
{
   SEARCHSTRING srchStr = { sizeof( SEARCHSTRING), (char*) str, start, !ins, CV_NAME };
   void *v;
   v = sendEvent( CM_SEARCHSTRING, &srchStr, after ? (PRECORDCORE) CMA_FIRST
                                                   : rcFromObj( after));

   JCnrRecord *rec = 0;

   if( (long)v == -1)
      cnrError( "search for a string");
   else if( v) rec = initialise( objFromRc( v));

   return rec;
}

JCnrRecordList &JContainer::matchingRecords( JCnrRecordList &list,
                                             const char *str, BOOL s, BOOL i)
{
   JCnrRecord *rec = 0;
   while( ( rec = matchingRecord( str, rec, s, i)) != 0)
      list.append( rec);
   return list;
}

// Sorting -------------------------------------------------------------------
struct cnrsortdata
{
   JCnrSorter &sorter;
   JContainer *cnr;
   cnrsortdata( JCnrSorter &s, JContainer *c) : sorter( s), cnr( c) {}
};

static int _System fnSort( PRECORDCORE, PRECORDCORE, void *);

JContainer &JContainer::sort( JCnrSorter &s)
{
   cnrsortdata data( s, this);
   BOOL rc = sendEvent( CM_SORTRECORD, JMP((void*)&fnSort), &data);
   if( !rc)
      cnrError( "Do a sort");
   return self;
}

static int _System fnSort( PRECORDCORE arg1, PRECORDCORE arg2, void *d)
{
   cnrsortdata *data = (cnrsortdata *) d;
   JCnrRecord *rec1 = objFromRc( arg1);
   JCnrRecord *rec2 = objFromRc( arg2);

   return data->sorter( data->cnr->initialise( rec1),
                        data->cnr->initialise( rec2) );
}

struct JCnrObjNameSorter : public JCnrSorter
{
   int operator () ( JCnrRecord *r1, JCnrRecord *r2)
   {
      return strcmp( r1->text(), r2->text());
   }
};

JContainer &JContainer::sortByName()
{
   JCnrObjNameSorter sorter;
   return sort( sorter);
}

// Inserting columns ---------------------------------------------------------
JContainer &JContainer::insert( JCnrColumn *col, JCnr::rec_insert r)
{
   assertParms( col, "JCnr::insert");
   FIELDINFOINSERT inserter = { sizeof( FIELDINFOINSERT), (PFIELDINFO) r,
                                true, 1 };
   BOOL rc = sendEvent( CM_INSERTDETAILFIELDINFO, col->data, &inserter);
   if( !rc)
      cnrError( "insert a column");
   return self;
}

JContainer &JContainer::insert( JCnrColumn *col, JCnrColumn *after)
{
   assertParms( col && after, "JCnr::insert");
   FIELDINFOINSERT inserter = { sizeof( FIELDINFOINSERT),
                                (PFIELDINFO) after->data, true, 1 };
   BOOL rc = sendEvent( CM_INSERTDETAILFIELDINFO, col->data, &inserter);
   if( !rc)
      cnrError( "insert a column");
   return self;
}

JContainer &JContainer::insert( JCnrColumnList &l, JCnr::rec_insert r)
{
   FIELDINFOINSERT inserter = { sizeof( FIELDINFOINSERT),
                                (PFIELDINFO) r, false, 1 };
   JCnrColumnList::cursor c( l);
   BOOL                   rc;

   for_cursor( c) {
      rc = sendEvent( CM_INSERTDETAILFIELDINFO, (*c.current())->data, &inserter);
      if( !rc) {
         cnrError( "insert a column");
         break;
      }
   }

   if( !rc) {
      rc = sendEvent( CM_INVALIDATEDETAILFIELDINFO);
      if( !rc)
         cnrError( "invalidate some columns");
   }

   return self;
}

JContainer &JContainer::insert( JCnrColumnList &l, JCnrColumn *after)
{
   FIELDINFOINSERT inserter = { sizeof( FIELDINFOINSERT),
                                (PFIELDINFO) after->data, false, 1 };
   JCnrColumnList::cursor c( l);
   BOOL                   rc;

   for_cursor( c) {
      rc = sendEvent( CM_INSERTDETAILFIELDINFO, (*c.current())->data, &inserter);
      if( !rc) {
         cnrError( "insert a column");
         break;
      }
   }

   if( !rc) {
      rc = sendEvent( CM_INVALIDATEDETAILFIELDINFO);
      if( !rc)
         cnrError( "invalidate some columns");
   }

   return self;
}

// Querying columns ----------------------------------------------------------
JCnrColumn *JContainer::firstColumn() const
{
   JCnrColumn *col = 0;
   void *v = sendEvent( CM_QUERYDETAILFIELDINFO, JMP(), CMA_FIRST);
   if( (long)v == -1)
      cnrError( "query a column");
   else if( v) col = ((JFieldInfo *) v)->me;
   return  col;
}

JCnrColumn *JContainer::lastColumn() const
{
   JCnrColumn *col = 0;

   void *v = sendEvent( CM_QUERYDETAILFIELDINFO, JMP(), CMA_LAST);

   if( (long)v == -1)
      cnrError( "query a column");
   if( v) col = ((JFieldInfo *) v)->me;

   return col;
}

JCnrColumnList &JContainer::allColumns( JCnrColumnList &l) const
{
   JCnrColumn *col = firstColumn();
   if( col) {
      l.append( col);
      while( ( col = col->next()) != 0)
         l.append( col);
   }
   return l;
}

// Scrolling the details view horizontally -----------------------------------
JContainer &JContainer::scrollLeftWindow( long amt)
{
   BOOL rc = sendEvent( CM_HORZSCROLLSPLITWINDOW, CMA_LEFT, amt);
   if( !rc)
      cnrError( "scroll the window");
   return self;
}

JContainer &JContainer::scrollRightWindow( long amt)
{
   BOOL rc = sendEvent( CM_HORZSCROLLSPLITWINDOW, CMA_RIGHT, amt);
   if( !rc)
      cnrError( "scroll the window");
   return self;
}

// Deltas --------------------------------------------------------------------
JContainer &JContainer::setDelta( long d)
{
   CNRINFO info;
   info.cDelta = d;
   setInfo( CMA_DELTA, &info);
   return self;
}

ulong JContainer::delta() const
{
   CNRINFO info;
   getInfo( &info);
   return info.cDelta;
}

// Context menus -------------------------------------------------------------
JContainer &JContainer::setRecordMenu( JMenu *menu)
{ menuRec = menu; return self; }

JContainer &JContainer::setContainerMenu( JMenu *menu)
{ menuCnr = menu; return self; }

// Help manager support ------------------------------------------------------
JContainer &JContainer::setHelpID( const JHelpID &id)
{ hlpID = id; return self; }

// Stopping default dragndrop actions ----------------------------------------
JContainer &JContainer::stopDrags()
{
   flStatus |= nodrags;
   return self;
}

JContainer &JContainer::stopDrops()
{
   flStatus |= nodrops;
   return self;
}

JContainer &JContainer::stopLazyDrags()
{
   flStatus |= nolazydrags;
   return self;
}

// event handling ------------------------------------------------------------
BOOL JContainer::event( const JCtlEvent &e)
{
   switch( e.notify()) {
      // just tell the record object. Nothing special to do.
      case CN_COLLAPSETREE:
      {
         JCnrRecord *rec = objFromRc( (void *) e.data());
         initialise( rec);
         return rec->collapsed();
      }
      case CN_EXPANDTREE:
      {
         JCnrRecord *rec = objFromRc( (void *) e.data());
         initialise( rec);
         return rec->expanded();
      }
      case CN_ENDEDIT:
      {
         PCNREDITDATA pThing = (PCNREDITDATA) e.data();
         if( pThing->pRecord && ( !pThing->pFieldInfo ||
             pThing->pFieldInfo->offStruct == (ulong)JCnrRecord::textOffset)) {
             JCnrRecord *rec = objFromRc( pThing->pRecord);
             initialise( rec);
             return rec->renamed();
         }
         return true;
      }

      // get a reply from the method; if true, open the record
      case CN_ENTER:
      {
         PNOTIFYRECORDENTER pThing = (PNOTIFYRECORDENTER) e.data();
         if( pThing->pRecord) {
            JCnrRecord *rec = objFromRc( pThing->pRecord);
            initialise( rec);
            if( !rec->opened())
               rec->setEmphasis( JCnr::open);
         }
         return true;
      }

      // call a virtual method in cnr
      case CN_KILLFOCUS:
         return lostFocus();
      case CN_SETFOCUS:
         return gainedFocus();
      case CN_SCROLL:
      {
         PNOTIFYSCROLL pThing = (PNOTIFYSCROLL) e.data();
         return scrolled( pThing->lScrollInc);
      }
      case CN_QUERYDELTA:
      {
         PNOTIFYDELTA pThing = (PNOTIFYDELTA) e.data();
         return deltaReached( (JCnr::deltaType) pThing->fDelta);
      }

      case CN_CONTEXTMENU:
      {
         // empty the previously sourced list
         sourced.empty();

         // check whether it's a request for rec or bg menu
         if( e.data() && menuRec) {
            JCnrRecord *rec = objFromRc( (void *)e.data());
            initialise( rec);

            // create a list to put the recs in
            JCnrRecordList l;

            // if menu on selected, source all selected recs,
            // if menu on deselected, just source that one
            if( rec->hasEmphasis( JCnr::selected))
               selectedRecords( l);
            else
               l.append( rec);

            // inform the last record that it's about to get menu'd.
            // This gives it a chance to alter the contents of the menu
            if( !rec->menu( l)) {
               // go thru' the list and source the elements, also adding to
               // the previously sourced list.
               JCnrRecordList::cursor c( l);

               for_cursor( c) {
                  JCnrRecord *rec = *c.current();
                  rec->setEmphasis( JCnr::sourced);
                  sourced.append( rec);
               }

               // show the menu
               menuRec->popUp( *parent());
            }
         } else if( !e.data() && menuCnr) {
            // give the cnr source emphasis & show the menu
            setSource();
            menuCnr->popUp( *parent());
         }
         return true;
      }

      // dragndrop things
      case CN_DRAGOVER:
      {
         if( !( flStatus & nodrops)) {
            PCNRDRAGINFO pThing = (PCNRDRAGINFO) e.data();
            JDragSet     dragset( pThing->pDragInfo);
            JDropOk      isOk;

            // if over record, ask record; else check if we can accept it
            if( pThing->pRecord)
               // !! ought this to be initialised?
               isOk = objFromRc( pThing->pRecord)->isDropOk( dragset);
            else
               isOk = isDropOk( dragset);

            e.setrc( JMR( (ushort) isOk.state, (ushort) isOk.op));
         }
         return true;
      }
      case CN_DROPNOTIFY:
         // this is the same as cn_drop, except the drag is lazy... doh.
      case CN_DROP:
      {
         if( !( flStatus & nodrops)) {
            // first check drop's okay, just to be on the safe side
            PCNRDRAGINFO pThing = (PCNRDRAGINFO) e.data();
            JDragSet     dragset( pThing->pDragInfo, true); // free strhandles
            JDropOk      isOk;

            // if over record, ask record; else check if we can accept it
            if( pThing->pRecord) {
               JCnrRecord *rec = initialise( objFromRc( pThing->pRecord));
               isOk = rec->isDropOk( dragset);
               if( isOk.state == JDrag::yes)
                  rec->drop( dragset);
            } else {
               isOk = isDropOk( dragset);
               if( isOk.state == JDrag::yes)
                  drop( dragset);
            }

            dragset.endConversation();
         }
         return true;
      }
      case CN_INITDRAG:
      {
         if( !( flStatus & nodrags)) {
            PCNRDRAGINIT pThing = (PCNRDRAGINIT) e.data();
            // check we're allowed to drag & a record (not bg)'s being dragged
            // If so: is record selected?
            //        - no: just drag the record
            //        -yes: drag all selected records
            if( pThing->pRecord) {
               JCnrRecordList list;
               JCnrRecord    *rec = initialise( objFromRc( pThing->pRecord));

               if( rec->hasEmphasis( JCnr::selected))
                  selectedRecords( list);
               else list.append( rec);

               JCnrRecordList::cursor c( list);
               JStdDragSet            dragset( this, list.elements());

               for_cursor( c) {
                  JDragItem *d = (*c.current())->initDrag();
                  if( !d) {
                     cnrError( "Get a dragitem");
                     return true; // argh...
                  }
                  d->setOffset( pThing->cx, pThing->cy);
                  d->setItemID( (ulong) *c.current());
                  dragset.addDragItem( *d);
                  delete d;
               }

               dragset.drag();
            }
         }
         return true;
      }
      case CN_PICKUP:
      {
         if( !( flStatus & nolazydrags)) {
            PCNRDRAGINIT pThing = (PCNRDRAGINIT) e.data();
            // check we're allowed to drag & a record (not bg)'s being dragged
            // If so: is record selected?
            //        - no: just drag the record
            //        -yes: drag all selected records
            if( pThing->pRecord) {
               JCnrRecordList list;
               JCnrRecord    *rec = objFromRc( pThing->pRecord);

               if( initialise( rec)->hasEmphasis( JCnr::selected))
                  selectedRecords( list);
               else list.append( rec);

               JCnrRecordList::cursor c( list);
               JLazyDragSet           dragset( this, list.elements());

               for_cursor( c) {
                  JCnrRecord *rec = *c.current();
                  rec->setEmphasis( JCnr::picked);
                  JDragItem *d = rec->initDrag();
                  if( !d) {
                     cnrError( "Get a dragitem");
                     return true;
                  }
                  d->setItemID( (ulong) rec);
                  dragset.addDragItem( *d);
                  delete d;
               }

               dragset.lazyDrag();
            }
         }
         return true;
      }

      // call help manager with id for drop-on-cnr
      case CN_DROPHELP:
      {
         PCNRDRAGINFO pThing = (PCNRDRAGINFO) e.data();
         JDragSet dragset( pThing->pDragInfo, true); // free strhandles
         if( HelpManager.helpModel() != JHelpManager::none) {
            JHelpID i = dropHelp( dragset);
            if( i.ID())
               HelpManager.showHelp( i);
         }
         dragset.endConversation();
         return true;
      }
      // call help manager with id for cnr or object, if they exist
      case CN_HELP:
      {
         if( HelpManager.helpModel() != JHelpManager::none) {
            PMINIRECORDCORE pRecord = (PMINIRECORDCORE) e.data();
            if( pRecord) {
               JCnrRecord *rec = initialise( objFromRc( pRecord));
               JHelpID i = rec->helpID();
               if( i.ID())
                  HelpManager.showHelp( i);
            } else {
               if( hlpID.ID() )
                  HelpManager.showHelp( hlpID);
            }
         }
         return true;
      }
      // just realloc the string
      case CN_REALLOCPSZ:
      {
         PCNREDITDATA pThing = (PCNREDITDATA) e.data();
         *(pThing->ppszText) = (char *) realloc( *(pThing->ppszText), pThing->cbText);
         e.setrc( true);
         return true;
      }
   }
   // stop all other bizarre notifies right here.
   return true;
}
