/*
 * JCnrCol.cpp
 *
 * Container column
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JCnr2.hpp"
#include "JCnrCol.hpp"
#include "JCnr.hpp"
#include <string.h>

#define cnrError( s) jlib_throw( new JException( err_cnr, JExn::base, s))

// style constants -----------------------------------------------------------
const ulong JCnrColumn::readOnly       = CFA_FIREADONLY;
const ulong JCnrColumn::separator      = CFA_SEPARATOR;
const ulong JCnrColumn::justify_bottom = CFA_BOTTOM;
const ulong JCnrColumn::justify_top    = CFA_TOP;
const ulong JCnrColumn::justify_middle = CFA_VCENTER;
const ulong JCnrColumn::align_right    = CFA_RIGHT;
const ulong JCnrColumn::align_left     = CFA_LEFT;
const ulong JCnrColumn::align_center   = CFA_CENTER;
const ulong JCnrColumn::normal         = CFA_FIREADONLY | CFA_SEPARATOR |
                                         CFA_VCENTER | CFA_CENTER;

// Creating ------------------------------------------------------------------
JCnrColumn::JCnrColumn( JContainer *ctr, long offset, JCnr::column_type t,
                        const char *title, ulong style, BOOL titro)
{
   assertParms( ctr, "JCnrColumn::ctor");
   cnr = ctr;
   void *v = cnr->sendEvent( CM_ALLOCDETAILFIELDINFO, 1);
   data = (JFieldInfo *) v;
   if( !data)
      cnrError( "alloc a column");
   else {
      // set up array
      data->length = sizeof( FIELDINFO);
      data->flField = (ulong) t | style | CFA_HORZSEPARATOR;
      data->flTitle = CFA_CENTER | CFA_VCENTER | ( titro ? CFA_FITITLEREADONLY : 0);
      data->title = strdup( title);
      // offset given is inside MyCnrRecord
      data->offset = offset + sizeof( MINIRECORDCORE);
      // put us in the user-data field (thoughtful cnr designers!)
      data->me = this;
   }
}

JCnrColumn::JCnrColumn( const JCnrColumn &copy)
{
   cnr = copy.cnr;
   void *v = cnr->sendEvent( CM_ALLOCDETAILFIELDINFO, 1);
   data = (JFieldInfo *) v;
   if( !data)
      cnrError( "alloc a column");
   else {
      // copy bits across
      memcpy( data, copy.data, sizeof( JFieldInfo));
      // set us
      data->me = this;
   }
}

// Destroying ----------------------------------------------------------------
JCnrColumn::~JCnrColumn()
{
   ::free( data->title);
   BOOL rc = cnr->sendEvent( CM_FREEDETAILFIELDINFO, &data, 1);
   if( !rc)
      cnrError( "free a column");
}

void JCnrColumn::remove( BOOL f)
{
   long rc = cnr->sendEvent( CM_REMOVEDETAILFIELDINFO, &data,
                             JMP( 1, CMA_INVALIDATE));
   if( rc == -1)
      cnrError( "remove a column");
   else if( f) free();
}

void JCnrColumn::free()
{
   delete this;
}

// Cursor-like methods -------------------------------------------------------
JCnrColumn *JCnrColumn::next() const
{
   JCnrColumn *col = 0;

   void *v = cnr->sendEvent( CM_QUERYDETAILFIELDINFO, data, CMA_NEXT);

   JFieldInfo *f = (JFieldInfo *) v;
   if( (long)f == -1)
      cnrError( "query a column");
   else if( f) col = f->me;

   return col;
}

JCnrColumn *JCnrColumn::previous() const
{
   JCnrColumn *col = 0;

   void *v = cnr->sendEvent( CM_QUERYDETAILFIELDINFO, data, CMA_PREV);

   JFieldInfo *f = (JFieldInfo *) v;
   if( (long)f == -1)
      cnrError( "query a column");
   else if( f) col = f->me;

   return col;
}

// Actions on the column title -----------------------------------------------
JCnrColumn &JCnrColumn::editTitle()
{
   if( !( data->flTitle & CFA_FITITLEREADONLY)) {
      CNREDITDATA dat = { sizeof( CNREDITDATA), cnr->handle(), 0,
                          (PFIELDINFO) data, 0, 0, CID_LEFTCOLTITLEWND };
      BOOL rc = cnr->sendEvent( CM_OPENEDIT, &dat);
      if( !rc)
         cnrError( "open an edit");
   }
   return self;
}

JCnrColumn &JCnrColumn::setTitle( const char *txt)
{
   ::free( data->title);
   data->title = strdup( txt);
   BOOL rc = cnr->sendEvent( CM_INVALIDATEDETAILFIELDINFO);
   if( !rc)
      cnrError( "invalidate a column");
   return self;
}

// Visibility of the column; it's still in the container ---------------------
JCnrColumn &JCnrColumn::show( BOOL sho)
{
   if( sho) data->flField &= ~CFA_INVISIBLE;
   else     data->flField |= CFA_INVISIBLE;
   BOOL rc = cnr->sendEvent( CM_INVALIDATEDETAILFIELDINFO);
   if( !rc)
      cnrError( "invalidate a column");
   return self;
}

// Splitbar things -----------------------------------------------------------
JCnrColumn &JCnrColumn::setSplitbar()
{
   CNRINFO info;
   info.pFieldInfoLast = (PFIELDINFO) data;
   BOOL rc = cnr->sendEvent( CM_SETCNRINFO, &info, CMA_PFIELDINFOLAST);
   if( !rc)
      cnrError( "set the info");
   return self;
}

JCnrColumn &JCnrColumn::setObjectColumn()
{
   CNRINFO info;
   info.pFieldInfoObject = (PFIELDINFO) data;
   BOOL rc = cnr->sendEvent( CM_SETCNRINFO, &info, CMA_PFIELDINFOOBJECT);
   if( !rc)
      cnrError( "set the info");
   return self;
}

// JList scaffolding (hmm) ---------------------------------------------------
int JCnrColumn::operator == ( const JCnrColumn &op2) const
{
   return data == op2.data;
}
