/*
 * JCnrRec.cpp
 *
 * Container record object
 * _________________________________________________________________________
 *
 *                     Part of JLib - John Fairhurst
 * _________________________________________________________________________
 *
 *
 */

#define _jlib_err
#include "Jos2.h"
#include "JCnrRec.hpp"
#include "JCnr.hpp"
#include "JDrgItem.hpp"
#include <string.h>

// Exceptions ----------------------------------------------------------------

#define cnrError( s) jlib_throw( new JException( err_cnr, JExn::base, s))

#define rcFromObj( s) ( (RECORDCORE *) (((char *)s) - sizeof( MINIRECORDCORE)))
#define mrcFromObj( s) ( (JMiniRCore*) (((char *)s) - sizeof( MINIRECORDCORE)))
#define objFromRc( s) ( (JCnrRecord *) (((char *)s) + sizeof( MINIRECORDCORE)))

JNoCnrContext::JNoCnrContext() : JException( err_nocnrobjcontext, JExn::cnrContext)
{}

BOOL JCnrRecord::check() const
{
   BOOL rc = true;
   if( !cnr) { jlib_throw( new JNoCnrContext()); rc = false; }
   return rc;
}

// Scaffolding for JList (don't really need this any more) -------------------
int JCnrRecord::operator == ( const JCnrRecord &arg2) const
{
   return text() == arg2.text();
}

// Creation ------------------------------------------------------------------
JCnrRecord::JCnrRecord( const char *text, ulong icon)
           : cnr( 0), core( mrcFromObj( this))
{
   // check we were created with 'new'
   if( core->size != sizeof( MINIRECORDCORE))
      jlib_throw( new JException( err_cnrnotnew, JExn::base));
   else {
      core->hIcon = icon; core->txtIcon = strdup( text);
   }
}

// the all-important operator new overloads; I hope this works...
void *JCnrRecord::operator new( size_t cbSize, JContainer *c)
{
   assertParms( c, "JCnrRec::operator new");
   void *v = c->sendEvent( CM_ALLOCRECORD, cbSize, 1);
   if( !v)
      cnrError( "alloc a record");

   return objFromRc( v);
}

void *JCnrRecord::operator new( size_t cbSize, const JContainer &c)
{
   assertParms( c, "JCnrRec::operator new");
   void *v = c.sendEvent( CM_ALLOCRECORD, cbSize, 1);
   if( !v)
      cnrError( "alloc a record");

   return objFromRc( v);
}

// Deletion & cleanup --------------------------------------------------------
void JCnrRecord::operator delete( void *)
{} // do nothing here; the record gets taken from the cnr & free'd elsewhere.

JCnrRecord::~JCnrRecord()
{
   ::free( core->txtIcon);
}

void JCnrRecord::remove( BOOL freeit)
{
   if( check()) {
   
      void *v = rcFromObj( this);
      long rc = cnr->sendEvent( CM_REMOVERECORD, &v, JMP( 1, CMA_INVALIDATE));
      if( rc == -1)
         cnrError( "remove a rec");
      else
         if( freeit) free();
   }
}

void JCnrRecord::free()
{
   if( check()) {
      void *v = rcFromObj( this);
      delete this; // umm....
      long  rc = cnr->sendEvent( CM_FREERECORD, &v, 1);
      if( !rc) {
         JPMExcep err( "");
         if( err.error() != PMERR_RECORD_CURRENTLY_INSERTED)
            cnrError( "free a rec");
      }
   }
}

// Access to data members ----------------------------------------------------
JStr JCnrRecord::text() const
{
   return JStr( core->txtIcon);
}

JCnrRecord &JCnrRecord::setText( const char *s)
{
   if( check()) {
      ::free( core->txtIcon);
      core->txtIcon = strdup( s);
      void *v = rcFromObj( this);
      BOOL rc = cnr->sendEvent( CM_INVALIDATERECORD, &v,
                                JMP( 1, CMA_TEXTCHANGED));
      if( !rc)
         cnrError( "invalidate a record");
   }
   return self;
}

JCnrRecord &JCnrRecord::setIcon( ulong ic)
{
   if( check()) {
      core->hIcon = ic;
      void *v = rcFromObj( this);
      BOOL rc = cnr->sendEvent( CM_INVALIDATERECORD, &v,
                                JMP( 1, CMA_NOREPOSITION));
      if( !rc)
         cnrError( "invalidate a record");
   }
   return self;
}

// Record rectangles ---------------------------------------------------------
JRect JCnrRecord::textRect() const
{
   JRect rcl;
   if( check()) {
      QUERYRECORDRECT qrr = { sizeof( QUERYRECORDRECT), rcFromObj( this),
                              false, CMA_TEXT };
      BOOL rc = cnr->sendEvent( CM_QUERYRECORDRECT, &rcl, &qrr);
      if( !rc)
         cnrError( "get a rec's rect");
   }
   return rcl;
}

JRect JCnrRecord::iconRect() const
{
   JRect rcl;
   if( check()) {
      QUERYRECORDRECT qrr = { sizeof( QUERYRECORDRECT), rcFromObj( this),
                              false, CMA_ICON };
      BOOL rc = cnr->sendEvent( CM_QUERYRECORDRECT, &rcl, &qrr);
      if( !rc)
         cnrError( "get a rec's rect");
   }
   return rcl;
}

JRect JCnrRecord::recordRect() const
{
   JRect rcl;
   if( check()) {
      QUERYRECORDRECT qrr = { sizeof( QUERYRECORDRECT), rcFromObj( this),
                              false, CMA_ICON | CMA_TEXT };
      BOOL rc = cnr->sendEvent( CM_QUERYRECORDRECT, &rcl, &qrr);
      if( !rc)
         cnrError( "get a rec's rect");
   }
   return rcl;
}

// Record emphasis access ----------------------------------------------------
BOOL JCnrRecord::hasEmphasis( JCnr::emphasis e)  const
{
   BOOL rc = false;
   if( check()) {
      void *v = rcFromObj( this);
      rc = cnr->sendEvent( CM_QUERYRECORDINFO, &v, 1);
      if( !rc)
         cnrError( "query the recordinfo");
      else rc = core->attributes & (ulong) e;
   }
   return rc;
}

JCnrRecord &JCnrRecord::setEmphasis( JCnr::emphasis e, BOOL set)
{
   if( check()) {
      BOOL rc = cnr->sendEvent( CM_SETRECORDEMPHASIS, rcFromObj( this),
                                JMP( set, (ushort) e));
      if( !rc)
         cnrError( "set a rec's emphasis");
   }
   return self;
}

JCnrRecord &JCnrRecord::setReadonly( BOOL f)
{
   if( f) core->attributes |= CRA_RECORDREADONLY;
   else   core->attributes &= ~CRA_RECORDREADONLY;
   return self;
}

BOOL JCnrRecord::isReadonly()
{
   return core->attributes & CRA_RECORDREADONLY;
}

// Parent records in tree view -----------------------------------------------
JCnrRecord &JCnrRecord::collapse()
{
   if( check()) {
      BOOL rc = cnr->sendEvent( CM_COLLAPSETREE, rcFromObj( this));
      if( !rc)
         cnrError( "collapse a tree");
   }
   return self;
}

JCnrRecord &JCnrRecord::expand()
{
   if( check()) {
      BOOL rc = cnr->sendEvent( CM_EXPANDTREE, rcFromObj( this));
      if( !rc)
         cnrError( "expand a tree");
   }
   return self;
}

JCnrRecord &JCnrRecord::setParent( JCnrRecord *newmom, BOOL sibs)
{
   if( check()) {
      TREEMOVE stupidName = { rcFromObj( this), rcFromObj( newmom),
                              (PRECORDCORE) CMA_FIRST, sibs };
      BOOL rc = cnr->sendEvent( CM_MOVETREE, &stupidName);
      if( !rc)
         cnrError( "move a tree");
   }
   return self;
}

// Open the edit MLE for the record ------------------------------------------
JCnrRecord &JCnrRecord::editText()
{
   if( check()) {
      CNREDITDATA data = { sizeof( CNREDITDATA), cnr->handle(),
                           rcFromObj( this), NULL, NULL, 0, 0 };
      BOOL rc = cnr->sendEvent( CM_OPENEDIT, &data);
      if( !rc)
         cnrError( "edit a record's text");
   }
   return self;
}

// Cursor-ey things; get related records -------------------------------------
JCnrRecord *JCnrRecord::next()
{
   JCnrRecord *rec = 0;
   if( check()) {
      void *v = cnr->sendEvent( CM_QUERYRECORD, rcFromObj( this),
                                JMP( CMA_NEXT, CMA_ITEMORDER) );
      if( (long) v == -1)
         cnrError( "query a rec");
      else if( v) rec = cnr->initialise( objFromRc( v));
   }
   return rec;
}

JCnrRecord *JCnrRecord::previous()
{
   JCnrRecord *rec = 0;
   if( check()) {
      void *v = cnr->sendEvent( CM_QUERYRECORD, rcFromObj( this),
                                JMP( CMA_PREV, CMA_ITEMORDER) );
      if( (long) v == -1)
         cnrError( "query a rec");
      else if( v) rec = cnr->initialise( objFromRc( v));
   }
   return rec;
}

JCnrRecord *JCnrRecord::parent()
{
   JCnrRecord *rec = 0;
   if( check()) {
      void *v = cnr->sendEvent( CM_QUERYRECORD, rcFromObj( this),
                                JMP( CMA_PARENT, CMA_ITEMORDER) );
      if( (long) v == -1)
         cnrError( "query a rec");
      else if( v) rec = cnr->initialise( objFromRc( v));
   }
   return rec;
}

JCnrRecord *JCnrRecord::firstChild()
{
   JCnrRecord *rec = 0;
   if( check()) {
      void *v = cnr->sendEvent( CM_QUERYRECORD, rcFromObj( this),
                                JMP( CMA_FIRSTCHILD, CMA_ITEMORDER) );
      if( (long) v == -1)
         cnrError( "query a rec");
      else if( v) rec = cnr->initialise( objFromRc( v));
   }
   return rec;
}

JCnrRecord *JCnrRecord::lastChild()
{
   JCnrRecord *rec = 0;
   if( check()) {
      void *v = cnr->sendEvent( CM_QUERYRECORD, rcFromObj( this),
                                JMP( CMA_LASTCHILD, CMA_ITEMORDER) );
      if( (long) v == -1)
         cnrError( "query a rec");
      else if( v) rec = cnr->initialise( objFromRc( v));
   }
   return rec;
}

JHelpID JCnrRecord::helpID() { return cnr->helpID(); }

// data offsets ---------------------------------------------------------------
long JCnrRecord::iconOffset = -4;
long JCnrRecord::textOffset = -8;

// Event notifies -------------------------------------------------------------
JDragItem *JCnrRecord::initDrag()
{
   JCustomItem *itm = new JCustomItem( JDrag::rmf( JDrag::rm( "blah"),
                                                   JDrag::rf( "blah")),
                                       (ulong) this, JIcon( core->hIcon));
   return itm;
}
